/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/

package com.freescale.system.browser.epl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Widget;

import com.freescale.system.browser2.SystemBrowserActivator2;
import com.freescale.system.browser.epl.ITabDescriptorEpl;
/**
 * Viewer which exposes and manages a set of tabbed sheets. Each tabbed
 * sheet page contains an instance of this viewer. On the left of the
 * viewer is a composite containing the tabs; on the right, the active 
 */
public class TabbedSystemViewerEpl extends StructuredViewer {
	/**
	 * The left part of the viewer; a control which exposes the tabs for all the
	 * sheets in the page.
	 */
	protected final TabbedListEpl fTabChooser;

	/**
	 * The collection of tab descriptors in the tab chooser. 
	 */
	protected List<ITabDescriptorEpl> fTabDescriptors = new ArrayList<>();

	/**
	 * Constructor
	 * 
	 * @param tabChooser the control which allows the user to choose a tab
	 */
	public TabbedSystemViewerEpl(TabbedListEpl tabChooser) {
		this.fTabChooser = tabChooser;
		hookControl(tabChooser);
	}

	/**
	 * Returns a tab descriptor by index. Returns null if out
	 * of range.
	 */
	public ITabDescriptorEpl getNthTab(int index) {
		if (index >= 0 && index < fTabDescriptors.size()) {
			return fTabDescriptors.get(index);
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#reveal(java.lang.Object)
	 */
	@Override
	public void reveal(Object element) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.Viewer#getControl()
	 */
	@Override
	public Control getControl() {
		return fTabChooser;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#doFindInputItem(java.lang.Object)
	 */
	@Override
	protected Widget doFindInputItem(Object element) {
		// Not implemented. Do we need to?
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#doFindItem(java.lang.Object)
	 */
	@Override
	protected Widget doFindItem(Object element) {
		// Not implemented. Do we need to?
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#doUpdateItem(org.eclipse.swt.widgets.Widget, java.lang.Object, boolean)
	 */
	@Override
	protected void doUpdateItem(Widget item, Object element, boolean fullMap) {
		// Not implemented. Do we need to?
		return;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#getSelectionFromWidget()
	 */
	@Override
	protected List<?> getSelectionFromWidget() {
		int index = fTabChooser.getSelectionIndex();
		if (index < 0) {
			return Collections.EMPTY_LIST;
		}
		List<Object> result = new ArrayList<>(1);
		result.add(getNthTab(index));
		return result;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#internalRefresh(java.lang.Object)
	 */
	@Override
	protected void internalRefresh(Object element) {// Not implemented. Do we need to?
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.StructuredViewer#setSelectionToWidget(java.util.List, boolean)
	 */
	@Override
	protected void setSelectionToWidget(@SuppressWarnings("rawtypes") List list, boolean reveal) {
		// We do not consider multiple selections. Only the first
		// element will represent the selection.
		
		if (list == null || list.size() == 0) { 
			// clear selection
			fTabChooser.deselectAll();
		} else {
			TabDescriptorEpl selectedTab = (TabDescriptorEpl)list.get(0);
			int i = 0;
			for (ITabDescriptorEpl tab : fTabDescriptors) {
				if (tab == selectedTab) {
					fTabChooser.select(i, reveal);
					return;
				}
				i++;
			}
			SystemBrowserActivator2.log(IStatus.ERROR, "Unexpected tab selection"); //$NON-NLS-1$
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.viewers.Viewer#inputChanged(java.lang.Object, java.lang.Object)
	 */
	@Override
	protected void inputChanged(Object input, Object oldInput) {
		if (input == null) { 
			return;
		}
		ITabDescriptorEpl[] children = (ITabDescriptorEpl[])getSortedChildren(input);
		fTabChooser.removeAll();
		fTabDescriptors.clear();
		for (final ITabDescriptorEpl child : children) {
			fTabDescriptors.add(child);
			mapElement(child, fTabChooser);
		}
		fTabChooser.setElements(children);
	}	
}
