/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright (c) 2010 Freescale Semiconductor Inc.  All rights reserved.
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/

package com.freescale.system.browser.epl;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.part.Page;

import com.freescale.system.browser.epl.TabbedSheetPageEpl;


/**
 * An abstract implementation of a section in a tab in the tabbed page. Any
 * class wishing to implement a section must subclass this class. The
 * <code>ISection</code> interface should not be used directly.
 */
public abstract class AbstractSectionEpl implements ISectionEpl {

	/**
	 * The standard label width when labels for sections line up on the left
	 * hand side of the composite.
	 */
	public static final int STANDARD_LABEL_WIDTH = 100;

	/**
	 * The tabbed sheet page
	 */
	private TabbedSheetPageEpl fTabbedSheetPage;

	private boolean fInvalidated;

	/**
	 * Get the widget factory for the property sheet page.
	 * 
	 * @return the widget factory.
	 */
	public TabbedWidgetFactoryEpl getWidgetFactory() {
		return fTabbedSheetPage.getWidgetFactory();
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.ISection#createControls(org.eclipse.swt.widgets.Composite, org.eclipse.ui.part.Page)
	 */
	@Override
	public void createControls(Composite parent, Page page) {
		fTabbedSheetPage = (TabbedSheetPageEpl)page;
	}

	/**
	 * The Page that houses this section's GUI. This is the instance passed into
	 * createControls();
	 * 
	 * @return the Page instance
	 */
	protected Page getPage() {
		return fTabbedSheetPage;
	}

	/**
	 * A subclass should invoke this implementation at the beginning of its
	 * implementation.
	 * 
	 * @see com.freescale.system.browser.ui.ISection#aboutToBeShown()
	 */
	@Override
	public void aboutToBeShown() {
		if (fInvalidated) {
			refresh();
		}
	}

	/**
	 * A subclass should invoke this implementation at the beginning of its
	 * implementation.
	 * 
	 * @see com.freescale.system.browser.ui.ISection#aboutToBeHidden()
	 */
	@Override
	public void aboutToBeHidden() {// empty default implementation 
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.internal.ui.tabbed.ISection#getMinimumHeight()
	 */
	@Override
	public int getMinimumHeight() {
		return SWT.DEFAULT;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.freescale.system.browser.ui.tabbed.ISection#shouldUseExtraSpace()
	 */
	@Override
	public boolean shouldUseExtraSpace() {
		return false;
	}
	
	/**
	 * A subclass should call super.refresh() at the end of its implementation
	 * 
	 * @see com.freescale.system.browser.ui.ISection#refresh()
	 */
	@Override
	public void refresh() {
		fInvalidated = false;
	}
	
	/** 
	 * A subclass will generally have no need to override this method. If it does
	 * it should make sure to invoke this implementation.
	 * 
	 * @see com.freescale.system.browser.ui.ISection#invalidate()
	 */
	@Override
	public void invalidate() {
		fInvalidated = true;
	}
}
