/**
 * Copyright 2017-2020 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;

/**
 * Class represents standard array GUI representation with items as columns. 
 * @author Juraj Ondruska
 */
public class ArrayControlStandardHorizontal extends AArrayControlStandard {
	/** Composite with the content */
	private @Nullable Composite contentComposite;
	/** Number of columns in the item composite */
	public static final int GUI_ITEM_COLS = ComponentSettingView.CONFIG_SET_COLS + 1;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlStandardHorizontal(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlStandard#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		// create main horizontally scrollable area
		ScrolledComposite scrolledComposite = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.BORDER);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
		// create the content in the content composite
		Composite contentCompositeLoc = createComposite(scrolledComposite, minColumnCount(children.size()));
		super.createMainControl(contentCompositeLoc);
		// configure the scrolled composite to display the content properly
		scrolledComposite.setContent(contentCompositeLoc);
		scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
		// add listener for proper resizing of the scrolled composite
		contentCompositeLoc.addListener(SWT.Resize, e -> {
			scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			scrolledComposite.requestLayout();
		});
		addScrollListener(contentCompositeLoc, composite);
		contentComposite = contentCompositeLoc;
		return scrolledComposite;
	}

	/**
	 * @param elementCount number of table elements to display
	 * @return minimal number of columns
	 */
	private int minColumnCount(int elementCount) {
		return elementCount + (isUiArrayFixedSpecified() ? 0 : 1);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		ScrolledComposite scrolledComposite = (ScrolledComposite) contentControl;
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			boolean layoutChange = updateMainCommonImpl(updateType, contentCompositeLoc);
			((GridLayoutComponents) contentCompositeLoc.getLayout()).numColumns = minColumnCount(arrayConfig.getChildren().size());
			// refresh GUI in case of some change
			if (layoutChange) {
				contentCompositeLoc.requestLayout();
				scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
			}
		}
		updateErrorDecoration(contentControl);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlStandard#recreateAddButton(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void recreateAddButton(@NonNull Composite composite) {
		createAddButton(composite, 1);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlStandard#createItem(org.eclipse.swt.widgets.Composite, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl)
	 */
	@Override
	protected void createItem(@NonNull Composite compositeForItem, @NonNull IChildControl childControl) {
		createItemGui(childControl, createComposite(compositeForItem, GUI_ITEM_COLS), ComponentSettingView.CONFIG_SET_COLS);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlStandard#disposeChild(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl)
	 */
	@Override
	protected void disposeChild(IChildControl childControl) {
		Composite itemGui = childControl.getContent();
		if (itemGui != null) {
			itemGui.dispose();
		}
		childControl.dispose();
	}
}
