/**
 * Copyright 2019-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Layout;

import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.data.setting.BoolSetting;
import com.nxp.swtools.resourcetables.model.data.setting.DynamicEnumSetting;
import com.nxp.swtools.resourcetables.model.data.setting.EnumSetting;
import com.nxp.swtools.resourcetables.model.data.setting.IBaseModel;
import com.nxp.swtools.resourcetables.model.data.setting.ISetting;
import com.nxp.swtools.resourcetables.model.data.setting.StructSetting;

/**
 * Class represents array of radio button groups. Usable only for enum, boolean or structure of enums or booleans
 * @author Tomas Rudolf - nxf31690
 */
public abstract class ArrayControlRadio extends AArrayControlInternal {
	/** Index used in case of not structured setting of array */
	public static final int NOT_STRUCTURED_SETTING_INDEX = -1;
	/** Composite with the content */
	protected @Nullable Composite contentComposite;
	/** Id of first child */
	protected static final @NonNull String ID_ZERO = "0"; //$NON-NLS-1$
	/** Stores information which child of structure should be shown as independent array */
	private int whichChildOfStructure = NOT_STRUCTURED_SETTING_INDEX;
	/** Links to other copies of this class created because of structure was given as input */
	private List<ArrayControlRadio> otherChildrenOfStructure = new ArrayList<>();

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlRadio(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/**
	 * Sets which child of structure should be used
	 * @param index of child in structure
	 */
	public void setWhichChildOfStructure(int index) {
		this.whichChildOfStructure = index;
	}

	/**
	 * Returns which child of structure should be used
	 * @return index of child in structure
	 */
	public int getWhichChildOfStructure() {
		return whichChildOfStructure;
	}

	/**
	 * Is given setting supported by this GUI representation
	 * @param setting to be checked
	 * @return {@code true} if given setting is supported, otherwise returns {@code false}
	 */
	public boolean isSettingSupported(IBaseModel setting) {
		boolean isEnum = setting instanceof EnumSetting;
		boolean isDynamicEnum = setting instanceof DynamicEnumSetting;
		boolean isBool = setting instanceof BoolSetting;
		return isEnum || isDynamicEnum || isBool;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		ISetting settingInsideArray = getChild().getInstanceSetting();
		if (!shouldShowMoreInstances()) {
			super.create(composite, colSpan);
		} else {
			// Create other settings
			if (settingInsideArray instanceof StructSetting) {
				StructSetting structSetting = (StructSetting) settingInsideArray;
				List<ISetting> settingsList = new ArrayList<>(structSetting.getSettings().values());
				if (isSettingSupported(settingsList.get(0))) {
					// First create this control with first setting
					setWhichChildOfStructure(0);
					super.create(composite, colSpan);
				}
				// Create rest of controls in parent composite
				for (int childIndex = 1; childIndex < settingsList.size(); childIndex++) {
					ISetting settingOfStructure = settingsList.get(childIndex);
					if (isSettingSupported(settingOfStructure)) {
						ArrayControlRadio radioGroup = createAnotherControl();
						otherChildrenOfStructure.add(radioGroup);
						radioGroup.setWhichChildOfStructure(childIndex);
						radioGroup.create(composite, colSpan);
					}
				}
			}
		}
	}

	/**
	 * @return another control of this type
	 */
	protected abstract ArrayControlRadio createAnotherControl();

	/*
	 * (non-Javadoc)
	 * @see
	 * com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#update(com.nxp.swtools.periphs.gui.view.componentsettings.
	 * IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		super.update(updateType);
		otherChildrenOfStructure.forEach(x -> x.update(updateType));
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateSelectionOfExpandGroupContent()
	 */
	@Override
	protected void updateSelectionOfExpandGroupContent() {
		// if no child in table, set the expand control instance null, otherwise set the tableComposite instance
		if (children.isEmpty()) {
			expandGroupContent = null;
		} else {
			expandGroupContent = contentComposite;
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateLabelContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateLabelContent(@NonNull Control label, @NonNull UpdateType updateType) {
		super.updateLabelContent(label, updateType);
		if (label instanceof Label) {
			Label labelLoc = (Label) label;
			if (getWhichChildOfStructure() == NOT_STRUCTURED_SETTING_INDEX) {
				super.updateLabelContent(label, updateType);
			} else {
				StructSetting structSetting = (StructSetting) getChild().getInstanceSetting();
				List<ISetting> settings = new ArrayList<>(structSetting.getSettings().values());
				String uiName = UtilsText.safeString(getChild().getUiName() + UtilsText.SPACE + UtilsText.DASH + UtilsText.SPACE
						+ settings.get(getWhichChildOfStructure()).getUINameString());
				labelLoc.setText(uiName);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlStandard#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		createLabelWithControls(composite);
		// create main horizontally scroll-able area
		ScrolledComposite scrolledComposite = new ScrolledComposite(composite, SWT.H_SCROLL | SWT.BORDER);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
		// create the content in the content composite
		getControlOptions().borderHidden(true);
		Composite contentCompositeLoc = createComposite(scrolledComposite, children.size());
		contentCompositeLoc.setLayout(createLayout(children.size()));
		contentCompositeLoc.setLayoutData(new GridDataComponents());
		contentComposite = contentCompositeLoc;
		// FIXME TomasR v99 Discuss getting back functionality for arrays of structures with enums/booleans
		createContent(contentCompositeLoc);
		// configure the scrolled composite to display the content properly
		scrolledComposite.setContent(contentCompositeLoc);
		// add listener for proper resizing of the scrolled composite
		contentCompositeLoc.addListener(SWT.Resize, e -> {
			contentCompositeLoc.pack();
			scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
		});
		addScrollListener(contentCompositeLoc, scrolledComposite);
		addScrollListener(scrolledComposite, composite);
		return scrolledComposite;
	}

	/**
	 * Creates layout for main composite
	 * @param columns in composite
	 * @return layout
	 */
	private static Layout createLayout(int columns) {
		GridLayoutComponents layout = new GridLayoutComponents(columns, false);
		layout.horizontalSpacing = 0;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.verticalSpacing = 0;
		return layout;
	}

	/**
	 * Returns number of columns required to show content
	 * @return number of columns
	 */
	protected int getColumnCount() {
		return children.size() + (isUiArrayFixedSpecified() ? 0 : 1) + 1 /* Left header */;
	}

	/**
	 * Check whether it is needed to show multiple array controls for inner setting of this array
	 * @return {@code true} if inner setting is structure and has more than one setting inside, otherwise returns {@code false}
	 */
	private boolean shouldShowMoreInstances() {
		// Ends recursion for sub-children of structure
		if (getWhichChildOfStructure() != NOT_STRUCTURED_SETTING_INDEX) {
			return false;
		}
		ISetting settingInsideArray = getChild().getInstanceSetting();
		if (settingInsideArray instanceof StructSetting) {
			StructSetting structSetting = (StructSetting) settingInsideArray;
			LinkedHashMap<@NonNull String, ISetting> settings = structSetting.getSettings();
			return settings.size() > 1;
		}
		return false;
	}

	/**
	 * Implementation of create main control in classes that extend this class
	 * @param composite composite in which the content should be created
	 * @param names list of names
	 * @param ids list of ids
	 */
	protected abstract void createContent(@NonNull Composite composite);

	/**
	 * Changes value of given config
	 * @param configToChange config for change
	 * @param value value which should be set
	 * @return {@code true} if value was changed successfully or {@code false} if not. Returns {@code false} when given config is not scalar
	 */
	protected boolean changeValue(@Nullable ISettingConfig configToChange, @NonNull String value) {
		if (configToChange instanceof ScalarConfig) {
			ScalarConfig scalarConfig = (ScalarConfig) configToChange;
			return getControllerWrapper().getController().setValue(scalarConfig, value, this.getClass());
		}
		return false;
	}

	/**
	 * Implementation of update main control in classes that extend this class
	 * @param updateType type of update
	 * @param composite composite in which the update occurs
	 * @return {@code true} if something changed, otherwise returns {@code false}
	 */
	protected abstract boolean updateMainCommonImpl(@NonNull UpdateType updateType, @NonNull Composite composite);

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control,
	 * com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		ScrolledComposite scrolledComposite = (ScrolledComposite) contentControl;
		Composite contentCompositeLoc = contentComposite;
		boolean layoutChange = false;
		if (updateType == UpdateType.NORMAL) {
			if (contentCompositeLoc != null) {
				layoutChange = updateMainCommonImpl(updateType, contentCompositeLoc);
				// refresh GUI in case of some change
				if (layoutChange) {
					contentCompositeLoc.requestLayout();
					scrolledComposite.setMinSize(contentCompositeLoc.computeSize(SWT.DEFAULT, SWT.DEFAULT));
				}
			}
		}
		updateErrorDecoration(contentControl);
		// FIXME TomasR v13 maintenance - Investigate. This could lead to executing multiple updates of arrays of structures with enums/booleans
		// Update others which were created by this class
		for (ArrayControlRadio other : otherChildrenOfStructure) {
			Control mainControlLoc = other.mainControl;
			if (mainControlLoc != null) {
				other.updateMainContent(mainControlLoc, updateType);
			}
		}
		// if no child in table, set the expand control instance null, otherwise set the scrolledComposite instance
		if (children.isEmpty()) {
			expandGroupContent = null;
		} else {
			expandGroupContent = scrolledComposite;
		}
		if ((contentCompositeLoc != null) && (updateType == UpdateType.NORMAL) && (layoutChange)) {
			ScrolledCompositeHelper.updateScrollSize(contentCompositeLoc);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateButtons(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateButtons(@Nullable UpdateType updateType) {
		if (!areButtonsEnabled()) {
			updateType = UpdateType.FORCE_DISABLE;
		}
		updateAddButton(updateType);
		updateRemoveButton(updateType);
		// Do not update up and down buttons as they should be invisible in this GUI representation
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#setAllSettingsTo(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.internal.ArrayControlItemMenu, com.nxp.swtools.periphs.gui.view.componentsettings.internal.IArrayControlItemMenuControl)
	 */
	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		// Not supported in this GUI representation
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public @Nullable ISettingConfig getSelection() {
		// FIXME TomasR v99 - Consider proper implementation of lastly selected radio group item
		// Always return the last one
		List<@NonNull ISettingConfig> arrayChildren = getChild().getChildren();
		return arrayChildren.get(arrayChildren.size() - 1);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		// Reset this value to unselected/single setting because of function create can be called multiple times
		setWhichChildOfStructure(NOT_STRUCTURED_SETTING_INDEX);
		for (ArrayControlRadio other : otherChildrenOfStructure) {
			other.dispose();
		}
		otherChildrenOfStructure.clear();
	}
}
