/**
 * Copyright 2018-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlFactory;
import com.nxp.swtools.periphs.gui.view.componentsettings.ComponentSettingView;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class represents standard array GUI representation Master (left) and Detail (right)
 * @author Tomas Rudolf
 * @author Viktar Paklonski
 * @author Juraj Ondruska
 */
public class ArrayControlMasterDetail extends AArrayControlGroup {
	/** Icon of remove-cross */
	static final @Nullable Image IMAGE_CROSS = ToolsImages.getImage(IToolsImages.ICON_REMOVE_CROSS_CIRCLE);
	/** Minimal count of items in Master table */
	private static final int MIN_ITEMS_IN_TABLE = 5;
	/** Maximal count of items in Master table */
	private static final int MAX_ITEMS_IN_TABLE = 24;
	/** Minimal width of the Master table */
	private static final int MIN_CHARS_IN_TABLE = 8;
	/** Composite for detail */
	private @Nullable Composite detailComposite;
	/** UI - composite */
	private @Nullable Composite uiComposite;
	/** Master tableViewer */
	@Nullable TableViewer masterViewer;
	/** The currently displayed setting - set after the control was displayed - should be disposed prior to displaying new control */
	private @Nullable IChildControl displayedControl;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlMasterDetail(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlStandard#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		Composite contentComposite = createComposite(composite, 1);
		contentComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		createLabelWithControls(contentComposite);
		Composite uiCompositeLoc = new Composite(contentComposite, SWT.NONE);
		uiCompositeLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, ComponentSettingView.CONFIG_SET_COLS, 1));
		uiComposite = uiCompositeLoc;
		GridLayoutComponents layout = new GridLayoutComponents(3, false);
		layout.marginWidth = layout.marginHeight = 0;
		uiCompositeLoc.setLayout(layout);
		createMasterPart(uiCompositeLoc);
		Label separator = new Label(uiCompositeLoc, SWT.SEPARATOR | SWT.VERTICAL);
		separator.setLayoutData(new GridDataComponents(SWT.CENTER, SWT.FILL, false, false));
		createDetailPart(uiCompositeLoc);
		createInfoLabel(contentComposite, ComponentSettingView.CONFIG_SET_COLS);
		restoreSelectionFromStorage();
		return contentComposite;
	}

	/**
	 * Create Detail part
	 * @param composite to create Detail part in
	 */
	private void createDetailPart(@NonNull Composite composite) {
		Composite detailPart = new Composite(composite, SWT.NONE);
		detailComposite = detailPart;
		detailPart.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, true));
		GridLayoutComponents detailPartLayout = new GridLayoutComponents(ComponentSettingView.CONFIG_SET_COLS, false);
		detailPartLayout.marginLeft = 3;
		detailPartLayout.horizontalSpacing = 8;
		detailPart.setLayout(detailPartLayout);
	}

	/**
	 * Create Master part
	 * @param composite to create Master part in
	 * @return Master part composite
	 */
	private @NonNull Composite createMasterPart(@NonNull Composite composite) {
		Composite masterPart = new Composite(composite, SWT.NONE);
		masterPart.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, false, true));
		GridLayoutComponents layout = new GridLayoutComponents(1, true);
		layout.marginWidth = layout.marginHeight = 0;
		layout.verticalSpacing = 0;
		masterPart.setLayout(layout);
		// create table
		Table masterContentTable = new Table(masterPart, SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION | SWT.RESIZE | SWT.H_SCROLL | SWT.V_SCROLL);
		TableLayout tableLayout = new TableLayout();
		masterContentTable.setLayout(tableLayout);
		SWTFactoryProxy.INSTANCE.setTestId(masterContentTable, TestIDs.PERIPHS_MASTER_TABLE + UtilsText.UNDERSCORE + arrayConfig.getId());
		masterContentTable.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, false, false));
		masterContentTable.setLinesVisible(true);
		TableViewer masterContent = new TableViewer(masterContentTable);
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(masterContent);
		masterViewer = masterContent;
		createChildColumn(masterContent);
		// create table context menu
		if (isUiArrayReorderSpecified() || !isUiArrayFixedSpecified()) {
			registerControl(new ArrayControlItemMenuContext(this, masterContent));
		}

		masterContent.setContentProvider(ArrayContentProvider.getInstance());
		setViewerInput(masterContent);
		updateColumnWidth(masterContent);
		masterContent.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				ISelection selection = event.getSelection();
				if (selection instanceof StructuredSelection) {
					Object selectedElement = ((StructuredSelection) selection).getFirstElement();
					if ((selectedElement instanceof IChildControl) || (selectedElement == null)) {
						IChildControl newSelection = (IChildControl) selectedElement;
						setSelectedChild(newSelection);
						updateDetailContent();
						updateLabelWithControls(getControlUpdateType());
					}
				}
			}
		});
		addKeyboardListener(masterContentTable);
		addScrollListener(masterContentTable, masterPart);
		return masterPart;
	}

	/**
	 * Select last item in the Master table.
	 */
	@Override
	public void selectLastItem() {
		IChildControl childToSelect = null;
		if (!children.isEmpty()) {
			childToSelect = children.get(children.size() - 1);
		}
		selectItem(childToSelect);
	}

	/**
	 * Set proper input to the table viewer.
	 * @param viewer to fill
	 */
	private void setViewerInput(@NonNull TableViewer viewer) {
		IChildControl previouslySelectedChild = getChildToSelect();
		viewer.setInput(new ArrayList<>(children));
		selectItem(previouslySelectedChild);
		int tableSize = Math.max(Math.min(viewer.getTable().getItemCount(), MAX_ITEMS_IN_TABLE), MIN_ITEMS_IN_TABLE);
		((GridDataComponents) Objects.requireNonNull(viewer.getTable().getLayoutData())).heightHint = viewer.getTable().getItemHeight() * tableSize;
		Objects.requireNonNull(viewer.getTable().getParent()).requestLayout();
	}

	/**
	 * Creates column with child names and status decoration images
	 * @param tableViewer to create column in
	 */
	private void createChildColumn(@NonNull TableViewer tableViewer) {
		TableViewerColumn tableViewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		tableViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			@Override
			public Image getImage(Object element) {
				if (element instanceof IChildControl) {
					return getStatusIcon(((IChildControl) element).getChild());
				}
				return null;
			}

			@Override
			public String getText(Object element) {
				if (element instanceof IChildControl) {
					ISettingConfig config = (ISettingConfig) ((IChildControl) element).getChild();
					String key = arrayConfig.getSettingKey(config, arrayConfig.getExpressionContext());
					return (key != null) ? key : config.getUiName();
				}
				return null;
			}

			@Override
			public Color getForeground(Object element) {
				if (element instanceof ISettingConfig) {
					if (!((ISettingConfig) element).isEnabled()) {
						return SwToolsColors.getColor(SwToolsColors.DISABLED_FG);
					}
				}
				return super.getForeground(element);
			}

			@Override
			public String getToolTipText(Object element) {
				if (element instanceof IChildControl) {
					ISettingConfig config = (ISettingConfig) ((IChildControl) element).getChild();
					return ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(config));
				}
				return null;
			}
		});
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateSelectionOfExpandGroupContent()
	 */
	@Override
	protected void updateSelectionOfExpandGroupContent() {
		// if no child in table, set the expand control instance null, otherwise set the uiComposite instance
		if (children.isEmpty()) {
			expandGroupContent = null;
		} else {
			expandGroupContent = uiComposite;
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		TableViewer masterViewerLoc = masterViewer;
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Iterator<IChildControl> childrenIterator = children.iterator();
			Iterator<ISettingConfig> arrayItemIterator = arrayConfig.getChildren().iterator();
			ISettingConfig arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			// dispose controls of all removed items
			while (childrenIterator.hasNext()) {
				IChildControl childControl = childrenIterator.next();
				updateControlOptions(childControl);
				if (!childControl.getChild().equals(arrayItem)) {
					// Remove child
					childrenIterator.remove();
					childControl.dispose();
				} else { // match
					arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
				}
			}
			Composite uiCompositeLoc = uiComposite;
			while (arrayItem != null) {
				addItem(arrayItem, updateType);
				arrayItem = arrayItemIterator.hasNext() ? arrayItemIterator.next() : null;
			}
			if (uiCompositeLoc != null) {
				Point size = uiCompositeLoc.getSize();
				boolean isChanged = !size.equals(new Point(0, 0));
				if ((KEPreferences.isAnimationsEnabled() && isChanged && updateType == UpdateType.NORMAL) && (children.isEmpty() || (children.size() == 1 && !uiCompositeLoc.isVisible()))) {
					setControlVisibleAnimation(uiCompositeLoc, null, !children.isEmpty());
				} else {
					setControlVisible(uiCompositeLoc, !children.isEmpty());
				}
			}
			updateMasterContent();
			if (masterViewerLoc != null) {
				masterViewerLoc.getTable().setEnabled(arrayConfig.isEnabled());
			}
			selectItem(getChildToSelect());
		} else {
			if (masterViewerLoc != null) {
				final Object input = masterViewerLoc.getInput();
				if (input instanceof List<?>) {
					masterViewerLoc.update(((List<?>) input).toArray(), null);
				} else {
					masterViewerLoc.refresh();
				}
			}
			updateDetailContent(UpdateType.PROBLEM_DECORATION);
			updateLabelWithControls(UpdateType.PROBLEM_DECORATION);
		}
	}

	/**
	 * Update the column width
	 * @param tableViewer The table viewer
	 */
	void updateColumnWidth(@NonNull TableViewer tableViewer) {
		// This type of array (Master-Detail arrays) has only one column
		TableColumn column = tableViewer.getTable().getColumn(0);
		column.pack();
		column.setWidth(Math.max(tableViewer.getTable().computeSize(getTableWidthHint(), tableViewer.getTable().getSize().y).x, column.getWidth()));
	}

	/**
	 * Update content of Master table
	 */
	void updateMasterContent() {
		TableViewer masterViewerLoc = masterViewer;
		if ((masterViewerLoc != null) && (!masterViewerLoc.getControl().isDisposed())) {
			setViewerInput(masterViewerLoc);
			updateColumnWidth(masterViewerLoc);
			Table table = masterViewerLoc.getTable();
			Object data = table.getLayoutData();
			if (getControlOptions().isTableRowsVisibleLimitSet() && data instanceof GridDataComponents) {
				GridDataComponents tableLayoutData = (GridDataComponents) data;
				int wantedMaxHeight = table.getItemHeight() * getControlOptions().getTableVisibleRows();
				Point size = table.computeSize(SWT.DEFAULT, SWT.DEFAULT);
				tableLayoutData.heightHint = Math.min(size.y, wantedMaxHeight);
			}
		}
	}

	/**
	 * Select given item in table
	 * @param control to be selected
	 */
	@Override
	public void selectItem(@Nullable IChildControl control) {
		Viewer masterViewerLoc = masterViewer;
		if ((masterViewerLoc != null) && (!masterViewerLoc.getControl().isDisposed())) {
			// XXX Do not block same selection as it will break "selected child" mechanism and updating of buttons along it
			ISelection selection = (control == null) ? StructuredSelection.EMPTY : new StructuredSelection(control);
			masterViewerLoc.setSelection(selection, true);
		}
	}

	/**
	 * Updates content of Detail part
	 */
	void updateDetailContent() {
		updateDetailContent(getControlUpdateType());
	}

	/**
	 * Updates content of Detail part
	 * @param updateType type of the update
	 */
	void updateDetailContent(@NonNull UpdateType updateType) {
		Composite detailCompositeLoc = detailComposite;
		if ((detailCompositeLoc == null) || (detailCompositeLoc.isDisposed())) { // the detail composite can be disposed by the child displayed in it
			if (uiComposite != null) {
				createDetailPart(uiComposite);
				detailCompositeLoc = detailComposite;
			}
		}
		if ((detailCompositeLoc != null) && (!detailCompositeLoc.isDisposed())) {
			detailCompositeLoc.setRedraw(false);
			IChildControl control = getSelectedChild();
			IChildControl displayedControlLoc = displayedControl;
			if ((control == displayedControlLoc) && (displayedControlLoc != null)) {
				if (displayedControlLoc.isDisposed()) {
					displayedControlLoc.create(detailCompositeLoc, ComponentSettingView.CONFIG_SET_COLS);
				}
				displayedControlLoc.update(updateType);
			} else {
				if (displayedControl != null) {
					displayedControl.dispose();
					displayedControl = null;
				}
				if (control != null) {
					updateControlOptions(control);
					control.create(detailCompositeLoc, ComponentSettingView.CONFIG_SET_COLS);
					control.update(updateType);
					displayedControl = control;
				}
			}
			detailCompositeLoc.requestLayout();
			ScrolledCompositeHelper.updateScrollSize(detailCompositeLoc);
			Composite mainCompositeLoc = (Composite) mainControl;
			if ((mainCompositeLoc != null) && (!mainCompositeLoc.isDisposed())) {
				mainCompositeLoc.requestLayout();
				mainCompositeLoc.redraw();
			}
			detailCompositeLoc.setRedraw(true);
		}
	}

	/**
	 * Add item to Master table
	 * @param arrayItem Newly added array item
	 * @param updateType type of update for newly created child
	 * @return {@code true} on success, {@code false} otherwise
	 */
	private boolean addItem(@NonNull ISettingConfig arrayItem, @NonNull UpdateType updateType) {
		IChildControl childControl = ChildControlFactory.create(arrayItem, null, controllerWrapper);
		if (childControl != null) {
			updateControlOptions(childControl);
			children.add(childControl);
			childControl.update(updateType);
			return true;
		}
		return false;
	}

	/**
	 * Updates control options of given childControl for master-detail purposes
	 * @param childControl to change
	 */
	protected void updateControlOptions(IChildControl childControl) {
		ControlOptions childControlOptions = childControl.getControlOptions();
		childControlOptions.labelHidden(true);
		if (childControl.getChild() instanceof IChildProvidable) { // hide border only in case of structures, arrays, etc.
			childControlOptions.borderHidden(true);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#setAllSettingsTo(com.nxp.swtools.periphs.gui.view.componentsettings.internal.ArrayControlItemMenu, com.nxp.swtools.periphs.gui.view.componentsettings.internal.IArrayControlItemMenuControl)
	 */
	@Override
	public void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		// not supported for master-detail array
	}

	/**
	 * @return preferred width of the table
	 */
	private int getTableWidthHint() {
		double length = MIN_CHARS_IN_TABLE;
		List<@NonNull Double> columnWidths = getControlOptions().getTableColumnWidths();
		if ((columnWidths != null) && !columnWidths.isEmpty()) {
			length = columnWidths.get(0).doubleValue();
		}
		return (int) (length * computeColumnWidthUnitSize(Objects.requireNonNull(uiComposite)));
	}

	/**
	 * Get currently selected setting in array
	 * @return currently selected setting in array or {@code null} if some problems appear
	 */
	@Override
	public @Nullable ISettingConfig getSelection() {
		TableViewer masterViewerLoc = masterViewer;
		if (masterViewerLoc != null) {
			ISelection selection = masterViewerLoc.getSelection();
			return super.getChildFromSelection(selection);
		}
		return null;
	}
}