/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.List;
import java.util.function.Supplier;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.PeripheralsToolTippableFormatter;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ArrayControl;
import com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions.ShowContentAs;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig.Representation;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableDescriptionAppendDecorator;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class represents common base for each of the array GUI representations (table/standard, vertical/horizontal).
 * @author Juraj Ondruska
 * @author Viktar Paklonski
 */
public abstract class AArrayControlInternal extends ChildProvidableControlBase {
	/** Count of columns in label composite */
	private static final int LABEL_CONTROL_COLUMNS_COUNT = 3;
	/** Space around whole title bar in pixels */
	private static final int TITLEBAR_SPACE_AROUND = 0;
	/** Offset between label and buttons in pixels */
	private static final int TITLEBAR_OFFSET = 10;
	/** Number of items in title-bar */
	private static final int TITLEBAR_NUMBER_OF_BUTTONS = 4;
	/** Button used for adding new items to the array */
	private @Nullable Button buttonAdd;
	/** Button used for removing items from the array */
	private @Nullable Button buttonRemove;
	/** Button used for moving items up in the array */
	private @Nullable Button buttonUp;
	/** Button used for moving items down in the array */
	private @Nullable Button buttonDown;
	/** The array configuration for which to create GUI */
	protected final @NonNull ArrayConfig arrayConfig;
	/** Label of array */
	private @Nullable Control label;
	/** Label with info about array */
	private @Nullable Label infoLabel;
	/** {@link ArrayControl} which created this delegate */
	private ArrayControl parentArrayControl;
	/** Combo box to switch representations */
	protected InstantSearchList representationCombo;
	/** Composite of the representation combo */
	protected Composite representationComposite;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected AArrayControlInternal(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
		this.arrayConfig = arrayConfig;
	}
	
	/**
	 * 
	 * @return the generic controllerWrapper
	 */
	public @NonNull IControllerWrapper getControllerWrapper() {
		return controllerWrapper;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		super.create(composite, colSpan);
		Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			// Override layout for label to fill entire row of grid
			labelControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, currentColSpan, 1));
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			// Override layout for main mainControl to fill entire row of grid
			mainControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, currentColSpan, 1));
		}
	}

	/**
	 * Creates info label
	 * @param composite to create in
	 * @param colSpan of label
	 */
	protected void createInfoLabel(@NonNull Composite composite, int colSpan) {
		Label infoLabelLoc = new Label(composite, SWT.NONE);
		infoLabelLoc.setText(Messages.get().AArrayControlInternal_AddItemInfoLabel);
		GridDataComponents layoutData = new GridDataComponents(SWT.LEFT, SWT.TOP, true, false, colSpan, 1);
		layoutData.exclude = true;
		infoLabelLoc.setVisible(false);
		infoLabelLoc.setLayoutData(layoutData);
		infoLabel = infoLabelLoc;
	}

	/**
	 * Sets visible flag on info label control
	 * @param visible
	 */
	protected void setInfoLabelVisible(boolean visible) {
		Label infoLabelLoc = infoLabel;
		if ((infoLabelLoc != null) && !infoLabelLoc.isDisposed()) {
			setControlVisible(infoLabelLoc, visible);
		}
	}

	/**
	 * Create main control.
	 * @param composite to create control in
	 * @return the newly created control
	 */
	@Override
	abstract public @Nullable Control createMainControl(@NonNull Composite composite);

	/**
	 * Get currently selected setting in array
	 * @return currently selected setting in array or {@code null} if some problems appear
	 */
	public @Nullable ISettingConfig getSelection() {
		// Should be overwritten
		return null;
	}

	/**
	 * Get child from given selection
	 * @param selection to be used
	 * @return child from selection or {@code null} if child cannot be found 
	 */
	public ISettingConfig getChildFromSelection(ISelection selection) {
		if (selection instanceof StructuredSelection) {
			StructuredSelection structuredSelection = (StructuredSelection) selection;
			Object firstElement = structuredSelection.getFirstElement();
			if (firstElement instanceof IChildControl) {
				IChildControl childControl = (IChildControl) firstElement;
				IChild childControlChild = childControl.getChild();
				if (childControlChild instanceof ISettingConfig) {
					return (ISettingConfig) childControlChild;
				}
			}
		}
		return null;
	}

	/**
	 * Select last item in array
	 */
	public void selectLastItem() {
		// Should be overwritten
	}

	/**
	 * Select given item in table
	 * @param control to be selected
	 */
	public void selectItem(IChildControl control) {
		// Should be overwritten
	}

	/**
	 * Update main content of a setting configuration, e.g. an editor.
	 * @param mainControlLoc the control to update (control which was returned by the {@link #createMainControl(Composite)} method)
	 * @param updateType whether to disable children (if there are some children) and all the controls
	 */
	@Override
	abstract public void updateMainContent(@NonNull Control mainControlLoc, @NonNull UpdateType updateType);

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		super.update(updateType);
		updateLabelWithControls(getChild().isEnabled() ? updateType : UpdateType.FORCE_DISABLE); // Update label at only on one place for all graphical representations
		setInfoLabelVisible((updateType != UpdateType.FORCE_DISABLE) && getChild().isEnabled() && !getControlOptions().isArrayFixed() && children.isEmpty());
		updateRepresentationComposite();
	}

	/**
	 * @return check whether indices of the array should be shown
	 */
	protected boolean shouldShowIndices() {
		return !getControlOptions().isArrayIndicesHidden();
	}

	/**
	 * Creates label with controls to add, remove and move items
	 * @param composite to create in
	 * @return created control or {@code null} if none was created
	 */
	public @Nullable Control createLabelWithControls(@NonNull Composite composite) {
		boolean showBar = !isUiLabelHiddenSpecified() || !isUiArrayFixedSpecified() || isUiArrayReorderSpecified();
		if (showBar) {
			// Create main content composite
			Composite contentComposite = new Composite(composite, SWT.NONE);
			contentComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false));
			GridLayoutComponents contentCompositeLayout = new GridLayoutComponents(LABEL_CONTROL_COLUMNS_COUNT, false);
			contentCompositeLayout.marginWidth = contentCompositeLayout.marginHeight = TITLEBAR_SPACE_AROUND;
			contentComposite.setLayout(contentCompositeLayout);
			// Create label
			if (!isUiLabelHiddenSpecified()) {
				Control labelLoc = super.createLabelControl(contentComposite);
				label = labelLoc;
				if (labelLoc != null) {
					labelLoc.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
					createErrorDecoration(labelLoc, SWT.LEFT | SWT.TOP);
					registerCopyPasteMenuOnLabel();
				}
			} else {
				contentCompositeLayout.numColumns--;
			}
			// Create buttons composite
			Composite buttonsComposite = new Composite(contentComposite, SWT.NONE);
			buttonsComposite.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.TOP, false, false));
			GridLayoutComponents buttonsCompositeLayout = new GridLayoutComponents(TITLEBAR_NUMBER_OF_BUTTONS, false);
			buttonsCompositeLayout.marginLeft = isUiLabelHiddenSpecified() ? 0 : TITLEBAR_OFFSET;
			buttonsComposite.setLayout(buttonsCompositeLayout);
			// Create buttons
			createAddButton(buttonsComposite, 1, () -> selectLastItem());
			createRemoveButton(buttonsComposite, 1, () -> getSelection());
			createUpButton(buttonsComposite, 1, () -> {
				ISettingConfig selection = getSelection();
				if (selection != null) {
					ProgressUtils.run(getMoveUpAction(selection));
				}
			});
			createDownButton(buttonsComposite, 1, () -> {
				ISettingConfig selection = getSelection();
				if (selection != null) {
					ProgressUtils.run(getMoveDownAction(selection));
				}
			});
			Composite representationCompositeLoc = new Composite(contentComposite, SWT.NONE);
			representationComposite = representationCompositeLoc;
			representationCompositeLoc.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.CENTER, true, false));
			GridLayoutComponents layout = new GridLayoutComponents(2, false);
			representationCompositeLoc.setLayout(layout);
			Label representationLabelLoc = new Label(representationCompositeLoc, SWT.NONE);
			representationLabelLoc.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
			representationLabelLoc.setText(Messages.get().AArrayControlInternal_RepresentationComboLabel);
			InstantSearchList representationComboLoc = createRepresentationSelectionCombo(representationCompositeLoc, SWT.NONE);
			representationComboLoc.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.CENTER, false, false));
			representationCombo = representationComboLoc;
			updateLabelWithControls(UpdateType.NORMAL);
			return contentComposite;
		}
		return null;
	}

	/**
	 * Returns action to do when item should be moved up in array
	 * @param selected child
	 * @return action runnable
	 */
	protected @NonNull IRunnableWithProgress getMoveUpAction(@NonNull ISettingConfig selected) {
		return m -> controllerWrapper.getController().moveItemFront(arrayConfig, selected, AArrayControlInternal.this);
	}

	/**
	 * Returns action to do when item should be moved down in array
	 * @param selected child
	 * @return action runnable
	 */
	protected @NonNull IRunnableWithProgress getMoveDownAction(@NonNull ISettingConfig selected) {
		return m -> controllerWrapper.getController().moveItemBack(arrayConfig, selected, AArrayControlInternal.this);
	}

	/**
	 * Updates label with controls
	 * @param updateType type of update
	 */
	public void updateLabelWithControls(@NonNull UpdateType updateType) {
		updateButtons(updateType);
		Control labelLoc = label;
		if ((labelLoc != null) && !labelLoc.isDisposed()) {
			updateLabelContent(labelLoc, updateType);
			setLabelInternalFontStyle(SWT.BOLD);
			setControlEnabled(labelLoc, updateType != UpdateType.FORCE_DISABLE, true);
			String labelToolTip = PeripheralsToolTippableFormatter.getToolTipText(new ToolTipableDescriptionAppendDecorator(new ToolTipableMarkdownDescriptionDecorator(
					getChild()), isExpandGroupVisible() ? UtilsText.EMPTY_STRING : UtilsText.XHTML_BR + UtilsText.htmlI(Messages.get().AArrayControlInternal_CollapsedArrayTooltipInfo)));
			updateLabelInternalTooltip(labelToolTip);
		}
	}

	/**
	 * Creates basic button with given testID, icon and runnable that will be executed when clicked on
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 * @param listener runnable that will be executed after clicking on button
	 * @param testId of button
	 * @param icon that should be displayed
	 * @return created button
	 */
	private static @NonNull Button createButton(@NonNull Composite contentComposite, int colSpan, @NonNull SelectionListener listener, final @NonNull String testId,
			final @Nullable Image icon) {
		Button button = new Button(contentComposite, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, testId);
		button.setImage(icon);
		button.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.BOTTOM, false, false, colSpan, 1));
		button.addSelectionListener(listener);
		setStateOfButton(button, false);
		return button;
	}

	/**
	 * Create button used for adding items to the array.
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 * @param afterAdd runnable that will be executed after adding
	 */
	protected void createAddButton(@NonNull Composite contentComposite, int colSpan, @Nullable Runnable afterAdd) {
		SelectionListener listener = new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				ProgressUtils.run((m) -> {
					controllerWrapper.getController().addItem(arrayConfig, AArrayControlInternal.this);
				});
				if (afterAdd != null) {
					afterAdd.run();
				}
			}
		};
		final String testId = TestIDs.PERIPHS_ARRAY_ADD_BUTTON + UtilsText.UNDERSCORE + arrayConfig.getId();
		final Image icon = ToolsImages.getImage(IToolsImages.ICON_ADD);
		Button button = createButton(contentComposite, colSpan, listener, testId, icon);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ArrayControlMasterDetail_AddNewItem);
		buttonAdd = button;
	}

	/**
	 * Create button used for adding items to the array.
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 */
	protected void createAddButton(@NonNull Composite contentComposite, int colSpan) {
		createAddButton(contentComposite, colSpan, null);
	}

	/**
	 * Create button used for removing items to the array.
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 * @param supplier supplier of item that should be removed
	 */
	protected void createRemoveButton(@NonNull Composite contentComposite, int colSpan, @NonNull Supplier<@Nullable ISettingConfig> supplier) {
		SelectionListener listener = new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				ISettingConfig childConfig = supplier.get();
				if (childConfig != null) {
					removeItem(childConfig);
				}
			}

		};
		final String testId = TestIDs.PERIPHS_ARRAY_REMOVE_BUTTON + UtilsText.UNDERSCORE + arrayConfig.getId();
		final Image icon = ToolsImages.getImage(IToolsImages.ICON_REMOVE_CROSS);
		Button button = createButton(contentComposite, colSpan, listener, testId, icon);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ArrayControl_ItemMenu_Remove);
		buttonRemove = button;
	}

	/**
	 * Removes given item from array config
	 * @param childConfig child to be removed
	 */
	void removeItem(@NonNull ISettingConfig childConfig) {
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().removeItem(arrayConfig, childConfig, AArrayControlInternal.this);
		});
	}

	/**
	 * Create button used for moving item up in array
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 * @param runnable that is executed when user clicks on button
	 */
	protected void createUpButton(@NonNull Composite contentComposite, int colSpan, @NonNull Runnable runnable) {
		SelectionListener listener = new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				runnable.run();
			}
		};
		final String testId = TestIDs.PERIPHS_ARRAY_UP_BUTTON + UtilsText.UNDERSCORE + arrayConfig.getId();
		final Image icon = getMoveUpIcon();
		Button button = createButton(contentComposite, colSpan, listener, testId, icon);
		if (!isUiArrayReorderSpecified()) {
			button.setEnabled(false);
		}
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, getMoveUpButtonDescription());
		buttonUp = button;
	}

	/**
	 * @return description for move up button
	 */
	protected @NonNull String getMoveUpButtonDescription() {
		return Messages.get().ArrayControl_ItemMenu_MoveUp;
	}

	/**
	 * @return icon for move up button
	 */
	protected @Nullable Image getMoveUpIcon() {
		return ToolsImages.getImage(IToolsImages.ICON_UP);
	}

	/**
	 * Create button used for moving item down in array
	 * @param contentComposite to create button in
	 * @param colSpan to assign to the grid layout data of the button
	 * @param runnable that is executed when user clicks on button
	 */
	protected void createDownButton(@NonNull Composite contentComposite, int colSpan, @NonNull Runnable runnable) {
		SelectionListener listener = new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				runnable.run();
			}
		};
		final String testId = TestIDs.PERIPHS_ARRAY_DOWN_BUTTON + UtilsText.UNDERSCORE + arrayConfig.getId();
		final Image icon = getMoveDownIcon();
		Button button = createButton(contentComposite, colSpan, listener, testId, icon);
		if (!isUiArrayReorderSpecified()) {
			button.setEnabled(false);
		}
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, getMoveDownDescription());
		buttonDown = button;
	}

	/**
	 * @return description for move down button
	 */
	protected @NonNull String getMoveDownDescription() {
		return Messages.get().ArrayControl_ItemMenu_MoveDown;
	}

	/**
	 * @return icon for move down button
	 */
	protected @Nullable Image getMoveDownIcon() {
		return ToolsImages.getImage(IToolsImages.ICON_DOWN);
	}

	/**
	 * Destroy button used for adding items to the array.
	 */
	protected void destroyAddButton() {
		if (buttonAdd != null) {
			buttonAdd.dispose();
		}
	}

	/**
	 * Checks option value "UI_ARRAY_FIXED"
	 * @return true if option is present (containing value or not), false otherwise
	 */
	protected boolean isUiArrayFixedSpecified() {
		return getControlOptions().isArrayFixed() || (getChild().getModelData().getSizeExpr() != null);
	}

	/**
	 * Checks option value "UI_ARRAY_REORDER"
	 * @return true if option is present (containing value or not), false otherwise
	 */
	protected boolean isUiArrayReorderSpecified() {
		return getControlOptions().isArrayReorder();
	}

	/**
	 * Checks option value "UI_LABEL_HIDDEN"
	 * @return true if option is present (containing value or not), false otherwise
	 */
	protected boolean isUiLabelHiddenSpecified() {
		return getControlOptions().isLabelHidden();
	}

	/**
	 * Update the enable state of the add button and all menu buttons given the update type and the UI_ARRAY_FIXED_OPTION.
	 * @param updateType type of the update or {@code null} if update type is unknown
	 */
	protected abstract void updateButtons(@Nullable UpdateType updateType);

	/**
	 * Update the enable state of the add button given the update type and the UI_ARRAY_FIXED option.
	 * @param updateType type of the update or {@code null} if update type is unknown
	 */
	protected void updateAddButton(@Nullable UpdateType updateType) {
		final boolean allowed = ((updateType != UpdateType.FORCE_DISABLE) && getChild().isEnabled());
		Button buttonLoc = buttonAdd;
		if ((buttonLoc != null) && !buttonLoc.isDisposed()) {
			buttonLoc.setEnabled(allowed && getChild().canAddItem());
			setStateOfButton(buttonLoc, !isUiArrayFixedSpecified());
		}
	}

	/**
	 * Sets state of button
	 * @param button state of which button should be set
	 * @param shouldBeVisible {@code true} if button should be visible or {@code false} if not
	 */
	protected static void setStateOfButton(@NonNull Button button, final boolean shouldBeVisible) {
		GridDataComponents layoutData = (GridDataComponents) button.getLayoutData();
		if (layoutData != null) {
			layoutData.exclude = false;
			button.setVisible(true);
			if (!shouldBeVisible) {
				layoutData.exclude = true;
				button.setVisible(false);
			}
		}
	}

	/**
	 * Update the enable state of the remove button given the update type and the UI_ARRAY_FIXED option.
	 * @param updateType type of the update or {@code null} if update type is unknown
	 */
	protected void updateRemoveButton(@Nullable UpdateType updateType) {
		final boolean allowed = ((updateType != UpdateType.FORCE_DISABLE) && getChild().isEnabled());
		Button buttonLoc = buttonRemove;
		if ((buttonLoc != null) && !buttonLoc.isDisposed()) {
			buttonLoc.setEnabled(allowed && getChild().canRemoveItem());
			setStateOfButton(buttonLoc, !isUiArrayFixedSpecified());
		}
	}

	/**
	 * Update the enable state of the remove button given the update type and the UI_ARRAY_REORDER option.
	 * @param updateType type of the update or {@code null} if update type is unknown
	 */
	protected void updateUpButton(@Nullable UpdateType updateType) {
		final boolean allowed = ((updateType != UpdateType.FORCE_DISABLE) && getChild().isEnabled());
		Button buttonLoc = buttonUp;
		if ((buttonLoc != null) && !buttonLoc.isDisposed()) {
			boolean selectedAny = (selectedChildIdx > -1);
			boolean selectedFirst = (selectedChildIdx == 0);
			buttonLoc.setEnabled(allowed && !selectedFirst && selectedAny);
			setStateOfButton(buttonLoc, isUiArrayReorderSpecified());
		}
	}

	/**
	 * Update the enable state of the remove button given the update type and the UI_ARRAY_REORDER option.
	 * @param updateType type of the update or {@code null} if update type is unknown
	 */
	protected void updateDownButton(@Nullable UpdateType updateType) {
		final boolean allowed = ((updateType != UpdateType.FORCE_DISABLE) && getChild().isEnabled());
		Button buttonLoc = buttonDown;
		if ((buttonLoc != null) && !buttonLoc.isDisposed()) {
			boolean selectedAny = (selectedChildIdx > -1);
			boolean selectedLast = (selectedChildIdx == (children.size() - 1));
			buttonLoc.setEnabled(allowed && !selectedLast && selectedAny);
			setStateOfButton(buttonLoc, isUiArrayReorderSpecified());
		}
	}

	/**
	 * Determines whether the control is first in the list of controls.
	 * @param setting control instance
	 * @return {@code true} if first, {@code false} otherwise 
	 */
	protected boolean isControlFirst(@Nullable IChildControl setting) {
		return children.isEmpty() ? false : children.get(0).equals(setting);
	}

	/**
	 * Determines whether the control is last in the list of controls.
	 * @param setting control instance
	 * @return {@code true} if last, {@code false} otherwise 
	 */
	protected boolean isControlLast(@Nullable IChildControl setting) {
		return children.isEmpty() ? false : children.get(children.size() - 1).equals(setting);
	}

	/**
	 * Move item 1 position toward the beginning of the array.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public void moveItemFront(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		final IChildControl setting = control.getSelectedItemHint();
		assert setting != null;
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().moveItemFront(arrayConfig, (ISettingConfig) setting.getChild(), caller); 
		});
	}

	/**
	 * Move item 1 position toward the end of the array.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public void moveItemBack(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		final IChildControl setting = control.getSelectedItemHint();
		assert setting != null;
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().moveItemBack(arrayConfig, (ISettingConfig) setting.getChild(), caller); 
		});
	}

	/**
	 * Move item to the beginning of the array.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public void moveItemBeginning(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		final IChildControl setting = control.getSelectedItemHint();
		assert setting != null;
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().moveItemBeginning(arrayConfig, (ISettingConfig) setting.getChild(), caller); 
		});
	}

	/**
	 * Move item to the end of the array.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public void moveItemEnd(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		final IChildControl setting = control.getSelectedItemHint();
		assert setting != null;
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().moveItemEnd(arrayConfig, (ISettingConfig) setting.getChild(), caller); 
		});
	}

	/**
	 * Remove item of the array.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public void removeItem(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control) {
		final IChildControl setting = control.getSelectedItemHint();
		assert setting != null;
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().removeItem(arrayConfig, (ISettingConfig) setting.getChild(), caller); 
		});
	}

	/**
	 * Returns config of currently selected item
	 * @param menuControl of the menu
	 * @return config of currently selected item
	 */
	public @Nullable ISettingConfig getCurrentItemConfig(@NonNull IArrayControlItemMenuControl menuControl) {
		final IChildControl control = menuControl.getSelectedItemHint();
		if (control == null) {
			return null;
		}
		IChild setting = control.getChild();
		if (setting instanceof ISettingConfig) {
			return (ISettingConfig) setting;
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#createComposite(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public @NonNull Composite createComposite(@NonNull Composite parentComposite, int numColumns) {	
		GridLayoutComponents layout = new GridLayoutComponents(numColumns, false);
		int style = getSwtStyle();
		if (getControlOptions().isBorderHidden()) {
			layout.marginWidth = 0;
		} else {
			layout.marginWidth = 8;		
		}
		layout.horizontalSpacing = 8;
		Composite composite = new Composite(parentComposite, style);	
		composite.setLayout(layout);
		return composite;	
	}
	
	/**
	 * Set all settings to the specified value.
	 * @param caller of the method (also used for identification purposes in generated events)
	 * @param control of the menu
	 */
	public abstract void setAllSettingsTo(@NonNull ArrayControlItemMenu caller, @NonNull IArrayControlItemMenuControl control);

	/**
	 * Adds listener for keyboard to given table
	 * @param table which should listen for keyboard
	 */
	protected void addKeyboardListener(@NonNull Table table) {
		table.addKeyListener(new KeyAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.KeyAdapter#keyPressed(org.eclipse.swt.events.KeyEvent)
			 */
			@Override
			public void keyPressed(KeyEvent e) {
				handleDeleteKey(table.getShell(), e);
			}
		});
	}

	/**
	 * Handles event when DELETE key is pressed
	 * @param shell in which the confirmation dialog should appear
	 * @param event KeyEvent that happened
	 */
	protected void handleDeleteKey(@Nullable Shell shell, @NonNull KeyEvent event) {
		if (event.keyCode == SWT.DEL) {
			ISettingConfig selection = getSelection();
			if (selection == null) {
				return;
			}
			boolean remove = MessageDialog.openQuestion(shell, Messages.get().AArrayControlInternal_RemoveItemTitle, Messages.get().AArrayControlInternal_RemoveItemConfirmationQuestion);
			if (remove) {
				removeItem(selection);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#getChild()
	 */
	@Override
	public @NonNull ArrayConfig getChild() {
		return (ArrayConfig) child;
	}

	/**
	 * Change value in the model.
	 * @param config to be changed
	 * @param newVal to set
	 */
	protected void changeModelValue(@NonNull ScalarConfig config, @NonNull String newVal) {
		Control mainControlLoc = mainControl;
		assert (mainControlLoc != null);
		ProgressUtils.run((m) -> {
			controllerWrapper.getController().setValue(config, newVal, mainControlLoc);
		});
	}

	/**
	 * Creates combo box to select wanted representation of array
	 * @param parentComposite in which to create the combo
	 * @param style SWT styles to add along SWT.BORDER
	 * @return the combo box control
	 */
	protected InstantSearchList createRepresentationSelectionCombo(@NonNull Composite parentComposite, int style) {
		InstantSearchList combo = new InstantSearchList(parentComposite, SWT.BORDER | style);
		SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_ARRAY_SWITCH_REPRESENTATION + getChild().getId());
		representationCombo = combo;
		updateRepresentationComposite();
		combo.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				Representation currentRepresentation = getChild().getRepresentation();
				Representation selectedRepresentation = Representation.getByLabel(UtilsText.safeString(e.text));
				if (selectedRepresentation != null && selectedRepresentation != currentRepresentation) {
					getChild().setRepresentation(selectedRepresentation);
					recreateGUI();
				}
			}
		});
		return combo;
	}

	/**
	 * Updates the representation composite visibility state and combo box items
	 */
	protected void updateRepresentationComposite() {
		Composite representationCompositeLoc = representationComposite;
		InstantSearchList representationComboLoc = representationCombo;
		if ((representationCompositeLoc != null) && (representationComboLoc != null) && !representationCompositeLoc.isDisposed() && !representationComboLoc.isDisposed()) {
			List<Representation> availableRepresentations = getControlOptions().getAvailableRepresentations();
			if (availableRepresentations != null) {
				if (availableRepresentations.size() > 1) {
					setControlVisible(representationCompositeLoc, true);
					@NonNull String[] labels = new @NonNull String[availableRepresentations.size()];
					for (int i = 0; i < availableRepresentations.size(); i++) {
						labels[i] = availableRepresentations.get(i).getLabel();
					}
					representationComboLoc.setItems(labels);
				} else {
					setControlVisible(representationCompositeLoc, false);
				}
				representationComboLoc.setSelection(getChild().getRepresentation().getLabel());
			} else {
				setControlVisible(representationCompositeLoc, false);
			}
		}
	}

	/**
	 * Recreates GUI of array
	 */
	protected void recreateGUI() {
		Composite parentLoc = parent;
		if (parentLoc != null) {
			@NonNull Control[] parentChildren = parentLoc.getChildren();
			int indexAfterMainControl = -1;
			for (int i = 0; i < parentChildren.length; i++) {
				if (parentChildren[i].equals(mainControl)) {
					indexAfterMainControl = i+1;
				}
			}
			Control controlAfterMainControl = null;
			if (indexAfterMainControl >= 0 && indexAfterMainControl < parentChildren.length) {
				controlAfterMainControl = parentChildren[indexAfterMainControl];
			}
			dispose();
			parentArrayControl.recreateDelegate(controlAfterMainControl);
		}
	}

	/**
	 * Moves controls before given control
	 * @param controlBefore
	 */
	public void moveBefore(Control controlBefore) {
		Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			labelControlLoc.moveAbove(controlBefore);
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			mainControlLoc.moveAbove(controlBefore);
		}
	}

	/**
	 * Sets parent {@link ArrayControl}
	 * @param arrayControl to be set
	 */
	public void setParentArrayControl(ArrayControl arrayControl) {
		this.parentArrayControl = arrayControl;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#createControlOptionsForChild(com.nxp.swtools.resourcetables.model.config.IChild)
	 */
	@Override
	protected @NonNull ControlOptions createControlOptionsForChild(@NonNull IChild subChild) {
		ControlOptions overrideOptions = new ControlOptions();
		Representation representation = getChild().getRepresentation();
		// if indices should be hidden, hide label of child
		if (getControlOptions().isArrayIndicesHidden()) {
			overrideOptions.labelHidden(true);
		}
		if (representation == Representation.TABS) {
			// if child should be shown as Tab and has own children
			if (subChild instanceof IChildProvidable) {
				overrideOptions.borderHidden(true); // hide its border (no reason to have border around whole tab content)
				overrideOptions.labelHidden(true); // hide its label (label is put into tab name)
			}
		}
		if (representation == Representation.HORIZONTAL_TABLE || representation == Representation.VERTICAL_TABLE) {
			overrideOptions = overrideOptions.borderHidden(true);
		}
		if (representation == Representation.HORIZONTAL_RADIOS || representation == Representation.VERTICAL_RADIOS) {
			overrideOptions = overrideOptions.showContentAs(ShowContentAs.RADIO_GROUP);
		}
		if (representation == Representation.MASTER_DETAIL) {
			overrideOptions = overrideOptions.labelHidden(true);
		}
		return overrideOptions;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#onExpandGroupVisibilityChange()
	 */
	@Override
	public void onExpandGroupVisibilityChange() {
		updateLabelWithControls(getChild().isEnabled() ? UpdateType.NORMAL : UpdateType.FORCE_DISABLE);
	}

	/**
	 * Checks whether the buttons should be enabled or not
	 * @return {@code true} when buttons are enabled, {@code false} otherwise
	 */
	protected boolean areButtonsEnabled() {
		return isExpandGroupVisible();
	}

	/**
	 * Checks if the selected control supports set all settings to functionality
	 * @param control to be checked
	 * @return {@code true} when the selected control supports the feature, {@code false} otherwise
	 */
	public boolean isSetAllSettingsToSupported(@NonNull IArrayControlItemMenuControl control) {
		IChild selectedSetting = control.getSelectedSettingHint();
		if (selectedSetting == null) {
			return false;
		}
		String typeName = selectedSetting.getTypeName();
		if (Type.INFO.toString().equalsIgnoreCase(typeName) || Type.VARIABLE.toString().equalsIgnoreCase(typeName)) {
			return false;
		}
		return true;
	}
}
