/**
 * Copyright 2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Class representing control of a combo-box setting with UI_SETTING_LINK option
 * @author Ionut-Cristian Tudose - nxf64017
 */
public class ScalarComboLinkControl extends ScalarComboControl {

	/** Number of columns used when UI_SETTING_LINK option is defined for a setting */
	private static final int REFERENCE_CONTROL_NUM_OF_COLUMNS = 2;

	/** Link button */
	private Button linkButton;

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarComboLinkControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#setEnabledStateToMainControl(boolean)
	 */
	@Override
	protected void setEnabledStateToMainControl(boolean enabled) {
		InstantSearchList comboLoc = combo;
		if ((comboLoc != null) && (comboLoc.getEnabled() != enabled)) {
			comboLoc.setEnabled(enabled);
		}
		Button linkButtonLoc = linkButton;
		if ((linkButtonLoc != null) && (linkButtonLoc.getEnabled() != enabled)) {
			linkButtonLoc.setEnabled(enabled);
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null && !mainControlLoc.getEnabled()) {
			mainControlLoc.setEnabled(true); // Enable composite to allow tooltips to be shown
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#createComposite(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Composite createComposite(@NonNull Composite composite) {
		Composite contentComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(REFERENCE_CONTROL_NUM_OF_COLUMNS, false);
		layout.marginWidth = layout.marginHeight = 0;
		contentComposite.setLayout(layout);
		return contentComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#createControls(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createControls(@NonNull Composite composite) {
		combo = createComboBox(composite);
		linkButton = createLinkButton(composite);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#addSpecificListeners()
	 */
	@Override
	protected void addSpecificListeners() {
		Button linkButtonLoc = linkButton;
		if (linkButtonLoc != null) {
			linkButtonLoc.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					Object optionValue = getChild().getOptionValue(SettingOptions.UI_SETTING_LINK);
					if (optionValue instanceof IChild) {
						handleLinkToSetting((IChild) optionValue, ScalarComboLinkControl.class);
					}
				}
			});
		}
	}

	/**
	 * Method used to create link button
	 * @param composite in which to create the link button
	 * @return link button
	 */
	private Button createLinkButton(@NonNull Composite composite) {
		Button button = new Button(composite, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_DYNAMIC_ENUM_LINK_BUTTON + child.getId());
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ScalarComboLinkControl_LinkButton_Tooltip);
		button.setImage(ToolsImages.getImage(IToolsImages.ICON_SIGNAL_FLOW_TOGGLE));
		button.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.FILL, false, false));
		return button;
	}

}
