/**
 * Copyright 2021-2022 NXP
 * Created: Feb 16, 2021
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.runtime.IPath;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.provider.configuration.ConfigChangeReason;
import com.nxp.swtools.provider.configuration.ISharedConfiguration;
import com.nxp.swtools.provider.configuration.SharedConfigurationAdapter;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.resourcetables.model.config.ChildContext;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Helper class for remembering selected child in providable settings
 * @author Tomas Rudolf - nxf31690
 */
// FIXME TomasR v13 maintenance - Unify storage helpers together. Lot of code is duplicated with different constants etc.
public class ProvidableSettingsSelectionStorageHelper {
	/** Logger of class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ProvidableSettingsSelectionStorageHelper.class);
	/** Name of the states element */
	private static final String SELECTIONS_ELEMENT_NAME = "selections"; //$NON-NLS-1$
	/** Name of the state element */
	private static final String SELECTION_ELEMENT_NAME = "selection"; //$NON-NLS-1$
	/** Name of the ID attribute */
	private static final String ID_ATTRIBUTE_NAME = "id"; //$NON-NLS-1$
	/** Name of the value attribute */
	private static final String VALUE_ATTRIBUTE_NAME = "value"; //$NON-NLS-1$
	/** editors content */
	public static final @NonNull String SETTING_SELECTIONS_KEY = "setting_selections_"; //$NON-NLS-1$
	/** Storage for XML memento */
	@NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Map of expanded states for ids */
	private @NonNull Map<@NonNull String, @Nullable String> collapsibleSettingsState = new HashMap<>();
	/** Location of current MEX */
	private @NonNull String mexFileLocation = UtilsText.EMPTY_STRING;

	/**
	 * Get singleton
	 * @return singleton of this class
	 */
	public static @NonNull ProvidableSettingsSelectionStorageHelper getInstance() {
		try {
			ProvidableSettingsSelectionStorageHelper helper = SWTFactoryProxy.INSTANCE.getSingletonInstance(ProvidableSettingsSelectionStorageHelper.class);
			return helper;
		} catch (InstantiationException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot obtain instance of a providable setting selection helper", e); //$NON-NLS-1$
		}
	}

	/**
	 * Constructor
	 */
	public ProvidableSettingsSelectionStorageHelper() {
		// Reset temporary location before saving to mex
		storageHelper.saveString(SETTING_SELECTIONS_KEY + UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING);
		SharedConfigurationFactory.getSharedConfigurationSingleton().addListener(new SharedConfigurationAdapter() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.provider.configuration.SharedConfigurationAdapter#configurationReloaded(com.nxp.swtools.provider.configuration.ISharedConfiguration, com.nxp.swtools.provider.configuration.ConfigChangeReason)
			 */
			@Override
			public void configurationReloaded(@NonNull ISharedConfiguration sharedConfig, @NonNull ConfigChangeReason reason) {
				switch (reason) {
					case LOAD_CONFIG: {
						mexFileLocation = getMexFilePath();
						loadSelections();
						break;
					}
					case NEW_CONFIG: {
						saveSelections();
						mexFileLocation = getMexFilePath();
						collapsibleSettingsState.clear();
						break;
					}
					default:
						// DO nothing
				}
			}
		});
	}

	/**
	 * Sets information about expanded state of the control to map of states
	 * @param control about which save the expanded state
	 * @param selectedChildId id of selected child
	 * @return {@code true} when state is saved, {@code false} otherwise
	 */
	public boolean setSelectedChild(@NonNull IChildControl control, @Nullable String selectedChildId) {
		String fullSettingId = getFullId(control);
		if (fullSettingId == null) {
			return false;
		}
		collapsibleSettingsState.put(fullSettingId, selectedChildId);
		saveSelections();
		return true;
	}

	/**
	 * Returns full id of given control or null in case of problems
	 * @param control to get full id from
	 * @return full id of the child in control, {@code null} when functional group or component instance cannot be found
	 */
	private static @Nullable String getFullId(@NonNull IChildControl control) {
		IChild child = control.getChild();
		ChildContext childContext = child.getChildContext();
		StringBuilder builder = new StringBuilder(100);
		IFunctionalGroup functionalGroup = childContext.getFunctionalGroup();
		if (functionalGroup == null) {
			return null;
		}
		builder.append(functionalGroup.getName()).append(UtilsText.DOT);
		IComponentInstanceConfig instance = childContext.getComponentInstanceConfig();
		if (instance == null) {
			return null;
		}
		builder.append(instance.getName()).append(UtilsText.DOT);
		builder.append(child.getId());
		String fullSettingId = builder.toString();
		return fullSettingId;
	}

	/**
	 * Gets selected child
	 * @param control that contains selection of child
	 * @return selected child or {@code null} when no selection was stored
	 */
	public @Nullable String getSelectedChild(@NonNull IChildControl control) {
		String fullId = getFullId(control);
		if (fullId == null) {
			return null;
		}
		return collapsibleSettingsState.get(fullId);
	}

	/**
	 * Check if given control has saved state
	 * @param control to be checked
	 * @return {@code true} if expansion state of the control is saved, {@code false} otherwise
	 */
	public boolean isStored(@NonNull IChildControl control) {
		String fullId = getFullId(control);
		if (fullId == null) {
			return false;
		}
		String selectedChildId = collapsibleSettingsState.get(fullId);
		return (selectedChildId != null);
	}

	/**
	 * Save current states to tool preferences
	 */
	private void saveSelections() {
		StringWriter writer = new StringWriter();
		try {
			XMLMemento xmlMemento = XMLMemento.createWriteRoot(SELECTIONS_ELEMENT_NAME); 
			for (Entry<@NonNull String, @Nullable String> entry : collapsibleSettingsState.entrySet()) {
				IMemento child = xmlMemento.createChild(SELECTION_ELEMENT_NAME);
				child.putString(ID_ATTRIBUTE_NAME, entry.getKey());
				String value = entry.getValue();
				if (value != null) {
					child.putString(VALUE_ATTRIBUTE_NAME, value);
				}
			}
			xmlMemento.save(writer);
			storageHelper.saveString(SETTING_SELECTIONS_KEY + mexFileLocation, UtilsText.safeString(writer.toString()));
		} catch (IOException e) {
			LOGGER.log(Level.INFO, "[TOOL] File exception occured. Detail: {0}", e.getMessage()); //$NON-NLS-1$
		}
	}

	/**
	 * Load states of current MEX file from tool preferences
	 */
	private void loadSelections() {
		try {
			if (UtilsText.isEmpty(mexFileLocation)) {
				return;
			}
			String mementoStrContent = storageHelper.loadString(SETTING_SELECTIONS_KEY + mexFileLocation, UtilsText.EMPTY_STRING);
			if (UtilsText.EMPTY_STRING.equals(mementoStrContent)) {
				return;
			}
			StringReader reader = new StringReader(mementoStrContent);
			// Prepare memento from storage
			XMLMemento xmlMemento = XMLMemento.createReadRoot(reader);
			// Read all children
			IMemento[] children = xmlMemento.getChildren();
			collapsibleSettingsState.clear();
			for (IMemento child : children) {
				String id = UtilsText.safeString(child.getString(ID_ATTRIBUTE_NAME));
				String value = child.getString(VALUE_ATTRIBUTE_NAME);
				collapsibleSettingsState.put(id, value);
			}
		} catch (WorkbenchException e) {
			LOGGER.log(Level.SEVERE, "[TOOL] XMLMemento exception occured. Detail: {0}", e.getMessage()); //$NON-NLS-1$
		}
	}

	/**
	 * Looks up for MEX location
	 * @return path to MEX file
	 */
	private static String getMexFilePath() {
		@Nullable IPath locationPath = SharedConfigurationFactory.getSharedConfigurationSingleton().getLocationPath();
		String projectLocation;
		if (locationPath == null) {
			projectLocation = UtilsText.EMPTY_STRING;
		} else {
			projectLocation = locationPath.toOSString();
		}
		return projectLocation;
	}

	/**
	 * Performs actions that are needed during shutdown
	 */
	public void onShutdown() {
		saveSelections();
	}

	/**
	 * Removes all entries from tool preferences file which are containing the given pattern
	 * @param infix - based on which the entries will be removed
	 */
	public void removeSettingStateEntries(String infix) {
		collapsibleSettingsState.keySet().removeIf(fullId -> fullId.contains(infix));
		saveSelections();
	}
}
