/**
 * Copyright 2018-2022 NXP
 * Created: Jun 13, 2018
 */

package com.nxp.swtools.periphs.gui.view.componentsettings;

import static com.nxp.swtools.resourcetables.model.data.SettingOptions.*;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig.Representation;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.SetConfig;
import com.nxp.swtools.resourcetables.model.data.Mode;
import com.nxp.swtools.resourcetables.model.data.setting.InfoSetting;
import com.nxp.swtools.utils.profiler.Profiler;

/**
 * UI options for specifying how controls should be displayed.
 * Null in fields is used as "not defined" - to avoid overwriting existing settings with default state when merging options
 * @author David Danaj (b57899/nxa30572)
 */
public final class ControlOptions {
	/** Integer with value 1 */
	private static final @NonNull Integer INTEGER_ONE = Integer.valueOf(1);
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ControlOptions.class);

	/**
	 * Enumeration of possible types how setting can be displayed
	 * @author David Danaj (b57899/nxa30572)
	 */
	public enum ShowContentAs {
		/** Show as FORM */
		@NonNull FORM(VALUE_SHOW_CONTENT_AS_FORM),
		/** Show as TABLE */
		@NonNull TABLE(VALUE_SHOW_CONTENT_AS_TABLE),
		/** Show as TABS */
		@NonNull TABS(VALUE_SHOW_CONTENT_AS_TABS),
		/** Show as RADIO_GROUP */
		@NonNull RADIO_GROUP(VALUE_SHOW_CONTENT_AS_RADIO_GROUP),
		/** Show as MASTER_DETAIL */
		@NonNull MASTER_DETAIL(VALUE_SHOW_CONTENT_AS_MASTER_DETAIL);

		/** String representation of ShowContentAs */
		private @NonNull String xmlName;

		/**
		 * Constructor.
		 * @param xmlName String representation of ShowContentAs
		 */
		private ShowContentAs(@NonNull String xmlName) {
			this.xmlName = xmlName;
		}

		/*
		 * (non-Javadoc)
		 * @see java.lang.Enum#toString()
		 */
		@Override
		public String toString() {
			return xmlName;
		}

		/**
		 * Get ShowContentAs value based on SettingOption string value.
		 * @param settingOption string to parse
		 * @return parsed ShowContentAs or {@code null} if unit was not recognized
		 */
		public static @Nullable ShowContentAs parse(@Nullable String settingOption) {
			if (settingOption != null) {
				for (ShowContentAs showContentAs : values()) {
					if (showContentAs.toString().equals(settingOption)) {
						return showContentAs;
					}
				}
			}
			return null;
		}
	}

	/**
	 * Enumeration of possible types how setting can be printed
	 * @author David Danaj (b57899/nxa30572)
	 */
	public enum PrintAs {
		/** Print as PLAIN */
		PLAIN(VALUE_PRINT_THIS_AS_PLAIN),
		/** Print as LINK */
		LINK(VALUE_PRINT_THIS_AS_LINK);

		/** String representation of PrintAs */
		@NonNull
		String xmlName;

		/**
		 * Constructor.
		 * @param xmlName String representation of PrintAs
		 */
		private PrintAs(@NonNull String xmlName) {
			this.xmlName = xmlName;
		}

		/*
		 * (non-Javadoc)
		 * @see java.lang.Enum#toString()
		 */
		@Override
		public String toString() {
			return xmlName;
		}

		/**
		 * Get PrintAs value based on SettingOption string value.
		 * @param settingOption string to parse
		 * @return parsed PrintAs or {@code null} if unit was not recognized
		 */
		public static @Nullable PrintAs parse(@Nullable String settingOption) {
			if (settingOption != null) {
				for (PrintAs printAs : values()) {
					if (printAs.toString().equals(settingOption)) {
						return printAs;
					}
				}
			}
			return null;
		}
	}

	// XXX null in fields is used as "not defined" - to avoid overwriting existing settings with default state when merging options
	/** Child for which this options are used */
	private @Nullable IChild child;
	/** ShowContentAs option */
	private @Nullable ShowContentAs showContentAs;
	/** PrintAs option */
	private @Nullable PrintAs printAs;
	/** how expand group should be shown */
	private @Nullable Boolean collapseGroup;
	/** should expand group be hidden */
	private @Nullable Boolean expandGroupHidden;
	/** should array be fixed */
	private @Nullable Boolean arrayFixed;
	/** should array allow reorder of items */
	private @Nullable Boolean arrayReorder;
	/** should array indices be hidden */
	private @Nullable Boolean arrayIndicesHidden;
	/** should array be displayed horizontally */
	private @Nullable Boolean arrayHorizontal;
	/** should label be hidden */
	private @Nullable Boolean labelHidden;
	/** should border be hidden */
	private @Nullable Boolean borderHidden;
	/** should provide a button to copy {@link InfoSetting} value */
	private @Nullable Boolean infoCopyButton;
	/** should {@link Mode} be hidden */
	private @Nullable Boolean componentModeHidden;
	/** widths of the table columns */
	private @Nullable List<@NonNull Double> tableColumnWidths;
	/** should show label in bold font */
	private @Nullable Boolean labelBold;
	/** should show break line */
	private @Nullable Boolean lineBreak;
	/** how many lines should be shown */
	private @Nullable Integer numOfLines;
	/** how many columns should be shown */
	private @Nullable Integer numOfColumns;
	/** should radio group show label of each possible value */
	private @Nullable Boolean radioGroupValueLabelHidden;
	/** should radio group show indices */
	private @Nullable Boolean radioGroupIndicesHidden;
	/** should radio group be layouted as horizontal */
	private @Nullable Boolean radioGroupHorizontal;
	/** should selected file be persisted when user selects it */
	private @Nullable Boolean persistOnSelection;
	/** hide sub-settings */
	private @Nullable Boolean controlHidden;
	/** Path to documentation page which should be opened */
	private @Nullable String openDocumentationPage;
	/** List of representations that are available */
	private List<@NonNull Representation> availableRepresentations;
	/** Number of visible rows of table */
	private @Nullable Integer tableVisibleRows = null;

	/**
	 * Empty constructor.
	 */
	public ControlOptions() {
		// empty on purpose
	}

	/**
	 * Create new instance by retrieving options from model. If options are not defined, set fields to default state.
	 * @param child setting for which control options are created
	 */
	public ControlOptions(@NonNull final IChild child) {
		this.child = child;
		refresh();
	}

	/**
	 * Refresh each option based on current state of setting option
	 */
	public void refresh() {
		IChild childLoc = child;
		if (childLoc != null) {
			Profiler profiler = Profiler.getInstance(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
			BigInteger index = profiler.start(ControlOptions.class, !profiler.isEnabled() ? UtilsText.EMPTY_STRING :
					MessageFormat.format("Control options refresh of child with id \"{0}\"", childLoc.getId())); //$NON-NLS-1$
			Object optionValue = childLoc.getOptionValue(SHOW_CONTENT_AS);
			if ((optionValue != null) && (optionValue instanceof String)) {
				showContentAs = ShowContentAs.parse((String) optionValue);
			} else {
				showContentAs = ShowContentAs.FORM; // set default
			}
			optionValue = childLoc.getOptionValue(PRINT_THIS_AS);
			if ((optionValue != null) && (optionValue instanceof String)) {
				printAs = PrintAs.parse((String) optionValue);
			} else {
				printAs = PrintAs.PLAIN; // set default
			}
			optionValue = childLoc.getOptionValue(UI_MULTI_LINE);
			if (optionValue instanceof String) {
				try {
					numOfLines = Integer.valueOf(UtilsText.safeToString(optionValue));
				} catch (@SuppressWarnings("unused") NumberFormatException e) {
					numOfLines = INTEGER_ONE;
					LOGGER.log(Level.SEVERE, "[DATA] Value \"{0}\" cannot be converted to number", optionValue); //$NON-NLS-1$
				}
			}
			optionValue = childLoc.getOptionValue(UI_MULTI_COLUMN);
			Integer defaultValue = INTEGER_ONE;
			if (childLoc instanceof SetConfig) {
				defaultValue = Integer.valueOf(2);
			}
			if (optionValue instanceof String) {
				try {
					Integer numOfColumnsLoc = Integer.valueOf(UtilsText.safeToString(optionValue));
					if (numOfColumnsLoc.compareTo(INTEGER_ONE) < 0) {
						numOfColumnsLoc = defaultValue;
					}
					numOfColumns = numOfColumnsLoc;
				} catch (@SuppressWarnings("unused") NumberFormatException e) {
					numOfColumns = defaultValue;
					LOGGER.log(Level.SEVERE, "[DATA] Value \"{0}\" cannot be converted to number", optionValue); //$NON-NLS-1$
				}
			} else {
				numOfColumns = defaultValue;
			}
			// if option is not set - it is set to default state (false)
			arrayFixed = Boolean.valueOf(childLoc.isOptionSet(UI_ARRAY_FIXED) || ((childLoc instanceof ArrayConfig) && (((ArrayConfig)childLoc).getModelData().getSizeExpr() != null)));
			arrayReorder = Boolean.valueOf(childLoc.isOptionSet(UI_ARRAY_REORDER));
			arrayIndicesHidden = Boolean.valueOf(childLoc.isOptionSet(UI_ARRAY_INDICES_HIDDEN));
			arrayHorizontal = Boolean.valueOf(childLoc.isOptionSet(UI_ARRAY_LAYOUT_HORIZONTAL));
			labelHidden = Boolean.valueOf(childLoc.isOptionSet(UI_LABEL_HIDDEN));
			borderHidden = Boolean.valueOf(childLoc.isOptionSet(UI_BORDER_HIDDEN));
			infoCopyButton = Boolean.valueOf(childLoc.isOptionSet(UI_INFO_COPY_BUTTON));
			componentModeHidden = Boolean.valueOf(childLoc.isOptionSet(UI_COMPONENT_MODE_HIDDEN));
			labelBold = Boolean.valueOf(childLoc.isOptionSet(UI_LABEL_BOLD));
			lineBreak = Boolean.valueOf(childLoc.isOptionSet(UI_LINE_BREAK));
			radioGroupValueLabelHidden = Boolean.FALSE;
			radioGroupIndicesHidden = Boolean.TRUE;
			radioGroupHorizontal = Boolean.valueOf(childLoc.isOptionSet(UI_RADIO_GROUP_HORIZONTAL));
			persistOnSelection = Boolean.valueOf(childLoc.isOptionSet(PERSIST_ON_SELECTION));
			optionValue = childLoc.getOptionValue(UI_TABLE_COLUMN_WIDTHS);
			if (optionValue instanceof String) {
				tableColumnWidths = parseDoubleArray((String) optionValue);
			}
			optionValue = childLoc.getOptionValue(UI_OPEN_DOCUMENTATION);
			if (optionValue instanceof String) {
				openDocumentationPage = (String) optionValue;
			} else {
				openDocumentationPage = null;
			}
			Object arrayTableRowsLimit = childLoc.getOptionValue(UI_ARRAY_TABLE_ROWS_VISIBLE);
			if (arrayTableRowsLimit instanceof String) {
				try {
					tableVisibleRows = Integer.valueOf((String) arrayTableRowsLimit);
				} catch (@SuppressWarnings("unused") NumberFormatException e) {
					LOGGER.log(Level.SEVERE, () -> MessageFormat.format("TOOL] Value {0} given for option {1} is not integer in setting {2}", //$NON-NLS-1$
							arrayTableRowsLimit, UI_ARRAY_TABLE_ROWS_VISIBLE, childLoc.getId()));
				}
			}
			optionValue = childLoc.getOptionValue(UI_ARRAY_REPRESENTATIONS);
			if (optionValue instanceof String) {
				String[] array = ((String) optionValue).split(UtilsText.COMMA);
				List<@NonNull Representation> list = new ArrayList<>(array.length);
				for (String id : array) {
					Representation representation = Representation.getById(id);
					if (representation != null) {
						list.add(representation);
					}
				}
				availableRepresentations = list;
			} else if (optionValue instanceof List<?>) {
				List<Object> list = (List<Object>) optionValue;
				List<@NonNull Representation> representations = new ArrayList<>(list.size());
				for (Object obj : list) {
					if (obj instanceof String) {
						Representation representation = Representation.getById((String) obj);
						if (representation != null) {
							representations.add(representation);
						}
					}
				}
				availableRepresentations = representations;
			}
			expandGroupHidden = Boolean.valueOf(childLoc.isOptionSet(UI_EXPAND_GROUP_HIDDEN));
			collapseGroup = Boolean.valueOf(childLoc.isOptionSet(UI_COLLAPSE_GROUP));
			profiler.stop(index, ChildControlBase.class, null);
			controlHidden = Boolean.valueOf(childLoc.isOptionSet(UI_CONTROL_HIDDEN));
		}
	}

	/**
	 * @return the arrayFixed
	 */
	public boolean isArrayFixed() {
		return Boolean.TRUE.equals(arrayFixed);
	}

	/**
	 * @return the arrayReorder
	 */
	public boolean isArrayReorder() {
		return Boolean.TRUE.equals(arrayReorder);
	}

	/**
	 * @return the arrayIndicesHidden
	 */
	public boolean isArrayIndicesHidden() {
		return Boolean.TRUE.equals(arrayIndicesHidden);
	}

	/**
	 * @return the arrayHorizontal
	 */
	public boolean isArrayHorizontal() {
		return Boolean.TRUE.equals(arrayHorizontal);
	}

	/**
	 * @return {@code true} when the limit on visible table rows was set, {@code false} otherwise
	 */
	public boolean isTableRowsVisibleLimitSet() {
		return tableVisibleRows != null;
	}

	/**
	 * @return the labelHidden
	 */
	public boolean isLabelHidden() {
		return Boolean.TRUE.equals(labelHidden);
	}

	/**
	 * @return the labelBold
	 */
	public boolean isLabelBold() {
		return Boolean.TRUE.equals(labelBold);
	}

	/**
	 * @return the lineBreak
	 */
	public boolean isLineBreak() {
		return Boolean.TRUE.equals(lineBreak);
	}

	/**
	 * @return the multi column flag
	 */
	public boolean isMultiColumn() {
		Integer numOfColumnsLoc = numOfColumns;
		if (numOfColumnsLoc != null) {
			return INTEGER_ONE.compareTo(numOfColumnsLoc) < 0;
		}
		return false;
	}

	/**
	 * @return the borderHidden
	 */
	public boolean isBorderHidden() {
		return Boolean.TRUE.equals(borderHidden);
	}

	/**
	 * @return whether {@link #infoCopyButton} equals {@code true}
	 */
	public boolean isInfoCopyButtonRequested() {
		return Boolean.TRUE.equals(infoCopyButton);
	}

	/**
	 * @return whether {@link #componentModeHidden} equals {@code true}
	 */
	public boolean isComponentModeHidden() {
		return Boolean.TRUE.equals(componentModeHidden);
	}

	/**
	 * @return whether {@link #radioGroupValueLabelHidden} equals {@code true}
	 */
	public boolean isRadioGroupValueLabelHidden() {
		return Boolean.TRUE.equals(radioGroupValueLabelHidden);
	}

	/**
	 * @return whether {@link #radioGroupIndicesHidden} equals {@code true}
	 */
	public boolean isRadioGroupIndicesHidden() {
		return Boolean.TRUE.equals(radioGroupIndicesHidden);
	}

	/**
	 * @return whether {@link #radioGroupHorizontal} equals {@code true}
	 */
	public boolean isRadioGroupHorizontal() {
		return Boolean.TRUE.equals(radioGroupHorizontal);
	}

	/**
	 * @return whether {@link #persistOnSelection} equals {@code true}
	 */
	public boolean isSelectionPersistent() {
		return Boolean.TRUE.equals(persistOnSelection);
	}

	/**
	 * @return whether {@link #showContentAs} is not specified
	 */
	public boolean shouldLayoutAsForm() {
		return ShowContentAs.FORM.equals(showContentAs);
	}

	/**
	 * @return whether {@link #showContentAs} equals {@code TABLE}
	 */
	public boolean shouldLayoutAsTable() {
		return ShowContentAs.TABLE.equals(showContentAs);
	}

	/**
	 * @return whether {@link #showContentAs} equals {@code TABS}
	 */
	public boolean shouldLayoutAsTabs() {
		return ShowContentAs.TABS.equals(showContentAs);
	}

	/**
	 * @return whether {@link #showContentAs} equals {@code MASTER_DETAIL}
	 */
	public boolean shouldLayoutAsMasterDetail() {
		return ShowContentAs.MASTER_DETAIL.equals(showContentAs);
	}

	/**
	 * @return whether {@link #showContentAs} equals {@code RADIO_GROUP}
	 */
	public boolean shouldLayoutAsRadioGroup() {
		return ShowContentAs.RADIO_GROUP.equals(showContentAs);
	}

	/**
	 * @return true if the expand group should be hidden
	 */
	public boolean isExpandGroupHidden() {
		return Boolean.TRUE.equals(expandGroupHidden);
	}

	/**
	 * @return true if the group should be collapse, false otherwise
	 */
	public boolean isGroupCollapsed() {
		return Boolean.TRUE.equals(collapseGroup);
	}

	/**
	 * @return true if the control is hidden, false otherwise
	 */
	public boolean isControlHidden() {
		return Boolean.TRUE.equals(controlHidden);
	}

	/**
	 * Sets the state of controlHidden option
	 * @param value which controlHidden will take
	 */
	public void controlHidden(boolean value) {
		this.controlHidden = Boolean.valueOf(value);
	}

	/**
	 * @return true if documentation page to be opened is specified, false otherwise
	 */
	public boolean isDocumentationPageToOpenSpecified() {
		return openDocumentationPage != null;
	}

	/**
	 * @return the arrayFixed
	 */
	public @Nullable Boolean getArrayFixed() {
		return arrayFixed;
	}

	/**
	 * @return the arrayReorder
	 */
	public @Nullable Boolean getArrayReorder() {
		return arrayReorder;
	}

	/**
	 * @return the arrayIndicesHidden
	 */
	public @Nullable Boolean getArrayIndicesHidden() {
		return arrayIndicesHidden;
	}

	/**
	 * @return the arrayHorizontal
	 */
	public @Nullable Boolean getArrayHorizontal() {
		return arrayHorizontal;
	}

	/**
	 * @return the labelHidden
	 */
	public @Nullable Boolean getLabelHidden() {
		return labelHidden;
	}

	/**
	 * @return the borderHidden
	 */
	public @Nullable Boolean getBorderHidden() {
		return borderHidden;
	}

	/**
	 * @return {@link #componentModeHidden}
	 */
	public @Nullable Boolean getComponentModeHidden() {
		return componentModeHidden;
	}

	/**
	 * @return the showContentAs
	 */
	public @Nullable ShowContentAs getShowContentAs() {
		return showContentAs;
	}

	/**
	 * @return the printAs
	 */
	public @Nullable PrintAs getPrintAs() {
		return printAs;
	}

	/**
	 * @return {@link #infoCopyButton}
	 */
	public @Nullable Boolean getInfoCopyButton() {
		return infoCopyButton;
	}

	/**
	 * @return {@link #tableColumnWidths}
	 */
	public @Nullable List<@NonNull Double> getTableColumnWidths() {
		return tableColumnWidths;
	}

	/**
	 * @return {@link #labelBold}
	 */
	public @Nullable Boolean getLabelBold() {
		return labelBold;
	}

	/**
	 * @return {@link #lineBreak}
	 */
	public @Nullable Boolean getLineBreak() {
		return lineBreak;
	}

	/**
	 * @return {@link #numOfLines}
	 */
	public @Nullable Integer getNumOfLines() {
		return numOfLines;
	}

	/**
	 * @return {@link #numOfColumns}
	 */
	public @Nullable Integer getNumOfColumns() {
		return numOfColumns;
	}

	/**
	 * @return {@link #openDocumentationPage}
	 */
	public @Nullable String getDocumentationPageToOpen() {
		return openDocumentationPage;
	}

	/**
	 * @return {@link #availableRepresentations}
	 */
	public @Nullable List<Representation> getAvailableRepresentations() {
		return availableRepresentations;
	}

	/**
	 * @return number of rows visible in the table. 0 when option was not set
	 */
	public int getTableVisibleRows() {
		return (tableVisibleRows != null) ? tableVisibleRows.intValue() : 0;
	}

	/**
	 * @param setShowContentAs set how content should be displayed
	 * @return this
	 */
	public @NonNull ControlOptions showContentAs(@NonNull final ShowContentAs setShowContentAs) {
		this.showContentAs = setShowContentAs;
		return this;
	}

	/**
	 * @param bold set if label should be displayed as bold
	 * @return this
	 */
	public @NonNull ControlOptions labelBold(final boolean bold) {
		this.labelBold = Boolean.valueOf(bold);
		return this;
	}

	/**
	 * @param breakLine set if line break should be added
	 * @return this
	 */
	public @NonNull ControlOptions lineBreak(final boolean breakLine) {
		this.lineBreak = Boolean.valueOf(breakLine);
		return this;
	}

	/**
	 * @param multi set number of lines to be shown
	 * @return this
	 */
	public @NonNull ControlOptions numOfLines(final int multi) {
		this.numOfLines = Integer.valueOf(multi);
		return this;
	}

	/**
	 * @param multi set number of columns to be shown
	 * @return this
	 */
	public @NonNull ControlOptions numOfColumns(final int multi) {
		this.numOfColumns = Integer.valueOf(multi);
		return this;
	}

	/**
	 * @param setPrintAs set how content should be printed
	 * @return this
	 */
	public @NonNull ControlOptions printAs(@NonNull final PrintAs setPrintAs) {
		this.printAs = setPrintAs;
		return this;
	}

	/**
	 * @param isArrayFixed set arrayFixed
	 * @return this
	 */
	public @NonNull ControlOptions arrayFixed(final boolean isArrayFixed) {
		this.arrayFixed = Boolean.valueOf(isArrayFixed);
		return this;
	}

	/**
	 * @param isArrayReorder set arrayReorder
	 * @return this
	 */
	public @NonNull ControlOptions arrayReorder(final boolean isArrayReorder) {
		this.arrayReorder = Boolean.valueOf(isArrayReorder);
		return this;
	}

	/**
	 * @param isArrayIndicesHidden set arrayIndicesHidden
	 * @return this
	 */
	public @NonNull ControlOptions arrayIndicesHidden(final boolean isArrayIndicesHidden) {
		this.arrayIndicesHidden = Boolean.valueOf(isArrayIndicesHidden);
		return this;
	}

	/**
	 * @param isArrayHorizontal set arrayHorizontal
	 * @return this
	 */
	public @NonNull ControlOptions arrayHorizontal(final boolean isArrayHorizontal) {
		this.arrayHorizontal = Boolean.valueOf(isArrayHorizontal);
		return this;
	}

	/**
	 * @param isLabelHidden set labelHidden
	 * @return this
	 */
	public @NonNull ControlOptions labelHidden(final boolean isLabelHidden) {
		this.labelHidden = Boolean.valueOf(isLabelHidden);
		return this;
	}

	/**
	 * @param isRadioGroupIndicesHidden set radioGroupIndicesHidden
	 * @return this
	 */
	public @NonNull ControlOptions radioGroupIndicesHidden(final boolean isRadioGroupIndicesHidden) {
		this.radioGroupIndicesHidden = Boolean.valueOf(isRadioGroupIndicesHidden);
		return this;
	}

	/**
	 * @param isRadioGroupValueLabelHidden set radioGroupValueLabelHidden
	 * @return this
	 */
	public @NonNull ControlOptions radioGroupValueLabelHidden(final boolean isRadioGroupValueLabelHidden) {
		this.radioGroupValueLabelHidden = Boolean.valueOf(isRadioGroupValueLabelHidden);
		return this;
	}

	/**
	 * @param isRadioGroupHorizontal set radioGroupHorizontal
	 * @return this
	 */
	public @NonNull ControlOptions radioGroupHorizontal(final boolean isRadioGroupHorizontal) {
		this.radioGroupHorizontal = Boolean.valueOf(isRadioGroupHorizontal);
		return this;
	}

	/**
	 * @param selection should be persistent
	 * @return this
	 */
	public @NonNull ControlOptions persistOnSelection(final boolean selection) {
		this.persistOnSelection = Boolean.valueOf(selection);
		return this;
	}

	/**
	 * @param isBorderHidden set borderHidden
	 * @return this
	 */
	public @NonNull ControlOptions borderHidden(final boolean isBorderHidden) {
		this.borderHidden = Boolean.valueOf(isBorderHidden);
		return this;
	}

	/**
	 * @param setTableColumnWidths set tableColumnWidths
	 * @return this
	 */
	public @NonNull ControlOptions tableColumnWidths(final @Nullable List<@NonNull Double> setTableColumnWidths) {
		this.tableColumnWidths = setTableColumnWidths;
		return this;
	}

	/**
	 * @param page set which page should be opened
	 * @return this
	 */
	public @NonNull ControlOptions openDocumentationPage(final @Nullable String page) {
		this.openDocumentationPage = page;
		return this;
	}

	/**
	 * @param list to be set as available representations
	 * @return this
	 */
	public @NonNull ControlOptions availableRepresentations(final @Nullable List<@NonNull Representation> list) {
		this.availableRepresentations = list;
		return this;
	}

	/**
	 * @param rows number of visible rows
	 * @return this
	 */
	public @NonNull ControlOptions tableVisibleRows(final int rows) {
		this.tableVisibleRows = Integer.valueOf(rows);
		return this;
	}

	/**
	 * Merge options into existing options. Defined options (not null) in argument overrides options in this object.
	 * @param overrideOptions ControlOptions to be merged into existing options
	 * @return this
	 */
	public @NonNull ControlOptions merge(@NonNull ControlOptions overrideOptions) {
		Field[] fields = this.getClass().getDeclaredFields();
		for (Field field : fields) {
			if ((field.getModifiers() & (Modifier.FINAL | Modifier.STATIC)) == 0) { // update only non-static, non-final fields
				// get value
				Object value;
				try {
					value = field.get(overrideOptions);
					// if value is defined, set it into this options
					if (value != null) {
						field.set(this, value);
					}
				} catch (IllegalArgumentException | IllegalAccessException e) {
					LOGGER.warning("[TOOL] Error in merging control options: " + e.getMessage()); //$NON-NLS-1$
				}
			}
		}
		return this;
	}

	/**
	 * Extracts float array content from string
	 * @param values comma-separated signed floats
	 * @return array of float values
	 */
	private static @NonNull List<@NonNull Double> parseDoubleArray(@NonNull String values) {
		final List<@NonNull Double> result = new ArrayList<>();
		final String[] doubles = values.split(UtilsText.COMMA);
		for (String doubleNumberString : doubles) {
			try {
				result.add(Double.valueOf(doubleNumberString.trim()));
			} catch (NumberFormatException e) {
				result.add(Double.valueOf(-1));
				LOGGER.log(Level.SEVERE, String.format("[DATA] Invalid array of doubles '%1s': %2s. Using value -1 instead", values, e.getMessage())); //$NON-NLS-1$
				break;
			}
		}
		return result;
	}
}
