/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.MenuDetectEvent;
import org.eclipse.swt.events.MenuDetectListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;

import com.nxp.swtools.common.ui.utils.swt.HtmlTooltip;
import com.nxp.swtools.common.ui.utils.swt.ITooltipProvider2;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions.ShowContentAs;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.data.XMLConstants;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Base class representing control of a structured setting configuration.
 * @author Juraj Ondruska
 */
public class ChildProvidableControlBase extends ChildControlBase {
	/** The children */
	protected final @NonNull List<@NonNull IChildControl> children = new ArrayList<>();
	/** Key for setting data (IChildControl) into CTabItems */
	public static final @NonNull String KEY_TAB_CHILD_CONTROL = ChildProvidableControlBase.class.getCanonicalName()
			+ ".key_tab_child_control"; //$NON-NLS-1$
	/** The currently selected child */
	protected @Nullable IChildControl selectedChild;
	/** Index of the currently selected child */
	protected int selectedChildIdx = -1;
	/** Tab folder with the children */
	protected @Nullable CTabFolder tabFolder;
	/** Selection listener for tab folder */
	private @Nullable SelectionListener tabSelectionListener;

	/**
	 * Constructor that will create children only when required
	 * @param childProvidable to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 * @param createChildren {@code true} when children should be created, {@code false} when not
	 */
	public ChildProvidableControlBase(@NonNull IChildProvidable childProvidable, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper, boolean createChildren) {
		super(childProvidable, controlOptions, controllerWrapper);
		if (createChildren) {
			for (IChild subChild : childProvidable.getChildren()) {
				ControlOptions overrideOptions = createControlOptionsForChild(subChild);
				IChildControl subChildControl = ChildControlFactory.create(subChild, overrideOptions, controllerWrapper);
				if (subChildControl != null) {
					children.add(subChildControl);
				}
			}
		}
	}

	/**
	 * Constructor that always create children
	 * @param childProvidable to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ChildProvidableControlBase(@NonNull IChildProvidable childProvidable, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		this(childProvidable, controlOptions, controllerWrapper, true);
	}

	/**
	 * Create Control Options for child based on options of its parent (calling object)
	 * @param subChild child for which control options are created
	 * @return ControlOptions for a child
	 */
	protected @NonNull ControlOptions createControlOptionsForChild(@NonNull IChild subChild) {
		ControlOptions overrideOptions = new ControlOptions();
		// if indices should be hidden, hide label of child
		if (getControlOptions().isArrayIndicesHidden()) {
			overrideOptions.labelHidden(true);
		}
		if (ShowContentAs.TABS.equals(getControlOptions().getShowContentAs())) {
			// if child should be shown as Tab and has own children
			if (subChild instanceof IChildProvidable) {
				overrideOptions.borderHidden(true); // hide its border (no reason to have border around whole tab content)
				overrideOptions.labelHidden(true); // hide its label (label is put into tab name)
			}
		} else if (ShowContentAs.TABLE.equals(getControlOptions().getShowContentAs())) {
			// hide border in table
			overrideOptions = overrideOptions.borderHidden(true);
		} else if (ShowContentAs.RADIO_GROUP.equals(getControlOptions().getShowContentAs())) {
			overrideOptions.showContentAs(ShowContentAs.RADIO_GROUP);
		} else if (ShowContentAs.MASTER_DETAIL.equals(getControlOptions().getShowContentAs())) {
			// hide indices/labels in detail
			overrideOptions = overrideOptions.labelHidden(true);
		}
		return overrideOptions;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#update(com.nxp.swtools.periphs.gui.editor.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		UpdateType updateChildrenType = child.isEnabled() ? updateType : UpdateType.FORCE_DISABLE;
		updateStoredUpdateType(updateChildrenType);
		if ((tabFolder != null) && (!tabFolder.isDisposed()) && (updateType != UpdateType.PROBLEM_DECORATION)) {
			updateTabs();
		}
		super.update(updateType);
		for (IChildControl childControl : children) {
			childControl.update(updateChildrenType);
		}
		// remove html tooltip from mainControl (tooltip was visible on empty places in composite) - MCUXCON-1778
		Control mainControlLoc = mainControl;
		if ((mainControlLoc != null) && (!mainControlLoc.isDisposed())) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(mainControlLoc, null);
		}
	}

	/**
	 * Update CTabFolder to match information from childrenControls. Creates CTabItems on correct indexes if they were not displayed and should be displayed. If
	 * CTabItems were displayed and should be hidden now, they are disposed.
	 */
	public void updateTabs() {
		final CTabFolder tabFolderLoc = tabFolder;
		SelectionListener tabSelectionListenerLoc = tabSelectionListener;
		if ((tabFolderLoc != null) && (tabSelectionListenerLoc != null)) {
			tabFolderLoc.removeSelectionListener(tabSelectionListenerLoc);
		}
		Iterator<? extends @NonNull IChild> childrenOfConfigIterator = getChild().getChildren().iterator();
		Iterator<@NonNull IChildControl> childrenIterator = children.iterator();
		IChild childOfConfig = childrenOfConfigIterator.hasNext() ? childrenOfConfigIterator.next() : null;
		IChildControl childControl = childrenIterator.hasNext() ? childrenIterator.next() : null;
		while (childControl != null) {
			// skip the settings that are variables, because no UI controls are created for variables
			while ((childOfConfig != null) && XMLConstants.VARIABLE.equals(childOfConfig.getTypeName())) {
				childOfConfig = childrenOfConfigIterator.hasNext() ? childrenOfConfigIterator.next() : null;
			}
			if (childControl.getChild().equals(childOfConfig)) {
				childOfConfig = childrenOfConfigIterator.hasNext() ? childrenOfConfigIterator.next() : null;
			} else {
				removeTabWithControl(childControl);
				childControl.dispose();
				childrenIterator.remove();
			}
			childControl = childrenIterator.hasNext() ? childrenIterator.next() : null;
		}
		while (childOfConfig != null) {
			ControlOptions overrideOptions = createControlOptionsForChild(childOfConfig);
			IChildControl subChildControl = ChildControlFactory.create(childOfConfig, overrideOptions, controllerWrapper);
			if (subChildControl != null) {
				children.add(subChildControl);
				createTabItem(subChildControl, ComponentSettingView.CONFIG_SET_COLS);
			}
			childOfConfig = childrenOfConfigIterator.hasNext() ? childrenOfConfigIterator.next() : null;
		}
		if (tabFolderLoc != null) {
			int tabIndex = 0;
			IChild childLoc;
			boolean shouldBeVisible;
			for (int i = 0; i < children.size(); i++) {
				childControl = children.get(i);
				childLoc = childControl.getChild();
				shouldBeVisible = childLoc.isAvailable();
				if (shouldBeVisible) {
					if (tabIndex >= tabFolderLoc.getItemCount()) {
						createTabItem(childControl, ComponentSettingView.CONFIG_SET_COLS, tabIndex);
					} else {
						CTabItem tabItem = tabFolderLoc.getItem(tabIndex);
						if (!tabItem.isDisposed()) {
							if (tabItem.getData(KEY_TAB_CHILD_CONTROL) != childControl) {
								createTabItem(childControl, ComponentSettingView.CONFIG_SET_COLS, tabIndex);
							} else {
								tabItem.setText(childControl.getChild().getUiName());
							}
						}
					}
					tabIndex++;
				} else {
					removeTabWithControl(childControl);
				}
			}
			if (tabSelectionListenerLoc != null) {
				tabFolderLoc.addSelectionListener(tabSelectionListenerLoc);
			}
			for (int i = 0; i < tabFolderLoc.getItemCount(); i++) {
				CTabItem tabItem = tabFolderLoc.getItem(i);
				Object tabChildControl = tabItem.getData(KEY_TAB_CHILD_CONTROL);
				if (!tabItem.isDisposed() && !children.contains(tabChildControl)) {
					if (tabChildControl instanceof Control) {
						((Control)tabChildControl).dispose();
					}
					tabItem.dispose();
				}
			}
			CTabItem tabToSelect = getTabToSelect();
			if (tabToSelect == null) {
				if (tabFolderLoc.getItemCount() > 0) {
					tabToSelect = tabFolderLoc.getItem(0);
				}
			}
			if ((tabToSelect != null) && !tabToSelect.isDisposed()) {
				tabFolderLoc.setSelection(tabToSelect);
				updateChildSelection(ComponentSettingView.CONFIG_SET_COLS);
			}
			if (!tabFolderLoc.isDisposed()) {
				updateErrorDecoration(tabFolderLoc);
				tabFolderLoc.requestLayout();
			}
		}
	}

	/**
	 * Removes tab, which contains provided childControl, from tab folder
	 * @param childControl that is presented in some tab in tab folder
	 */
	private void removeTabWithControl(IChildControl childControl) {
		final CTabFolder tabFolderLoc = tabFolder;
		if (tabFolderLoc == null) {
			return;
		}
		for (CTabItem tabItem : tabFolderLoc.getItems()) {
			if (!tabItem.isDisposed() && (tabItem.getData(KEY_TAB_CHILD_CONTROL) == childControl)) {
				tabItem.dispose();
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ChildControlBase#updateErrorDecoration
	 */
	@Override
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		super.updateErrorDecoration(controlWithDecoration);
		if (controlWithDecoration instanceof CTabFolder) {
			for (CTabItem tabItem : ((CTabFolder) controlWithDecoration).getItems()) {
				Object data = tabItem.getData(KEY_TAB_CHILD_CONTROL);
				if (data instanceof IChildControl) {
					IChildControl childControl = (IChildControl) data;
					IChild tabChild = childControl.getChild();
					int highestSeverityProblemLevel = tabChild.getHighestSeverityProblemLevel();
					if ((highestSeverityProblemLevel >= ErrorLevels.MIN_LEVEL)
							&& (highestSeverityProblemLevel <= ErrorLevels.MAX_LEVEL)) {
						tabItem.setImage(ToolsImages.getStatusDecoratorImg(highestSeverityProblemLevel));
					} else {
						tabItem.setImage(null);
					}
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.AChildControl#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		for (IChildControl childControl : children) {
			childControl.dispose();
		}
		tabFolder = null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.AChildControl#getChild()
	 */
	@Override
	public @NonNull IChildProvidable getChild() {
		return ((IChildProvidable) child);
	}

	/**
	 * @return unmodifiable {@link #children}
	 */
	public @NonNull List<@NonNull IChildControl> getChildren() {
		return CollectionsUtils.safeList(Collections.unmodifiableList(children));
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createMainControl
	 */
	@Override
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		Composite contentComposite;
		if (ShowContentAs.TABS.equals(getControlOptions().getShowContentAs())) {
			contentComposite = createControlTabs(composite, ComponentSettingView.CONFIG_SET_COLS, new SelectionAdapter() {
				/*
				 * (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					updateChildSelection(ComponentSettingView.CONFIG_SET_COLS);
					ScrolledCompositeHelper.updateScrollSize(composite);
				}
			});
		} else {
			contentComposite = createControlStandard(composite, ComponentSettingView.CONFIG_SET_COLS);
		}
		return (contentComposite == composite) ? null : contentComposite;
	}

	/**
	 * Helper method to create correct composite according to options set.
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @return created composite
	 */
	public @NonNull Composite createComposite(@NonNull Composite parentComposite, int numColumns) {
		return createComposite(parentComposite, numColumns, this, true);
	}

	/**
	 * Helper method to create correct composite according to options set.
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @param childControl child for which composite should be created
	 * @param optimize flag. Set to true if optimizations are allowed, otherwise set false
	 * @return created composite
	 */
	public @NonNull Composite createComposite(@NonNull Composite parentComposite, int numColumns,
			@NonNull IChildControl childControl, boolean optimize) {
		GridLayoutComponents layout = new GridLayoutComponents(numColumns, false);
		// new composite has to be created for settings that should be shown as tabs
		if (optimize && (childControl.getControlOptions().isBorderHidden())
				&& (!ShowContentAs.TABS.equals(getControlOptions().getShowContentAs()))) {
			if (!getControlOptions().isLabelHidden() && !getControlOptions().isExpandGroupHidden()) { // Expand group button is visible
				layout.marginWidth = 8;
				return createNewComposite(parentComposite, childControl, layout);
			}
			return parentComposite;
		} else {
			layout.marginWidth = 8;
		}
		return createNewComposite(parentComposite, childControl, layout);
	}

	/**
	 * Creates new composite for the children
	 * @param parentComposite composite in which to create this one
	 * @param childControl the control of a child
	 * @param layout the layout for the composite
	 * @return new composite created in given parent composite
	 */
	private static @NonNull Composite createNewComposite(@NonNull Composite parentComposite, @NonNull IChildControl childControl, @NonNull GridLayoutComponents layout) {
		// FIXME TomasR v13 maintenance - Check if the layout may be created in this function instead of passing it
		int style = getSwtStyle(childControl);
		layout.horizontalSpacing = 8;
		Composite composite = new Composite(parentComposite, style);
		composite.setLayout(layout);
		return composite;
	}

	/**
	 * Create standard control. Composite and children in it
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @return created composite
	 */
	public @NonNull Composite createControlStandard(@NonNull Composite parentComposite, int numColumns) {
		Composite contentComposite = createComposite(parentComposite, numColumns);
		fillTopOfMainControl(contentComposite, numColumns);
		for (IChildControl childControl : children) {
			childControl.create(contentComposite, numColumns);
		}
		return contentComposite;
	}

	/**
	 * Fill top part of the main control.
	 * @param composite to fill
	 * @param numColumns number of columns of the composite
	 */
	protected void fillTopOfMainControl(@SuppressWarnings("unused") @NonNull Composite composite,
			@SuppressWarnings("unused") int numColumns) {
		// nothing to do
	}

	/**
	 * Create control as a CTabFolder. Children are created as individual CTabItems with their own composite inside the CTabFolder
	 * @param parentComposite parent composite in which new composite should be created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @param selectionListener listener for selection event of tabs
	 * @return created composite
	 */
	public @NonNull Composite createControlTabs(@NonNull Composite parentComposite, int numColumns, @NonNull SelectionListener selectionListener) {
		fillTopOfMainControl(parentComposite, numColumns);
		CTabFolder tabFolderLoc = new CTabFolder(parentComposite, SWT.BORDER);
		tabFolder = tabFolderLoc;
		createHtmlTooltipFolder(tabFolderLoc);
		tabFolderLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, true));
		tabFolderLoc.addSelectionListener(selectionListener);
		tabFolderLoc.addMenuDetectListener(new MenuDetectListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.MenuDetectListener#menuDetected(org.eclipse.swt.events.MenuDetectEvent)
			 */
			@Override
			public void menuDetected(MenuDetectEvent e) {
				CTabFolder tabFolderFromEvent = (CTabFolder) e.widget;
				Point tabRelativeLocation = tabFolderLoc.toControl(new Point(e.x, e.y));
				CTabItem clickedTab = tabFolderFromEvent.getItem(tabRelativeLocation);
				if (clickedTab == null) {
					return;
				}
				Object data = clickedTab.getData(KEY_TAB_CHILD_CONTROL);
				if (!(data instanceof ChildControlBase)) {
					return;
				}
				ChildControlBase childControl = (ChildControlBase) data;
				IChild childInChildControl = childControl.getChild();
				if (!(childInChildControl instanceof ISettingConfig)) {
					return;
				}
				ISettingConfig settingConfig = (ISettingConfig) childInChildControl;
				Menu menu = createCopyPasteMenu(tabFolderLoc, settingConfig);
				tabFolderFromEvent.setMenu(menu);
				menu.setLocation(new Point(e.x, e.y));
				menu.setVisible(true);
			}
		});
		tabSelectionListener = selectionListener;
		restoreSelectionFromStorage();
		return tabFolderLoc;
	}

	/**
	 * Selects lastly selected child according to child selection helper. When no child was selected previously
	 * Must be called only in create functions!
	 */
	protected void restoreSelectionFromStorage() {
		ProvidableSettingsSelectionStorageHelper selectionStorageHelper = ProvidableSettingsSelectionStorageHelper.getInstance();
		if (selectionStorageHelper.isStored(this)) {
			String childId = selectionStorageHelper.getSelectedChild(this);
			IChildControl childToSelect = childId == null? null : CollectionsUtils.nullableOptionalGet(children.stream().filter(x -> x.getChild().getName().equals(childId)).findAny());
			if (childToSelect != null) {
				setSelectedChild(childToSelect);
			}
		}
	}

	/**
	 * Set selected child based on the tab selection.
	 * @param colSpan number of columns set to newly created grid layout for new composite
	 */
	public void updateChildSelection(int colSpan) {
		if (tabFolder != null) {
			final CTabItem selection = tabFolder.getSelection();
			if (selection != null) {
				Object itemData = selection.getData(KEY_TAB_CHILD_CONTROL);
				if (itemData instanceof IChildControl) {
					IChildControl previouslySelectedChild = getSelectedChild();
					IChildControl childControl = (IChildControl) itemData;
					setSelectedChild(childControl);
					if (childControl.isDisposed()
							// FIXME TomasR v13 maintenance - Check why the config set was created (is not disposed) and controls were not yet created
							|| ((childControl instanceof ConfigSetControl) && getControlOptions().shouldLayoutAsTabs())) { // Config set control shown as tabs is not created yet, but is not disposed anymore
						Control selectionControl = selection.getControl();
						if (selectionControl instanceof Composite) {
							disposeContentOfOtherTabs(childControl);
							//UCT-3858 when composite is created override options to have label and border hidden when parent is show as tabs 
							ControlOptions options = createControlOptionsForChild(childControl.getChild());
							childControl.getControlOptions().merge(options);
							childControl.create((Composite) selectionControl, colSpan);
							childControl.update(getControlUpdateType());
						}
					}
					// Update GUI only when selection changed to save performance
					if ((previouslySelectedChild == null) || !previouslySelectedChild.equals(itemData)) {
						// these two lines must be placed here in this order. otherwise, if you switch from a tab with less controls to a tab with more, the
						// additional controls will not be displayed; show scroll bar if controls don't fit the window
						Composite control = (Composite) Objects.requireNonNull(selection.getControl());
						Objects.requireNonNull(control.getParent()).requestLayout();
						control.layout();
						Objects.requireNonNull(Objects.requireNonNull(parent).getParent()).pack();
					}
				}
			}
		}
	}

	/**
	 * Disposes controls of all children in the tab folder except the given one
	 * @param controlToKeep childControl that should be kept
	 */
	private void disposeContentOfOtherTabs(@NonNull IChildControl controlToKeep) {
		for (IChildControl childLoc : children) {
			if ((childLoc != controlToKeep) && !childLoc.isDisposed()) {
				childLoc.dispose();
			}
		}
	}

	/**
	 * Get the tab which should be selected
	 * @return the tab to be selected or {@code null} if no tab should be selected
	 */
	private @Nullable CTabItem getTabToSelect() {
		IChildControl childToSelect = getChildToSelect();
		if (tabFolder != null) {
			for (CTabItem item : tabFolder.getItems()) {
				if (!item.isDisposed() && (item.getData(KEY_TAB_CHILD_CONTROL) == childToSelect)) {
					return item;
				}
			}
		}
		return null;
	}

	/**
	 * @return the currently selected child or {@code null} if no child is selected
	 */
	public @Nullable IChildControl getSelectedChild() {
		return selectedChild;
	}

	/**
	 * Get child which should be selected. The method tries to check whether an item is available and returns it. The preferences are: 1. The lastly selected
	 * item; 2. Item with the same name as the lastly selected item; 3. Item with the same index as the lastly selected item; If none of the above is available
	 * then this method returns the first available item behind the lastly selected item; If no item behind the lastly selected item is available then the
	 * nearest item before the lastly selected item is returned; If none of the above is available then the first available item is returned; Otherwise
	 * {@code null} is returned
	 * @return the child to be selected or {@code null} if no child is available
	 */
	public @Nullable IChildControl getChildToSelect() {
		// try to locate the lastly selected child in the current list of children
		int newSelectionIdx = 0;
		IChildControl selectedChildLoc = selectedChild;
		int lastSelectionIdx = children.indexOf(selectedChildLoc);
		if ((lastSelectionIdx < 0) && (selectedChildLoc != null)) {
			// the lastly selected item is not in the current list, try to find item with the same name
			String selectedChildName = selectedChildLoc.getChild().getName();
			IChildControl childWithSameName = CollectionsUtils.nullableOptionalGet(
					children.stream().filter(ch -> ch.getChild().getName().equals(selectedChildName)).findFirst());
			lastSelectionIdx = children.indexOf(childWithSameName);
			if (lastSelectionIdx < 0) {
				// child with the same name was not found, use the last selection index
				lastSelectionIdx = selectedChildIdx;
			}
		}
		// update index so it is in range [0 .. children.size]
		if (lastSelectionIdx >= children.size()) {
			lastSelectionIdx = children.size() - 1;
		}
		if (lastSelectionIdx < 0) {
			lastSelectionIdx = 0;
		}
		// select appropriate child from the current children list based on the previous selection (preferences: 1. the lastly selected item,
		// 2. first available item behind the lastly selected item, 3. first available item before the lastly selected item)
		if ((lastSelectionIdx >= 0) && (lastSelectionIdx < children.size())) {
			newSelectionIdx = lastSelectionIdx;
			if (!children.get(lastSelectionIdx).getChild().isAvailable()) {
				while ((newSelectionIdx < children.size()) && !children.get(newSelectionIdx).getChild().isAvailable()) {
					newSelectionIdx++;
				}
				if (newSelectionIdx >= children.size()) {
					newSelectionIdx = lastSelectionIdx - 1;
					while ((newSelectionIdx >= 0) && !children.get(newSelectionIdx).getChild().isAvailable()) {
						newSelectionIdx--;
					}
				}
			}
		}
		return ((newSelectionIdx >= 0) && (newSelectionIdx < children.size())) ? children.get(newSelectionIdx) : null;
	}

	/**
	 * Create new CTabItem.
	 * @param childControl childControl for which tab is created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 */
	public void createTabItem(@NonNull IChildControl childControl, int numColumns) {
		createTabItem(childControl, numColumns, -1);
	}

	/**
	 * Create new CTabItem.
	 * @param childControl childControl for which tab is created
	 * @param numColumns number of columns set to newly created grid layout for new composite
	 * @param index the position on which should be CTabItem created in a parent. If index is out of bounds of the parent, new CTabItem is created at the end.
	 */
	public void createTabItem(@NonNull IChildControl childControl, int numColumns, int index) {
		CTabFolder tabFolderLoc = tabFolder;
		if (tabFolderLoc != null) {
			if ((index < 0) || (index > tabFolderLoc.getItemCount())) {
				index = tabFolderLoc.getItemCount();
			}
			CTabItem tabItem = new CTabItem(tabFolderLoc, SWT.NONE, index);
			Composite contentComposite = createComposite(tabFolderLoc, numColumns, childControl, true);
			tabItem.setControl(contentComposite);
			tabItem.setText(childControl.getChild().getUiName());
			tabItem.setData(KEY_TAB_CHILD_CONTROL, childControl);
			tabItem.setToolTipText(UtilsText.EMPTY_STRING);
			contentComposite.requestLayout();
		}
	}

	/**
	 * Set the currently selected child.
	 * @param selectedChild to set
	 */
	public void setSelectedChild(@Nullable IChildControl selectedChild) {
		this.selectedChild = selectedChild;
		selectedChildIdx = children.indexOf(selectedChild);
		ProvidableSettingsSelectionStorageHelper.getInstance().setSelectedChild(this, selectedChild != null ? selectedChild.getChild().getName() : null);
	}

	/**
	 * Select the tab containing the child to select
	 * @param selectedChild to set
	 */
	public void selectChildTab(@NonNull IChildControl selectedChild) {
		final CTabFolder tabFolderLoc = tabFolder;
		if (tabFolderLoc != null) {
			for (int i = 0; i < tabFolderLoc.getItems().length; i++) {
				Object itemData = tabFolderLoc.getItem(i).getData(KEY_TAB_CHILD_CONTROL);
				if (selectedChild.equals(itemData)) {
					tabFolderLoc.setSelection(i);
					return;
				}
			}
		}
	}

	/**
	 * Enables html tooltip for CTabFolder
	 * @param folder the folder instance
	 */
	private static void createHtmlTooltipFolder(@NonNull final CTabFolder folder) {
		HtmlTooltip htmlTooltip = SWTFactoryProxy.INSTANCE.createHtmlTooltip(folder, new ITooltipProvider2() {
			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.swt.ITooltipProvider2#getToolTipText(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public String getToolTipText(Event event) {
				CTabItem item = folder.getItem(new Point(event.x, event.y));
				if (item != null) {
					Object property = item.getData(KEY_TAB_CHILD_CONTROL);
					if (property instanceof IChildControl) {
						return ToolTipableFormatter.getToolTipText(
								new ToolTipableMarkdownDescriptionDecorator(((IChildControl) property).getChild()));
					}
				}
				return null;
			}

			/*
			 * (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.swt.ITooltipProvider#getToolTipText()
			 */
			@Override
			public String getToolTipText() {
				return null;
			}
		});
		folder.setData("SWTFactory.HtmlToolTip", htmlTooltip); //$NON-NLS-1$ // FIXME TomasR v99 - Create proper implementation in SWTFactory
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#isCollapsible()
	 */
	@Override
	public boolean isCollapsible() {
		IChildControl parentChildControl = getParentChildControl();
		if (parentChildControl != null && parentChildControl.getControlOptions().isMultiColumn()) {
			return false;
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#setParentControl(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl)
	 */
	@Override
	public void setParentControl(IChildControl parentChildControl) {
		super.setParentControl(parentChildControl);
		for (IChildControl childLoc : children) {
			childLoc.setParentControl(this);
		}
	}
}
