/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.nxp.swtools.common.ui.utils.GcUtils;
import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.ui.utils.swt.widgets.ReadOnlyText;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.PeripheralsToolTippableFormatter;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.DocumentationView;
import com.nxp.swtools.periphs.gui.view.DocumentationViewHelper;
import com.nxp.swtools.periphs.gui.view.SubSettingsDialog;
import com.nxp.swtools.provider.analytics.ActionAnalyticsBuilder;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.ChildValidationHelper;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.profiler.Profiler;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.tooltip.ToolTipableAmpersandDescriptionDecorator;
import com.nxp.swtools.utils.tooltip.ToolTipableHtmlErrorWarningInfoDecorator;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownValueDescriptionDecorator;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.utils.ValidationProblemsMenuHelper;

import animations.swt.EaseSelector;
import animations.swt.SizeChanger;

/**
 * Base class representing control of a setting configuration.
 * @author Juraj Ondruska
 */
public class ChildControlBase implements IChildControl {
	/** Count of base columns in label composite */
	protected static final int LABEL_COMPOSITE_BASE_COLUMN_COUNT = 5;
	/** Logger of the class */
	protected static final Logger LOGGER = LogManager.getLogger(ChildControlBase.class);
	/** Minimum amount of time how long the animation will be performed (ms) */
	private final int MIN_ANIMATION_DURATION = 500;
	/** Maximum amount of time how long the animation will be performed (ms) */
	private final int MAX_ANIMATION_DURATION = 1000;
	/** Key for the error decoration of a control (set as data of the control). Use with {@link Control#getData(String)} */
	@Deprecated
	public static final @NonNull String KEY_ERROR_DECORATION = ControlDecorationUtils.KEY_ERROR_DECORATION;
	/** The controller */
	public final @NonNull IControllerWrapper controllerWrapper;
	// FIXME TomasR v14 maintenance - Consider replacing direct reference to setting in model (which may be outdated) by ID relative to root - Outdated setting may cause wrong value in cache
	/** The setting configuration */
	protected final @NonNull IChild child;
	/** The composite in which the control is created */
	protected @Nullable Composite parent;
	/** Control of the label composite */
	protected @Nullable Control labelControl;
	/** Control of the label */
	protected @Nullable Label labelControlInternal;
	/** Quick solver icon */
	protected Label quickSolverButton;
	/** Button that opens link to documentation */
	private Label documentationLinkButton;
	/** Expand Group button */
	protected @Nullable Label expandGroupButton;
	/** The group that should be expanded/collapsed */
	protected @Nullable Control expandGroupContent;
	/** Expand Group visible value */
	protected boolean expandGroupVisible = true;
	/** The main mainControl */
	protected @Nullable Control mainControl;
	/** Control of the main content */
	protected @Nullable Control mainControlInternal;
	/** UI options */
	protected @NonNull ControlOptions controlOptions;
	/** signals whether {@link #dispose()} has been called */
	protected boolean disposed;
	/** ColSpan given in create method */
	protected int currentColSpan = 0;
	/** String representing the ampersand character (&) */
	private static final @NonNull String AMPERSAND = "&"; //$NON-NLS-1$
	/** String representing replacement for the ampersand character (&) */
	private static final @NonNull String AMPERSAND_REPLACEMENT = "&&"; //$NON-NLS-1$
	/** The current active sub-settings dialog */
	protected @Nullable SubSettingsDialog activeSubSettingDialog;
	/** sub-settings icon */
	protected @Nullable Label subSettingsButton;
	/** Parent ChildControl */
	private @Nullable IChildControl parentChildControl;
	/** Control update type */
	protected @NonNull UpdateType controlUpdateType = UpdateType.NORMAL;

	/**
	 * @param child the setting configuration to create content for
	 * @param controlOptions UI options for this child
	 * @param controllerWrapper containing the generic controller
	 */
	public ChildControlBase(@NonNull IChild child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		this.child = child;
		this.controllerWrapper = controllerWrapper;
		this.controlOptions = controlOptions;
		this.disposed = true;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		parent = composite;
		final int maxEms = 25;
		currentColSpan = colSpan;
		Control labelControlLoc = (getControlOptions().isLabelHidden()) ? null : createLabelControl(composite);
		labelControl = labelControlLoc;
		Control mainControlLoc = createMainControl(composite);
		if ((getControlOptions().isLabelHidden()) && (mainControlLoc != null) && (!getControlOptions().isBorderHidden())) {
			createErrorDecoration(mainControlLoc, SWT.TOP | SWT.LEFT);
		}
		mainControl = mainControlLoc;
		int labelSpan = (mainControlLoc == null) ? colSpan : 1;
		int controlColSpan = (labelControlLoc == null) ? colSpan : (colSpan - 1);
		if ((labelControlLoc != null) && (labelControlLoc.getLayoutData() == null)) {
			GridDataComponents labelControlLayoutData = new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, labelSpan, 1);
			labelControlLayoutData.maxWidth = GcUtils.getEmWidth(labelControlLoc, true) * maxEms;
			labelControlLoc.setLayoutData(labelControlLayoutData);
		}
		if ((mainControlLoc != null) && (mainControlLoc.getLayoutData() == null)) {
			GridDataComponents mainControlLayoutData = new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, controlColSpan, 1);
			mainControlLoc.setLayoutData(mainControlLayoutData);
		}
		expandGroupContent = mainControlLoc;
		disposed = false;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#update(com.nxp.swtools.periphs.gui.editor.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		Control labelControlLoc = labelControl;
		Control mainControlLoc = mainControl;
		if (((labelControlLoc == null) || labelControlLoc.isDisposed()) && ((mainControlLoc == null) || mainControlLoc.isDisposed())) {
			return; // nothing to update (no UI controls)
		}
		SubSettingsDialog subSettingDialogLoc = activeSubSettingDialog;
		if (subSettingDialogLoc != null) {
			IChildControl childControl = subSettingDialogLoc.getChildControl();
			if (childControl != null) {
				childControl.update(UpdateType.SUB_SETTINGS_DIALOG);
			}
		}
		Control labelControlInternalLoc = labelControlInternal;
		Control mainControlInternalLoc = mainControlInternal;

		Profiler profiler = Profiler.getInstance(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
		BigInteger index = profiler.start(ChildControlBase.class,
				!profiler.isEnabled() ? UtilsText.EMPTY_STRING : MessageFormat.format("GUI update of \"{0}\"", getChild().getId())); //$NON-NLS-1$
		getControlOptions().refresh();
		boolean visible = child.isAvailable() && !child.isControlHidden();
		// this is here to differentiate if the update was called for sub-settings in dialog(to make them visible)
		if (updateType == UpdateType.SUB_SETTINGS_DIALOG) {
			visible = child.isAvailable();
		}
		boolean enabled = (updateType != UpdateType.FORCE_DISABLE) && child.isEnabled();
		UpdateType updateChildrenType = child.isEnabled() ? updateType : UpdateType.FORCE_DISABLE;
		updateStoredUpdateType(updateChildrenType);
		boolean layoutChange = false;
		String toolTipText = visible ? PeripheralsToolTippableFormatter.getToolTipText(new ToolTipableMarkdownValueDescriptionDecorator(
				new ToolTipableMarkdownDescriptionDecorator(new ToolTipableAmpersandDescriptionDecorator(new ToolTipableHtmlErrorWarningInfoDecorator(child))))) : UtilsText.EMPTY_STRING;
		if ((labelControlLoc != null) && !labelControlLoc.isDisposed()) {
			layoutChange = labelControlLoc.isVisible() != visible;
			updateLabelEnableAndVisibleState(enabled, visible);
			updateSubSettingsButton(enabled);
			SWTFactoryProxy.INSTANCE.setTestId(labelControlLoc, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
			if (labelControlInternalLoc != null) {
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelControlInternalLoc, toolTipText);
				if (labelControlInternalLoc.getParent() != null) {
					setParentControlEnabled(labelControlInternalLoc.getParent(), true);
				}
			}
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelControlLoc, toolTipText);
			if (visible) {
				updateLabelContent(labelControlLoc, updateChildrenType);
			}
		}
		if ((mainControlLoc != null) && !mainControlLoc.isDisposed()) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(mainControlLoc, toolTipText);
			// FIXME TomasR v13 maintenance - Controls that are invisible because they are on other tab/collapsed etc. are requesting layout without a reason
			layoutChange = layoutChange || (mainControlLoc.isVisible() != visible);
			setEnabledStateToMainControl(enabled);
			if (mainControlInternalLoc != null) {
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(mainControlInternalLoc, toolTipText);
			}
			mainControlLoc.setVisible(visible);
			((GridDataComponents) Objects.requireNonNull(mainControlLoc.getLayoutData())).exclude = !visible;
			SWTFactoryProxy.INSTANCE.setTestId(mainControlLoc, TestIDs.PERIPHS_SETTING_CONTROL + child.getId());
			if (mainControlInternalLoc != null) {
				SWTFactoryProxy.INSTANCE.setTestId(mainControlInternalLoc, TestIDs.PERIPHS_SETTING_CONTROL + child.getId());
			}
			if (visible) {
				updateMainContent(mainControlLoc, updateChildrenType);
			}
			updateSelectionOfExpandGroupContent();
			Control expandGroupContentLoc = expandGroupContent;
			if ((expandGroupContentLoc != null) && !getControlOptions().isExpandGroupHidden() && !isExpandGroupVisible()) {
				setControlVisible(expandGroupContentLoc, isExpandGroupVisible());
				layoutChange = true;
			}
		}
		if (layoutChange && (parent != null)) {
			parent.requestLayout();
		}
		profiler.stop(index, ChildControlBase.class, null);
	}

	/**
	 * Updates stored update type if it is either NORMAL or FORCE_DISABLE
	 * @param type to be checked and stored
	 */
	protected void updateStoredUpdateType(UpdateType type) {
		if ((type == UpdateType.NORMAL) || (type == UpdateType.FORCE_DISABLE)) {
			controlUpdateType = type;
		}
	}

	/**
	 * Updates selection of expandGroupContent variable
	 */
	protected void updateSelectionOfExpandGroupContent() {
		// Intentionally empty
	}

	/**
	 * Sets given enabled state to the main control
	 * @param enabled flag to be set
	 */
	protected void setEnabledStateToMainControl(boolean enabled) {
		Control mainControlLoc = mainControl;
		Control mainControlInternalLoc = mainControlInternal;
		if (mainControlLoc != null) {
			if (mainControlInternalLoc != null) {
				// Main control is wrapped by composite
				if (mainControlInternalLoc.getEnabled() != enabled) {
					mainControlInternalLoc.setEnabled(enabled);
				}
				// Enable the outer composite to support tooltips on disabled settings
				setParentControlEnabled(mainControlInternalLoc.getParent(), true);
			} else {
				// Main control is not wrapped and must be set enabled this way
				if (mainControlLoc.getEnabled() != enabled) {
					mainControlLoc.setEnabled(enabled);
				}
				setParentControlEnabled(mainControlLoc.getParent(), true);
			}
		}
	}

	/**
	 * Update error decoration attached to the control by the previous call of {@link #createErrorDecoration(Control, int)}
	 * @param controlWithDecoration with the decoration to update
	 */
	protected void updateErrorDecoration(@NonNull Control controlWithDecoration) {
		if (!isDisposed()) {
			final StringBuilder result = new StringBuilder();
			int statusLevel = getStatusLevel();
			if (getChild().isSynchronizationEnabled()) {
				if (statusLevel == ErrorLevels.LEVEL_SUCCESS) {
					statusLevel = ErrorLevels.LEVEL_INFORMATION;
				}
				result.append(Messages.get().ToolTipableFormater_SyncValue);
				result.append(UtilsText.LF);
			}
			String status = getDecoratorStatus(statusLevel);
			if (statusLevel >= ErrorLevels.LEVEL_ERROR) {
				result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Errors);
			} else if (statusLevel == ErrorLevels.LEVEL_WARNING) {
				result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Warnings);
			} else if (statusLevel == ErrorLevels.LEVEL_INFORMATION) {
				result.append(com.nxp.swtools.utils.Messages.get().ToolTipableFormater_Infos);
			}
			String dependenciesMessageInTree = ValidationHelper.getDependenciesMessageInTree(child, statusLevel);
			final String statusAndDependencyMessage = ChildValidationHelper.mergeProblems(status, dependenciesMessageInTree);
			if (!UtilsText.isEmpty(statusAndDependencyMessage)) {
				result.append(UtilsText.LF);
				result.append(statusAndDependencyMessage);
			}
			String decoratorDescription = null;
			String quickFixDescription = null;
			if (result.length() > 0) {
				decoratorDescription = result.toString();
				result.insert(0, Messages.get().QuickFixButton_Description + UtilsText.LF);
				quickFixDescription = result.toString();
			}
			if (decoratorDescription != null) {
				ControlDecorationUtils.setImage(controlWithDecoration, ToolsImages.getStatusDecoratorImg(statusLevel));
			}
			updateQuickSolver(quickFixDescription);
			ControlDecorationUtils.updateControlDecoration(controlWithDecoration, decoratorDescription);
		}
	}

	/**
	 * Returns status used for decorator. Can be overridden to add additional status by specific implementations.
	 * @param statusLevel level of status to be returned
	 * @return status of current child
	 */
	protected String getDecoratorStatus(int statusLevel) {
		String status = null;
		if (statusLevel == ErrorLevels.LEVEL_ERROR) {
			status = child.getError();
		} else if (statusLevel == ErrorLevels.LEVEL_WARNING) {
			status = child.getWarning();
		} else if (statusLevel == ErrorLevels.LEVEL_INFORMATION) {
			status = child.getInfo();
		}
		return status;
	}

	/**
	 * Returns status level of current child. Can be overridden by specific implementation to modify the level of reported problem.
	 * @return one of constants from {@link ErrorLevels}
	 */
	protected int getStatusLevel() {
		final int problemLevelOfChild = ValidationHelper.getHighestSeveritySettingValidationProblemLevelInTree(child);
		int statusLevel = ErrorLevels.LEVEL_SUCCESS;
		if ((child.getError() != null) || (problemLevelOfChild >= ErrorLevels.LEVEL_ERROR)) {
			statusLevel = ErrorLevels.LEVEL_ERROR;
		} else if ((child.getWarning() != null) || (problemLevelOfChild == ErrorLevels.LEVEL_WARNING)) {
			statusLevel = ErrorLevels.LEVEL_WARNING;
		} else if ((child.getInfo() != null) || (problemLevelOfChild == ErrorLevels.LEVEL_INFORMATION)) {
			statusLevel = ErrorLevels.LEVEL_INFORMATION;
		}
		return statusLevel;
	}

	/**
	 * Update status of quick solver button
	 * @param description to be set to the quick solver button
	 */
	protected void updateQuickSolver(@Nullable String description) {
		Label quickSolverLoc = quickSolverButton;
		if ((quickSolverLoc != null) && !quickSolverLoc.isDisposed()) {
			int statusLevel = ValidationHelper.getHighestSeveritySettingValidationProblemLevel(getChild());
			List<@NonNull IBaseProblem> problems = new ArrayList<>();
			problems.addAll(ValidationHelper.getSettingValidationProblems(getChild(), ErrorLevels.LEVEL_FATAL));
			problems.addAll(ValidationHelper.getSettingValidationProblems(getChild(), ErrorLevels.LEVEL_ERROR));
			problems.addAll(ValidationHelper.getSettingValidationProblems(getChild(), ErrorLevels.LEVEL_WARNING));
			List<@NonNull IBaseProblem> mergedProblems = new ArrayList<>(problems.size());
			ValidationProblemsMenuHelper.mergeProblems(problems, mergedProblems);
			boolean visible = false;
			if (!problems.isEmpty()) {
				MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
				menuMgr.setRemoveAllWhenShown(true);
				menuMgr.addMenuListener(manager -> {
					for (IBaseProblem problem : mergedProblems) {
						if (problem instanceof IValidationProblem) {
							IValidationProblem validationProblem = (IValidationProblem) problem;
							MenuManager problemMenuManager = new MenuManager(validationProblem.getDependency().getDescription());
							ValidationProblemsMenuHelper.createPopupMenu(validationProblem, problemMenuManager, false);
							manager.add(problemMenuManager);
						}
					}
				});
				Menu oldMenu = quickSolverLoc.getMenu();
				if (oldMenu!= null) {
					oldMenu.dispose();
				}
				Menu menu = menuMgr.createContextMenu(quickSolverLoc);
				quickSolverLoc.setMenu(menu);
				visible = true;
			}
			quickSolverLoc.setVisible(visible);
			Object layoutData = quickSolverLoc.getLayoutData();
			if (layoutData instanceof GridDataComponents) {
				GridDataComponents data = (GridDataComponents) layoutData;
				data.exclude = !visible;
			}
			switch (statusLevel) {
				case ErrorLevels.LEVEL_FATAL:
				case ErrorLevels.LEVEL_ERROR:
				case ErrorLevels.LEVEL_WARNING: {
					quickSolverLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_AUTO_RESOLVE_PROBLEM));
					break;
				}
				case ErrorLevels.LEVEL_INFORMATION:
				case ErrorLevels.LEVEL_SUCCESS:
				default: {
					quickSolverLoc.setImage(null);
					break;
				}
			}
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(quickSolverLoc, description);
			quickSolverLoc.requestLayout();
		}
	}

	/**
	 * Updates the status of the sub-settings button.
	 * @param enabled state of sub-settings button.
	 */
	protected void updateSubSettingsButton(boolean enabled) {
		Label subSettingsButtonLoc = subSettingsButton;
		if ((subSettingsButtonLoc != null) && !subSettingsButtonLoc.isDisposed()) {
			List<@NonNull ISettingConfig> relatedSettings = getSubSettingsChildren();
			boolean visible = false;
			if (!relatedSettings.isEmpty()) {
				visible = true;
				MenuManager menuMgr = new MenuManager();
				menuMgr.setRemoveAllWhenShown(true);
				menuMgr.addMenuListener(new IMenuListener() {
					/*
					 * (non-Javadoc)
					 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
					 */
					@Override
					public void menuAboutToShow(IMenuManager manager) {
						for (int i = 0; i < relatedSettings.size(); i++) {
							ISettingConfig subSetting = relatedSettings.get(i);
							manager.add(ActionAnalyticsBuilder.action(() -> openSubSettingsDialog(subSetting)).id(subSetting.getId())
									.text(subSetting.getUiName()).enabled(child.isEnabled()).build());
						}
					}
				});
				subSettingsButtonLoc.setMenu(menuMgr.createContextMenu(subSettingsButtonLoc));
			}
			subSettingsButtonLoc.setVisible(visible);
			subSettingsButtonLoc.setEnabled(enabled);
			Object layoutData = subSettingsButtonLoc.getLayoutData();
			if (layoutData instanceof GridDataComponents) {
				GridDataComponents data = (GridDataComponents) layoutData;
				data.exclude = !visible;
			}
			subSettingsButtonLoc.requestLayout();
		}
	}

	/**
	 * Create error decoration for the given control. The decoration is hidden by default. Its state should be updated by calling
	 * {@link #updateErrorDecoration(Control)}
	 * @param controlToDecorate control for which to create decoration
	 * @param position of the decoration, possible values are defined in {@link ControlDecoration#ControlDecoration(Control, int)}
	 */
	protected void createErrorDecoration(@NonNull Control controlToDecorate, int position) {
		ControlDecorationUtils.createErrorDecoration(controlToDecorate, position);
	}

	/**
	 * Update content of a label.
	 * @param label the control to update
	 * @param updateType whether to disable children (if there are some children)
	 */
	protected void updateLabelContent(@NonNull Control label, @NonNull UpdateType updateType) {
		if ((label instanceof Composite)) {
			Label labelControlLoc = labelControlInternal;
			if (labelControlLoc != null) {
				if (updateType != UpdateType.PROBLEM_DECORATION) {
					labelControlLoc.setText(getChild().getUiName().replaceAll(AMPERSAND, AMPERSAND_REPLACEMENT));
					if (getControlOptions().isLabelBold()) {
						setLabelInternalFontStyle(SWT.BOLD);
					}
					// Replace max width of label by max width of it's composite to enable wrapping
					Object compositeLayoutData = label.getLayoutData();
					Object labelLayoutData = labelControlLoc.getLayoutData();
					if ((compositeLayoutData != null) && (labelLayoutData != null)) {
						((GridDataComponents) labelLayoutData).maxWidth = ((GridDataComponents) compositeLayoutData).maxWidth;
					}
				}
				updateErrorDecoration(labelControlLoc);
				updateDocumentationButton();
			}
		}
		if ((label instanceof Label)) { // FIXME TomasR v13 maintenance - this should not be used by any control -> remove it
			if (updateType != UpdateType.PROBLEM_DECORATION) {
				((Label) label).setText(child.getUiName());
			}
			updateErrorDecoration(label);
		}
		boolean enable = !UpdateType.FORCE_DISABLE.equals(updateType);
		if (expandGroupButton != null) {
			expandGroupButton.setEnabled(enable);
		}
	}

	/**
	 * Update tooltip of label
	 * @param tooltip text for tooltip
	 */
	protected void updateLabelInternalTooltip(@Nullable String tooltip) {
		Label labelControlInternalLoc = labelControlInternal;
		if ((labelControlInternalLoc != null) && !labelControlInternalLoc.isDisposed()) {
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelControlInternalLoc, tooltip);
		}
	}

	/**
	 * Update font style of label
	 * @param style to be set
	 */
	protected void setLabelInternalFontStyle(int style) {
		Label labelControlInternalLoc = labelControlInternal;
		if ((labelControlInternalLoc != null) && !labelControlInternalLoc.isDisposed()) {
			FontFactory.changeStyle(labelControlInternalLoc, style);
		}
	}

	/**
	 * Update enabled and visible state of label
	 * @param enabled state
	 * @param visible state
	 */
	protected void updateLabelEnableAndVisibleState(boolean enabled, boolean visible) {
		Control labelControlLoc = labelControl;
		if ((labelControlLoc != null) && !labelControlLoc.isDisposed()) {
			labelControlLoc.setVisible(visible);
			((GridDataComponents) Objects.requireNonNull(labelControlLoc.getLayoutData())).exclude = !visible;
		}
		Control labelControlInternalLoc = labelControlInternal;
		if ((labelControlInternalLoc != null) && !labelControlInternalLoc.isDisposed()) {
			if (labelControlInternalLoc.getEnabled() != enabled) {
				labelControlInternalLoc.setEnabled(enabled);
			}
			labelControlInternalLoc.setVisible(visible);
		}
	}

	/**
	 * Update main content of a setting configuration, e.g. an editor.
	 * @param contentControl the mainControl to update
	 * @param updateType whether to disable children (if there are some children)
	 */
	protected void updateMainContent(@NonNull Control contentControl, @NonNull UpdateType updateType) {
		if ((contentControl instanceof Text) && (updateType != UpdateType.PROBLEM_DECORATION)) {
			((ReadOnlyText) contentControl).setText(child.getUiName());
		}
		updateErrorDecoration(contentControl);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#dispose()
	 */
	@Override
	public void dispose() {
		Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			labelControlLoc.dispose();
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null) {
			mainControlLoc.dispose();
		}
		labelControl = null;
		mainControl = null;
		disposed = true;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#isDisposed()
	 */
	@Override
	public boolean isDisposed() {
		return disposed;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.IChildControl#getChild()
	 */
	@Override
	public @NonNull IChild getChild() {
		return child;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#getContent()
	 */
	@Override
	public @Nullable Composite getContent() {
		return parent;
	}

	/**
	 * Creates the label composite.
	 * @param composite in which to create the label composite
	 * @return the newly created label composite or {@code null} if there should not be label
	 */
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		Composite labelComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(getLabelCompositeColumnCount(), false);
		layout.marginHeight = 0;
		layout.verticalSpacing = 0;
		layout.marginWidth = 0;
		labelComposite.setLayout(layout);
		if (isCollapsible() && !getControlOptions().isExpandGroupHidden()) {
			createCollapseButton(labelComposite);
		}
		createSubSettingsButton(labelComposite);
		createLabelControlInternal(labelComposite);
		createQuickSolverButton(labelComposite);
		createDocumentationLinkButton(labelComposite);
		createLabelControlAdditionalControls(labelComposite);
		return labelComposite;
	}

	/**
	 * @return the count of columns for label composite
	 */
	protected int getLabelCompositeColumnCount() {
		return LABEL_COMPOSITE_BASE_COLUMN_COUNT;
	}

	/**
	 * Create additional controls in label composite. Override if additional controls are needed for some settings
	 * @param composite to which add the controls
	 */
	protected void createLabelControlAdditionalControls(@SuppressWarnings("unused") @NonNull Composite composite) {
		// Intentionally empty
	}

	/**
	 * Creates the label control of the setting
	 * @param composite in which to create the button
	 */
	protected void createLabelControlInternal(Composite composite) {
		Label label = new Label(composite, SWT.WRAP);
		label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, false, false, 1, 1));
		SWTFactoryProxy.INSTANCE.setTestId(label, TestIDs.PERIPHS_SETTING_LABEL + child.getId());
		label.setText(child.getUiName().replaceAll(AMPERSAND, AMPERSAND_REPLACEMENT));
		ControlDecorationUtils.createErrorDecoration(label, SWT.LEFT | SWT.TOP);
		labelControlInternal = label;
	}

	/**
	 * Creates button which contains context menu with quick solution to problem that relates to the setting
	 * @param composite in which to create the button
	 */
	protected void createQuickSolverButton(Composite composite) {
		Label quickSolverButtonLoc = new Label(composite, SWT.NONE);
		GridDataComponents quickSolverLayoutData = new GridDataComponents();
		quickSolverLayoutData.exclude = true;
		quickSolverButtonLoc.setVisible(false);
		quickSolverButtonLoc.setLayoutData(quickSolverLayoutData);
		quickSolverButton = quickSolverButtonLoc;
	}

	/**
	 * Creates button which opens documentation view to a specific page when clicked
	 * @param composite in which to create the button
	 */
	protected void createDocumentationLinkButton(Composite composite) {
		Label button = new Label(composite, SWT.NONE);
		GridDataComponents layoutData = new GridDataComponents();
		button.setLayoutData(layoutData);
		button.setImage(ToolsImages.getImage(IToolsImages.ICON_QUESTION_MARK));
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_OPEN_DOCUMENTATION_LINK_FROM_SETTING + getChild().getId());
		button.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				String page = getControlOptions().getDocumentationPageToOpen();
				if (page == null) {
					return;
				}
				IWorkbenchWindow workbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				if (workbenchWindow == null) {
					return;
				}
				IWorkbenchPage workbenchPage = workbenchWindow.getActivePage();
				IComponentInstanceConfig instance = getChild().getChildContext().getComponentInstanceConfig();
				if (instance == null) {
					return;
				}
				String componentType = instance.getComponentTypeId();
				String pageInComponent = page;
				int separatorPosition = page.indexOf(DocumentationView.OTHER_COMPONENT_SEPARATOR);
				if (separatorPosition != -1) {
					ConfigurationComponentTypeId targetTypeId = DocumentationView.getTargetTypeId(page, controllerWrapper);
					if (targetTypeId != null) {
						componentType = targetTypeId.getTypeId();
						pageInComponent = page.substring(separatorPosition + 2);
					}
				}
				DocumentationView.open(workbenchPage, componentType, true);
				DocumentationView.openPage(pageInComponent);
			}
		});
		documentationLinkButton = button;
	}

	/**
	 * Updates the documentation button
	 */
	protected void updateDocumentationButton() {
		Label button = documentationLinkButton;
		if (button != null) {
			boolean documentationPageToOpenSpecified = getControlOptions().isDocumentationPageToOpenSpecified();
			button.setVisible(documentationPageToOpenSpecified);
			if (documentationPageToOpenSpecified) {
				String page = getControlOptions().getDocumentationPageToOpen();
				if (page != null) {
					String pageName = DocumentationViewHelper.getNameOfPage(getChild(), page);
					if (pageName != null) {
						SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, MessageFormat.format(Messages.get().ComponentSettingView_DocumentationViewPage, pageName));
					}
				}
			}
			Object data = button.getLayoutData();
			if (data instanceof GridDataComponents) {
				GridDataComponents layoutData = (GridDataComponents) data;
				layoutData.exclude = !documentationPageToOpenSpecified;
			}
		}
	}

	/**
	 * Creates button which collapses the setting's control on click
	 * @param composite in which to create the button
	 */
	protected void createCollapseButton(Composite composite) {
		boolean visible = !getControlOptions().isGroupCollapsed();
		if (CollapsibleSettingsStorageHelper.getInstance().isStored(this)) {
			visible = CollapsibleSettingsStorageHelper.getInstance().isExpanded(this);
		}
		Label expandGroupButtonLoc = new Label(composite, SWT.NONE);
		expandGroupButtonLoc.setVisible(true);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(expandGroupButtonLoc, Messages.get().ComponentSettingView_ExpandCollapseButton);
		SWTFactoryProxy.INSTANCE.setTestId(expandGroupButtonLoc, TestIDs.PERIPHS_COLLAPSE_SETTING_LABEL + getChild().getId());
		GridDataComponents gd = new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false);
		expandGroupButtonLoc.setLayoutData(gd);
		if (!visible) {
			expandGroupButtonLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_RIGHT));
		} else {
			expandGroupButtonLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_DOWN));
		}
		expandGroupVisible = visible;
		saveExpandedState();
		expandGroupButtonLoc.addMouseListener(new MouseAdapter() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.swt.events.MouseAdapter#mouseUp(org.eclipse.swt.events.MouseEvent)
			 */
			@Override
			public void mouseUp(MouseEvent e) {
				super.mouseUp(e);
				Control expandGroupLocal = expandGroupContent;
				if ((expandGroupLocal != null) && !expandGroupLocal.isDisposed()) {
					expandGroupVisible = !expandGroupVisible;
					if (KEPreferences.isAnimationsEnabled()) {
						setControlVisibleAnimation(expandGroupLocal, expandGroupButtonLoc, expandGroupVisible);
					} else {
						setControlVisible(expandGroupLocal, expandGroupVisible);
						updateLayout();
						onExpandGroupVisibilityChange();
						saveExpandedState();
					}
					// update button image
					if (expandGroupVisible) {
						expandGroupButtonLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_DOWN));
					} else {
						expandGroupButtonLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_RIGHT));
					}
				}
			}
		});
		expandGroupButton = expandGroupButtonLoc;
	}

	/**
	 * Saves the current expanded state to helper
	 */
	private void saveExpandedState() {
		if (isExpandGroupVisible()) {
			CollapsibleSettingsStorageHelper.getInstance().expandControl(ChildControlBase.this);
		} else {
			CollapsibleSettingsStorageHelper.getInstance().collapseControl(ChildControlBase.this);
		}
	}

	/**
	 * Checks whether the expand group is visible or not
	 * @return {@code true} when the expand group is visible, {@code false} otherwise
	 */
	protected boolean isExpandGroupVisible() {
		return expandGroupVisible;
	}

	/**
	 * Executed after the expand group visibility changes
	 */
	protected void onExpandGroupVisibilityChange() {
		// Intentionally empty
	}

	/**
	 * Creates the sub-settings button which opens the sub-settings dialog
	 * @param composite in which to create the button
	 */
	protected void createSubSettingsButton(Composite composite) {
		Label subSettingsButtonLoc = new Label(composite, SWT.NONE);
		GridDataComponents buttonGridData = new GridDataComponents(SWT.LEFT, SWT.BOTTOM, false, false);
		buttonGridData.exclude = true;
		subSettingsButtonLoc.setVisible(false);
		subSettingsButtonLoc.setLayoutData(buttonGridData);
		subSettingsButtonLoc.setImage(ToolsImages.getImage(IToolsImages.ICON_SETTINGS));
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(subSettingsButtonLoc, Messages.get().ToolTip_SubSettingsButton);
		SWTFactoryProxy.INSTANCE.setTestId(subSettingsButtonLoc, TestIDs.PERIPHS_SUB_SETTINGS_BUTTON + child.getId());
		subSettingsButton = subSettingsButtonLoc;
	}

	/**
	 * Create main mainControl.
	 * @param composite to create mainControl in
	 * @return the newly created mainControl or {@code null} if there should not be content
	 */
	public @Nullable Control createMainControl(@NonNull Composite composite) {
		Text text = new Text(composite, getSwtStyle() | SWT.READ_ONLY | SWT.WRAP);
		text.setText(child.getUiName());
		return text;
	}

	/**
	 * Returns a list with sub-settings for a given child.
	 * @return list with sub-settings for the given child.
	 */
	protected @NonNull List<@NonNull ISettingConfig> getSubSettingsChildren() {
		List<@NonNull ISettingConfig> childSubSettings = new ArrayList<>();
		Object optionValue = child.getOptionValue(SettingOptions.RELATED_SETTINGS);
		if (optionValue != null) {
			if (optionValue instanceof List) {
				List<?> settings = (List<?>) optionValue;
				for (Object obj : settings) {
					if (obj instanceof ISettingConfig) {
						childSubSettings.add((ISettingConfig) obj);
					} else {
						LOGGER.log(Level.WARNING, "The received item is not an instance of " + ISettingConfig.class.getName()); //$NON-NLS-1$
					}
				}
			} else {
				LOGGER.log(Level.SEVERE, String.format(
						"The current type is not supported for RELATED_SETTINGS option of setting: %1s. Only the array setting instances are supported.", //$NON-NLS-1$
						getChild()));
			}
		} else if (child.isOptionSet(SettingOptions.RELATED_SETTINGS_AUTO) && (child instanceof ISettingConfig)) {
			List<@NonNull ISettingConfig> relatedSettings = ((ISettingConfig) child).getAllRelatedSettings();
			if (relatedSettings != null) {
				childSubSettings = relatedSettings;
			}
		}
		return childSubSettings;
	}

	/**
	 * Creates and opens the sub-settings dialog
	 * @param subSetting to create inside the dialog
	 */
	protected void openSubSettingsDialog(@NonNull IChild subSetting) {
		Display currentDisplay = Display.getCurrent();
		if (currentDisplay != null) {
			Shell activeShell = currentDisplay.getActiveShell();
			if (activeShell != null) {
				SubSettingsDialog diag = new SubSettingsDialog(activeShell, subSetting, controllerWrapper, new ControlOptions(subSetting));
				activeSubSettingDialog = diag;
				diag.open();
			}
		}
	}

	/**
	 * Get integer style bits for SWT based on defined options (currently only hideBorder).
	 * @return SWT style bits
	 */
	public int getSwtStyle() {
		return getSwtStyle(this);
	}

	/**
	 * Get integer style bits for SWT based on defined options (currently only hideBorder).
	 * @param childControl child for which style should be calculated
	 * @return SWT style bits
	 */
	public static int getSwtStyle(@NonNull IChildControl childControl) {
		if (childControl.getControlOptions().isBorderHidden()) {
			return SWT.NONE;
		} else {
			return SWT.BORDER;
		}
	}

	/**
	 * Returns current status icon
	 * @param element to get status from
	 * @return current status icon or {@code null} when there is no problem
	 */
	public static @Nullable Image getStatusIcon(@NonNull IChild element) {
		if (element.getError() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_ERROR);
		} else if (element.getWarning() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING);
		} else if (element.getInfo() != null) {
			return ToolsImages.getStatusIcon(ErrorLevels.LEVEL_INFORMATION);
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#getControlOptions()
	 */
	@Override
	public @NonNull ControlOptions getControlOptions() {
		return this.controlOptions;
	}

	/**
	 * Adds scroll listener that scrolls with ScrolledComposite which is parent of contentContainer
	 * @param control to which add this scroll listener
	 * @param childOfScrolledComposite child of ScrolledComposite which should be scrolled
	 */
	public static void addScrollListener(@NonNull Control control, @NonNull Composite childOfScrolledComposite) {
		control.addListener(SWT.MouseWheel, new Listener() {
			@Override
			public void handleEvent(Event event) {
				event.doit = false; // Do not execute handle in original event listener
				ScrolledCompositeHelper.scrollByNumberOfLines(childOfScrolledComposite, -event.count, ScrolledCompositeHelper.DEFAULT_SCROLL_INCREMENT);
			}
		});
	}

	/**
	 * Register copy and paste menu on label
	 */
	protected void registerCopyPasteMenuOnLabel() {
		Label labelControlLoc = labelControlInternal;
		if (labelControlLoc != null) {
			registerCopyPasteMenu(labelControlLoc);
		}
	}

	/**
	 * Registers copy and paste menu to right mouse button click
	 * @param control that will support copy and paste
	 */
	protected void registerCopyPasteMenu(@NonNull Control control) {
		ISettingConfig setting = null;
		if (getChild() instanceof ISettingConfig) {
			setting = (ISettingConfig) getChild();
			final ISettingConfig finalSetting = setting;
			control.setMenu(createCopyPasteMenu(control, finalSetting));
		}
	}

	/**
	 * Creates the copy and paste menu. Contains following actions: copy, paste, paste as new item
	 * @param control on which the menu will be created
	 * @param setting which can be copied/can be pasted into
	 * @return menu object
	 */
	protected Menu createCopyPasteMenu(@NonNull Control control, final @NonNull ISettingConfig setting) {
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			@Override
			public void menuAboutToShow(IMenuManager manager) {
				String clipboardString = UtilsText.EMPTY_STRING;
				final Clipboard clipBoard = new Clipboard(control.getDisplay());
				Object clipboardContent = clipBoard.getContents(TextTransfer.getInstance());
				if (clipboardContent instanceof String) {
					clipboardString = (String) clipboardContent;
				}
				final String clipboardStringFinal = clipboardString;
				// Copy
				manager.add(ActionAnalyticsBuilder.action(new Runnable() {
					@Override
					public void run() {
						String stringCopyResult = controllerWrapper.getController().copy(setting);
						if (UtilsText.isEmpty(stringCopyResult)) {
							LOGGER.log(Level.SEVERE, "Copy did not produce String for the clipboard"); //$NON-NLS-1$
							return;
						}
						clipBoard.setContents(new Object[] { stringCopyResult }, new Transfer[] { TextTransfer.getInstance() });
					}
				}).id(TestIDs.PERIPHS_CHILD_CONTROL_COPY).text(Messages.get().ChildControl_Copy).image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
						.enabled(controllerWrapper.getController().canCopy(setting)).build());
				// Paste
				manager.add(ActionAnalyticsBuilder.action(new Runnable() {
					@Override
					public void run() {
						controllerWrapper.getController().paste(setting, clipboardStringFinal);
					}
				}).id(TestIDs.PERIPHS_CHILD_CONTROL_PASTE).text(Messages.get().ChildControl_Paste).image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
						.enabled(controllerWrapper.getController().canPaste(setting, clipboardStringFinal)).build());
				// Paste as new item
				if (setting instanceof ArrayConfig) {
					ArrayConfig finalArray = (ArrayConfig) setting;
					manager.add(ActionAnalyticsBuilder.action(new Runnable() {
						@Override
						public void run() {
							controllerWrapper.getController().pasteAsNewItem(finalArray, clipboardStringFinal);
						}
					}).id(TestIDs.PERIPHS_CHILD_CONTROL_PASTE_AS_NEW_ITEM).text(Messages.get().ChildControl_Paste_AsNewItem)
							.image(ToolsImages.getImageDescriptor(IToolsImages.ICON_COPY))
							.enabled(controllerWrapper.getController().canPasteAsNewItem(finalArray, clipboardStringFinal)).build());
				}
			}
		});
		Menu createdMenu = menuMgr.createContextMenu(control);
		return createdMenu;
	}

	/**
	 * Sets visibility of control
	 * @param control to be set
	 * @param visible flag
	 */
	protected void setControlVisible(@NonNull Control control, boolean visible) {
		Object data = control.getLayoutData();
		if (data instanceof GridDataComponents) {
			GridDataComponents layoutData = (GridDataComponents) data;
			layoutData.exclude = !visible;
			control.setVisible(visible);
		}
	}

	/**
	 * Sets visibility of control with animation
	 * @param control to be animate
	 * @param button expand/collapse button
	 * @param visible flag
	 */
	protected void setControlVisibleAnimation(@NonNull Control control, @Nullable Label button, boolean visible) {
		Point optimalSize = control.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		int duration = countDuration(optimalSize.y);
		int height;
		if (!visible) {
			height = 0;
		} else {
			Object data = control.getLayoutData();
			if (data instanceof GridDataComponents) {
				GridDataComponents layoutData = (GridDataComponents) data;
				layoutData.exclude = !visible;
			}
			updateLayout();
			height = optimalSize.y;
		}
		control.getDisplay().asyncExec(() -> {
			if (!control.isDisposed()) {
				if (visible) {
					control.setSize(control.getSize().x, 0);
				}
				SizeChanger sizeChanger = new SizeChanger(control, control.getSize().x, height, false, EaseSelector.SINE) {
					/*
					 * (non-Javadoc)
					 * @see animations.swt.ChangerBase#animationStartCallback()
					 */
					@Override
					public void animationStartCallback() {
						if (button != null) {
							button.setEnabled(false);
						}
						if (visible) {
							control.getDisplay().asyncExec(() -> {
								control.setVisible(visible);
								onExpandGroupVisibilityChange();
								saveExpandedState();
							});
						}
					}

					/*
					 * (non-Javadoc)
					 * @see animations.swt.ChangerBase#animationDoneCallback()
					 */
					@Override
					public void animationDoneCallback() {
						if (button != null) {
							button.setEnabled(true);
						}
						if (!visible) {
							setControlVisible(control, visible);
							updateLayout();
						}
						onExpandGroupVisibilityChange();
						saveExpandedState();
					}
				};
				sizeChanger.play(duration);
			}
		});
	}

	/**
	 * Count duration of animation based on height of composite
	 * @param height of composite
	 * @return duration (ms)
	 */
	private int countDuration(int height) {
		Display display = Display.getCurrent();
		if (display != null) {
			height = (height * 100) / display.getDPI().y;
		}
		if (height > 1) {
			int duration = (int) Math.log(height) * 140;
			if (duration < MIN_ANIMATION_DURATION) {
				return MIN_ANIMATION_DURATION;
			}
			if (duration > MAX_ANIMATION_DURATION) {
				return MAX_ANIMATION_DURATION;
			}
			return duration;
		} else {
			return MIN_ANIMATION_DURATION;
		}
	}

	/**
	 * Update layout when composite is expanded or collapsed
	 */
	protected void updateLayout() {
		Composite parentLoc = parent;
		if (parentLoc != null) {
			ScrolledCompositeHelper.updateScrollSize(parentLoc);
			parentLoc.requestLayout();
			Composite parentComposite = parentLoc.getParent();
			if (parentComposite != null) {
				parentComposite.layout();
			}
		}
	}

	/**
	 * Sets the same enabled state to the control and to its children if it has any. Important: Use wisely. Setting enabled to all children can be
	 * counterproductive when children will set different state in their call to this function. Will cause performance issues when enable state will
	 * be set for the same control multiple times.
	 * @param control to which set the enabled state
	 * @param enabled state
	 * @param recursive flag
	 */
	protected void setControlEnabled(@NonNull Control control, boolean enabled, boolean recursive) {
		if (recursive && (control instanceof Composite)) {
			Composite composite = (Composite) control;
			for (Control childControl : composite.getChildren()) {
				setControlEnabled(childControl, enabled, recursive);
			}
		}
		if (control.getEnabled() != enabled) {
			control.setEnabled(enabled);
		}
	}

	/**
	 * Sets the same enabled state to the parents of the given composite and to the composite itself
	 * @param composite of whose enabled state will be set
	 * @param enabled state
	 */
	protected void setParentControlEnabled(@Nullable Composite composite, boolean enabled) {
		if ((composite != null)) {
			if (composite.getEnabled() != enabled) {
				composite.setEnabled(enabled);
			}
			Composite compositeParent = composite.getParent();
			if ((compositeParent != null) && (compositeParent.isEnabled() != enabled)) {
				// There is parent composite in tree of controls and the path is still not set to the same enabled state
				setParentControlEnabled(compositeParent, enabled);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl#setParentControl(com.nxp.swtools.periphs.gui.view.componentsettings.
	 * IChildControl)
	 */
	@Override
	public void setParentControl(IChildControl parentChildControl) {
		this.parentChildControl = parentChildControl;
	}

	/**
	 * Get child control of parent
	 * @return child control of parent or {@code null} when it was not set
	 */
	public @Nullable IChildControl getParentChildControl() {
		return this.parentChildControl;
	}

	/**
	 * @return the control {@link UpdateType}
	 */
	public @NonNull UpdateType getControlUpdateType() {
		return controlUpdateType;
	}

	/**
	 * Handles opening of instance view of linked child
	 * If linked setting is from global config set then global settings view will be opened.
	 * @param linkedChild
	 * @param caller
	 */
	protected void handleLinkToSetting(IChild linkedChild, @NonNull Object caller) {
		IComponentConfig componentConfig = linkedChild.getChildContext().getComponentConfig();
		if (componentConfig != null) {
			IConfigSetConfig globalConfigSet = componentConfig.getGlobalConfigSet();
			IConfigSetConfig configSetConfig = linkedChild.getChildContext().getConfigSetConfig();
			if ((globalConfigSet != null) && (configSetConfig == globalConfigSet)) { // Link is to global config set
				controllerWrapper.getGUIController().openGlobalViewOfChild(linkedChild);
			} else { // Link is to child in some functional group
				IFunctionalGroup functionalGroup = linkedChild.getChildContext().getFunctionalGroup();
				if (functionalGroup == null) {
					return;
				}
				controllerWrapper.getController().setFunctionalGroup(functionalGroup, true, caller);
				controllerWrapper.getGUIController().openViewOfChild(linkedChild);
			}
		}
	}
}
