/**
 * Copyright 2021 NXP
 * Created: Mar 12 2021
 */
package com.nxp.swtools.periphs.gui.view;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.expression.IContext;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplateProperties;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.Description;

/**
 * Entry representation in the {@link AddComponentDialog}.
 * Always contains component. When contains template then this is a template entry.
 * @author Tomas Rudolf - nxf31690
 */
public class AddComponentDialogEntry {
	/** Component which this entry represents */
	private @NonNull ConfigurationComponentTypeId component;
	/** Properties of template if this entry represents component template */
	private @Nullable ComponentTemplateProperties templateProperties = null;
	/** Parent entry if this entry is not from root level */
	private @Nullable AddComponentDialogEntry parent = null;
	/** List of component templates of current component if this is template entry */
	private @NonNull List<AddComponentDialogEntry> templateEntries = Collections.emptyList();

	/**
	 * Constructor - component entry
	 * @param component
	 */
	public AddComponentDialogEntry(@NonNull ConfigurationComponentTypeId component) {
		this.component = component;
	}

	/**
	 * Constructor - template entry
	 * @param parent entry
	 * @param templateProperties properties of the template
	 */
	public AddComponentDialogEntry(@NonNull AddComponentDialogEntry parent, @NonNull ComponentTemplateProperties templateProperties) {
		this(parent.getComponent());
		this.templateProperties = templateProperties;
		this.parent = parent;
	}

	/**
	 * @return component
	 */
	public @NonNull ConfigurationComponentTypeId getComponent() {
		return component;
	}

	/**
	 * @return component template when this is template entry, {@code null} otherwise
	 */
	public @Nullable ComponentTemplateProperties getTemplateProperties() {
		return templateProperties;
	}

	/**
	 * Get not null component template.
	 * Internal usage only!
	 * @return component template
	 */
	private @NonNull ComponentTemplateProperties getTemplatePropertiesNonNull() {
		assert templateProperties!= null : "Template is null. Add check before calling this function"; //$NON-NLS-1$
		return templateProperties;
	}

	/**
	 * @return {@code true} when this is template entry, {@code false} otherwise
	 */
	public boolean isTemplateEntry() {
		return templateProperties != null;
	}

	/**
	 * @return {@code true} when the component is hidden component, {@code false} otherwise
	 */
	public boolean isComponentHidden() {
		return getComponent().getConfigurationComponent().isHidden();
	}

	/**
	 * @return {@code true} when this is component entry and there are templates for this component, {@code false} otherwise
	 */
	public boolean hasTemplates() {
		return !templateEntries.isEmpty();
	}

	/**
	 * @return parent entry if it exists, {@code null} otherwise
	 */
	public @Nullable AddComponentDialogEntry getParent() {
		return parent;
	}

	/**
	 * @return list of all template entries if this is component entry, empty list when no templates are available or this is template entry
	 */
	public @NonNull List<AddComponentDialogEntry> getTemplateEntries() {
		return CollectionsUtils.unmodifiableList(templateEntries);
	}

	/**
	 * Sets template entries
	 * @param entries to be set
	 */
	public void setTemplateEntries(@NonNull List<AddComponentDialogEntry> entries) {
		this.templateEntries = new ArrayList<>(entries);
	}

	/**
	 * Adds one template entry
	 * @param entry to be added
	 * @return @see {@link List#add(Object)}
	 */
	public boolean addTemplateEntry(@NonNull AddComponentDialogEntry entry) {
		if (!(templateEntries instanceof ArrayList)) {
			templateEntries = new ArrayList<>();
		}
		return templateEntries.add(entry);
	}

	/**
	 * @param context to resolve expression
	 * @return name of the template if this is template entry or name of the component
	 */
	public @NonNull String getName(@NonNull IContext context) {
		if (isTemplateEntry()) {
			return getTemplatePropertiesNonNull().getName();
		}
		return getComponent().getConfigurationComponent().getLabel(context);
	}

	/**
	 * @param context to resolve expression
	 * @return description of the template if this is template entry or description of the component
	 */
	public @Nullable String getDescription(@NonNull IContext context) {
		if (isTemplateEntry()) {
			return getTemplatePropertiesNonNull().getDescription();
		}
		Description description = getComponent().getConfigurationComponent().getDescription();
		if (description == null) {
			return null;
		}
		return description.getDescription(context);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return Objects.hash(component, templateProperties, parent);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj == this) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (this.getClass() != obj.getClass()) {
			return false;
		}
		AddComponentDialogEntry other = (AddComponentDialogEntry) obj;
		if (!Objects.equals(this.component, other.component)) {
			return false;
		}
		if (!Objects.equals(this.templateProperties, other.templateProperties)) {
			return false;
		}
		if (!Objects.equals(this.parent, other.parent)) {
			return false;
		}
		return true;
	}
}
