/**
  * Copyright 2022 NXP
  * Created Nov 14, 2022
 */
package com.nxp.swtools.periphs.gui;

import static com.nxp.swtools.common.utils.text.UtilsText.*;

import java.util.Objects;

import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.utils.Messages;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.tooltip.IToolTipable;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;

/**
 * Tooltip formatter for Peripherals tool specific needs
 * @author Tomas Rudolf - nxf31690
 */
public class PeripheralsToolTippableFormatter extends ToolTipableFormatter {

	/**
	 * Get text of tool-tip.
	 * @param tooltipable with info displayed in tool-tip
	 * @return text of tool-tip
	 */
	public static String getToolTipText(IToolTipable tooltipable) {
		StringBuilder result = new StringBuilder();
		// create name
		result.append(createIdentification(tooltipable));
		// create status
		String status = createStatus(tooltipable, null /* no message */, ErrorLevels.LEVEL_SUCCESS /* invalid level (ignored) */);
		// create value
		String value = createValue(tooltipable);
		if (!(isEmpty(status) && isEmpty(value))) {
			// add line break to visual separate
			result.append(XHTML_BR);
			// add content of status and value
			result.append(status);
			result.append(value);
		}
		return trimExcessBreakLine(UtilsText.safeToString(result));
	}

	/**
	 * Create header of the tool-tip, i.e. name, ID and type of the tooltipable.
	 * @param tooltipable with info displayed in tool-tip
	 * @return text of the
	 */
	protected static String createIdentification(IToolTipable tooltipable) {
		StringBuilder result = new StringBuilder();
		// add name
		String name = tooltipable.getUiName();
		if (!isEmpty(name)) {
			result.append(htmlB(name));
		}
		if (shouldShowInternalIds()) {
			String idPart = createTypeAndId(tooltipable);
			if (!isEmpty(idPart)) {
				result.append(SPACE).append(UtilsText.LEFT_BRACKET);
				result.append(idPart);
				result.append(UtilsText.RIGHT_BRACKET);
			}
		}
		String description = tooltipable.getDescription();
		if (result.length() > 0 && (description == null || description.isEmpty())) {
			result.append(XHTML_BR);
		}
		// description
		if (description != null && !description.isEmpty()) {
			if (!(description.startsWith(XHTML_BR) || description.startsWith(XHTML_P))) {
				// Do not add new line in case there is already line break in the description
				result.append(XHTML_BR);
			}
			result.append(description).append(XHTML_BR);
		}
		return UtilsText.safeToString(result);
	}

	/**
	 * Checks whether internal IDs should be shown in the tooltip
	 * @return {@code true} when internal IDs should be shown, {@code false} when not
	 */
	protected static final boolean shouldShowInternalIds() {
		return KEPreferences.shouldInternalIdsBeShown() || SWToolsProperties.isVerificationOn();
	}

	/**
	 * Create value info section, i.e. name of the currently selected value and its description.
	 * @param tooltipable with info displayed in tool-tip
	 * @return text of the value info section
	 */
	protected static String createValue(IToolTipable tooltipable) {
		boolean showValue = false;
		boolean showValueDescription = false;
		if (shouldShowInternalIds()) {
			// Value can be shown if it is not empty
			if (!isEmpty(tooltipable.getValueName())) {
				showValue = true;
			}
			// Value description can be shown if it is not empty and differs from value
			if ((!isEmpty(tooltipable.getValueDescription()) && !Objects.equals(tooltipable.getValueName(), tooltipable.getValueDescription()))) {
				showValueDescription = true;
			}
		} else {
			// Show only value description if it is not empty
			if (!isEmpty(tooltipable.getValueDescription())) {
				showValueDescription = true;
			}
		}
		return createValueInternal(tooltipable, showValue, showValueDescription);
	}

	/**
	 * Creates the Value section string based on control properties
	 * @param tooltipable which provides information
	 * @param showValue {@code true} if internal value should be included, {@code false} otherwise
	 * @param showValueDescription {@code true} if description of value should be included, {@code false} otherwise
	 * @return content of Value section
	 */
	protected static String createValueInternal(IToolTipable tooltipable, boolean showValue, boolean showValueDescription) {
		StringBuilder result = new StringBuilder();
		// Show value section header only if at lest value or it's description is shown
		if (showValue || showValueDescription) {
			result.append(htmlB(Messages.get().ToolTipableFormater_Value)).append(XHTML_BR);
		}
		if (showValue) {
			result.append(tooltipable.getValueName());
		}
		// Show separation when both value and value description will be shown
		if (showValue && showValueDescription) {
			result.append(SPACE).append(EN_DASH).append(SPACE);
		}
		if (showValueDescription) {
			result.append(tooltipable.getValueDescription());
		}
		if (result.length() > 0) {
			result.append(XHTML_BR);
		}
		return UtilsText.safeToString(result);
	}
}