/**
/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * Copyright 2019-2020 NXP
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    chris.gross@us.ibm.com - initial API and implementation
 *    Chuck.Mastrandrea@sas.com - wordwrapping in bug 222280
 *    smcduff@hotmail.com       - wordwrapping in bug 222280
 *    Marty Jones<martybjones@gmail.com> - custom header/footer font in bug 293743
 *    Marek Trmac<marek.trmac@nxp.com> - class DefaultColumnHeaderRenderer re-implemented to display text vertically; Jan/2019
 *******************************************************************************/
package com.nxp.swtools.derivative.rcp;

import org.eclipse.nebula.widgets.grid.GridColumn;
import org.eclipse.nebula.widgets.grid.GridHeaderRenderer;
import org.eclipse.nebula.widgets.grid.internal.SortArrowRenderer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Display;

/**
 * This class is custom copy of org.eclipse.nebula.widgets.grid.internal.DefaultColumnHeaderRenderer that was updated to draw text vertically
 */
public class GridHeaderRendererVertical extends GridHeaderRenderer {

	/** Default max size */
	public static final Point MAX_SIZE = new Point(50, 120);

	/** Margin */
	int leftMargin = 6;

	/** Margin */
	int rightMargin = 6;

	/** Margin */
	int topMargin = 6;

	/** Margin */
	int bottomMargin = 6;

	/** Margin */
	int arrowMargin = 6;

	/** Space between image and text */
	int imageSpacing = 3;

	/** Renderer */
	private SortArrowRenderer arrowRenderer = new SortArrowRenderer();

	/** Maximum size of the cell */
	Point maxSize;

	/** Header color provider */
	IHeaderColumnColorProvider colorProvider;

	/**
	 * Draws an image vertically (rotates plus or minus 90 degrees)
	 * <dl>
	 * <dt><b>Styles: </b></dt>
	 * <dd>UP, DOWN</dd>
	 * </dl>
	 *
	 * @param image
	 *            the image to draw
	 * @param x
	 *            the x coordinate of the top left corner of the drawing
	 *            rectangle
	 * @param y
	 *            the y coordinate of the top left corner of the drawing
	 *            rectangle
	 * @param paintBackground
	 *            set to <code>false</code> to render the background
	 *            transparent. Needed for example to render the background with
	 *            an image or gradient with another painter so the text drawn
	 *            here should have no background.
	 * @param gc
	 *            the GC on which to draw the image
	 * @param style
	 *            the style (SWT.UP or SWT.DOWN)
	 *            <p>
	 *            Note: Only one of the style UP or DOWN may be specified.
	 *            </p>
	 * Reused from org.eclipse.nebula.widgets.nattable.painter.cell.GraphicsUtils
	 */
	public static void drawVerticalImage(Image image, int x, int y,
			boolean paintBackground, GC gc, int style) {
		// Get the current display
		Display display = Display.getCurrent();
		if (display == null) {
			SWT.error(SWT.ERROR_THREAD_INVALID_ACCESS);
		}

		// Use the image's data to create a rotated image's data
		ImageData sd = image.getImageData();
		int width4data = sd.height;
		int height4data = sd.width;
		ImageData dd = new ImageData(width4data, height4data, sd.depth, sd.palette);
		dd.transparentPixel = sd.transparentPixel;

		// set the defined background color to be transparent
		if (!paintBackground) {
			dd.transparentPixel = sd.palette.getPixel(gc.getBackground().getRGB());
		}

		// Determine which way to rotate, depending on up or down
		boolean up = (style & SWT.UP) == SWT.UP;

		// Run through the horizontal pixels
		for (int sx = 0; sx < sd.width; sx++) {
			// Run through the vertical pixels
			for (int sy = 0; sy < sd.height; sy++) {
				// Determine where to move pixel to in destination image data
				int dx = up ? sy : sd.height - sy - 1;
				int dy = up ? sd.width - sx - 1 : sx;
				// Swap the x, y source data to y, x in the destination
				dd.setPixel(dx, dy, sd.getPixel(sx, sy));
			}
		}

		// Create the vertical image
		Image vertical = new Image(display, dd);

		// Draw the vertical image onto the original GC
		gc.drawImage(vertical, x, y);

		// Dispose the vertical image
		vertical.dispose();
	}

	/**
	 * Draws text vertically (rotates plus or minus 90 degrees). Uses the
	 * current font, color, and background.
	 * <dl>
	 * <dt><b>Styles: </b></dt>
	 * <dd>UP, DOWN</dd>
	 * </dl>
	 *
	 * @param string
	 *            the text to draw
	 * @param x
	 *            the x coordinate of the top left corner of the drawing
	 *            rectangle
	 * @param y
	 *            the y coordinate of the top left corner of the drawing
	 *            rectangle
	 * @param underline
	 *            set to <code>true</code> to render the text underlined
	 * @param strikethrough
	 *            set to <code>true</code> to render the text strikethrough
	 * @param paintBackground
	 *            set to <code>false</code> to render the background
	 *            transparent. Needed for example to render the background with
	 *            an image or gradient with another painter so the text drawn
	 *            here should have no background.
	 * @param gc
	 *            the GC on which to draw the text
	 * @param style
	 *            the style (SWT.UP or SWT.DOWN)
	 *            <p>
	 *            Note: Only one of the style UP or DOWN may be specified.
	 *            </p>
	 * Reused from org.eclipse.nebula.widgets.nattable.painter.cell.GraphicsUtils
	 */
	public static void drawVerticalText(String string, int x, int y,
			boolean underline, boolean strikethrough, boolean paintBackground,
			GC gc, int style) {
		// Get the current display
		Display display = Display.getCurrent();
		if (display == null) {
			SWT.error(SWT.ERROR_THREAD_INVALID_ACCESS);
		}

		// Determine string's dimensions
		// FontMetrics fm = gc.getFontMetrics();
		Point pt = gc.textExtent(string.trim());

		if ((pt.x <= 0) || (pt.y <= 0)) {
			return;
		}
		// Create an image the same size as the string
		Image stringImage = new Image(display, pt.x, pt.y);

		// Create a GC so we can draw the image
		GC stringGc = new GC(stringImage);

		// Set attributes from the original GC to the new GC
		stringGc.setAntialias(gc.getAntialias());
		stringGc.setTextAntialias(gc.getTextAntialias());
		stringGc.setForeground(gc.getForeground());
		stringGc.setBackground(gc.getBackground());
		stringGc.setFont(gc.getFont());

		// Fill the image with the specified background color
		// to avoid white spaces if the text does not fill the
		// whole image (e.g. on new lines)
		stringGc.fillRectangle(0, 0, pt.x, pt.y);

		// Draw the text onto the image
		stringGc.drawText(string, 0, 0);

		// draw underline and/or strikethrough
		if (underline || strikethrough) {
			// check and draw underline and strikethrough separately so it is
			// possible to combine both
			if (underline) {
				// y = start y of text + font height
				// - half of the font descent so the underline is between the
				// baseline and the bottom
				int underlineY = pt.y
						- (stringGc.getFontMetrics().getDescent() / 2);
				stringGc.drawLine(0, underlineY, pt.x, underlineY);
			}

			if (strikethrough) {
				// y = start y of text + half of font height + ascent so lower
				// case characters are
				// also strikethrough
				int strikeY = (pt.y / 2)
						+ (stringGc.getFontMetrics().getLeading() / 2);
				stringGc.drawLine(0, strikeY, pt.x, strikeY);
			}
		}

		// Draw the image vertically onto the original GC
		drawVerticalImage(stringImage, x, y, paintBackground, gc, style);

		// Dispose the new GC
		stringGc.dispose();

		// Dispose the image
		stringImage.dispose();
	}

	/**
	 * Constructor
	 * @param maxSize maximal size
	 */
	public GridHeaderRendererVertical(Point maxSize, IHeaderColumnColorProvider colorProvider) {
		this.colorProvider = colorProvider;
		this.maxSize = maxSize;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Point computeSize(GC gc, int wHint, int hHint, Object value) {
		GridColumn column = (GridColumn)value;

		gc.setFont(column.getHeaderFont());

		int x = leftMargin + gc.getFontMetrics().getHeight() + rightMargin;
		int y = topMargin;

		if (column.getImage() != null) {
			x += column.getImage().getBounds().width + imageSpacing;
		}
		y += gc.stringExtent(column.getText()).x + bottomMargin;

		if (column.getImage() != null) {
			y = Math.max(y, topMargin + column.getImage().getBounds().height + bottomMargin);
		}

		x += computeControlSize(column).y;

		return new Point(Integer.min(x, maxSize.x), Integer.min(y, maxSize.y));
	}

	/**
	 * Background color for the selected column
	 * @param column being drawn
	 * @return background color
	 */
	Color getBackgroundColor(GridColumn column) {
		Color result = (colorProvider != null) ? colorProvider.getBackgroundColor(column) : null;
		return (result != null) ? result : getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);
	}

	/**
	 * Background color for the selected column
	 * @param column being drawn
	 * @return background color
	 */
	Color getForegroundColor(GridColumn column) {
		Color result = (colorProvider != null) ? colorProvider.getForegroundColor(column) : null;
		return (result != null) ? result : getDisplay().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void paint(GC gc, Object value) {
		GridColumn column = (GridColumn)value;

		// set the font to be used to display the text.
		gc.setFont(column.getHeaderFont());

		boolean flat = (column.getParent().getCellSelectionEnabled() && !column.getMoveable());

		boolean drawSelected = ((isMouseDown() && isHover()));

		gc.setBackground(getBackgroundColor(column));

		if (flat && isSelected()){
			gc.setBackground(column.getParent().getCellHeaderSelectionBackground());
		}

		Rectangle bounds = getBounds();

		gc.fillRectangle(bounds.x, bounds.y, bounds.width, bounds.height);

		int pushedDrawingOffset = 0;
		if (drawSelected) {
			pushedDrawingOffset = 1;
		}

		int x = leftMargin;

		if (column.getImage() != null) {
			int y = bottomMargin;

			if (column.getHeaderControl() == null ) {
				y = (bounds.y + pushedDrawingOffset + bounds.height) - bottomMargin - column.getImage().getBounds().height;
			}

			gc.drawImage(column.getImage(), bounds.x + x + pushedDrawingOffset, y);
			x += column.getImage().getBounds().width + imageSpacing;
		}

		gc.setForeground(getForegroundColor(column));

		int y = bounds.y + topMargin;

		String text = column.getText();

		text = DefaultColumnGroupHeaderRenderer.getShortStr(gc, text, bounds.height - topMargin - bottomMargin - pushedDrawingOffset).trim();

		Point textSize = gc.textExtent(text);
		Point textPos = new Point((bounds.x + x + pushedDrawingOffset) - 1, bounds.y + pushedDrawingOffset);
		switch (column.getAlignment()) {
			case SWT.RIGHT:
				textPos.y += (topMargin + pushedDrawingOffset);
				break;
			case SWT.CENTER:
				textPos.y += (bounds.height - textSize.x) / 2;
				break;
			default:
				textPos.y += (bounds.height - textSize.x) - topMargin;
		}

		drawVerticalText(text, textPos.x, textPos.y, false, false, false, gc, SWT.UP);

		if (column.getSort() != SWT.NONE) {
			if( column.getHeaderControl() == null ) {
				y = bounds.y + ((bounds.height - arrowRenderer.getBounds().height) / 2) + 1;
			} else {
				y = bounds.y + ((bounds.height - computeControlSize(column).y - arrowRenderer.getBounds().height) / 2) + 1;
			}

			arrowRenderer.setSelected(column.getSort() == SWT.UP);
			if (drawSelected) {
				arrowRenderer.setLocation(((bounds.x + bounds.width) - arrowMargin - arrowRenderer.getBounds().width) + 1, y);
			} else {
				if (column.getHeaderControl() == null) {
					y = bounds.y + ((bounds.height - arrowRenderer.getBounds().height) / 2);
				} else {
					y = bounds.y + ((bounds.height - computeControlSize(column).y - arrowRenderer.getBounds().height) / 2);
				}
				arrowRenderer.setLocation((bounds.x + bounds.width) - arrowMargin - arrowRenderer.getBounds().width, y);
			}
			arrowRenderer.paint(gc, null);
		}

		if (!flat) {

			if (drawSelected) {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
			} else {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_HIGHLIGHT_SHADOW));
			}

			gc.drawLine(bounds.x, bounds.y, (bounds.x + bounds.width) - 1, bounds.y);
			gc.drawLine(bounds.x, bounds.y, bounds.x, (bounds.y + bounds.height) - 1);

			if (!drawSelected) {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_LIGHT_SHADOW));
				gc.drawLine(bounds.x + 1, bounds.y + 1, (bounds.x + bounds.width) - 2, bounds.y + 1);
				gc.drawLine(bounds.x + 1, bounds.y + 1, bounds.x + 1, (bounds.y + bounds.height) - 2);
			}

			if (drawSelected) {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
			} else {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_DARK_SHADOW));
			}
			gc.drawLine((bounds.x + bounds.width) - 1, bounds.y, (bounds.x + bounds.width) - 1, (bounds.y + bounds.height) - 1);
			gc.drawLine(bounds.x, (bounds.y + bounds.height) - 1, (bounds.x + bounds.width) - 1, (bounds.y + bounds.height) - 1);

			if (!drawSelected) {
				gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
				gc.drawLine((bounds.x + bounds.width) - 2, bounds.y + 1,
						(bounds.x + bounds.width) - 2, (bounds.y + bounds.height) - 2);
				gc.drawLine(bounds.x + 1, (bounds.y + bounds.height) - 2,
						(bounds.x + bounds.width) - 2, (bounds.y + bounds.height) - 2);
			}

		} else {
			gc.setForeground(getDisplay().getSystemColor(SWT.COLOR_WIDGET_DARK_SHADOW));

			gc.drawLine((bounds.x + bounds.width) - 1, bounds.y, (bounds.x + bounds.width) - 1, (bounds.y + bounds.height) - 1);
			gc.drawLine(bounds.x, (bounds.y + bounds.height) - 1, (bounds.x + bounds.width) - 1, (bounds.y + bounds.height) - 1);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDisplay(Display display)
	{
		super.setDisplay(display);
		arrowRenderer.setDisplay(display);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean notify(int event, Point point, Object value)
	{
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Rectangle getTextBounds(Object value, boolean preferred)
	{
		GridColumn column = (GridColumn)value;

		int x = leftMargin;

		if (column.getImage() != null)
		{
			x += column.getImage().getBounds().width + imageSpacing;
		}



		GC gc = new GC(column.getParent());
		gc.setFont(column.getParent().getFont());
		int y = topMargin; //getBounds().height - bottomMargin - gc.getFontMetrics().getHeight();

		Rectangle bounds = new Rectangle(x,y,0,0);

		Point p = gc.stringExtent(column.getText());

		bounds.width = p.y;

		if (preferred)
		{
			bounds.height = p.x;
		}
		else
		{
			int width = getBounds().height - y;
			if (column.getSort() == SWT.NONE)
			{
				width -= bottomMargin;
			}
			else
			{
				width -= arrowMargin + arrowRenderer.getSize().x + arrowMargin;
			}
			bounds.height = width;
		}


		gc.dispose();

		return bounds;
	}

	/**
	 * @return the bounds reserved for the control
	 */
	@Override
	protected Rectangle getControlBounds(Object value, boolean preferred) {
		Rectangle bounds = getBounds();
		GridColumn column = (GridColumn) value;
		Point controlSize = computeControlSize(column);

		int y = (getBounds().y + getBounds().height) - bottomMargin - controlSize.y;

		return new Rectangle(bounds.x+3,y,bounds.width-6,controlSize.y);
	}

	/**
	 * Calculate size of edit control
	 * @param column
	 * @return calculated size
	 */
	private static Point computeControlSize(GridColumn column) {
		if( column.getHeaderControl() != null ) {
			return column.getHeaderControl().computeSize(SWT.DEFAULT, SWT.DEFAULT);
		}
		return new Point(0,0);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.nebula.widgets.grid.GridHeaderRenderer#setWordWrap(boolean)
	 */
	@Override
	public void setWordWrap(boolean value) {
		super.setWordWrap(false); //word wrap cannot be enabled
	}
}
