/* The purpose of this program is to compare dejagnu log files */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/stat.h>
#include <glib.h>

typedef struct {
  int n_pass;
  int n_fail;
  int n_unres;
  int n_unsup;
  int n_xpass;
  int n_xfail;
  int n_lines;
  int n_size;
  char **lines;
  GHashTable *testnames;
  GHashTable *results;
} log;

enum {
  R_NONE, R_XPASS, R_XFAIL, R_PASS, R_FAIL, R_UNRES, R_UNSUP
};
char *rname[] = {
  "NONE ", "XPASS", "XFAIL", "PASS ", "FAIL ", "UNRES", "UNSUP"
};

char *files[32];
FILE *fin[32];
log fdata[32];
int n_files;

int f_debug;
int f_comp;
int f_filter;
int f_strict;
int f_dump;

int n_matches;
int n_changes;
int n_newres;
int n_strict;

void
usage (void)
{
  fprintf (stdout, "Usage: [<options>] <list of files>\n");
  fprintf (stdout, "Where <options> is one of:\n");
  fprintf (stdout, "    -h        Print this info\n");
  fprintf (stdout, "    -debug    Verbose output\n");
  fprintf (stdout, "    -cmp      Compare two files\n");
  fprintf (stdout, "    -strict   Compare and don't report UNRESOLVED/[PASS|FAIL] differences\n");
  fprintf (stdout, "              ... and don't report PASS/UNSUPPORTED present in one side only either\n");
  fprintf (stdout, "    -f        Disable filtering of strange lines\n");
  fprintf (stdout, "    -dump     Filter log file and dump to stdout\n");
  fprintf (stdout, "Version 1.3\n");
}

/* Copy a string to end into a variable size array of strings */
int
copyline (int i, char *s)
{
  int nl = fdata[i].n_lines;
  int sz = fdata[i].n_size;

  if (s == NULL)
    return 0;

  if (nl == sz) {
    /* Increase allocation */
    if (sz == 0) {
      fdata[i].n_size = 4096;
      fdata[i].lines = (char **) malloc (4096 * sizeof (char *));
    } else {
      fdata[i].n_size = sz * 2;
      fdata[i].lines = (char **) realloc (fdata[i].lines, sz * 2 * sizeof (char *));
    }
  }
  fdata[i].lines[nl++] = strdup (s);
  fdata[i].n_lines = nl;

  return 1;
}

/* Return modified line, or NULL if to be removed */
char *
filterline (char *p)
{
  char *line, *l[1024];
  char cp[4096];
  int nf;
  char *s1, *s2;
  int i, f;
  int modified;

  if (f_filter == 0)
    return p;

  /* Break line into tokens */
  line = strdup (p);
  nf = 0;
  l[nf] = strtok (line, " ");
  while (l[nf] != NULL) {
    nf++;
    l[nf] = strtok (NULL, " ");
  }

  /* Remove lines of the form: */
  if (nf == 7 && strcmp (l[2], "is") == 0 && strcmp (l[5], "expected") == 0
      && ((strcmp (l[3], "optimized") == 0 && strcmp (l[4], "away,") == 0)
	  || (strcmp (l[3], "not") == 0 && strcmp (l[4], "computable,") == 0))) {
    if (f_debug) fprintf (stdout, "removing: %s", p);
    return NULL;
  }
  if (nf == 6 && (strcmp (l[1], "e") == 0 || strcmp (l[1], "b") == 0
		  || strcmp (l[1], "c") == 0 || strcmp (l[1], "w") == 0 || strcmp (l[1], "t") == 0
		  || strcmp (l[1], "o") == 0
		  || strcmp (l[1], "v") == 0 || strcmp (l[1], "ret") == 0)
      && strcmp (l[2], "is") == 0
      && strcmp (l[4], "not") == 0) {
    if (f_debug) fprintf (stdout, "removing: %s", p);
    return NULL;
  }
  if (nf == 4 && strcmp (l[2], "is") == 0
      && (strcmp (l[1], "c") == 0 || strcmp (l[1], "v") == 0 || strcmp (l[1], "e") == 0
	  || strcmp (l[1], "w") == 0 || strcmp (l[1], "o") == 0 || strcmp (l[1], "n") == 0
	  || strcmp (l[1], "i") == 0 || strcmp (l[1], "j") == 0 || strcmp (l[1], "k") == 0
	  || strcmp (l[1], "t") == 0 || strcmp (l[1], "first") == 0 || strcmp (l[1], "last") == 0
	  || strcmp (l[1], "begin") == 0 || strcmp (l[1], "end") == 0 || strcmp (l[1], "ret") == 0
	  || strcmp (l[1], "argc") == 0 || strcmp (l[1], "0x40") == 0 || strcmp (l[1], "b") == 0
	  || strcmp (l[1], "ab") == 0 || strcmp (l[1], "ac") == 0 || strcmp (l[1], "msg") == 0
	  || strcmp (l[1], "&i") == 0 || strcmp (l[1], "-1") == 0)) {
    if (f_debug) fprintf (stdout, "removing: %s", p);
    return NULL;
  }
  /* If line has a filename, remove dir path */
  modified = 0;
  for (f = 1; f <= 4; f+=3) {
    if (nf > f && strncmp (l[f], "/local", 6) == 0) {
      /* remove the path of the filename */
      s1 = l[f] + strlen (l[f]) - 1;
      while (*s1 != '/')
	s1--;
      l[f] = s1 + 1;
      modified = 1;
    }
  }
  /* If lines has a address, change it to ........ */
  for (f = 4; f < nf-6; f++) {
    if (strcmp(l[f], "load") == 0 && strcmp(l[f+1], "of") == 0 && strcmp(l[f+2], "address") == 0) {
      s1 = l[f+3] + 2;
      while (*s1 != '\0')
	*s1++ = '.';
      modified = 1;
    }
  }
  for (f = 4; f < nf-6; f++) {
    if (strcmp(l[f], "virtual") == 0 && strcmp(l[f+1], "base") == 0 && strcmp(l[f+2], "of") == 0 && strcmp(l[f+3], "address") == 0) {
      s1 = l[f+4] + 2;
      while (*s1 != '\0')
	*s1++ = '.';
      modified = 1;
    }
  }
  for (f = 4; f < nf-4; f++) {
    if (strcmp(l[f], "bytes") == 0 && strcmp(l[f+1], "at") == 0 && strcmp(l[f+2], "address") == 0) {
      s1 = l[f+3] + 2;
      while (*s1 != '\0')
	*s1++ = '.';
      modified = 1;
    }
  }
  for (f = 2; f < nf-6; f++) {
    if (strcmp(l[f], "output") == 0 && strcmp(l[f+1], "pattern") == 0 && strcmp(l[f+2], "test,") == 0
	&& strcmp(l[f+3], "is") == 0 && strncmp(l[f+4], "==", 2) == 0) {
      s1 = l[f+4] + 2;
      *s1 = '\0';
      modified = 1;
    }
  }
      
  if (modified != 0) {
    /* return a new string built from the rest of it */
    s1 = cp;
    for (i = 0; i < nf; i++) {
      s2 = l[i];
      while (*s2 != '\0')
	*s1++ = *s2++;
      *s1++ = ' ';
    }
    s1--;
    *s1 = '\0';
    return strdup (cp);
  }
  
  return p;
}

void
collect_testname (int i, char *s, long res)
{
  char cp[4096];
  char *tname, *tresult;
  int l;

  if (s == NULL)
    return;
  strcpy (cp, s);
  tresult = strtok (cp, " ");
  tresult += strlen(tresult) + 1;
  l = strlen(tresult);
  if (tresult[l - 1] == '\n') {
    tresult[l - 1] = '\0';
  }
  g_hash_table_insert (fdata[i].results, strdup(tresult), (void *)res);

  tname = strtok (NULL, " ");
  if (tname == NULL) {
    printf ("This line should have at least 2 fields: %s\n", s);
    exit (0);
  }
  l = strlen(tname);
  if (tname[l - 1] == '\n') {
    tname[l - 1] = '\0';
  }
  g_hash_table_insert (fdata[i].testnames, strdup(tname), (void *)res);
}

/* Load an entire file on memory, save only the lines with results */
void
load_log (int i)
{
  char line[4096];
  char *filtered;
  char *s;
  char *p;

  fdata[i].testnames = g_hash_table_new (g_str_hash, g_str_equal);
  fdata[i].results = g_hash_table_new (g_str_hash, g_str_equal);

  while ((s=fgets(line, 4096, fin[i])) != NULL) {
    if ((p=strstr (line , "XPASS: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_XPASS);
      if (copyline (i, filtered) != 0) fdata[i].n_xpass++;

    } else if ((p=strstr (line , "XFAIL: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_XFAIL);
      if (copyline (i, filtered) != 0) fdata[i].n_xfail++;

    } else    if ((p=strstr(line, "PASS: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_PASS);
      if (copyline (i, filtered) != 0) fdata[i].n_pass++;

    } else if ((p=strstr (line , "FAIL: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_FAIL);
      if (copyline (i, filtered) != 0) fdata[i].n_fail++;

    } else if ((p=strstr (line , "UNRESOLVED: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_UNRES);
      if (copyline (i, filtered) != 0) fdata[i].n_unres++;

    } else if ((p=strstr (line , "UNSUPPORTED: ")) != NULL) {
      filtered = filterline(p);
      collect_testname(i, filtered, R_UNSUP);
      if (copyline (i, filtered) != 0) fdata[i].n_unsup++;
    }
  }

  if (f_debug) {
    printf ("Loading %d testnames:%d results:%d\n", i,
	    g_hash_table_size(fdata[i].testnames),
	    g_hash_table_size(fdata[i].results));
  }
}

void
unload_log (int i)
{
  free (fdata[i].lines);
  memset (fdata + i, 0, sizeof (log));
}

void
sort_log (int i)
{
  log *l;
  char **lines;
  int p1, p2;
  char *s1, *s2;

  l = fdata + i;
  /* Allocate a sorted array */
  lines = (char **) malloc (l->n_size * sizeof (char *));
  p1 = 0;

  /* List all PASS first */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "PASS", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  /* List all FAIL next */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "FAIL", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  /* List all UNRESOLVED next */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "UNRE", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  /* List all UNSUPPORTED next */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "UNSU", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  /* List all XPASS next */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "XFAI", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  /* List all XFAIL next */
  p2 = 0;
  while (p2 != l->n_lines) {
    if (strncmp (l->lines[p2], "XFAI", 4) == 0)
      lines[p1++] = l->lines[p2];
    p2++;
  }
  l->lines = lines;
}

void
report_count (int i)
{
  int l;
  char *p, *q;

  /* We want to tabulate results at column 40
     If file names are longer than this, we truncate,
     otherwise we fill the space with tabs */
  
  p = files[i];
  q = p + strlen (p);
  while (*(q) != '/' && q != p)
    q--;

  l = strlen (q);
  if (l > 40) {
    q += (l - 40);
    l = 40;
  }

  fprintf (stdout, "%s", q);
  while (l < 40) {
    fprintf (stdout, "\t");
    l += 8;
  }
  fprintf (stdout, "PASS %6d     FAIL %4d     UNRES%4d     ",
	   fdata[i].n_pass, fdata[i].n_fail, fdata[i].n_unres);
  fprintf (stdout, "UNSUP%4d     XPASS%4d     XFAIL%4d\n",
	   fdata[i].n_unsup, fdata[i].n_xpass, fdata[i].n_xfail);
}

void
dump_testname (gpointer key, gpointer value, gpointer data)
{
  printf ("Testname %s\n", key);
}

void
dump_log (int i)
{
  log *l;
  int j;

  l = fdata + i;
  for (j = 0; j < l->n_lines; j++) {
    fprintf (stdout, "%s", l->lines[j]);
  }
  g_hash_table_foreach (fdata[i].testnames, (GHFunc)dump_testname, NULL);
  printf ("Number of individual tests: %d\n", g_hash_table_size(fdata[i].testnames));
}

void
check_hash (gpointer key, gpointer value, gpointer data)
{
  GHashTable *t2;
  long res1;
  long res2;

  res1 = (long) value;
  t2 = (GHashTable *)data;
  
  res2 = (long) g_hash_table_lookup (t2, key);
  if (res2 == 0) {
    if (f_strict == 0 || (res1 != R_PASS && res1 != R_UNSUP)) {
      fprintf(stdout, "-%s %s\n\n", rname[res1], key);
      n_strict++;
    }
    n_changes++;
  } else if (res1 != res2) {
    if (f_strict == 0
	|| !((res1 == R_UNRES && (res2 == R_PASS || res2 == R_FAIL))
	     || (res2 == R_UNRES && (res1 == R_PASS || res1 == R_FAIL)))) {
      fprintf(stdout, "-%s %s\n", rname[res1], key);
      fprintf(stdout, "+%s %s\n\n", rname[res2], key);
      n_strict++;
    }
    n_changes++;
  } else {
    n_matches++;
  }
}

void
new_tests (gpointer key, gpointer value, gpointer data)
{
  GHashTable *t1;
  long res1;
  long res2;

  res2 = (long) value;
  t1 = (GHashTable *)data;
  
  res1 = (long) g_hash_table_lookup (t1, key);
  if (res1 == 0) {
    if (f_strict == 0 || (res2 != R_PASS && res2 != R_UNSUP)) {
      fprintf(stdout, "+%s %s\n\n", rname[res2], key);
    }
    n_newres++;
  }
}

void
compare_log (int l1, int l2)
{
  GHashTable *t1, *t2;

  t1 = fdata[l1].results;
  t2 = fdata[l2].results;

  n_matches = 0;
  n_changes = 0;
  n_newres = 0;
  n_strict = 0;
  g_hash_table_foreach (t1, check_hash, t2);
  g_hash_table_foreach (t2, new_tests, t1);
  fprintf (stdout, "    Total matches:      %d\n", n_matches);
  fprintf (stdout, "    Total changes:      %d\n", n_changes);
  fprintf (stdout, "    Total non reported: %d\n", n_changes - n_strict);
  fprintf (stdout, "    Total new results:  %d\n", n_newres);
}

char *
get_fname (char *dir, char *fn)
{
  struct dirent **namelist;
  int n;
  int i;

  n = scandir (dir, &namelist, 0, alphasort);

  for (i = 0; i < n; i++) {
    if (fnmatch (fn, namelist[i]->d_name, FNM_PERIOD) == 0)
      return namelist[i]->d_name;
  }
  return NULL;
}

void
compare_all (char *s)
{
  char *p0, *p1;
  char fn0[1024];
  char fn1[1024];

  p0 = get_fname (files[0], s);
  p1 = get_fname (files[1], s);
  sprintf (fn0, "%s/%s", files[0], p0);
  sprintf (fn1, "%s/%s", files[1], p1);
  fin[2] = fopen (fn0, "r");
  fin[3] = fopen (fn1, "r");
  if ((fin[2] == NULL && fin[3] != NULL) || (fin[2] != NULL && fin[3] == NULL)) {
    fprintf (stdout, "   Only one of directories has %s\n", s);
    if (fin[2] != NULL) fclose (fin[2]);
    if (fin[3] != NULL) fclose (fin[3]);
    return;
  }
  if ((fin[2] == NULL && fin[3] == NULL))
    return;
  fprintf (stdout, "\nComparing logs: %s\n", s);
  fprintf (stdout, "==============================\n");
  load_log (2);
  load_log (3);
  compare_log (2, 3);
  fclose (fin[2]);
  fclose (fin[3]);
  unload_log (2);
  unload_log (3);
}

int
is_dir (char *fn)
{
  struct stat fs;
  int res;

  res = stat (fn, &fs);
  if (res == 0 && S_ISDIR (fs.st_mode))
    return 1;
  return 0;
}

void
load_dump_or_count (int first)
{
  int i;

  /* Process files */
  for (i = first; i < n_files; i++) {
    load_log (i);
    sort_log (i);
  }

  /* Process dump or count */
  if (f_dump != 0) {
    for (i = first; i < n_files; i++)
      dump_log (i);
  } else {
    for (i = first; i < n_files; i++)
      report_count (i);
  }
}

void
load_filelist (void)
{
  FILE *f;
  char line[1024];

  f = fopen ("loglist", "r");
  while (fgets(line, 1024, f) != NULL) {
    line[strlen(line)-1] = '\0';
    files[n_files] = strdup (line);
    n_files++;
  }
}

void
check_inputfiles (int first)
{
  int i;

  /* Check that we can open all input files */
  for (i = first; i < n_files; i++) {
    if (i == 32) {
      fprintf (stdout, "Maximum number of files exceeded\n");
      exit (0);
    }      
    fin[i] = fopen (files[i], "r");
    if (fin[i] == NULL) {
      fprintf (stdout, "Cannot open file:\n%s\n", files[i]);
      exit (0);
    }
  }
}

int
main (int argc, char **argv)
{
  int i;
  char buffer[1024];

  n_files = 0;
  f_debug = 0;
  f_comp = 0;
  f_filter = 1;
  f_strict = 0;
  f_dump = 0;

  /* Scan command line arguments */
  if (argc <= 1) {
    usage ();
    exit (0);
  }

  for (i = 1; i < argc; i++) {
    if (strcmp (argv[i], "-h") == 0) {
      usage ();
      exit (0);
    } else if (strcmp (argv[i], "-debug") == 0) {
      f_debug = 1;
    } else if (strcmp (argv[i], "-cmp") == 0) {
      f_comp = 1;
    } else if (strcmp (argv[i], "-strict") == 0) {
      f_comp = 1;
      f_strict = 1;
    } else if (strcmp (argv[i], "-f") == 0) {
      f_filter = 0;
    } else if (strcmp (argv[i], "-dump") == 0) {
      f_dump = 1;
    } else {
      files[n_files] = argv[i];
      n_files++;
    }
  }

  if (n_files <= 0) {
    fprintf (stdout, "At least one file must be given\n");
    exit (0);
  }

  /* Show what we got on input */
  if (f_debug != 0) {
    fprintf (stdout, "Number of input files: %d\n", n_files);
    for (i = 0; i < n_files; i++)
      fprintf (stdout, "log file: %s\n", files[i]);
    fprintf (stdout, "Operations: ");
    if (f_comp != 0) fprintf (stdout, "compare ");
    if (f_filter != 0) fprintf (stdout, "filter ");
    fprintf (stdout, "\n");
  }

  check_inputfiles (0);

  if (n_files == 1) {
    /* Exactly one argument */
    if (is_dir (files[0])) {
      sprintf (buffer, "find %s -path '*/obj_gcc/gcc/testsuite/*/*.log' -or -path '*/obj_gcc/*/*/testsuite/*.log' > loglist",
	       files[0]);
      system (buffer);
      load_filelist ();
      system ("rm -f loglist");
      check_inputfiles (1);
      load_dump_or_count (1);
    } else {
      load_dump_or_count (0);
    }
  } else if (n_files == 2) {
    /* Exaclty two arguments */
    if (is_dir (files[0])) {
      if (! is_dir (files[1])) {
	fprintf (stdout, "Either specify two directories or two files\n  x1=%s\n  x2=%s\n",
		 files[0], files[1]);
	exit (0);
      }
      /* Two directories */
      compare_all ("*gcc.log");
      compare_all ("*gfortran.log");
      compare_all ("*g++.log");
      compare_all ("*objc.log");
      compare_all ("*libffi.log");
      compare_all ("*libgomp.log");
      compare_all ("*libjava.log");
      compare_all ("*libmudflap.log");
      compare_all ("*libstdc++.log");

      compare_all ("*910.gcc.log");
      compare_all ("*910.gfortran.log");
      compare_all ("*910.g++.log");

      compare_all ("*915.gcc.log");
      compare_all ("*915.gfortran.log");
      compare_all ("*915.g++.log");
    } else {
      if (is_dir (files[1])) {
	fprintf (stdout, "Either specify two directories or two files\n  x1=%s\n  x2=%s\n",
		 files[0], files[1]);
	exit (0);
      }
      /* Two regular files */
      if (f_comp != 0) {
	for (i = 0; i < n_files; i++) {
	  load_log (i);
	  sort_log (i);
	}
	compare_log (0, 1);
      } else {
	load_dump_or_count (0);
      }
    }
  } else {
    load_dump_or_count (0);
  }
}
