#include "MAC57D54H.h"
#include "mc.h"

static void clock_init(void)
{
#ifdef ATE
    /* Set XOSC for bypass mode */
    FXOSC.CTL.R |= 0x80000000;
#endif
    /* turn on XOSC */
    mode_clock_enable(MC_DRUN, MC_FXOSC);
    enter_mode(MC_DRUN);
    while (mode_transition_ongoing()) {}

    /* 320 MHz PLL0 from 40 MHz XOSC */
    auxclk_src_set(0, MC_FXOSC);
    pll_cfg(MC_PLL0, 2, 64, 0);

    /* turn on PLL0 */
    mode_clock_enable(MC_DRUN, MC_PLL0);
    enter_mode(MC_DRUN);
    while (mode_transition_ongoing()) {}

    /* DDR/SDR clock = 320 Mhz */
    sysclk_div_set(0, 1);
    sysclk_div_enable(0);
    /* CA5 clock = 320 Mhz */
    sysclk_div_set(1, 1);
    sysclk_div_enable(1);
    /* CM4/Platform = 160 Mhz */
    sysclk_div_set(2, 3);
    sysclk_div_enable(2);
    /* Peripherals = 80 Mhz */
    sysclk_div_set(3, 7);
    sysclk_div_enable(3);
    /* IOP clock = 80 Mhz */
    sysclk_div_set(4, 7);
    sysclk_div_enable(4);
    /* 2X platform clock */
    sysclk_div_set(5, 1);
    sysclk_div_enable(5);

    /* set PLL0 to sysclk */
    mode_sysclk_src_set(MC_DRUN, MC_PLL0);
    enter_mode(MC_DRUN);
    while (mode_transition_ongoing()) {}

    /* GPUx2 = 320MHz from PLL0 */
    auxclk_src_set(1, MC_PLL0);
    auxclk_div_set(1, 0, 1);
    auxclk_div_enable(1, 0);

    /* Enable all peripherals in DRUN using percfg 0 */
    run_percfg_mode_enable(0, MC_DRUN);
    enter_mode(MC_DRUN);
    while (mode_transition_ongoing()) {}
}

static void memory_burst_copy(uint32_t dst, uint32_t src, uint32_t num)
{
    if (num < 32) return;

    DMA_0.TCD[0].SADDR.R         = src;
    DMA_0.TCD[0].ATTR.R          = 0x0505;
    DMA_0.TCD[0].SOFF.R          = 0x0020;
    DMA_0.TCD[0].NBYTES.MLNO.R   = num;
    DMA_0.TCD[0].SLAST.R         = 0;
    DMA_0.TCD[0].DADDR.R         = dst;
    DMA_0.TCD[0].CITER.ELINKNO.R = 1;
    DMA_0.TCD[0].DOFF.R          = 32;
    DMA_0.TCD[0].DLASTSGA.R      = 0;
    DMA_0.TCD[0].BITER.ELINKNO.R = 1;
    DMA_0.TCD[0].CSR.R           = 1;

    while (DMA_0.TCD[0].CSR.B.START == 1) {}
    while (DMA_0.TCD[0].CSR.B.ACTIVE == 1) {}
}

static void memory_burst_clear(uint32_t addr, uint32_t num)
{
    int i;

    if (num < 64) return;

    memory_burst_copy(addr, 0x3f000000, 32);

    for (i = 0; i < 8; i++) {
        *((uint32_t *)addr + i) = 0;
    }

    DMA_0.TCD[0].SADDR.R         = addr;
    DMA_0.TCD[0].ATTR.R          = 0x0505;
    DMA_0.TCD[0].SOFF.R          = 0x0020;
    DMA_0.TCD[0].NBYTES.MLNO.R   = num - 32;
    DMA_0.TCD[0].SLAST.R         = 0;
    DMA_0.TCD[0].DADDR.R         = addr + 32;
    DMA_0.TCD[0].CITER.ELINKNO.R = 1;
    DMA_0.TCD[0].DOFF.R          = 32;
    DMA_0.TCD[0].DLASTSGA.R      = 0;
    DMA_0.TCD[0].BITER.ELINKNO.R = 1;
    DMA_0.TCD[0].CSR.R           = 1;

    while (DMA_0.TCD[0].CSR.B.START == 1) {}
    while (DMA_0.TCD[0].CSR.B.ACTIVE == 1) {}
}

static void memory_init_m0(void)
{
    /* linker defined symbols */
    extern uint32_t _cm0p_raw_s, _cm0p_raw_e, _cm0p_start_addr;

    /* clear IOP RAM */
    memory_burst_clear(0x3E400000, (32 * 1024));

    /* copy CM0+ binary to the correct place in memory */
    memory_burst_copy((uint32_t)&_cm0p_start_addr,
                      (uint32_t)&_cm0p_raw_s,
                      (uint32_t)&_cm0p_raw_e - (uint32_t)&_cm0p_raw_s);

}

static void memory_init_a5(void)
{
    /* linker defined symbols */
    extern uint32_t _ca5_raw_s, _ca5_raw_e, _ca5_start_addr;

       /* clear SRAM: ECC SRAM initialization  */
    memory_burst_clear(0x3EF00000, (1024 * 1024)); 
    
    /* Copy CA5 only if it shall be located in RAM */
    if ((int)&_ca5_start_addr == 0x3EF00000)
    {
        /* copy CA5 binary to the correct place in memory */
        memory_burst_copy((uint32_t)&_ca5_start_addr,
                          (uint32_t)&_ca5_raw_s,
                          (uint32_t)&_ca5_raw_e - (uint32_t)&_ca5_raw_s);
    }
}

static void core_init(void)
{
    /* linker defined symbols */
#if defined (CORE_M0)
    extern uint32_t _cm0p_start_addr;
#endif
#if defined (CORE_A5)
    extern uint32_t _ca5_start_addr;
#endif
    /* enable all cores in all modes */
#if defined (CORE_A5)
    MC_ME_CCTL1 = 0x00FF;
#endif

#if defined (CORE_M0)
    MC_ME_CCTL2 = 0x20FF;
#endif

    
    /* set boot address for all cores */
#if defined (CORE_A5)
    core_boot_addr_set(MC_CA5, (uint32_t)&_ca5_start_addr);
#endif
 
#if defined (CORE_M0)
    core_boot_addr_set(MC_CM0P, (uint32_t)&_cm0p_start_addr);
#endif
    /* reset all cores on next mode change */
#if defined (CORE_A5)
    core_rmc_enable(MC_CA5);
#endif
 
#if defined (CORE_M0)
    core_rmc_enable(MC_CM0P);
#endif

    enter_mode(MC_DRUN);
    
    while (mode_transition_ongoing()) {}
}

void init_load_cores ()
{
    clock_init();
#if defined (CORE_M0)
    memory_init_m0();
#endif
#if defined (CORE_A5)
    memory_init_a5();
#endif
    core_init();
    return;
}
