#ifndef MC_H
#define MC_H

enum mode {
    MC_RESET    = 0,
    MC_SAFE     = 2,
    MC_DRUN     = 3,
    MC_RUN0     = 4,
    MC_RUN1     = 5,
    MC_RUN2     = 6,
    MC_RUN3     = 7,
    MC_STOP0    = 10,
    MC_STANDBY0 = 13,
};

enum core {
    MC_CM4,
    MC_CA5,
    MC_CM0P,
};

enum clock {
    MC_FIRC  = 0,
    MC_FXOSC = 1,
    MC_SIRC  = 2,
    MC_PLL0  = 4,
    MC_PLL1  = 5,
    MC_PLL2  = 6,
    MC_PLL3  = 7,
};

/*
 * Enable a particular mode
 */
void mode_enable(enum mode mode);

/*
 * Disable a particular mode
 */
void mode_disable(enum mode mode);

/* 
 * Enable a clock for a particular mode 
 */
void mode_clock_enable(enum mode mode, enum clock clock);

/*
 * Disable a clock for a particular mode
 */
void mode_clock_disable(enum mode mode, enum clock clock);

/*
 * Get the configuration status of a clock in a particular mode.
 *
 * Returns 1 if the clock is enabled or 0 if the clock is disabled.
 */
int mode_clock_is_enabled(enum mode mode, enum clock clock);

/* 
 * Set the system clock for a mode
 */
void mode_sysclk_src_set(enum mode mode, enum clock clock);

/*
 * Get the system clock for a mode
 */
enum clock mode_sysclk_src_get(enum mode mode);

/* 
 * Enter a mode
 */
void enter_mode(enum mode mode);

/*
 * Get the current mode
 */
enum mode current_mode(void);

/*
 * See if a mode transition is in progress. Returns 1 if a transition is in
 * progress, 0 otherwise.
 */
int mode_transition_ongoing(void);

/* 
 * Enable a mode in a particular run peripheral config
 */
void run_percfg_mode_enable(int pc, enum mode mode);

/*
 * Disable a mode in a particular run peripheral config
 */
void run_percfg_mode_disable(int pc, enum mode mode);

/* 
 * Enable a mode in a particular low power peripheral config
 */
void lp_percfg_mode_enable(int pc, enum mode mode);

/* 
 * Disable a mode in a particular low power peripheral config
 */
void lp_percfg_mode_disable(int pc, enum mode mode);

/*
 * Get enabled/disabled status of a mode in a run peripheral config
 *
 * Returns 1 if the mode is enabled; 0 otherwise.
 */
int run_percfg_mode_is_enabled(int pc, enum mode mode);

/*
 * Get enabled/disabled status of a mode in a low-power peripheral config
 *
 * Returns 1 if the mode is enabled; 0 otherwise.
 */
int lp_percfg_mode_is_enabled(int pc, enum mode mode);

/* 
 * Set the run peripheral config for a particular peripheral control register
 */
void perctl_run_percfg_set(int perctl, int pc);

/* 
 * Set the low-power peripheral config for a particular peripheral control
 * register
 */
void perctl_lp_percfg_set(int perctl, int pc);

/* 
 * Get the run peripheral config for a particular peripheral control register
 */
int perctl_run_percfg_get(int perctl);

/*
 * Get the low-power peripheral config for a particular peripheral control
 * register
 */
int perctl_lp_percfg_get(int perctl);

/* 
 * Get core status. Returns 1 if core is running, 0 otherwise
 */
int core_running(enum core core);

/* 
 * Enable core for a particular mode 
 */
void core_mode_enable(enum core core, enum mode mode);

/* 
 * Enable core for a particular mode 
 */
void core_mode_disable(enum core core, enum mode mode);

/*
 * Get the on/off status for a core in a particular mode. Returns 1 if the core
 * is enabled for that mode, 0 otherwise.
 */
int core_mode_is_enabled(enum core core, enum mode mode);

/* 
 * Set the boot address for a core 
 */
void core_boot_addr_set(enum core core, unsigned int addr);

/*
 * Get the boot address for a core
 */
unsigned int core_boot_addr_get(enum core core);

/* 
 * Set/clear RMC bit (reset core on next mode change)
 */
void core_rmc_enable(enum core core);

/*
 * Get the status of the RMC bit (reset core on next mode change)
 */
int core_rmc_is_enabled(enum core core);

/* 
 * Set the system clock divider
 */
void sysclk_div_set(int divider, int divn);

/*
 * Get the system clock divider
 */
int sysclk_div_get(int divider);

/*
 * Enable the system clock divider
 */
void sysclk_div_enable(int divider);

/*
 * Disable the system clock divider
 */
void sysclk_div_disable(int divider);
 
/*
 * Check if the system clock divider is enabled.
 *
 * Returns 1 if the divider is enabled; 0 otherwise.
 */
int sysclk_div_is_enabled(int divider);

/*
 * Get the current system clock
 */
enum clock current_sysclk(void);

/* 
 * Set the auxclk source 
 */
void auxclk_src_set(int auxclk, enum clock src);

/*
 * Get the auxclk source
 */
enum clock auxclk_src_get(int auxclk);

/* 
 * Set auxclk divider value
 */
void auxclk_div_set(int auxclk, int divider, int divn);

/* 
 * Get auxclk divider value 
 */
int auxclk_div_get(int auxclk, int divider);

/*
 * Enable the auxiliary clock divider
 */
void auxclk_div_enable(int auxclk, int divider);

/*
 * Disable the auxiliary clock divider
 */
void auxclk_div_disable(int auxclk, int divider);

/*
 * Check if the auxclk divider is enabled
 *
 * Returns 1 if auxclk dvider is on; 0 otherwise.
 */
int auxclk_div_is_enabled(int auxclk, int divider);

/* 
 * update clock dividers 
 */
void update_clock_div(void);

/* 
 * Check if clock dividers are updating. Returns 1 if clocks are updating, 0
 * otherwise.
 */
int clock_div_updating(void);

/* 
 * Configure dividers for pll. Only supports non-FM configuration.
 */
void pll_cfg(enum clock pll, int prediv, int mfd, int rfdphi);

#endif
