/*
 * File:		arm_cm4.c
 * Purpose:		Generic high-level routines for ARM Cortex M4 processors
 *
 * Notes:
 */

#include "arm_cm4.h"
#include "MAC57D54H.h"

/***********************************************************************/
/*
 * Change the value of the vector table offset register to the specified value.
 *
 * Parameters:
 * vtor     new value to write to the VTOR
 */

void write_vtor (uint32_t vtor)
{
  /* Write the VTOR with the new value */
  CM4.VTOR.R = vtor;
}
/***********************************************************************/
/*
 * Initialize the NVIC to enable the specified IRQ.
 * 
 * NOTE: The function only initializes the NVIC to enable a single IRQ. 
 * Interrupts will also need to be enabled in the ARM core. This can be 
 * done using the EnableInterrupts macro.
 *
 * Parameters:
 * irq    irq number to be enabled (the irq number NOT the vector number)
 */

void enable_irq (uint32_t irq)
{
  uint32_t div;
  
  /* Make sure that the IRQ is an allowable number. Right now up to 111 is 
   * used.
   */
  if (irq > 111) return;
  
  /* Determine which of the NVICISERs corresponds to the irq */
  div = irq/32;
  
  switch (div)
  {
    case 0x0:
      CM4.NVIC_ICPR[0].R = 1 << (irq%32);
      CM4.NVIC_ISER[0].R = 1 << (irq%32);
      break;
    case 0x1:
      CM4.NVIC_ICPR[1].R = 1 << (irq%32);
      CM4.NVIC_ISER[1].R = 1 << (irq%32);
      break;
    case 0x2:
      CM4.NVIC_ICPR[2].R = 1 << (irq%32);
      CM4.NVIC_ISER[2].R = 1 << (irq%32);
      break;
    case 0x3:
      CM4.NVIC_ICPR[3].R = 1 << (irq%32);
      CM4.NVIC_ISER[3].R = 1 << (irq%32);
      break;
  }              
}
/***********************************************************************/
/*
 * Initialize the NVIC to disable the specified IRQ.
 * 
 * NOTE: The function only initializes the NVIC to disable a single IRQ. 
 * If you want to disable all interrupts, then use the DisableInterrupts
 * macro instead. 
 *
 * Parameters:
 * irq    irq number to be disabled (the irq number NOT the vector number)
 */

void disable_irq (uint32_t irq)
{
  uint32_t div;
  
  /* Make sure that the IRQ is an allowable number. Right now up to 111 is 
   * used.
   */
  if (irq > 111) return;
  
  /* Determine which of the NVICICERs corresponds to the irq */
  div = irq/32;
  
  switch (div)
  {
    case 0x0:
      CM4.NVIC_ICER[0].R = 1 << (irq%32);
      break;
    case 0x1:
      CM4.NVIC_ICER[1].R = 1 << (irq%32);
      break;
    case 0x2:
      CM4.NVIC_ICER[2].R = 1 << (irq%32);
      break;
    case 0x3:
      CM4.NVIC_ICER[3].R = 1 << (irq%32);
      break;              
  }              
}
/***********************************************************************/
/*
 * Initialize the NVIC to set specified IRQ priority.
 * 
 * NOTE: The function only initializes the NVIC to set a single IRQ priority. 
 * Interrupts will also need to be enabled in the ARM core. This can be 
 * done using the EnableInterrupts macro.
 *
 * Parameters:
 * irq    irq number to be enabled (the irq number NOT the vector number)
 * prio   irq priority. 0-15 levels. 0 max priority
 */

void set_irq_priority (uint32_t irq, uint8_t prio)
{
  /*irq priority pointer*/
  //uint32_t prio_reg;
  
  /* Make sure that the IRQ is an allowable number. Right now up to 111 is 
   * used.
   */
  if (irq > 111) return;

  if (prio > 15) return;        /* Must be ls nibble */
  
  /* Assign priority to IRQ */
  CM4.NVIC_IPR[irq].R = prio<<4;
}
/***********************************************************************/

/***********************************************************************/
/*
 * Route the interrupt to M4.
 * 
 * Parameters:
 * irq    irq number to be routed
 */

uint16_t route_interrupt (uint32_t irq, uint16_t cpu)
{
  /* Make sure that the IRQ is an allowable number. Right now up to 111 is 
   * used.
   */
  if (irq > 111) return 0;

  MSCM.IRSPRC[irq].R = cpu;
  return (uint16_t)MSCM.IRSPRC[irq].R;
}
/***********************************************************************/

