/*
*                    Copyright (c), NXP Semiconductors
*
*                       (C) NXP Semiconductors 2018
*
*         All rights are reserved. Reproduction in whole or in part is
*        prohibited without the written consent of the copyright owner.
*    NXP reserves the right to make changes without notice at any time.
*   NXP makes no warranty, expressed, implied or statutory, including but
*   not limited to any implied warranty of merchantability or fitness for any
*  particular purpose, or that the use will not infringe any third party patent,
*   copyright or trademark. NXP must not be liable for any loss or damage
*                            arising from its use.
*/

#ifndef NDEF_REC_TEXT_H_
#define NDEF_REC_TEXT_H_

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* ********************************************************************************************
* Includes
* ******************************************************************************************* */
#include <stdint.h>
#include "NDEF_Status.h"
#include "NDEF_Record.h"

/** \defgroup text Text record
*
* \brief
* @{
*/


/* ********************************************************************************************
* Types/Structure Declarations
* ******************************************************************************************* */

/**
* \brief Text encoding enumeration
*/
typedef enum
{
	Encoding_UTF8 = 0x00,
	Encoding_UTF16= 0x01
}NDEF_Rec_TextEncoding_t;


/**
* \brief Text record structure
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;				/**< Internal record information. */
	NDEF_Record_Header_t sHeader;				/**< Record header structure. */
	uint8_t   bStatusByte;						/**< Status byte of the payload. */
	uint8_t * pLanguageCode;					/**< Pointer to Language code allocated on heap . */
	uint8_t * pPlainText;						/**< Pointer to Plain text allocated on heap. */
}NDEF_Rec_Text_t;

/* ********************************************************************************************
* Function Prototypes
* ******************************************************************************************* */
/**
 * \brief  This function will allocate Text record on heap (including all parameters of variable length)  and return a handle by which the record can be referenced. Text encoding is implicitly set to utf-8 with limitiation to ASCII subset - this function is basically a wrapper of NDEF_Rec_Text_CreateByArray which allows user to input the text as string instead of arrays.
 *		   pLangCode represents the IANA  language code  of the plaintext - null terminated string.
 *		   Note: This API function is intended to be used only with ASCII subset of utf-8 encoding. In case the full Unicode character support is desired,  NDEF_Rec_Text_CreateByArray should be used.
 *	
 *
 * @param[in] pLangCode			Language code string - null terminated.
 * @param[in] pPlainText		PlainText string - null terminated .
 *
 * @return Record handle
 * @retval In case of error returns NULL.
 */
extern NDEF_Rec_Text_t * NDEF_Rec_Text_CreateByStr(const char * pLangCode, const char * pPlainText);


/**
* \brief  This function will allocate Text record on heap(including all parameters of variable length)  and return a handle by which the record can be referenced. Text encoding needs to be explicitely set via eEncoding enumeration while reflecting the actuall encoding of the pPlainText buffer content. 
*		  pLangCode represents the IANA  language code  of the plaintext - null terminated string.
*		  
* @param[in] pLangCode			Language code string - null terminated.
* @param[in] eEncoding			Text encoding.
* @param[in] pPlainText			Pointer to PlainText array.
* @param[in] dwLength			Byte length of PlainText.
*
* @return Record handle
* @retval In case of error returns NULL.
*/
extern NDEF_Rec_Text_t * NDEF_Rec_Text_CreateByArray(const char * pLangCode, NDEF_Rec_TextEncoding_t eEncoding, const uint8_t * pPlainText, uint32_t dwLength);

/**
 * \brief  This function is used to get the PlainText from a record specified by handle.
 *		   Note: This API function is intended to be used only with ASCII subset. For now the library doesnt distinguish between ASCII and non ASCII encoding of parsed message text, therefore if only ASCII characters are expected this function could be used, otherwise NDEF_Rec_Text_GetPlainTextArray should be used.
 *
 * @param[in] 		pRecord	   				Record handle.
 * @param[out] 		pText   				Pointer to output Text array - output will be NULL terminated.
 * @param[in] 	    dwLength   				Size of pText output array.

 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 * @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer.
 */

extern Status_t NDEF_Rec_Text_GetPlainTextStr(const NDEF_Rec_Text_t * pRecord, char * pText, uint32_t  dwLength);


/**
* \brief This function is used to get the PlainText from a record specified by handle.
*
* @param[in] 		pRecord	   				Record handle.
* @param[out] 		pText   				Pointer to output Text array.
* @param[in] 	    dwLength   				Size of pText output array.

*
* @return Status of the API
* @retval NDEF_STATUS_SUCCESS				Function returned successfully.
* @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
* @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer.
*/
extern Status_t NDEF_Rec_Text_GetPlainTextArray(const NDEF_Rec_Text_t * pRecord, uint8_t * pText, uint32_t  dwLength);


/**
 * \brief This function is used to return the PlainText length from a record specified by handle.
 *
 * @param[in] pRecord	   		Record handle
 *
 * @return Length of PlainText
 */
extern uint32_t NDEF_Rec_Text_GetPlainTextLen(const NDEF_Rec_Text_t * pRecord);


/**
 * \brief This function is used to return the EcodingType of a record specified by handle.
 *
 * @param[in] 		pRecord	   				Record handle
 *
 * @return Enumeration of TextEncdoing - NDEF_Rec_TextEncoding_t
 */
extern NDEF_Rec_TextEncoding_t  NDEF_Rec_Text_GetEncoding(const NDEF_Rec_Text_t * pRecord);


/**
* \brief This function is used to return the Languagecode length from a record specified by handle.
*
* @param[in] pRecord	   		Record handle
*
* @return Length of the LanguageCode
*/
extern uint8_t NDEF_Rec_Text_GetLanguageLen(const NDEF_Rec_Text_t * pRecord);

/**
 * \brief This function is used to get the LanguageCode from a record specified by handle.
 *
 * @param[in] 		pRecord	   				Text record handle.
 * @param[out] 		pLangCode   			Pointer to output LanguageCode buffer - output will be NULL terminated.
 * @param[in] 		wLength   				Size of Language code output buffer.


 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 * @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer.
 */
extern Status_t NDEF_Rec_Text_GetLanguageStr(const NDEF_Rec_Text_t * pRecord, char * pLangCode, uint16_t  wLength);

#ifdef __cplusplus
} /* Extern C */
#endif

/** @}
* end of text Text record
*/


#endif /* NDEF_REC_TEXT_H_ */

