/*
*                    Copyright (c), NXP Semiconductors
*
*                       (C) NXP Semiconductors 2018
*
*         All rights are reserved. Reproduction in whole or in part is
*        prohibited without the written consent of the copyright owner.
*    NXP reserves the right to make changes without notice at any time.
*   NXP makes no warranty, expressed, implied or statutory, including but
*   not limited to any implied warranty of merchantability or fitness for any
*  particular purpose, or that the use will not infringe any third party patent,
*   copyright or trademark. NXP must not be liable for any loss or damage
*                            arising from its use.
*/

#ifndef NDEF_REC_BLUETOOTHSSP_H_
#define NDEF_REC_BLUETOOTHSSP_H_

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* ********************************************************************************************
* Includes
* ******************************************************************************************* */
#include <stdint.h>
#include "NDEF_Status.h"
#include "NDEF_Record.h"


/** \defgroup Btssp Bluetooth Secure Simple Pairing
*
* \brief
* @{
*/

/* ********************************************************************************************
* MACROS/Defines
* ******************************************************************************************* */
#define BTSSP_DEVICE_ADDR_LEN	6
#define BTSSP_OOB_DATA_LENGTH_LEN      2

#define BTSSP_LE_DEVICE_ADDR_LENGTH_LEN         1
#define BTSSP_LE_DEVICE_ADDR_DATA_TYPE		 0x1BU
#define BTSSP_LE_DEVICE_ADDR_DATA_TYPE_LEN   1
#define BTSSP_LE_DEVICE_ADDR_LEN             7
#define BTSSP_LE_DEVICE_ADDR_TOTAL_LEN    (BTSSP_LE_DEVICE_ADDR_LENGTH_LEN + BTSSP_LE_DEVICE_ADDR_DATA_TYPE_LEN + BTSSP_LE_DEVICE_ADDR_LEN)

#define BTSSP_LE_ROLE_LENGTH_LEN  1
#define BTSSP_LE_ROLE_DATA_TYPE  0x1CU
#define BTSSP_LE_ROLE_DATA_TYE_LEN 1
#define BTSSP_LE_ROLE_LEN 1
#define BTSSP_LE_ROLE_TOTAL_LEN (BTSSP_LE_ROLE_LENGTH_LEN + BTSSP_LE_ROLE_DATA_TYE_LEN + BTSSP_LE_ROLE_LEN)

/* ********************************************************************************************
* Types/Structure Declarations
* ******************************************************************************************* */

/**
* \brief Address type of low energy record
*/
typedef enum
{
	PublicAddress = 0x00,
	RandomAddress,
	ErrorAddressType
}AddressType_t;


/**
* \brief Device role of low energy record
*/
typedef enum
{
	PeripheralOnly = 0x00,	  
	CentralOnly,
	PeripheralPrefered,
	CentralPrefered,
	ErrorRole
}DeviceRole_t;

/**
* \brief Bluetooth secure simple pairing record stucture for BR/EDR devices.
*/
typedef struct
{
	RecordInfo_t INT_sRecordInfo;					/**< Internal record information. */
	NDEF_Record_Header_t sHeader;					/**< Record header structure. */
	uint16_t  wOobDataLen;							/**< OutOfBandData length. */
	uint8_t   aDeviceAddress[BTSSP_DEVICE_ADDR_LEN];/**< Device address array[6]. */
	uint8_t * pOptionalOobData;						/**< Pointer to  optional OutOfBandData - EIR formatted OOB data(BT CORE specification). */
}NDEF_Rec_BtsspBrEdr_t;


/**
* \brief Bluetooth secure simple pairing record stucture for LE devices.
*/
typedef struct
{	
	RecordInfo_t INT_sRecordInfo;							/**< Internal record information. */
	NDEF_Record_Header_t sHeader;							/**< Record header structure. */
	uint8_t aDeviceAddress[BTSSP_LE_DEVICE_ADDR_LEN];	    /**< BLE device address[6]. */
	DeviceRole_t eLeRole;									/**< Low energy role. */
	uint8_t * pOptionalOobData;								/**< Pointer to optional OutOfBandData - AD formatted OOB data (BT CORE specification). */	
}NDEF_Rec_BtsspLe_t;


/* ********************************************************************************************
* Function Prototypes
* ******************************************************************************************* */

/**
 * \brief This function will allocate Bluetooth secure simple pairing BREDR record on heap  and return a handle by which the record can be referenced. 
 *
 * @param[in] 	  pDeviceAdress  Pointer to DeviceAddress[6]
 *
 * @return Record handle
 * @retval In case of error returns NULL
 */

extern NDEF_Rec_BtsspBrEdr_t * NDEF_Rec_Btssp_CreateBrEdr(const uint8_t * pDeviceAdress);


/**
* \brief This function will allocate Bluetooth secure simple pairing Low Energy record on heap  and return a handle by which the record can be referenced. 
*
* @param[in] 	  pDeviceAdress  Pointer to DeviceAddress[6]
*
* @return Record handle
* @retval In case of error returns NULL.
*/

extern NDEF_Rec_BtsspLe_t * NDEF_Rec_Btssp_CreateLe(const uint8_t * pDeviceAdress, AddressType_t eAddressType, DeviceRole_t eRole);


/**
 * \brief This function will allocate OptionalOobData on heap, and set the referece into record specified by handle. Optional Oob Data are non-mandatory data used within BREDR or LE records which need to be formated to either EIR(BREDR) or AD(LE) format by application layer, and then assigned to record. For information about EIR or AD format see Bluetooth Core specification.
 *
 * @param[in] pRecord	   		Record handle(either BREDR or LE).
 * @param[in] pOptionalOobData  Pointer to OutOfBandData.
 * @param[in] wOobDataLength    Length of OptionalOobData  
 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 * @retval NDEF_STATUS_ERROR_NULL_PTR		Failed malloc.
 */
extern Status_t NDEF_Rec_Btssp_SetOptionalOobData(void * pRecord, const uint8_t * pOptionalOobData, uint16_t wOobDataLength);


/**
 * \brief This generic function is used to get the OptionalOutOfBandDataLen of a record specified by handle. Works for both BREDR and BLE records. 
 *		  In case of BREDR, the return value reflects the length of  OptionalOobData  - it its not equal to wOobDataLen structure member of NDEF_Rec_BtsspBrEdr_t. In case OobDataLen is needed, it might be calculated from return value with following formula:  OobDataLen = return_value +BTSSP_DEVICE_ADDR_LEN + BTSSP_OOB_DATA_LEN.
 *
 * @param[in] pRecord	 Pointer to Btssp record structure.
 *
 * @return Length of OptionalOutOfBandData
 */
extern uint16_t NDEF_Rec_Btssp_GetOptionalOobDataLen(const void * pRecord);

/**
 * \brief This generic function is used to get the OptionalOutOfBandData of a record specified by handle. Works for both BREDR and BLE records.
 *
 * @param[in] 	  pRecord   Record handle(either BREDR or LE).
 * @param[in] 	  wLength	Size of OutOfBandData buffer.
 * @param[out] 	  pOptionalOobData  Pointer to OutOfBandData.
 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 *  @retval NDEF_STATUS_ERROR_BUFF_OVF		Output won't fit into buffer. */
extern Status_t NDEF_Rec_Btssp_GetOptionalOobData(const void * pRecord, uint8_t * pOptionalOobData, uint16_t wLength);

/**
 * \brief  This generic function is used to get the DeviceAddress of a record specified by handle. Works for both BREDR and BLE records.
 *
 * @param[in] 	  pRecord  		 Record handle(either BREDR or LE).
 * @param[out] 	  pDeviceAdress  Pointer to DeviceAddress[6]
 *
 * @return Status of the API
 * @retval NDEF_STATUS_SUCCESS				Function returned successfully.
 * @retval NDEF_STATUS_ERROR_INVALID_PARAM	Null/Invalid function parameter.
 */

extern Status_t NDEF_Rec_Btssp_GetDeviceAddress(const void * pRecord,  uint8_t * pDeviceAdress);


/**
* \brief  This function is used to return the Low enrgy address type of a record specified by handle.
*
* @param[in] 	  pRecord BLE handle.
*
* @return Low energy address type
*/

extern AddressType_t NDEF_Rec_Btssp_GetLeAddressType(NDEF_Rec_BtsspLe_t * pRecord);

/**
* \brief  This function is used to return the Low energy role of a record specified by handle.
*
* @param[in]     pRecord BLE handle.
*
* @return Low Energy role
*/

extern DeviceRole_t NDEF_Rec_Btssp_GetLeRole(NDEF_Rec_BtsspLe_t * pRecord);


/** @}
* end of Btssp Bluetooth secure simple pairing
*/


#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* NDEF_REC_BLUETOOTHSSP_H_ */
