/*
 * File:		gpio_main.c
 * Purpose:		GPIO Example/Tests
 *
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"
#include "uif.h"
#include "clock/clock.h"
#include "uart/uart.h"
#include "mcf5213_gpio.h"

typedef char bool;
#define false 0
#define true 1
/********************************************************************/
/*
 * Setup user interface
 */
const char BANNER[] = "\rGPIO Debugger\n";
const char PROMPT[] = "GPIO> ";

void cmd_sysclk (int, char **);
void cmd_dump (int, char **);
void cmd_setreg8  (int, char **);
void cmd_setreg16  (int, char **);
void cmd_setreg32  (int, char **);
void cmd_setpin  (int, char **);
void cmd_getport  (int, char **);
void cmd_getportp  (int, char **);

UIF_CMD UIF_CMDTAB[] =
{
    UIF_CMDS_ALL
    {"dump",0,5,0,cmd_dump,"Display the GPIO registers","[p|d|s|c|a]"},
    {"setreg",2,2,0,cmd_setreg32,"Writes to the GPIO register","<reg offset> <value>"},
    {"setreg8",2,2,0,cmd_setreg8,"Writes to the GPIO register","<reg offset> <value>"},
    {"setreg16",2,2,0,cmd_setreg16,"Writes to the GPIO register","<reg offset> <value>"},
    {"setreg32",2,2,0,cmd_setreg32,"Writes to the GPIO register","<reg offset> <value>"},
    {"setpin",3,3,0,cmd_setpin,"Writes to the GPIO register pin","<reg> <pin> <0/1>"},
    {"getport",1,1,0,cmd_getport,"Reads to the GPIO PORT register","<reg>"},
    {"getportp",1,1,0,cmd_getport,"Reads to the GPIO PORTP/SET register","<reg>"},
};

void setshow_baud (int, char **);

UIF_SETCMD UIF_SETCMDTAB[] =
{
    {"baud", 0,1,setshow_baud,""},
};

const int UIF_NUM_CMD    = UIF_CMDTAB_SIZE;
const int UIF_NUM_SETCMD = UIF_SETCMDTAB_SIZE;

/*
 * System clock variables
 */
int baud = 19200;

/********************************************************************/
void 
main (void)
{
	mcf5xxx_irq_enable();
	
    printf(BANNER);
    printf(HELPMSG);

    while (TRUE)
    {
        printf(PROMPT);
        run_cmd();
    }
}

/********************************************************************/
void
cmd_dump (int argc, char **argv)
{
    int i;
    bool port = false;
    bool ddr = false;
    bool set = false;
    bool clr = false;
    bool par = false;
    if (argc == 1)
    {
    	port = true;
    	ddr = true;
    	set = true;
    	clr = true;
    	par = true;
    }
    else
	for (i=1; i<argc;i++)
	{
		switch (argv[i][0])
		{
			case 'p':
			case 'P':
			    port = true;
			    break;
			case 'd':
			case 'D':
			    ddr = true;
			    break;
			case 's':
			case 'S':
			    set = true;
			    break;
			case 'c':
			case 'C':
			    clr = true;
			    break;
			case 'a':
			case 'A':
			    par = true;
			    break;
			default:
			    break;
		}
	}

	if (port)
	{
		printf("PORTNQ   = %#02X\n",MCF_GPIO_PORTNQ);
    	printf("PORTDD   = %#02X\n",MCF_GPIO_PORTDD);
    	printf("PORTAN   = %#02X\n",MCF_GPIO_PORTAN);
    	printf("PORTAS   = %#02X\n",MCF_GPIO_PORTAS);
 	   	printf("PORTQS   = %#02X\n",MCF_GPIO_PORTQS);
 	   	printf("PORTTA   = %#02X\n",MCF_GPIO_PORTTA);
 	   	printf("PORTTC   = %#02X\n",MCF_GPIO_PORTTC);
 	   	printf("PORTTD   = %#02X\n",MCF_GPIO_PORTTD);
 	   	printf("PORTUA   = %#02X\n",MCF_GPIO_PORTUA);
 	   	printf("PORTUB   = %#02X\n",MCF_GPIO_PORTUB);
 	   	printf("PORTUB   = %#02X\n",MCF_GPIO_PORTUC);
	}
	if (ddr)
	{
    	printf("DDRNQ    = %#02X\n",MCF_GPIO_DDRNQ );
    	printf("DDRDD    = %#02X\n",MCF_GPIO_DDRDD );
    	printf("DDRAN    = %#02X\n",MCF_GPIO_DDRAN );
    	printf("DDRAS    = %#02X\n",MCF_GPIO_DDRAS );
   		printf("DDRQS    = %#02X\n",MCF_GPIO_DDRQS );
  	  	printf("DDRTA    = %#02X\n",MCF_GPIO_DDRTA );
    	printf("DDRTC    = %#02X\n",MCF_GPIO_DDRTC );
   		printf("DDRTD    = %#02X\n",MCF_GPIO_DDRTD );
    	printf("DDRUA    = %#02X\n",MCF_GPIO_DDRUA );
    	printf("DDRUB    = %#02X\n",MCF_GPIO_DDRUB );
    	printf("DDRUC    = %#02X\n",MCF_GPIO_DDRUC );
	}
	if (set)
	{
    	printf("SETNQ    = %#02X\n",MCF_GPIO_SETNQ );
    	printf("SETDD    = %#02X\n",MCF_GPIO_SETDD );
    	printf("SETAN    = %#02X\n",MCF_GPIO_SETAN );
    	printf("SETAS    = %#02X\n",MCF_GPIO_SETAS );
    	printf("SETQS    = %#02X\n",MCF_GPIO_SETQS );
    	printf("SETTA    = %#02X\n",MCF_GPIO_SETTA );
    	printf("SETTC    = %#02X\n",MCF_GPIO_SETTC );
    	printf("SETTD    = %#02X\n",MCF_GPIO_SETTD );
    	printf("SETUA    = %#02X\n",MCF_GPIO_SETUA );
    	printf("SETUB    = %#02X\n",MCF_GPIO_SETUB );
    	printf("SETUC    = %#02X\n",MCF_GPIO_SETUC );
	}
	if (clr)
	{
    	printf("CLRNQ    = %#02X\n",MCF_GPIO_CLRNQ );
    	printf("CLRDD    = %#02X\n",MCF_GPIO_CLRDD );
    	printf("CLRAN    = %#02X\n",MCF_GPIO_CLRAN );
    	printf("CLRAS    = %#02X\n",MCF_GPIO_CLRAS );
    	printf("CLRQS    = %#02X\n",MCF_GPIO_CLRQS );
    	printf("CLRTA    = %#02X\n",MCF_GPIO_CLRTA );
    	printf("CLRTC    = %#02X\n",MCF_GPIO_CLRTC );
    	printf("CLRTD    = %#02X\n",MCF_GPIO_CLRTD );
    	printf("CLRUA    = %#02X\n",MCF_GPIO_CLRUA );
    	printf("CLRUB    = %#02X\n",MCF_GPIO_CLRUB );
    	printf("CLRUC    = %#02X\n",MCF_GPIO_CLRUC );
	}
	if (par)
	{
    	printf("PNQPAR   = %#02X\n",MCF_GPIO_PNQPAR);
    	printf("PDDPAR   = %#02X\n",MCF_GPIO_PDDPAR);
    	printf("PANPAR   = %#02X\n",MCF_GPIO_PANPAR);
    	printf("PASPAR   = %#02X\n",MCF_GPIO_PASPAR);
    	printf("PQSPAR   = %#04X\n",MCF_GPIO_PQSPAR);
    	printf("PTAPAR   = %#02X\n",MCF_GPIO_PTAPAR);
    	printf("PTCPAR   = %#02X\n",MCF_GPIO_PTCPAR);
    	printf("PTDPAR   = %#02X\n",MCF_GPIO_PTDPAR);
    	printf("PUAPAR   = %#02X\n",MCF_GPIO_PUAPAR);
    	printf("PUBPAR   = %#02X\n",MCF_GPIO_PUBPAR);
    	printf("PUCPAR   = %#02X\n",MCF_GPIO_PUCPAR);
    	printf("PSRR     = %#08X\n",MCF_PAD_PSRR  );
    	printf("PSDR     = %#08X\n",MCF_PAD_PDSR  );
	}
}

/********************************************************************/
void
cmd_getport (int argc, char **argv)
{
    int success, reg;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 2) && (argv[1] != NULL))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        address += reg;
        printf("PORT**(%d)   = %#08X\n",reg, (*(vuint8*)address));
    }
}

/********************************************************************/
void
cmd_getportp (int argc, char **argv)
{
    int success, reg;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 2) && (argv[1] != NULL))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        address += reg;
        printf("PORTP**(%d)   = %#08X\n",reg, (*(vuint8*)address));
    }
}

/********************************************************************/
void
cmd_setreg8 (int argc, char **argv)
{
    int success, value, reg;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 3) && ((argv[1] != NULL) && (argv[2] != NULL)))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        value = get_value(argv[2],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        address += reg;
        (*(vuint8*)address) = (uint8)value;
    }
}

/********************************************************************/
void
cmd_setreg16 (int argc, char **argv)
{
    int success, value, reg;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 3) && ((argv[1] != NULL) && (argv[2] != NULL)))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        value = get_value(argv[2],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        address += reg;
        (*(vuint16*)address) = (uint16)value;
    }
}

/********************************************************************/
void
cmd_setreg32 (int argc, char **argv)
{
    int success, value, reg;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 3) && ((argv[1] != NULL) && (argv[2] != NULL)))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        value = get_value(argv[2],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[2]);
            return;
        }
        address += reg;
        (*(vuint32*)address) = value;
    }
}

/********************************************************************/
void
cmd_setpin (int argc, char **argv)
{
    int success, pin, reg, value;
    unsigned int address = (unsigned int)(&MCF_GPIO_REGBASE);
    if ((argc == 4) && ((argv[1] != NULL) && (argv[2] != NULL) && (argv[3] != NULL)))
    {
        reg = get_value(argv[1],&success,16);
        if (success == 0)
        {
            printf(INVALUE,argv[1]);
            return;
        }
        pin = get_value(argv[2],&success,10);
        if ((success == 0) || (pin > 7))
        {
            printf(INVALUE,argv[2]);
            return;
        }
        value = get_value(argv[3],&success,10);
        if((success == 0) || (value!=0 && value!=1))
        {
            printf(INVALUE,argv[3]);
            return;
        }
        address += reg;
        if(value==1)
        	(*(vuint8*)address) |= (0x1 << pin);
        else
        	(*(vuint8*)address) &= (~(0x1 << pin));	
    }
}


/********************************************************************/
void
setshow_baud (int argc, char **argv)
{
}
/********************************************************************/
