/*
 * File:    mcf5213.c
 * Purpose: MCF521x specific routines   
 *
 * Notes:       
 * 
 * License:     All software covered by license agreement in -
 *              docs/Freescale_Software_License.pdf
 */

#include "common.h"

extern int d0_reset;
extern int d1_reset;

/********************************************************************/
void
cpu_startup (void)
{
#ifdef DEBUG_PRINT
    /*
     * Determine cause(s) of reset
     */
    printf("\n\n");
    if (MCF_SCM_CRSR & MCF_SCM_CRSR_CWDR)
        printf("Core Watchdog Time Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_LVD)
        printf("Low Voltage Detect Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_SOFT)
        printf("Software Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_WDR)
        printf("Watchdog Timer Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_POR)
        printf("Power-on Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_EXT)
        printf("External Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_LOC)
        printf("Loss of Clock Reset\n");
    if (MCF_CIM_RSR & MCF_CIM_RSR_LOL)
        printf("Loss of PLL Lock Reset\n");

    /*
     * Identify which MCU of which revision we are executing on
     */
    switch ((MCF_CIM_CIR & 0xFFC0) >> 6)
    {
        case 0x3C:
            printf("MCF5211 ");
            break;
        case 0x42:
            printf("MCF5212 ");
            break;
        case 0x43:
            printf("MCF5213 ");
            break;
        default:
            printf("Unknown ");
            break;
    }
    printf("Rev. %d\n",(MCF_CIM_CIR & 0x003F));
#endif

#ifdef DEBUG_PRINT_D0D1
    /*
     * Print out the core integration information
     */
   mcf5xxx_interpret_d0d1(d0_reset,d1_reset);
#endif

    /* 
     * Enable on-chip modules to access internal SRAM 
     */
    MCF_SCM_RAMBAR = 0
        | MCF_SCM_RAMBAR_BA(SRAM_ADDRESS)
        | MCF_SCM_RAMBAR_BDE;
}
/********************************************************************/
void
cpu_handle_interrupt (int vector)
{
    if (vector < 64 || vector > 192)
        return;
    
    if (vector >= 64 && vector <= 71)
        board_handle_interrupt(vector);
    else
        printf("User Defined Vector #%d\n",vector);
}
/********************************************************************/
/*
 * Pause for the specified number of micro-seconds.
 * Uses DTIM3 as a timer
 */
void
cpu_pause(int usecs)
{
    /* Enable the DMA Timer 3 */
    MCF_DTIM3_DTRR = (vuint32)(usecs - 1);
    MCF_DTIM3_DTER = MCF_DTIM_DTER_REF;
    MCF_DTIM3_DTMR = 0
        | MCF_DTIM_DTMR_PS(sys_clk_khz / 1000)
        | MCF_DTIM_DTMR_ORRI
        | MCF_DTIM_DTMR_FRR
        | MCF_DTIM_DTMR_CLK_DIV1
        | MCF_DTIM_DTMR_RST;

    while ((MCF_DTIM3_DTER & MCF_DTIM_DTER_REF) == 0) 
    {};
    
    /* Disable the timer */
    MCF_DTIM3_DTMR = 0;
}
/********************************************************************/
/*
 * These functions enable and disable the PST and DDATA BDM signals.
 *
 * Disabling PST/DDATA conserves power and reduces EMI.  Enabling these
 * signals enables real-time trace capabilities.  Some BDM interface
 * cables use the PST signals to detect when the processor is halted
 * (PST[3:0] = 1111).
 *
 * The PST/DDATA signals can only be disabled by setting the CSR[PCD]
 * bit. The CSR is only writeable via the BDM or the WDEBUG instrcution
 *
 * On the MCF5213, the PSTCLK is not disabled by setting the CSR[PCD] bit
 * Instead it is controlled by the SYNCR[DISCLK] bit. The CLOCK driver
 * provides functions to control the PSTCLK/CLKOUT signal.
 *
 * The MCF5213 also provides an ALLPST signal. This signal is the logical
 * AND of PST[3:0] and is used by BDM interface cables to detect when
 * the processor is halted. The ALLPST signal is also controlled
 * by these functions.
 */

uint16 wdebug_pstddata_off[4] = {0x2C80, 0x0002, 0x0000, 0x0000};
uint16 wdebug_pstddata_on[4]  = {0x2C80, 0x0000, 0x0000, 0x0000};

void
cpu_pstddata_disable(void)
{
    mcf5xxx_exe_wdebug(wdebug_pstddata_off);
}

void
cpu_pstddata_enable(void)
{
    mcf5xxx_exe_wdebug(wdebug_pstddata_on);
}
/********************************************************************/
