/******************************************************************************
*
*       COPYRIGHT (c) 2001-2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
*       The code is the property of Motorola.
*
*       The copyright notice above does not evidence any
*       actual or intended publication of such source code.
*
* Filename:     $Source: /proj/cvsroot/mgt/MGT5200/apps/AC97Sample/modules/PSC_UART/pscuart5200.c,v $
* Author:       $Author: ra6707 $
* Locker:       $Locker:  $
* State:        $State: Exp $
* Revision:     $Revision: 1.1 $
*
* Functions:    
*
* History:      Use the RCS command rlog to display revision history
*               information.
*
* Description:  
*
* Notes:                
*
******************************************************************************/

#include <string.h>

#include "ppctypes.h"

#include "configure.h"

#if !(defined(MGT5200) ^ defined(MPC5200B))
	#error Either MGT5200 or MPC5200B has to be defined.
#endif

#if defined(MPC5200B)
	#include "mpc5200b/mpc5200b.h"
	#include "mpc5200b/gpio_std.h"
	#include "mpc5200b/psc.h"
#elif defined(MGT5200)
	#include "mgt5200/mgt5200.h"
	#include "mgt5200/gpio_std.h"
	#include "mgt5200/psc.h"
#endif

#include "core5200.h"
#include "freq5200.h"
#include "pscuart5200.h"

/*! \file pscuart5200.c
 * This file contains the C-code of the PSCUART5200 module implementing the
 * basic hardware abstraction layer. The abstraction layer needed by
 * CodeWarriors MSL is implemented in uart.c.
 */

psc_regs *psc = NULL;

/*! \brief
 * Initialize the PSC UART.
 *
 * This function must be called before any other PSC UART function is called.
 *
 * \param	baudRate	Baud rate to initialize UART with.
 */
#if (defined PSC_UART_NOINIT_dBUG) && (defined dBUG)
void pscuartInit (unsigned long baudRate)
{
#pragma unused (baudRate)
	switch (PSC_UART_PORT) {
		case 1:
			psc = (void *)(readMBAR () + MBAR_PSC1);
			break;
		case 2:
			psc = (void *)(readMBAR () + MBAR_PSC2);
			break;
		case 3:
			psc = (void *)(readMBAR () + MBAR_PSC3);
			break;
		case 4:
			psc = (void *)(readMBAR () + MBAR_PSC4);
			break;
		case 5:
			psc = (void *)(readMBAR () + MBAR_PSC5);
			break;
		case 6:
			psc = (void *)(readMBAR () + MBAR_PSC6);
			break;
		default:
			psc = NULL;
			break;	
	}
}

#else

void pscuartInit (unsigned long baudRate)
{
	gpio_std_regs *gpio;
	uint16 divider;

	gpio = (gpio_std_regs *) (readMBAR () + MBAR_GPIO_STD);

	switch (PSC_UART_PORT) {
		case 1:
			psc = (void *)(readMBAR () + MBAR_PSC1);
			/*
			 * Set GPIO port mapping for PSC1 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFFFFFFF7) | 0x00000004;
			break;
		case 2:
			psc = (void *)(readMBAR () + MBAR_PSC2);
			/*
			 * Set GPIO port mapping for PSC2 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFFFFFF7F) | 0x00000040;
			break;
		case 3:
			psc = (void *)(readMBAR () + MBAR_PSC3);
			/*
			 * Set GPIO port mapping for PSC3 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFFFFF0FF) | 0x00000400;
			break;
		case 4:
			psc = (void *)(readMBAR () + MBAR_PSC4);
			/*
			 * Set GPIO port mapping for PSC4 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFFF0FFFF) | 0x00080000;
			break;
		case 5:
			psc = (void *)(readMBAR () + MBAR_PSC5);
			/*
			 * Set GPIO port mapping for PSC5 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFFF0FFFF) | 0x000B0000;
			break;
		case 6:
			psc = (void *)(readMBAR () + MBAR_PSC6);
			/*
			 * Set GPIO port mapping for PSC6 to UART
			 */
			gpio->port_config = (gpio->port_config & 0xFF0FFFFF) | 0x00500000;
			break;
		default:
			psc = NULL;
			break;	
	}

	if (psc == NULL) {
		return;
	}

	/*
	 * Calculate baud rate divider setting
	 */
#ifdef PSC_UART_BAUDRATE
	baudRate = PSC_UART_BAUDRATE;
#endif
#ifdef IPBI_FREQ
	divider = (uint16) (((IPBI_FREQ / baudRate) + 16) / 32);
#else
	divider = (uint16) (((freqIPBI () / baudRate) + 16) / 32);
#endif

	psc->SICR = 0x00;				/* SICR: SIM2 = uart mode,dcd does not affect rx */
	psc->SR_CSR = 0xDD00;			/* CSR: RX/TX baud rate from timers */
	psc->CTUR = (uint8) ((divider >> 8) & 0xFF);	/* CTUR: divide counter upper byte */
	psc->CTLR = (uint8) (divider & 0xFF);	/* CTLR: divide counter lower byte */
	psc->CR = 0x20;					/* CR: reset RX and RXFIFO */
	psc->CR = 0x30;					/* CR: reset TX and TXFIFO */
	psc->CR = 0x40;					/* CR: reset error status */
	psc->CR = 0x50;					/* CR: reset break change int */
	psc->CR = 0x10;					/* CR: reset MR pointer */
	psc->IPCR_ACR = 0x00;			/* ACR: disable state change CTS/ DCD interrupts */
#if defined HARDWARE_FLOW_CTRL
	psc->ModeReg = 0xF3;			/* MR1: RX controls RTS, 8bit data, no parity */
	psc->ModeReg = 0x17;			/* MR2: normal mode, 1stop bit, CTS controls TX */
#else
	psc->ModeReg = 0x73;			/* MR1: 8bit data, no parity */
	psc->ModeReg = 0x07;			/* MR2: normal mode,1stop bit */
#endif
	psc->OP1 = 1;					/* OP1: enable RTS to send */
	psc->ISR_IMR = 0x0000;			/* IMR: Mask RxRDY and TxRDY from causing an interrupt */
	psc->RFALARM = 0xF8;			/* RFALARM: */
	psc->TFALARM = 0xF8;			/* TFALARM: */
	psc->CR = 0x05;					/* CR: enable TX and RX. */			
}

#endif

/*! \brief
 * See if a character is available at the PSC UART and if so retreive it.
 *
 * \return Character or error code:
 * - -1, no character available or interface not initialized
 * - else, the first character of the FIFO.
 */
int pscuartPollChar (void)
{
	char c;
	
	if (psc == NULL) {
		return -1;
	}
	
	if (((psc->SR_CSR) & 0x0100) != 0) {	/* receiver ready? */
		c = *((char *) &psc->RB_TB);		/* get character */

#ifdef RCV_CR_TO_LF
		if (c == '\r') {
			c = '\n';
		}
#endif


#ifdef RCV_ECHO
		pscuartPutChar (c);	
#endif
		return c;
	} else {
		return -1;							/* return error */
	}
}
/*! \brief Retrieve a character from the PSC UART.
 *
 * If none is available wait for one.
 *
 * \return Character or error code:
 * - -1, interface not initialized
 * - else, the first character of the FIFO
 */
int pscuartGetChar (void)
{
	char c;
	
	if (psc == NULL) {
		return -1;
	}
	
	while (((psc->SR_CSR) & 0x0100) == 0) {};	/* wait for receiver ready */
	c =  *((char *) &psc->RB_TB);				/* get character */
	
#ifdef RCV_CR_TO_LF
	if (c == '\r') {
		c = '\n';
	}
#endif

#ifdef RCV_ECHO
	pscuartPutChar (c);
	if (c == 0x08) {			/* Backspace */
		pscuartPutChar (' ');
		pscuartPutChar (c);
	}
#endif
	
	return c;
}

/*! \brief Write a character to the PSC UART.
 *
 * If the FIFO is full, wait until there is space.
 *
 * \param	c	Character to transmit.
 */
void pscuartPutChar (char c)
{
	if (psc == NULL) {
		return;
	}

#ifdef SND_LF_TO_CRLF
	if (c == '\n') {
		while (((psc->SR_CSR) & 0x0400) == 0) {};	/* wait for transmitter ready */
		*((char *) &psc->RB_TB) = '\r';				/* put character */
	}		
#endif
	
	while (((psc->SR_CSR) & 0x0400) == 0) {};	/* wait for transmitter ready */
	*((char *) &psc->RB_TB) = c;				/* put character */
}
