/******************************************************************************
*
*       COPYRIGHT (c) 2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
* Filename:     time.c
* Author:       Mark Jonas
* Revision:     1.0
*
* History:      2003-May-30: First release.
*
* Description:  This file contains function to measure time on a MPC5200
*               processor using the Timebase registers of the core.
*
* Notes:        These functions will fail if the program runs for more than
*               ~130 seconds without resetting the timer (timeReset()) if
*               a 64 bit division is not available. Else these functions will
*               fail if the program runs for more than (2^32)-1 ms (more than
*               8 years) without resetting the timer. See timeRead().
*
******************************************************************************/

#include "ppctypes.h"

#include "mpc5200.h"
#include "xlarb.h"

#include "time.h"
#include "core.h"
#include "freq.h"

/*--------------------------------------------------------------------------
   Function    : void timeInit (void)
   Description : Initialize timer module. This function must be called
                 before calling any other timer module function.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
void timeInit (void)
{
	xlarb_regs *xlarb;
	
	xlarb = (xlarb_regs *) (readMBAR () + MBAR_XLARB);
	
	xlarb->config |= 0x00002000UL; /* Enable Timebase counter */

	timeReset ();
}

/*--------------------------------------------------------------------------
   Function    : void timeReset (void)
   Description : Reset timer to zero.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
void timeReset (void)
{
	writeTBx (0,0);
}

/*--------------------------------------------------------------------------
   Function    : uint32 timeRead (void)
   Description : Return time expired since last timer reset in Milliseconds.
   Parameter(s): none
   Returns     : Expired time.
  --------------------------------------------------------------------------*/
uint32 timeRead (void)
{
/* This only works if a 64 Bit division routine is available:
	return (uint32) (readTBx () / (freqXLB () / 4000)); */

/* Workaround works for up to 130 seconds @ 132 MHz XLB */
	return (uint32) (((uint32) (readTBx () & 0xFFFFFFFFUL)) / (freqXLB () / 4000));	
}

/*--------------------------------------------------------------------------
   Function    : void timeWaitms (uint32 ms)
   Description : Wait for the time specified in Milliseconds.
   Parameter(s): ms - Time to wait in Milliseconds.
   Returns     : nothing
  --------------------------------------------------------------------------*/
void timeWaitms (uint32 ms)
{
	uint32 now;
	
	now = timeRead ();
	while (now + ms > timeRead ()) {};
}
