/******************************************************************************
*
*       COPYRIGHT (c) 2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
* Filename:     init.S
* Author:       Mark Jonas
* Revision:     1.0
*
* History:      2003-May-30: First release.
*
* Description:  This file contains generic boot code for the MPC5200 processor.
*               Once the procssor is set up the code branches to the main()
*               function of a C program compiled for EABI compatibility. A
*               minimal __eabi function is supplied in this file.
*
* Notes:        
*
******************************************************************************/

#include "init.h"

.global reset_handler
.global __eabi

.text

/*--------------------------------------------------------------------------
   Function    : reset_handler
   Description : Handler for Reset Exception. Not intended to be used as a
                 software reset point by branching to it. Function does not
                 return. Calls main() using the rfi instruction.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
reset_handler:
	// Set r0 to 0.
	lis r0,0x0000

	// NOTE:
	// After a reset MSR state is all exceptions disabled,
	// address translation off, FP disabled.

	// Invalidate BATs to avoid error states because the state of
	// the BATs is unknown after reset.
	mtspr	ibat0u,r0
	mtspr	ibat1u,r0
	mtspr	ibat2u,r0
	mtspr	ibat3u,r0
	mtspr	ibat4u,r0
	mtspr	ibat5u,r0
	mtspr	ibat6u,r0
	mtspr	ibat7u,r0
	isync
	mtspr	dbat0u,r0
	mtspr	dbat1u,r0
	mtspr	dbat2u,r0
	mtspr	dbat3u,r0
	mtspr	dbat4u,r0
	mtspr	dbat5u,r0
	mtspr	dbat6u,r0
	mtspr	dbat7u,r0
	isync
	
	// Setup MBAR mirror register. MBAR is at 0x80000000 after
	// reset.
	lis	r3,0x80000000@h
	ori	r3,r3,0x80000000@l
	mtspr	mbar,r3

	// Switch to execution from CS0
	bl	init_cs0

	// Initialize SDRAM controller.
	bl	init_sdram

#if MMU_ON == 1
	// If the code specifies that we're going to use the MMU, branch to
	// to the setup function that handles setting up the BATs and
	// invalidating TLB entries.
	//
	// NOTE: We've done nothing with the segment registers, so we need to
	// be sure that all memory accessed by this code and by the user
	// program is represented in the BATs. Otherwise, we might get
	// some spurious translations.
	bl	setup_bats
	sync
	bl	address_translation_on
	sync
#endif
	
	// Note: This code is run from reset, so we assume that there is no
	// data that needs to be flushed from the cache. This code only
	// invalidates and enables the caches, it does not flush!
	//
	// Note: The caches are enabled *after* the relocation in order
	// to help avoid cache preloading.
	
	// Relocate the text, data, and bss segments.
	bl	relocate_image

#if DCACHE_ON == 1
	bl	invalidate_and_enable_dcache
#endif

#if ICACHE_ON == 1
	bl	invalidate_and_enable_icache
#endif

	// Load address of main().
	// It's loaded in SRR0 and will be the branch target of the
	// rfi instruction. The rfi is the last instruction executed
	// from Flash. Starting with main() instructions are fetched
	// from SDRAM.
	lis	r3,main@h
	ori	r3,r3,main@l
	mtspr	srr0,r3
	
	// Set the MSR.
	// It's loaded in SRR1 and will be copied into MSR by the rfi
	// instruction.	
	lis	r3,0x0000
	ori	r3,r3,0x3900	// turn on fp,
				// enable fp & machine check exceptions
				// exception prefix 0x00000000		
#if MMU_ON == 1
	ori	r3,r3,0x0030	// turn on I and D translation
#endif
	mtspr	srr1,r3

	// Put something in LR so when main() returns it jumps
	// somewhere safe. Here it is an endless loop.
	lis	r3,exit@h
	ori	r3,r3,exit@l
	mtlr	r3

	// Set up stack pointer for the user application
	lis	r1,STACK_LOC@h	// STACK_LOC defined in ppcinit.h
	ori	r1,r1,STACK_LOC@l
	
	// make sure the word the stack pointer points to is NULL
	lis	r3,0x0000
	stw	r3,0(r1)
	
	// go to main ()
	rfi

exit:
	b	exit

/*--------------------------------------------------------------------------
   Function    : init_cs0
   Description : Initialize CS0 and switch to execution from there.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
init_cs0:
	// Save LR to R31.
	mflr	r31
	
	// R3 points to internal peripheral space.
	mfspr	r3,mbar

	// Enable CS module and external bus error signalling.
	addi	r4,0,(LPC_CS_CTRL_EBEE | LPC_CS_CTRL_ME)@l	// how unsigned?
	sth	r4,LPC_CS_CTRL(r3)
	eieio

	// Set CS0 start and stop address.
	lis	r4,(FLASH_START>>16)@h
	ori	r4,r4,(FLASH_START>>16)@l
	stw	r4,IPBI_CS0_START(r3)
	lis	r4,(FLASH_STOP>>16)@h
	ori	r4,r4,(FLASH_STOP>>16)@l
	stw	r4,IPBI_CS0_STOP(r3)
	eieio
	
	// Set optimized CS settings.
	// TODO
	
	// Check for boot low or boot high (IP bit in MSR).
	mfmsr	r4
	andi.	r4,r4,0x0040
	bgt	boot_high
	
boot_low:
	// CSBoot on, CS0 on
	lis	r4,(IPBI_CTRL_CSBOOT | IPBI_CTRL_CS0)@h
	ori	r4,r4,(IPBI_CTRL_CSBOOT | IPBI_CTRL_CS0)@l
	stw	r4,IPBI_CTRL(r3)
	eieio

	// New start address of Flash
	lis	r4,FLASH_START@h
	ori	r4,r4,FLASH_START@l

	// Adjust return address
	add	r31,r31,r4

	// Switch from executing in CSBoot space in CS0 space with
	// an absolute jump.
	lis	r5,boot_high@h
	ori	r5,r5,boot_high@l
	add	r5,r5,r4
	mtlr	r5
	blr

boot_high:
	// CSBoot off, CS0 on
	lis	r4,IPBI_CTRL_CS0@h
	ori	r4,r4,IPBI_CTRL_CS0@l
	stw	r4,IPBI_CTRL(r3)
	eieio

	// return
	mtlr	r31
	blr

/*--------------------------------------------------------------------------
   Function    : init_sdram
   Description : Initialize SDRAM controller.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
init_sdram:
	// R3 points to internal peripheral space.
	mfspr	r3,mbar

	// Set SDRAM CS0 Configuration Register.
	lis	r4,((RAM_START & 0xFFF00000) | RAM_SIZE)@h
	ori	r4,r4,((RAM_START & 0xFFF00000) | RAM_SIZE)@l
	stw	r4,IPBI_SDRAM_CONF0(r3)

	// NOTE:
	// IceCube+ only uses SDRAM CS0. If this code is run on a
	// board which use CS1 code to setup CS1 must be added here.
	// If CS1 is used the I/O pin must be configured so in the
	// GPIO Port Configuration Register.

	eieio

	// Write Config 1.
	lis	r4,CONFIG1@h
	ori	r4,r4,CONFIG1@l
	stw	r4,SDRAM_CONF1(r3)
	
	// Write Config 2.
	lis	r4,CONFIG2@h
	ori	r4,r4,CONFIG2@l
	stw	r4,SDRAM_CONF2(r3)
	
	// Write Control
	lis	r4,(CONTROL | 0x80000000)@h
	ori	r4,r4,(CONTROL | 0x80000000)@l
	stw	r4,SDRAM_CTRL(r3)
	
	eieio
	
	// Issue precharge all.
	lis	r4,(CONTROL | 0x80000002)@h
	ori	r4,r4,(CONTROL | 0x80000002)@l
	stw	r4,SDRAM_CTRL(r3)
	
	eieio
	
#if DDR == 1
	// Write Extended Mode.
	li	r4,EMODE
	sth	r4,SDRAM_MODE(r3)

	eieio
#endif

	// Write Mode.
	li	r4,(MODE | 0x0400)
	sth	r4,SDRAM_MODE(r3)

	
	eieio
		
	// Issue precharge all followed by a refresh.
	lis	r4,(CONTROL | 0x80000002)@h
	ori	r4,r4,(CONTROL | 0x80000002)@l
	stw	r4,SDRAM_CTRL(r3)

	eieio

	lis	r4,(CONTROL | 0x80000004)@h
	ori	r4,r4,(CONTROL | 0x80000004)@l
	stw	r4,SDRAM_CTRL(r3)
	
	eieio
	
	// Write Mode.
	li	r4,MODE
	sth	r4,SDRAM_MODE(r3)

	eieio
	
	// Write Control.
	lis	r4,CONTROL@h
	ori	r4,r4,CONTROL@l
	stw	r4,SDRAM_CTRL(r3)
	
	eieio

	// Write Tap Delay.
	li	r4,TAP_DEL
	stb	r4,CDM_PORSTCFG(r3)
	
	eieio

	// return
	blr

/*--------------------------------------------------------------------------
   Function    : relocate_image
   Description : Copy code, data and bss sections to SDRAM. All sections
                 must have a size which is a multiple of 64 bytes.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
relocate_image:
	// Check for boot low or boot high (IP bit in MSR).
	// If booting low, execution starts at 0x00000000 which corresponds FLASH_START + 0x0100 = 0xFF800100.
	// If booting high, execution starts at BOOT_HIGH_ADDR + 0x0100 = 0xFFF00100.
	mfmsr	r3
	andi.	r3,r3,0x0040
	bgt	high_offset
	
	lis	r6,FLASH_START@h
	ori	r6,r6,FLASH_START@l
	b	relocate_text

high_offset:	
	lis	r6,BOOT_HIGH_ADDR@h
	ori	r6,r6,BOOT_HIGH_ADDR@l

relocate_text:
	// Copy text section from _text_start in Flash (r3) to

	// _text_start in SDRAM (r4)
	lis	r3,_text_start@h
	ori	r3,r3,_text_start@l
	mr	r4,r3
	add	r3,r3,r6
	
	// Are they the same? No need to relocate if so.
	cmplw	r3,r4
	beq	relocate_data
	
	// Calculate number of 64 byte chunks to copy.
	lis	r5,_text_size@h
	ori	r5,r5,_text_size@l
	srwi.	r5,r5,6			// divide by 64
	beq	relocate_data		// if zero bytes to copy, skip

	// Copy loop
	mtctr	r5
cont_text:
	lmw	r16,0(r3)
	stmw	r16,0(r4)
	addi	r3,r3,64
	addi	r4,r4,64
	bdnz	cont_text
	
relocate_data:
	// Copy data section from _data_start in Flash (r3) to
	// _data_start in SDRAM (r4)
	lis	r3,_data_start@h
	ori	r3,r3,_data_start@l
	mr	r4,r3
	add	r3,r3,r6
	
	// Are they the same? No need to relocate if so.
	cmplw	r3,r4
	beq	clear_bss
	
	// Calculate number of 64 byte chunks to copy.
	lis	r5,_data_size@h
	ori	r5,r5,_data_size@l
	srwi.	r5,r5,6			// divide by 64
	beq	clear_bss		// if zero bytes to copy, skip

	// Copy loop
	mtctr	r5
cont_data:
	lmw	r16,0(r3)
	stmw	r16,0(r4)
	addi	r3,r3,64
	addi	r4,r4,64
	bdnz	cont_data
	
// This clear_bss code can be removed if you're sure you never
// depend on unitialized data being 0.
clear_bss:
	// Clear bss section in SDRAM.
	lis	r4,_bss_start@h
	ori	r4,r4,_bss_start@l

	// Provide zeros.
	lis	r16,0x0000
	mr	r17,r16
	mr	r18,r16
	mr	r19,r16
	mr	r20,r16
	mr	r21,r16
	mr	r22,r16
	mr	r23,r16
	mr	r24,r16
	mr	r25,r16
	mr	r26,r16
	mr	r27,r16
	mr	r28,r16
	mr	r29,r16
	mr	r30,r16
	mr	r31,r16

	// Calculate number of 64 byte chunks to clear.
	lis	r5,_bss_size@h
	ori	r5,r5,_bss_size@l
	srwi.	r5,r5,6			// divide by 64
	beq	fini			// if zero bytes to clear, skip
	
	// Clear loop
	mtctr	r5
cont_bss:
	stmw	r16,0(r4)
	addi	r4,r4,64
	bdnz	cont_bss

fini:
	sync

	// return from relocate_image
	blr

/*--------------------------------------------------------------------------
   Function    : setup_bats
   Description : Set the BATs as defined in init.h. The MMU should be turned
                 off before this code is run and re-enabled afterwards.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
setup_bats:
	lis	r4,IBAT0L_VAL@h
	ori	r4,r4,IBAT0L_VAL@l
	lis	r3,IBAT0U_VAL@h
	ori	r3,r3,IBAT0U_VAL@l
	mtspr	ibat0l,r4
	mtspr	ibat0u,r3
	isync

	lis	r4,DBAT0L_VAL@h
	ori	r4,r4,DBAT0L_VAL@l
	lis	r3,DBAT0U_VAL@h
	ori	r3,r3,DBAT0U_VAL@l
	mtspr	dbat0l,r4
	mtspr	dbat0u,r3
	isync

	lis	r4,IBAT1L_VAL@h
	ori	r4,r4,IBAT1L_VAL@l
	lis	r3,IBAT1U_VAL@h
	ori	r3,r3,IBAT1U_VAL@l
	mtspr	ibat1l,r4
	mtspr	ibat1u,r3
	isync

	lis	r4,DBAT1L_VAL@h
	ori	r4,r4,DBAT1L_VAL@l
	lis	r3,DBAT1U_VAL@h
	ori	r3,r3,DBAT1U_VAL@l
	mtspr	dbat1l,r4
	mtspr	dbat1u,r3
	isync

	lis	r4,IBAT2L_VAL@h
	ori	r4,r4,IBAT2L_VAL@l
	lis	r3,IBAT2U_VAL@h
	ori	r3,r3,IBAT2U_VAL@l
	mtspr	ibat2l,r4
	mtspr	ibat2u,r3
	isync

	lis	r4,DBAT2L_VAL@h
	ori	r4,r4,DBAT2L_VAL@l
	lis	r3,DBAT2U_VAL@h
	ori	r3,r3,DBAT2U_VAL@l
	mtspr	dbat2l,r4
	mtspr	dbat2u,r3
	isync

	lis	r4,IBAT3L_VAL@h
	ori	r4,r4,IBAT3L_VAL@l
	lis	r3,IBAT3U_VAL@h
	ori	r3,r3,IBAT3U_VAL@l
	mtspr	ibat3l,r4
	mtspr	ibat3u,r3
	isync

	lis	r4,DBAT3L_VAL@h
	ori	r4,r4,DBAT3L_VAL@l
	lis	r3,DBAT3U_VAL@h
	ori	r3,r3,DBAT3U_VAL@l
	mtspr	dbat3l,r4
	mtspr	dbat3u,r3
	isync

	lis	r4,IBAT4L_VAL@h
	ori	r4,r4,IBAT4L_VAL@l
	lis	r3,IBAT4U_VAL@h
	ori	r3,r3,IBAT4U_VAL@l
	mtspr	ibat4l,r4
	mtspr	ibat4u,r3
	isync

	lis	r4,DBAT4L_VAL@h
	ori	r4,r4,DBAT4L_VAL@l
	lis	r3,DBAT4U_VAL@h
	ori	r3,r3,DBAT4U_VAL@l
	mtspr	dbat4l,r4
	mtspr	dbat4u,r3
	isync

	lis	r4,IBAT5L_VAL@h
	ori	r4,r4,IBAT5L_VAL@l
	lis	r3,IBAT5U_VAL@h
	ori	r3,r3,IBAT5U_VAL@l
	mtspr	ibat5l,r4
	mtspr	ibat5u,r3
	isync

	lis	r4,DBAT5L_VAL@h
	ori	r4,r4,DBAT5L_VAL@l
	lis	r3,DBAT5U_VAL@h
	ori	r3,r3,DBAT5U_VAL@l
	mtspr	dbat5l,r4
	mtspr	dbat5u,r3
	isync

	lis	r4,IBAT6L_VAL@h
	ori	r4,r4,IBAT6L_VAL@l
	lis	r3,IBAT6U_VAL@h
	ori	r3,r3,IBAT6U_VAL@l
	mtspr	ibat6l,r4
	mtspr	ibat6u,r3
	isync

	lis	r4,DBAT6L_VAL@h
	ori	r4,r4,DBAT6L_VAL@l
	lis	r3,DBAT6U_VAL@h
	ori	r3,r3,DBAT6U_VAL@l
	mtspr	dbat6l,r4
	mtspr	dbat6u,r3
	isync

	lis	r4,IBAT7L_VAL@h
	ori	r4,r4,IBAT7L_VAL@l
	lis	r3,IBAT7U_VAL@h
	ori	r3,r3,IBAT7U_VAL@l
	mtspr	ibat7l,r4
	mtspr	ibat7u,r3
	isync

	lis	r4,DBAT7L_VAL@h
	ori	r4,r4,DBAT7L_VAL@l
	lis	r3,DBAT7U_VAL@h
	ori	r3,r3,DBAT7U_VAL@l
	mtspr	dbat7l,r4
	mtspr	dbat7u,r3
	isync

	// BATs are now set up, now invalidate tlb entries
	lis	r3,0x0000

	lis	r5,0x2	// set up high bound of 0x00020000 for 603e
	isync

	// Recall that in order to invalidate TLB entries, the value issued to
	// tlbie must increase the value in bits 15:19(603e)
	// by one each iteration.
tlblp:
	tlbie	r3
	sync

	addi	r3,r3,0x1000
	cmp	0,0,r3,r5	// check if all TLBs invalidated yet
	blt	tlblp
	blr

/*--------------------------------------------------------------------------
   Function    : invalidate_and_enable_dcache
   Description : Invalidate data chache and then enable it.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
invalidate_and_enable_dcache:
	mfspr	r5,hid0
	ori	r5,r5,0x4400
	sync
	mtspr	hid0,r5
	rlwinm	r6,r5,0,22,20
	mtspr	hid0,r6
	blr

/*--------------------------------------------------------------------------
   Function    : invalidate_and_enable_icache
   Description : Invalidate instruction chache and then enable it.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
invalidate_and_enable_icache:
	mfspr	r5,hid0
	ori	r5,r5,0x8800
	isync
	mtspr	hid0,r5
	rlwinm	r6,r5,0,21,19
	mtspr	hid0,r6
	blr

/*--------------------------------------------------------------------------
   Function    : address_translation_on
   Description : Enable address translation using the MMU.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
address_translation_on:
	mfmsr	r5
	ori	r5,r5,0x0030
	mtmsr	r5
	isync
	blr

/*--------------------------------------------------------------------------
   Function    : __eabi
   Description : Replaces standard __eabi(). This is a minimal __eabi,
                 because we don't require anything to happen here other than
                 setting up the SDA pointers.
   Parameter(s): none
   Returns     : nothing
  --------------------------------------------------------------------------*/
__eabi:
	// Get small data area locations as per PPC EABI.
	// See http://motorola.com/semiconductors (search on EABI)
	// for more information.

	lis	r13,_SDA_BASE_@h
	ori	r13,r13,_SDA_BASE_@l
	lis	r2,_SDA2_BASE_@h
	ori	r2,r2,_SDA2_BASE_@l
	
	blr

.end