/*
 * Copyright 2013, 2016 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.palI14443p4a
{
    #region BASE

    #region INTERFACE
    /// <summary>
    /// Interface definition of interfaces a PAL object needs to implement.
    /// </summary>
    public interface IPalI14443p4a
    {
        /// <summary>
        /// Perform Rats.
        /// </summary>
        /// <param name="bFsdi"></param>
        /// <param name="bCid"></param>
        /// <param name="pAts"></param>
        /// <returns></returns>
        Status_t Rats (
            byte bFsdi,       /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,        /**< [In] Card Identifier; 0-14 */
            out byte[] pAts   /**< [Out] Answer to Select; uint8_t[255] */
            );

        /// <summary>
        /// Perform Pps.
        /// </summary>
        /// <param name="bDri"></param>
        /// <param name="bDsi"></param>
        /// <returns></returns>
        Status_t Pps (
            byte bDri,         /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            );

        /// <summary>
        /// Perform ActivateCard.
        /// </summary>
        /// <param name="bFsdi"></param>
        /// <param name="bCid"></param>
        /// <param name="bDri"></param>
        /// <param name="bDsi"></param>
        /// <param name="pAts"></param>
        /// <returns></returns>
        Status_t ActivateCard (
            byte bFsdi,       /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,        /**< [In] Card Identifier; 0-14 */
            byte bDri,        /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,        /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            out byte[] pAts   /**< [Out] Answer to Select; uint8_t[255] */
            );

        /// <summary>
        /// Perform GetProtocolParams.
        /// </summary>
        /// <param name="pAtqs"></param>
        /// <returns></returns>
        Status_t GetProtocolParams (
            out byte pCidEnabled,      /**< [Out] Unequal '0' if Card Identifier is enabled. */
            out byte pCid,             /**< [Out] Card Identifier. */
            out byte pNadSupported,    /**< [Out] Node Address Support; Unequal '0' if supported. */
            out byte pFwi,             /**< [Out] Frame Waiting Integer. */
            out byte pFsdi,            /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            out byte pFsci             /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            );
    }

    #endregion
    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Rats (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            byte[] pAts         /**< [Out] Answer to Select; uint8_t[1-254] */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Pps (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi           /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_ActivateCard (
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            byte bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            byte[] pAts         /**< [Out] Answer to Select; uint8_t[1-254] */
            );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_GetProtocolParams (
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ref byte pCidEnabled,   /**< [Out] Unequal '0' if Card Identifier is enabled. */
            ref byte pCid,          /**< [Out] Card Identifier. */
            ref byte pNadSupported, /**< [Out] Node Address Support; Unequal '0' if supported. */
            ref byte pFwi,          /**< [Out] Frame Waiting Integer. */
            ref byte pFsdi,         /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            ref byte pFsci          /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Rats (
            byte bFsdi,     /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,      /**< [In] Card Identifier; 0-14 */
            out byte[] pAts /**< [Out] Answer to Select; uint8_t[1-254] */
            )
        {
            Status_t status;
            pAts = new byte[254];
            status = phpalI14443p4a_Rats ( m_pDataParams, bFsdi, bCid, pAts );

            /* Resize Ats buffer to actual Ats size */
            if ( status.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize<byte> ( ref pAts, pAts[0] );
            }

            return status;
        }

        public Status_t Pps (
            byte bDri,  /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi   /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            )
        {
            return phpalI14443p4a_Pps ( m_pDataParams, bDri, bDsi );
        }

        public Status_t ActivateCard (
            byte bFsdi,     /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,      /**< [In] Card Identifier; 0-14 */
            byte bDri,      /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,      /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            out byte[] pAts /**< [Out] Answer to Select; uint8_t[1-254] */
            )
        {
            Status_t status;
            pAts = new byte[254];
            status = phpalI14443p4a_ActivateCard ( m_pDataParams, bFsdi, bCid, bDri, bDsi, pAts );

            /* Resize Ats buffer to actual Ats size */
            if ( status.Equals ( Error_Gen.SUCCESS ) )
            {
                Array.Resize<byte> ( ref pAts, pAts[0] );
            }

            return status;
        }

        public Status_t GetProtocolParams (
            out byte pCidEnabled,   /**< [Out] Unequal '0' if Card Identifier is enabled. */
            out byte pCid,          /**< [Out] Card Identifier. */
            out byte pNadSupported, /**< [Out] Node Address Support; Unequal '0' if supported. */
            out byte pFwi,          /**< [Out] Frame Waiting Integer. */
            out byte pFsdi,         /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            out byte pFsci          /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            )
        {
            pCidEnabled = 0;
            pCid = 0;
            pNadSupported = 0;
            pFwi = 0;
            pFsdi = 0;
            pFsci = 0;

            return phpalI14443p4a_GetProtocolParams (
                m_pDataParams,
                ref pCidEnabled,
                ref pCid,
                ref pNadSupported,
                ref pFwi,
                ref pFsdi,
                ref pFsci );
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI14443p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!             */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer */
            public byte bCidSupported;      /**< Cid Support indicator; Unequal '0' if supported.           */
            public byte bNadSupported;      /**< Nad Support indicator; Unequal '0' if supported.           */
            public byte bCid;               /**< Card Identifier; Ignored if bCidSupported is equal '0'.    */
            public byte bBitRateCaps;       /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4     */
            public byte bFwi;               /**< Frame Waiting Integer.                                     */
            public byte bFsci;              /**< PICC Frame Size Integer; 0-0xC                               */
            public byte bFsdi;              /**< (Current) PCD Frame Size Integer; 0-0xC                      */
            public byte bDri;               /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3       */
            public byte bDsi;               /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3          */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Sw_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p4a_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI14443p4a_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region SamAV2_X

    public class SamAV2_X : palI14443p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;             /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;  /**< Pointer to the parameter structure of the underlying layer. */
            byte bCidSupported;     /**< Cid Support indicator; Unequal '0' if supported. */
            byte bNadSupported;     /**< Nad Support indicator; Unequal '0' if supported. */
            byte bCid;              /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
            byte bBitRateCaps;      /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4. */
            byte bFwi;              /**< Frame Waiting Integer. */
            byte bFsci;             /**< PICC Frame Size Integer; 0-8. */
            byte bDri;              /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi;              /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_SamAV2_X_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.SamAV2 pHal )
        {
            return phpalI14443p4a_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV2_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV2_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3_X

    public class SamAV3_X : palI14443p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;             /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;  /**< Pointer to the parameter structure of the underlying layer. */
            byte bCidSupported;     /**< Cid Support indicator; Unequal '0' if supported. */
            byte bNadSupported;     /**< Nad Support indicator; Unequal '0' if supported. */
            byte bCid;              /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
            byte bBitRateCaps;      /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4. */
            byte bFwi;              /**< Frame Waiting Integer. */
            byte bFsci;             /**< PICC Frame Size Integer; 0-8. */
            byte bDri;              /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3. */
            byte bDsi;              /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_SamAV3_X_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.SamAV3 pHal )
        {
            return phpalI14443p4a_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
#endif

    #region Rd710

    public class Rd710 : palI14443p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!             */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Rd710_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,      /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Rd710 pHal )
        {
            return phpalI14443p4a_Rd710_Init (
                                            ref m_DataParamsInt[0],
                                            ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                            pHal.m_pDataParams );
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Rd710 pHal)
        {
            return phpalI14443p4a_Rd710_Init(
                                            ref m_DataParamsInt[0],
                                            (ushort)wDataParamSize,
                                            pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Rd710 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Rd710 ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region PCSC

    public class PCSC : palI14443p4a.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY!             */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer */
            public IntPtr pBalDataParams;   /**< pointer to the parameter structure of the underlying pcsc bal. Is used to access the pcsc parameter */
            public IntPtr pHalPcscDataParams;   /**< pointer to the parameter structure of the underlying pcsc hal. Is used to access the pcsc parameter */
            public byte bCidSupported;      /**< Cid Support indicator; Unequal '0' if supported. */
            public byte bNadSupported;      /**< Nad Support indicator; Unequal '0' if supported. */
            public byte bCid;               /**< Card Identifier; Ignored if bCidSupported is equal '0'. */
            public byte bBitRateCaps;       /**< TA(1) byte of ATS; ISO/IEC 14443-4:2008, Section 5.2.4. */
            public byte bFwi;               /**< Frame Waiting Integer. */
            public byte bFsci;              /**< PICC Frame Size Integer; 0-0xC; */
            public byte bFsdi;              /**< (Current) PCD Frame Size Integer; 0-0xC; */
            public byte bDri;               /**< (Current) Divisor Receive (PCD to PICC) Integer; 0-3; */
            public byte bDsi;               /**< (Current) Divisor Send (PICC to PCD) Integer; 0-3; */
            public byte bActStatus;         /**< Get the Activated Status; */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Pcsc_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init ( Hal.Generic pHal )
        {
            return phpalI14443p4a_Pcsc_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI14443p4a_Pcsc_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public PCSC ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        ~PCSC ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL
    #region Callback
    /// <summary>
    /// Class for Callback layer initialization interface and data params.
    /// </summary>
    public class Callback : palI14443p4a.Generic
    {
        #region Data Structure
        /// <summary>
        /// PAL-ISO14443P4A parameter structure for Callback reader.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pRatsCb;
            IntPtr pPpsCb;
            IntPtr pActivateCardCb;
            IntPtr pGetProtocolParamsCb;
            IntPtr pCbContext;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Callback_Init (
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pRatsCb,
            IntPtr pPpsCb,
            IntPtr pActivateCardCb,
            IntPtr pGetProtocolParamsCb,
            IntPtr pCbContext );
        #endregion DLL Imports

        #region DELEGATES

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort RatsDelegate (
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            IntPtr pAts,        /**< [Out] Answer to Select; uint8_t[255] */
            IntPtr pContext     /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort PpsDelegate (
            byte bDri,         /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,         /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            IntPtr pContext    /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort ActivateCardDelegate (
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            byte bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            IntPtr pAts,        /**< [Out] Answer to Select; uint8_t[255] */
            IntPtr pContext     /**< [Out] User Defined Context. */
            );

        [UnmanagedFunctionPointer ( CallingConvention.Cdecl )]
        private delegate ushort GetProtocolParamsDelegate (
            ref byte pCidEnabled,      /**< [Out] Unequal '0' if Card Identifier is enabled. */
            ref byte pCid,             /**< [Out] Card Identifier. */
            ref byte pNadSupported,    /**< [Out] Node Address Support; Unequal '0' if supported. */
            ref byte pFwi,             /**< [Out] Frame Waiting Integer. */
            ref byte pFsdi,            /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            ref byte pFsci,            /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            IntPtr pContext            /**< [Out] User Defined Context. */
            );

        #endregion DELEGATES

        #region CALLBACKS

        private ushort cbRats (
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            IntPtr pAts,        /**< [Out] Answer to Select; uint8_t[255] */
            IntPtr pContext     /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtsTmp;

            if ( pAts == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P4A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.Rats ( bFsdi, bCid, out pAtsTmp );

            if ( pAtsTmp != null && pAtsTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtsTmp.Length > 255 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P4A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtsTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAts, dIndex, pAtsTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbPps (
            byte bDri,         /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,         /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            IntPtr pContext    /**< [Out] User Defined Context. */
            )
        {
            // Execute callback function
            return m_ExtPal.Pps ( bDri, bDsi );
        }

        private ushort cbActivateCard (
            byte bFsdi,         /**< [In] Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames) */
            byte bCid,          /**< [In] Card Identifier; 0-14 */
            byte bDri,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
            byte bDsi,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
            IntPtr pAts,        /**< [Out] Answer to Select; uint8_t[255] */
            IntPtr pContext     /**< [Out] User Defined Context. */
            )
        {
            Status_t status;
            byte[] pAtsTmp;

            if ( pAts == IntPtr.Zero )
            {
                return new Status_t ( Error_CompCode.PAL_ISO14443P4A, Error_Comm.INVALID_PARAMETER );
            }

            // Execute callback function
            status = m_ExtPal.ActivateCard ( bFsdi, bCid, bDri, bDsi, out pAtsTmp );

            if ( pAtsTmp != null && pAtsTmp.Length != 0 )
            {
                // Buffer overflow check
                if ( pAtsTmp.Length > 255 )
                {
                    return new Status_t ( Error_CompCode.PAL_ISO14443P4A, Error_Comm.BUFFER_OVERFLOW );
                }

                for ( int dIndex = 0; dIndex < pAtsTmp.Length; dIndex++ )
                {
                    Marshal.WriteByte ( pAts, dIndex, pAtsTmp[dIndex] );
                }
            }
            return status;
        }

        private ushort cbGetProtocolParams (
            ref byte pCidEnabled,      /**< [Out] Unequal '0' if Card Identifier is enabled. */
            ref byte pCid,             /**< [Out] Card Identifier. */
            ref byte pNadSupported,    /**< [Out] Node Address Support; Unequal '0' if supported. */
            ref byte pFwi,             /**< [Out] Frame Waiting Integer. */
            ref byte pFsdi,            /**< [Out] PCD Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            ref byte pFsci,            /**< [Out] PICC Frame Size Integer; 0-8 (0xC for SW implementation to support 4k frames). */
            IntPtr pContext            /**< [Out] User Defined Context. */
            )
        {
            return m_ExtPal.GetProtocolParams ( out pCidEnabled, out pCid, out pNadSupported, out pFwi, out pFsdi, out pFsci );
        }

        #endregion

        #region Initialization
        // Storage for internal callbacks
        private List<Delegate> m_Callbacks;
        private palI14443p4a.IPalI14443p4a m_ExtPal;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p4a.IPalI14443p4a ExtPal
            )
        {
            return Init ( ExtPal, IntPtr.Zero );
        }

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="ExtPal"></param>
        /// <param name="cbContext"></param>
        /// <returns></returns>
        public Status_t Init (
            palI14443p4a.IPalI14443p4a ExtPal,
            IntPtr cbContext
            )
        {
            // Allocate internal callback delegates
            m_Callbacks = new List<Delegate> ();
            m_Callbacks.Add ( new RatsDelegate ( cbRats ) );
            m_Callbacks.Add ( new PpsDelegate ( cbPps ) );
            m_Callbacks.Add ( new ActivateCardDelegate ( cbActivateCard ) );
            m_Callbacks.Add ( new GetProtocolParamsDelegate ( cbGetProtocolParams ) );

            // Store reference to external hal object
            m_ExtPal = ExtPal;

            // Call init. function
            return phpalI14443p4a_Callback_Init (
                ref m_DataParamsInt[0],
                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[0] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[1] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[2] ),
                Marshal.GetFunctionPointerForDelegate ( m_Callbacks[3] ),
                cbContext );
        }

        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Callback ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Callback ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping
    }
    #endregion Callback

    #region Sam_X
    public class Sam_X : Generic
    {
        #region Data Structure
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pHalDataParams;
            byte bCidSupported;
            byte bNadSupported;
            byte bCid;
            byte bBitRateCaps;
            byte bFwi;
            byte bFsci;
            byte bDri;
            byte bDsi;
        };
        #endregion

        #region Dll Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phpalI14443p4a_Sam_X_Init ( ref DataParams_t m_pDataParams,
            ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        public Status_t Init ( Hal.Sam oHal )
        {
            return phpalI14443p4a_Sam_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        public Sam_X ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        ~Sam_X ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion
    }
    #endregion
#endif
}
