/*
 * Copyright 2015 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// SamAV3 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class SamAV3 : Hal.Generic
    {
        #region Macros
        public const int DEFAULT_TIMEOUT = 150;     /**< Default timeout in microseconds    */
        public const int SHADOW_COUNT = 0x000F;     /**< Number of shadowed configurations. */

        public const int RESERVED_TX_BUFFER_LEN = 6;    /**< Amount of needed and reserved memory for the protocol overhead. */
        public const int RESERVED_RX_BUFFER_LEN = 2;    /**< Amount of needed and reserved memory for the protocol overhead. */

        public const int MFP_AUTHENTICATE_PART2_CAPBILITIY_SIZE = 6;
        #endregion Macros

        #region Enumerations
        #region Error Codes
        /// <summary>
        /// SamAV3 Custom Error Codes for the respective status codes returned by Sam hardwre.
        /// For the comments mentioned with RESP.* refer Sam AV3 document.
        /// </summary>
        public enum Error : byte
        {
            #region Error Codes Compatible to Sam AV2
            /// <summary>
            /// EEProm failure.
            /// Custom error code for RESP.ISO6400
            /// </summary>
            HW_EEPROM = ( CustomCodes.ERROR_BEGIN + 0 ),

            /// <summary>
            /// RC failure.
            /// Custom error code for RESP.ISO6401
            /// </summary>
            HW_RC5XX = ( CustomCodes.ERROR_BEGIN + 1 ),

            /// <summary>
            /// Key creation failure.
            /// Custom error code for RESP.ISO6501
            /// </summary>
            KEY_CREATE_FAILED = ( CustomCodes.ERROR_BEGIN + 2 ),

            /// <summary>
            /// SAM key reference error.
            /// Custom error code for RESP.ISO6502
            /// </summary>
            KEY_REF_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 3 ),

            /// <summary>
            /// SAM key usage counter error.
            /// Custom error code for RESP.SO6503
            /// </summary>
            KEY_KUC_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 4 ),

            /// <summary>
            /// Memory failure.
            /// Custom error code for RESP.ISO6581
            /// </summary>
            HW_EE_HIGH_VOLTAGE = ( CustomCodes.ERROR_BEGIN + 5 ),

            /// <summary>
            /// Wrong length; no further indication.
            /// Custom error code for RESP.ISO6700
            /// </summary>
            ISO7816_WRONG_LENGTH_LC = ( CustomCodes.ERROR_BEGIN + 6 ),

            /// <summary>
            /// Command aborted - final chained command expected; running command aborted, new command ignored.
            /// Custom error code for RESP.ISO6883
            /// </summary>
            INCOMPLETE_CHAINING = ( CustomCodes.ERROR_BEGIN + 7 ),

            /// <summary>
            /// Security status not satisfied.
            /// Custom error code for RESP.ISO6982
            /// </summary>
            INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 8 ),

            /// <summary>
            /// Referenced data invalid.
            /// Custom error code for RESP.ISO6984
            /// </summary>
            KEY_INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 9 ),

            /// <summary>
            /// Conditions of use not satisfied.
            /// Custom error code for RESP.ISO6985
            /// </summary>
            COND_USE_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 10 ),

            /// <summary>
            /// Command not allowed.
            /// Custom error code for RESP.ISO6986
            /// </summary>
            ISO7816_COMMAND_NOT_ALLOWED = ( CustomCodes.ERROR_BEGIN + 11 ),

            /// <summary>
            /// Wrong datafield parameter.
            /// Custom error code for RESP.ISO6A80
            /// </summary>
            ISO7816_WRONG_PARAMS_FOR_INS = ( CustomCodes.ERROR_BEGIN + 12 ),

            /// <summary>
            /// Key version not found.
            /// Custom error code for RESP.ISO6A82
            /// </summary>
            KEY_VERSION_INVALID = ( CustomCodes.ERROR_BEGIN + 13 ),

            /// <summary>
            /// Host protocol error.
            /// Custom error code for RESP.ISO6A84
            /// </summary>
            HOST_PROTECTION = ( CustomCodes.ERROR_BEGIN + 14 ),

            /// <summary>
            /// Incorrect parameters P1-P2.
            /// Custom error code for RESP.ISO6A86
            /// </summary>
            ISO7816_WRONG_P1P2 = ( CustomCodes.ERROR_BEGIN + 15 ),

            /// <summary>
            /// Incorrect Le value.
            /// Custom error code for RESP.ISO6C00
            /// </summary>
            ISO7816_WRONG_LE = ( CustomCodes.ERROR_BEGIN + 16 ),

            /// <summary>
            /// Instruction code not supported or invalid or not available
            /// in the current state of the SAM.
            /// Custom error code for RESP.ISO6D00
            /// </summary>
            ISO7816_UNKNOWN_INS = ( CustomCodes.ERROR_BEGIN + 17 ),

            /// <summary>
            /// Class not supported.
            /// Custom error code for RESP.ISO6E00
            /// </summary>
            ISO7816_UNKNOWN_CLASS = ( CustomCodes.ERROR_BEGIN + 18 ),

            /// <summary>
            /// Correct execution - authentication failed.
            /// Custom error code for RESP.ISO901E
            /// </summary>
            CRYPTO = ( CustomCodes.ERROR_BEGIN + 19 ),

            /// <summary>
            /// Correct execution - MIFARE Plus PICC protocol error.
            /// Custom error code for RESP.ISO90BE
            /// </summary>
            MIFARE_PLUS_CRYPTO = ( CustomCodes.ERROR_BEGIN + 20 ),

            /// <summary>
            /// Correct execution - error code returned by MIFARE Plus PICC.
            /// Custom error code for RESP.ISO90BF
            /// </summary>
            MIFARE_PLUS_GEN = ( CustomCodes.ERROR_BEGIN + 21 ),

            /// <summary>
            /// Correct execution - UID not complete.
            /// Custom error code for RESP.ISO90C0
            /// </summary>
            ISO_UID_INCOMPLETE = ( CustomCodes.ERROR_BEGIN + 22 ),

            /// <summary>
            /// Correct execution - error code returned by DESFire PICC.
            /// Custom error code for RESP.ISO90DF
            /// </summary>
            DESFIRE_GEN = ( CustomCodes.ERROR_BEGIN + 23 ),

            /// <summary>
            /// Incorrect block number.
            /// Custom error code for RESP.ISO90EB
            /// </summary>
            ISO_WRONG_BNR = ( CustomCodes.ERROR_BEGIN + 25 ),

            /// <summary>
            /// Invalid format.
            /// Custom error code for RESP.ISO90EC
            /// </summary>
            ISO_INVALID_FORMAT = ( CustomCodes.ERROR_BEGIN + 26 ),

            /// <summary>
            /// Invalid parameter.
            /// Custom error code for RESP.ISO90ED
            /// </summary>
            ISO_INVALID_PARAMETER = ( CustomCodes.ERROR_BEGIN + 27 ),

            /// <summary>
            /// Correct execution - error code returned by PICC.
            /// Custom error code for RESP.ISO90EF
            /// </summary>
            MIFARE_GEN = ( CustomCodes.ERROR_BEGIN + 28 ),

            /// <summary>
            /// MIFARE (R) NACK 0 received.
            /// Custom error code for RESP.ISO90F0
            /// </summary>
            MIFARE_NAK0 = ( CustomCodes.ERROR_BEGIN + 29 ),

            /// <summary>
            /// MIFARE (R) NACK 1 received.
            /// Custom error code for RESP.ISO90F1
            /// </summary>
            MIFARE_NAK1 = ( CustomCodes.ERROR_BEGIN + 30 ),

            /// <summary>
            /// MIFARE (R) NACK 4 received.
            /// Custom error code for RESP.ISO90F4
            /// </summary>
            MIFARE_NAK4 = ( CustomCodes.ERROR_BEGIN + 31 ),

            /// <summary>
            /// MIFARE (R) NACK 5 received.
            /// Custom error code for RESP.ISO90F5
            /// </summary>
            MIFARE_NAK5 = ( CustomCodes.ERROR_BEGIN + 32 ),

            /// <summary>
            /// Correct execution - more date expected.
            /// Custom error code for RESP.ISO90AF
            /// </summary>
            OK_CHAINING_ACTIVE = ( CustomCodes.ERROR_BEGIN + 35 ),
            #endregion Error Codes Compatible to Sam AV2

            #region Sam AV3 Specific Error Codes
            /// <summary>
            /// Incompatible referenced data.
            /// Custom error code for RESP.ISO6781
            /// </summary>
            INCOMPLETE_REFERENCE_DATA = ( CustomCodes.ERROR_BEGIN + 36 ),

            /// <summary>
            /// Security status not satisfied.
            /// Custom error code for RESP.ISO6782
            /// </summary>
            SECURITY_STATUS_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 37 ),

            /// <summary>
            /// Secure messaging not supported.
            /// Custom error code for RESP.ISO6882
            /// </summary>
            SECURE_MESSAGING_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 38 ),

            /// <summary>
            /// Command chaining not supported.
            /// Custom error code for RESP.ISO6884
            /// </summary>
            COMMAND_CHAINING_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 39 ),

            /// <summary>
            /// Incorrect length.
            /// Custom error code for RESP.ISO6983
            /// </summary>
            INCORRECT_LENGTH = ( CustomCodes.ERROR_BEGIN + 40 ),

            /// <summary>
            /// Incorrect secure messaging data objects.
            /// Custom error code for RESP.ISO6988
            /// </summary>
            INCORRECT_SECURE_MESSAGING_DATA = ( CustomCodes.ERROR_BEGIN + 41 ),

            /// <summary>
            /// Function not supported.
            /// Custom error code for RESP.ISO6A81
            /// </summary>
            FUNCTION_NOT_SUPPORTED = ( CustomCodes.ERROR_BEGIN + 42 ),

            /// <summary>
            /// Record not found.
            /// Custom error code for RESP.ISO6A83
            /// </summary>
            RECORD_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 43 ),

            /// <summary>
            /// Referenced data or reference data not found.
            /// Custom error code for RESP.ISO6A88
            /// </summary>
            REFERENCED_DATA_NOT_FOUND = ( CustomCodes.ERROR_BEGIN + 44 ),

            /// <summary>
            /// No precise diagnosis.
            /// Custom error code for RESP.ISO6F00
            /// </summary>
            NO_PRECISE_DIAGNOSIS = ( CustomCodes.ERROR_BEGIN + 45 ),

            /// <summary>
            /// Any status bytes, as provided by the programmable logic..
            /// Custom error code for RESP.ISOxxxx
            /// </summary>
            PROGRAMMABLE_LOGIC = ( CustomCodes.ERROR_BEGIN + 46 ),

            /// <summary>
            /// Correct execution - more date expected (SAM-Host secure messaging to be applied on each command exchange)
            /// </summary>
            OK_CHAINING_ACTIVE_EXT = ( CustomCodes.ERROR_BEGIN + 47 )
            #endregion Sam AV3 Specific Error Codes
        }
        #endregion Error Codes

        #region Generic Enumerations
        #region LFI
        /// <summary>
        /// Enumeration filed to indicate the Last Frame Indicator.
        /// </summary>
        public enum LFI : byte
        {
            /// <summary> Indication of last frame. </summary>
            LAST_FRAME = 0x00,

            /// <summary> Indication of intermediate frame. More data will be sent to SAM. </summary>
            CHAINED_FRAME = 0xAF
        }
        #endregion LFI

        #region Logical Channel
        /// <summary>
        /// The locigal channel to be used for communicating with Sam. This will internally update the Class byte. These logical channel
        /// can be used to communicate with 4 differrent PICC's. Each channel maintains its own states.
        /// </summary>
        public enum LogicalChannel : byte
        {
            /// <summary> Logical Channel 0.</summary>
            LC0 = 0x00,

            /// <summary> Logical Channel 1.</summary>
            LC1 = 0x01,

            /// <summary> Logical Channel 2.</summary>
            LC2 = 0x02,

            /// <summary> Logical Channel 3.</summary>
            LC3 = 0x03
        }
        #endregion Logical Channel

        #region Configuration
        /// <summary>
        /// Gets or Sets the configuration parameter.
        /// </summary>
        public enum Config : int
        {
            /// <summary>Get / Set the Hostmode; (e.g. #PHHAL_HW_SAMAV2_HC_AV1_MODE).</summary>
            HOSTMODE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,

            /// <summary>
            /// Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
            /// Default is #PH_OFF; Only applicable in NonX-Mode.
            ///</summary>
            DISABLE_NONX_CFG_MAPPING,

            /// <summary>
            /// Configure communication type. Refer <see cref="CommunicationType"/> for supported values.
            /// </summary>
            COMMUNICATION_TYPE = 0x0060,

            /// <summary>
            /// Configure I2C Slave Address.
            /// </summary>
            I2C_SLAVE_ADDRESS = 0x0061,

            /// <summary>
            /// Configure I2C BitRate. Refer <see cref="I2C_BitRate"/> for supported values.
            /// </summary>
            I2C_BITRATE = 0x0062,

            /// <summary>
            /// Configure I2C Timeout.
            /// </summary>
            I2C_TIMEOUT_MS = 0x0063
        }
        #endregion Configuration

        #region Operational Modes
        /// <summary>
        /// Operation mode of Sam.
        /// </summary>
        public enum OpMode : byte
        {
            /// <summary> Option to iniitalize Sam communication in Non X ( S ) mode. </summary>
            NON_X = 0x00,

            /// <summary> Option to iniitalize Sam communication in X mode with RC523 reader. </summary>
            X_RC523 = 0x01,

            /// <summary> Option to iniitalize Sam communication in X mode with RC663 reader. </summary>
            X_RC663 = 0x02
        }
        #endregion Operational Modes

        #region Custom Exchange bits
        /// <summary>
        /// Custom exchange bit used for custom exchaining.
        /// </summary>
        public enum CustomExchangeBit : int
        {
            NO_ENCIPHERING_BIT = 0x0010,
            NO_DECIPHERING_BIT = 0x0020
        }
        #endregion Custom Exchange bits
        #endregion  Generic Enumerations

        #region Command specific enumerations
        #region Host Communication
        #region Cmd.SAM_LockUnlock
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_LockUnlock command.
        /// </summary>
        public enum LockUnlock : byte
        {
            /// <summary> Option mask for Sub-command type as unlock. </summary>
            UNLOCK = 0x00,

            /// <summary> Option mask for Sub-command type as lock without specifying unlock key. </summary>
            LOCK_NO_KEY = 0x01,

            /// <summary> Option mask for Sub-command type as lock with specifying unlock key. </summary>
            LOCK_KEY = 0x02,

            /// <summary> Option mask for Sub-command type as activate MIFARE SAM to AV3. </summary>
            ACTIVATE_SAM = 0x03,

            /// <summary> Option mask for Sub-command type as unlock PL. </summary>
            UNLOCK_PL = 0x04
        }
        #endregion Cmd.SAM_LockUnlock

        #region Cmd.SAM_AuthenticateHost
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_AuthenticateHost command.
        /// </summary>
        public enum HostMode : byte
        {
            /// <summary> Option mask for protection mode as plain. </summary>
            PLAIN = 0x00,

            /// <summary> Option mask for protection mode as MAC protection. </summary>
            MAC = 0x01,

            /// <summary> Option mask for protection mode as Full protection. </summary>
            FULL = 0x02
        }

        /// <summary>
        /// Option macros to indicate the version of Sam.
        /// </summary>
        public enum HcMode : byte
        {
            /// <summary> Option to represent that the Sam hardware is in AV2 mode or its an AV2 hardware. </summary>
            AV2 = 0x02,

            /// <summary> Option to represent that the Sam hardware is in AV3 mode. This is application for both activated and unactivated state. </summary>
            AV3 = 0x03
        }
        #endregion Cmd.SAM_AuthenticateHost
        #endregion Host Communication

        #region Security and Configuration
        #region Cmd.SAM_DisableCrypto
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_DisableCrypto command.
        /// </summary>
        public enum DisableCrypto : ushort
        {
            /// <summary> Option mask for Disable Crypto with no change for programming mask bit. </summary>
            NO_CHANGE = 0x0000,

            /// <summary> Option mask for Disable Crypto to disable Desfire Key change. </summary>
            DES_PICC_CHANGE_KEY = 0x0800,

            /// <summary> Option mask for Disable Crypto to disable the decryption of data. </summary>
            DECRYPTION = 0x1000,

            /// <summary> Option mask for Disable Crypto to disable encryption of data. </summary>
            ENCRYPTION = 0x2000,

            /// <summary> Option mask for Disable Crypto to disable verification of MAC. </summary>
            MAC_VERIFICATION = 0x4000,

            /// <summary> Option mask for Disable Crypto to disable generation of MAC. </summary>
            MAC_GENERATION = 0x8000
        }
        #endregion Cmd.SAM_DisableCrypto

        #region Cmd.SAM_ActivateOfflineKey
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_ActivateOffline command.
        /// </summary>
        public enum LRPUpdate
        {
            /// <summary>
            /// Option mask for ActivateOffline with P1 information Bit 1 set to zero in case of
            /// non AES_128 LPR keytype.
            /// </summary>
            RFU = 0,

            /// <summary>
            /// Option mask for ActivateOffline with LRP Update keys to generate One
            /// updated key (KeyID.LRPUpdate).
            /// </summary>
            ONE = 0,

            /// <summary>
            /// Option mask for ActivateOffline with LRP Update keys to generate Two
            /// updated key (KeyID.LRPMACUpdate and KeyID.LRPENCUpdate).
            /// </summary>
            TWO = 2
        }
        #endregion Cmd.SAM_ActivateOfflineKey

        #region Cmd.SAM_LoadInitVector
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.LoadInitVector command.
        /// </summary>
        public enum ModeIndicator : byte
        {
            /// <summary> Option mask for Load Init Vector to set the IV. </summary>
            SET_IV = 0x00,

            /// <summary> Option mask for Load Init Vector to set the LRP_EncCtr. </summary>
            SET_LRP_ENC_CTR = 0x01
        }
        #endregion Cmd.SAM_LoadInitVector

        #region Cmd.SAM_KillAuthentication
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_KillAuthentication command.
        /// </summary>
        public enum KillAuth : byte
        {
            /// <summary> Option mask for killing any authentication on corresponding LC. </summary>
            FULL,

            /// <summary> Option mask for killing PICC or offline key activation but preserving any Host Authentication. </summary>
            PARTIAL
        }
        #endregion Cmd.SAM_KillAuthentication

        #region Cmd.SAM_SetConfiguration
        /// <summary>
        /// Option macros for Sam AV3 Security and Configuration Cmd.SAM_SetConfiguration command.
        /// </summary>
        public enum SetConfiguration : byte
        {
            /// <summary> Option mask for exchanging the historical bytes. </summary>
            HISTORICAL_BYTES = 0x00,

            /// <summary> Option mask for exchanging the reader IC configuration. </summary>
            READER_IC_CONFIG = 0x01,

            /// <summary> Option mask for exchanging the I2C processing clock speed configuration. </summary>
            I2C_CLOCK_SPEED = 0x02,

            /// <summary> Option mask for exchanging the Default EMV Change keys configuration. </summary>
            EMV_CHANGE_KEYS = 0x03,

            /// <summary> Option mask for exchanging the Full ATR configuration. </summary>
            FULL_ATR = 0x04,
        }

        /// <summary>
        /// The configuration data if SetConfiguration Option is READER_IC_CONFIG.
        /// </summary>
        public enum ReaderIC : byte
        {
            /// <summary> Option mask for exchanging the reader IC configuration as RC512. </summary>
            READER_IC_CONFIG_RC512 = 0x01,

            /// <summary> Option mask for exchanging the reader IC configuration as RC523. </summary>
            READER_IC_CONFIG_RC523 = 0x02,

            /// <summary> Option mask for exchanging the reader IC configuration as RC663. </summary>
            READER_IC_CONFIG_RC663 = 0x03,
        }
        #endregion Cmd.SAM_SetConfiguration
        #endregion Security and Configuration

        #region Key Management
        #region Cmd.SAM_ChangeKeyEntry, Cmd.SAM_ChangeKUCEntry
        /// <summary>
        /// Option macros for Sam AV3 Key Management Cmd.SAM_ChangeKeyEntry command.
        /// </summary>
        public enum ProMas : byte
        {
            #region Cmd.SAM_ChangeKeyEntry
            /// <summary>Option mask for updating key number and version A.</summary>
            UPDATE_KEY_VA = 0x80,

            /// <summary>Option mask for updating key number and version B.</summary>
            UPDATE_KEY_VB = 0x40,

            /// <summary>Option mask for updating key number and version C.</summary>
            UPDATE_KEY_VC = 0x20,

            /// <summary>Option mask for updating Desfire application identifier.</summary>
            UPDATE_DF_AID = 0x10,

            /// <summary>Option mask for updating key number and version of change entry key (CEK).</summary>
            UPDATE_KEY_CEK = 0x08,

            /// <summary>Option mask for updating reference key usage counter.</summary>
            UPDATE_REF_NO_KUC = 0x04,

            /// <summary>Option mask for updating SET and Extended SET.</summary>
            UPDATE_SET_EXTSET = 0x02,

            /// <summary>Option mask for specifying the versions in the data field after SET and ExtSET.</summary>
            INCLUDE_VERSION = 0x01,
            #endregion Cmd.SAM_ChangeKeyEntry

            #region Cmd.SAM_ChangeKUCEntry
            /// <summary>Option mask for updating the limit.</summary>
            UPDATE_LIMIT = 0x80,

            /// <summary>Option mask for updating CKUC key number.</summary>
            UPDATE_KEY_NO_CKUC = 0x40,

            /// <summary>Option mask for updating the VCKUC.</summary>
            UPDATE_VCKUC = 0x20,
            #endregion Cmd.SAM_ChangeKUCEntry
        }
        #endregion Cmd.SAM_ChangeKeyEntry, Cmd.SAM_ChangeKUCEntry

        #region Cmd.SAM_GetKeyEntry
        /// <summary>
        /// Option macros for Sam AV3 Key Management Cmd.SAM_GetKeyEntry command.
        /// </summary>
        public enum KeyEntryFormat : byte
        {
            /// <summary>Option mask for enabling Sam AV2 Format as key entry format.</summary>
            SAM_AV2 = 0x00,

            /// <summary>Option mask for enabling New Format as key entry format.</summary>
            NEW = 0x01,

            /// <summary>Option mask for receiving the Ram key.</summary>
            RAM_KEY = 0x00
        }
        #endregion Cmd.SAM_GetKeyEntry

        #region Cmd.SAM_DumpSessionKey, Cmd.SAM_DumpSecretKey
        /// <summary>
        /// Option macros for Sam AV3 Key Management Cmd.SAM_DumpSessionKey and Cmd.SAM_DumpSecretKey command.
        /// </summary>
        public enum DumpMode : byte
        {
            /// <summary>Option mask for dumping session or seceret keys in plain.</summary>
            PLAIN = 0x00,

            /// <summary>Option mask for dumping session or seceret keys in enciphered.</summary>
            ENCIPHERED = 0x01,

            /// <summary>Option mask for disabling the diversification of the provided key.</summary>
            DIVERSIFICATION_OFF = 0x00,

            /// <summary>Option mask for enabling the diversification of the provided key.</summary>
            DIVERSIFICATION_ON = 0x02
        }
        #endregion Cmd.SAM_DumpSessionKey, Cmd.SAM_DumpSecretKey

        #region Cmd.SAM_EncipherKeyEntry
        /// <summary>
        /// Option macros for Sam AV3 Key Management Cmd.SAM_EncipherKeyEntry command.
        /// </summary>
        public enum EncipherKeyEntry : byte
        {
            DIV_OFF = 0x00, /**< Option mask to exclude the diversification input in command frame. */
            DIV_ON = 0x01,  /**< Option mask to include the diversification input in command frame. */
            SAM_UID_OFF = 0x00, /**< Option mask to exclude the Sam UID in command frame. */
            SAM_UID_ON = 0x02   /**< Option mask to include the Sam UID in command frame. */
        }
        #endregion Cmd.SAM_EncipherKeyEntry

        #region Cmd.SAM_Generate_MFCLicMAC
        /// <summary>
        /// Option macros for Sam AV3 Key Management Cmd.SAM_Generate_MFCLicMAC command.
        /// </summary>
        public enum MFCLicMAC : byte
        {
            /// <summary> Option mask to exclude the diversification input in command frame. </summary>
            DIV_OFF = 0x00,

            /// <summary> Option mask to include the diversification input in command frame. </summary>
            DIV_ON = 0x01
        }
        #endregion Cmd.SAM_Generate_MFCLicMAC
        #endregion Key Management

        #region Data Processing
        #region Cmd.SAM_ApplySM
        /// <summary>
        ///
        /// </summary>
        public enum Offset : byte
        {
            /// <summary> . </summary>
            EXCLUDE = 0x00,

            /// <summary> . </summary>
            INCLUDE = 0x80
        }
        #endregion Cmd.SAM_ApplySM

        #region Cmd.SAM_ApplySM, Cmd.SAM_RemoveSM
        /// <summary>
        /// Option macros for Sam AV3 Data Processing Cmd.SAM_Apply_SM and Cmd.SAM_Remove_SM command
        /// </summary>
        public enum CommMode : byte
        {
            /// <summary> Option mask for communication mode as plain for Cmd.SAM_Apply_SM command. </summary>
            PLAIN = 0x00,

            /// <summary> Option mask for communication mode as MAC for Cmd.SAM_Apply_SM and Cmd.SAM_Remove_SM command. </summary>
            MAC = 0x10,

            /// <summary> Option mask for communication mode as FULL for Cmd.SAM_Apply_SM and Cmd.SAM_Remove_SM command. </summary>
            FULL = 0x30
        }
        #endregion Cmd.SAM_ApplySM, Cmd.SAM_RemoveSM

        #region Cmd.SAM_VerifyMAC, Cmd.SAM_GenerateMAC
        /// <summary>
        /// Option macros for Sam AV3 Data Processing Cmd.SAM_Verify_MAC and Cmd.SAM_GenerateMAC command.
        /// </summary>
        public enum TruncationMode : byte
        {
            /// <summary> Option mask for truncation mode as standard trunction. </summary>
            STANDARD = 0x00,

            /// <summary> Option mask for truncation mode as MFP trunction. </summary>
            MFP = 0x80
        }
        #endregion Cmd.SAM_VerifyMAC, Cmd.SAM_GenerateMAC

        #region Cmd.SAM_GenerateMAC
        /// <summary>
        /// Option macros for Sam AV3 Data Processing Cmd.SAM_GenerateMAC command.
        /// </summary>
        public enum IncludeLC : byte
        {
            /// <summary> Option mask for inclusion of LC in the command frame. </summary>
            INCLUDE = 0x80
        }
        #endregion Cmd.SAM_VerifyMAC, Cmd.SAM_GenerateMAC

        #region Cmd.SAM_DecipherData
        /// <summary>
        /// Option macros for Sam AV3 Data Processing Cmd.SAM_Decipher_Data command.
        /// </summary>
        public enum Length : byte
        {
            /// <summary> Option mask for excluding the length intformation in the command frame. </summary>
            EXCLUDE = 0x00,

            /// <summary> Option mask for including the length intformation in the command frame. </summary>
            INCLUDE = 0x80
        }
        #endregion Cmd.SAM_DecipherData
        #endregion Data Processing

        #region Public Key Infrastructure
        #region Hash Algorithm
        /// <summary>
        /// Option macros for Sam AV3 Public Key Infrastructure command.
        /// </summary>
        public enum PKI_HashAlgo
        {
            /// <summary> Option mask for SHA 1 hashing algorithm to be used. </summary>
            HASH_ALGORITHM_SHA_1 = 0x00,

            /// <summary> Option mask for SHA 224 hashing algorithm to be used. </summary>
            HASH_ALGORITHM_SHA_224 = 0x01,

            /// <summary> Option mask for SHA 256 hashing algorithm to be used. </summary>
            HASH_ALGORITHM_SHA_256 = 0x03
        }
        #endregion

        #region RSA
        #region Cmd.PKI_DecipherData
        /// <summary>
        /// Option macros for Sam AV3 Cmd.PKI_DecipherData command.
        /// </summary>
        public enum PKI_DecipherData
        {
            /// <summary> Option mask for a framing the first frame of Decipher Data command. </summary>
            FIRST_FRAME = 0x80
        }
        #endregion Cmd.PKI_DecipherData
        #endregion RSA
        #endregion Public Key Infrastructure

        #region Virtual Card
        #region SelectVC
        /// <summary>
        /// Option macros for Sam AV3 VCA / PC Virtual Card Select command.
        /// </summary>
        public enum VCA_Select : byte
        {
            /// <summary> Default option mask to disable the diversification of VcSelect MAC and ENC key. </summary>
            DEFAULT = 0x00,

            /// <summary> Option mask to disable the diversification of VC SelectENC key. </summary>
            ENC_KEY_DIV_OFF = 0x00,

            /// <summary> Option mask to perform diversification of VC SelectENC key using the diversification input provided. </summary>
            ENC_KEY_DIV_INPUT = 0x01,

            /// <summary> Option mask to disable the diversification of VC SelectMAC key. </summary>
            MAC_KEY_DIV_OFF = 0x00,

            /// <summary> Option mask to perform diversification of VC SelectMAC key using the diversification input provided. </summary>
            MAC_KEY_DIV_INPUT = 0x02,

            /// <summary> Option mask to perform diversification of VC SelectMAC key using Virtual Card Identifier. </summary>
            MAC_KEY_DIV_VCUID = 0x04,

            /// <summary> Option mask to perform diversification of VC SelectMAC key using Virtual Card Identifier and Diversification input provided. </summary>
            MAC_KEY_DIV_INPUT_VCUID = 0x06,

            /// <summary> Option mask for complete VC selection in 1 part. </summary>
            VARIANT_PART1 = 0x00,

            /// <summary> Option mask for complete VC selection in 2 part. </summary>
            VARIANT_PART2 = 0x08,
        }
        #endregion

        #region Proximity Check
        /// <summary>
        /// Option macros for Sam AV3 VCA / PC Proximity check command.
        /// </summary>
        public enum VCA_ProximityCheck : byte
        {
            /// <summary> Option mask to disable diversification of ProximityCheck key. </summary>
            DIVERSIFICATION_OFF = 0x00,

            /// <summary> Option mask to perform diversification of ProximityCheck key using the diversification input provided. </summary>
            DIVERSIFICATION_ON = 0x01,

            /// <summary> Option mask to perform Normal Cmd.VerifyPC processing. </summary>
            NORMAL_PROCESSING = 0x00,

            /// <summary> Option mask to perform Random Cmd.VerifyPC processing. </summary>
            RANDOM_PROCESSING = 0x02,

            /// <summary> Option mask to perform Proximity Check in native format. </summary>
            NATIVE_FORMAT = 0x00,

            /// <summary> Option mask to perform Proximity Check in Iso7816-4 wrapped format. </summary>
            WRAPPED_FORMAT = 0x04
        }
        #endregion
        #endregion Virtual Card

        #region MIFARE DESFire
        #region ISO Mode Selection
        /// <summary>
        /// MIFARE DESfire options to update the P2 information byte.
        /// </summary>
        public enum DF_ISOMode : byte
        {
            /// <summary> Sam Desfire ISO mode selection for Native command set. </summary>
            NATIVE = 0x00,

            /// <summary> Sam Desfire ISO mode selection for ISO 7816-4 command set. </summary>
            ISO7816_4 = 0x40,

            /// <summary> Sam Desfire ISO mode selection for Iso complaint Authentication. </summary>
            ISO_AUTHENTICATION = 0x80
        }
        #endregion ISO Mode Selection

        #region Key Diversification
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of Authenticate command.
        /// </summary>
        public enum DF_Diversification : byte
        {
            /// <summary> Sam Desfire key derivation disabled. </summary>
            DIVERSIFICATION_OFF = 0x00,

            /// <summary> Sam Desfire key derivation enabled. </summary>
            DIVERSIFICATION_ON = 0x01
        }
        #endregion Key Diversification

        #region Crypto Configuration
        /// <summary>
        /// MIFARE DESfire options to update the P2 information byte.
        /// </summary>
        public enum DF_CryptoConfig : byte
        {
            /// <summary> Sam Desfire communication mode as Plain. </summary>
            COMM_MODE_PLAIN = 0x00,

            /// <summary> Sam Desfire communication mode as MAC. </summary>
            COMM_MODE_MAC = 0x10,

            /// <summary> Sam Desfire communication mode as Full. </summary>
            COMM_MODE_FULL = 0x30,

            /// <summary> Sam DESFire chaining mode as DESFire application chaining. </summary>
            DESFIRE_CHAINING = 0x00,

            /// <summary> Sam DESFire chaining mode as ISO/IEC 14443-4 chaining. </summary>
            ISO_CHAINING = 0x40,

            /// <summary> Sam Desfire DESFire EV2 mode as legacy mode (Sam AV2). </summary>
            LEGACY_MODE = 0x00,

            /// <summary> Sam Desfire DESFire EV2 mode as EV2 mode. </summary>
            EXTENDED_OFFSET = 0x80
        }
        #endregion Crypto Configuration

        #region Authenticate command
        #region Authentication Type
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of Authenticate command.
        /// </summary>
        public enum DF_AuthType : byte
        {
            /// <summary> Sam Desfire Authentication mode as D40 and EV1. </summary>
            D40_EV1 = 0x00,

            /// <summary> Sam Desfire Authentication mode as EV2 First. </summary>
            EV2_FIRST_AUTH = 0x80,

            /// <summary> Sam Desfire Authentication mode as EV2 Non First. </summary>
            EV2_NON_FIRST_AUTH = 0xC0
        }
        #endregion Authentication Type

        #region Auth Mode
        /// <summary>
        /// Option macros for Key to be used for when EV2 Authentication is selected
        /// </summary>
        public enum DF_AuthMode : byte
        {
            /// <summary> Sam DESFire Auth mode as EV2. </summary>
            EV2 = 0x00,

            /// <summary> Sam DESFire Auth mode as LRP. </summary>
            LRP = 0x01
        }
        #endregion Auth Mode

        #region Secure Messaging
        /// <summary>
        /// Option macros for secure messaging.
        /// </summary>
        public enum DF_SecureMessaging : byte
        {
            /// <summary> Sam DESFire Allow Secure messaging. </summary>
            ALLOW = 0x00,

            /// <summary> Sam DESFire Suppress Secure messaging. </summary>
            SUPPRESS = 0x20
        }
        #endregion Secure Messaging

        #region Key Derivation Info
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of Authenticate command.
        /// </summary>
        public enum DF_KeyDerInfo : byte
        {
            /// <summary> Sam Desfire key derivation type enabled. </summary>
            ENABLED = 0x00,

            /// <summary> Sam Desfire key derivation type as NO KDF or key derivation disabled. </summary>
            NO_KDF = 0x20
        }
        #endregion Key Derivation Info

        #region Key Diversification Mode
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of Authenticate command.
        /// </summary>
        public enum DF_KeyDivMode : byte
        {
            /// <summary> Sam Desfire key derivation type as AV1. </summary>
            AV1 = 0x00,

            /// <summary> Sam Desfire key derivation type as AV1 double encryption round. </summary>
            AV1_DOUBLE_ENCRYPTION = 0x00,

            /// <summary> Sam Desfire key derivation type as AV1 single encryption round. </summary>
            AV1_SINGLE_ENCRYPTION = 0x08,

            /// <summary> Sam Desfire key derivation type as AV2. </summary>
            AV2 = 0x10,

            /// <summary> Sam Desfire key derivation type as RFU. </summary>
            RFU = 0x18
        }
        #endregion Key Diversification Mode

        #region Key Selection
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of Authenticate command.
        /// </summary>
        public enum DF_KeySel : byte
        {
            /// <summary> Sam Desfire key selection by key entry number. </summary>
            KEY_ENTRY_NUMBER = 0x00,

            /// <summary> Sam Desfire key selection by DESFIRE key number. </summary>
            DESFIRE_NUMBER_KEY_NUMBER = 0x02
        }
        #endregion Key Selection
        #endregion Authenticate command

        #region ChangeKey command
        /// <summary>
        /// MIFARE DESfire options to update the P1 information byte of ChangeKey command.
        /// </summary>
        public enum DF_KeyCompMthod : byte
        {
            /// <summary> Sam Desfire diversification method as AV1. </summary>
            DIV_METHOD_AV1 = 0x00,

            /// <summary> Sam Desfire diversification method as AV2. </summary>
            DIV_METHOD_AV2 = 0x20,

            /// <summary> Sam Desfire diversification method of current key for AV1 as double encryption. </summary>
            CURRENT_KEY_DOUBLE_ENCRYPTION = 0x00,

            /// <summary> Sam Desfire diversification method of current key for AV1 as single encryption. </summary>
            CURRENT_KEY_SINGLE_ENCRYPTION = 0x10,

            /// <summary> Sam Desfire diversification method of new key for AV1 as double encryption. </summary>
            NEW_KEY_DOUBLE_ENCRYPTION = 0x00,

            /// <summary> Sam Desfire diversification method of new key for AV1 as single encryption. </summary>
            NEW_KEY_SINGLE_ENCRYPTION = 0x08,

            /// <summary> Sam Desfire diversification usage for current key is disabled. </summary>
            CURRENT_KEY_OFF = 0x00,

            /// <summary> Sam Desfire diversification usage for current key is enabled. </summary>
            CURRENT_KEY_ON = 0x04,

            /// <summary> Sam Desfire diversification usage for new key is disabled. </summary>
            NEW_KEY_OFF = 0x00,

            /// <summary> Sam Desfire diversification usage for new key is enabled. </summary>
            NEW_KEY_ON = 0x02,

            /// <summary> Sam Desfire crypto computation mode are different for target and auth keys. </summary>
            DIFFERENT_KEY = 0x00,

            /// <summary> Sam Desfire crypto computation mode are same for target and auth keys. </summary>
            SAME_KEY = 0x01
        }

        /// <summary>
        /// MIFARE DESfire options to update the P2 information byte of ChangeKey command.
        /// </summary>
        public enum DF_Configuration : byte
        {
            /// <summary> Sam Desfire Change Key command type as Change Key. </summary>
            CHANGE_KEY = 0x00,

            /// <summary> Sam Desfire Change Key command type as Change Key EV2. </summary>
            CHANGE_KEY_EV2 = 0x20,

            /// <summary> Sam Desfire PICC Master key update to exclude key type in cryptogram. </summary>
            EXCLUDE_KEYTYPE = 0x00,

            /// <summary> Sam Desfire PICC Master key update to include key type in cryptogram. This byte will be right shifted 2 time to update the bit 4. </summary>
            INCLUDE_KEYTYPE = 0x10
        }
        #endregion ChangeKey command

        #region CreateTMFile command
        #region FileOption
        /// <summary>
        /// Option for Cmd.DESFire_CreateTMFilePICC command.
        /// </summary>
        public enum DF_FileOption : byte
        {
            /// <summary> Sam Desfire communication mode as Plain. </summary>
            PLAIN = 0x00,

            /// <summary> Sam Desfire communication mode as MAC. </summary>
            MAC = 0x01,

            /// <summary> Sam Desfire communication mode as FULL. </summary>
            FULL = 0x03
        }
        #endregion FileOption

        #region TMKeyOption
        /// <summary>
        /// Option for Cmd.DESFire_CreateTMFilePICC command.
        /// </summary>
        public enum DF_TMKeyOption : byte
        {
            /// <summary> Sam Desfire TMKeyOption as AES. </summary>
            AES = 0x02
        }
        #endregion TMKeyOption
        #endregion CreateTMFile command
        #endregion MIFARE DESFire

        #region MIFARE Plus
        #region Authenticate command
        /// <summary>
        /// MIFARE Plus options to update the P1 information byte of Authenticate command. These flags are common for both X
        /// and S mode MIFARE Plus Authenticate command.
        /// </summary>
        public enum MFP_Authenticate : byte
        {
            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFY_OFF = 0x00,

            /// <summary> Option to enable the key diversification.</summary>
            DIVERSIFY_ON = 0x01,

            /// <summary> Option to perform First authentication.</summary>
            FIRST_AUTH = 0x00,

            /// <summary> Option to perform NonFirst (following) authentication.</summary>
            NONFIRST_AUTH = 0x02,

            /// <summary> Option to set the key derivation info for SL0 or SL1 layer.</summary>
            SL1_NO_KDF = 0x00,

            /// <summary> Option to set the key derivation info for SL3 layer.</summary>
            SL3_KDF = 0x0C
        }
        #endregion Authenticate command

        #region Sector Switch Authenticate command
        /// <summary>
        /// MIFARE Plus options to update the P1 information byte of Sector Switch Authenticate command. These flags are common
        /// for both X and S mode MIFARE Plus SectorSwitchAuthenticate command.
        /// </summary>
        public enum MFP_SectorSwitchAuth : ushort
        {
            /// <summary> Option to disable the Sector Switch key diversification.</summary>
            SECTOR_SWITCH_DIV_OFF = 0x00,

            /// <summary> Option to enable the Sector Switch key diversification.</summary>
            SECTOR_SWITCH_DIV_ON = 0x01,

            /// <summary> Option to disable the Sector key diversification.</summary>
            SECTOR_DIV_OFF = 0x00,

            /// <summary> Option to enable the Sector key diversification.</summary>
            SECTOR_DIV_ON = 0x02,

            /// <summary> Option to disable the Master Sector key diversification.</summary>
            MASTER_SECTOR_DIV_OFF = 0x00,

            /// <summary> Option to enable the Master Sector key diversification with given sector number.</summary>
            MASTER_SECTOR_DIV_ON = 0x04,

            /// <summary> Option to buffer the KeyBlocks information.</summary>
            BUFFER_KEY_BLOCKS = 0x10,

            /// <summary> Option to buffer the Diversification input information.</summary>
            BUFFER_DIV_INPUT = 0x20
        }
        #endregion Sector Switch Authenticate command

        #region PDC Authenticate command
        /// <summary>
        /// MIFARE Plus options to update the P1 information byte of PDC Authenticate command. These flags are common
        /// for both X and S mode MIFARE Plus PDCAuthenticate command.
        /// </summary>
        public enum PDC_Auth : byte
        {
            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFICATION_OFF = 0x00,

            /// <summary> Option to indicate the Key diversification selection as RFU.</summary>
            DIVERSIFICATION_RFU = 0x01,

            /// <summary> Option to indicate the UpgradeKey derivation form ICUpgradeKey given UpgradeInfo.</summary>
            DERIVE_UPGRADE_KEY = 0x02,

            /// <summary> Option to indicate the diversification of YearUpgradeKey with the given DivInput
            /// and then derive the actual UpgradeKey with UpgradeInfo.
            /// </summary>
            DIVERSIFY_YEAR_UPGRADE_KEY = 0x03
        }
        #endregion PDC Authenticate command

        #region CombinedRead command
        /// <summary>
        /// MIFARE Plus options macros for differentiating between command and response for Cmd.SAM_CombinedReadMFP command
        /// </summary>
        public enum MFP_CombinedRead : byte
        {
            /// <summary> Option value for Combined Write - Command. </summary>
            COMMAND = 0x00,

            /// <summary> Option mask for Combined Write - Response. </summary>
            RESPONSE = 0x01,

            /// <summary> Option value for Combined Read - Both Command and Response. </summary>
            BOTH = 0x02
        }
        #endregion CombinedRead command

        #region CombinedWrite command
        /// <summary>
        /// MIFARE Plus options macros for differentiating between command / response and for
        /// specifying the Plain data in response for Cmd.SAM_CombinedWrite command.
        /// </summary>
        public enum MFP_CombinedWrite : byte
        {
            /// <summary> Option value for Combined Write - Command. </summary>
            COMMAND = 0x00,

            /// <summary> Option mask for Combined Write - Response. </summary>
            RESPONSE = 0x01,

            /// <summary> Option macro to return the plain data in response. </summary>
            RETURN_PLAIN = 0x00,

            /// <summary> Option macro to skip the plain data in response. </summary>
            SKIP_PLAIN = 0x02
        }
        #endregion CombinedWrite command

        #region ChangeKey command
        /// <summary>
        /// MIFARE Plus options to update the P1 information byte of ChangeKey command. These flags are common for both X
        /// and S mode MIFARE Plus ChangeKey command.
        /// </summary>
        public enum MFP_ChangeKey : byte
        {
            /// <summary> Option value for ChangeKey - Command. </summary>
            COMMAND = 0x00,

            /// <summary> Option mask for ChangeKey - Response. </summary>
            RESPONSE = 0x01,

            /// <summary> Option to disable the key diversification.</summary>
            DIVERSIFY_OFF = 0x00,

            /// <summary> Option to enable the key diversification.</summary>
            DIVERSIFY_ON = 0x02,
        }
        #endregion ChangeKey command
        #endregion MIFARE Plus

        #region MIFARE Classic
        #region Cmd.SAM_AuthenticateMIFARE, Cmd.MF_Authenticate
        /// <summary>
        /// Enumeration fields for Cmd.SAM_AuthenticateMifare, Cmd.MF_Authenticate
        /// </summary>
        public enum MFC_Authenticate : byte
        {
            /// <summary> Option to exclude the diversification block in the command frame. </summary>
            DIV_OFF = 0x00,

            /// <summary> Option to include the diversification block in the command frame. </summary>
            DIV_ON = 0x01
        }
        #endregion Cmd.SAM_AuthenticateMIFARE, Cmd.MF_Authenticate

        #region Cmd.MF_AuthenticatedRead, Cmd.MF_AuthenticatedWrite
        #region Cmd.MF_AuthenticatedWrite
        /// <summary>
        /// Enumeration fields for Cmd.MF_AuthenticatedWrite
        /// </summary>
        public enum TMData : ushort
        {
            /// <summary> Option mask for not exchanging the LE byte to SAM. </summary>
            NOT_RETURNED = 0x0000,

            /// <summary> Option mask for exchanging the LE byte to SAM for retrieval of TMC and TMV information. </summary>
            RETURNED = 0x0080
        }
        #endregion Cmd.MF_AuthenticatedWrite

        /// <summary>
        /// Enumeration fields for Cmd.MF_AuthenticatedRead and Cmd.MF_AuthenticatedWrite
        /// </summary>
        public enum MFC_CmdSetting : byte
        {
            /// <summary> Includes the Key information in the command frame. </summary>
            AUTH_RW_KEY_INFO_AVAILABLE = 0x00,

            /// <summary> Excludes the Key information in the command frame. Last key to apply for authenticating the block.</summary>
            AUTH_RW_KEY_INFO_NOT_AVAILABLE = 0x01,

            /// <summary> Key diversification is off.</summary>
            AUTH_RW_CMD_SET_DIV_OFF = 0x00,

            /// <summary> Key diversification is on.</summary>
            AUTH_RW_CMD_SET_DIV_ON = 0x02
        }
        #endregion Cmd.MF_AuthenticatedRead, Cmd.MF_AuthenticatedWrite

        #region Cmd.MF_Write
        /// <summary>
        /// Enumeration fields for Cmd.MF_Write
        /// </summary>
        public enum MFC_OpMode : byte
        {
            /// <summary> Perform Write for MIFARE 1k and 4k, plain compatible write (16 byte data blocks). </summary>
            WRITE_CLASSIC = 0x00,

            /// <summary> Perform Write for MIFARE Ultralight(4 byte data blocks). </summary>
            WRITE_ULTRALIGHT = 0x01,
        }
        #endregion Cmd.MF_Write

        #region Cmd.MF_Write, Cmd.MF_ValueWrite, Cmd.MF_Increment, Cmd.MF_Decrement, Cmd.MF_Restore
        /// <summary>
        /// Enumeration fields for Cmd.MF_ValueWrite
        /// </summary>
        public enum MFC_TMData : byte
        {
            /// <summary> Option mask for not exchanging the LE byte to SAM. </summary>
            NOT_RETURNED = 0x00,

            /// <summary> Option mask for exchanging the LE byte to SAM for retrieval of TMC and TMV information. </summary>
            RETURNED = 0x80
        }
        #endregion Cmd.MF_Write, Cmd.MF_ValueWrite, Cmd.MF_Increment, Cmd.MF_Decrement, Cmd.MF_Restore

        #region Cmd.SAM_ChangeKeyMIFARE, Cmd.MF_ChangeKey
        /// <summary>
        /// Enumeration options for MIFARE Classic X and S mode communication.
        /// </summary>
        public enum MFC_ChangeKey : byte
        {
            #region Cmd.SAM_ChangeKeyMIFARE, Cmd.MF_ChangeKey
            /// <summary> Diversification is disabled. This is the default value. </summary>
            CHANGE_KEY_MIFARE_DIV_OFF = 0x00,

            /// <summary> Diversification of Key A is enabled. </summary>
            CHANGE_KEY_MIFARE_DIV_A_ON = 0x02,

            /// <summary> Diversification of Key A is enabled. </summary>
            CHANGE_KEY_MIFARE_DIV_B_ON = 0x04,
            #endregion Cmd.SAM_ChangeKeyMIFARE, Cmd.MF_ChangeKey

            #region Cmd.SAM_ChangeKeyMIFAREDump
            /// <summary> Dump the key as plain. </summary>
            CHANGE_KEY_MIFARE_PLAIN = 0x00,

            /// <summary> Dump the key as enciphered. </summary>
            CHANGE_KEY_MIFARE_ENCRYPT = 0x01,
            #endregion Cmd.SAM_ChangeKeyMIFAREDump
        }
        #endregion Cmd.SAM_ChangeKeyMIFARE, Cmd.MF_ChangeKey
        #endregion MIFARE Classic

        #region MIFARE Ultralight
        /// <summary>
        /// Option macros for the information bytes of MIFARE ultralight X command.
        /// </summary>
        public enum MFULOption : byte
        {
            #region Option for Cmd.ULC_AuthenticatePICC
            /// <summary>
            /// Enumeration fields for Cmd.ULC_AuthenticatePICC
            /// For including or excluding the diversification input in command frame.
            /// </summary>
            ULC_AUTHENTICATE_DIV_OFF = 0x00,
            ULC_AUTHENTICATE_DIV_ON = 0x01,

            /// <summary>
            /// Enumeration fields for Cmd.ULC_AuthenticatePICC
            /// For including or excluding the LE in the command frame.
            /// </summary>
            ULC_AUTHENTICATE_EXCLUDE_LE = 0x00,
            ULC_AUTHENTICATE_INCLUDE_LE = 0x02
            #endregion
        }
        #endregion MIFARE Ultralight

        #region MIFARE Common
        #region ISOMode
        /// <summary>
        /// MIFARE Common options to update the P2 information byte.
        /// </summary>
        public enum TMRI_IsoMode : byte
        {
            /// <summary> Sam CommitReaderID ISO mode selection for Native command set. </summary>
            NATIVE = 0x00,

            /// <summary> Sam CommitReaderID ISO mode selection for ISO 7816-4 command set. </summary>
            ISO7816_4 = 0x40
        }
        #endregion ISOMode

        #region State
        /// <summary>
        /// Options to differentiate the state of PICC authentication.
        /// </summary>
        public enum State : byte
        {
            /// <summary> Option to indicate the PICC state as MIFARE Plus EV1. </summary>
            MFP = 0x00,

            /// <summary> Option to indicate the PICC state as MIFARE DESFire EV2. </summary>
            DESFIRE = 0x01
        }
        #endregion State
        #endregion MIFARE Common

        #region ISO / IEC-29167-10
        #region Cmd.SAM_AuthenticateTAM
        /// <summary>
        /// Option macros to differentiate between part 1 and part2 command exchange
        /// </summary>
        public enum Tam : byte
        {
            /// <summary> Option for exchanging Key and Div input and receive the IChallange as response. </summary>
            GET_RND = 0x00,

            /// <summary> Option for exchanging TResponse and validating it. </summary>
            PROCESS_TRESPONE = 0x01
        }
        #endregion Cmd.SAM_AuthenticateTAM

        #region Cmd.SAM_AuthenticateMAM
        /// <summary>
        /// Option macros for updating the PurposeMAM2 information of Cmd.SAM_AuthenticteMAM
        /// </summary>
        public enum Mam : byte
        {
            /// <summary> Option to indicate PurposeMAM2 value as None. Here only authentication will be performed.</summary>
            NONE = 0x00,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy disable until HF reset. </summary>
            DISABLE_PRIVACY_HF_RESET = 0x08,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy enable. </summary>
            ENABLE_PRIVACY = 0x09,

            /// <summary> Option to indicate PurposeMAM2 value as Privacy disable. </summary>
            DISABLE_PRIVACY = 0x0A,

            /// <summary> Option to indicate PurposeMAM2 value as Destroy. </summary>
            DESTROY = 0x0B
        }
        #endregion Cmd.SAM_AuthenticateMAM
        #endregion ISO / IEC-29167-10

        #region ISO14443-3
        #region Cmd.ISO14443-3_Request_Wakeup
        /// <summary>
        /// Option for Cmd.ISO14443-3_Request_Wakeup command.
        /// </summary>
        public enum CmdCode : byte
        {
            /// <summary> Request command code. </summary>
            REQA = 0x26,

            /// <summary> Wakeup command code. </summary>
            WUPA = 0x52
        }
        #endregion Cmd.ISO14443-3_Request_Wakeup

        #region Cmd.ISO14443-3_ActivateIdle
        /// <summary>
        /// Option for Cmd.ISO14443-3_ActivateIdle command.
        /// </summary>
        public enum ActivateIdle : byte
        {
            /// <summary> Default option mask for ActivateIdle. </summary>
            DEFAULT = 0x00,

            /// <summary> Option flag for applying the ATQA filter. </summary>
            APPLY_ATQA = 0x01,

            /// <summary> Option flag for applying the SAK filter. </summary>
            APPLY_SAK = 0x02,

            /// <summary> Option flag for setting the time to wait. </summary>
            APPLY_TIME = 0x04
        }
        #endregion Cmd.ISO14443-3_ActivateIdle
        #endregion ISO14443-3

        #region ISO14443-4
        /// <summary>
        /// Option for Cmd.ISO14443-4_Deselect command.
        /// </summary>
        public enum Deselect : byte
        {
            /// <summary> Option to not free CID is deselect fails. </summary>
            DO_NOT_FREE_CID = 0x00,

            /// <summary> Option to forcefully free CID in any case. </summary>
            FORCE_FREE_CID = 0x01
        }
        #endregion ISO14443-4
        #endregion  Command specific enumerations

        /// <summary>
        /// Option macros for the information bytes of Public Key Infrastructure command.
        /// </summary>
        public enum PKIOption : byte
        {
            #region Option for Cmd.PKI_GenerateKeyPair
            /// <summary>
            /// Enumeration fileds for P1, Bit0 information byte of Cmd.PKI_GenerateKeyPair.
            /// Public exponent selection
            /// </summary>
            RANDOM_E = 0x00,
            HOST_E = 0x01,
            #endregion

            #region Option for Cmd.PKI_ImportKey, Cmd.PKI_ImportEccKey
            /// <summary>
            /// Enumeration fileds for P1, Bit0 information byte of Cmd.PKI_ImportKey
            /// and Cmd.PKI_ImportEccKey
            /// Update selection
            /// </summary>
            UPDATE_KEY_SETTINGS_VALUE = 0x00,
            UPDATE_KEY_SETTINGS_ONLY = 0x01,
            #endregion

            #region Option for Cmd.PKI_GenerateKeyPair, Cmd.PKI_ImportKey
            /// <summary>
            /// Enumeration fields for P1, Bit 1 information bytes of Cmd.PKI_GenerateKeyPair,
            /// Cmd.PKI_ImportKey
            /// Include Access Entry Key (SAM AV3 feature)
            /// </summary>
            P1_AEK_EXCLUDE = 0x00,
            P1_AEK_INCLUDE = 0x02,
            #endregion

            #region Option for Cmd.PKI_ExportPrivateKey and Cmd.PKI_ExportPublicKey
            /// <summary>
            /// Enumeration fields for P2, Bit 7 information bytes of Cmd.PKI_ExportPrivateKey,
            /// Cmd.PKI_ExportPublicKey
            /// Include Access Entry Key (SAM AV3 feature)
            /// </summary>
            P2_AEK_EXCLUDE = 0x00,
            P2_AEK_INCLUDE = 0x80,
            #endregion

            #region Option for Cmd.PKI_UpdateKeyEntries
            /// <summary>
            /// Enumeration fields for Cmd.PKI_UpdateKeyEntries.
            /// For including Le byte and receiving UpdateACK information from Sam hardware.
            /// </summary>
            ACK_EXCLUDE = 0x00,
            ACK_INCLUDE = 0x80,
            #endregion

            #region Option for Cmd.PKI_UpdateKeyEntries, Cmd.PKI_GenerateHash, Cmd.PKI_EncipherKeyEntries
            /// <summary>
            /// Enumeration fields for Cmd.PKI_UpdateKeyEntries, Cmd.PKI_GenerateHash,
            /// Cmd.PKI_EncipherKeyEntries
            /// Hashing algorithm selection (for padding MGFs and digital signature)
            /// </summary>
            HASH_ALGORITHM_SHA_1 = 0x00,
            HASH_ALGORITHM_SHA_224 = 0x01,
            HASH_ALGORITHM_SHA_256 = 0x03,
            HASH_ALGORITHM_SHA_RFU = 0x02,
            #endregion

            #region Option for Cmd.PKI_EncipherKeyEntries
            /// <summary>
            /// Enumeration fields for Cmd.PKI_EncipherKeyEntries
            /// Use Key Diversification
            /// </summary>
            DIVERSIFICATION_OFF = 0x00,
            DIVERSIFICATION_ON = 0x10,
            #endregion

            #region Option for Cmd.PKI_GenerateHash and Cmd.PKI_DecipherData
            /// <summary>
            /// Enumeration fields for Cmd.PKI_GenerateHash and Cmd.PKI_DecipherData
            /// First frame indication.
            /// </summary>
            FIRST = 0x04,
            #endregion

            #region Option for Cmd.PKI_ImportEccCurve
            /// <summary>
            /// Enumeration fileds for P1, Bit0 information byte of Cmd.PKI_ImportEccCurve
            /// Update selection
            /// </summary>
            UPDATE_CURVE_SETTINGS_VALUE = 0x00,
            UPDATE_CURVE_SETTINGS_ONLY = 0x01,
            #endregion

            #region Option for Cmd.PKI_ImportCaPk
            /// <summary>
            /// Enumeration fileds for P1 information byte of Cmd.PKI_ImportCaPk
            /// Last Frame Indicator
            /// </summary>
            LFI_LAST = 0x00,
            LFI_CHAINED = 0xAF,

            /// <summary>
            /// Enumeration fileds for P2, Bit7 information byte of Cmd.PKI_ImportCaPk
            /// Update selection
            /// </summary>
            IMPORT_KEY_SETTINGS_VALUE = 0x00,
            IMPORT_KEY_SETTINGS_ONLY = 0x80,
            #endregion

            #region Option for Cmd.RemoveCaPk
            /// <summary>
            /// Enumeration fileds for OfflineAck reception for Cmd.PKI_RemoveCaPk
            /// OfflineAcknowledgment reception.
            /// </summary>
            OFFLINE_ACK_RECEPTION_OFF = 0x00,
            OFFLINE_ACK_RECEPTION_ON = 0x01,
            #endregion

            #region Option for Cmd.PKI_ExportCaPk
            /// <summary>
            /// Enumeration fileds for P1 information byte of Cmd.PKI_ExportCaPk
            /// Export selection
            /// </summary>
            EXPORT_KEY_SETTINGS_VALUE = 0x00,
            EXPORT_KEY_SETTINGS_ONLY = 0x80,
            #endregion
        }
        #endregion Enumerations

        #region Data Structure
        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the lower layers parameter structure. </summary>
            public IntPtr pBalDataParams;

            /// <summary> Pointer to the hal data params of a reader. NULL in case of X-Mode. </summary>
            public IntPtr pReaderHalDataParams;

            /// <summary> Pointer to the KeyStore used for Host Authentication. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the ENC crypto layers parameter structure. </summary>
            public IntPtr pENCCryptoDataParams;

            /// <summary> Pointer to the MAC crypto layers parameter structure. </summary>
            public IntPtr pMACCryptoDataParams;

            /// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to the ENC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. </summary>
            public IntPtr pPLUpload_ENCCryptoDataParams;

            /// <summary> Pointer to the MAC crypto layers parameter structure. This will be used for ProgrammableLogic feature only. </summary>
            public IntPtr pPLUpload_MACCryptoDataParams;

            /// <summary> Command counter for Secure Messaging. </summary>
            public uint Cmd_Ctr;

            /// <summary>
            /// Indicated the one of the below configurations.
            ///		0x02: Sam AV2 hardware is used.
            ///		0x03: Sam AV3 hardware is used with sam in Un-Activated or Activated state
            ///	</summary>
            public byte bHostMode;

            /// <summary> The current Authtentication Type used for SM. </summary>
            public byte bAuthType;

            /// <summary> Store the current authentication key. </summary>
            public byte bKeyNo;

            /// <summary> Command Data pending for encryption. </summary>
            public fixed byte bPendingEncCmdData[16];

            /// <summary> Length of pending command data to encrypt. </summary>
            public byte bPendingEncCmdDataLength;

            /// <summary> Initvector for pending CMD-Data encryption. </summary>
            public fixed byte bPendingCmdIv[16];

            /// <summary> Command Data pending for MACing. </summary>
            public fixed byte bPendingMacCmdData[16];

            /// <summary> Length of pending command data to MAC. </summary>
            public byte bPendingMacCmdDataLength;

            /// <summary> Initvector for pending CMD-Data MACing. </summary>
            public fixed byte bPendingCmdMac[16];

            /// <summary> Response Data pending for MACing. </summary>
            public fixed byte bPendingMacRespData[16];

            /// <summary> Length of pending receive data to MAC. </summary>
            public byte bPendingMacRespDataLength;

            /// <summary> Intermediate response MAC. </summary>
            public fixed byte bPendingRespMac[16];

            /// <summary> Initvector for pending receive-Data decryption. </summary>
            public fixed byte bPendingRespIv[16];

            /// <summary> Pending (unreturned) response data. </summary>
            public fixed byte bPendingRespData[16];

            /// <summary> Length of pending response data. </summary>
            public byte bPendingRespDataLength;

            /// <summary> Type of secure messaging for current command. </summary>
            public byte bCmdSM;

            /// <summary> Type of secure messaging for current response. </summary>
            public byte bRespSM;

            /// <summary> Whether command chaining is active or not. </summary>
            public byte bCommandChaining;

            /// <summary> Whether response chaining is active or not. </summary>
            public byte bResponseChaining;

            /// <summary> SAM UID. </summary>
            public fixed byte bUid[7];

            /// <summary> Whether CMAC mode is enabled in the master key. </summary>
            public byte bMasterKeyCmacMode;

            /// <summary> Operation mode; One of the below values.
            ///		NxpRdLibNet.Hal.SamAV3.OpMode.NON_X
            ///		NxpRdLibNet.Hal.SamAV3.OpMode.X_RC523
            ///		NxpRdLibNet.Hal.SamAV3.OpMode.X_RC663</summary>
            public byte bOpMode;

            /// <summary> Logical channel number to select for Authenthication. </summary>
            public byte bLogicalChannel;

            /// <summary> Pointer to global transmit buffer used by the Exchange() function. </summary>
            public IntPtr pTxBuffer;

            /// <summary> Size of the global transmit buffer. </summary>
            public ushort wTxBufSize;

            /// <summary> Number of valid bytes for exchange within the transmit buffer. </summary>
            public ushort wTxBufLen;

            /// <summary> Number of valid bytes for other commands within the transmit buffer. </summary>
            public ushort wTxBufLen_Cmd;

            /// <summary> Pointer to global receive buffer used by the Exchange() function. </summary>
            public IntPtr pRxBuffer;

            /// <summary> Size of the global receive buffer. </summary>
            public ushort wRxBufSize;

            /// <summary> Number of valid bytes within the receive buffer. </summary>
            public ushort wRxBufLen;

            /// <summary> Starting position within the global receive buffer. </summary>
            public ushort wRxBufStartPos;

            /// <summary> Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). </summary>
            public ushort wTxBufStartPos;

            /// <summary> Configuration shadow; Stores configuration for current cardtype. </summary>
            public fixed ushort wCfgShadow[SHADOW_COUNT];

            /// <summary> Type of card for which the hal is configured for. </summary>
            public byte bCardType;

            /// <summary>
            /// Unit of current timeout value (either NxpRdLibNet.Hal.Config.TIMEOUT_VALUE_US or
            /// NxpRdLibNet.Hal.Config.TIMEOUT_VALUE_MS).
            /// </summary>
            public byte bTimeoutUnit;

            /// <summary> Field-Off-Time in milliseconds. </summary>
            public ushort wFieldOffTime;

            /// <summary> Field-Recovery-Time in milliseconds. </summary>
            public ushort wFieldRecoveryTime;

            /// <summary> Storage for additional error information. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Disables/Enables time measurement. </summary>
            public ushort wTimingMode;

            /// <summary> Current timing value. </summary>
            public uint dwTimingUs;

            /// <summary> Contains information about MIFARE Crypto enabled state. </summary>
            public byte bMifareCryptoDisabled;

            /// <summary> Storage for NxpRdLibNet.Hal.Config.RFRESET_ON_TIMEOUT setting. </summary>
            public byte bRfResetAfterTo;

            /// <summary> Storage for NxpRdLibNet.Hal.SamAV3.Config.DISABLE_NONX_CFG_MAPPING setting. </summary>
            public byte bDisableNonXCfgMapping;

            /// <summary> Current timing value backup for PC. </summary>
            public ushort wFdtPc;

            /// <summary>
            /// Buffer to store the complete segment information of PLUpload command.
            /// The buffer should have a memory size equivalent to the complete PLUpload Segment(s) size.
            /// </summary>
            public IntPtr pPLUploadBuff;

            /// <summary> The size of bytes available in PLUploadBuff buffer. </summary>
            public ushort wPLUploadBuffLen;

            /// <summary> Programmable Logic Initial session key for macing the data. </summary>
            public fixed byte aPLUploadSessMAC0[32];

            /// <summary> The current key type used for crypto operations. </summary>
            public byte bPLUploadKeyType;
        };
        #endregion Data Structure

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SamAV3 ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SamAV3 ()
        {
            // Free Buffers
            if ( this.m_pTxBuffer.IsAllocated )
            {
                this.m_pTxBuffer.Free ();
            }
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }
            if ( this.m_pPLUploadBuf.IsAllocated )
            {
                this.m_pPLUploadBuf.Free ();
            }
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Mapping

        #region Dll Import
        #region Initialization
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pBalDataParams, IntPtr pReaderHalDataParams,
            IntPtr pKeyStoreDataParams, IntPtr pCryptoENCDataParams, IntPtr pCryptoMACDataParams, IntPtr pCryptoRngDataParams, IntPtr pPLUpload_CryptoENCDataParams,
            IntPtr pPLUpload_CryptoMACDataParams, byte bOpMode, byte bLogicalChannel, IntPtr pTxBuffer, ushort wTxBufSize, IntPtr pRxBuffer, ushort wRxBufSize,
            IntPtr pPLUploadBuf );
        #endregion Initialization

        #region Host Communication
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_LockUnlock ( ref DataParams_t pDataParams, byte bCmdType, ushort wRdKeyNo, ushort wRdKeyVer,
            byte bSamKeyNo, byte bSamKeyVer, byte bUnlockKeyNo, byte bUnlockKeyVer, uint dwMaxChainBlocks );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateHost ( ref DataParams_t pDataParams, byte bHostMode, ushort wRdKeyNo,
            ushort wRdKeyV, byte bSamKeyNo, byte bSamKeyVer );
        #endregion Host Communication

        #region Security and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GetVersion ( ref DataParams_t pDataParams, byte[] pVersion, ref byte pVersionLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DisableCrypto ( ref DataParams_t pDataParams, ushort wProMas );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte[] pDivInput,
            byte bDivInputLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_LoadInitVector ( ref DataParams_t pDataParams, byte bOption, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_KillAuthentication ( ref DataParams_t pDataParams, byte bOption );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_SelectApplication ( ref DataParams_t pDataParams, byte[] pDF_Aid );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GetRandom ( ref DataParams_t pDataParams, byte bExpLen, byte[] pRnd );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_Sleep ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_SetConfiguration ( ref DataParams_t m_pDataParams, byte bOption, byte[] pData, byte bDataLen );
        #endregion Security and Configuration

        #region Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo, byte bProMas, byte[] bKeyData,
            byte bKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyEntryOffline ( ref DataParams_t pDataParams, byte bKeyNo, byte bProMas, ushort wChangeCtr,
            byte[] pOfflineCryptogram, byte bOfflineCryptogramLen, byte bEnableOfflineAck, byte[] pOfflineAck );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GetKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo, byte bMode, byte[] pKeyEntry,
            ref byte pKeyEntryLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKUCEntry ( ref DataParams_t pDataParams, byte bKucNo, byte bProMas, byte[] pKucData,
            byte bKucDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKUCEntryOffline ( ref DataParams_t pDataParams, byte bKucNo, byte bProMas, ushort wChangeCtr,
            byte[] pOfflineCryptogram, byte bOfflineCryptogramLen, byte bEnableOfflineAck, byte[] pOfflineAck );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GetKUCEntry ( ref DataParams_t pDataParams, byte bKucNo, byte[] pKucEntry );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DumpSecretKey ( ref DataParams_t pDataParams, byte bCrypto, byte bKeyNo, byte bKeyV, byte[] pDivInput,
            byte bDivInputLen, byte[] pSecretKey, ref byte pSecretKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DumpSessionKey ( ref DataParams_t pDataParams, byte bCrypto, byte[] pSessionKey, ref byte pSessionKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DisableKeyEntry ( ref DataParams_t pDataParams, byte bKeyNo, byte[] pOfflineCryptogram, byte bCryptogramLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DisableKeyEntryOffline ( ref DataParams_t pDataParams, byte bKeyNo, ushort wChangeCtr, byte[] pOfflineCryptogram,
            byte bOfflineCryptogramLen, byte bEnableOfflineAck, byte[] pOfflineAck );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_EncipherKeyEntry ( ref DataParams_t pDataParams, byte bPersoKeyNo, byte bKeyNo, byte bOption, ushort wPersoCtr,
            byte[] pDivInput, byte bDivInputLen, byte[] pOfflineCryptogram, ref byte bOfflineCryptogramLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DeriveKey ( ref DataParams_t pDataParams, byte bSrcKeyNo, byte bSrcKeyVer, byte bDstKeyNo, byte[] pDeriveIn,
            byte bDeriveInLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GenerateMFCLicMAC ( ref DataParams_t pDataParams, byte bOption, byte bInputLen, byte[] pInput, byte bKeyCount,
            byte[] pMFCSectorKeys, byte[] pMFUID, byte[] pMFCLicMAC );
        #endregion Key Management

        #region Data Processing
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ApplySM ( ref DataParams_t pDataParams, ushort wOption, byte bCommMode, byte bOffset, byte bCmdCtrIncr,
            byte[] pTxData, byte bTxDataLen, ref IntPtr ppRxData, ref ushort pRxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_RemoveSM ( ref DataParams_t pDataParams, ushort wOption, byte bCommMode, byte[] pTxData,
            byte bTxDataLen, ref IntPtr ppRxData, ref ushort pRxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_VerifyMAC ( ref DataParams_t pDataParams, ushort wOption, byte bNum, byte[] pTxData,
            byte bTxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( ref DataParams_t pDataParams, ushort wOption, byte bNum, byte[] pTxData, byte bTxDataLen,
            ref IntPtr ppRxData, ref ushort pRxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DecipherData ( ref DataParams_t pDataParams, ushort wOption, byte[] pEncData, ushort bEncDataLen,
            byte[] pLength, ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_EncipherData ( ref DataParams_t pDataParams, ushort wOption, byte[] pPlainData, byte bPlainDataLen,
            byte bOffset, ref IntPtr ppEncData, ref ushort pEncDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData ( ref DataParams_t pDataParams, ushort wOption, byte[] pEncData, ushort bEncDataLen,
            ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData ( ref DataParams_t pDataParams, ushort wOption, byte[] pPlainData, byte bPlainDataLen,
            ref IntPtr ppEncData, ref ushort pEncDataLen );

        #endregion Data Processing

        #region Public Key Infrastructure
        #region RSA
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_GenerateKeyPair ( ref DataParams_t pDataParams, byte bOption, byte bPKI_KeyNo, ushort wPKI_Set,
            byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen, ushort wPKI_eLen, byte[] pPKI_e );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ImportKey ( ref DataParams_t pDataParams, byte bOption, byte bPKI_KeyNo, ushort wPKI_Set,
            byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen, ushort wPKI_eLen,
            ushort wPKI_PLen, ushort wPKI_QLen, byte[] pPKI_N, byte[] pPKI_e, byte[] pPKI_p, byte[] pPKI_q, byte[] pPKI_dP, byte[] pPKI_dQ,
            byte[] pPKI_ipq );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ExportPrivateKey ( ref DataParams_t pDataParams, ushort wOption, byte bPKI_KeyNo,
            ref IntPtr ppKeyData, ref ushort pKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ExportPublicKey ( ref DataParams_t pDataParams, ushort wOption, byte bPKI_KeyNo,
            ref IntPtr ppKeyData, ref ushort pKeyDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_UpdateKeyEntries ( ref DataParams_t pDataParams, byte bOption, byte bNoOfKeyEntries,
            byte bHashingAlg, byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Ack, byte[] pKeyFrame, ushort wKeyFrameLen, ref IntPtr ppUpdateACK,
            ref ushort wUpdateACKLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_EncipherKeyEntries ( ref DataParams_t pDataParams, ushort wOption, byte bNoOfKeyEntries, byte bHashingAlg,
            byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Dec, byte bPKI_KeyNo_Verif, ushort wPerso_Ctr, byte[] pKeyEntries, byte bKeyEntriesLen,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppEncKeyFrame_Sign, ref ushort pEncKeyFrame_Sign_Len );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_GenerateHash ( ref DataParams_t pDataParams, ushort wOption, byte bHashingAlg, uint dwMLen, byte[] pMessage,
            ushort wMsgLen, ref IntPtr ppHash, ref ushort pHashLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_GenerateSignature ( ref DataParams_t pDataParams, byte bHashingAlg, byte bPKI_KeyNo_Sign, byte[] pHash,
            byte bHashLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_SendSignature ( ref DataParams_t pDataParams, ref IntPtr ppSignature, ref ushort pSignatureLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_VerifySignature ( ref DataParams_t pDataParams, byte bPKI_KeyNo_Verif, byte bHashingAlg, byte[] pHash,
            byte bHashLen, byte[] pSignature, ushort wSignatureLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_EncipherData ( ref DataParams_t pDataParams, byte bHashingAlg, byte bPKI_KeyNo_Enc, byte[] pPlainData,
            ushort wPlainDataLen, ref IntPtr ppEncData, ref ushort pEncDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_DecipherData ( ref DataParams_t pDataParams, ushort wOption, byte bHashingAlg, byte bPKI_KeyNo_Dec, byte[] pEncData,
            ushort wEncDataLen, ref IntPtr ppPlainData, ref ushort pPlainDataLen );
        #endregion RSA

        #region ECC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ImportEccKey ( ref DataParams_t pDataParams, byte bOption, byte bECC_KeyNo, ushort wECC_Set,
            byte bECC_KeyNoCEK, byte bECC_KeyVCEK, byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK, ushort wECC_Len, byte[] pECC_xy,
            ushort wECC_xyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ImportEccCurve ( ref DataParams_t pDataParams, byte bOption, byte bECC_CurveNo, byte bECC_KeyNoCCK,
            byte bECC_KeyVCCK, byte bECC_N, byte bECC_M, byte[] pECC_Prime, byte[] pECC_ParamA, byte[] pECC_ParamB, byte[] pECC_Px, byte[] pECC_Py, byte[] pECC_Order );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ExportEccPublicKey ( ref DataParams_t pDataParams, byte bECC_KeyNo, ref ushort pECC_Set, ref byte pECC_KeyNoCEK,
            ref byte pECC_KeyVCEK, ref byte pECC_RefNoKUC, ref byte pECC_KeyNoAEK, ref byte pECC_KeyVAEK, ref ushort pECC_Len, ref IntPtr ppECC_xy, ref byte pECC_xyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_VerifyEccSignature ( ref DataParams_t pDataParams, byte bECC_KeyNo, byte bECC_CurveNo, byte bLen, byte[] pMessage,
            byte[] pSignature, ushort wSignatureLen );
        #endregion ECC

        #region EMV
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ImportCaPk ( ref DataParams_t pDataParams, byte bOption, byte[] pCaPkData, byte bCaPkDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ImportCaPkOffline ( ref DataParams_t pDataParams, byte bReceiveOfflineAck, byte bOption,
            byte[] pOfflineCryptogram, byte bOfflineCryptogramLen, ref IntPtr ppOfflineAck, ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_RemoveCaPk ( ref DataParams_t pDataParams, byte[] pRID, byte bPkID );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_RemoveCaPkOffline ( ref DataParams_t pDataParams, byte bReceiveOfflineAck,
            byte[] pOfflineCryptogram, byte bOfflineCryptogramLen, ref IntPtr ppOfflineAck, ref ushort pOfflineAckLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_ExportCaPk ( ref DataParams_t pDataParams, ushort wOption, byte[] pRID, byte bPkID,
            ref IntPtr ppKeyEntry, ref ushort pKeyEntryLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_LoadIssuerPk ( ref DataParams_t pDataParams, byte bLFI, ushort wOption, byte[] pData, byte bDataLen,
            byte[] pIssureID, byte[] pExpDate, byte[] pSerialNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PKI_LoadIccPk ( ref DataParams_t pDataParams, byte bLFI, ushort wOption, byte[] pData, byte bDataLen,
            byte[] pPAN, byte[] pExpDate, byte[] pSerialNo, ref byte pAlgoPk );
        #endregion EMV
        #endregion Public Key Infrastructure

        #region Virtual Card
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_SelectVC ( ref DataParams_t pDataParams, byte bOption, byte bEncKeyNo, byte bEncKeyVer,
            byte bMacKeyNo, byte bMacKeyVer, byte[] pData, byte bDataLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pPPCData, byte bPPCDataLen, byte[] pPCData, byte bPCDataLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppMac, ref ushort pMacLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part2 ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen, ref byte pPiccRetCode );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_VCA_Select ( ref DataParams_t pDataParams, byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo,
            byte bMacKeyVer, byte[] pIID, byte bIIDLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse, ref ushort pRespLen, ref ushort pPiccRetCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_VCA_ProximityCheck ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte bNumOfRand,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppResponse, ref ushort pRespLen );
        #endregion X Mode
        #endregion Virtual Card

        #region MIFARE DESFire
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte bAuthMode, byte[] pDivInput, byte bDivInputLen, byte[] pCardResponse, byte bCardRespLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2 ( ref DataParams_t pDataParams, byte bPiccErrorCode, byte[] pCardResponse,
            byte bCardRespLen, byte[] pPDcap2, byte[] pPCDcap2, ref byte pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_IsoAuthenticatePICC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, byte[] pCardResponse, byte bCardRespLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_IsoAuthenticatePICC_Part2 ( ref DataParams_t pDataParams, byte[] pCardResponse, byte bCardRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyPICC ( ref DataParams_t pDataParams, byte bCryptoMethod, byte bConfig, byte bKeySetNo,
            byte bDFKeyNo, byte bCurrKeyNo, byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppSamResponse,
            ref ushort pSamRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_CreateTMFilePICC ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte bFileNo,
            byte bFileOption, byte[] pAccessRights, byte bTMKeyOptions, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppSamResponse, ref ushort pSamRespLen );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC ( ref DataParams_t pDataParams, byte bOption, byte bISOMode, byte bDFKeyNo,
            byte bKeyNo, byte bKeyVer, byte bPCDcap2InLen, byte[] pPCDcap2In, byte[] pDivInput, byte bDivInputLen, byte[] pPDcap2, byte[] pPCDcap2,
            byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC ( ref DataParams_t pDataParams, byte bKeyCompMeth, byte bConfig, byte bKeySetNo,
            byte bDFKeyNo, byte bCurrKeyNo, byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_DESFire_WriteX ( ref DataParams_t pDataParams, ushort wOption, byte bCrypto, byte[] pData, byte bDataLen,
            byte[] pPiccReturnCode, ref byte pErrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_DESFire_ReadX ( ref DataParams_t pDataParams, ushort wOption, byte bCrypto, byte[] pAppData, byte bAppDataLen,
            ref IntPtr ppResponse, ref ushort pRespLen, byte[] pPiccReturnCode, ref byte pErrLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_DESFire_CreateTMFilePICC ( ref DataParams_t pDataParams, byte bOption, byte bISOMode, byte bKeyNo, byte bKeyVer,
            byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bTMKeyOptions, byte[] pDivInput, byte bDivInputLen, byte[] pPiccReturnCode );
        #endregion X Mode
        #endregion MIFARE DESFire

        #region MIFARE Plus
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pPDChal, byte bPDChalLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPCDChalResp, ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode, byte[] pPDResp,
            byte bPDRespLen, ref IntPtr pPDCap2, ref IntPtr pPCDCap2, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthSectorSwitchMFP_Part1 ( ref DataParams_t pDataParams, byte bOption, byte[] pPDChal, byte bPDChalLen,
            ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer, byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount, byte[] pKeyBlocks, byte bKeyBlocksLen,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPCDChalResp, ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthSectorSwitchMFP_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode, byte[] pPDResp,
            byte bPDRespLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticatePDC_Part1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, byte[] pPDChal,
            byte bPDChalLen, byte[] pUpgradeInfo, byte bLen, byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPCDChalResp, ref ushort pPCDChalRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticatePDC_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode, byte[] pPDResp, byte bPDRespLen,
            ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP ( ref DataParams_t pDataParams, byte bLFI, ushort wOption, byte[] pData, byte bDataLen,
            ref IntPtr ppOutput, ref ushort pOutputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP ( ref DataParams_t pDataParams, ushort wOption, byte[] pData, byte bDataLen,
            ref IntPtr ppOutput, ref ushort pOutputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP ( ref DataParams_t pDataParams, byte bOption, byte[] pData, byte bDataLen,
            ref IntPtr ppProtectedData, ref ushort pProtectedDataLen, ref byte pPiccReturnCode );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_Authenticate ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, ushort wBlockNo, byte[] pPcdCapsIn,
            byte bPcdCapsInLen, byte[] pDivInput, byte bDivInputLen, byte[] pPcdCapsOut, byte[] pPdCaps, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch ( ref DataParams_t pDataParams, ushort wOption, ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer,
            byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount, byte[] pKeyBlocks, byte bKeyBlocksLen, byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_PDC_Authenticate ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer, ushort wUpgradeKey,
            byte[] pUpgradeInfo, byte bUpgradeInfoLen, byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_CombinedRead ( ref DataParams_t pDataParams, ushort wOption, byte[] pReadCmd, byte bReadCmdLen,
            ref IntPtr ppData, ref ushort pDataLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_CombinedWrite ( ref DataParams_t pDataParams, ushort wOption, byte[] pData, byte bDataLen, byte[] aTMC,
            byte[] aTMV, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_ChangeKey ( ref DataParams_t pDataParams, byte bOption, byte bCmdCode, ushort wBlockNo, byte bKeyNo,
            byte bKeyVer, byte[] pDivInput, byte bDivInputLen, ref byte pPiccReturnCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MFP_WritePerso ( ref DataParams_t pDataParams, ushort wOption, byte[] pBlocks, byte bBlockLen, ref byte pPiccReturnCode );
        #endregion X Mode
        #endregion MIFARE Plus

        #region MIFARE Classic
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part1 ( ref DataParams_t pDataParams, byte bOption, byte[] pMFUID, byte bKeyNo,
            byte bKeyVer, byte bKeyType, byte bMFBlockNo, byte bDivBlockNo, byte[] pNumberRB, byte bNumRBLen, ref IntPtr ppEncToken, ref ushort pEncTokenLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part2 ( ref DataParams_t pDataParams, byte[] pEncToken, byte bEncTokenLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyMIFARE ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVerA,
            byte bKeyVerB, byte[] pAccCond, byte[] pMFUID, byte bDivBlockNo, ref IntPtr ppProtData, ref ushort pProtDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_ChangeKeyMIFAREDump ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte bKeyType, byte[] pMFUID, byte bDivBlockNo, ref IntPtr ppSecretKey, ref ushort pSecretKeyLen );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Authenticate ( ref DataParams_t pDataParams, byte bOption, byte[] pMFUID, byte bKeyNo, byte bKeyVer,
            byte bKeyType, byte bMFBlockNo, byte bDivBlockNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Read ( ref DataParams_t pDataParams, byte[] pBlocks, byte bBlocksLen, ref IntPtr ppData,
            ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Write ( ref DataParams_t pDataParams, byte bOption, byte[] pData, byte bDataLen,
            ref IntPtr ppTMData, ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_ValueWrite ( ref DataParams_t pDataParams, byte bOption, byte[] pBlocks, byte bBlocksLen,
            ref IntPtr ppTMData, ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Increment ( ref DataParams_t pDataParams, byte bOption, byte[] pBlocks, byte bBlocksLen,
            ref IntPtr ppTMData, ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Decrement ( ref DataParams_t pDataParams, byte bOption, byte[] pBlocks, byte bBlocksLen,
            ref IntPtr ppTMData, ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_Restore ( ref DataParams_t pDataParams, byte bOption, byte[] pBlocks, byte bBlocksLen,
            ref IntPtr ppTMData, ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_AuthenticatedRead ( ref DataParams_t pDataParams, ushort wOption, byte[] pMFUid, byte bCmdSettings,
            byte bKeyNo, byte bKeyVer, byte bKeyType, byte bAuthBlockNo, byte bDivBlockNo, byte[] pBlocks, byte bBlocksLen, ref IntPtr ppData, ref ushort pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_AuthenticatedWrite ( ref DataParams_t pDataParams, ushort wOption, byte[] pMFUid, byte bCmdSettings,
            byte bKeyNo, byte bKeyVer, byte bKeyType, byte bAuthBlockNo, byte bDivBlockNo, byte[] pBlocks, byte bBlocksLen, ref IntPtr ppTMData,
            ref ushort pTMDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_MF_ChangeKey ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVerA, byte bKeyVerB,
            byte bMFBlockNo, byte[] pAccCond, byte[] pMFUID, byte bDivBlockNo );
        #endregion X Mode
        #endregion MIFARE Classic

        #region MIFARE Ultralight
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part1 ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, ref IntPtr ppPwd, ref ushort pPwdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part2 ( ref DataParams_t pDataParams, ushort wPACK );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_UL_PwdAuthUL ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer, byte[] pDivInput,
            byte bDivInputLen, ref byte pStatusCode );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_ULC_AuthenticatePICC ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pDivInput, byte bDivInputLen, ref byte pStatusCode );
        #endregion X Mode
        #endregion MIFARE Ultralight

        #region MIFARE Common
        #region S Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part1 ( ref DataParams_t pDataParams, byte bState, ushort wBlockNr,
            ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part2 ( ref DataParams_t pDataParams, byte bPiccErrCode, byte[] pData, byte bDataLen,
            ref byte pPiccReturnCode );
        #endregion S Mode

        #region X Mode
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_TMRI_CommitReaderID ( ref DataParams_t pDataParams, byte bISOMode, byte bState, ushort wBlockNr,
            ref IntPtr ppEncTMRI, ref ushort pEncTMRILen, ref byte pStatusCode );
        #endregion X Mode
        #endregion MIFARE Common

        #region ISO / IEC 29167 - 10
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1 ( ref DataParams_t pDataParams, byte bOption, byte bKeyNo, byte bKeyVer,
            byte[] pData, byte bDataLen, ref IntPtr ppIChallange, ref ushort pIChallangeLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM2 ( ref DataParams_t pDataParams, byte bOption, ushort wKeyNo, ushort wKeyVer,
            byte[] pData, byte bDataLen, byte bBlockSize, byte bBlockCount, byte bProtMode, ref IntPtr ppResponse, ref ushort pResponseLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM1 ( ref DataParams_t pDataParams, byte bKeyNo, byte bKeyVer, byte[] pData,
            byte bDataLen, byte bPurposeMAM2, ref IntPtr ppIChallange, ref ushort pIChallangeLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM2 ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen, ref IntPtr ppIResponse,
            ref ushort pIResponseLen );
        #endregion ISO / IEC 29167 - 10

        #region EMV
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_EMVCo_RecoverStaticData ( ref DataParams_t pDataParams, ushort wOption, byte[] bSignedDynamicAppData,
            byte bSignedDynamicAppDataLen, ref IntPtr pRxBuffer, ref byte bRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_EMVCo_RecoverDynamicData ( ref DataParams_t pDataParams, ushort wOption, byte[] bSignedDynamicAppData,
            byte bSignedDynamicAppDataLen, ref IntPtr pRxBuffer, ref byte bRxLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_EMVCo_EncipherPin ( ref DataParams_t pDataParams, byte[] pPinBlock, byte[] pIccNumber,
            ref IntPtr bRxBuffer, ref byte pRxLength );
        #endregion EMV

        #region Programmable Logic
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_PLExec ( ref DataParams_t pDataParams, byte bLFI, byte[] pPLData, byte bPLDataLen,
            ref IntPtr ppPLResp, ref ushort pPLResp );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_SAM_PLUpload ( ref DataParams_t pDataParams, byte bIsFirstFrame, byte bIsFinalFrame, ushort wUploadCtr,
            byte bKeyNo, byte bKeyVer, byte[] pPLCode, ushort wPLCodeLen, byte[] aPLReKey, byte bPLReKeyLen );
        #endregion

        #region Reader Chips
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_RC_ReadRegister ( ref DataParams_t pDataParams, byte[] RegAddr, byte RegAddrLen, byte[] Value );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_RC_WriteRegister ( ref DataParams_t pDataParams, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_RC_RFControl ( ref DataParams_t pDataParams, ushort wTime );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_RC_Init ( ref DataParams_t pDataParams, byte bLoadReg );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_RC_LoadRegisterValueSet ( ref DataParams_t pDataParams, byte bStoreReg, byte[] pData, byte bDataLen );
        #endregion Reader Chips

        #region ISO14443-3
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_RequestA_Wakeup ( ref DataParams_t pDataParams, byte bCmdCode, byte[] pAtqa );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_AnticollisionSelect ( ref DataParams_t pDataParams, byte[] pSelCodes, byte bSelCodesLen,
            ref byte pSak, byte[] pUid, ref byte pUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_ActivateIdle ( ref DataParams_t pDataParams, byte bOption, byte bNumCards, ushort wTime,
            byte[] pAtqaIn, byte[] pSakIn, ref IntPtr ppResponse, ref ushort pRespLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_ActivateWakeUp ( ref DataParams_t pDataParams, byte[] pUid, byte bUidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_HaltA ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_3_TransparentExchange ( ref DataParams_t pDataParams, byte[] pTxBuf, byte bTxLen,
            byte bTxBitLen, ref IntPtr ppRxBuf, ref ushort pRxLen, ref byte pRxBitLen );
        #endregion

        #region ISO14443-4
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_RATS_PPS ( ref DataParams_t pDataParams, byte bCidIn, byte bDsiIn, byte bDriIn,
            ref byte pCidOut, ref byte pDsiOut, ref byte pDriOut, byte[] pAts );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_Init ( ref DataParams_t pDataParams, byte bCid, byte bDri, byte bDsi,
            byte bFwi, byte bFsci );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange ( ref DataParams_t pDataParams, ushort wOption, byte[] pAppDataIn,
            byte bLenAppData, ref IntPtr ppAppDataOut, ref ushort pAppDataOutLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_PresenceCheck ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_Deselect ( ref DataParams_t pDataParams, byte bFreeCID );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_X_ISO14443_4_FreeCid ( ref DataParams_t pDataParams, byte[] pCid, byte bCidLen );
        #endregion ISO14443-4

        #region Utilities
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_DetectMode ( ref DataParams_t m_pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_SamAV3_Cmd_7816Exchange ( IntPtr pDataParams, ushort wOption, byte[] pTxBuffer, ushort wTxLength,
            ref IntPtr ppRxBuffer, ref ushort pRxLength );
        #endregion Utilities
        #endregion Dll Import

        #region Wrapper Functions
        #region Initialization
        private byte[] m_bHalBuffer;
        private GCHandle m_pTxBuffer;
        private GCHandle m_pRxBuffer;
        private GCHandle m_pPLUploadBuf;

        /// <summary>
        /// Interface to initialize the SAM AV3 componenet. The Tx and Rx buffer size if set to 261 bytes (256 + 5 bytes BAL header / Footer.)
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the hal data params of a reader. NULL in case of X-Mode.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="oPLUpload_ENCCrypto">Pointer to the ENC crypto layers parameter structure. This will be used for ProgrammableLogic feature only.</param>
        /// <param name="oPLUpload_MACCrypto">Pointer to the MAC crypto layers parameter structure. This will be used for ProgrammableLogic feature only.</param>
        /// <param name="eMode">Command counter for Secure Messaging.</param>
        /// <param name="eLc">The desired operation mode.
        ///							NxpRdLibNet.Hal.SamAV3.NON_X
        ///							NxpRdLibNet.Hal.SamAV3.X_RC523
        ///							NxpRdLibNet.Hal.SamAV3.X_RC663</param>
        /// <param name="aPLUploadBuf">Buffer for PL operations.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Bal.Generic oBal, Hal.Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto, CryptoSym.Generic oMACCrypto,
            CryptoRng.Generic oCryptoRng, CryptoSym.Generic oPLUpload_ENCCrypto, CryptoSym.Generic oPLUpload_MACCrypto, OpMode eMode, LogicalChannel eLc,
            byte[] aPLUploadBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;

            if ( oBal != null ) pBalPtr = oBal.m_pDataParams;
            if ( oReaderHal != null ) pReaderHalPtr = oReaderHal.m_pDataParams;
            if ( oKeyStore != null ) pKeyStorePtr = oKeyStore.m_pDataParams;
            if ( oENCCrypto != null ) pENCCryptoPtr = oENCCrypto.m_pDataParams;
            if ( oMACCrypto != null ) pMACCryptoPtr = oMACCrypto.m_pDataParams;
            if ( oCryptoRng != null ) pCryptoRngPtr = oCryptoRng.m_pDataParams;
            if ( oPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = oPLUpload_ENCCrypto.m_pDataParams;
            if ( oPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = oPLUpload_MACCrypto.m_pDataParams;

            // Free Buffer
            if ( this.m_pTxBuffer.IsAllocated ) this.m_pTxBuffer.Free ();
            if ( this.m_pRxBuffer.IsAllocated ) this.m_pRxBuffer.Free ();
            if ( this.m_pPLUploadBuf.IsAllocated ) this.m_pPLUploadBuf.Free ();

            // Use only one Buffer with fixed length
            m_bHalBuffer = new byte[261];
            m_pTxBuffer = GCHandle.Alloc ( m_bHalBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( m_bHalBuffer, GCHandleType.Pinned );
            m_pPLUploadBuf = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );

            return phhalHw_SamAV3_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pBalPtr, pReaderHalPtr, pKeyStorePtr,
                pENCCryptoPtr, pMACCryptoPtr, pCryptoRngPtr, pPLUpload_ENCCryptoPtr, pPLUpload_MACCryptoPtr, ( byte ) eMode, ( byte ) eLc, m_pTxBuffer.AddrOfPinnedObject (),
                261, m_pRxBuffer.AddrOfPinnedObject (), 261, m_pPLUploadBuf.AddrOfPinnedObject () );
        }

        /// <summary>
        /// Interface to initialize the SAM AV3 componenet.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="oReaderHal">Pointer to the hal data params of a reader. NULL in case of X-Mode.</param>
        /// <param name="oKeyStore">Pointer to the KeyStore used for Host Authentication.</param>
        /// <param name="oENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="oMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="oCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="oPLUpload_ENCCrypto">Pointer to the ENC crypto layers parameter structure. This will be used for ProgrammableLogic feature only.</param>
        /// <param name="oPLUpload_MACCrypto">Pointer to the MAC crypto layers parameter structure. This will be used for ProgrammableLogic feature only.</param>
        /// <param name="eMode">Command counter for Secure Messaging.</param>
        /// <param name="eLc">The desired operation mode.
        ///							NxpRdLibNet.Hal.SamAV3.NON_X
        ///							NxpRdLibNet.Hal.SamAV3.X_RC523
        ///							NxpRdLibNet.Hal.SamAV3.X_RC663</param>
        ///	<param name="aTxBuffer">Buffer for handling the messages to be sent to SAM.</param>
        ///	<param name="aRxBuffer">Buffer for handling the messages received from SAM.</param>
        /// <param name="aPLUploadBuf">Buffer for PL operations.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Bal.Generic oBal, Hal.Generic oReaderHal, KeyStore.Generic oKeyStore, CryptoSym.Generic oENCCrypto, CryptoSym.Generic oMACCrypto,
            CryptoRng.Generic oCryptoRng, CryptoSym.Generic oPLUpload_ENCCrypto, CryptoSym.Generic oPLUpload_MACCrypto, OpMode eMode, LogicalChannel eLc,
             byte[] aTxBuffer, byte[] aRxBuffer, byte[] aPLUploadBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;

            if ( oBal != null ) pBalPtr = oBal.m_pDataParams;
            if ( oReaderHal != null ) pReaderHalPtr = oReaderHal.m_pDataParams;
            if ( oKeyStore != null ) pKeyStorePtr = oKeyStore.m_pDataParams;
            if ( oENCCrypto != null ) pENCCryptoPtr = oENCCrypto.m_pDataParams;
            if ( oMACCrypto != null ) pMACCryptoPtr = oMACCrypto.m_pDataParams;
            if ( oCryptoRng != null ) pCryptoRngPtr = oCryptoRng.m_pDataParams;
            if ( oPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = oPLUpload_ENCCrypto.m_pDataParams;
            if ( oPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = oPLUpload_MACCrypto.m_pDataParams;

            // Free Buffer
            if ( this.m_pTxBuffer.IsAllocated ) this.m_pTxBuffer.Free ();
            if ( this.m_pRxBuffer.IsAllocated ) this.m_pRxBuffer.Free ();
            if ( this.m_pPLUploadBuf.IsAllocated ) this.m_pPLUploadBuf.Free ();

            // Use only one Buffer with fixed length
            m_pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );
            m_pPLUploadBuf = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );

            return phhalHw_SamAV3_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pBalPtr, pReaderHalPtr, pKeyStorePtr,
                pENCCryptoPtr, pMACCryptoPtr, pCryptoRngPtr, pPLUpload_ENCCryptoPtr, pPLUpload_MACCryptoPtr, ( byte ) eMode, ( byte ) eLc, m_pTxBuffer.AddrOfPinnedObject (),
                ( ushort ) aTxBuffer.Length, m_pRxBuffer.AddrOfPinnedObject (), ( ushort ) aRxBuffer.Length, m_pPLUploadBuf.AddrOfPinnedObject () );
        }

        /// <summary>
        /// Interface to initialize the SAM AV3 componenet. The Tx and Rx buffer size if set to 261 bytes (256 + 5 bytes BAL header / Footer.)
        /// </summary>
        ///
        /// <param name="pBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="pReaderHal">Pointer to the hal data params of a reader. NULL in case of X-Mode.</param>
        /// <param name="pKeyStore">Pointer to the KeyStore used for Host Authentication.</param>
        /// <param name="pENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="pMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="pCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="Mode">Command counter for Secure Messaging.</param>
        /// <param name="Lc">The desired operation mode.
        ///							NxpRdLibNet.Hal.SamAV3.NON_X
        ///							NxpRdLibNet.Hal.SamAV3.X_RC523
        ///							NxpRdLibNet.Hal.SamAV3.X_RC663</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Bal.Generic pBal, Hal.Generic pReaderHal, KeyStore.Generic pKeyStore, CryptoSym.Generic pENCCrypto, CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng, OpMode Mode, LogicalChannel Lc )
        {
            return Init ( pBal, pReaderHal, pKeyStore, pENCCrypto, pMACCrypto, pCryptoRng, null, null, Mode, Lc, new byte[0] );
        }

        /// <summary>
        /// Interface to initialize the SAM AV3 componenet.
        /// </summary>
        ///
        /// <param name="pBal">Pointer to the lower layers parameter structure.</param>
        /// <param name="pReaderHal">Pointer to the hal data params of a reader. NULL in case of X-Mode.</param>
        /// <param name="pKeyStore">Pointer to the KeyStore used for Host Authentication.</param>
        /// <param name="pENCCrypto">Pointer to the ENC crypto layers parameter structure.</param>
        /// <param name="pMACCrypto">Pointer to the MAC crypto layers parameter structure.</param>
        /// <param name="pCryptoRng">Pointer to the parameter structure of the CryptoRng layer.</param>
        /// <param name="Mode">Command counter for Secure Messaging.</param>
        /// <param name="Lc">The desired operation mode.
        ///							NxpRdLibNet.Hal.SamAV3.NON_X
        ///							NxpRdLibNet.Hal.SamAV3.X_RC523
        ///							NxpRdLibNet.Hal.SamAV3.X_RC663</param>
        ///	<param name="aTxBuffer">Buffer for handling the messages to be sent to SAM.</param>
        ///	<param name="aRxBuffer">Buffer for handling the messages received from SAM.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Bal.Generic pBal, Hal.Generic pReaderHal, KeyStore.Generic pKeyStore, CryptoSym.Generic pENCCrypto, CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng, OpMode Mode, LogicalChannel Lc, byte[] aTxBuffer, byte[] aRxBuffer )
        {
            return Init ( pBal, pReaderHal, pKeyStore, pENCCrypto, pMACCrypto, pCryptoRng, null, null, Mode, Lc, aTxBuffer, aRxBuffer, new byte[0] );
        }

#if DEBUG
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init ( int wDataParamSize, Bal.Generic pBal, Hal.Generic pReaderHal, KeyStore.Generic pKeyStore, CryptoSym.Generic pENCCrypto, CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng, CryptoSym.Generic pPLUpload_ENCCrypto, CryptoSym.Generic pPLUpload_MACCrypto, OpMode Mode, LogicalChannel Lc, int wTxBufferSize,
            byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer, byte[] aPLUploadBuf )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;
            IntPtr pPLUpload_ENCCryptoPtr = IntPtr.Zero;
            IntPtr pPLUpload_MACCryptoPtr = IntPtr.Zero;

            if ( pBal != null ) pBalPtr = pBal.m_pDataParams;
            if ( pReaderHal != null ) pReaderHalPtr = pReaderHal.m_pDataParams;
            if ( pKeyStore != null ) pKeyStorePtr = pKeyStore.m_pDataParams;
            if ( pENCCrypto != null ) pENCCryptoPtr = pENCCrypto.m_pDataParams;
            if ( pMACCrypto != null ) pMACCryptoPtr = pMACCrypto.m_pDataParams;
            if ( pCryptoRng != null ) pCryptoRngPtr = pCryptoRng.m_pDataParams;
            if ( pPLUpload_ENCCrypto != null ) pPLUpload_ENCCryptoPtr = pPLUpload_ENCCrypto.m_pDataParams;
            if ( pPLUpload_MACCrypto != null ) pPLUpload_MACCryptoPtr = pPLUpload_MACCrypto.m_pDataParams;

            // Free Buffers
            if ( this.m_pTxBuffer.IsAllocated )
                this.m_pTxBuffer.Free ();
            if ( this.m_pRxBuffer.IsAllocated )
                this.m_pRxBuffer.Free ();
            if ( this.m_pPLUploadBuf.IsAllocated )
                this.m_pPLUploadBuf.Free ();

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc ( bTxBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( bRxBuffer, GCHandleType.Pinned );
            m_pPLUploadBuf = GCHandle.Alloc ( aPLUploadBuf, GCHandleType.Pinned );

            return phhalHw_SamAV3_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, pBalPtr, pReaderHalPtr, pKeyStorePtr, pENCCryptoPtr, pMACCryptoPtr,
                pCryptoRngPtr, pPLUpload_ENCCryptoPtr, pPLUpload_MACCryptoPtr, ( byte ) Mode, ( byte ) Lc, m_pTxBuffer.AddrOfPinnedObject (), ( ushort ) wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize, m_pPLUploadBuf.AddrOfPinnedObject () );
        }

        public Status_t Init ( int wDataParamSize, Bal.Generic pBal, Hal.Generic pReaderHal, KeyStore.Generic pKeyStore, CryptoSym.Generic pENCCrypto, CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng, OpMode Mode, LogicalChannel Lc, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer )
        {
            return Init ( wDataParamSize, pBal, pReaderHal, pKeyStore, pENCCrypto, pMACCrypto, pCryptoRng, null, null, Mode, Lc, wTxBufferSize, bTxBuffer,
                wRxBufferSize, bRxBuffer, new byte[0] );
        }
#endif
        #endregion Initialization

        #region Host Communication
        /// <summary>
        /// Lock or Unlock the SAM. Its important to perform Host Authentication after successfull LockUnlock operation to set the new session keys.
        /// Host Authentication is required because, LockUnlock interface will utilize the same Cryptoparams which was utilized by Host Authentication
        /// interface. Since the Cryptoparams are common, the keys will be updated by LockUnlock interface and the exisiting Host Authentication session
        /// keys will not be available.
        /// </summary>
        ///
        /// <param name="bMode">Sub Command type.
        ///						NxpRdLibNet.Hal.SamAV3.LockUnlock.UNLOCK
        ///						NxpRdLibNet.Hal.SamAV3.LockUnlock.LOCK_NO_KEY
        ///						NxpRdLibNet.Hal.SamAV3.LockUnlock.LOCK_KEY
        ///						NxpRdLibNet.Hal.SamAV3.LockUnlock.ACTIVATE_SAM
        ///						NxpRdLibNet.Hal.SamAV3.LockUnlock.UNLOCK_PL</param>
        /// <param name="bRdKeyNo">Key reference number to be used in software key store.</param>
        /// <param name="bRdKeyVer">Key version to be used in software keystore.</param>
        /// <param name="bSamKeyNo">Key reference number in hardware keytsore.</param>
        /// <param name="bSamKeyVer">Key version to be used in hardware key store.</param>
        /// <param name="bUnlockKeyNo">Unlock Key Number to be used in hardware key store (only used when bCmdType = 0x01).</param>
        /// <param name="bUnlockKeyVer">Unlock Key Version to be used in hardware key store (only used when bCmdType = 0x01).</param>
        /// <param name="dwMaxChainBlocks">Maximal message size under command chaining in MAC or Full Protection (only used during switch from AV2 to AV3).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_LockUnlock ( byte bMode, byte bRdKeyNo, byte bRdKeyVer, byte bSamKeyNo, byte bSamKeyVer, byte bUnlockKeyNo, byte bUnlockKeyVer, uint dwMaxChainBlocks )
        {
            return phhalHw_SamAV3_Cmd_SAM_LockUnlock ( ref m_DataParamsInt[0], bMode, bRdKeyNo, bRdKeyVer, bSamKeyNo, bSamKeyVer, bUnlockKeyNo, bUnlockKeyVer, dwMaxChainBlocks );
        }

        /// <summary>
        /// Mutual 3-pass-AV2 or 3-Pass AV3 authentication between Host and SAM.
        /// </summary>
        ///
        /// <param name="bHostMode">Type of Protection mode to be applied.
        ///						NxpRdLibNet.Hal.SamAV3.HostMode.PLAIN
        ///						NxpRdLibNet.Hal.SamAV3.HostMode.MAC
        ///						NxpRdLibNet.Hal.SamAV3.HostMode.FULL</param>
        /// <param name="bRdKeyNo">Key reference number to be used in software key store.</param>
        /// <param name="bRdKeyVer">Key version to be used in software keystore.</param>
        /// <param name="bSamKeyNo">Key reference number in hardware keytsore.</param>
        /// <param name="bSamKeyVer">Key version to be used in hardware key store.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateHost ( byte bHostMode, byte bRdKeyNo, byte bRdKeyVer, byte bSamKeyNo, byte bSamKeyVer )
        {
            return phhalHw_SamAV3_Cmd_SAM_AuthenticateHost ( ref m_DataParamsInt[0], bHostMode, bRdKeyNo, bRdKeyVer, bSamKeyNo, bSamKeyVer );
        }
        #endregion Host Communication

        #region Security and Configuration
        /// <summary>
        /// Get version information from the SAM.
        /// </summary>
        ///
        /// <param name="aVersion">Buffer containing the read version.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GetVersion ( out byte[] aVersion )
        {
            aVersion = new byte[31]; // Max Version Lenght is 31 bytes
            byte bVersionLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_GetVersion ( ref m_DataParamsInt[0], aVersion, ref bVersionLen );

            Array.Resize<byte> ( ref aVersion, ( int ) bVersionLen );

            return oStatus;
        }

        /// <summary>
        /// Disable cryto-related features of the SAM permanently and is irreversible.
        /// </summary>
        ///
        /// <param name="wProMas">Sub Command type.
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.NO_CHANGE
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.DES_PICC_CHANGE_KEY
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.DECRYPTION
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.ENCRYPTION
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.MAC_VERIFICATION
        ///							NxpRdLibNet.Hal.SamAV3.DisableCrypto.MAC_GENERATION</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DisableCrypto ( ushort wProMas )
        {
            return phhalHw_SamAV3_Cmd_SAM_DisableCrypto ( ref m_DataParamsInt[0], wProMas );
        }

        /// <summary>
        /// Activation of an OfflineCrypto or an OfflineChange Key. This command is available in both AV2 and AV3 mode.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information if Keytype is AES_128LRP.
        ///							NxpRdLibNet.Hal.SamAV3.LRPUpdate.ONE
        ///							NxpRdLibNet.Hal.SamAV3.LRPUpdate.TWO
        ///							NxpRdLibNet.Hal.SamAV3.LRPUpdate.RFU if keytype is other than AES_128LRP.</param>
        /// <param name="bKeyNo">Key reference number in hardware keytsore.</param>
        /// <param name="bKeyVer">Key version to be used in hardware key store.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ActivateOfflineKey ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput )
        {
            return phhalHw_SamAV3_Cmd_SAM_ActivateOfflineKey ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );

        }

        /// <summary>
        /// Load an Init Vector for the next cryptographic operation into the SAM.
        /// </summary>
        ///
        /// <param name="bOption">One of the below option.
        ///							NxpRdLibNet.Hal.SamAV3.ModeIndicator.SET_IV
        ///							NxpRdLibNet.Hal.SamAV3.ModeIndicator.SET_LRP_ENC_CTR</param>
        /// <param name="aData">Data based on the option selected.
        ///							If NxpRdLibNet.Hal.SamAV3.ModeIndicator.SET_IV,
        ///								8 bytes of initializtion vector for DES or 16 bytes of initializtion vector for AES.
        ///
        ///
        ///							If NxpRdLibNet.Hal.SamAV3.ModeIndicator.SET_LRP_ENC_CTR, 16 bytes Encryption counter.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_LoadInitVector ( byte bOption, byte[] aData )
        {
            return phhalHw_SamAV3_Cmd_SAM_LoadInitVector ( ref m_DataParamsInt[0], bOption, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Kill all active authentications in this logical channel.
        /// </summary>
        ///
        /// <param name="bOption">The type of authentication to be killed.
        ///							NxpRdLibNet.Hal.SamAV3.KillAuth.FULL
        ///							NxpRdLibNet.Hal.SamAV3.KillAuth.PARTIAL</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_KillAuthentication ( byte bOption )
        {
            return phhalHw_SamAV3_Cmd_SAM_KillAuthentication ( ref m_DataParamsInt[0], bOption );
        }

        /// <summary>
        /// Select an application by the DF_AID
        /// </summary>
        ///
        /// <param name="aDF_Aid">Desfire application identifier (3 bytes).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_SelectApplication ( byte[] aDF_Aid )
        {
            return phhalHw_SamAV3_Cmd_SAM_SelectApplication ( ref m_DataParamsInt[0], aDF_Aid );
        }

        /// <summary>
        /// Gets a random number.
        /// </summary>
        ///
        /// <param name="bExpLen">The length of random bytes expected form Sam hardware.</param>
        /// <param name="aRnd">The random number returned by Sam.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GetRandom ( byte bExpLen, out byte[] aRnd )
        {
            aRnd = new byte[bExpLen];
            return phhalHw_SamAV3_Cmd_SAM_GetRandom ( ref m_DataParamsInt[0], bExpLen, aRnd );
        }

        /// <summary>
        /// Set the SAM into power down mode.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_Sleep ()
        {
            return phhalHw_SamAV3_Cmd_SAM_Sleep ( ref m_DataParamsInt[0] );

        }

        /// <summary>
        /// Used to update SAM configuration settings.
        /// </summary>
        ///
        /// <param name="bOption">Configuration setting ID. Define length and content of the Data parameter.
        ///							NxpRdLibNet.Hal.SamAV3.SetConfiguration.HISTORICAL_BYTES
        ///							NxpRdLibNet.Hal.SamAV3.SetConfiguration.READER_IC_CONFIG
        ///							NxpRdLibNet.Hal.SamAV3.SetConfiguration.I2C_CLOCK_SPEED
        ///							NxpRdLibNet.Hal.SamAV3.SetConfiguration.EMV_CHANGE_KEYS
        ///							NxpRdLibNet.Hal.SamAV3.SetConfiguration.FULL_ATR</param>
        /// <param name="aData">Configuration setting data.
        ///						If NxpRdLibNet.Hal.SamAV3.SetConfiguration.HISTORICAL_BYTES, the historical bytes should be exchanged.
        ///
        ///						If NxpRdLibNet.Hal.SamAV3.SetConfiguration.READER_IC_CONFIG, one of the below values
        ///								NxpRdLibNet.Hal.SamAV3.ReaderIC.READER_IC_CONFIG_RC512
        ///								NxpRdLibNet.Hal.SamAV3.ReaderIC.READER_IC_CONFIG_RC523
        ///								NxpRdLibNet.Hal.SamAV3.ReaderIC.READER_IC_CONFIG_RC663
        ///
        ///						If NxpRdLibNet.Hal.SamAV3.SetConfiguration.I2C_CLOCK_SPEED, one of the below values
        ///							0x00 - 0x01: Contact external Clock.
        ///							0x04       : 0.5 MHz.
        ///							0x05       : 1 MHz.
        ///							0x06       : 2 MHz.
        ///							0x07       : 3 MHz.
        ///							0x08       : 4 MHz.
        ///							0x09       : 6 MHz.
        ///							0x0A       : 8 MHz.
        ///							0x0B       : 12 MHz. (Default)
        ///							0x0C       : 16 MHz.
        ///							0x0F       : Free Running mode.
        ///
        ///						If NxpRdLibNet.Hal.SamAV3.SetConfiguration.EMV_CHANGE_KEYS,
        ///							4 bytes key information as EMV_DefaultKeyNoCEK1, EMV_DefaultVCEK1,
        ///							EMV_DefaultKeyNoCEK2, EMV_DefaultVCEK2</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_SetConfiguration ( byte bOption, byte[] aData )
        {
            return phhalHw_SamAV3_Cmd_SAM_SetConfiguration ( ref m_DataParamsInt[0], bOption, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion Security and Configuration

        #region Key Management
        /// <summary>
        /// Change a symmetric key entry in the key table of the SAM.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be changed (00h to 7Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///						  option can be combined by using bitwise OR operator.
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VA
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VB
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_DF_AID
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_CEK
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_REF_NO_KUC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_SET_EXTSET
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.INCLUDE_VERSION</param>
        /// <param name="aKeyData">Buffer containing the key data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyEntry ( byte bKeyNo, byte bProMas, byte[] aKeyData )
        {
            return phhalHw_SamAV3_Cmd_SAM_ChangeKeyEntry ( ref m_DataParamsInt[0], bKeyNo, bProMas, aKeyData,
                ( byte ) ( ( aKeyData == null ) ? 0 : aKeyData.Length ) );
        }

        /// <summary>
        /// Change a Offline symmetric key entry in the key entry table of the SAM.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be changed (00h to 7Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///						  option can be combined by using bitwise OR operator.
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VA
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VB
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_VC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_DF_AID
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_CEK
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_REF_NO_KUC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_SET_EXTSET
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.INCLUDE_VERSION</param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCryptogram">Offline Cryptogram returned from SAM (EncKeyEntry[80] + OfflineMAC[8])</param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///										0: Disable (The OfflineAck buffer will be null)
        ///										1: Enable (OfflineAck buffer will be filed with information)</param>
        /// <param name="aOfflineAck">Offline Acknowledge information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyEntryOffline ( byte bKeyNo, byte bProMas, ushort wChangeCtr, byte[] aOfflineCryptogram, byte bEnableOfflineAck,
            out byte[] aOfflineAck )
        {
            aOfflineAck = new byte[8]; /* Fixed length of Offline ACK 8 bytes */
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyEntryOffline ( ref m_DataParamsInt[0], bKeyNo, bProMas, wChangeCtr, aOfflineCryptogram,
                (byte) ( ( aOfflineCryptogram == null ) ? 0 : aOfflineCryptogram.Length ), bEnableOfflineAck, aOfflineAck );

            if ( bEnableOfflineAck.Equals ( 0 ) || !oStatus.Equals ( new Status_t () ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Get information about a key entry.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be returned (00h to 7Fh).</param>
        /// <param name="bMode">Key entry format to be used.
        ///							NxpRdLibNet.Hal.SamAV3.KeyEntryFormat.SAM_AV2
        ///							NxpRdLibNet.Hal.SamAV3.KeyEntryFormat.NEW
        ///							NxpRdLibNet.Hal.SamAV3.KeyEntryFormat.RAM_KEY</param>
        /// <param name="aKeyEntry">Buffer containing the information about the key entry.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GetKeyEntry ( byte bKeyNo, byte bMode, out byte[] aKeyEntry )
        {
            byte bKeyEntryLen = 0;

            aKeyEntry = new byte[16]; // Max key information length is 16 bytes

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_GetKeyEntry ( ref m_DataParamsInt[0], bKeyNo, bMode, aKeyEntry, ref bKeyEntryLen );

            Array.Resize<byte> ( ref aKeyEntry, ( int ) bKeyEntryLen );
            return oStatus;
        }

        /// <summary>
        /// Change the key usage counter (KUC). Selection is done by its reference number.
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be updated (00h to 0Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///						  option can be combined by using bitwise OR operator.
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_LIMIT
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_NO_CKUC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_VCKUC</param>
        /// <param name="aKucData">Buffer containing the KUC data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKUCEntry ( byte bKucNo, byte bProMas, byte[] aKucData )
        {
            return phhalHw_SamAV3_Cmd_SAM_ChangeKUCEntry ( ref m_DataParamsInt[0], bKucNo, bProMas, aKucData,
                ( byte ) ( ( aKucData == null ) ? 0 : aKucData.Length ) );
        }

        /// <summary>
        /// Change a Offline KUC entry in the key entry table of the SAM.
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be updated (00h to 0Fh).</param>
        /// <param name="bProMas">Program mask indicating the fields that should be changed. All the below
        ///						  option can be combined by using bitwise OR operator.
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_LIMIT
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_KEY_NO_CKUC
        ///							NxpRdLibNet.Hal.SamAV3.ProMas.UPDATE_VCKUC</param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCryptogram">Offline Cryptogram returned from SAM (EncKUCEntry[16] + OfflineMAC[8])</param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///										0: Disable (The OfflineAck buffer will be null)
        ///										1: Enable (OfflineAck buffer will be filed with information)</param>
        /// <param name="aOfflineAck">Offline Acknowledge information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKUCEntryOffline ( byte bKucNo, byte bProMas, ushort wChangeCtr, byte[] aOfflineCryptogram, byte bEnableOfflineAck,
            out byte[] aOfflineAck )
        {
            aOfflineAck = new byte[8]; /* Fixed length of Offline ACK 8 bytes */
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKUCEntryOffline ( ref m_DataParamsInt[0], bKucNo, bProMas, wChangeCtr, aOfflineCryptogram,
                ( byte ) ( ( aOfflineCryptogram == null ) ? 0 : aOfflineCryptogram.Length ), bEnableOfflineAck, aOfflineAck );

            if ( bEnableOfflineAck.Equals ( 0 ) || !oStatus.Equals ( new Status_t () ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Get information about a key usage counter (KUC).
        /// </summary>
        ///
        /// <param name="bKucNo">Reference number of the key usage counter to be returned (00h to 0Fh).</param>
        /// <param name="aKucEntry">Buffer containing the KUC entry. This buffer has to be 10 bytes long.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GetKUCEntry ( byte bKucNo, out byte[] aKucEntry )
        {
            aKucEntry = new byte[10];
            return phhalHw_SamAV3_Cmd_SAM_GetKUCEntry ( ref m_DataParamsInt[0], bKucNo, aKucEntry );
        }

        /// <summary>
        /// Dump the current session key.
        /// </summary>
        ///
        /// <param name="bCrypto">Crypto settings.
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.PLAIN
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.ENCIPHERED</param>
        /// <param name="aSessionKey">Buffer containig the session key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DumpSessionKey ( byte bCrypto, out byte[] aSessionKey )
        {
            byte bSessionKeyLen = 0;

            aSessionKey = new byte[40]; // Max SessionKey Lenght is 24 bytes but when a MIFARE Plus SL3 Session key is dumped the length is 32
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_DumpSessionKey ( ref m_DataParamsInt[0], bCrypto, aSessionKey, ref bSessionKeyLen );

            /* Resize SessionKey buffer to actual SessionKey size */
            Array.Resize<byte> ( ref aSessionKey, ( int ) bSessionKeyLen );

            return oStatus;

        }

        /// <summary>
        /// Retrive a PICC key stored in the key table. This command is only available in AV2 and AV3.
        /// </summary>
        ///
        /// <param name="bCrypto">Crypto settings.
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.PLAIN
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.ENCIPHERED
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.Hal.SamAV3.DumpMode.DIVERSIFICATION_ON</param>
        /// <param name="bKeyNo">Reference number of the key entry to be dumped.</param>
        /// <param name="bKeyVer">Reference version of the key entry to be dumped.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aSecretKey">Buffer containing the plain secret key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DumpSecretKey ( byte bCrypto, byte bKeyNo, byte bKeyVer, byte[] aDivInput, out byte[] aSecretKey )
        {
            byte bSecretKeyLen = 0;

            aSecretKey = new byte[32];
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_DumpSecretKey ( ref m_DataParamsInt[0], bCrypto, bKeyNo, bKeyVer, aDivInput,
                (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aSecretKey, ref bSecretKeyLen );

            /* Resize SecretKey buffer to actual SecretKey size */
            Array.Resize<byte> ( ref aSecretKey, ( int ) bSecretKeyLen );

            return oStatus;
        }

        /// <summary>
        /// Disable a key entry.
        /// </summary>
        ///
        /// <param name="bKeyNo">Number of the key entry to be disabled (00h to 7Fh).</param>
        /// <param name="aOfflineCryptogram">Buffer containing the cryptogram for offline key deactivation.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DisableKeyEntry ( byte bKeyNo, byte[] aOfflineCryptogram )
        {
            return phhalHw_SamAV3_Cmd_SAM_DisableKeyEntry ( ref m_DataParamsInt[0], bKeyNo, aOfflineCryptogram,
                ( byte ) ( ( aOfflineCryptogram == null ) ? 0 : aOfflineCryptogram.Length ) );
        }

        /// <summary>
        /// Disable a key entry using offline cryptogram.
        /// </summary>
        ///
        /// <param name="bKeyNo">Reference number of the key entry to be disabled (00h to 7Fh).</param>
        /// <param name="wChangeCtr">Change Counter to avoid replay attacks.</param>
        /// <param name="aOfflineCryptogram">Offline Cryptogram returned from SAM (EncGoldField[16] + OfflineMAC[8])</param>
        /// <param name="bEnableOfflineAck">To Enable reception of Offline Acknowledge
        ///										0: Disable (The OfflineAck buffer will be null)
        ///										1: Enable (OfflineAck buffer will be filed with information)</param>
        /// <param name="aOfflineAck">Offline Acknowledge information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DisableKeyEntryOffline ( byte bKeyNo, ushort wChangeCtr, byte[] aOfflineCryptogram, byte bEnableOfflineAck,
            out byte[] aOfflineAck )
        {
            aOfflineAck = new byte[8]; /* Fixed length of Offline ACK 8 bytes */
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_DisableKeyEntryOffline ( ref m_DataParamsInt[0], bKeyNo, wChangeCtr, aOfflineCryptogram,
                ( byte ) ( ( aOfflineCryptogram == null ) ? 0 : aOfflineCryptogram.Length ), bEnableOfflineAck, aOfflineAck );

            if ( bEnableOfflineAck.Equals ( 0 ) || !oStatus.Equals ( new Status_t () ) )
                aOfflineAck = null;

            return oStatus;
        }

        /// <summary>
        /// Is a Personalization SAM command, used to prepare a cryptogram (according to Offline change protection) for the
        /// OfflineChange key on a target SAM.
        /// </summary>
        ///
        /// <param name="bPersoKeyNo">Reference number of the key entry with Perso enabled.</param>
        /// <param name="bKeyNo">Reference number of the key entry to be encrypted.</param>
        /// <param name="bOption">Option to include Key diversification and SAM UID in command.
        ///							NxpRdLibNet.Hal.SamAV3.EncipherKeyEntry.DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.EncipherKeyEntry.DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.EncipherKeyEntry.SAM_UID_OFF
        ///							NxpRdLibNet.Hal.SamAV3.EncipherKeyEntry.SAM_UID_ON</param>
        /// <param name="wPersoCtr">Change Counter to avoid replay attacks</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aOfflineCryptogram">Offline Cryptogram returned from SAM (EncKeyEntry[80] + OfflineMAC[8])</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_EncipherKeyEntry ( byte bPersoKeyNo, byte bKeyNo, byte bOption, ushort wPersoCtr, byte[] aDivInput, out byte[] aOfflineCryptogram )
        {
            byte bOfflineCryptogramLen = 0;

            aOfflineCryptogram = new byte[88]; /* Max length of Offline Cryptogram is 88 bytes (EncKeyEntry[80] + OfflineMAC[8]) */
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_EncipherKeyEntry ( ref m_DataParamsInt[0], bPersoKeyNo, bKeyNo, bOption, wPersoCtr, aDivInput,
                (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aOfflineCryptogram, ref bOfflineCryptogramLen );

            /* Resize OfflineCryptogram buffer to actual OfflineCryptogram size */
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
                Array.Resize<byte> ( ref aOfflineCryptogram, ( int ) bOfflineCryptogramLen );

            return oStatus;
        }

        /// <summary>
        /// Is used to derive a key from a source key (in other contexts often called master key) based on a CMAC operation. In a MIFARE context,
        /// this command can be used to support session key generations for the Transaction MAC and Secure Dynamic Messaging features, for backend
        /// (and/or reader) interpretation and validation of the cryptograms created by the PICC.
        /// </summary>
        /// <param name="bSrcKeyNo">Reference key number of the key entry (00h to 7Fh).</param>
        /// <param name="bSrcKeyVer">Reference key version of the key entry (00h to FFh).</param>
        /// <param name="bDstKeyNo">Reference key number of the ram key entry (E0h to E3h).</param>
        /// <param name="aDeriveIn">The derivation input for deriving the key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DeriveKey ( byte bSrcKeyNo, byte bSrcKeyVer, byte bDstKeyNo, byte[] aDeriveIn )
        {
            return phhalHw_SamAV3_Cmd_SAM_DeriveKey ( ref m_DataParamsInt[0], bSrcKeyNo, bSrcKeyVer, bDstKeyNo, aDeriveIn,
                ( byte ) ( ( aDeriveIn == null ) ? 0 : aDeriveIn.Length ) );
        }

        /// <summary>
        /// Is used to generate the MIFARE License key.
        /// </summary>
        ///
        /// <param name="bOption">One of the below values.
        ///							NxpRdLibNet.Hal.SamAV3.DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.DIV_ON</param>
        /// <param name="bInputLen">The length of bytes available in Input buffer.</param>
        /// <param name="aInput">Length N of the input data for the MAC computation.</param>
        /// <param name="bKeyCount">Length of bytes available in MFCSectorKeys buffer.</param>
        /// <param name="aMFCSectorKeys">Length N of the input data for the MAC computation.</param>
        /// <param name="aMFUID">MIFARE standard UID. Here the last four bytes of the UID should be passed
        ///						 regardless of 4 or 7 byte UID. This is an optional parameter, so can be NULL.</param>
        /// <param name="aMFCLicMAC">Generated MIFARE Classic License MAC information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GenerateMFCLicKey ( byte bOption, byte bInputLen, byte[] aInput, byte bKeyCount, byte[] aMFCSectorKeys, byte[] aMFUID,
            out byte[] aMFCLicMAC )
        {
            aMFCLicMAC = new byte[8];
            return phhalHw_SamAV3_Cmd_SAM_GenerateMFCLicMAC ( ref m_DataParamsInt[0], bOption, bInputLen, aInput, bKeyCount, aMFCSectorKeys, aMFUID,
                aMFCLicMAC );
        }
        #endregion Key Management

        #region Data Processing
        /// <summary>
        /// Applys the DESFire EV2 Secure Messaging in S-mode on the provided DESFire EV2 command according to the required
        /// mode and the currently activated session keys. The required protection mode is selected via the command parameter bCommMode.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="bCommMode">Communication mode to be used.
        ///								NxpRdLibNet.SamAV3.CommMode.PLAIN
        ///								NxpRdLibNet.SamAV3.CommMode.MAC
        ///								NxpRdLibNet.SamAV3.CommMode.FULL</param>
        /// <param name="bOffset">Command offset. Index of the first byte in data field of the PICC command data.</param>
        /// <param name="bCmdCtrIncr">Command counter increment value. Value by which to increase the CmdCtr.</param>
        /// <param name="aTxData">Plain data to be protected according to the communication mode specified.</param>
        /// <param name="aRxData">The protected data returned by Sam according to communication mode specified.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ApplySM ( ushort wOption, byte bCommMode, byte bOffset, byte bCmdCtrIncr, byte[] aTxData, out byte[] aRxData )
        {
            Status_t oStatus;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_SAM_ApplySM ( ref m_DataParamsInt[0], wOption, bCommMode, bOffset, bCmdCtrIncr, aTxData,
                ( byte ) ( ( aTxData == null ) ? 0 : aTxData.Length ), ref ppRxData, ref wRxDataLen );

            aRxData = MarshalCopy ( oStatus, ppRxData, wRxDataLen );
            return oStatus;
        }

        /// <summary>
        /// Removes the DESFire EV2 Secure Messaging in S-mode ([22]) from the provided PICC response payload according
        /// to the required mode and the currently activated session keys. The required protection mode is selected via the
        /// command parameter bCommMode.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="bCommMode">Communication mode to be used.
        ///								NxpRdLibNet.SamAV3.CommMode.MAC
        ///								NxpRdLibNet.SamAV3.CommMode.FULL</param>
        /// <param name="aTxData">The complete data received form the PICC including the status code.</param>
        /// <param name="aRxData">The plain data returned by Sam according to communication mode specified.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_RemoveSM ( ushort wOption, byte bCommMode, byte[] aTxData, out byte[] aRxData )
        {
            Status_t oStatus;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_SAM_RemoveSM ( ref m_DataParamsInt[0], wOption, bCommMode, aTxData,
                ( byte ) ( ( aTxData == null ) ? 0 : aTxData.Length ), ref ppRxData, ref wRxDataLen );

            aRxData = MarshalCopy ( oStatus, ppRxData, wRxDataLen );
            return oStatus;
        }

        /// <summary>
        /// Verifies the MAC which was sent by the PICC or any other system based on the given MACed plain text data
        /// and the currently valid cryptographic key.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="bNum">Type of truncation mode to be applied if AES key type is used.
        ///							NxpRdLibNet.SamAV3.TruncationMode.STANDARD
        ///							NxpRdLibNet.SamAV3.TruncationMode.MFP
        ///
        ///							Number of MAC bytes to check in the plain data buffer.
        ///							Number of Bytes should be combined with one of the above two options.</param>
        /// <param name="aTxData">Plain data including the MAC to be checked.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_VerifyMAC ( ushort wOption, byte bNum, byte[] aTxData )
        {
            return phhalHw_SamAV3_Cmd_SAM_VerifyMAC ( ref m_DataParamsInt[0], wOption, bNum, aTxData,
                ( byte ) ( ( aTxData == null ) ? 0 : aTxData.Length ) );
        }

        /// <summary>
        /// Generates a MAC which is meant to be sent to the PICC or any other system based on the given plain text data
        /// and the currently valid cryptographic key.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.
        ///							NxpRdLibNet.SamAV3.IncludeLC.INCLUDE should be used for inclusion of
        ///							LC when NxpRdLibNet.ExchangeOptions.BUFFER_FIRST is used and TxDataLen is not available but
        ///							TxDataLen is part of NxpRdLibNet.ExchangeOptions.BUFFER_CONT or NxpRdLibNet.ExchangeOptions.BUFFER_LAST</param>
        /// <param name="bNum">Type of truncation mode to be applied if AES key type is used.
        ///							NxpRdLibNet.SamAV3.TruncationMode.STANDARD
        ///							NxpRdLibNet.SamAV3.TruncationMode.MFP
        ///
        ///							Number of MAC bytes to check in the plain data buffer.
        ///							Number of Bytes should be combined with one of the above two options.</param>
        /// <param name="aTxData">Plain data to be maced. Can be null if there is no data. </param>
        /// <param name="aRxData">The generated MAC returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_GenerateMAC ( ushort wOption, byte bNum, byte[] aTxData, out byte[] aRxData )
        {
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_GenerateMAC ( ref m_DataParamsInt[0], wOption, bNum, aTxData,
                (byte) ( ( aTxData == null ) ? 0 : aTxData.Length ), ref ppRxData, ref wRxDataLen );

            aRxData = MarshalCopy ( oStatus, ppRxData, wRxDataLen );
            return oStatus;
        }

        /// <summary>
        /// Deciphers data packages sent by a PICC, any other system or a MIFARE card based on the currently valid
        /// cryptographic key and returns plain data to the host.
        /// </summary>
        ///
        /// <param name="wOption">Option for including the length information in command frame.
        ///							NxpRdLibNet.SamAV3.Length.EXCLUDE
        ///							NxpRdLibNet.SamAV3.Length.INCLUDE
        ///
        ///						  Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="aEncData">Encrypted data to be deciphered.</param>
        /// <param name="dwLength">Overall length of encrypted input data. This 3 byte value is only used if indicated by Option.</param>
        /// <param name="aPlainData">Deciphered data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DecipherData ( ushort wOption, byte[] aEncData, int dwLength, out byte[] aPlainData )
        {
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            byte[] aLength = new byte[3];
            aLength[0] = ( byte ) dwLength;
            aLength[1] = ( byte ) ( dwLength >> 8 );
            aLength[2] = ( byte ) ( dwLength >> 16 );

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_DecipherData ( ref m_DataParamsInt[0], wOption, aEncData, (byte) ( ( aEncData == null ) ? 0 : aEncData.Length ),
                aLength, ref ppPlainData, ref wPlainDataLen );

            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );
            return oStatus;
        }

        /// <summary>
        /// Enciphers data packages which are meant to be sent to a PICC or any other system based on the given
        /// plain text data and the currently valid cryptographic key.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="aPlainData">Plain data to be enciphered.</param>
        /// <param name="bOffset">Offset into the input data indicating the first data byte to be enciphered.</param>
        /// <param name="aEncData">Enciphered data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_EncipherData ( ushort wOption, byte[] aPlainData, byte bOffset, out byte[] aEncData )
        {
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_EncipherData ( ref m_DataParamsInt[0], wOption, aPlainData,
                (byte) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), bOffset, ref ppPlainData, ref wPlainDataLen );

            aEncData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );
            return oStatus;
        }

        /// <summary>
        /// Decrypts data received from any other system based on the given cipher text data and the currently valid cryptographic
        /// OfflineCrypto Key. The valid key has been activated using a valid key activation (Cmd.SAM_ActivateOfflineKey).
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="aEncData">Encrypted data to be deciphered.</param>
        /// <param name="aPlainData">Deciphered data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_DecipherOfflineData ( ushort wOption, byte[] aEncData, out byte[] aPlainData )
        {
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_DecipherOfflineData ( ref m_DataParamsInt[0], wOption, aEncData,
                (byte) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref ppPlainData, ref wPlainDataLen );

            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );
            return oStatus;
        }

        /// <summary>
        /// Encrypts data received from any other system based on the given cipher text data and the currently valid cryptographic
        /// OfflineCrypto Key. The valid key has been activated using a valid key activation (Cmd.SAM_ActivateOfflineKey).
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING should be used to exchange chunks of data.</param>
        /// <param name="aPlainData">Plain data to be enciphered.</param>
        /// <param name="aEncData">Enciphered data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_EncipherOfflineData ( ushort wOption, byte[] aPlainData, out byte[] aEncData )
        {
            IntPtr ppEncData = IntPtr.Zero;
            ushort wEncDataLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_EncipherOfflineData ( ref m_DataParamsInt[0], wOption, aPlainData,
                (byte) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), ref ppEncData, ref wEncDataLen );

            aEncData = MarshalCopy ( oStatus, ppEncData, wEncDataLen );
            return oStatus;
        }
        #endregion Data Processing

        #region Public Key Infrastructure
        #region RSA
        /// <summary>
        /// Create an RSA key pair. This command is available for AV2 and AV3 version(s).
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.RANDOM_E
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.HOST_E
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P1_AEK_EXCLUDE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P1_AEK_INCLUDE</param>
        /// <param name="bPKI_KeyNo">Reference number of the key entry where the created key should be stored (00h to 01h).</param>
        /// <param name="wPKI_Set">Configuration settings of the created key entry</param>
        /// <param name="bPKI_KeyNoCEK">Reference number to the change key of the created key entry.</param>
        /// <param name="bPKI_KeyVCEK">Version of the change key of the created key entry.</param>
        /// <param name="bPKI_RefNoKUC">Reference number to the KUC of the created key entry.</param>
        /// <param name="bPKI_KeyNoAEK">Reference number to the created access key entry.
        ///									0xFE		: No Access Restrictions
        ///									0xFF		: Entry Disabled
        ///									0x00 - 0x7F : Access key entry number</param>
        /// <param name="bPKI_KeyVAEK">Version of the created acces key entry</param>
        /// <param name="wPKI_NLen">Length of the modulus N (multiple of 8 and in [32;256]).</param>
        /// <param name="wPKI_eLen">Length of the exponent e (multiple of 4 and in [4;256]).</param>
        /// <param name="aPKI_e">Buffer containing the exponent e provided by user or host</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_GenerateKeyPair ( byte bOption, byte bPKI_KeyNo, ushort wPKI_Set, byte bPKI_KeyNoCEK,
            byte bPKI_KeyVCEK, byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen,
            ushort wPKI_eLen, byte[] aPKI_e )
        {
            return phhalHw_SamAV3_Cmd_PKI_GenerateKeyPair ( ref m_DataParamsInt[0], bOption, bPKI_KeyNo, wPKI_Set,
                bPKI_KeyNoCEK, bPKI_KeyVCEK, bPKI_RefNoKUC, bPKI_KeyNoAEK, bPKI_KeyVAEK, wPKI_NLen, wPKI_eLen, aPKI_e );
        }

        /// <summary>
        /// Import a public or private RSA key.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_ONLY
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P1_AEK_EXCLUDE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P1_AEK_INCLUDE</param>
        /// <param name="bPKI_KeyNo">Reference Number of the key entry to be imported (00h to 01h if private key is included
        ///							 or 00h to 02h if not included).</param>
        /// <param name="wPKI_Set">Configuration settings of the imported key entry. It indicates wheter a private of public
        ///						   key shall be imported.</param>
        /// <param name="bPKI_KeyNoCEK">Reference number to the change key of the created key entry.
        ///									0xFE		: No Restrictions
        ///									0xFF		: Entry Locked
        ///									0x00 - 0x7F : Restricted to specific permanent KST Key Entry</param>
        /// <param name="bPKI_KeyVCEK">Version of the change key of the created key entry.</param>
        /// <param name="bPKI_RefNoKUC">Reference number to the KUC of the created key entry.</param>
        /// <param name="bPKI_KeyNoAEK">Reference number to the created access key entry.
        ///									0xFE		: No Access Restrictions
        ///									0xFF		: Entry Disabled
        ///									0x00 - 0x7F : Access key entry number</param>
        /// <param name="bPKI_KeyVAEK">Version of the created acces key entry</param>
        /// <param name="wPKI_NLen">Length of the modulus N (multiple of 8 and in [32;256]).</param>
        /// <param name="wPKI_eLen">Length of the exponent e (multiple of 4 and in [4;256]).</param>
        /// <param name="wPKI_PLen">Length of prime P.</param>
        /// <param name="wPKI_QLen">Length of prime Q.</param>
        /// <param name="aPKI_N">Buffer containing the Modulus N.</param>
        /// <param name="aPKI_e">Buffer containing the Exponent e</param>
        /// <param name="aPKI_p">Buffer containing the Prime P.</param>
        /// <param name="aPKI_q">Buffer containing the Prime P.</param>
        /// <param name="aPKI_dP">Parameter dP padded up to a length of wPKI_PLen.</param>
        /// <param name="aPKI_dQ">Parameter dQ padded up to a length of wPKI_QLen.</param>
        /// <param name="aPKI_ipq">Inverse P(-1) mod Q padded up to a length of wPKI_QLen.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ImportKey ( byte bOption, byte bPKI_KeyNo, ushort wPKI_Set, byte bPKI_KeyNoCEK, byte bPKI_KeyVCEK,
            byte bPKI_RefNoKUC, byte bPKI_KeyNoAEK, byte bPKI_KeyVAEK, ushort wPKI_NLen, ushort wPKI_eLen, ushort wPKI_PLen,
            ushort wPKI_QLen, byte[] aPKI_N, byte[] aPKI_e, byte[] aPKI_p, byte[] aPKI_q, byte[] aPKI_dP, byte[] aPKI_dQ,
            byte[] aPKI_ipq )
        {
            return phhalHw_SamAV3_Cmd_PKI_ImportKey ( ref m_DataParamsInt[0], bOption, bPKI_KeyNo, wPKI_Set, bPKI_KeyNoCEK,
                bPKI_KeyVCEK, bPKI_RefNoKUC, bPKI_KeyNoAEK, bPKI_KeyVAEK, wPKI_NLen, wPKI_eLen, wPKI_PLen, wPKI_QLen,
                aPKI_N, aPKI_e, aPKI_p, aPKI_q, aPKI_dP, aPKI_dQ, aPKI_ipq );
        }

        /// <summary>
        /// Export the private part of an RSA key pair. This command is only available for AV2 and AV3 version(s).
        /// </summary>
        ///
        /// <param name="wOption">Option for AEK selection and differentiating between first part and last part of data.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P2_AEK_EXCLUDE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P2_AEK_INCLUDE
        ///
        ///							Should be combined with the above option.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT (for receiving the first part of data)
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING (for receiving the final part of data)</param>
        /// <param name="bPKI_KeyNo">Reference number of the key entry to be exported (00h to 01h).</param>
        /// <param name="aKeyEntry">Buffer containing the received key data.</param>
        ///
        /// <returns>Returns Success status for successfull operation, command completed.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ExportPrivateKey ( ushort wOption, byte bPKI_KeyNo, out byte[] aKeyEntry )
        {
            Status_t oStatus;
            IntPtr ppKeyData = IntPtr.Zero;
            ushort pKeyDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_ExportPrivateKey ( ref m_DataParamsInt[0], wOption, bPKI_KeyNo, ref ppKeyData, ref pKeyDataLen );

            byte[] RxBuffer = MarshalCopy ( oStatus, ppKeyData, pKeyDataLen );
            aKeyEntry = RxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Export the public part of an RSA key pair. This command is only available for AV2 and AV3 version(s).
        /// </summary>
        ///
        /// <param name="wOption">Option for AEK selection and differentiating between first part and last part of data.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P2_AEK_EXCLUDE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.P2_AEK_INCLUDE
        ///
        ///							Should be combined with the above option.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT (for receiving the first part of data)
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING (for receiving the final part of data)</param>
        /// <param name="bPKI_KeyNo">Reference number of the key entry to be exported (00h to 01h).</param>
        /// <param name="aKeyEntry">Buffer containing the received key data.</param>
        ///
        /// <returns>Returns Success status for successfull operation, command completed.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ExportPublicKey ( ushort wOption, byte bPKI_KeyNo, out byte[] aKeyEntry )
        {
            Status_t oStatus;
            IntPtr ppKeyData = IntPtr.Zero;
            ushort pKeyDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_ExportPublicKey ( ref m_DataParamsInt[0], wOption, bPKI_KeyNo, ref ppKeyData, ref pKeyDataLen );

            byte[] aRxBuffer = MarshalCopy ( oStatus, ppKeyData, pKeyDataLen );
            aKeyEntry = aRxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Change up to 3 symmetric key entries by using PKI.
        /// </summary>
        ///
        /// <param name="bOption">Option to receive the UploadACK.
        ///										NxpRdLibNet.ExchangeOptions.DEFAULT (for receiving the first part of data)
        ///										NxpRdLibNet.ExchangeOptions.RXCHAINING (for receiving the final part of data)
        ///
        ///									Should be combined with the above option.
        ///									Option to include Le byte and Acknowledge key number to command frame for receiving
        ///									the UpdateAck data from Sam hardware.
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.ACK_EXCLUDE
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.ACK_INCLUDE</param>
        /// <param name="bNoOfKeyEntries">Number of symmetric key entries to update.
        ///									0x00: RFU;
        ///									0x01: 1 Key Entry;
        ///									0x02: 2 Key Entry;
        ///									0x03: 3 Key Entry;</param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="bPKI_KeyNo_Enc">Reference number of the RSA key entry to be used for decryption (00h to 01h).</param>
        /// <param name="bPKI_KeyNo_Sign">Reference number of the RSA key entry to be used for signature verification (00h to 02h).</param>
        /// <param name="bPKI_KeyNo_Ack">Reference number of the RSA key entry to be used for acknowledge signature generation (00h to 01h).</param>
        /// <param name="aKeyFrame">Buffer containing the RSA encrypted key entries and the signature.</param>
        /// <param name="aUpdateACK">Buffer containing the RSA encrypted Acknowledge signature.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_UpdateKeyEntries ( byte bOption, byte bNoOfKeyEntries, byte bHashingAlg,
            byte bPKI_KeyNo_Enc, byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Ack, byte[] aKeyFrame, out byte[] aUpdateACK )
        {
            Status_t oStatus;
            IntPtr ppUpdateACK = IntPtr.Zero;
            ushort pUpdateACKLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_UpdateKeyEntries ( ref m_DataParamsInt[0], bOption, bNoOfKeyEntries, bHashingAlg,
                bPKI_KeyNo_Enc, bPKI_KeyNo_Sign, bPKI_KeyNo_Ack, aKeyFrame, ( ushort ) ( ( aKeyFrame == null ) ? 0 : aKeyFrame.Length ), ref ppUpdateACK,
                ref pUpdateACKLen );

            byte[] aRxBuffer = MarshalCopy ( oStatus, ppUpdateACK, pUpdateACKLen );
            aUpdateACK = aRxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Prepare a cryptogram (according to Asymmetric Offline Change Cryptogram) for the PKI offline update of
        /// KST key entries on a target SAM. This command is only available for AV3 version(s).
        /// </summary>
        ///
        /// <param name="wOption">Option to differentiate between the command frame to be exchanged.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT (for receiving the first part of data);
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING (for receiving the final part of data);
        ///
        ///						  For enabling or disabling of key diversification. Should be combined with
        ///						  the above options \n
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.DIVERSIFICATION_OFF;
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.DIVERSIFICATION_ON; </param>
        /// <param name="bNoOfKeyEntries">Number of symmetric key entries to update.
        ///									0x00: RFU;
        ///									0x01: 1 Key Entry;
        ///									0x02: 2 Key Entry;
        ///									0x03: 3 Key Entry;</param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="bPKI_KeyNo_Enc">Reference number of the RSA key entry to be used for encryption (00h to 02h).</param>
        /// <param name="bPKI_KeyNo_Sign">Reference number of the RSA key entry to be used for signature verification (00h to 01h).</param>
        /// <param name="bPKI_KeyNo_Dec">Reference number of the RSA key entry to be used for decryption (00h to 01h).</param>
        /// <param name="bPKI_KeyNo_Verif">Reference number of the RSA key entry to be used for signature verification (00h to 02h).</param>
        /// <param name="wPerso_Ctr">Targeted offline change counter data.</param>
        /// <param name="aKeyEntries">Set of 01h - 03h reference number(s) of
        ///								Perso key entry: 00h - 7Fh (NVRam key) or E0h - E3h (Ram key);
        ///								key number     : 00h - 7Fh</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input).</param>
        /// <param name="aEncKeyFrame_Sign"></param>
        ///
        /// <returns>Returns Success status for successfull operation, command completed.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_EncipherKeyEntries ( ushort wOption, byte bNoOfKeyEntries, byte bHashingAlg, byte bPKI_KeyNo_Enc,
            byte bPKI_KeyNo_Sign, byte bPKI_KeyNo_Dec, byte bPKI_KeyNo_Verif, ushort wPerso_Ctr, byte[] aKeyEntries,
            byte[] aDivInput, out byte[] aEncKeyFrame_Sign )
        {
            Status_t oStatus;
            IntPtr ppEncKeyFrame_Sign = IntPtr.Zero;
            ushort pEncKeyFrame_Sign_Len = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_EncipherKeyEntries ( ref m_DataParamsInt[0], wOption, bNoOfKeyEntries, bHashingAlg,
                bPKI_KeyNo_Enc, bPKI_KeyNo_Sign, bPKI_KeyNo_Dec, bPKI_KeyNo_Verif, wPerso_Ctr, aKeyEntries,
                ( byte ) ( ( aKeyEntries != null ) ? aKeyEntries.Length : 0 ), aDivInput,
                ( byte ) ( ( aDivInput != null ) ? aDivInput.Length : 0 ), ref ppEncKeyFrame_Sign, ref pEncKeyFrame_Sign_Len );

            byte[] aRxBuffer = MarshalCopy ( oStatus, ppEncKeyFrame_Sign, pEncKeyFrame_Sign_Len );
            aEncKeyFrame_Sign = aRxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Generate Hash dataframe from Data.
        /// </summary>
        ///
        /// <param name="wOption">Option parameter: \n
        ///						  Local buffering is implemented by the flags
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST;
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT;
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST (the command is sent to the SAM);
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING (The LFI is set to AFh if the flag);
        ///
        ///						 On the first frame of the command chain the flag NxpRdLibNet.Hal.SamAV3.PKIOption.FIRST_FRAME
        ///						 has to be set to force the appending of the length of the overall message. </param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="dwMLen">Overall message length (4 byte).</param>
        /// <param name="aMessage">Message chunk to be hashed.</param>
        /// <param name="aHash">Buffer containing the hash after sending the last message chunk.</param>
        ///
        /// <returns>Returns Success status for successfull operation, command completed.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_GenerateHash ( ushort wOption, byte bHashingAlg, uint dwMLen, byte[] aMessage, out byte[] aHash )
        {
            Status_t oStatus;
            IntPtr ppHash = IntPtr.Zero;
            ushort pHashLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_GenerateHash ( ref m_DataParamsInt[0], wOption, bHashingAlg, dwMLen, aMessage,
                ( ushort ) ( ( aMessage == null ) ? 0 : aMessage.Length ), ref ppHash, ref pHashLen );

            byte[] aRxBuffer = MarshalCopy ( oStatus, ppHash, pHashLen );
            aHash = aRxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Generate Hash dataframe from Data.
        /// </summary>
        ///
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="bPKI_KeyNo_Sign">Number of the key entry to be used for signing (00h to 01h).</param>
        /// <param name="bHash">Hash message to be signed.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_GenerateSignature ( byte bHashingAlg, byte bPKI_KeyNo_Sign, byte[] bHash )
        {
            return phhalHw_SamAV3_Cmd_PKI_GenerateSignature ( ref m_DataParamsInt[0], bHashingAlg, bPKI_KeyNo_Sign, bHash,
                ( byte ) ( ( bHash != null ) ? bHash.Length : 0 ) );
        }

        /// <summary>
        /// Get a previously generated signature. This command is only available for AV2 and AV3 version(s).
        /// </summary>
        ///
        /// <param name="aSignature">The signature received from Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_SendSignature ( out byte[] aSignature )
        {
            Status_t wStatus;
            IntPtr ppSignature = IntPtr.Zero;
            ushort pSignatureLen = 0;

            wStatus = phhalHw_SamAV3_Cmd_PKI_SendSignature ( ref m_DataParamsInt[0], ref ppSignature, ref pSignatureLen );

            aSignature = MarshalCopy ( wStatus, ppSignature, pSignatureLen );

            return wStatus;
        }

        /// <summary>
        /// Verify a hash / signature pair with a given RSA key.
        /// </summary>
        ///
        /// <param name="bPKI_KeyNo_Verif">Key reference number of the PKI Key Entry to be used for the cryptogram signature verification (00h to 02h)</param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="aHashData">Hash data.</param>
        /// <param name="aSignature">RSA digital signature.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_VerifySignature ( byte bPKI_KeyNo_Verif, byte bHashingAlg, byte[] aHashData, byte[] aSignature )
        {
            return phhalHw_SamAV3_Cmd_PKI_VerifySignature ( ref m_DataParamsInt[0], bPKI_KeyNo_Verif, bHashingAlg, aHashData,
                ( byte ) ( ( aHashData == null ) ? 0 : aHashData.Length ), aSignature, ( ushort ) ( ( aSignature == null ) ? 0 : aSignature.Length ) );
        }

        /// <summary>
        /// Performs the offline encryption of plain RSA data. This is only supported by
        /// Sam AV3 version.
        /// </summary>
        ///
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_SHA_225
        ///									NxpRdLibNet.Hal.SamAV3.PKIOption.HASH_ALGORITHM_RFU</param>
        /// <param name="bPKI_KeyNo_Enc">Reference number of the RSA key entry to be used for encryption (00h to 02h).</param>
        /// <param name="aPlainData">RSA Plain Data to be encrypted.</param>
        /// <param name="aEncData">The RSA encrypted data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_EncipherData ( byte bHashingAlg, byte bPKI_KeyNo_Enc, byte[] aPlainData, out byte[] aEncData )
        {
            Status_t wStatus;
            IntPtr ppEncData = IntPtr.Zero;
            ushort pEncDataLen = 0;

            wStatus = phhalHw_SamAV3_Cmd_PKI_EncipherData ( ref m_DataParamsInt[0], bHashingAlg, bPKI_KeyNo_Enc,
                aPlainData, ( ushort ) ( ( aPlainData == null ) ? 0 : aPlainData.Length ), ref ppEncData, ref pEncDataLen );

            aEncData = MarshalCopy ( wStatus, ppEncData, pEncDataLen );

            return wStatus;
        }

        /// <summary>
        /// Performs the offline decryption of encrypted RSA data. This is only supported by
        /// Sam AV3 version.
        /// </summary>
        ///
        /// <param name="wOption">Option parameter:
        ///							Local buffering is implemented by the flags
        ///								NxpRdLibNet.ExchangeOptions.DEFAULT
        ///								NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///
        ///							Usage:
        ///								For chainned data
        ///									For First Frame     : Use flags NxpRdLibNet.Hal.SamAV3.PKI_DecipherData.FIRST_FRAME | NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///									For Next N Frame(s) : Use flags NxpRdLibNet.ExchangeOptions.TXCHAINING
        ///									For Last Frame      : Use flags NxpRdLibNet.ExchangeOptions.DEFAULT
        ///
        ///								For non chainned data i.e. only single frame use NxpRdLibNet.Hal.SamAV3.PKI_DecipherData.FIRST_FRAME | NxpRdLibNet.ExchangeOptions.DEFAULT</param>
        /// <param name="bHashingAlg">Hashing algorithm selection (for padding MGFs and digital signature).
        ///									NxpRdLibNet.Hal.SamAV3.PKI_HashAlgo.HASH_ALGORITHM_SHA_1
        ///									NxpRdLibNet.Hal.SamAV3.PKI_HashAlgo.HASH_ALGORITHM_SHA_224
        ///									NxpRdLibNet.Hal.SamAV3.PKI_HashAlgo.HASH_ALGORITHM_SHA_225</param>
        /// <param name="bPKI_KeyNo_Dec">Reference number of the RSA key entry to be used for decryption (00h to 01h).</param>
        /// <param name="aEncData">The RSA encrypted data to be decrypted.</param>
        /// <param name="aPlainData">RSA Plain Data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation, command completed.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_DecipherData ( ushort wOption, byte bHashingAlg, byte bPKI_KeyNo_Dec, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t wStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort pPlainDataLen = 0;

            wStatus = phhalHw_SamAV3_Cmd_PKI_DecipherData ( ref m_DataParamsInt[0], wOption, bHashingAlg, bPKI_KeyNo_Dec,
                aEncData, ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref ppPlainData, ref pPlainDataLen );

            aPlainData = MarshalCopy ( wStatus, ppPlainData, pPlainDataLen );

            return wStatus;
        }
        #endregion RSA

        #region ECC
        /// <summary>
        /// Imports the ECC public key to Key Storage. This is only supported by Sam AV3 version.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_ONLY</param>
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be update (00h to 07h).</param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///									0xFE		: No Restrictions
        ///									0xFF		: Entry Locked
        ///									0x00 - 0x7F : Restricted to specific permanent KST Key Entry</param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///									0xFE		: No Access Restrictions
        ///									0xFF		: Entry Disabled
        ///									0x00 - 0x7F : Access key entry number</param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="aECC_xy">Public key point coordinate. Ranges from 33 - 65 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ImportEccKey ( byte bOption, byte bECC_KeyNo, ushort wECC_Set, byte bECC_KeyNoCEK, byte bECC_KeyVCEK,
            byte bECC_RefNoKUC, byte bECC_KeyNoAEK, byte bECC_KeyVAEK, ushort wECC_Len, byte[] aECC_xy )
        {
            return phhalHw_SamAV3_Cmd_PKI_ImportEccKey ( ref m_DataParamsInt[0], bOption, bECC_KeyNo, wECC_Set, bECC_KeyNoCEK,
                bECC_KeyVCEK, bECC_RefNoKUC, bECC_KeyNoAEK, bECC_KeyVAEK, wECC_Len, aECC_xy,
                ( byte ) ( aECC_xy != null ? aECC_xy.Length : 0 ) );
        }

        /// <summary>
        /// Imports the full ECC Curve description to the ECC Curve Storage Table. This is
        /// only supported by Sam AV3 version.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_CURVE_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_CURVE_SETTINGS_ONLY</param>
        /// <param name="bECC_CurveNo">Curve reference number of the ECC curve entry to update (00h to 03h).</param>
        /// <param name="bECC_KeyNoCCK">Key reference number to change the curve entry.
        ///									0xFE		: No Restrictions
        ///									0xFF		: Entry Locked
        ///									0x00 - 0x7F : Restricted to specific permanent KST Key Entry.</param>
        /// <param name="bECC_KeyVCCK">Key version to change curve entry.</param>
        /// <param name="bECC_N">Size of the field in bytes. Ranges from 16 to 32 bytes.</param>
        /// <param name="bECC_M">Size of the order in bytes. Ranges from 16 to 32 bytes.</param>
        /// <param name="aECC_Prime">Prime, field definition: ECC_N bytes.</param>
        /// <param name="aECC_ParamA">Curve parameter (a): ECC_N bytes.</param>
        /// <param name="aECC_ParamB">Curve parameter (b): ECC_N bytes.</param>
        /// <param name="aECC_Px">x-coordinate of basepoint: ECC_N bytes.</param>
        /// <param name="aECC_Py">y-coordinate of basepoint: ECC_N bytes.</param>
        /// <param name="aECC_Order">Order of basepoint: ECC_M bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ImportEccCurve ( byte bOption, byte bECC_CurveNo, byte bECC_KeyNoCCK, byte bECC_KeyVCCK, byte bECC_N,
            byte bECC_M, byte[] aECC_Prime, byte[] aECC_ParamA, byte[] aECC_ParamB, byte[] aECC_Px, byte[] aECC_Py, byte[] aECC_Order )
        {
            return phhalHw_SamAV3_Cmd_PKI_ImportEccCurve ( ref m_DataParamsInt[0], bOption, bECC_CurveNo, bECC_KeyNoCCK, bECC_KeyVCCK,
                bECC_N, bECC_M, aECC_Prime, aECC_ParamA, aECC_ParamB, aECC_Px, aECC_Py, aECC_Order );
        }

        /// <summary>
        /// Exports the ECC public key from Key Storage. This is only supported by Sam AV3 version.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be update (00h to 07h).</param>
        /// <param name="wECC_Set">Configuration settings of the key entry.</param>
        /// <param name="bECC_KeyNoCEK">Key reference number of change entry key.
        ///									0xFE		: No Restrictions
        ///									0xFF		: Entry Locked
        ///									0x00 - 0x7F : Restricted to specific permanent KST Key Entry</param>
        /// <param name="bECC_KeyVCEK">Key version of change entry key.</param>
        /// <param name="bECC_RefNoKUC">Reference number of key usage counter.</param>
        /// <param name="bECC_KeyNoAEK">Key reference number of access entry key.
        ///									0xFE		: No Access Restrictions
        ///									0xFF		: Entry Disabled
        ///									0x00 - 0x7F : Access key entry number</param>
        /// <param name="bECC_KeyVAEK">Key version of access entry key</param>
        /// <param name="bECC_KeyVAEK">Version of the created acces key entry.</param>
        /// <param name="wECC_Len">ECC bit field size in bytes.</param>
        /// <param name="aECC_xy">Public key point coordinate. Ranges from 33 - 65 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ExportEccPublicKey ( byte bECC_KeyNo, out ushort wECC_Set, out byte bECC_KeyNoCEK, out byte bECC_KeyVCEK,
            out byte bECC_RefNoKUC, out byte bECC_KeyNoAEK, out byte bECC_KeyVAEK, out ushort wECC_Len, out byte[] aECC_xy )
        {
            Status_t oStatus;
            IntPtr ppECC_xy = IntPtr.Zero;
            byte pECC_xyLen = 0;

            wECC_Set = 0;
            bECC_KeyNoCEK = 0;
            bECC_KeyVCEK = 0;
            bECC_RefNoKUC = 0;
            bECC_KeyNoAEK = 0;
            bECC_KeyVAEK = 0;
            wECC_Len = 0;
            aECC_xy = null;

            oStatus = phhalHw_SamAV3_Cmd_PKI_ExportEccPublicKey ( ref m_DataParamsInt[0], bECC_KeyNo, ref wECC_Set, ref bECC_KeyNoCEK,
                ref bECC_KeyVCEK, ref bECC_RefNoKUC, ref bECC_KeyNoAEK, ref bECC_KeyVAEK, ref wECC_Len, ref ppECC_xy, ref pECC_xyLen );

            byte[] RxBuffer = MarshalCopy ( oStatus, ppECC_xy, pECC_xyLen );
            aECC_xy = RxBuffer;

            return oStatus;
        }

        /// <summary>
        /// The command verifies the correctness of an ECC signature (i.e.: NXPOriginalitySignature) obtained from the
        /// product to verify. The signature is computed according to Elliptic Curve DSA (ECDSA) algorithm. This is only
        /// supported by Sam AV3 version.
        /// </summary>
        ///
        /// <param name="bECC_KeyNo">The key reference number of the ECC key entry to be used for signature verification (00h to 07h).</param>
        /// <param name="bECC_CurveNo">The curve reference number of the ECC curve entry to be used for signature verification (00h to 03h).</param>
        /// <param name="bLen">Length in bytes of the message to verify.</param>
        /// <param name="aMessage">Signed input data.</param>
        /// <param name="aSignature">The ECC digital signature where N is 2  ECC_Len of ECC_KeyNo key entry.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_VerifyEccSignature ( byte bECC_KeyNo, byte bECC_CurveNo, byte bLen, byte[] aMessage, byte[] aSignature )
        {
            return phhalHw_SamAV3_Cmd_PKI_VerifyEccSignature ( ref m_DataParamsInt[0], bECC_KeyNo, bECC_CurveNo, bLen, aMessage,
                aSignature, ( ushort ) ( ( aSignature != null ) ? aSignature.Length : 0 ) );
        }
        #endregion ECC

        #region EMV
        /// <summary>
        /// Imports a CaPk key to Sam hardware. This command is used to import a Certificate Authority Public Key in the SAM for a given
        /// RID (Registered Application Provider Identifier)
        /// </summary>
        ///
        /// <param name="wOption">Option for P2 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_ONLY</param>
        /// <param name="aCaPkData">Certification Authority Public Key Related Data.
        ///							buffer should have the complete data based on the selected
        ///							option.</param>
        /// <param name="bCaPkDataLen">Certification Authority Public Key Related Data length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ImportCaPk ( byte bOption, byte[] aCaPkData, byte bCaPkDataLen )
        {
            return phhalHw_SamAV3_Cmd_PKI_ImportCaPk ( ref m_DataParamsInt[0], bOption, aCaPkData, bCaPkDataLen );
        }

        /// <summary>
        /// Imports a offlineCaPk key to Sam hardware. This command is used to import a Certificate Authority Public Key in the SAM
        /// </summary>
        ///
        /// <param name="bReceiveOfflineAck">Flag to exchange LE and receive the offline acknowledgement.
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.OFFLINE_ACK_RECEPTION_OFF
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.OFFLINE_ACK_RECEPTION_OFF</param>
        /// <param name="wOption">Option for P2 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_ONLY</param>
        /// <param name="aOfflineCryptogram">Offline change cryptogram. This data buffer should have the
        ///									 complete data based on the selected option.</param>
        /// <param name="bOfflineCryptogramLen">Length of bytes available in OfflineCryptogram buffer.</param>
        /// <param name="bOfflineMacLen">Length of bytes available in OfflineMac buffer.</param>
        /// <param name="aOfflineAck">Offline remove acknowledge as MACt(Kcm, 0x90 || 0x00 || INS || Change_Ctr || RID || PkID ||
        ///							  SAMUID).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ImportCaPkOffline ( byte bReceiveOfflineAck, byte bOption, byte[] aOfflineCryptogram,
            byte bOfflineCryptogramLen, out byte[] aOfflineAck )
        {
            Status_t oStatus = null;
            IntPtr ppOfflineAck = IntPtr.Zero;
            ushort pOfflineAckLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_ImportCaPkOffline ( ref m_DataParamsInt[0], bReceiveOfflineAck, bOption, aOfflineCryptogram,
                bOfflineCryptogramLen, ref ppOfflineAck, ref pOfflineAckLen );

            aOfflineAck = null;
            aOfflineAck = MarshalCopy ( oStatus, ppOfflineAck, pOfflineAckLen );

            return oStatus;
        }

        /// <summary>
        /// Removes a CaPk key to Sam hardware. This command is used to permanently remove a Certificate Authority Public Key from the
        /// EMV key storage table (EMV_KST). The command shall clear and reset the permanent storage containg the CAPk related data.
        /// </summary>
        ///
        /// <param name="aRID">Registered Application provider Identifier. This buffer should contain 5 bytes of RID.</param>
        /// <param name="bPkID">Certification Authority Public Key Index.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_RemoveCaPk ( byte[] aRID, byte bPkID )
        {
            return phhalHw_SamAV3_Cmd_PKI_RemoveCaPk ( ref m_DataParamsInt[0], aRID, bPkID );
        }

        /// <summary>
        /// Removes a CaPk key to Sam hardware. This command is used to permanently remove a Certificate Authority Public Key from the
        /// EMV key storage table (EMV_KST). The command shall clear and reset the permanent storage containg the CAPk related data.
        /// </summary>
        ///
        /// <param name="bReceiveOfflineAck">Flag to exchange LE and receive the offline acknowledgement.
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.OFFLINE_ACK_RECEPTION_OFF
        ///										NxpRdLibNet.Hal.SamAV3.PKIOption.OFFLINE_ACK_RECEPTION_OFF</param>
        /// <param name="pOfflineCryptogram">Offline change cryptogram. This data buffer should have the
        ///									 complete data based on the selected option.</param>
        /// <param name="bOfflineCryptogramLen">Length of bytes available in OfflineCryptogram buffer.</param>
        /// <param name="aOfflineAck">Offline remove acknowledge as MACt(Kcm, 0x90 || 0x00 || INS || Change_Ctr || RID || PkID ||
        ///							  SAMUID).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_RemoveCaPkOffline ( byte bReceiveOfflineAck, byte[] aOfflineCryptogram, byte bEncCaPkRefLen,
            byte bOfflineCryptogramLen, out byte[] aOfflineAck )
        {
            Status_t oStatus = null;
            IntPtr ppOfflineAck = IntPtr.Zero;
            ushort pOfflineAckLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_RemoveCaPkOffline ( ref m_DataParamsInt[0], bReceiveOfflineAck, aOfflineCryptogram,
                bEncCaPkRefLen, ref ppOfflineAck, ref pOfflineAckLen );

            aOfflineAck = null;
            aOfflineAck = MarshalCopy ( oStatus, ppOfflineAck, pOfflineAckLen );

            return oStatus;
        }

        /// <summary>
        /// Exports a CaPk key to Sam hardware. This command is used export a Certificate Authority Public Key from the EMV CA public keys
        /// key set entry (KST). The CA Pk entry is selected with the RID and PkID.
        /// </summary>
        ///
        /// <param name="wOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_VALUE
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.UPDATE_KEY_SETTINGS_ONLY
        ///
        ///						  Bufferring options to command exchange and CaPk keyentry reception
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///								To frame and exchange the Export CaPk command information.
        ///								Here the command header, RID, PkID and LE will be exchanged.
        ///
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING
        ///								To exchange command header only and for reception of parts of
        ///								keyentry.</param>
        /// <param name="aRID">Registered Application provider Identifier. This buffer should contain 5 bytes of RID.</param>
        /// <param name="bPkID">Certification Authority Public Key Index.</param>
        /// <param name="aKeyEntry">CAPk key entry information returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_ExportCaPk ( ushort wOption, byte[] aRID, byte bPkID, out byte[] aKeyEntry )
        {
            Status_t oStatus = null;
            IntPtr ppKeyEntry = IntPtr.Zero;
            ushort pKeyEntryLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_PKI_ExportCaPk ( ref m_DataParamsInt[0], wOption, aRID, bPkID, ref ppKeyEntry, ref pKeyEntryLen );

            aKeyEntry = null;
            aKeyEntry = MarshalCopy ( oStatus, ppKeyEntry, pKeyEntryLen );

            return oStatus;
        }

        /// <summary>
        /// Loads a Issuer Public Key. This command is used to load an Issuer Public Key. The SAM only accepts Issuer Public Key Certificate
        /// signed by the selected Certification Authority identified by RID and PkID. The required Certification Authority Public Key must be
        /// stored in the CA Key Storage otherwise Cmd.PKI_LoadIssuerPk will fail.
        /// </summary>
        ///
        /// <param name="bLFI">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.LFI_LAST
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.LFI_CHAINED</param>
        /// <param name="wOption">Bufferring options for exchanging the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///								To frame and exchange the complete payload information. If this
        ///								flag is passed, this is one complete and single frame. There will
        ///								not be more data to be sent by the HOST to Sam hardware, meaning
        ///								LFI parameter equals #PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///								To buffer initial set of payload information.
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///								To buffer the intermediate set of payload information.
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///								To buffer the final set of payload information.
        ///
        ///						  Note: For option NxpRdLibNet.ExchangeOptions.BUFFER_FIRST, NxpRdLibNet.ExchangeOptions.BUFFER_CONT and
        ///								NxpRdLibNet.ExchangeOptions.BUFFER_LAST LFI parameter depends on the maximum frame size of Sam hardware.
        ///								The payload information shall have RID, PkID, CertLen, PkCert, PkRemLen, PkRem and PkExp.</param>
        /// <param name="aData">The payload to be exchanged or bufferred.</param>
        /// <param name="bDataLen">Length of bytes available in Data buffer.</param>
        /// <param name="aIssureID">Issuer Identifier. This buffer will have 4 bytes of ID.</param>
        /// <param name="aExpDate">Certificate Expiration Date (MMYY). The buffer will have 2 bytes of data.</param>
        /// <param name="aSerialNo">Certificate Serial Number. The buffer will have 3 bytes of serial number.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_LoadIssuerPk ( byte bLFI, ushort wOption, byte[] aData, byte bDataLen, out byte[] aIssureID,
            out byte[] aExpDate, out byte[] aSerialNo )
        {
            Status_t oStatus = null;
            byte[] aIsserID_Tmp = new byte[4];
            byte[] aExpDate_Tmp = new byte[2];
            byte[] aSerialNo_Tmp = new byte[3];

            oStatus = phhalHw_SamAV3_Cmd_PKI_LoadIssuerPk ( ref m_DataParamsInt[0], bLFI, wOption, aData, bDataLen, aIsserID_Tmp,
                aExpDate_Tmp, aSerialNo_Tmp );

            aIssureID = null;
            aExpDate = null;
            aSerialNo = null;
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aIssureID = ( byte[] ) aIsserID_Tmp.Clone ();
                aExpDate = ( byte[] ) aExpDate_Tmp.Clone ();
                aSerialNo = ( byte[] ) aSerialNo_Tmp.Clone ();
            }

            return oStatus;
        }

        /// <summary>
        /// Loads a ICC Public Key. This command is used to load an ICC Public Key or an ICC PIN Encipherment Public Key. The SAM only
        /// accepts Public Key Certificate signed by the Issuer identified by the Issuer Public Key previously loaded with Cmd.PKI_LoadIssuerPk
        /// Cmd.PKI_LoadIccPk fails if the Issuer Public Key is not previously loaded, or has not signed the ICC (PIN Encipherment) Public Key
        /// Certificate.
        /// </summary>
        ///
        /// <param name="bLFI">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.LFI_LAST
        ///							NxpRdLibNet.Hal.SamAV3.PKIOption.LFI_CHAINED</param>
        /// <param name="wOption">Bufferring options for exchanging the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///								To frame and exchange the complete payload information. If this
        ///								flag is passed, this is one complete and single frame. There will
        ///								not be more data to be sent by the HOST to Sam hardware, meaning
        ///								LFI parameter equals #PHHAL_HW_SAMAV3_ISO7816_LAST_FRAME
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///								To buffer initial set of payload information.
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///								To buffer the intermediate set of payload information.
        ///
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///								To buffer the final set of payload information.
        ///
        ///						  Note: For option NxpRdLibNet.ExchangeOptions.BUFFER_FIRST, NxpRdLibNet.ExchangeOptions.BUFFER_CONT and
        ///								NxpRdLibNet.ExchangeOptions.BUFFER_LAST LFI parameter depends on the maximum frame size of Sam hardware.
        ///								The payload information shall have CertLen, PkCert, PkRemLen, PkRem, PkExp and SData.</param>
        /// <param name="aData">The payload to be exchanged or bufferred.</param>
        /// <param name="bDataLen">Length of bytes available in Data buffer.</param>
        /// <param name="aPAN">Application PAN. This buffer will have 10 bytes of information.</param>
        /// <param name="aExpDate">Certificate Expiration Date (MMYY). The buffer will have 2 bytes of data.</param>
        /// <param name="aSerialNo">Certificate Serial Number. The buffer will have 3 bytes of serial number.</param>
        /// <param name="bAlgoPk">ICC Public Key Algorithm Indicator. The buffer will have 1 bytes of algorithm information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PKI_LoadIccPk ( byte bLFI, ushort wOption, byte[] aData, byte bDataLen, out byte[] aPAN,
            out byte[] aExpDate, out byte[] aSerialNo, out byte bAlgoPk )
        {
            Status_t oStatus = null;
            byte[] aPAN_Tmp = new byte[10];
            byte[] aExpDate_Tmp = new byte[2];
            byte[] aSerialNo_Tmp = new byte[3];

            bAlgoPk = 0;
            oStatus = phhalHw_SamAV3_Cmd_PKI_LoadIccPk ( ref m_DataParamsInt[0], bLFI, wOption, aData, bDataLen, aPAN_Tmp,
                aExpDate_Tmp, aSerialNo_Tmp, ref bAlgoPk );

            aPAN = null;
            aExpDate = null;
            aSerialNo = null;
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aPAN = ( byte[] ) aPAN_Tmp.Clone ();
                aExpDate = ( byte[] ) aExpDate_Tmp.Clone ();
                aSerialNo = ( byte[] ) aSerialNo_Tmp.Clone ();
            }

            return oStatus;
        }
        #endregion EMV
        #endregion Public Key Infrastructure

        #region Virtual Card
        #region S Mode
        /// <summary>
        /// Performs Virtual card selection based on the response of ISOSelect command.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by bitwise oring.
        ///							Option for diversification of VCSelectENCKey.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.ENC_KEY_DIV_OFF
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.ENC_KEY_DIV_INPUT
        ///
        ///							Option for diversification of VCSelectMACKey.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_OFF
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_INPUT
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_VCUID
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_INPUT_VCUID
        ///
        ///							Option to disable the diversification for VcSelect MAC and ENC keys.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.DEFAULT</param>
        /// <param name="bEncKeyNo">ence key number to be used in hardware keystore for VCSelectENC key.</param>
        /// <param name="bEncKeyVer">Reference key version to be used in hardware keystore for VCSelectENC key.</param>
        /// <param name="bMacKeyNo">Reference key number to be used in hardware keystore for VCSelectMAC key.</param>
        /// <param name="bMacKeyVer">Reference key version to be used in hardware keystore for VCSelectMAC key.</param>
        /// <param name="aData">Cmd.ISOSelect response payload(32 Bytes) without TLV headers, including the VC related data.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response to Challenge as MACt(DivKey(KVCSelMAC)), RndChal || VCData). </param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_SelectVC ( byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer, byte[] aData, byte[] aDivInput,
            out byte[] aResponse )
        {
            ushort pRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_SelectVC ( ref m_DataParamsInt[0], bOption, bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer,
                aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ), aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                ref ppResponse, ref pRespLen );

            aResponse = null;
            if ( !pRespLen.Equals ( 0 ) )
            {
                aResponse = new byte[pRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, pRespLen );
            }

            return oStatus;
        }
        public Status_t Cmd_SAM_AV3_SelectVC ( byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer,
                    byte[] aData, byte[] aDivInput, out byte[] aRespData )
        {
            return Cmd_SAM_SelectVC ( bOption, bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer, aData, aDivInput, out aRespData );
        }

        /// <summary>
        /// Performs Proximity Check Part 1 command execution.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information.
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.DIVERSIFICATION_ON</param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key vesion to be used in hardware keystore.</param>
        /// <param name="aPPCData">Cmd.PreparePC response data.
        ///							For DESFIRE PICC: Option (1byte) || PubRespTime (2byte) [|| PPS1 (1byte)]
        ///							For PLUS PICC   : Option (1byte) || PubRespTime (2byte) [|| PPS1 (1byte)] [|| ActBitRate (N byte)]</param>
        /// <param name="aPCData">Response and challenge bytes exchanged during the proximity check protocol as
        ///						  (pRndR1 || pRndC1) || ... || (pRndR8 || pRndC8)</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aMac">The MAC to be exchanged to the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ProximityCheck_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPPCData, byte[] aPCData, byte[] aDivInput, out byte[] aMac )
        {
            ushort pMacLen = 0;
            IntPtr ppMac = IntPtr.Zero;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aPPCData,
                ( byte ) ( ( aPPCData == null ) ? 0 : aPPCData.Length ), aPCData, ( byte ) ( ( aPCData == null ) ? 0 : aPCData.Length ), aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref ppMac, ref pMacLen );

            aMac = null;
            if ( !pMacLen.Equals ( 0 ) )
            {
                aMac = new byte[pMacLen];
                Marshal.Copy ( ppMac, aMac, 0, pMacLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs Proximity Check Part 2 command execution.
        /// </summary>
        ///
        /// <param name="aData">The data to be exchaned to SAM hardware.
        ///							For PICC Error  : The PICC error code should be passed.
        ///							For PICC Success: The MAC received from PICC should be passed.</param>
        /// <param name="bPiccRetCode">The response code PICC echoed back by the Sam hardware.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ProximityCheck_Part2 ( byte[] aData, out byte bPiccRetCode )
        {
            bPiccRetCode = 0;
            return phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part2 ( ref m_DataParamsInt[0], aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref bPiccRetCode );
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Performs Virtual card selection in X-Mode.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by bitwise oring.
        ///							Option for diversification of VCSelectENCKey.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.ENC_KEY_DIV_OFF
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.ENC_KEY_DIV_INPUT
        ///
        ///							Option for diversification of VCSelectMACKey.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_OFF
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_INPUT
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_VCUID
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.MAC_KEY_DIV_INPUT_VCUID
        ///
        ///							Option to disable the diversification for VcSelect MAC and ENC keys.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.DEFAULT
        ///
        ///							VC selection can be done in 1-part of 2-parts.
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.VARIANT_PART1
        ///								NxpRdLibNet.Hal.SamAV3.VCA_Select.VARIANT_PART2</param>
        /// <param name="bEncKeyNo">ence key number to be used in hardware keystore for VCSelectENC key.</param>
        /// <param name="bEncKeyVer">Reference key version to be used in hardware keystore for VCSelectENC key.</param>
        /// <param name="bMacKeyNo">Reference key number to be used in hardware keystore for VCSelectMAC key.</param>
        /// <param name="bMacKeyVer">Reference key version to be used in hardware keystore for VCSelectMAC key.</param>
        /// <param name="aIID">The Installation Identifier (IID) to be selected.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response received from SAM based on the variants and AuthVCMandatory settings or PICC error
        ///								AuthVC Not Mandatory & Variant 1 : Conf0 (1byte), FCI (N bytes)
        ///								AuthVC Mandatory & Variant 1     : Conf1 (1byte), VC Data (16 bytes)
        ///								AuthVC Mandatory & Variant 2     : VC Data (16 bytes)</param>
        ///	<param name="wPiccRetCode">The status code returned from the PICC. This will be applicable for both the variants.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_VCA_Select ( byte bOption, byte bEncKeyNo, byte bEncKeyVer, byte bMacKeyNo, byte bMacKeyVer, byte[] aIID, byte[] aDivInput,
            out byte[] aResponse, out ushort wPiccRetCode )
        {
            ushort pRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            wPiccRetCode = 0;
            ppResponse = Marshal.AllocHGlobal ( 50 );
            Status_t oStatus = phhalHw_SamAV3_Cmd_VCA_Select ( ref m_DataParamsInt[0], bOption, bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer, aIID,
                ( byte ) ( ( aIID == null ) ? 0 : aIID.Length ), aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref ppResponse,
                ref pRespLen, ref wPiccRetCode );

            aResponse = null;
            if ( !pRespLen.Equals ( 0 ) )
            {
                aResponse = new byte[pRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, pRespLen );
            }

            Marshal.FreeHGlobal ( ppResponse );

            return oStatus;
        }

        /// <summary>
        /// Performs Proximity Check command execution in X-Mode.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information.
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.DIVERSIFICATION_ON
        ///
        ///						  Option to indicate Cmd.VerifyPC processing.
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.NATIVE_FORMAT
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.WRAPPED_FORMAT
        ///
        ///						  The command format to be used to communicated to PICC.
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.NORMAL_PROCESSING
        ///							NxpRdLibNet.Hal.SamAV3.VCA_ProximityCheck.RANDOM_PROCESSING</param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key vesion to be used in hardware keystore.</param>
        /// <param name="bNumOfRand">Maximum number of random bytes sent in one Cmd.ProxmityCheck</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aResponse">Response received from SAM.
        ///								For PICC Error  : The PICC error code will be returned.
        ///								For PICC Success: The PPCDataLen and PPCData will be returned.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_VCA_ProximityCheck ( byte bOption, byte bKeyNo, byte bKeyVer, byte bNumOfRand, byte[] aDivInput, out byte[] aResponse )
        {
            ushort pRespLen = 0;
            IntPtr ppResponse = IntPtr.Zero;

            Status_t oStatus = phhalHw_SamAV3_Cmd_VCA_ProximityCheck ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, bNumOfRand, aDivInput,
                (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref ppResponse, ref pRespLen );

            aResponse = null;
            if ( !pRespLen.Equals ( 0 ) )
            {
                aResponse = new byte[pRespLen];
                Marshal.Copy ( ppResponse, aResponse, 0, pRespLen );
            }

            return oStatus;
        }
        #endregion X Mode
        #endregion Virtual Card

        #region MIFARE DESFire
        #region S Mode
        /// <summary>
        /// Performs first part of encryption and decryption of data received from Card and to be sent to Card. Here the Encrypted RndB data will
        /// be sent to Sam hardware. Sam hardware will Decrypt the data and Encrpyt RndA with RndB'. This encrypted RndA and RndB' will be
        /// returned to the caller for further transmission to the Card.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by bitwise oring.
        ///							The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_AuthenticateOption"/>
        ///							Option for Authentication mode and Authentication type
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.D40_EV1
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.EV2_FIRST_AUTH
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.EV2_NON_FIRST_AUTH
        ///
        ///							Option for Key Derivation Information and Key Diversification Types.
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.ENABLED
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.NO_KDF
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_DOUBLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_SINGLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV2
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.RFU
        ///
        ///							Option for Key Selection
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.KEY_ENTRY_NUMBER
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.DESFIRE_NUMBER_KEY_NUMBER
        ///
        ///							Option for Key Diversification
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_OFF
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_ON
        ///						  </param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key version to be used in hardware keystore.</param>
        /// <param name="bAuthMode">The type of key to be used for EV2 authentication.
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthMode.EV2
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthMode.LRP</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aCardResponse">Encrypted RndB data returned by the card.</param>
        /// <param name="aSamResponse">Encrypted RndA and RndB' returned by the Sam hardware.</param>
        ///
        /// <returns>Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticatePICC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte bAuthMode, byte[] aDivInput, byte bDivInputLen,
            byte[] aCardResponse, out byte[] aSamResponse )
        {
            ushort pSamRespLen = 0;
            IntPtr ppSamResponse = IntPtr.Zero;

            aSamResponse = null;
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, bAuthMode, aDivInput,
                bDivInputLen, aCardResponse, (byte) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ), ref ppSamResponse, ref pSamRespLen );

            if ( !pSamRespLen.Equals ( 0 ) )
            {
                aSamResponse = new byte[pSamRespLen];
                Marshal.Copy ( ppSamResponse, aSamResponse, 0, pSamRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs second part of decryption of data received from Card. Here the Encrypted RndA' data will be sent to Sam hardware. Sam hardware
        /// will Decrypt the data and extract PCD and PD Capabilities for EV2 First Auth and null in case of rest of Authentication modes. This PCD
        /// and PD information will be returned to the caller. Also the status code of Card will be returned to the caller in case of error.
        /// </summary>
        ///
        /// <param name="bPiccErrorCode">Status code returned by the Card in case of failure else 90 for success.</param>
        /// <param name="aCardResponse">Encrypted RndA' returned by the Sam hardware.</param>
        /// <param name="aPCDCap2">Decrypted 6 bytes PCD capabilities returned by Sam hardware for EV2 First Auth, NULL for others.</param>
        /// <param name="aPDCap2">Decrypted 6 bytes PD capabilities returned by Sam hardware for EV2 First Auth, NULL for others.</param>
        /// <param name="bStatusCode">Status code from Desfire PICC if available else zero.</param>
        ///
        /// <returns>PH_ERR_SUCCESS Operation successful.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticatePICC_Part2 ( byte bPiccErrorCode, byte[] aCardResponse, out byte[] aPDcap2, out byte[] aPCDcap2, out byte bStatusCode )
        {
            bStatusCode = 0;
            aPCDcap2 = new byte[6];
            aPDcap2 = new byte[6];

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2 ( ref m_DataParamsInt[0], bPiccErrorCode, aCardResponse, (byte) ( ( aCardResponse != null ) ?
                aCardResponse.Length : 0 ), aPDcap2, aPCDcap2, ref bStatusCode );

            if ( aCardResponse != null )
            {
                if ( !aCardResponse.Length.Equals ( 32 ) )
                {
                    aPCDcap2 = null;
                    aPDcap2 = null;
                }
            }

            return oStatus;
        }

        /// <summary>
        /// Performs first part of encryption and decryption of data received from Card and to be sent to Card. Here the Encrypted RndB data will be sent to
        /// Sam hardware. Sam hardware will Decrypt the data and Encrpyt RndA with RndB'. This encrypted RndA and RndB' will be returned to the caller for
        /// further transmission to the Card.
        /// </summary>
        ///
        /// <param name="bOption">Option to update the P1 information. The options can be combined by bitwise oring.
        ///							The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_AuthenticateOption"/>
        ///							Option for Key Derivation Information and Key Diversification Types.
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.ENABLED
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.NO_KDF
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_DOUBLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_SINGLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV2
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.RFU
        ///
        ///							Option for Key Selection
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.KEY_ENTRY_NUMBER
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.DESFIRE_NUMBER_KEY_NUMBER
        ///
        ///							Option for Key Diversification
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_OFF
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_ON
        ///						  </param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key version to be used in hardware keystore.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aCardResponse">Encrypted RndB data returned by the card.</param>
        /// <param name="aSamResponse">Encrypted RndA and RndB' returned by the Sam hardware.</param>
        ///
        /// <returns>Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen, byte[] aCardResponse,
            out byte[] aSamResponse )
        {
            ushort pSamRespLen = 0;
            IntPtr ppSamResponse = IntPtr.Zero;

            aSamResponse = null;
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_IsoAuthenticatePICC_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aDivInput, bDivInputLen,
                aCardResponse, (byte) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ), ref ppSamResponse, ref pSamRespLen );

            if ( !pSamRespLen.Equals ( 0 ) )
            {
                aSamResponse = new byte[pSamRespLen];
                Marshal.Copy ( ppSamResponse, aSamResponse, 0, pSamRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs second part of decryption of data received from Card. Here the Encrypted RndA' data will be sent to Sam hardware. Sam hardware will
        /// Decrypt the data and compare with the one which it generated. If it succeeds SUCCESS will be returned by SAM else error code will be returned.
        /// </summary>
        ///
        /// <param name="aCardResponse">Encrypted RndA' returned by the Sam hardware.</param>
        ///
        /// <returns>PH_ERR_SUCCESS Operation successful.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part2 ( byte[] aCardResponse )
        {
            return phhalHw_SamAV3_Cmd_SAM_IsoAuthenticatePICC_Part2 ( ref m_DataParamsInt[0], aCardResponse,
                ( byte ) ( ( aCardResponse != null ) ? aCardResponse.Length : 0 ) );
        }

        /// <summary>
        /// Performs key change for the specified current key to a new key. The crypto operation of the key to be
        /// changed will be calculated by SAM hardware. This crypto data will then sent to card to perform Change
        /// Key operations.
        /// </summary>
        ///
        /// <param name="bCryptoMethod">Multiple option from <seealso cref="NxpRdLibNet.Hal.SamAV3.DesfireSModeOption"/>.
        ///								These options shoulde be formed by Oring the fields of the enum. Options can be
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.DIVERSIFICATION_METHOD_AV1
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.DIVERSIFICATION_METHOD_AV2
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_SAMAV2_RFU
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_CURRENT_KEY_DOUBLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_CURRENT_KEY_SINGLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_NEW_KEY_DOUBLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_NEW_KEY_SINGLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_CURRENT_KEY_OFF
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_CURRENT_KEY_ON
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_NEW_KEY_OFF
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KDF_NEW_KEY_ON
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.CRYPTO_MODE_TARGET_AUTH_KEY_DIFFERENT
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.CRYPTO_MODE_TARGET_AUTH_KEY_SAME</param>
        /// <param name="bConfig">
        ///						  To perform ChangeKey command or ChangeKeyEV2 command. Use from one of the below
        ///						   options.
        ///							NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.CMD_TYPE_CHANGE_KEY
        ///						    NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.CMD_TYPE_CHANGE_KEY_EV2
        ///							NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.MASTER_KEY_UPDATE_EXCLUDE_KEYTYPE
        ///							NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.MASTER_KEY_UPDATE_INCLUDE_KEYTYPE</param>
        /// <param name="bKeySetNo">Key set number to which the key to be changed belongs to.</param>
        /// <param name="bDFKeyNo">Block number of the key available in the card. This will be used while
        ///						   exchanging the command with card. The lower nibble will be used for P2
        ///						   information byte if command type is ChangeKey and the complete byte
        ///						   will be used if command type is ChangeKeyEV2.</param>
        /// <param name="bCurrKeyNo">Present key address in hardware key store.</param>
        /// <param name="bCurrKeyVer">Present key version to be used.</param>
        /// <param name="bNewKeyNo">New key address in hardware key store to which the present key will be changed.</param>
        /// <param name="bNewKeyVer">New key version to be used.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aSamResponse">Cryptogram holding key data.</param>
        ///
        /// <returns>PH_ERR_SUCCESS Operation successful.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyPICC ( byte bCryptoMethod, byte bConfig, byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo, byte bCurrKeyVer,
            byte bNewKeyNo, byte bNewKeyVer, byte[] aDivInput, byte bDivInputLen, out byte[] aSamResponse )
        {
            ushort pSamRespLen = 0;
            IntPtr ppSamResponse = IntPtr.Zero;

            aSamResponse = null;
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyPICC ( ref m_DataParamsInt[0], bCryptoMethod, bConfig, bKeySetNo, bDFKeyNo, bCurrKeyNo, bCurrKeyVer,
                bNewKeyNo, bNewKeyVer, aDivInput, bDivInputLen, ref ppSamResponse, ref pSamRespLen );

            if ( !pSamRespLen.Equals ( 0 ) )
            {
                aSamResponse = new byte[pSamRespLen];
                Marshal.Copy ( ppSamResponse, aSamResponse, 0, pSamRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs creation of cryptogram for Transaction MAC file creation.
        /// </summary>
        ///
        /// <param name="bOption">Option for updating the P1 information of Sam frame.
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KEY_DIVERSIFICATION_OFF
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.KEY_DIVERSIFICATION_ON
        ///						  </param>
        /// <param name="bKeyNo">Key address in hardware key store</param>
        /// <param name="bKeyVer">Key version to be used.</param>
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bFileOption">Options for the targeted file. Should be one of the below options.
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.FILE_OPTION_PLAIN
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.FILE_OPTION_MAC
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.FILE_OPTION_FULL
        ///						  </param>
        /// <param name="pAccessRights">Set of access conditions. Should be one of the below values
        ///								Read	: Bits 15 - 12;
        ///											0x00 - 0x0D: Key no of a Application key;
        ///											0x0E		 : Free Access;
        ///											0x0F		 : RFU;
        ///								Write	: Bits 11 - 08
        ///											0x0F	     : RFU (Should be the default value always);
        ///								ReadWrite: Bits 07 - 04
        ///											0x00 - 0x0D: Key index within the active key set (AKS);
        ///											0x0E	     : RFU;
        ///											0x0F	     : Commit Reader ID command disabled.
        ///								Change	: Bits 03 - 00
        ///											0x00 - 0x0D: Key no of a Application key;
        ///											0x0E		 : Free Access;
        ///											0x0F		 : RFU;</param>
        /// <param name="bTMKeyOptions">Option for Application Transaction MAC file.
        ///									NxpRdLibNet.Hal.SamAV3.DesfireSModeOption.TM_KEY_OPTION_AES_KEY
        ///									Other values are RFU
        ///						  </param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aSamResponse">Cryptogram holding Transaction MAC Key data</param>
        ///
        /// <returns>PH_ERR_SUCCESS Operation successful.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_CreateTMFilePICC ( byte bOption, byte bKeyNo, byte bKeyVer, byte bFileNo, byte bFileOption, byte[] pAccessRights,
            byte bTMKeyOptions, byte[] aDivInput, byte bDivInputLen, out byte[] aSamResponse )
        {
            ushort pSamRespLen = 0;
            IntPtr ppSamResponse = IntPtr.Zero;

            aSamResponse = null;
            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_CreateTMFilePICC ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, bFileNo, bFileOption,
                pAccessRights, bTMKeyOptions, aDivInput, bDivInputLen, ref ppSamResponse, ref pSamRespLen );

            if ( !pSamRespLen.Equals ( 0 ) )
            {
                aSamResponse = new byte[pSamRespLen];
                Marshal.Copy ( ppSamResponse, aSamResponse, 0, pSamRespLen );
            }

            return oStatus;
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Perform an authentication procedure between SAM AV3 and DESFire.
        /// </summary>
        ///
        /// <param name="bAuthMode">Option to update the P1 information. The options can be combined by bitwise oring.
        ///							The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_AuthenticateOption"/>
        ///							Option for Authentication mode and Authentication type
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.D40_EV1
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.EV2_FIRST_AUTH
        ///								NxpRdLibNet.Hal.SamAV3.DF_AuthType.EV2_NON_FIRST_AUTH
        ///
        ///							Option for Key Derivation Information and Key Diversification Types.
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.ENABLED
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDerInfo.NO_KDF
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_DOUBLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV1_SINGLE_ENCRYPTION
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.AV2
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeyDivMode.RFU
        ///
        ///							Option for Key Selection
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.KEY_ENTRY_NUMBER
        ///								NxpRdLibNet.Hal.SamAV3.DF_KeySel.DESFIRE_NUMBER_KEY_NUMBER
        ///
        ///							Option for Key Diversification
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_OFF
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_ON
        ///							</param>
        /// <param name="bISOMode">ISO mode to be used.
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.NATIVE
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.ISO7816_4
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.ISO_AUTHENTICATION</param>
        /// <param name="bDFKeyNo">DESFire Key Number to be used for authentication.</param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key version to be used in hardware keystore.</param>
        /// <param name="aPCDcap2In">Input PCD capabilites to be exchanged. Should be as mentioned below.
        ///								For Non First authentication the buffer should be null.
        ///								For First Authentication:
        ///									NULL for no input PCD capabilities.
        ///									Buffer with size 1 - 6 with PCD input capabilities information.
        ///									Buffer with size 1 having the value as 0xFF for Default PcdCaps information.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aPcdCapOut">Buffer containing the output PCD capabilities. This will contain 6 bytes of valid PCD information.</param>
        /// <param name="aPDcap2">Buffer containing the output PD capabilities. This will contain 6 bytes of valid PD information.</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. This will of 1 byte in length for Native error code and two byte for rest.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_DESFire_AuthenticatePICC ( byte bAuthMode, byte bISOMode, byte bDFKeyNo, byte bKeyNo, byte bKeyVer, byte[] aPCDcap2In, byte[] aDivInput,
            out byte[] aPDcap2, out byte[] aPCDcap2, out byte[] aPiccErrorCode )
        {
            aPCDcap2 = new byte[6];
            aPDcap2 = new byte[6];
            aPiccErrorCode = new byte[2];
            byte bPCDcap2InLen = 0;
            byte[] aPCDcap2In_Tmp = null;

            /* Frame the Input PCD capabilities length and the buffer. */
            /* For FirstAuth with no input capabilities or NonFirstAuth. */
            if ( aPCDcap2In == null )
            {
                bPCDcap2InLen = 0;
                aPCDcap2In_Tmp = null;
            }
            else
            {
                /* For FirstAuth and Default PCD capabilities. */
                if ( aPCDcap2In.Length.Equals ( 1 ) && aPCDcap2In[0].Equals ( 0xFF ) )
                {
                    bPCDcap2InLen = 0xFF;
                    aPCDcap2In_Tmp = null;
                }
                /* Reset of the PCD capabilities information. */
                else
                {
                    bPCDcap2InLen = ( byte ) aPCDcap2In.Length;
                    aPCDcap2In_Tmp = ( byte[] ) aPCDcap2In.Clone ();
                }
            }

            Status_t oStatus = phhalHw_SamAV3_Cmd_DESFire_AuthenticatePICC ( ref m_DataParamsInt[0], bAuthMode, bISOMode, bDFKeyNo, bKeyNo, bKeyVer,
                bPCDcap2InLen, aPCDcap2In_Tmp, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPDcap2, aPCDcap2, aPiccErrorCode );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bISOMode.Equals ( ( byte ) DF_ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            if ( !( bAuthMode & ( byte ) DF_AuthType.EV2_FIRST_AUTH ).Equals ( ( byte ) DF_AuthType.EV2_FIRST_AUTH ) )
            {
                aPCDcap2 = null;
                aPDcap2 = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Change a key of a DESFire PICC.
        /// </summary>
        ///
        /// <param name="bKeyCompMeth">Options for P1 information byte
        ///							   The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_ChangeKeyCompilationOption"/>
        ///							   Key diversification method
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.DIV_METHOD_AV1
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.DIV_METHOD_AV2
        ///
        ///							   Sam AV1 and Sam AV2 Key diversification method
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.CURRENT_KEY_DOUBLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.CURRENT_KEY_SINGLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.NEW_KEY_DOUBLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.NEW_KEY_SINGLE_ENCRYPTION
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.CURRENT_KEY_OFF
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.CURRENT_KEY_ON
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.NEW_KEY_OFF
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.NEW_KEY_ON
        ///
        ///							   Cryptogram computation mode
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.DIFFERENT_KEY
        ///									NxpRdLibNet.Hal.SamAV3.DF_KeyCompMthod.SAME_KEY</param>
        /// <param name="bConfig">Options for P2 information byte.
        ///						  The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_ChangeKeyConfigurationOption"/>
        ///						  ISO mode to be used.
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.NATIVE
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.ISO7816_4
        ///
        ///						  Command Type.
        ///								NxpRdLibNet.Hal.SamAV3.DF_Configuration.CHANGE_KEY
        ///								NxpRdLibNet.Hal.SamAV3.DF_Configuration.CHANGE_KEY_EV2
        ///
        ///						  PICC master key update.
        ///								NxpRdLibNet.Hal.SamAV3.DF_Configuration.EXCLUDE_KEYTYPE
        ///								NxpRdLibNet.Hal.SamAV3.DF_Configuration.INCLUDE_KEYTYPE</param>
        /// <param name="bKeySetNo">Key set number to which the key to be changed belongs to.</param>
        /// <param name="bDFKeyNo">Block number of the key available in the card. This will be used while exchanging the command with card. The lower nibble will be
        ///						   used for P2 information byte if command type is ChangeKey and the complete byte will be used if command type is ChangeKeyEV2.</param>
        /// <param name="bCurrKeyNo">Reference key number to be used in hardware keystore for CurrentKey.</param>
        /// <param name="bCurrKeyVer">Reference key version to be used in hardware keystore for CurrentKey.</param>
        /// <param name="bNewKeyNo">Reference key number to be used in hardware keystore for NewKey.</param>
        /// <param name="bNewKeyVer">Reference key version to be used in hardware keystore for NewKey.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. This will of 1 byte in length for Native error code and two byte for rest.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_DESFire_ChangeKeyPICC ( byte bKeyCompMeth, byte bConfig, byte bKeySetNo, byte bDFKeyNo, byte bCurrKeyNo,
            byte bCurrKeyVer, byte bNewKeyNo, byte bNewKeyVer, byte[] aDivInput, out byte[] aPiccErrorCode )
        {
            aPiccErrorCode = new byte[2];
            Status_t oStatus = phhalHw_SamAV3_Cmd_DESFire_ChangeKeyPICC ( ref m_DataParamsInt[0], bKeyCompMeth, bConfig, bKeySetNo, bDFKeyNo, bCurrKeyNo, bCurrKeyVer,
                bNewKeyNo, bNewKeyVer, aDivInput, (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPiccErrorCode );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bConfig.Equals ( ( byte ) DF_ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            return oStatus;
        }

        /// <summary>
        /// Write data to a DESFire encrypted or MACed.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///						  To buffer the data locally use the below flags.
        ///								NxpRdLibNet.ExchangeOptions.DEFAULT
        ///								NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///								NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///								NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///
        ///						  To exchange data in chaining mode use NxpRdLibNet.ExchangeOptions.TXCHAINING buffering flag.</param>
        /// <param name="bCryptoConfig">Option to set the P2 information byte.
        ///								The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_ReadWrite_CryptoConfiguration"/>
        ///								Extended Offset
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.LEGACY_MODE
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.EXTENDED_OFFSET
        ///
        ///								Chaining configuration
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.DESFIRE_CHAINING
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.ISO_CHAINING
        ///
        ///								Communication Mode.
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_PLAIN
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_MAC
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_FULL
        ///
        ///								If Legacy Mode, bit 3-0 should be updated with proper offset.
        ///								If Legacy Mode, bit 3-0 is RFU.</param>
        /// <param name="aData">The data to be written to the DESFire PICC.
        ///							If LEGACY_MODE, the buffer should contain only the PICC related data.
        ///							If EXTENDED_OFFSET, the buffer should contain offset followed by the PICC related data.\n</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. This will of 1 byte in length for Native error code and two byte for rest.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_DESFire_WriteX ( ushort wOption, byte bCryptoConfig, byte[] aData, out byte[] aPiccErrorCode )
        {
            byte bErrLen = 0;
            aPiccErrorCode = new byte[2];

            Status_t oStatus = phhalHw_SamAV3_Cmd_DESFire_WriteX ( ref m_DataParamsInt[0], ( ushort ) wOption, ( byte ) bCryptoConfig, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), aPiccErrorCode, ref bErrLen );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                Array.Resize ( ref aPiccErrorCode, bErrLen );
            }
            else
                aPiccErrorCode = null;

            return oStatus;
        }

        /// <summary>
        /// Read data from a DESFire encrypted or MACed.
        /// </summary>
        ///
        /// <param name="wOption">Buffering options.
        ///						  To buffer the initial command information. Use this flag to buffer the Length
        ///						  information also in case of FULL (P2 = 0x30) communication mode and more data
        ///						  is expected.
        ///								NxpRdLibNet.ExchangeOptions.DEFAULT
        ///
        ///						  To receive data in chaining mode use NxpRdLibNet.ExchangeOptions.RXCHAINING buffering flag.</param>
        /// <param name="bCryptoConfig">Option to set the P2 information byte.
        ///								The byte can also be framed by using a helper class <seealso cref="NxpRdLibNet.Hal.SamAV3.DESFire_ReadWrite_CryptoConfiguration"/>
        ///								Communication Mode.
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_PLAIN
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_MAC
        ///									NxpRdLibNet.Hal.SamAV3.DF_CryptoConfig.COMM_MODE_FULL</param>
        /// <param name="aAppData">The following information should be passed.
        ///								3 bytes length information in case if Communication mode is FULL and more data is expected.
        ///								Complete PICC command header and data to be sent to PICC for initial exchange.
        ///								DESFire Chaining command code in case more data is expected.</param>
        /// <param name="aResponse">The data received from Sam hardware.</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. This will of 1 byte in length for Native error code and two byte for rest.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_DESFire_ReadX ( ushort wOption, byte bCryptoConfig, byte[] aAppData, out byte[] aResponse, out byte[] aPiccErrorCode )
        {
            byte bErrLen = 0;
            aPiccErrorCode = new byte[2];
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRespLen = 0;

            aResponse = null;
            Status_t oStatus = phhalHw_SamAV3_Cmd_DESFire_ReadX ( ref m_DataParamsInt[0], (ushort) wOption, (byte) bCryptoConfig, aAppData,
                (byte) ( ( aAppData == null ) ? 0 : aAppData.Length ), ref ppResponse, ref wRespLen, aPiccErrorCode, ref bErrLen );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                Array.Resize ( ref aPiccErrorCode, bErrLen );
            }
            else
            {
                if ( ppResponse != IntPtr.Zero )
                {
                    aResponse = new byte[wRespLen];
                    aResponse = MarshalCopy ( oStatus, ppResponse, wRespLen );
                }

                aPiccErrorCode = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs Create Transaction MAC file.
        /// </summary>
        ///
        /// <param name="bOption">Options for P1 information byte.
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_OFF
        ///								NxpRdLibNet.Hal.SamAV3.DF_Diversification.DIVERSIFICATION_ON</param>
        /// <param name="bISOMode">Options for P2 information byte.
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.NATIVE
        ///								NxpRdLibNet.Hal.SamAV3.DF_ISOMode.ISO7816_4</param>
        /// <param name="bKeyNo">Reference key number to be used in hardware keystore.</param>
        /// <param name="bKeyVer">Reference key version to be used in hardware keystore.</param>
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bFileOption">Options for the targeted file.
        ///								NxpRdLibNet.Hal.SamAV3.DF_FileOption.PLAIN
        ///								NxpRdLibNet.Hal.SamAV3.DF_FileOption.MAC
        ///								NxpRdLibNet.Hal.SamAV3.DF_FileOption.FULL</param>
        /// <param name="aAccessRights">Access conditions to be applied for the file. Refer DESFire EV2 datasheet for access
        ///								rights information. This should be two bytes long.</param>
        /// <param name="bTMKeyOptions">Option for the TransactionMAC Key. NxpRdLibNet.Hal.SamAV3.DF_TMKeyOption.AES is currently
        ///								supported option as per DESFireEV2 datasheet.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aPiccErrorCode">Error code returned by PICC. This will of 1 byte in length for Native error code and two byte for rest.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_DESFire_CreateTMFilePICC ( byte bOption, byte bISOMode, byte bKeyNo, byte bKeyVer, byte bFileNo, byte bFileOption, byte[] aAccessRights,
            byte bTMKeyOptions, byte[] aDivInput, out byte[] aPiccErrorCode )
        {
            aPiccErrorCode = new byte[2];

            Status_t oStatus = phhalHw_SamAV3_Cmd_DESFire_CreateTMFilePICC ( ref m_DataParamsInt[0], bOption, bISOMode, bKeyNo, bKeyVer, bFileNo, bFileOption, aAccessRights,
                    bTMKeyOptions, aDivInput, (byte) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPiccErrorCode );

            if ( oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.DESFIRE_GEN ) ) )
            {
                if ( bISOMode.Equals ( ( byte ) DF_ISOMode.NATIVE ) )
                    Array.Resize ( ref aPiccErrorCode, 1 );
            }
            else
                aPiccErrorCode = null;

            return oStatus;
        }
        #endregion X Mode
        #endregion MIFARE DESFire

        #region MIFARE Plus
        #region S Mode
        /// <summary>
        /// Perform a MFP Authenticate command part1. This command will generate a 16 byte random number with the one received from
        /// card and return an 32 byte encrypted data which will be sent to card.
        /// </summary>
        ///
        /// <param name="bOption">Option for Authenticate command.
        ///							NxpRdLibNet.Hal.SamAV3.SamAV3.MFP_Authenticate.DIVERSIFY_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.DIVERSIFY_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.FIRST_AUTH
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.NONFIRST_AUTH
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.SL1_NO_KDF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.SL3_KDF</param>
        /// <param name="bKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="aPDChal">Buffer containing the challange generated by the card.</param>
        /// <param name="bPDChalLen">Length of bytes available in PDChal buffer</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be sent to card.</param>
        ///
        /// <returns>Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateMFP_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPDChal, byte bPDChalLen, byte[] aDivInput,
            byte bDivInputLen, out byte[] aPCDChalResp )
        {
            Status_t oStatus;
            ushort wPCDChalRespLen = 0;
            IntPtr ppPCDChalResp = IntPtr.Zero;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aPDChal,
                bPDChalLen, aDivInput, bDivInputLen, ref ppPCDChalResp, ref wPCDChalRespLen );

            aPCDChalResp = null;
            aPCDChalResp = MarshalCopy ( oStatus, ppPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        ///  Perform a MFP Authenticate command part2. This command will decrypt the response received from card and will return
        ///  the PC capabilities and PCD capabilites.
        /// </summary>
        /// <param name="bPiccErrCode">Error Code sent by the MFP card. For success it should be 0x90.</param>
        /// <param name="aPDResp">Buffer containing the input received from card.</param>
        /// <param name="bPDRespLen">Length of bytes available in PDResp buffer</param>
        /// <param name="aPDCap2">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
        /// <param name="aPCDCap2">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateMFP_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen, out byte[] aPDCap2, out byte[] aPCDCap2,
            out byte bPiccReturnCode )
        {
            Status_t oStatus;
            IntPtr ppPDCap2Int = IntPtr.Zero;
            IntPtr ppPCDCap2Int = IntPtr.Zero;

            bPiccReturnCode = 0;
            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMFP_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aPDResp,
                bPDRespLen, ref ppPDCap2Int, ref ppPCDCap2Int, ref bPiccReturnCode );

            aPDCap2 = null;
            aPCDCap2 = null;
            aPDCap2 = MarshalCopy ( oStatus, ppPDCap2Int, 6 );
            aPCDCap2 = MarshalCopy ( oStatus, ppPCDCap2Int, 6 );

            return oStatus;
        }

        /// <summary>
        /// Perform a MFP Sector Switch Authenticate command part1. This command will generate a 16 byte random number with the one received from
        /// card and return an 32 byte encrypted data which will be sent to card.
        /// </summary>
        ///
        /// <param name="bOption">Option to set the P1 information bytes.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON</param>
        /// <param name="aPDChal">Buffer containing the challange generated by the card.</param>
        /// <param name="bPDChalLen">Length of bytes available in PDChal buffer.</param>
        /// <param name="wSSKeyBNr">PICC Sector Switch key block number to be used for authentication.</param>
        /// <param name="bSSKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bSSKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="bMSKeyNo">Key number to be used from Sam keystore if master sector key in enabled.</param>
        /// <param name="bMSKeyVer">Key version to be used from Sam keystore if master sector key in enabled.</param>
        /// <param name="bSectorCount">Number of sectors to be moved to higher security level</param>
        /// <param name="aKeyBlocks"></param>
        /// <param name="bKeyBlocksLen"></param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be sent to card.</param>
        ///
        /// <returns>Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthSectorSwitchMFP_Part1 ( byte bOption, byte[] aPDChal, byte bPDChalLen, ushort wSSKeyBNr, byte bSSKeyNo,
            byte bSSKeyVer, byte bMSKeyNo, byte bMSKeyVer, byte bSectorCount, byte[] aKeyBlocks, byte bKeyBlocksLen, byte[] aDivInput,
            byte bDivInputLen, out byte[] aPCDChalResp )
        {
            Status_t oStatus;
            ushort wPCDChalRespLen = 0;
            IntPtr ppPCDChalResp = IntPtr.Zero;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthSectorSwitchMFP_Part1 ( ref m_DataParamsInt[0], bOption, aPDChal, bPDChalLen, wSSKeyBNr, bSSKeyNo, bSSKeyVer,
                bMSKeyNo, bMSKeyVer, bSectorCount, aKeyBlocks, bKeyBlocksLen, aDivInput, bDivInputLen, ref ppPCDChalResp, ref wPCDChalRespLen );

            aPCDChalResp = null;
            aPCDChalResp = MarshalCopy ( oStatus, ppPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a MFP Sector Switch Authenticate Authenticate command part2. This command will decrypt the response received from
        /// card and will return success status if the challanges matches.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by the MFP card. For success it should be 0x90.</param>
        /// <param name="aPDResp">Buffer containing the input received from card.</param>
        /// <param name="bPDRespLen">Length of bytes available in PDResp buffer.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthSectorSwitchMFP_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_SAM_AuthSectorSwitchMFP_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aPDResp, bPDRespLen,
                ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a Post delivery configuration Authenticate command part1. This command will generate a 16 byte random number
        /// with the one received from card and return an 32 byte encrypted data which will be sent to card.
        /// </summary>
        ///
        /// <param name="bOption">Option to set the P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFICATION_RFU
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DERIVE_UPGRADE_KEY
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFY_YEAR_UPGRADE_KEY</param>
        /// <param name="bKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="aPDChal">Buffer containing the input PcdCaps.</param>
        /// <param name="bPDchalLen">Length of bytes available in PDChal buffer.</param>
        /// <param name="aUpgradeInfo">Upgrade information of the target product state.</param>
        /// <param name="bLen">Length of bytes available in UpgradeInfo buffer.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aPCDChalResp">Buffer containing the PCD challenge response to be sent to card.</param>
        ///
        /// <returns>Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticatePDC_Part1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aPDChal, byte bPDchalLen, byte[] aUpgradeInfo,
            byte bLen, byte[] aDivInput, byte bDivInputLen, out byte[] aPCDChalResp )
        {
            Status_t oStatus;
            ushort wPCDChalRespLen = 0;
            IntPtr ppPCDChalResp = IntPtr.Zero;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticatePDC_Part1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aPDChal, bPDchalLen,
                aUpgradeInfo, bLen, aDivInput, bDivInputLen, ref ppPCDChalResp, ref wPCDChalRespLen );

            aPCDChalResp = null;
            aPCDChalResp = MarshalCopy ( oStatus, ppPCDChalResp, wPCDChalRespLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a Post delivery configuration Authenticate command part2. This command will decrypt the response received from
        /// card and will return success status if the challanges matches.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by the MFP card. For success it should be 0x90.</param>
        /// <param name="aPDResp">Buffer containing the input received from card.</param>
        /// <param name="bPDRespLen">Length of bytes available in PDResp buffer.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticatePDC_Part2 ( byte bPiccErrCode, byte[] aPDResp, byte bPDRespLen, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_SAM_AuthenticatePDC_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aPDResp,
                bPDRespLen, ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Read command. This command is used to perform GetVersion, ReadSignature and all Read
        /// related operations.
        /// </summary>
        ///
        /// <param name="bLFI">Option for updating the P2 information of Sam frame.
        ///							NxpRdLibNet.Hal.SamAV3.LFI_LAST_FRAME
        ///							NxpRdLibNet.Hal.SamAV3.CHAINED_FRAME</param>
        /// <param name="wOption">Option flag to buffer the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT		: Framing the single payload which has all the information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST: For buffering command information and first set of data information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT	: For buffering Intermediate set of data information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST	: For buffering Final set of data information.
        ///
        ///						  The below flags should be used to switch between command and resposne.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedResd.COMMAND
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedRead.RESPONSE
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedRead.Both
        ///
        ///						  The below flags should be used to update the Plain data in response.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.RETURN_PLAIN
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.SKIP_PLAIN</param>
        /// <param name="aData">The data to be sent to Sam hardware. Should be one of the following.
        ///							For Command:
        ///								For READxyz  : Read Cmd (1byte) + BNR (2byte) + No.Of Blocks (1byte)
        ///								For GetV     : GetVersion command (1byte)
        ///								For Read_Sign: Read_Sign Cmd (1byte) + Address (1btye)
        ///
        ///							For Response:
        ///								Maced / Encrypted data
        ///								Error Code
        ///
        ///							For Command + Response:
        ///								READxyU + BNr + No.Of Blocks + RC + Data + MAC (Optional)
        ///								RC</param>
        /// <param name="bDataLen">Length of bytes available in Data buffer</param>
        /// <param name="aOutput">The complete information received from PICC.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success Chaining status for successfull chaining operation.
        ///			 Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_CombinedReadMFP ( byte bLFI, ushort wOption, byte[] aData, byte bDataLen, out byte[] aOutput, out byte bPiccReturnCode )
        {
            Status_t oStatus;
            ushort wOutputLen = 0;
            IntPtr ppOutput = IntPtr.Zero;

            bPiccReturnCode = 0;
            oStatus = phhalHw_SamAV3_Cmd_SAM_CombinedReadMFP ( ref m_DataParamsInt[0], bLFI, wOption, aData, bDataLen, ref ppOutput, ref wOutputLen,
                ref bPiccReturnCode );

            aOutput = null;
            aOutput = MarshalCopy ( oStatus, ppOutput, wOutputLen );

            return oStatus;
        }

        /// <summary>
        /// Perform a MFP CombinedWrite command. This command is common for Write, WriteValue, Increment, Decrement, IncrementTransfer,
        /// DecrementTransfer, Transfer and Restore.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT		: Framing the single payload which has all the information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST: For buffering command information and first set of data information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT	: For buffering Intermediate set of data information.
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST	: For buffering Final set of data information.
        ///
        ///						  The below flags should be used to switch between command and resposne.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.COMMAND
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.RESPONSE
        ///
        ///						  The below flags should be used to update the Plain data in response.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.RETURN_PLAIN
        ///							NxpRdLibNet.Hal.SamAV3.MFP_CombinedWrite.SKIP_PLAIN</param>
        /// <param name="aData">The data to be sent to Sam hardware. Should be one of the following.
        ///							For Command:
        ///								For WRITExy : Write Cmd (1byte) + BNR (2byte) + PlainData (N * 16 bytes)
        ///								For INCx    : Inc Cmd (1byte) + BNR (2byte) + PlainData (4bytes)
        ///								For DECx    : Dec Cmd (1byte) + BNR (2byte) + PlainData (4bytes)
        ///								For INCTx   : Inc Transfer Cmd (1byte) + BNR (4byte) + PlainData (4bytes)
        ///								For DECTx   : Dec Transfer Cmd (1byte) + BNR (4byte) + PlainData (4bytes)
        ///								For	 TRFx    : Transfer Cmd (1byte) + BNR (2byte)
        ///								For RESx    : Restore Cmd (1byte) + BNR (2byte)
        ///							For Response:
        ///								RC
        ///								RC + MAC
        ///								RC + TMC + TMV
        ///								RC + TMC + TMV + MAC</param>
        /// <param name="bDataLen">Length of bytes available in Data buffer</param>
        /// <param name="aOutput">The complete information received from PICC.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_CombinedWriteMFP ( ushort wOption, byte[] aData, byte bDataLen, out byte[] aOutput, out byte bPiccReturnCode )
        {
            Status_t oStatus;
            ushort wOutputLen = 0;
            IntPtr ppOutput = IntPtr.Zero;

            bPiccReturnCode = 0;
            oStatus = phhalHw_SamAV3_Cmd_SAM_CombinedWriteMFP ( ref m_DataParamsInt[0], wOption, aData, bDataLen, ref ppOutput, ref wOutputLen,
                ref bPiccReturnCode );

            aOutput = null;
            aOutput = MarshalCopy ( oStatus, ppOutput, wOutputLen );

            return oStatus;
        }

        /// <summary>
        /// Prepare the MFP command for key change. This command will return the protected data to be written to card.
        /// </summary>
        ///
        /// <param name="bOption">Option to set the P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.COMMAND
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.RESPONSE
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.DIVERSIFY_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.DIVERSIFY_ON</param>
        /// <param name="aData">The information to be exchange to Sam hardware.
        ///							For Command : The buffer should contain CmdCode, KeyBNo, KeyNo, KeyVer and DivInput.
        ///							For Response: The buffer should contain PICC Status and MAC.</param>
        /// <param name="bDataLen">Length of bytes available in Data buffer.</param>
        /// <param name="aProtectedData">The protected key data from SAM hardware.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyMFP ( byte bOption, byte[] aData, byte bDataLen, out byte[] aProtectedData, out byte bPiccReturnCode )
        {
            Status_t oStatus;
            ushort wProtDataLen = 0;
            IntPtr ppProtectedData = IntPtr.Zero;

            bPiccReturnCode = 0;
            oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyMFP ( ref m_DataParamsInt[0], bOption, aData, bDataLen, ref ppProtectedData, ref wProtDataLen,
                ref bPiccReturnCode );

            aProtectedData = null;
            aProtectedData = MarshalCopy ( oStatus, ppProtectedData, wProtDataLen );
            return oStatus;
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Perform a MIFARE Plus Authenticate command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">Option for Authenticate command.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.DIVERSIFY_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.DIVERSIFY_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.FIRST_AUTH
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.NONFIRST_AUTH
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.SL1_NO_KDF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_Authenticate.SL3_KDF</param>
        /// <param name="bKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="wBlockNo">PICC block number to be used for authentication.</param>
        /// <param name="aPcdCapsIn">Buffer containing the input PcdCaps.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="aPcdCapsOut">Buffer containing the Output PCD capabilities. This will be of 6 bytes.</param>
        /// <param name="aPdCaps">Buffer containing the Output PD capabilities. This will be of 6 bytes.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_Authenticate ( byte bOption, byte bKeyNo, byte bKeyVer, ushort wBlockNo, byte[] aPcdCapsIn, byte[] aDivInput,
            out byte[] aPcdCapsOut, out byte[] aPdCaps, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            aPcdCapsOut = new byte[6];
            aPdCaps = new byte[6];
            return phhalHw_SamAV3_Cmd_MFP_Authenticate ( ref m_DataParamsInt[0], ( byte ) bOption, bKeyNo, bKeyVer, wBlockNo, aPcdCapsIn,
                ( byte ) ( ( aPcdCapsIn == null ) ? 0 : aPcdCapsIn.Length ), aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                aPcdCapsOut, aPdCaps, ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Sector Switch Authenticate command in X mode.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST
        ///							NxpRdLibNet.MFP_SectorSwitchAuth.BUFFER_KEY_BLOCKS
        ///							NxpRdLibNet.MFP_SectorSwitchAuth.BUFFER_DIV_INPUT
        ///
        ///							Option to set the P1 information bytes.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.SECTOR_DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON</param>
        /// <param name="wSSKeyBNr">PICC Sector Switch key block number to be used for authentication.</param>
        /// <param name="bSSKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bSSKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="bMSKeyNo">Key number to be used from Sam keystore if master sector key in enabled.</param>
        /// <param name="bMSKeyVer">Key version to be used from Sam keystore if master sector key in enabled.</param>
        /// <param name="bSectorCount">Number of sectors to be switched inside the PICC.</param>
        /// <param name="aKeyBlocks">Buffer containing the PICC KeyB block number and reference key number and
        ///							 version to be used form Sam keystore.
        ///								The format of the buffer should be,
        ///								KeyBxBNr_1 + KeyNo_1 + KeyVer_1, KeyBxBNr_2 + KeyNo_2 + KeyVer_2, ...,
        ///								KeyBxBNr_N + KeyNo_N + KeyVer_N, Where N equal to the SectorCount information.</param>
        /// <param name="aDivInput">Buffer containing the diversification inputs to be used for diversifying the key.
        ///								The format of the buffer should be,
        ///								SSKeyDivLen + SSKeyDivInput + KeyBxDivLen + KeyBxDivInput</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_AuthSectorSwitch ( ushort wOption, ushort wSSKeyBNr, byte bSSKeyNo, byte bSSKeyVer, byte bMSKeyNo,
            byte bMSKeyVer, byte bSectorCount, byte[] aKeyBlocks, byte[] aDivInput, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_MFP_AuthSectorSwitch ( ref m_DataParamsInt[0], wOption, wSSKeyBNr, bSSKeyNo, bSSKeyVer,
                bMSKeyNo, bMSKeyVer, bSectorCount, aKeyBlocks, ( byte ) ( ( aKeyBlocks == null ) ? 0 : aKeyBlocks.Length ), aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Post Delivery configuration Authenticate command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">Option to set the P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFICATION_OFF
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFICATION_RFU
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DERIVE_UPGRADE_KEY
        ///							NxpRdLibNet.Hal.SamAV3.PDC_Auth.DIVERSIFY_YEAR_UPGRADE_KEY</param>
        /// <param name="bKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="wUpgradeKey">PICC UpgradeKey to be used for authentication.</param>
        /// <param name="aUpgradeInfo">Upgrade information of the target product state.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_PDC_Authenticate ( byte bOption, byte bKeyNo, byte bKeyVer, ushort wUpgradeKey, byte[] aUpgradeInfo, byte[] aDivInput,
            out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_PDC_Authenticate ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, wUpgradeKey, aUpgradeInfo,
                ( byte ) ( ( aUpgradeInfo == null ) ? 0 : aUpgradeInfo.Length ), aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ),
                ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Read command in X mode. This command is used to perform GetVersion, ReadSignature and all Read
        /// related operations.
        /// </summary>
        ///
        /// <param name="wOption">Option for receiving the next data chunk if previous status was chaining.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT: To exchange the MifarePlus PICC commands.
        ///							NxpRdLibNet.ExchangeOptions.RXCHAINING: The next set of data will be received.</param>
        /// <param name="aReadCmd">The different types of command to be sent.
        ///								GetVersion: GetVersion cmd (1byte)
        ///								ReadSig   : Read Signature cmd (1byte) + Address (1byte)
        ///								Read      : Read cmd (1byte) + BlockNr (2byte) + NoBlocks (1byte) \n</param>
        /// <param name="aData">The information returned by Sam hardware for the mentioned command in ReadCmd buffer.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Succcess Chaining for chaining response.
        ///			 Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_CombinedRead ( ushort wOption, byte[] aReadCmd, out byte[] aData, out byte bPiccReturnCode )
        {
            Status_t oStatus;
            IntPtr ppData = IntPtr.Zero;
            ushort wDataLen = 0;

            bPiccReturnCode = 0;
            oStatus = phhalHw_SamAV3_Cmd_MFP_CombinedRead ( ref m_DataParamsInt[0], ( ushort ) wOption, aReadCmd, ( byte ) ( ( aReadCmd == null ) ? 0 : aReadCmd.Length ),
                ref ppData, ref wDataLen, ref bPiccReturnCode );

            aData = null;
            if ( ppData != IntPtr.Zero )
            {
                aData = new byte[wDataLen];
                aData = MarshalCopy ( oStatus, ppData, wDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Plus Combined Write command in X mode. This command performs Write, Increment, Decrement, Transfer,
        /// Restore, IncrementTransfer and DecrementTransfer commands of the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST</param>
        /// <param name="aData">The data to be sent to Sam hardware. Should be one of the following.
        ///							For WRITExy : Write Cmd (1byte) + BNR (2byte) + PlainData (N * 16 bytes)
        ///							For INCx    : Inc Cmd (1byte) + BNR (2byte) + PlainData (4bytes)
        ///							For DECx    : Dec Cmd (1byte) + BNR (2byte) + PlainData (4bytes)
        ///							For INCTx   : Inc Transfer Cmd (1byte) + BNR (4byte) + PlainData (4bytes)
        ///							For DECTx   : Dec Transfer Cmd (1byte) + BNR (4byte) + PlainData (4bytes)
        ///							For TRFx    : Transfer Cmd (1byte) + BNR (2byte)
        ///							For RESx    : Restore Cmd (1byte) + BNR (2byte)</param>
        /// <param name="aTMC">Only available is the block is a TMProtected block. The buffer will have 4
        ///					   bytes of Transaction MAC counter information.</param>
        /// <param name="aTMV">Only available is the block is a TMProtected block. The buffer will have 8
        ///					   bytes of Transaction MAC value.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_CombinedWrite ( ushort wOption, byte[] aData, out byte[] aTMC, out byte[] aTMV, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            aTMC = new byte[4];
            aTMV = new byte[8];
            return phhalHw_SamAV3_Cmd_MFP_CombinedWrite ( ref m_DataParamsInt[0], wOption, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ),
                aTMC, aTMV, ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus ChangeKey command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">Option to set the P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.DIVERSIFY_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFP_ChangeKey.DIVERSIFY_ON</param>
        /// <param name="bCmdCode">The write comamnd code to be used for writting the key. (0xA0 or 0xA1)</param>
        /// <param name="wBlockNo">PICC block number to be used for changing the key.</param>
        /// <param name="bKeyNo">Key number to be used from Sam keystore.</param>
        /// <param name="bKeyVer">Key version to be used from Sam keystore.</param>
        /// <param name="aDivInput">Diversification Input used to diversify the key.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_ChangeKey ( byte bOption, byte bCmdCode, ushort wBlockNo, byte bKeyNo, byte bKeyVer, byte[] aDivInput, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_MFP_ChangeKey ( ref m_DataParamsInt[0], bOption, bCmdCode, wBlockNo, bKeyNo, bKeyVer, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), ref bPiccReturnCode );
        }

        /// <summary>
        /// Perform a MIFARE Plus Write Perso command in X mode.
        /// </summary>
        ///
        /// <param name="wOption">Option flag to buffer the payload information.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_FIRST
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_CONT
        ///							NxpRdLibNet.ExchangeOptions.BUFFER_LAST</param>
        /// <param name="aBlocks">Buffer containing the Block and Data pair to be written to card by Sam hardware.
        ///						  Should be holding an array of block number and data like
        ///						  BNR_1 + Data, BNR_2 + Data, BNR_3 + Data, ..., BNR_N + Data
        ///						  BNR_x should be 2 bytes and Data should 16 bytes.</param>
        /// <param name="bPiccReturnCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MFP_WritePerso ( ushort wOption, byte[] aBlocks, out byte bPiccReturnCode )
        {
            bPiccReturnCode = 0;
            return phhalHw_SamAV3_Cmd_MFP_WritePerso ( ref m_DataParamsInt[0], wOption, aBlocks, ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ),
                ref bPiccReturnCode );
        }
        #endregion X Mode
        #endregion

        #region MIFARE Classic
        #region S Mode
        /// <summary>
        /// Performs the MIFARE Classic authentication in S mode. This interfaces exchanges the first
        /// part of random challange received from PICC to Sam hardware.
        /// </summary>
        ///
        /// <param name="bOption">Specify whether diversification block number should be exchanged or not.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_Authenticate.DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFC_Authenticate.DIV_ON</param>
        /// <param name="aMFUID">MIFARE classic UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVer">Key Version of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyType">Type of key to be used.
        ///							NxpRdLibNet.Hal.KeyType.KEYA
        ///							NxpRdLibNet.Hal.KeyType.KEYB</param>
        /// <param name="bMFBlockNo">MIFARE block number used for authentication.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        /// <param name="aNumberRB">Should have the 5bytes (4 bytes of Rnd + 1 byte of Parity) of information (Token RB) returned by PICC.</param>
        /// <param name="aEncToken">The encrypted (Token	AB) information to be send to PICC. This will have 9 bytes (8 bytes of encrypted
        ///							information + 1 byte of Parity)</param>
        ///
        /// <returns>Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateMIFARE_Part1 ( byte bOption, byte[] aMFUID, byte bKeyNo, byte bKeyVer, byte bKeyType, byte bMFBlockNo,
            byte bDivBlockNo, byte[] aNumberRB, out byte[] aEncToken )
        {
            Status_t oStatus;
            IntPtr ppEncToken = IntPtr.Zero;
            ushort pEncTokenLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part1 ( ref m_DataParamsInt[0], bOption, aMFUID, bKeyNo, bKeyVer, bKeyType, bMFBlockNo,
                bDivBlockNo, aNumberRB, ( byte ) ( ( aNumberRB == null ) ? 0 : aNumberRB.Length ), ref ppEncToken, ref pEncTokenLen );

            byte[] RxBuffer = MarshalCopy ( oStatus, ppEncToken, pEncTokenLen );
            aEncToken = RxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Performs the MIFARE Classic authentication in S mode. This interfaces exchanges the second
        /// part of random challange received from PICC to Sam hardware.
        /// </summary>
        ///
        /// <param name="pEncToken">Should have the 5bytes (4 bytes of Rnd + 1 byte of Parity) of information (Token BA) returned by PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateMIFARE_Part2 ( byte[] aEncToken )
        {
            return phhalHw_SamAV3_Cmd_SAM_AuthenticateMIFARE_Part2 ( ref m_DataParamsInt[0], aEncToken,
                ( byte ) ( ( aEncToken == null ) ? 0 : aEncToken.Length ) );
        }

        /// <summary>
        /// Performs the MIFARE Classic key change in S mode.
        /// </summary>
        ///
        /// <param name="bOption">Specify whether diversification block number and UID should be exchanged or not.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_A_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_B_ON</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVerA">Key Version of MIFARE key A available in Sam keystore.</param>
        /// <param name="bKeyVerB">Key Version of MIFARE key B available in Sam keystore.</param>
        /// <param name="aAccCond">MIFARE classic access conditions.</param>
        /// <param name="aMFUID">MIFARE classic UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        /// <param name="aProtData">The protected information to be sent to PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyMIFARE ( byte bOption, byte bKeyNo, byte bKeyVerA, byte bKeyVerB, byte[] aAccCond, byte[] aMFUID,
            byte bDivBlockNo, out byte[] aProtData )
        {
            Status_t oStatus;
            IntPtr ppProtData = IntPtr.Zero;
            ushort pProtDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyMIFARE ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVerA, bKeyVerB, aAccCond,
                aMFUID, bDivBlockNo, ref ppProtData, ref pProtDataLen );

            byte[] RxBuffer = MarshalCopy ( oStatus, ppProtData, pProtDataLen );
            aProtData = RxBuffer;

            return oStatus;
        }

        /// <summary>
        /// Performs the MIFARE Classic key dump in S mode.
        /// </summary>
        ///
        /// <param name="bOption">Specify whether diversification block number and UID should be exchanged or not.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_PLAIN
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_ENCRYPT
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_A_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_B_ON</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVer">Key Version of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyType">Type of key to be used.
        ///							NxpRdLibNet.Hal.KeyType.KEYA
        ///							NxpRdLibNet.Hal.KeyType.KEYB</param>
        /// <param name="aMFUID">MIFARE classic UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        /// <param name="aSecretKey">The Secret key information to be sent to PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_ChangeKeyMIFAREDump ( byte bOption, byte bKeyNo, byte bKeyVer, byte bKeyType, byte[] aMFUID, byte bDivBlockNo,
            out byte[] aSecretKey )
        {
            Status_t oStatus;
            IntPtr ppSecretKey = IntPtr.Zero;
            ushort pSecretKeyLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_SAM_ChangeKeyMIFAREDump ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, bKeyType, aMFUID,
                bDivBlockNo, ref ppSecretKey, ref pSecretKeyLen );

            byte[] RxBuffer = MarshalCopy ( oStatus, ppSecretKey, pSecretKeyLen );
            aSecretKey = RxBuffer;

            return oStatus;
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Perform a MIFARE Classic Authenticate command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">Specify whether diversification block number should be exchanged or not.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_Authenticate.DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFC_Authenticate.DIV_ON</param>
        /// <param name="aMFUID">MIFARE classic UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVer">Key Version of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyType">Type of key to be used.
        ///							NxpRdLibNet.Hal.KeyType.KEYA
        ///							NxpRdLibNet.Hal.KeyType.KEYB</param>
        /// <param name="bMFBlockNo">MIFARE block number used for authentication.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Authenticate ( byte bOption, byte[] aMFUID, byte bKeyNo, byte bKeyVer, byte bKeyType, byte bMFBlockNo,
            byte bDivBlockNo )
        {
            return phhalHw_SamAV3_Cmd_MF_Authenticate ( ref m_DataParamsInt[0], bOption, aMFUID, bKeyNo, bKeyVer, bKeyType, bMFBlockNo, bDivBlockNo );
        }

        /// <summary>
        /// Perform a MIFARE Classic Read command in X mode.
        /// </summary>
        ///
        /// <param name="aBlocks">The block numbers from where the data should be read.
        ///						  Should be holding an array of block numbers like
        ///						  BNR_1, BNR_2, BNR_3, ..., BNR_N</param>
        /// <param name="aData">Data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Read ( byte[] aBlocks, out byte[] aData )
        {
            Status_t oStatus;
            IntPtr ppData = IntPtr.Zero;
            ushort pDataLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_MF_Read ( ref m_DataParamsInt[0], aBlocks, ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ),
                ref ppData, ref pDataLen );

            aData = null;
            if ( ( ppData != IntPtr.Zero ) && ( pDataLen > 0 ) )
            {
                aData = new byte[pDataLen];
                aData = MarshalCopy ( oStatus, ppData, pDataLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Classic Write command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">Specifying the product type targeted for writing the data.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_OpMode.WRITE_CLASSIC
        ///							NxpRdLibNet.Hal.SamAV3.MFC_OpMode.WRITE_ULTRALIGHT
        ///
        ///						  For exchanging the LE byte. To be ored with above options
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aData">Buffer containing the Block and Data pair to be written to card by Sam hardware.
        ///						Should be holding an array of block number and data like
        ///						BNR_1 + Data, BNR_2 + Data, BNR_3 + Data, ..., BNR_N + Data
        ///						Data could be 4 bytes or 16 bytes depending on the option specified.</param>
        ///	<param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of Ultralight or not a TMProtected block in case
        ///						  of Classic.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Write ( byte bOption, byte[] aData, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_Write ( ref m_DataParamsInt[0], bOption, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref ppTMData, ref pTMDataLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Classic Write Value command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">For exchanging the LE byte
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aBlocks">Buffer containing the Block, Value and Address pair to be written to card by Sam hardware.
        ///						  Should be holding an array of block number, value and address like
        ///						  BNR_1 + Value + Address, BNR_2 + Value + Address, ..., BNR_N + Value + Address</param>
        /// <param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of not a TMProtected block.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_ValueWrite ( byte bOption, byte[] aBlocks, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_ValueWrite ( ref m_DataParamsInt[0], bOption, aBlocks,
                ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ), ref ppTMData, ref pTMDataLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Classic Increment command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">For exchanging the LE byte
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aBlocks">Buffer containing the Source Block, Destination Block and Value pair to be
        ///						  written to card by Sam hardware. Should be holding an array of source block number,
        ///						  destination block number and value like
        ///						  SRC_BNR_1 + DST_BNR_1 + Value, SRC_BNR_2 + DST_BNR_2 + Value, ...,
        ///						  SRC_BNR_N + DST_BNR_N + Value</param>
        /// <param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of not a TMProtected block.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Increment ( byte bOption, byte[] aBlocks, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_Increment ( ref m_DataParamsInt[0], bOption, aBlocks,
                ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ), ref ppTMData, ref pTMDataLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Classic Decrement command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">For exchanging the LE byte
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aBlocks">Buffer containing the Source Block, Destination Block and Value pair to be
        ///						  written to card by Sam hardware. Should be holding an array of source block number,
        ///						  destination block number and value like
        ///						  SRC_BNR_1 + DST_BNR_1 + Value, SRC_BNR_2 + DST_BNR_2 + Value, ...,
        ///						  SRC_BNR_N + DST_BNR_N + Value</param>
        /// <param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of not a TMProtected block.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Decrement ( byte bOption, byte[] aBlocks, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_Decrement ( ref m_DataParamsInt[0], bOption, aBlocks,
                ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ), ref ppTMData, ref pTMDataLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Classic Restore command in X mode.
        /// </summary>
        ///
        /// <param name="bOption">For exchanging the LE byte
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aBlocks">Buffer containing the Source Block and Destination Block pair to be
        ///						  written to card by Sam hardware. Should be holding an array of source
        ///						  block number and destination block numbe like
        ///						  SRC_BNR_1 + DST_BNR_1, SRC_BNR_2 + DST_BNR_2, ..., SRC_BNR_N + DST_BNR_N</param>
        /// <param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of not a TMProtected block.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_Restore ( byte bOption, byte[] aBlocks, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_Restore ( ref m_DataParamsInt[0], bOption, aBlocks,
                ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ), ref ppTMData, ref pTMDataLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Authenticate followed by a MIFARE Read command in X mode.
        /// </summary>
        ///
        /// <param name="wOption">Option mask for this command. On every call a single data cycle is buffered.
        ///						  A cycle is one sequence of CmdSettingss | KeyNo | KeyV | KeyType | AuthBlockNo |
        ///						  DivBlockNo| NumBlocks | Blocks
        ///
        ///						  One of the below mentioned blocks to be used.
        ///						  NxpRdLibNet.ExchangeOptions.DEFAULT      (Single command frame to be exchanged with Sam hardware)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_FIRST (The header and the initial paylod information will be bufferred internally.)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_CONT  (The final payload if available and bufferred information will be sent to the SAM hardware)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_LAST  (The bufferred information will be sent to the SAM hardware)</param>
        /// <param name="aMFUid">MIFARE standard UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.
        ///						 This buffer is only used if wOption is set to NxpRdLibNet.ExchangeOptions.DEFAULT or NxpRdLibNet.ExchangeOptions.BUFFER_FIRST.</param>
        /// <param name="bCmdSettings">One of the below mentioned options to be used to update the CmdSettings
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_KEY_INFO_NOT_AVAILABLE
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_KEY_INFO_AVAILABLE
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_CMD_SET_DIV_OFF
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_CMD_SET_DIV_ON</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVer">Key Version of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyType">Type of key to be used.
        ///							NxpRdLibNet.Hal.KeyType.KEYA
        ///							NxpRdLibNet.Hal.KeyType.KEYB</param>
        /// <param name="bAuthBlockNo">MIFARE block number used for authentication.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        /// <param name="aBlocks">The block numbers from where the data should be read.</param>
        /// <param name="aData">Data returned by Sam hardware.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_AuthenticatedRead ( ushort wOption, byte[] aMFUid, byte bCmdSettings, byte bKeyNo, byte bKeyVer, byte bKeyType,
            byte bAuthBlockNo, byte bDivBlockNo, byte[] aBlocks, out byte[] aData )
        {
            Status_t oStatus;
            IntPtr ppData = IntPtr.Zero;
            ushort pDataLen = 0;

            aData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_AuthenticatedRead ( ref m_DataParamsInt[0], wOption, aMFUid, bCmdSettings, bKeyNo,
                bKeyVer, bKeyType, bAuthBlockNo, bDivBlockNo, aBlocks, ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ),
                ref ppData, ref pDataLen );

            /* Copy the Data to parameter if bytes are available. */
            if ( !pDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppData, pDataLen );
                aData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform a MIFARE Authenticate followed by a MIFARE Write command in X mode.
        /// </summary>
        ///
        /// <param name="wOption">Option mask for this command. On every call a single data cycle is buffered.
        ///						  A cycle is one sequence of CmdSettingss | KeyNo | KeyV | KeyType | AuthBlockNo |
        ///						  DivBlockNo| NumBlocks | Blocks
        ///
        ///						  One of the below mentioned blocks to be used.
        ///						  NxpRdLibNet.ExchangeOptions.DEFAULT      (Single command frame to be exchanged with Sam hardware)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_FIRST (The header and the initial paylod information will be bufferred internally.)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_CONT  (The final payload if available and bufferred information will be sent to the SAM hardware)
        ///						  NxpRdLibNet.ExchangeOptions.BUFFER_LAST  (The bufferred information will be sent to the SAM hardware)
        ///
        ///						  One of the below option ORED with above ones for LE byte exchange.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.NOT_RETURNED
        ///							NxpRdLibNet.Hal.SamAV3.MFC_TMData.RETURNED</param>
        /// <param name="aMFUid">MIFARE standard UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.
        ///						 This buffer is only used if wOption is set to NxpRdLibNet.ExchangeOptions.DEFAULT or NxpRdLibNet.ExchangeOptions.BUFFER_FIRST.</param>
        /// <param name="bCmdSettings">One of the below mentioned options to be used to update the CmdSettings
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_KEY_INFO_NOT_AVAILABLE
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_KEY_INFO_AVAILABLE
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_CMD_SET_DIV_OFF
        ///									NxpRdLibNet.Hal.SamAV3.MFC_CmdSetting.AUTH_RW_CMD_SET_DIV_ON</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVer">Key Version of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyType">Type of key to be used.
        ///							NxpRdLibNet.Hal.KeyType.KEYA
        ///							NxpRdLibNet.Hal.KeyType.KEYB</param>
        /// <param name="bAuthBlockNo">MIFARE block number used for authentication.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        /// <param name="aBlocks">Buffer containing the Block and Data pair to be written to card by Sam hardware.
        ///						Should be holding an array of block number and data like
        ///						BNR_1 + Data, BNR_2 + Data, BNR_3 + Data, ..., BNR_N + Data
        ///						Data could be 4 bytes or 16 bytes depending on the option specified.</param>
        ///	<param name="aTMData">The Transaction Mac Value and Counter of each block returned by MIFARE Classic PICC if
        ///						  its a TMProtected block. Null in case of Ultralight or not a TMProtected block in case
        ///						  of Classic.</param>
        ///
        /// <returns>Returns Success for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_AuthenticatedWrite ( ushort wOption, byte[] aMFUid, byte bCmdSettings, byte bKeyNo, byte bKeyVer, byte bKeyType,
            byte bAuthBlockNo, byte bDivBlockNo, byte[] aBlocks, out byte[] aTMData )
        {
            Status_t oStatus;
            IntPtr ppTMData = IntPtr.Zero;
            ushort pTMDataLen = 0;

            aTMData = null;

            oStatus = phhalHw_SamAV3_Cmd_MF_AuthenticatedWrite ( ref m_DataParamsInt[0], wOption, aMFUid, bCmdSettings, bKeyNo,
                bKeyVer, bKeyType, bAuthBlockNo, bDivBlockNo, aBlocks, ( byte ) ( ( aBlocks == null ) ? 0 : aBlocks.Length ),
                ref ppTMData, ref pTMDataLen );

            /* Copy the Data to parameter if bytes are available. */
            if ( !pTMDataLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppTMData, pTMDataLen );
                aTMData = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs the MIFARE Classic key change in S mode.
        /// </summary>
        ///
        /// <param name="bOption">Specify whether diversification block number and UID should be exchanged or not.
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_A_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFC_ChangeKey.CHANGE_KEY_MIFARE_DIV_B_ON</param>
        /// <param name="bKeyNo">Key number of MIFARE key available in Sam keystore.</param>
        /// <param name="bKeyVerA">Key Version of MIFARE key A available in Sam keystore.</param>
        /// <param name="bKeyVerB">Key Version of MIFARE key B available in Sam keystore.</param>
        /// <param name="bMFBlockNo"></param>
        /// <param name="aAccCond">MIFARE classic access conditions.</param>
        /// <param name="aMFUID">MIFARE classic UID. Here the last four bytes of the UID should be passed regardless of 4 or 7 byte UID.</param>
        /// <param name="bDivBlockNo">Block number to be used for Key diversification in Sam.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_MF_ChangeKey ( byte bOption, byte bKeyNo, byte bKeyVerA, byte bKeyVerB, byte bMFBlockNo, byte[] aAccCond,
            byte[] aMFUID, byte bDivBlockNo )
        {
            return phhalHw_SamAV3_Cmd_MF_ChangeKey ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVerA, bKeyVerB, bMFBlockNo,
                aAccCond, aMFUID, bDivBlockNo );
        }
        #endregion
        #endregion MIFARE Classic

        #region MIFARE Ultralight
        #region S Mode
        /// <summary>
        ///   Performs PwdAuthUL command execution part 1 in S mode. The last part includes the header, Key number, Key Version
        ///   Diversification input and the Message response is received.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number of Upload key (Ku) in software keystore.</param>
        /// <param name="bKeyVer">Key version of Upload key (Ku) in software keystore.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input)</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="aPwd">Overall message.</param>
        ///
        /// <returns>Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_PwdAuthUL_Part1 ( byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen, out byte[] aPwd )
        {
            Status_t oStatus;
            IntPtr ppPwd = IntPtr.Zero;
            ushort pPwdLen = 0;

            aPwd = null;

            oStatus = phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part1 ( ref m_DataParamsInt[0], bKeyNo, bKeyVer, aDivInput, bDivInputLen,
                ref ppPwd, ref pPwdLen );

            /* Copy the TMData to parameter if bytes are available. */
            if ( !pPwdLen.Equals ( 0 ) )
            {
                byte[] RxBuffer = MarshalCopy ( oStatus, ppPwd, pPwdLen );
                aPwd = RxBuffer;
            }

            return oStatus;
        }

        /// <summary>
        /// Performs PwdAuthUL command execution part 2 in S mode. The last part includes the header and the response received
        /// </summary>
        ///
        /// <param name="pPACK">Passwork Authentication Acknowledge.</param>
        /// <param name="bPACKLen">Length of the Passwork Authentication Acknowledge</param>

        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_PwdAuthUL_Part2 ( ushort wPack )
        {
            return phhalHw_SamAV3_Cmd_SAM_PwdAuthUL_Part2 ( ref m_DataParamsInt[0], wPack );
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Performs PwdAuthUL command execution in X mode.Includes the header, Key number, Key Version, Diversification input.
        /// </summary>
        /// <param name="bKeyNo">Key number available in Sam keystore.</param>
        /// <param name="bKeyVer">Key version available in Sam keystore.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input)</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        /// <param name="bStatusCode">Status code returned by PICC</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_UL_PwdAuthUL ( byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen, out byte bStatusCode )
        {
            bStatusCode = 0;
            Status_t oStatus = phhalHw_SamAV3_Cmd_UL_PwdAuthUL ( ref m_DataParamsInt[0], bKeyNo, bKeyVer, aDivInput, bDivInputLen, ref bStatusCode );

            return oStatus;
        }

        /// <summary>
        /// Performs AuthenticatePICC command execution in X mode.Includes the header, Key number, Key Version,
        /// Diversification input based on Option Field.
        /// </summary>
        ///
        /// <param name="bOption">Option for P1 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.MFULOption.ULC_AUTHENTICATE_DIV_OFF
        ///							NxpRdLibNet.Hal.SamAV3.MFULOption.ULC_AUTHENTICATE_DIV_ON
        ///							NxpRdLibNet.Hal.SamAV3.MFULOption.ULC_AUTHENTICATE_EXCLUDE_LE
        ///							NxpRdLibNet.Hal.SamAV3.MFULOption.ULC_AUTHENTICATE_INCLUDE_LE</param>
        /// <param name="bKeyNo">Key number available in Sam keystore.</param>
        /// <param name="bKeyVer">Key version available in Sam keystore.</param>
        /// <param name="aDivInput">Diversification input for key diversification. (1 to 31 byte(s) input)</param>
        /// <param name="bDivInputLen">Length of diversification input used to diversify the key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_ULC_AuthenticatePICC ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aDivInput, byte bDivInputLen, out byte bStatusCode )
        {
            bStatusCode = 0;
            Status_t oStatus = phhalHw_SamAV3_Cmd_ULC_AuthenticatePICC ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aDivInput, bDivInputLen, ref bStatusCode );

            return oStatus;
        }
        #endregion X Mode
        #endregion MIFARE Ultralight

        #region MIFARE Common
        #region S Mode
        /// <summary>
        /// Performs CommitReaderID command execution part 1 in S mode. The first part includes the header, block number if
        /// its MifarePlus state or only header if its Desfire state.
        /// </summary>
        ///
        /// <param name="bState">Options for framing the command. Below options should be used
        ///							NxpRdLibNet.Hal.SamAV3.State.MFP
        ///							NxpRdLibNet.Hal.SamAV3.State.DESFIRE</param>
        /// <param name="wBlockNr">Two bytes of block number if the state is MFP.</param>
        /// <param name="aResponse">Transaction MAC Reader ID and MAC to be sent to MIFARE Plus or Desfire PICC</param>
        ///
        /// <returns>Returns active chaining status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_CommitReaderID_Part1 ( byte bState, ushort wBlockNr, out byte[] aResponse )
        {
            ushort pTMRILen = 0;
            IntPtr ppTMRI = IntPtr.Zero;

            aResponse = null;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part1 ( ref m_DataParamsInt[0], bState, wBlockNr, ref ppTMRI, ref pTMRILen );

            aResponse = MarshalCopy ( oStatus, ppTMRI, pTMRILen );

            return oStatus;
        }

        /// <summary>
        /// Performs CommitReaderID command execution part 2 in S mode. The last part includes the header and the response received
        /// from Mfifare Plus or Desfire PICC.
        /// </summary>
        ///
        /// <param name="bPiccErrCode">Error Code sent by the MFP card. For success it should be 0x90.</param>
        /// <param name="aData">The complete data (ENCTMRI and MAC) received from MIFARE Plus or Desfire PICC.</param>
        /// <param name="bDataLen">Length of the data returned by MIFARE Plus or Desfire PICC</param>
        /// <param name="bPiccRetCode">Error code returned by MIFARE PICC. This will of 1 byte in length.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_CommitReaderID_Part2 ( byte bPiccErrCode, byte[] aData, byte bDataLen, out byte bPiccRetCode )
        {
            bPiccRetCode = 0;
            return phhalHw_SamAV3_Cmd_SAM_CommitReaderID_Part2 ( ref m_DataParamsInt[0], bPiccErrCode, aData, bDataLen, ref bPiccRetCode );
        }
        #endregion S Mode

        #region X Mode
        /// <summary>
        /// Performs CommitReaderID command execution in X mode. If success is returned the PICC return code will
        /// have 0x00 as the value else the actual error code.
        /// </summary>
        ///
        /// <param name="bISOMode">Options for P2 information byte.
        ///							NxpRdLibNet.Hal.SamAV3.TMRI_IsoMode.NATIVE
        ///							NxpRdLibNet.Hal.SamAV3.TMRI_IsoMode.ISO7816_4</param>
        /// <param name="bState">The PICC state in which the Sam is currenlty authenticated to.
        ///							NxpRdLibNet.Hal.SamAV3.State.MFP
        ///							NxpRdLibNet.Hal.SamAV3.State.DESFIRE</param>
        /// <param name="wBlockNr">Block to be used if PICC state is MIFARE Plus EV1.</param>
        /// <param name="aEncTMRI">Encrypted TM Reader ID as returned by PICC</param>
        /// <param name="bPiccRetCode">Status code returned by PICC</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_TMRI_CommitReaderID ( byte bISOMode, byte bState, ushort wBlockNr, out byte[] aEncTMRI, out byte bPiccRetCode )
        {
            ushort pEncTMRILen = 0;
            IntPtr ppEncTMRI = IntPtr.Zero;

            bPiccRetCode = 0;
            aEncTMRI = null;

            Status_t oStatus = phhalHw_SamAV3_Cmd_TMRI_CommitReaderID ( ref m_DataParamsInt[0], bISOMode, bState, wBlockNr, ref ppEncTMRI, ref pEncTMRILen,
                ref bPiccRetCode );

            aEncTMRI = MarshalCopy ( oStatus, ppEncTMRI, pEncTMRILen );

            return oStatus;
        }
        #endregion X Mode
        #endregion MIFARE Common

        #region ISO / IEC 29167 - 10
        /// <summary>
        /// Generates 10 bytes of random challange to be given to card. Also Decrypts the TResponse received from
        /// card and verifies the decrypted data for authenticity.
        /// received from card.
        /// </summary>
        ///
        /// <param name="bOption">The command to be framed for SAM.
        ///						  NxpRdLibNet.Hal.SamAV3.Tam.GET_RND
        ///						  NxpRdLibNet.Hal.SamAV3.Tam.PROCESS_TRESPONE</param>
        /// <param name="bKeyNo">Key reference number in hardware keytsore.</param>
        /// <param name="bKeyVer">Key reference version in hardware keytsore.</param>
        /// <param name="aData">Diversification Input or TResponse received from card.</param>
        /// <param name="aIChallange">The IChallange to be sent to card. This will contain 10 bytes of random challange data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AuthenticateTAM1 ( byte bOption, byte bKeyNo, byte bKeyVer, byte[] aData, out byte[] aIChallange )
        {
            Status_t oStatus;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM1 ( ref m_DataParamsInt[0], bOption, bKeyNo, bKeyVer, aData,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pRxBufferInt, ref wRxLength );

            aIChallange = MarshalCopy ( oStatus, pRxBufferInt, wRxLength );

            return oStatus;
        }

        /// <summary>
        ///	Generates 10 bytes of random challange to be given to card. Also Decrypts the TResponse received from
        ///	card and verifies the decrypted data for authenticity and provides the custom data received form card.
        /// </summary>
        /// <param name="bOption">The command to be framed for SAM.
        ///						  \c 0x00: Get Randon Number
        ///						  \c 0x01: Verify </param>
        /// <param name="wKeyNo">Key address in SAM.</param>
        /// <param name="wKeyVer">Key version in SAM.</param>
        /// <param name="aData">Diversification Input or TResponse received from card.</param>
        /// <param name="bBlockSize">To select the size of custom data block to be used.
        ///							 The value should either be 0x00 for 16 bit block size or 0x01 for 64 bit
        ///							 block size. As per ISO 29167</param>
        /// <param name="bBlockCount">To select the custom data block to be used from the offset specified.
        ///							  The BlockCount range is from 1 - 16.</param>
        /// <param name="bProtMode">To specify the mode of operation to be used for encryption/decryption.
        ///							The ProtMode ranges form 0 - 3. As per ISO 29167</param>
        /// <param name="aResponse">The IChallange to be sent to card. This will contain 10 bytes of random challange data.
        ///							Or the Custom Data received from card.</param>
        /// <returns>Returns SUCCESS or Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_AV3_AuthenticateTAM2 ( byte bOption, ushort wKeyNo, ushort wKeyVer, byte[] aData, byte bBlockSize,
            byte bBlockCount, byte bProtMode, out byte[] aResponse )
        {
            Status_t oStatus;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;


            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateTAM2 ( ref m_DataParamsInt[0], bOption, wKeyNo, wKeyVer, aData,
                ( byte ) aData.Length, bBlockSize, bBlockCount, bProtMode, ref pRxBufferInt, ref wRxLength );

            if ( ( pRxBufferInt != IntPtr.Zero ) && ( wRxLength > 0 ) )
            {
                aResponse = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, aResponse, 0, wRxLength );
            }
            else
            {
                aResponse = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Generates 10 bytes of random challange to be given to card.
        /// </summary>
        ///
        /// <param name="bKeyNo">Key reference number in hardware keytsore.</param>
        /// <param name="bKeyVer">Key reference version in hardware keytsore.</param>
        /// <param name="aData">Diversification Input.</param>
        /// <param name="bPurposeMAM2">Purpose MAM 2 data. A 4 bit value.
        ///						  NxpRdLibNet.Hal.SamAV3.Mam.NONE
        ///						  NxpRdLibNet.Hal.SamAV3.Mam.DISABLE_PRIVACY_HF_RESET
        ///						  NxpRdLibNet.Hal.SamAV3.Mam.ENABLE_PRIVACY
        ///						  NxpRdLibNet.Hal.SamAV3.Mam.DISABLE_PRIVACY
        ///						  NxpRdLibNet.Hal.SamAV3.Mam.DESTROY</param>
        /// <param name="aIChallange">The IChallange to be sent to card. This will contain 10 bytes of random challange data.</param>
        public Status_t Cmd_SAM_AuthenticateMAM1 ( byte bKeyNo, byte bKeyVer, byte[] aData, byte bPurposeMAM2, out byte[] aIChallange )
        {
            Status_t oStatus;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;


            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM1 ( ref m_DataParamsInt[0], bKeyNo, bKeyVer, aData, bPurposeMAM2,
                ( byte ) ( ( aData == null ) ? 0 : aData.Length ), ref pRxBufferInt, ref wRxLength );

            aIChallange = MarshalCopy ( oStatus, pRxBufferInt, wRxLength );

            return oStatus;
        }

        /// <summary>
        /// Decrypts the TResponse received from card and verifies the decrypted data for authenticity. Also
        /// frames the IResponse with will be sent to the card.
        /// </summary>
        ///
        /// <param name="aData">TResponse received from card.</param>
        /// <param name="aIResponse">The IResponse generated by SAM that will be sent to card. This will contain 16 bytes of data.</param>
        public Status_t Cmd_SAM_AuthenticateMAM2 ( byte[] aData, out byte[] aIResponse )
        {
            Status_t oStatus;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            oStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticateMAM2 ( ref m_DataParamsInt[0], aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ),
                ref pRxBufferInt, ref wRxLength );

            aIResponse = MarshalCopy ( oStatus, pRxBufferInt, wRxLength );

            return oStatus;
        }
        #endregion ISO / IEC 29167 - 10

        #region EMV
        /// <summary>
        /// Performs EMV Offline Static Data Authentication. This command recovers the static application data for the Static Data
        /// Authentication.
        /// </summary>
        ///
        /// <param name="wOption">Option for exchanging the data in chaining or in one frame.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING</param>
        /// <param name="aSignedStaticAppData">Signed Static Application Data.</param>
        /// <param name="aResponse">Buffer containing Hash Algorithm Indicator, Data Authentication Code and Hash Result.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_EMVCo_RecoverStaticData ( ushort wOption, byte[] aSignedStaticAppData, out byte[] aResponse )
        {
            IntPtr ppResponse = IntPtr.Zero;
            byte bRespLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_EMVCo_RecoverStaticData ( ref m_DataParamsInt[0], wOption, aSignedStaticAppData,
                (byte) ( ( aSignedStaticAppData == null ) ? 0 : aSignedStaticAppData.Length ), ref ppResponse, ref bRespLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, bRespLen );
            return oStatus;
        }

        /// <summary>
        /// Performs EMV Offline Dynamic Data Authentication. This command recovers the dynamic application data for
        /// the Dynamic Data Authentication or Combined Dynamic Data Authentication with Application Cryptogram Generation.
        /// The ICC Public Key or ICC PIN Encipherment Public Key must have been loaded previously with
        /// <seealso cref="NxpRdLibNet.Hal.SamAV3.Cmd_PKI_LoadIccPk"/>.
        /// </summary>
        ///
        /// <param name="wOption">Option for exchanging the data in chaining or in one frame.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING</param>
        /// <param name="aSignedDynamicAppData">Signed Dynamic Application Data.</param>
        /// <param name="aResponse">Buffer containing Hash Algorithm Indicator, ICC Dynamic Data Length, ICC Dynamic Data and Hash result.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_EMVCo_RecoverDynamicData ( ushort wOption, byte[] aSignedDynamicAppData, out byte[] aResponse )
        {
            IntPtr ppResponse = IntPtr.Zero;
            byte bRespLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_EMVCo_RecoverDynamicData ( ref m_DataParamsInt[0], wOption, aSignedDynamicAppData,
                (byte) ( ( aSignedDynamicAppData == null ) ? 0 : aSignedDynamicAppData.Length ), ref ppResponse, ref bRespLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, bRespLen );
            return oStatus;
        }

        /// <summary>
        /// Performs EMV Personal Identification Number Encipherment. This command is used to encipher the PIN to support offline PIN
        /// verification by the ICC. The ICC Public Key or ICC PIN Encipherment Public Key must have been loaded previously with
        /// <seealso cref="NxpRdLibNet.Hal.SamAV3.Cmd_PKI_LoadIccPk"/>.
        /// </summary>
        ///
        /// <param name="bPinBlock">Pin Block Number.</param>
        /// <param name="bIccNumber">ICC Unpredictable Number.</param>
        /// <param name="aEncPin">Enciphered PIN Data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_EMVCo_EncipherPin ( byte[] bPinBlock, byte[] bIccNumber, out byte[] aEncPin )
        {
            IntPtr ppEncPin = IntPtr.Zero;
            byte bEncPinLen = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_EMVCo_EncipherPin ( ref m_DataParamsInt[0], bPinBlock, bIccNumber, ref ppEncPin, ref bEncPinLen );

            aEncPin = MarshalCopy ( oStatus, ppEncPin, bEncPinLen );
            return oStatus;
        }
        #endregion EMV

        #region Programmable Logic
        /// <summary>
        /// Performs Programmable Logic execution.
        /// </summary>
        ///
        /// <param name="bLFI">Option for updating the P1 information of Sam frame.
        ///							NxpRdLibNet.Hal.SamAV3.LFI.LAST_FRAME
        ///							NxpRdLibNet.Hal.SamAV3.LFI.CHAINED_FRAME</param>
        /// <param name="aPLData">Programmable Logic command data.</param>
        /// <param name="bPLDataLen">Length of Programmable Logic command data.</param>
        /// <param name="aPLResp">Buffer holding the Programmable Logic response data.</param>
        ///
        /// <returns>Returns active chaining or Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_PLExec ( byte bLFI, byte[] aPLData, byte bPLDataLen, out byte[] aPLResp )
        {
            ushort pPLRespLen = 0;
            IntPtr ppPLResp = IntPtr.Zero;

            aPLResp = null;

            Status_t oStatus = phhalHw_SamAV3_Cmd_SAM_PLExec ( ref m_DataParamsInt[0], bLFI, aPLData, bPLDataLen, ref ppPLResp, ref pPLRespLen );
            if ( !pPLRespLen.Equals ( 0 ) )
            {
                aPLResp = new byte[pPLRespLen];
                Marshal.Copy ( ppPLResp, aPLResp, 0, pPLRespLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Performs Programmable Logic code upload.
        /// </summary>
        ///
        /// <param name="bIsFirstFrame">Option to represent the first frame where the UploadCtr will be exchanged.
        ///									0x01: To represent the frame as first frame.
        ///									0x00: To represent the frame as intermediate frame.</param>
        /// <param name="bIsFinalFrame">Option to represent the last frame. If set, the LE byte will be exchanged and
        ///								PLUploadACK will be received from SAM and will be validated internally.
        ///									0x01: To represent the final frame and request for PLUploadACK.
        ///									0x00: To represent the intermediate frame.</param>
        ///	<param name="wUploadCtr">The upload counter value. This should be greater than the one availabe in SAM.</param>
        /// <param name="bKeyNo">Key number of Upload key (Ku) in software keystore.</param>
        /// <param name="bKeyVer">Key version of Upload key (Ku) in software keystore.</param>
        /// <param name="aPLCode">Plain Programmable Logic code.</param>
        /// <param name="wPLCodeLen">Plain Programmable Logic code length.</param>
        /// <param name="aPLReKey">The Rekey to be used for next crypto segment. This should have the next SessionENC
        ///						   key followed by the SessionMAC key.</param>
        ///	<param name="bPLReKeyLen">Length of bytes available in PLReKey buffer. The length should be equal to double.</param>
        ///
        /// <returns>Returns active chaining or Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_SAM_PLUpload ( byte bIsFirstFrame, byte bIsFinalFrame, ushort wUploadCtr, byte bKeyNo, byte bKeyVer,
            byte[] aPLCode, ushort wPLCodeLen, byte[] aPLReKey, byte bPLReKeyLen )
        {
            byte[] aPlCodeTmp = new byte[wPLCodeLen];
            aPLCode.CopyTo ( aPlCodeTmp, 0 );

            return phhalHw_SamAV3_Cmd_SAM_PLUpload ( ref m_DataParamsInt[0], bIsFirstFrame, bIsFinalFrame, wUploadCtr, bKeyNo, bKeyVer,
                aPlCodeTmp, wPLCodeLen, aPLReKey, bPLReKeyLen );
        }
        #endregion

        #region Reader Chip
        /// <summary>
        /// Read the content of one or more register(s) of the connected reader chip.
        /// </summary>
        ///
        /// <param name="bRegAddr">Address for the registers to be read.</param>
        /// <param name="aValue">Register(s) content in the same order as the command data field address(es).</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_RC_ReadRegister ( byte[] bRegAddr, out byte[] aValue )
        {
            aValue = new byte[( ( bRegAddr == null ) ? 0 : bRegAddr.Length )];
            return phhalHw_SamAV3_Cmd_RC_ReadRegister ( ref m_DataParamsInt[0], bRegAddr, ( byte ) ( ( bRegAddr == null ) ? 0 : bRegAddr.Length ), aValue );
        }

        /// <summary>
        /// Write the content of one or more register(s) of the connected reader chip.
        /// </summary>
        ///
        /// <param name="aData">The registers to tbe updated. This buffer should contain register address followed by value.
        ///						The format should be as mentioned below.
        ///						RegAdd + Value + RegAdd + Value + ... + RegAdd + Value</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_RC_WriteRegister ( byte[] aData )
        {
            return phhalHw_SamAV3_Cmd_RC_WriteRegister ( ref m_DataParamsInt[0], aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Turn the radio frequency field On or Off
        /// </summary>
        ///
        /// <param name="dwTime">Shut-down time for the RF field, zero turns the field off.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_RC_RFControl ( int dwTime )
        {
            return phhalHw_SamAV3_Cmd_RC_RFControl ( ref m_DataParamsInt[0], ( ushort ) dwTime );
        }

        /// <summary>
        /// Initializes the reader chip.
        /// </summary>
        ///
        /// <param name="bLoadReg">The Load register information.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_RC_Init ( byte bLoadReg )
        {
            return phhalHw_SamAV3_Cmd_RC_Init ( ref m_DataParamsInt[0], bLoadReg );
        }

        /// <summary>
        /// Stores a customer defined register value set.
        /// </summary>
        ///
        /// <param name="bStoreReg">Number of register value set to be used for storing the values.</param>
        /// <param name="aData">List of Register descriptions. This buffer should be as meniotned below.
        ///						If TD1AR1070: SpeedID + ItemCount + RegAddr + RegValue
        ///						If TD1AR2060: RegAddr + RegValue</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_RC_LoadRegisterValueSet ( byte bStoreReg, byte[] aData )
        {
            return phhalHw_SamAV3_Cmd_RC_LoadRegisterValueSet ( ref m_DataParamsInt[0], bStoreReg, aData, ( byte ) ( ( aData == null ) ? 0 : aData.Length ) );
        }
        #endregion Reader Chip

        #region ISO14443-3
        /// <summary>
        /// Perform a request or wake-up command (type A).
        /// </summary>
        ///
        /// <param name="bCmdCode">Command code.
        ///							NxpRdLibNet.Hal.SamAV3.CmdCode.REQA
        ///							NxpRdLibNet.Hal.SamAV3.CmdCode.WUPA</param>
        /// <param name="aAtqa">AtqA returned by the card. The buffer has to be 2 bytes long.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_RequestA_Wakeup ( byte bCmdCode, out byte[] aAtqa )
        {
            aAtqa = new byte[2];
            return phhalHw_SamAV3_Cmd_X_ISO14443_3_RequestA_Wakeup ( ref m_DataParamsInt[0], bCmdCode, aAtqa );
        }

        /// <summary>
        /// Perform a bit-wise anticollision and select. (type A).
        /// </summary>
        ///
        /// <param name="aSelCodes">Buffer containing the SEL sequence bytes (93h, 95h and 97h).</param>
        /// <param name="bSak">Pointer to the 1 byte Select Acknowledge reveived from card.</param>
        /// <param name="aUid">Buffer containing the received UID. This buffer has to be 10 bytes long.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_AnticollisionSelect ( byte[] aSelCodes, out byte bSak, out byte[] aUid )
        {
            Status_t oStatus;
            aUid = new byte[10];
            byte bUidLen = 0;
            bSak = 0;

            oStatus = phhalHw_SamAV3_Cmd_X_ISO14443_3_AnticollisionSelect ( ref m_DataParamsInt[0], aSelCodes,
                ( byte ) ( ( aSelCodes == null ) ? 0 : aSelCodes.Length ), ref bSak, aUid, ref bUidLen );

            /* Resize pUid buffer to actual bVersion size */
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error.ISO_UID_INCOMPLETE ) )
                Array.Resize<byte> ( ref aUid, ( int ) bUidLen );

            return oStatus;
        }

        /// <summary>
        /// Perform one or several request - anticollision - select sequences (type A).
        /// </summary>
        ///
        /// <param name="bOption">Option parameter.
        ///							NxpRdLibNet.Hal.SamAV3.ActivateIdle.DEFAULT
        ///							NxpRdLibNet.Hal.SamAV3.ActivateIdle.APPLY_ATQA
        ///							NxpRdLibNet.Hal.SamAV3.ActivateIdle.APPLY_SAK
        ///							NxpRdLibNet.Hal.SamAV3.ActivateIdle.APPLY_TIME</param>
        /// <param name="bNumCards">Holds the number of cards to activate.</param>
        /// <param name="dwTime">Time to wait for a card.</param>
        /// <param name="aAtqaIn">Buffer containing the AtqA filter. This buffer has to be 4 bytes long.</param>
        /// <param name="aSakIn">Buffer containing the Sak filter. This buffer has to be 2 bytes long.</param>
        /// <param name="stCardInfo">Pointer to the buffer containing the received data.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_ActivateIdle ( byte bOption, byte bNumCards, int dwTime, byte[] aAtqaIn, byte[] aSakIn,
            out ActivateIdleParams[] stCardInfo )
        {
            Status_t oStatus;
            IntPtr pRxBufferInt = IntPtr.Zero;
            ushort pRxLen = 0;
            byte[] aResponse;

            oStatus = phhalHw_SamAV3_Cmd_X_ISO14443_3_ActivateIdle ( ref m_DataParamsInt[0], bOption, bNumCards, ( ushort ) dwTime, aAtqaIn,
                aSakIn, ref pRxBufferInt, ref pRxLen );

            if ( pRxLen > 0 && oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                aResponse = new byte[pRxLen];
                Marshal.Copy ( pRxBufferInt, aResponse, 0, pRxLen );

                if ( bNumCards > 1 )
                    stCardInfo = new ActivateIdleParams[aResponse[0]];
                else
                    stCardInfo = new ActivateIdleParams[bNumCards];

                if ( bNumCards == 1 )
                {
                    stCardInfo[0].Atqa = new byte[2];
                    Array.Copy ( aResponse, stCardInfo[0].Atqa, 2 );
                    stCardInfo[0].Sak = aResponse[2];
                    stCardInfo[0].Uid = new byte[aResponse[3]];
                    Array.Copy ( aResponse, 4, stCardInfo[0].Uid, 0, aResponse[3] );
                }
                else
                {
                    int dwInfoOffset = 1;
                    for ( int i = 0; i < aResponse[0]; i++ )
                    {
                        stCardInfo[i].Atqa = new byte[2];
                        Array.Copy ( aResponse, dwInfoOffset, stCardInfo[i].Atqa, 0, 2 );
                        stCardInfo[i].Sak = aResponse[dwInfoOffset + 2];
                        stCardInfo[i].Uid = new byte[aResponse[dwInfoOffset + 3]];
                        Array.Copy ( aResponse, dwInfoOffset + 4, stCardInfo[i].Uid, 0, aResponse[dwInfoOffset + 3] );
                        dwInfoOffset = dwInfoOffset + 4 + stCardInfo[i].Uid.Length;
                    }
                }
            }
            else
                stCardInfo = new ActivateIdleParams[0];

            return oStatus;
        }

        /// <summary>
        /// Perform a reactivate and select for a card in halt state (type A).
        /// </summary>
        ///
        /// <param name="aUid">UID of the card to reactivate.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_ActivateWakeUp ( byte[] aUid )
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_3_ActivateWakeUp ( ref m_DataParamsInt[0], aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ) );
        }

        /// <summary>
        /// Perform a halt command (type A).
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_HaltA ()
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_3_HaltA ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Perform a transparent exchange command (type A).
        /// </summary>
        ///
        /// <param name="aTxBuffer">Buffer containing the data to be sent.</param>
        /// <param name="bTxBitLen">Number of bytes to be sent to the card.</param>
        /// <param name="aRxBuffer">Pointer to the buffer containing the received data.</param>
        /// <param name="bRxBitLen">Amount of valid bits in the last byte in case of an incomplete byte.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_3_TransparentExchange ( byte[] aTxBuffer, byte bTxBitLen, out byte[] aRxBuffer, out byte bRxBitLen )
        {
            Status_t oStatus;
            ushort wRxLen = 0;
            IntPtr pRxBuf = IntPtr.Zero;
            bRxBitLen = 0;

            oStatus = phhalHw_SamAV3_Cmd_X_ISO14443_3_TransparentExchange ( ref m_DataParamsInt[0], aTxBuffer, ( byte ) ( ( aTxBuffer == null ) ? 0 : aTxBuffer.Length ),
                bTxBitLen, ref pRxBuf, ref wRxLen, ref bRxBitLen );

            aRxBuffer = MarshalCopy ( oStatus, pRxBuf, wRxLen );
            return oStatus;
        }
        #endregion ISO14443-3

        #region ISO14443-4
        /// <summary>
        /// Perform a combined RATS and PPS to prepare a card for T=CL data exchange.
        /// </summary>
        ///
        /// <param name="bCidIn">CID to be sent.</param>
        /// <param name="bDsiIn">DSI to be sent.</param>
        /// <param name="bDriIn">DRI to be sent.</param>
        /// <param name="bCidOut">CID used (1 byte).</param>
        /// <param name="bDsiOut">DSI used (1 byte).</param>
        /// <param name="bDriOut">DRI used (1 byte).</param>
        /// <param name="aAts">Pointer to a buffer containing the received ATS. The length of the
        ///					   ATS can be found on the first position.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_RATS_PPS ( byte bCidIn, byte bDsiIn, byte bDriIn, out byte bCidOut, out byte bDsiOut, out byte bDriOut,
            out byte[] aAts )
        {
            aAts = new byte[256];
            bCidOut = 0;
            bDsiOut = 0;
            bDriOut = 0;

            Status_t oStatus = phhalHw_SamAV3_Cmd_X_ISO14443_4_RATS_PPS ( ref m_DataParamsInt[0], bCidIn, bDsiIn, bDriIn, ref bCidOut, ref bDsiOut,
                ref bDriOut, aAts );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
                Array.Resize ( ref aAts, aAts[0] );
            else
                aAts = null;

            return oStatus;
        }

        /// <summary>
        /// Perform a init of ISO-14443-4 layer (init T=CL protocol).
        /// </summary>
        ///
        /// <param name="bCid">CID to apply.</param>
        /// <param name="bDri">Bit rate PCD -> PICC.
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_106
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_212
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_424
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_848</param>
        /// <param name="bDsi">Bit rate PICC -> PCD.
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_106
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_212
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_424
        ///						NxpRdLibNet.Hal.RfDataRate.DATARATE_848</param>
        /// <param name="bFwi">Frame waiting time indicator.</param>
        /// <param name="bFsci">Frame size card indicator.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_Init ( byte bCid, byte bDri, byte bDsi, byte bFwi, byte bFsci )
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_4_Init ( ref m_DataParamsInt[0], bCid, bDri, bDsi, bFwi, bFsci );
        }

        /// <summary>
        /// Perform an exchange bytes according ISO14443-4 T=CL protocol.
        /// </summary>
        ///
        /// <param name="wOption">Bufferring Option.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT   : To send and receive complete data
        ///							NxpRdLibNet.ExchangeOptions.TXCHAINING: To excahnge intermediate data</param>
        /// <param name="aAppDataIn">Buffer containing application data to sent.</param>
        /// <param name="aAppDataOut">Pointer to buffer containing the data returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_Exchange ( ushort wOption, byte[] aAppDataIn, out byte[] aAppDataOut )
        {
            ushort wAppDataOutLen = 0;
            IntPtr ppAppDataOut = IntPtr.Zero;

            Status_t oStatus = phhalHw_SamAV3_Cmd_X_ISO14443_4_Exchange ( ref m_DataParamsInt[0], wOption, aAppDataIn, (byte) ( ( aAppDataIn == null ) ? 0 : aAppDataIn.Length ),
                ref ppAppDataOut, ref wAppDataOutLen );

            aAppDataOut = MarshalCopy ( oStatus, ppAppDataOut, wAppDataOutLen );
            return oStatus;
        }

        /// <summary>
        /// Check if an activated card is still in the field.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_PresenceCheck ()
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_4_PresenceCheck ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Perform a Deselect command.
        /// </summary>
        ///
        /// <param name="bFreeCid">Bitmask for deselect option.
        ///							NxpRdLibNet.Hal.SamAV3.Deselect.DO_NOT_FREE_CID
        ///							NxpRdLibNet.Hal.SamAV3.Deselect.FORCE_FREE_CID</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_Deselect ( byte bFreeCid )
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_4_Deselect ( ref m_DataParamsInt[0], bFreeCid );
        }

        /// <summary>
        /// Free one or more currently assigned CIDs.
        /// </summary>
        ///
        /// <param name="aCid">Buffer containing all CIDs to be freed.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Cmd_X_ISO14443_4_FreeCid ( byte[] aCid )
        {
            return phhalHw_SamAV3_Cmd_X_ISO14443_4_FreeCid ( ref m_DataParamsInt[0], aCid,
                ( byte ) ( ( aCid == null ) ? 0 : aCid.Length ) );
        }
        #endregion ISO14443-4

        #region Utilities
        /// <summary>
        /// Detects the mode of the SAM and store in the internal buffer. This function should be called prior to all other functions even before
        /// Host Authentication. It will be either SAM AV2 or SAM AV3 mode.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DetectMode ()
        {
            return phhalHw_SamAV3_DetectMode ( ref m_DataParamsInt[0] );
        }

        /// <summary>
        /// Exchange commands with the SAM. This function is used to excahnge raw ISO7816 commands.
        /// </summary>
        ///
        /// <param name="wOption">One of the bufferring bits.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT
        ///							NxpRdLibNet.ExchangeOptions.FIRST
        ///							NxpRdLibNet.ExchangeOptions.CONT
        ///							NxpRdLibNet.ExchangeOptions.LAST</param>
        /// <param name="aTxBuffer">The ISO7816 command information to be exchanged.</param>
        /// <param name="aRxBuffer">The response received from Sam. This will contain only the payload if available.</param>
        /// <returns></returns>
        public Status_t Cmd_SAM_AV3_7816Exchange ( int wOption, byte[] aTxBuffer, out byte[] aRxBuffer )
        {
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            Status_t oStatus = phhalHw_SamAV3_Cmd_7816Exchange ( m_pDataParams, (ushort) wOption, aTxBuffer, (ushort) ( ( aTxBuffer == null ) ? 0 : aTxBuffer.Length ),
                ref pRxBufferInt, ref wRxLength );

            if ( ( pRxBufferInt != IntPtr.Zero ) && ( wRxLength > 0 ) )
            {
                aRxBuffer = new byte[wRxLength];
                Marshal.Copy ( pRxBufferInt, aRxBuffer, 0, wRxLength );
            }
            else
            {
                aRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Set the configuration.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set.</param>
        /// <param name="wValue">The value to be set for the mentioned confiuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( Config wConfig, int wValue )
        {
            return SetConfig ( ( Hal.Config ) wConfig, wValue );
        }

        /// <summary>
        /// Get the configuration.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to get.</param>
        /// <param name="wValue">The value returned for the mentioned confiuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( Config wConfig, out int wValue )
        {
            return GetConfig ( ( Hal.Config ) wConfig, out wValue );
        }
        #endregion Utilities
        #endregion Wrapper Functions

        #region Support classes
        #region Key Management
        public enum DesOption : byte
        {
            DESFIRE4 = 0x00,
            ISO_CRC16,
            ISO_CRC32
        }

        #region KUCEntry
        public class KucEntry
        {
            public uint Limit;
            public byte KeyNoCKUC;
            public byte KeyVCKUC;
            public uint Value;

            public KucEntry ()
            {
                Limit = 0xFFFFFFFF;
                KeyNoCKUC = 0xFE;
                KeyVCKUC = 0x00;
            }

            public KucEntry ( byte[] aKuc )
            {
                if ( aKuc == null )
                    return;

                if ( aKuc.Length < 10 )
                    return;

                Limit = ( uint ) ( ( aKuc[3] << 24 ) | ( aKuc[2] << 16 ) | ( aKuc[1] << 8 ) | aKuc[0] );
                KeyNoCKUC = aKuc[4];
                KeyVCKUC = aKuc[5];
                Value = ( uint ) ( ( aKuc[9] << 24 ) | ( aKuc[8] << 16 ) | ( aKuc[7] << 8 ) | aKuc[6] );
            }

            public byte[] ToByteArray ()
            {
                byte[] aKUCEntryBuffer = new byte[6];
                aKUCEntryBuffer[0] = ( byte ) ( Limit );
                aKUCEntryBuffer[1] = ( byte ) ( Limit >> 8 );
                aKUCEntryBuffer[2] = ( byte ) ( Limit >> 16 );
                aKUCEntryBuffer[3] = ( byte ) ( Limit >> 24 );
                aKUCEntryBuffer[4] = KeyNoCKUC;
                aKUCEntryBuffer[5] = KeyVCKUC;

                return aKUCEntryBuffer;
            }
        }
        #endregion KUCEntry

        #region Cmd.SAM_ChangeKeyEntry
        #region KeyEntry
        public class KeyEntry
        {
            #region KeyTypes
            public enum KeyType : int
            {
                DESFIRE = 0x0000,
                ISO_10116 = 0x0008,
                MIFARE = 0x0010,
                THREE_KEY_TRIPLE_DES = 0x0018,
                AES_128 = 0x0020,
                AES_192 = 0x0028,
                DES_ISO_CRC32 = 0x0030,
                AES_256 = 0x0038,
                AES128_LRP = 0x0040
            }
            #endregion KeyTypes

            #region KeyClass
            public enum KeyClass : int
            {
                HOST = 0x00,
                PICC = 0x01,
                OFFLINE_CHANGE = 0x02,
                OFFLINE_CRYPTO = 0x04,
                OFFLINE_UPLOAD = 0x05,
                OFFLINE_PERSO = 0x06,
            }
            #endregion KeyClass

            #region SET Configuration
            public class Set
            {
                private int dwSet;

                public static implicit operator Set ( int dwSet )
                {
                    Set MyInstance = new Set ();
                    MyInstance.dwSet = dwSet;
                    return MyInstance;
                }

                public static implicit operator int ( Set oSet )
                {
                    return oSet.dwSet;
                }

                public bool AllowDumpSessionKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0001;
                        else dwSet &= ~0x0001;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0001 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool KeepIV
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0004;
                        else dwSet &= ~0x0004;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0004 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool PLKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0080;
                        else dwSet &= ~0x0080;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0080 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public KeyType KeyType
                {
                    set
                    {
                        dwSet &= 0xFF87;
                        dwSet |= ( int ) value;
                    }
                    get
                    {
                        return ( KeyType ) ( dwSet & 0x0078 );
                    }
                }

                public bool AuthKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0100;
                        else dwSet &= ~0x0100;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0100 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableKeyEntry
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0200;
                        else dwSet &= ~0x0200;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0200 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool LockKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0400;
                        else dwSet &= ~0x0400;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0400 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableChangeKeyPICC
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0800;
                        else dwSet &= ~0x0800;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x0800 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableDecryption
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x1000;
                        else dwSet &= ~( int ) 0x1000;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x1000 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableEncryption
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x2000;
                        else dwSet &= ~( int ) 0x2000;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x2000 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableVerifyMac
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x4000;
                        else dwSet &= ~( int ) 0x4000;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x4000 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableGenerateMac
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x8000;
                        else dwSet &= ~( int ) 0x8000;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x8000 ) != 0x0000 ) return true;
                        else return false;
                    }
                }
            }
            #endregion SET Configuration

            #region ExtSET Configuration
            public class ExtSet
            {
                int dwExtSet;

                public static implicit operator ExtSet ( int dwExtSet )
                {
                    ExtSet MyInstance = new ExtSet ();
                    MyInstance.dwExtSet = dwExtSet;
                    return MyInstance;
                }

                public static implicit operator int ( ExtSet oExtSet )
                {
                    return oExtSet.dwExtSet;
                }

                public KeyClass KeyClass
                {
                    set
                    {
                        dwExtSet &= 0x00F8;
                        dwExtSet |= ( int ) value;
                    }
                    get
                    {
                        return ( KeyClass ) ( dwExtSet & 0x0007 );
                    }
                }

                public bool AllowDumpSecret
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0008;
                        else dwExtSet &= ~0x0008;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0008 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool DiversifiedRestriction
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0010;
                        else dwExtSet &= ~0x0010;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0010 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool PersonalizeSAM
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0020;
                        else dwExtSet &= ~0x0020;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0020 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool ForceKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0100;
                        else dwExtSet &= ~0x0100;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0100 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool ForceKeyChangeInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0200;
                        else dwExtSet &= ~0x0200;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0200 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool ForceSessionKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0400;
                        else dwExtSet &= ~0x0400;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0400 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool AllowDumpSecretKeyInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x0800;
                        else dwExtSet &= ~0x0800;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x0800 ) != 0x00 ) return true;
                        else return false;
                    }
                }

                public bool AllowDumpSessionKeyInternalHost
                {
                    set
                    {
                        if ( value ) dwExtSet |= 0x1000;
                        else dwExtSet &= ~0x1000;
                    }
                    get
                    {
                        if ( ( dwExtSet & 0x1000 ) != 0x00 ) return true;
                        else return false;
                    }
                }
            }
            #endregion ExtSET Configuration

            #region Properties
            public byte[] KeyA;
            public byte[] KeyB;
            public byte[] KeyC;
            public byte[] DF_AID;
            public byte DF_KeyNo;
            public byte KeyNoCEK;
            public byte KeyVCEK;
            public byte RefNoKUC;
            public Set SET;
            public byte VersionA;
            public byte VersionB;
            public byte VersionC;
            public ExtSet ExtSET;
            public byte KeyNoAEK;
            public byte KeyVAEK;
            #endregion Properties

            public KeyEntry ()
            {
                SET = new Set ();
                ExtSET = new ExtSet ();

                DF_AID = new byte[3];

                RefNoKUC = 0xFF;
                KeyNoCEK = 0xFE;
                KeyNoAEK = 0xFE;
            }

            public KeyEntry ( KeyType eKeyType, KeyClass eKeyClass, bool boIsRamKey = false )
            {
                SET = new Set ();
                SET.KeyType = eKeyType;

                ExtSET = new ExtSet ();
                ExtSET.KeyClass = eKeyClass;

                if ( eKeyClass.Equals ( KeyClass.HOST ) )
                    SET.AuthKey = true;

                VersionA = 0x00;
                VersionB = ( byte ) ( boIsRamKey ? 0x00 : 0x01 );
                VersionC = ( byte ) ( boIsRamKey ? 0x00 : 0x02 );

                if ( eKeyType.Equals ( KeyType.AES_256 ) )
                {
                    KeyA = new byte[48];
                    VersionB = 0x00;
                    VersionC = 0x00;
                }
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.THREE_KEY_TRIPLE_DES ) )
                {
                    KeyA = new byte[24];
                    KeyB = new byte[24];
                    VersionC = 0x00;
                }
                else
                {
                    KeyA = new byte[16];
                    KeyB = new byte[16];
                    KeyC = new byte[16];
                }

                DF_AID = new byte[3];

                RefNoKUC = 0xFF;
                KeyNoCEK = 0xFE;
                KeyNoAEK = 0xFE;
            }

            public KeyEntry ( byte[] aKeyInfoBuffer, bool boIsAV3, bool boIsRamKey = false )
            {
                byte bOffset = 0;

                SET = new Set ();
                bOffset = ( byte ) ( boIsAV3 ? ( aKeyInfoBuffer.Length - 6 ) : ( aKeyInfoBuffer.Length - 3 ) );
                SET = ( int ) ( ( aKeyInfoBuffer[bOffset + 1] << 8 ) | aKeyInfoBuffer[bOffset] );

                ExtSET = new ExtSet ();
                bOffset = ( byte ) ( boIsAV3 ? ( aKeyInfoBuffer.Length - 4 ) : ( aKeyInfoBuffer.Length - 1 ) );
                ExtSET = ( int ) ( ( boIsAV3 ? ( aKeyInfoBuffer[bOffset + 1] << 8 ) : 0 ) | aKeyInfoBuffer[bOffset] );

                DF_AID = new byte[3];

                if ( SET.KeyType.Equals ( KeyType.AES_256 ) )
                    KeyA = new byte[48];
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.THREE_KEY_TRIPLE_DES ) )
                {
                    KeyA = new byte[24];
                    KeyB = new byte[24];
                }
                else
                {
                    KeyA = new byte[16];
                    KeyB = new byte[16];
                    KeyC = new byte[16];
                }

                if ( !boIsRamKey )
                {
                    bOffset = 0;
                    VersionA = aKeyInfoBuffer[bOffset++];
                    VersionB = aKeyInfoBuffer[bOffset++];

                    if ( !SET.KeyType.Equals ( KeyType.AES_256 ) && !SET.KeyType.Equals ( KeyType.AES_192 ) &&
                        !SET.KeyType.Equals ( KeyType.THREE_KEY_TRIPLE_DES ) )
                        VersionC = aKeyInfoBuffer[bOffset++];

                    Array.Copy ( aKeyInfoBuffer, bOffset, DF_AID, 0, 3 );
                    bOffset += 3;

                    DF_KeyNo = aKeyInfoBuffer[bOffset++];
                }

                if ( boIsRamKey )
                    bOffset = 0;

                KeyNoCEK = aKeyInfoBuffer[bOffset++];
                KeyVCEK = aKeyInfoBuffer[bOffset++];
                RefNoKUC = aKeyInfoBuffer[bOffset++];

                if ( boIsAV3 )
                {
                    bOffset += 4;
                    KeyNoAEK = aKeyInfoBuffer[bOffset++];
                    KeyVAEK = aKeyInfoBuffer[bOffset++];
                }
            }

            public byte[] ToByteArray ( bool boIsAV3 )
            {
                byte bOffset = 0;
                byte[] aKeyInfoBuffer = new byte[boIsAV3 ? 64 : 61];

                if ( SET.KeyType.Equals ( KeyType.AES_256 ) )
                    System.Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 48 );
                else if ( SET.KeyType.Equals ( KeyType.AES_192 ) || SET.KeyType.Equals ( KeyType.THREE_KEY_TRIPLE_DES ) )
                {
                    System.Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 24 );
                    System.Array.Copy ( KeyB, 0, aKeyInfoBuffer, 24, 24 );
                }
                else
                {
                    System.Array.Copy ( KeyA, 0, aKeyInfoBuffer, 0, 16 );
                    System.Array.Copy ( KeyB, 0, aKeyInfoBuffer, 16, 16 );
                    System.Array.Copy ( KeyC, 0, aKeyInfoBuffer, 32, 16 );
                }
                bOffset = 48;

                System.Array.Copy ( DF_AID, 0, aKeyInfoBuffer, bOffset, 3 );
                bOffset += 3;

                aKeyInfoBuffer[bOffset++] = DF_KeyNo;
                aKeyInfoBuffer[bOffset++] = KeyNoCEK;
                aKeyInfoBuffer[bOffset++] = KeyVCEK;
                aKeyInfoBuffer[bOffset++] = RefNoKUC;
                aKeyInfoBuffer[bOffset++] = ( byte ) ( SET );
                aKeyInfoBuffer[bOffset++] = ( byte ) ( SET >> 8 );
                aKeyInfoBuffer[bOffset++] = VersionA;
                aKeyInfoBuffer[bOffset++] = VersionB;
                aKeyInfoBuffer[bOffset++] = VersionC;

                aKeyInfoBuffer[bOffset++] = ( byte ) ExtSET;
                if ( boIsAV3 )
                {
                    aKeyInfoBuffer[bOffset++] = ( byte ) ( ExtSET >> 8 );
                    aKeyInfoBuffer[bOffset++] = KeyNoAEK;
                    aKeyInfoBuffer[bOffset++] = KeyVAEK;
                }

                return aKeyInfoBuffer;
            }
        }
        #endregion KeyEntry

        #region Program Mask bits
        public class ChangeKeyEntry_ProMas
        {
            private int dwProMask;

            public static implicit operator ChangeKeyEntry_ProMas ( byte promas )
            {
                ChangeKeyEntry_ProMas MyInstance = new ChangeKeyEntry_ProMas ();
                MyInstance.dwProMask = promas;
                return MyInstance;
            }

            public static implicit operator byte ( ChangeKeyEntry_ProMas promas )
            {
                return ( byte ) promas.dwProMask;
            }

            public bool UpdateKeyVa
            {
                set
                {
                    if ( value ) dwProMask |= 0x80;
                    else dwProMask &= ~0x80;
                }
                get
                {
                    if ( ( dwProMask & 0x80 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeyVb
            {
                set
                {
                    if ( value ) dwProMask |= 0x40;
                    else dwProMask &= ~0x40;
                }
                get
                {
                    if ( ( dwProMask & 0x40 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeyVc
            {
                set
                {
                    if ( value ) dwProMask |= 0x20;
                    else dwProMask &= ~0x20;
                }
                get
                {
                    if ( ( dwProMask & 0x20 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateDFAid_DFKeyNo
            {
                set
                {
                    if ( value ) dwProMask |= 0x10;
                    else dwProMask &= ~0x10;
                }
                get
                {
                    if ( ( dwProMask & 0x10 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeyNoCEK_KeyVCEK
            {
                set
                {
                    if ( value ) dwProMask |= 0x08;
                    else dwProMask &= ~0x08;
                }
                get
                {
                    if ( ( dwProMask & 0x08 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateRefNoKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x04;
                    else dwProMask &= ~0x04;
                }
                get
                {
                    if ( ( dwProMask & 0x04 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateSET_ExtSET
            {
                set
                {
                    if ( value ) dwProMask |= 0x02;
                    else dwProMask &= ~0x02;
                }
                get
                {
                    if ( ( dwProMask & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool KeyVersionsSentSeparately
            {
                set
                {
                    if ( value ) dwProMask |= 0x01;
                    else dwProMask &= ~0x01;
                }
                get
                {
                    if ( ( dwProMask & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion Program Mask bits

        #region Option
        public class ChangeKeyEntry_Option
        {
            private int dwOption;

            public static implicit operator ChangeKeyEntry_Option ( byte bOption )
            {
                ChangeKeyEntry_Option MyInstance = new ChangeKeyEntry_Option ();
                MyInstance.dwOption = bOption;
                return MyInstance;
            }

            public static implicit operator byte ( ChangeKeyEntry_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool SentPlain
            {
                set
                {
                    if ( value ) dwOption |= 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion Option
        #endregion Cmd.SAM_ChangeKeyEntry

        #region Cmd.SAM_ChangeKUCEntry
        #region Program Mask bits
        public class ChangeKUCEntry_ProMas
        {
            private int dwProMask;

            public static implicit operator ChangeKUCEntry_ProMas ( byte bProMask )
            {
                ChangeKUCEntry_ProMas MyInstance = new ChangeKUCEntry_ProMas ();
                MyInstance.dwProMask = bProMask;
                return MyInstance;
            }

            public static implicit operator byte ( ChangeKUCEntry_ProMas oProMask )
            {
                return ( byte ) oProMask.dwProMask;
            }

            public bool UpdateLimit
            {
                set
                {
                    if ( value ) dwProMask |= 0x80;
                    else dwProMask &= ~0x80;
                }
                get
                {
                    if ( ( dwProMask & 0x80 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeyNoCKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x40;
                    else dwProMask &= ~0x40;
                }
                get
                {
                    if ( ( dwProMask & 0x40 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeyVCKUC
            {
                set
                {
                    if ( value ) dwProMask |= 0x20;
                    else dwProMask &= ~0x20;
                }
                get
                {
                    if ( ( dwProMask & 0x20 ) != 0x00 ) return true;
                    else return false;
                }
            }

        }
        #endregion Program Mask bits

        #region Option
        public class ChangeKUCEntry_Option
        {
            private int dwOption;

            public static implicit operator ChangeKUCEntry_Option ( byte bOption )
            {
                ChangeKUCEntry_Option MyInstance = new ChangeKUCEntry_Option ();
                MyInstance.dwOption = bOption;
                return MyInstance;
            }

            public static implicit operator byte ( ChangeKUCEntry_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool SentPlain
            {
                set
                {
                    if ( value ) dwOption |= 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion Option
        #endregion Cmd.SAM_ChangeKUCEntry

        #region Cmd.SAM_DumpSessionKey
        public class DumpSessionKey_Crypto
        {
            private int wCrypto;

            public static implicit operator DumpSessionKey_Crypto ( byte bCrypto )
            {
                DumpSessionKey_Crypto MyInstance = new DumpSessionKey_Crypto ();
                MyInstance.wCrypto = bCrypto;
                return MyInstance;
            }

            public static implicit operator byte ( DumpSessionKey_Crypto oCrypto )
            {
                return ( byte ) oCrypto.wCrypto;
            }

            public bool Encrypt
            {
                set
                {
                    if ( value ) wCrypto |= 0x01;
                    else wCrypto &= ~0x01;
                }
                get
                {
                    if ( ( wCrypto & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public byte LogicalChannelNo
            {
                set
                {
                    wCrypto &= 0xF9;
                    wCrypto |= ( byte ) ( ( value & 0x03 ) << 1 );
                }
                get
                {
                    return ( byte ) ( ( wCrypto & 0x06 ) >> 1 );
                }
            }
        }
        #endregion Cmd.SAM_DumpSessionKey

        #region Cmd.SAM_DumpSecretKey
        public class DumpSecretKey_Crypto
        {
            private int dwCrypto;

            public static implicit operator DumpSecretKey_Crypto ( byte bCrypto )
            {
                DumpSecretKey_Crypto MyInstance = new DumpSecretKey_Crypto ();
                MyInstance.dwCrypto = bCrypto;
                return MyInstance;
            }

            public static implicit operator byte ( DumpSecretKey_Crypto oCrypto )
            {
                return ( byte ) oCrypto.dwCrypto;
            }

            public bool EncryptedDump
            {
                set
                {
                    if ( value ) dwCrypto |= 0x01;
                    else dwCrypto &= ~0x01;
                }
                get
                {
                    if ( ( dwCrypto & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool DiversifiedDump
            {
                set
                {
                    if ( value ) dwCrypto |= 0x02;
                    else dwCrypto &= ~0x02;
                }
                get
                {
                    if ( ( dwCrypto & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion Cmd.SAM_DumpSecretKey
        #endregion Key Management

        #region Data Processing
        public class VerifyGenerateMAC_Num
        {
            private int Num;

            public static implicit operator VerifyGenerateMAC_Num ( byte Num )
            {
                VerifyGenerateMAC_Num MyInstance = new VerifyGenerateMAC_Num ();
                MyInstance.Num = Num;
                return MyInstance;
            }

            public static implicit operator byte ( VerifyGenerateMAC_Num Num )
            {
                return ( byte ) Num.Num;
            }

            public bool useMFPTruncation
            {
                set
                {
                    if ( value )
                    {
                        Num = 0x80;
                    }
                    else
                    {
                        Num &= ~0x80;
                    }
                }
                get
                {
                    if ( ( Num & 0x80 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public byte numMACBytes
            {
                set
                {
                    Num &= 0xE0;
                    Num |= ( byte ) ( value & 0x1F );
                }
                get
                {
                    return ( byte ) ( ( Num & 0x1F ) );
                }
            }
        }

        public class DecipherData_Option
        {
            private int Option;
            private bool first, cont, last;

            public static implicit operator DecipherData_Option ( int Option )
            {
                DecipherData_Option MyInstance = new DecipherData_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int ( DecipherData_Option Option )
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if ( Option.first )
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if ( Option.cont )
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if ( Option.last )
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setTXChaining
            {
                set
                {
                    if ( value )
                        Option |= ExchangeOptions.TXCHAINING;
                    else
                        Option &= ~ExchangeOptions.TXCHAINING;
                }
                get
                {
                    if ( ( Option & ExchangeOptions.TXCHAINING ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool setRXChaining
            {
                set
                {
                    if ( value )
                        Option |= ExchangeOptions.RXCHAINING;
                    else
                        Option &= ~ExchangeOptions.RXCHAINING;
                }
                get
                {
                    if ( ( Option & ExchangeOptions.RXCHAINING ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if ( value )
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if ( value )
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if ( value )
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }

            public bool withEncDataLength
            {
                set
                {
                    if ( value )
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ( ( Option & 0x04 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }
        #endregion Data Processing

        #region Public Key Infrastructure
        #region RSA
        #region KeyEntry
        public class PKI_RSA_KeyEntry
        {
            #region SET Configuration
            public class Set
            {
                private int dwSet;

                public static implicit operator Set ( ushort dwSet )
                {
                    Set MyInstance = new Set ();
                    MyInstance.dwSet = dwSet;
                    return MyInstance;
                }

                public static implicit operator ushort ( Set oSet )
                {
                    return ( ushort ) oSet.dwSet;
                }

                public bool IncludePrivateKey
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0001;
                        else dwSet &= ~0x0001;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0001 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool AllowPrivateKeyExport
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0002;
                        else dwSet &= ~0x0002;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0002 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableKeyEntry
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0004;
                        else dwSet &= ~0x0004;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0004 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableEncryptionHandling
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0008;
                        else dwSet &= ~0x0008;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0008 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool DisableSignatureHandling
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0010;
                        else dwSet &= ~0x0010;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0010 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool EnableUpdateKeyEntries
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0020;
                        else dwSet &= ~0x0020;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0020 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool PrivateKeyRepresentation
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0040;
                        else dwSet &= ~0x0040;
                    }
                    get
                    {
                        if ( ( dwSet & 0x0040 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool EnableEncipherKeyEntries
                {
                    set
                    {
                        if ( value ) dwSet |= 0x0080;
                        else dwSet &= ~0x0080;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x0080 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool ForceKeyUsageInternalHost
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x0100;
                        else dwSet &= ~( int ) 0x0100;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x0100 ) != 0x0000 ) return true;
                        else return false;
                    }
                }

                public bool ForceKeyChangeInternalHost
                {
                    set
                    {
                        if ( value ) dwSet |= ( int ) 0x0200;
                        else dwSet &= ~( int ) 0x0200;
                    }
                    get
                    {
                        if ( ( dwSet & ( int ) 0x0200 ) != 0x0000 ) return true;
                        else return false;
                    }
                }
            }
            #endregion SET Configuration

            #region Properties
            public ushort NLen;
            public ushort eLen;
            public ushort pLen;
            public ushort qLen;
            public byte[] N;
            public byte[] e;
            public byte[] p;
            public byte[] q;
            public byte[] dP;
            public byte[] dQ;
            public byte[] ipq;
            public byte KeyNoCEK;
            public byte KeyVCEK;
            public byte RefNoKUC;
            public Set SET;
            public byte KeyNoAEK;
            public byte KeyVAEK;
            #endregion Properties

            public PKI_RSA_KeyEntry ()
            {
                SET = new Set ();

                RefNoKUC = 0xFF;
                KeyNoCEK = 0x00;
                KeyNoAEK = 0x00;
            }

            public PKI_RSA_KeyEntry ( byte[] aKeyInfoBuffer, bool boAEK_Available = false )
            {
                ushort wOffset = 0;

                SET = new Set ();
                SET |= ( ushort ) ( aKeyInfoBuffer[wOffset++] << 8 );
                SET = aKeyInfoBuffer[wOffset++];

                KeyNoCEK = aKeyInfoBuffer[wOffset++];
                KeyVCEK = aKeyInfoBuffer[wOffset++];
                RefNoKUC = aKeyInfoBuffer[wOffset++];

                if ( boAEK_Available )
                {
                    KeyNoAEK = aKeyInfoBuffer[wOffset++];
                    KeyVAEK = aKeyInfoBuffer[wOffset++];
                }

                NLen = ( ushort ) ( aKeyInfoBuffer[wOffset++] << 8 );
                NLen |= aKeyInfoBuffer[wOffset++];

                eLen = ( ushort ) ( aKeyInfoBuffer[wOffset++] << 8 );
                eLen |= aKeyInfoBuffer[wOffset++];

                if ( SET.IncludePrivateKey )
                {
                    pLen = ( ushort ) ( aKeyInfoBuffer[wOffset++] << 8 );
                    pLen |= aKeyInfoBuffer[wOffset++];

                    qLen = ( ushort ) ( aKeyInfoBuffer[wOffset++] << 8 );
                    qLen |= aKeyInfoBuffer[wOffset++];
                }

                N = new byte[NLen];
                Array.Copy ( aKeyInfoBuffer, wOffset, N, 0, NLen );
                wOffset += NLen;

                e = new byte[eLen];
                Array.Copy ( aKeyInfoBuffer, wOffset, e, 0, eLen );
                wOffset += eLen;

                if ( SET.IncludePrivateKey )
                {
                    p = new byte[pLen];
                    Array.Copy ( aKeyInfoBuffer, wOffset, p, 0, pLen );
                    wOffset += pLen;

                    q = new byte[qLen];
                    Array.Copy ( aKeyInfoBuffer, wOffset, q, 0, qLen );
                    wOffset += qLen;

                    dP = new byte[pLen];
                    Array.Copy ( aKeyInfoBuffer, wOffset, dP, 0, pLen );
                    wOffset += pLen;

                    dQ = new byte[qLen];
                    Array.Copy ( aKeyInfoBuffer, wOffset, dQ, 0, qLen );
                    wOffset += qLen;

                    ipq = new byte[qLen];
                    Array.Copy ( aKeyInfoBuffer, wOffset, ipq, 0, qLen );

                    RemovePaddingBE ( ref dP );
                    RemovePaddingBE ( ref dQ );
                    RemovePaddingBE ( ref ipq );
                }
            }

            public byte[] ToByteArray ( bool boAEK_Available = false )
            {
                List<byte> aData = new List<byte> ();

                aData.Add ( ( byte ) ( SET >> 8 ) );
                aData.Add ( ( byte ) SET );

                aData.Add ( KeyNoCEK );
                aData.Add ( KeyVCEK );

                if ( boAEK_Available )
                {
                    aData.Add ( KeyNoAEK );
                    aData.Add ( KeyVAEK );
                }

                aData.Add ( ( byte ) ( NLen >> 8 ) );
                aData.Add ( ( byte ) NLen );

                aData.Add ( ( byte ) ( eLen >> 8 ) );
                aData.Add ( ( byte ) eLen );

                if ( SET.IncludePrivateKey )
                {
                    aData.Add ( ( byte ) ( pLen >> 8 ) );
                    aData.Add ( ( byte ) pLen );

                    aData.Add ( ( byte ) ( qLen >> 8 ) );
                    aData.Add ( ( byte ) qLen );
                }

                aData.AddRange ( N );
                aData.AddRange ( e );

                if ( SET.IncludePrivateKey )
                {
                    aData.AddRange ( p );
                    aData.AddRange ( q );
                    aData.AddRange ( Pad2LengthBE ( dP, pLen ) );
                    aData.AddRange ( Pad2LengthBE ( dQ, qLen ) );
                    aData.AddRange ( Pad2LengthBE ( ipq, qLen ) );
                }

                return ( aData.Count.Equals ( 0 ) ? null : aData.ToArray () );
            }

            private static void RemovePaddingBE ( ref byte[] aData )
            {
                for ( int iOffset = 0; iOffset < aData.Length; iOffset++ )
                {
                    if ( aData[iOffset] != 0 )
                    {
                        Array.Copy ( aData, iOffset, aData, 0, aData.Length - iOffset );
                        Array.Resize ( ref aData, aData.Length - iOffset );
                        break;
                    }
                }
            }

            private static byte[] Pad2LengthBE ( byte[] aData, int dwLength )
            {
                if ( aData.Length >= dwLength )
                    return aData;
                else
                {
                    byte[] buffer = new byte[dwLength];
                    Array.Copy ( aData, 0, buffer, dwLength - aData.Length, aData.Length );
                    return buffer;
                }
            }
        }
        #endregion KeyEntry

        public class PKI_RSA_GenerateKeyPair_Option
        {
            private int dwOption;

            public static implicit operator PKI_RSA_GenerateKeyPair_Option ( byte bOption )
            {
                PKI_RSA_GenerateKeyPair_Option oInstance = new PKI_RSA_GenerateKeyPair_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            public static implicit operator byte ( PKI_RSA_GenerateKeyPair_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool IncludeAEK
            {
                set
                {
                    if ( value ) dwOption = 0x02;
                    else dwOption &= ~0x02;
                }
                get
                {
                    if ( ( dwOption & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool ExponentProvidedByHost
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }

        public class PKI_RSA_ImportKey_Option
        {
            private int dwOption;

            public static implicit operator PKI_RSA_ImportKey_Option ( byte bOption )
            {
                PKI_RSA_ImportKey_Option oInstance = new PKI_RSA_ImportKey_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            public static implicit operator byte ( PKI_RSA_ImportKey_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool IncludeAEK
            {
                set
                {
                    if ( value ) dwOption = 0x02;
                    else dwOption &= ~0x02;
                }
                get
                {
                    if ( ( dwOption & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UpdateKeySettingsOnly
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion RSA

        #region ECC
        public class PKI_ECC_ImportKey_Option
        {
            private int dwOption;

            public static implicit operator PKI_ECC_ImportKey_Option ( byte bOption )
            {
                PKI_ECC_ImportKey_Option oInstance = new PKI_ECC_ImportKey_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            public static implicit operator byte ( PKI_ECC_ImportKey_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool UpdateKeySettingsOnly
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }

        public class PKI_ECC_ImportCurve_Option
        {
            private int dwOption;

            public static implicit operator PKI_ECC_ImportCurve_Option ( byte bOption )
            {
                PKI_ECC_ImportCurve_Option oInstance = new PKI_ECC_ImportCurve_Option ();
                oInstance.dwOption = bOption;
                return oInstance;
            }

            public static implicit operator byte ( PKI_ECC_ImportCurve_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool UpdateKeySettingsOnly
            {
                set
                {
                    if ( value ) dwOption = 0x01;
                    else dwOption &= ~0x01;
                }
                get
                {
                    if ( ( dwOption & 0x01 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion ECC
        #endregion Public Key Infrastructure

        #region Virtual Card
        #region Select
        public class VCA_Select_Option
        {
            private int dwOption;

            public static implicit operator VCA_Select_Option ( byte bOption )
            {
                VCA_Select_Option oMyInstance = new VCA_Select_Option ();
                oMyInstance.dwOption = bOption;
                return oMyInstance;
            }

            public static implicit operator byte ( VCA_Select_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool UseDivInput_ENCKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.ENC_KEY_DIV_INPUT;
                    else
                        dwOption &= ~( int ) VCA_Select.ENC_KEY_DIV_INPUT;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_Select.ENC_KEY_DIV_INPUT ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool UseDivInput_MACKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_INPUT;
                    else
                        dwOption &= ~( int ) VCA_Select.MAC_KEY_DIV_INPUT;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_INPUT ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool UseVCUID_MACKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_VCUID;
                    else
                        dwOption &= ~( int ) VCA_Select.MAC_KEY_DIV_VCUID;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_VCUID ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool UseVCUID_DivInput_MACKeyDiversification
            {
                set
                {
                    if ( value )
                    {
                        dwOption |= ( int ) VCA_Select.MAC_KEY_DIV_INPUT_VCUID;
                    }
                    else
                    {
                        if ( UseVCUID_MACKeyDiversification ) dwOption &= 0xFD;
                        else dwOption &= 0xF9;
                    }
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_Select.MAC_KEY_DIV_INPUT_VCUID ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool Use2PartVariantSupport
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_Select.VARIANT_PART2;
                    else
                        dwOption &= ~( int ) VCA_Select.VARIANT_PART2;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_Select.VARIANT_PART2 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }
        #endregion Select

        #region ProximityCheck
        public class VCA_ProximityCheck_Option
        {
            private int dwOption;

            public static implicit operator VCA_ProximityCheck_Option ( byte bOption )
            {
                VCA_ProximityCheck_Option oMyInstance = new VCA_ProximityCheck_Option ();
                oMyInstance.dwOption = bOption;
                return oMyInstance;
            }

            public static implicit operator byte ( VCA_ProximityCheck_Option oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool UseKeyDiversification
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.DIVERSIFICATION_ON;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.DIVERSIFICATION_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_ProximityCheck.DIVERSIFICATION_ON ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool UseRandomProcessing
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.RANDOM_PROCESSING;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.RANDOM_PROCESSING;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_ProximityCheck.RANDOM_PROCESSING ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool UseISO7816WrappedFormat
            {
                set
                {
                    if ( value )
                        dwOption |= ( int ) VCA_ProximityCheck.WRAPPED_FORMAT;
                    else
                        dwOption &= ~( int ) VCA_ProximityCheck.WRAPPED_FORMAT;
                }
                get
                {
                    if ( ( dwOption & ( int ) VCA_ProximityCheck.WRAPPED_FORMAT ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }
        #endregion ProximityCheck
        #endregion Virtual Card

        #region MIFARE DESFire
        #region Authentication
        public class DESFire_AuthenticateOption
        {
            private int dwOption;

            public static implicit operator DESFire_AuthenticateOption ( byte bOption )
            {
                DESFire_AuthenticateOption oObject = new DESFire_AuthenticateOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            public static implicit operator byte ( DESFire_AuthenticateOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool UseDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_Diversification.DIVERSIFICATION_ON;
                    else dwOption &= ~( int ) DF_Diversification.DIVERSIFICATION_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_Diversification.DIVERSIFICATION_ON ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool KeySelectionByDesfireKeyNumber
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeySel.DESFIRE_NUMBER_KEY_NUMBER;
                    else dwOption &= ~( int ) DF_KeySel.DESFIRE_NUMBER_KEY_NUMBER;
                }
                get
                {
                    if ( ( dwOption & 0x02 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseKeyDiversificationOneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyDivMode.AV1_SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) DF_KeyDivMode.AV1_SINGLE_ENCRYPTION;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyDivMode.AV1_SINGLE_ENCRYPTION ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseAV2KeyDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyDivMode.AV2;
                    else dwOption &= ~( int ) DF_KeyDivMode.AV2;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyDivMode.AV2 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool SupressSecureMessaging
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_SecureMessaging.SUPPRESS;
                    else dwOption &= ~( int ) DF_SecureMessaging.SUPPRESS;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_SecureMessaging.SUPPRESS ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool AuthenticateEV2First
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_AuthType.EV2_FIRST_AUTH;
                    else dwOption &= ~( int ) DF_AuthType.EV2_FIRST_AUTH;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_AuthType.EV2_FIRST_AUTH ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool AuthenticateEV2NonFirst
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_AuthType.EV2_NON_FIRST_AUTH;
                    else dwOption &= ~( ( int ) DF_AuthType.EV2_NON_FIRST_AUTH & 0x7F );
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_AuthType.EV2_NON_FIRST_AUTH ) != 0x00 ) return true;
                    else return false;
                }
            }
        }
        #endregion

        #region ChangeKey
        public class DESFire_ChangeKeyCompilationOption
        {
            private int dwOption;

            public static implicit operator DESFire_ChangeKeyCompilationOption ( byte bOption )
            {
                DESFire_ChangeKeyCompilationOption MyInstance = new DESFire_ChangeKeyCompilationOption ();
                MyInstance.dwOption = bOption;
                return MyInstance;
            }

            public static implicit operator byte ( DESFire_ChangeKeyCompilationOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool PICCTargetAndAuthKeySame
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.SAME_KEY;
                    else dwOption &= ~( int ) DF_KeyCompMthod.SAME_KEY;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.SAME_KEY ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool DiversifyNewKey
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.NEW_KEY_ON;
                    else dwOption &= ~( int ) DF_KeyCompMthod.NEW_KEY_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.NEW_KEY_ON ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool DiversifyCurrentKey
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.CURRENT_KEY_ON;
                    else dwOption &= ~( int ) DF_KeyCompMthod.CURRENT_KEY_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.CURRENT_KEY_ON ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseNewKeyDiversificationOneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.NEW_KEY_SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) DF_KeyCompMthod.NEW_KEY_SINGLE_ENCRYPTION;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.NEW_KEY_SINGLE_ENCRYPTION ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseCurrentKeyDiversificationOneRound
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.CURRENT_KEY_SINGLE_ENCRYPTION;
                    else dwOption &= ~( int ) DF_KeyCompMthod.CURRENT_KEY_SINGLE_ENCRYPTION;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.CURRENT_KEY_SINGLE_ENCRYPTION ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseSamAV2Diversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_KeyCompMthod.DIV_METHOD_AV2;
                    else dwOption &= ~( int ) DF_KeyCompMthod.DIV_METHOD_AV2;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_KeyCompMthod.DIV_METHOD_AV2 ) != 0x00 ) return true;
                    else return false;
                }
            }
        }

        public class DESFire_ChangeKeyConfigurationOption
        {
            private int dwOption;

            public static implicit operator DESFire_ChangeKeyConfigurationOption ( byte bOption )
            {
                DESFire_ChangeKeyConfigurationOption oObject = new DESFire_ChangeKeyConfigurationOption ();
                oObject.dwOption = bOption;
                return oObject;
            }

            public static implicit operator byte ( DESFire_ChangeKeyConfigurationOption oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool IncludeKeyType
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_Configuration.INCLUDE_KEYTYPE;
                    else dwOption &= ~( int ) DF_Configuration.INCLUDE_KEYTYPE;
                }

                get
                {
                    if ( ( dwOption & ( int ) DF_Configuration.INCLUDE_KEYTYPE ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool ChangeKeyEV2
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_Configuration.CHANGE_KEY_EV2;
                    else dwOption &= ~( int ) DF_Configuration.CHANGE_KEY_EV2;
                }

                get
                {
                    if ( ( dwOption & ( int ) DF_Configuration.CHANGE_KEY_EV2 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool UseISO7816_4_APDU
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_ISOMode.ISO7816_4;
                    else dwOption &= ~( int ) DF_ISOMode.ISO7816_4;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_ISOMode.ISO7816_4 ) != 0x00 ) return true;
                    else return false;
                }
            }

            public byte DESFireKeyNo
            {
                set
                {
                    dwOption &= 0xF0;
                    dwOption |= ( byte ) ( value & 0x0F );
                }
                get
                {
                    return ( byte ) ( dwOption & 0x0F );
                }
            }
        }
        #endregion ChangeKey

        #region WriteX and ReadX
        public class DESFire_ReadWrite_CryptoConfiguration
        {
            private int dwOption;

            public static implicit operator DESFire_ReadWrite_CryptoConfiguration ( byte bOption )
            {
                DESFire_ReadWrite_CryptoConfiguration oObject = new DESFire_ReadWrite_CryptoConfiguration ();
                oObject.dwOption = bOption;
                return oObject;
            }

            public static implicit operator byte ( DESFire_ReadWrite_CryptoConfiguration oOption )
            {
                return ( byte ) oOption.dwOption;
            }

            public bool ExtendedOffset
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_CryptoConfig.EXTENDED_OFFSET;
                    else dwOption &= ~( int ) DF_CryptoConfig.EXTENDED_OFFSET;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_CryptoConfig.EXTENDED_OFFSET ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool ISOChaining
            {
                set
                {
                    if ( value ) dwOption |= ( int ) DF_CryptoConfig.ISO_CHAINING;
                    else dwOption &= ~( int ) DF_CryptoConfig.ISO_CHAINING;
                }
                get
                {
                    if ( ( dwOption & ( int ) DF_CryptoConfig.ISO_CHAINING ) != 0x00 ) return true;
                    else return false;
                }
            }

            public DF_CryptoConfig CommMode
            {
                set
                {
                    dwOption &= 0xCF;
                    dwOption |= ( byte ) value;
                }
                get
                {
                    return ( DF_CryptoConfig ) ( dwOption & 0x30 );
                }
            }

            public byte Offset
            {
                set
                {
                    dwOption &= 0xF0;
                    dwOption |= ( byte ) ( value & 0x0F );
                }
                get
                {
                    return ( byte ) ( dwOption & 0x0F );
                }
            }
        }
        #endregion WriteX and ReadX
        #endregion MIFARE DESFire

        #region MIFARE Plus
        public class AuthenticateMFP_Option
        {
            public enum KDF : byte
            {
                NO_KDF = 0x00,
                SL3_KDF = 0x03
            }

            private int Option;

            public static implicit operator AuthenticateMFP_Option ( byte Option )
            {
                AuthenticateMFP_Option MyInstance = new AuthenticateMFP_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte ( AuthenticateMFP_Option Option )
            {
                return ( byte ) Option.Option;
            }

            public bool authenticateFirst
            {
                set
                {
                    if ( value )
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ( ( Option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool mifareEnciphered
            {
                set
                {
                    if ( value )
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ( ( Option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public KDF kdf
            {
                set
                {
                    Option &= 0xF3;
                    Option |= ( ( byte ) value << 2 );
                }
                get
                {
                    return ( KDF ) ( ( Option & 0x0C ) >> 2 );
                }
            }
        }

        public class AuthSectorSwitch_Option
        {
            private int dwOption;

            public static implicit operator AuthSectorSwitch_Option ( byte bOption )
            {
                AuthSectorSwitch_Option MyInstance = new AuthSectorSwitch_Option ();
                MyInstance.dwOption = bOption;
                return MyInstance;
            }

            public static implicit operator byte ( AuthSectorSwitch_Option Option )
            {
                return ( byte ) Option.dwOption;
            }

            public bool ApplySectorNumberKeyDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) MFP_SectorSwitchAuth.MASTER_SECTOR_DIV_ON ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool EnableSectorKeysDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) MFP_SectorSwitchAuth.SECTOR_DIV_ON ) != 0x00 ) return true;
                    else return false;
                }
            }

            public bool EnableSectorSwitchKeysDiversification
            {
                set
                {
                    if ( value ) dwOption |= ( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON;
                    else dwOption &= ~( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON;
                }
                get
                {
                    if ( ( dwOption & ( int ) MFP_SectorSwitchAuth.SECTOR_SWITCH_DIV_ON ) != 0x00 ) return true;
                    else return false;
                }
            }
        }

        public class CombinedReadWriteMfp_Option
        {
            private int Option;
            private bool first, cont, last;

            public enum CmdResp : byte
            {
                command = 0x00,
                response = 0x04,
                command_response = 0x08
            }

            public static implicit operator CombinedReadWriteMfp_Option ( int Option )
            {
                CombinedReadWriteMfp_Option MyInstance = new CombinedReadWriteMfp_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int ( CombinedReadWriteMfp_Option Option )
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if ( Option.first )
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if ( Option.cont )
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if ( Option.last )
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if ( value )
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if ( value )
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if ( value )
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }

            public CmdResp commandResponse
            {
                set
                {
                    Option &= 0xFFF3;
                    Option |= ( int ) value;
                }
                get
                {
                    return ( CmdResp ) ( Option & 0x000C );
                }

            }
        }

        public enum ChangeKeyMfp_Cmd : byte
        {
            writeEnc = 0xA0,
            writeEncWithRespMAC = 0xA1
        }
        #endregion MIFARE Plus

        #region MIFARE Classic
        public class MfRead_Option
        {
            private int option;

            public static implicit operator MfRead_Option ( byte option )
            {
                MfRead_Option MyInstance = new MfRead_Option ();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte ( MfRead_Option option )
            {
                return ( byte ) option.option;
            }

            public bool multiblockRead
            {
                set
                {
                    if ( value )
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ( ( option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfWrite_Option
        {
            private int option;
            private bool first, cont, last;

            public static implicit operator MfWrite_Option ( int option )
            {
                MfWrite_Option MyInstance = new MfWrite_Option ();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator int ( MfWrite_Option option )
            {
                option.option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                option.option &= ~ExchangeOptions.BUFFERED_BIT;

                if ( option.first )
                    option.option |= ExchangeOptions.BUFFER_FIRST;
                else if ( option.cont )
                    option.option |= ExchangeOptions.BUFFER_CONT;
                else if ( option.last )
                    option.option |= ExchangeOptions.BUFFER_LAST;

                return ( int ) option.option;
            }

            public bool mifareUlWrite
            {
                set
                {
                    if ( value )
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ( ( option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool multiblockWrite
            {
                set
                {
                    if ( value )
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ( ( option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if ( value )
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if ( value )
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if ( value )
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }
        }

        public class MfAuthenticateReadWrite_CmdSetting
        {
            private int option;

            public static implicit operator MfAuthenticateReadWrite_CmdSetting ( byte option )
            {
                MfAuthenticateReadWrite_CmdSetting MyInstance = new MfAuthenticateReadWrite_CmdSetting ();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte ( MfAuthenticateReadWrite_CmdSetting option )
            {
                return ( byte ) option.option;
            }

            public bool keyInformationProvided
            {
                set
                {
                    if ( value )
                        option &= ~0x01;
                    else
                        option |= 0x01;
                }
                get
                {
                    if ( ( option & 0x01 ) != 0x00 )
                        return false;
                    else
                        return true;
                }
            }

            public bool useKeyDiversification
            {
                set
                {
                    if ( value )
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ( ( option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfAuthenticate_Option
        {
            private int option;

            public static implicit operator MfAuthenticate_Option ( byte option )
            {
                MfAuthenticate_Option MyInstance = new MfAuthenticate_Option ();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte ( MfAuthenticate_Option option )
            {
                return ( byte ) option.option;
            }

            public bool useKeyDiversification
            {
                set
                {
                    if ( value )
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ( ( option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfChangeKey_KeyCompMeth
        {
            private int option;

            public static implicit operator MfChangeKey_KeyCompMeth ( byte option )
            {
                MfChangeKey_KeyCompMeth MyInstance = new MfChangeKey_KeyCompMeth ();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte ( MfChangeKey_KeyCompMeth option )
            {
                return ( byte ) option.option;
            }

            public bool diversifyKeyA
            {
                set
                {
                    if ( value )
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ( ( option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyB
            {
                set
                {
                    if ( value )
                        option |= 0x04;
                    else
                        option &= ~0x04;
                }
                get
                {
                    if ( ( option & 0x04 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKeyMifare_Option
        {
            private int Option;

            public static implicit operator ChangeKeyMifare_Option ( byte Option )
            {
                ChangeKeyMifare_Option MyInstance = new ChangeKeyMifare_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte ( ChangeKeyMifare_Option Option )
            {
                return ( byte ) Option.Option;
            }

            public bool retrieveKey
            {
                set
                {
                    if ( value )
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ( ( Option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyA
            {
                set
                {
                    if ( value )
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ( ( Option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyB
            {
                set
                {
                    if ( value )
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ( ( Option & 0x04 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool retrieveEncrypted
            {
                set
                {
                    if ( value )
                        Option |= 0x08;
                    else
                        Option &= ~0x08;
                }
                get
                {
                    if ( ( Option & 0x08 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public class AuthenticateMifare_Option
        {
            private int Option;

            public static implicit operator AuthenticateMifare_Option ( byte Option )
            {
                AuthenticateMifare_Option MyInstance = new AuthenticateMifare_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte ( AuthenticateMifare_Option Option )
            {
                return ( byte ) Option.Option;
            }

            public bool useKeyDiversification
            {
                set
                {
                    if ( value )
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ( ( Option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }
        #endregion MIFARE Classic

        #region ISO14443-3
        public class ActivateIdle_Option
        {
            private int Option;

            public static implicit operator ActivateIdle_Option ( byte Option )
            {
                ActivateIdle_Option MyInstance = new ActivateIdle_Option ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte ( ActivateIdle_Option Option )
            {
                return ( byte ) Option.Option;
            }

            public bool applyAtqaFilter
            {
                set
                {
                    if ( value )
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ( ( Option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool applySakFilter
            {
                set
                {
                    if ( value )
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ( ( Option & 0x02 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }

            public bool applyTime
            {
                set
                {
                    if ( value )
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ( ( Option & 0x04 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }

        public struct ActivateIdleParams
        {
            public byte[] Atqa; /**< ATQA of card */
            public byte Sak;    /**< SAK of card */
            public byte[] Uid;  /**< UID of card */
        };
        #endregion ISO14443-3

        #region ISO14443-4
        public class Deselect_FreeCid
        {
            private int Option;

            public static implicit operator Deselect_FreeCid ( byte Option )
            {
                Deselect_FreeCid MyInstance = new Deselect_FreeCid ();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte ( Deselect_FreeCid Option )
            {
                return ( byte ) Option.Option;
            }

            public bool forceFreeing
            {
                set
                {
                    if ( value )
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ( ( Option & 0x01 ) != 0x00 )
                        return true;
                    else
                        return false;
                }
            }
        }
        #endregion ISO14443-4
        #endregion Support Classes

        #region Support Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error_Gen.SUCCESS_CHAINING ) || oStatus.Equals ( Error_Gen.SUCCESS_INCOMPLETE_BYTE ) ||
                oStatus.Equals ( new Status_t ( Error_CompCode.HAL, Error.OK_CHAINING_ACTIVE ) ) ) && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( pBuffer != IntPtr.Zero )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }
        #endregion Support Methods

        #region DataParams Members
        public byte[] Uid
        {
            get
            {
                byte[] aUid = new byte[7];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 7; i++ )
                        {
                            aUid[i] = pDataParams->bUid[i];
                        }
                    }
                }
                return aUid;
            }
        }
        #endregion
    }
}
#endif
