﻿/*
 * Copyright 2019, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// ProxiLAB specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class ProxiLAB : Hal.Generic
    {
        #region DEFINES

        public const int DEFAULT_TIMEOUT_MS = 5;        /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0F;          /**< Number of shadowed configurations. */

        public enum Config : ushort
        {
            DEMODULATORINPUT = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN), /**< Set the input of the demodulator one of ProxiLABDemodulatorInput */
            POWERLEVEL,                                                /**< Set the power level of the ProxiLab output (0..1023) */
            DISPLAY_VERBOSE,                                           /**< Enable or disable the verbose output (default is enabled) */
            CLOCK_SOURCE,                                              /**< select the source of the clock. one of ProxiLABClockSource */
            MODULATOR_OUTPUT,                                          /**< select the rf output mode. one of ProxiLABModulatorOutput */
            DEMODULATOR_AUTO_THRESHOLD,                                /**< Reader Rx DemodulatorAutoThreshold is enabled or disabled (Values PH_ON, PH_OFF) */
            DEMODULATOR_THRESHOLD                                      /**< Reader Rx DemodulatorThreshold from (0..255) */
        }

        /*public enum IODirection : int
        {
            DIRECTION_OUTPUT = 0,
            DIRECTION_INPUT = 1
        }

        public enum Output : int
        {
            OUTPUT_CONSTANT_LOW = 0,
            OUTPUT_CONSTANT_HIGH = 1,
            OUTPUT_HIGH_IMPEDANCE = 2,
            OUTPUT_TRIGGER = 8,
            OUTPUT_SPLUSE = 9,
            OUTPUT_RF_POWER = 16,
            OUTPUT_RF_POWER_ON_SYNC = 17,
            OUTPUT_RF_POWER_OFF_SYNC = 18,
            EXTRACT_CLK = 31,
            OUTPUT_PCD_IO = 32,
            OUTPUT_PCD_FRAME = 33,
            OUTPUT_PCD_SOF = 34,
            OUTPUT_PCD_EOF = 35,
            OUTPUT_PCD_BIT_SYNC = 36,
            OUTPUT_PCD_BYTE_SYNC = 37,
            OUTPUT_PCD_PARITY_ERROR = 38,
            OUTPUT_PCD_CRC_ERROR = 39,
            OUTPUT_PICC_IO = 48,
            OUTPUT_PICC_FRAME = 49,
            OUTPUT_PICC_MANCHESTER = 50,
            OUTPUT_PICC_SUBCARRIER = 51,
            OUTPUT_PICC_PHASE = 52,
            OUTPUT_PICC_SOF = 53,
            OUTPUT_PICC_EOF = 54,
            OUTPUT_PICC_BIT_SYNC = 55,
            OUTPUT_PICC_BYTE_SYNC = 56,
            OUTPUT_PICC_COLLISION_ERROR = 57,
            OUTPUT_PICC_PARITY_ERROR = 58,
            OUTPUT_PICC_CRC_ERROR = 59,
            OUTPUT_PICC_TIMEOUT_ERROR = 60,
            OUTPUT_PICC_FFT_SYNC = 61,
            OUTPUT_PCD_A_FRAME = 288,
            OUTPUT_PCD_B_FRAME = 289,
            OUTPUT_PCD_FELICA_FRAME = 290,
            OUTPUT_PCD_ISO15693_FRAME = 291,
            OUTPUT_SECURITY_PULSE = 292,
            OUTPUT_PICC_MODULATOR_IO = 293
        }*/

        public enum ClockSource : ushort
        {
            INT = 0x00,      /**< Use embedded PLL. */
            EXT_DIG = 0x01,  /**< Use SMA3 as a clock input. The clock frequency is 2 times the carrier frequency. Ex : Fc = 13,56MHz => Digital Clock = 27,12MHz. */
            EXT_ANA = 0x02   /**< Use SMA1 as a carrier input. The voltage range is [1V;1,5V]. There is no processing on this signal. */
        }

        public enum Demodulator : ushort
        {
            RF_OUT = 0x00,      /**< Selects the RF OUT antenna (standard case: the antenna transmits and receives). */
            RF_IN = 0x01,       /**< Selects RF IN connector (the RF OUT antenna transmits, and any other probe is connected to RF IN to receive). */
            BRIDGE = 0x03,      /**< Selects RF IN HDMI connector (test assembly case: the RF OUT antenna transmits, and the bridge is connected to HDMI to receive). */
            BRIDGE_SSB = 0x05,  /**< Selects RF IN HDMI connector with SSB filter (test assembly case: the RF OUT antenna transmits, and the bridge is connected to HDMI to receive). */
            SMA2 = 0x04         /**< Selects SMA2 (TTL) connector. */
        }

        /**
        * \name Modulator Output Configuration Values
        */
        public enum ModulatorOutput : ushort
        {
            RF_OUT              = 0x00, /**< Selects RF OUT as output. */
            AWG_OUT             = 0x01, /**< Selects AWG OUT as output. */
            RF_OUT_DUAL         = 0x02, /**< Enables both outputs. */
            AWG_CARRIER_RF_OUT  = 0x03  /**< Selects AWG_CARRIER_RF_OUT. */
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]

        public unsafe struct DataParams_t
        {
            public ushort wId;                               /**< Layer ID for this HAL component, NEVER MODIFY! */
		    public IntPtr pBalDataParams;                    /**< pointer to the lower layers parameter structure */
		    public IntPtr pTxBuffer;                         /**< Pointer to global transmit buffer used by the Exchange() function. */
		    public ushort wTxBufSize;                        /**< Size of the global transmit buffer. */
		    public ushort wTxBufLen;                         /**< Number of valid bytes within the transmit buffer. */
		    public IntPtr pRxBuffer;                         /**< Pointer to global receive buffer used by the Exchange() function. */
		    public ushort wRxBufSize;                        /**< Size of the global receive buffer. */
		    public ushort wRxBufLen;                         /**< Number of valid bytes within the receive buffer. */
		    public ushort wRxBufStartPos;                    /**< Starting position within the global receive buffer. */
		    public ushort wTxBufStartPos;                    /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
		    public byte bCardType;                           /**< Type of card for which the hal is configured for. */
            public byte bTimeoutUnit;                        /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public fixed ushort wCfgShadow[SHADOW_COUNT];    /**< Configuration shadow; Stores configuration for current cardtype. */
		    /*public ushort wTimingMode;                     /**< Current timing measurement mode. */
		    public ushort wAdditionalInfo;                   /**< Storage for additional error information. */
		    public ushort wFieldOffTime;                     /**< Field-Off-Time in milliseconds. */
		    public ushort wFieldRecoveryTime;                /**< Field-Recovery-Time in milliseconds. */
		    public uint dwPowerLevel;                        /**< Power Level of RF Field */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ProxiLAB_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize               /**< [In] Size of the global receive buffer. */
            );

        #endregion

        #region CUSTOM COMMANDS
        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.ProxiLAB pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_ProxiLAB_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            return Init(wDataParamSize, pBal, wTxBufferSize, m_bTxBuffer, wRxBufferSize, m_bRxBuffer);
        }
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_ProxiLAB_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public ProxiLAB()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~ProxiLAB()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
#endif
