/*
 * Copyright 2013, 2016, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// MFCCRYPTO specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class MfcCrypto : Hal.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            /** Enable or disable MFCCrypto layer. */
            ENABLE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            ushort wId;                 /**< Layer ID for this HAL component, NEVER MODIFY! */
            IntPtr pHalDataParams;      /**< Pointer to the lower layers parameter structure. */
            fixed byte rA_buf[4];       /**< Initial RA value. */
            fixed byte rBp_buf[4];      /**< Initial RB value. */
            fixed byte cryptoReg[48];   /**< The CRYPTO LFSR. */
            fixed byte randReg[16];     /**< The PRNG LFSR for RB. */
            IntPtr pTxBuffer;           /**< Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize;          /**< Size of the global transmit buffer. */
            ushort wTxBufLen;           /**< Number of valid bytes within the transmit buffer. */
            byte bCryptoState;          /**< Current crypto state. */
            byte bActivated;            /**< Indicates whether this layer is activated or not. */
            uint uIdForSpFeed1;         /**< UID to unit32. */
            byte bEnabled;              /**< Disables/Enables Crypto. */
            byte bRxBits;               /**< Number of valid bits in the receive buffer. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize               /**< [In] Size of the global transmit buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Cmd_EncodeAuth1(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
            byte bAuthType,                 /**< [In] Authentication type. */
            byte bBlockAddr,                /**< [In] Block address. */
            byte[] pAuth1Msg                /**< [Out] Encrypted authentication 1 message, ready to send. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Cmd_DecodeAuth1(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
            byte[] pAuth1Reply,             /**< [In] Authentication reply. */
            ushort wAuth1ReplyLength,       /**< [In] Authentication reply length. */
            byte bAuth1ReplyBits,           /**< [In] Authentication reply bit length of the last byte. */
            byte[] pUid,                    /**< [In] Uid. */
            byte[] pKey,                    /**< [In] Key. */
            byte[] pRndA,                   /**< [In] Random A. */
            byte[] pRndB                    /**< [Out] Random B. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Cmd_EncodeAuth2(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
            byte[] pAuth2Msg                /**< [Out] Encrypted authentication 2 message, ready to send. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Cmd_DecodeAuth2(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
            byte[] pAuth2Reply,             /**< [In] Authentication reply. */
            ushort wAuth2ReplyLength,       /**< [In] Authentication reply length. */
            byte bAuth2ReplyBits,           /**< [In] Authentication reply bit length of the last byte. */
            byte[] pRndB2                   /**< [Out] Random B'. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_MfcCrypto_Cmd_CodeData(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
            byte[] pDataIn,                 /**< [In] Input buffer. */
            uint dwDataBitLen,              /**< [In] Buffer bit length. */
            byte[] pDataOut                 /**< [Out] Output buffer. */
            );

        #endregion

        #region CUSTOM COMMANDS

        public Status_t Cmd_EncodeAuth1(
            byte bAuthType,         /**< [In] Authentication type. */
            byte bBlockAddr,        /**< [In] Block address. */
            out byte[] pAuth1Msg    /**< [Out] Encrypted authentication 1 message, ready to send. */
            )
        {
            Status_t status;

            pAuth1Msg = new byte[5];
            status = phhalHw_MfcCrypto_Cmd_EncodeAuth1(ref m_DataParamsInt[0], (byte)bAuthType, bBlockAddr, pAuth1Msg);

            if (!status.Equals(new Status_t()))
            {
                pAuth1Msg = null;
            }

            return status;
        }

        public Status_t Cmd_DecodeAuth1(
            byte[] pAuth1Reply,         /**< [In] Authentication reply. */
            int wAuth1ReplyLength,      /**< [In] Authentication reply length. */
            byte bAuth1ReplyBits,       /**< [In] Authentication reply bit length of the last byte. */
            byte[] pUid,                /**< [In] Uid. */
            byte[] pKey,                /**< [In] Key. */
            byte[] pRndA,               /**< [In] Random A. */
            out byte[] pRndB            /**< [Out] Random B. */
            )
        {
            Status_t status;

            pRndB = new byte[4];
            status = phhalHw_MfcCrypto_Cmd_DecodeAuth1(ref m_DataParamsInt[0], pAuth1Reply, (ushort)wAuth1ReplyLength, bAuth1ReplyBits, pUid, pKey, pRndA, pRndB);

            if (!status.Equals(new Status_t()))
            {
                pRndB = null;
            }

            return status;
        }

        public Status_t Cmd_EncodeAuth2(
            out byte[] pAuth2Msg    /**< [Out] Encrypted authentication 2 message, ready to send. */
            )
        {
            Status_t status;

            pAuth2Msg = new byte[9];
            status = phhalHw_MfcCrypto_Cmd_EncodeAuth2(ref m_DataParamsInt[0], pAuth2Msg);

            if (!status.Equals(new Status_t()))
            {
                pAuth2Msg = null;
            }

            return status;
        }

        public Status_t Cmd_DecodeAuth2(
            byte[] pAuth2Reply,         /**< [In] Authentication reply. */
            int wAuth2ReplyLength,      /**< [In] Authentication reply length. */
            byte bAuth2ReplyBits,       /**< [In] Authentication reply bit length of the last byte. */
            out byte[] pRndB2           /**< [Out] Random B'. */
            )
        {
            Status_t status;

            pRndB2 = new byte[4];
            status = phhalHw_MfcCrypto_Cmd_DecodeAuth2(ref m_DataParamsInt[0], pAuth2Reply, (ushort)wAuth2ReplyLength, bAuth2ReplyBits, pRndB2);

            if (!status.Equals(new Status_t()))
            {
                pRndB2 = null;
            }

            return status;
        }

        public Status_t Cmd_CodeData(
            byte[] pDataIn,     /**< [In] Input buffer. */
            long dwDataBitLen,  /**< [In] Buffer bit length. */
            out byte[] pDataOut /**< [Out] Output buffer. */
            )
        {
            Status_t status;
            int DataByteLength = (int)((dwDataBitLen % 8 == 0) ? (dwDataBitLen >> 3) : ((dwDataBitLen >> 3) + 1));
            pDataOut = new byte[DataByteLength];
            status = phhalHw_MfcCrypto_Cmd_CodeData(ref m_DataParamsInt[0], pDataIn, (uint)dwDataBitLen, pDataOut);

            if (!status.Equals(new Status_t()))
            {
                pDataOut = null;
            }

            return status;
        }

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Hal.Generic pHal, int wTxBufferSize)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }

            // Allocate TxBuffer
            m_bTxBuffer = new byte[wTxBufferSize];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_MfcCrypto_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pHal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public MfcCrypto()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~MfcCrypto()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
