/*
 * Copyright 2013, 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Callback HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Callback : Hal.Generic
    {
        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct DataParams_t
        {
            ushort wId;
            IntPtr pExchangeCb;
            IntPtr pWriteRegisterCb;
            IntPtr pReadRegisterCb;
            IntPtr pApplyProtocolSettingsCb;
            IntPtr pMfcAuthenticateCb;
            IntPtr pMfcAuthenticateKeyCb;
            IntPtr pSetConfigCb;
            IntPtr pGetConfigCb;
            IntPtr pFieldOnCb;
            IntPtr pFieldOffCb;
            IntPtr pFieldResetCb;
            IntPtr pWaitCb;
            IntPtr pCbContext;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Callback_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pExchangeCb,
            IntPtr pWriteRegisterCb,
            IntPtr pReadRegisterCb,
            IntPtr pApplyProtocolSettingsCb,
            IntPtr pMfcAuthenticateCb,
            IntPtr pMfcAuthenticateKeyCb,
            IntPtr pSetConfigCb,
            IntPtr pGetConfigCb,
            IntPtr pFieldOnCb,
            IntPtr pFieldOffCb,
            IntPtr pFieldResetCb,
            IntPtr pWaitCb,
            IntPtr pCbContext
            );

        #endregion

        #region DELEGATES

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ExchangeDelegate(
            ushort wOption,         /**< [In] Option parameter. */
            IntPtr pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] received data */
            ref ushort pRxLength,   /**< [Out] number of received data bytes */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort WriteRegisterDelegate(
            byte bAddress,      /**< [In] Register Address */
            byte bValue,        /**< [In] Register Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ReadRegisterDelegate(
            byte bAddress,      /**< [In] Register Address */
            ref byte pValue,    /**< [Out] Register Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort ApplyProtocolSettingsDelegate(
            byte bCardType, /**< [In] Type of card for which the hal should be configured for */
            IntPtr pContext /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate ushort MfcAuthenticateKeyNoDelegate(
            byte bBlockNo,          /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
            ushort wKeyNo,          /**< [In] Key number to be used in authentication. */
            ushort wKeyVersion,     /**< [In] Key version to be used in authentication. */
            IntPtr pUid,            /**< [In] Serial number of current cascade level; byte[4]. */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate ushort MfcAuthenticateDelegate(
            byte bBlockNo,      /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,   /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
            IntPtr pKey,        /**< [In] Key to be used in authentication. */
            IntPtr pUid,        /**< [In] Serial number of current cascade level; byte[4]. */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort SetConfigDelegate(
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue,      /**< [In] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
        );


        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort GetConfigDelegate(
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue,  /**< [Out] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort FieldOnDelegate(
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort FieldOffDelegate(
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort FieldResetDelegate(
            IntPtr pContext             /**< [Out] User Defined Context */
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate ushort WaitDelegate(
            byte bUnit,             /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            ushort wTimeout,        /**< [In] Timeout value */
            IntPtr pContext         /**< [Out] User Defined Context */
            );

        #endregion

        #region CALLBACKS

        private ushort cbExchange(
                    ushort wOption,         /**< [In] Option parameter. */
                    IntPtr pTxBuffer,       /**< [In] data to transmit */
                    ushort wTxLength,       /**< [In] length of input data */
                    ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
                    ref ushort pRxLength,   /**< [Out] number of received data bytes */
                    IntPtr pContext         /**< [Out] User Defined Context */
                    )
        {
            Status_t status;
            byte[] pTxBufferTmp = new byte[wTxLength];
            byte[] pRxBufferTmp;
            int dwRxOffset;

            // Prepare input buffer(s)
            for (int dIndex = 0; dIndex < wTxLength; dIndex++)
            {
                pTxBufferTmp[dIndex] = Marshal.ReadByte(pTxBuffer, dIndex);
            }

            // Reset Receive length
            pRxLength = 0;

            // Execute callback function
            status = m_ExtHal.Exchange((int)wOption, pTxBufferTmp, out pRxBufferTmp);

            // Return pointer to global buffer
            if (pRxBufferTmp != null)
            {
                if (pRxBufferTmp.Length != 0)
                {
                    // Buffer overflow check
                    if (pRxBufferTmp.Length > m_RxBuffer.Length)
                    {
                        return new Status_t(Error_CompCode.HAL, Error_Comm.BUFFER_OVERFLOW);
                    }

                    // Get RxStartPos
                    var getStatus = m_ExtHal.GetConfig(NxpRdLibNet.Hal.Config.RXBUFFER_STARTPOS, out dwRxOffset);
                    if (!getStatus.Equals(new Status_t()))
                        return getStatus;

                    // Copy only new Rx-Contents to persistent HAL buffer (to prevent override of already restored data)
                    System.Array.Copy(pRxBufferTmp, dwRxOffset, m_RxBuffer, dwRxOffset, pRxBufferTmp.Length - dwRxOffset);

                    // Set Output Buffer to persistent HAL buffer pointer
                    ppRxBuffer = m_pRxBuffer.AddrOfPinnedObject();

                    // Set number of received bytes
                    pRxLength = (ushort)pRxBufferTmp.Length;
                }
            }

            return status;
        }

        private ushort cbWriteRegister(
            byte bAddress,      /**< [In] Register Address */
            byte bValue,        /**< [In] Register Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            // Execute callback function
            return m_ExtHal.WriteRegister(bAddress, bValue);
        }

        private ushort cbReadRegister(
            byte bAddress,      /**< [In] Register Address */
            ref byte pValue,    /**< [Out] Register Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            // Execute callback function
            return m_ExtHal.ReadRegister(bAddress, out pValue);
        }

        private ushort cbApplyProtocolSettings(
            byte bCardType, /**< [In] Type of card for which the hal should be configured for */
            IntPtr pContext /**< [Out] User Defined Context */
            )
        {
            // Execute callback function
            return m_ExtHal.ApplyProtocolSettings((CardType)bCardType);
        }

        private ushort cbMfcAuthenticateKeyNo(
            byte bBlockNo,          /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,       /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
            ushort wKeyNo,          /**< [In] Key number to be used in authentication. */
            ushort wKeyVersion,     /**< [In] Key version to be used in authentication. */
            IntPtr pUid,            /**< [In] Serial number of current cascade level; byte[4]. */
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            byte[] pUidTmp = null;

            if (pUid != IntPtr.Zero)
            {
                pUidTmp = new byte[4];
                for (int dIndex = 0; dIndex < 4; dIndex++)
                {
                    pUidTmp[dIndex] = Marshal.ReadByte(pUid, dIndex);
                }
            }
            // Execute callback function
            return m_ExtHal.MfcAuthenticateKeyNo(bBlockNo, bKeyType, wKeyNo, wKeyVersion, pUidTmp);
        }

        private ushort cbMfcAuthenticate(
            byte bBlockNo,      /**< [In] Blocknumber on card to authenticate to. */
            KeyType bKeyType,   /**< [In] Either #PHHAL_HW_MFC_KEYA or #PHHAL_HW_MFC_KEYB */
            IntPtr pKey,        /**< [In] Key to be used in authentication. */
            IntPtr pUid,        /**< [In] Serial number of current cascade level; byte[4]. */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            byte[] pKeyTmp = null;
            byte[] pUidTmp = null;

            if (pUid != IntPtr.Zero)
            {
                pUidTmp = new byte[4];
                for (int dIndex = 0; dIndex < 4; dIndex++)
                {
                    pUidTmp[dIndex] = Marshal.ReadByte(pUid, dIndex);
                }
            }

            // Prepare input buffer(s)
            if (pKey != IntPtr.Zero)
            {
                pKeyTmp = new byte[6];
                for (int dIndex = 0; dIndex < 6; dIndex++)
                {
                    pKeyTmp[dIndex] = Marshal.ReadByte(pKey, dIndex);
                }
            }

            // Execute callback function
            return m_ExtHal.MfcAuthenticate(bBlockNo, bKeyType, pKeyTmp, pUidTmp);
        }

        private ushort cbSetConfig(
            ushort wConfig, /**< [In] Configuration Identifier */
            ushort wValue,  /**< [In] Configuration Value */
            IntPtr pContext /**< [Out] User Defined Context */
                                          )
        {
            // Execute callback function
            return m_ExtHal.SetConfig((Config)wConfig, (int)wValue);
        }

        private ushort cbGetConfig(
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue,  /**< [Out] Configuration Value */
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            Status_t status;

            // Execute callback function
            int HelperpValue = 0;
            status = m_ExtHal.GetConfig((Config)wConfig, out HelperpValue);
            pValue = (ushort)HelperpValue;

            return status;
        }

        private ushort cbFieldOn(
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            Status_t status;

            // Execute callback function
            status = m_ExtHal.FieldOn();

            return status;
        }

        private ushort cbFieldOff(
            IntPtr pContext     /**< [Out] User Defined Context */
            )
        {
            Status_t status;

            // Execute callback function
            status = m_ExtHal.FieldOff();

            return status;
        }

        private ushort cbFieldReset(
            IntPtr pContext             /**< [Out] User Defined Context */
            )
        {
            Status_t status;

            // Execute callback function
            status = m_ExtHal.FieldReset();

            return status;
        }

        private ushort cbWait(
            byte bUnit,             /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            ushort wTimeout,        /**< [In] Timeout value */
            IntPtr pContext         /**< [Out] User Defined Context */
            )
        {
            Status_t status;

            // Execute callback function
            status = m_ExtHal.Wait(bUnit, wTimeout);

            return status;
        }

        #endregion

        #region INIT

        // Storage for internal callbacks
        private List<Delegate> m_Callbacks;
        private Hal.IHal m_ExtHal;
        private byte[] m_RxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="extInterface"></param>
        /// <returns></returns>
        public Status_t Init(
            Hal.IHal ExtHal,
            int RxBufferSize
            )
        {
            return Init(ExtHal, RxBufferSize, IntPtr.Zero);
        }

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="extInterface"></param>
        /// <returns></returns>
        public Status_t Init(
            Hal.IHal ExtHal,
            int RxBufferSize,
            IntPtr cbContext
            )
        {
            // Allocate internal callback delegates
            m_Callbacks = new List<Delegate>();
            m_Callbacks.Add(new ExchangeDelegate(cbExchange));
            m_Callbacks.Add(new WriteRegisterDelegate(cbWriteRegister));
            m_Callbacks.Add(new ReadRegisterDelegate(cbReadRegister));
            m_Callbacks.Add(new ApplyProtocolSettingsDelegate(cbApplyProtocolSettings));
            m_Callbacks.Add(new MfcAuthenticateKeyNoDelegate(cbMfcAuthenticateKeyNo));
            m_Callbacks.Add(new MfcAuthenticateDelegate(cbMfcAuthenticate));
            m_Callbacks.Add(new SetConfigDelegate(cbSetConfig));
            m_Callbacks.Add(new GetConfigDelegate(cbGetConfig));
            m_Callbacks.Add(new FieldOnDelegate(cbFieldOn));
            m_Callbacks.Add(new FieldOffDelegate(cbFieldOff));
            m_Callbacks.Add(new FieldResetDelegate(cbFieldReset));
            m_Callbacks.Add(new WaitDelegate(cbWait));

            // Store reference to external hal object
            m_ExtHal = ExtHal;

            // Free allocated RxBuffer if any
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Create RxBuffer
            m_RxBuffer = new byte[RxBufferSize];
            this.m_pRxBuffer = GCHandle.Alloc(this.m_RxBuffer, GCHandleType.Pinned);

            // Call init. function
            return phhalHw_Callback_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[0]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[1]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[2]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[3]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[4]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[5]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[6]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[7]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[8]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[9]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[10]),
                Marshal.GetFunctionPointerForDelegate(m_Callbacks[11]),
                cbContext);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Callback()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Callback()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }

            // Free allocated RxBuffer
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        private DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
