/*
 * Copyright 2013, 2017, 2019, 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlMstAmpOsc
{
    #region BASE

    #region DEFINES
    public enum Error : byte
    {
        /// <summary> entered field strength too high </summary>
        FIELD_TOO_HIGH = (CustomCodes.ERROR_BEGIN),
        /// <summary> entered field strength too low </summary>
        FIELD_TOO_LOW,
        /// <summary> Error in preset field strength values causing a gap in amplifiers chacracteristic. </summary>
        FIELD_GAP,
        /// <summary> Desired presision can not be reached, lower precision restriction</summary>
        FIELD_PRECISION_NOT_ACHIEVEABLE,
        /// <summary> Desired presision can not be reached, lower precision restriction. </summary>
        NOT_CAL,
        /// <summary> Failed in finding the right gain values. </summary>
        COULD_NOT_FIND_GAIN_VALS,
        /// <summary> Antenna type not supported. </summary>
        INVALID_ANTENNA_TYPE,
        /// <summary> Indicates that the channel values did not match (Sense coil > Cal Coil). </summary>
        CHANNEL_VALUE_MISSMATCH
    }
    public enum Config : ushort
    {
        PRECISION = 0,      /**< Specifies the desired precision. */
        CHECK_PRECISION,    /**< Specifies if the desired precesion is checked,if the precion can't be reached a recalibration is done if this does not work out an error is thrown. */
        MIN_FIELD,          /**< Defines the minimum field strength that can be set (read only). */
        MAX_FIELD,          /**< Defines the maximum field strength that can be set (read only). */
        AMPLIFIER_GAIN      /**< Used to read out the used gain  (read only). */
    };

    #endregion

    public abstract class Generic
    {

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_SetConfig(
                             IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                             ushort wIdentifier,			/**< [In] Specifies parameter identifier. */
                             ushort wValue				    /**< [In] Specifies parameter value. */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_GetConfig(
                             IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                             ushort wIdentifier,			/**< [In] Specifies parameter identifier. */
                             ref ushort pwValue				/**< [In] Specifies parameter value. */
                             );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_InitMstAmpOsc(
                             IntPtr pDataParams 			            /**< [In] Pointer to this layer's parameter structure. */
                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_Cal(
                                     IntPtr pDataParams				/**< [In] Pointer to this layer's parameter structure. */
                                     );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_SetFieldStrength(
                                     IntPtr pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
                                     ushort wFieldStrength		/**< [In] Specifies the field strength. */
                                     );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t SetConfig(
                                    ushort wIdentifier,
                                    ushort wValue
                                       )
        {
            return phdlMstAmpOsc_SetConfig(m_pDataParams, wIdentifier, wValue);
        }

        public Status_t GetConfig(
                                    ushort wIdentifier,
                                    out ushort wValue
                                       )
        {
            Status_t status;
            ushort wTemp = 0;
            status = phdlMstAmpOsc_GetConfig(m_pDataParams, wIdentifier, ref wTemp);
            wValue = wTemp;
            return status;
        }

        public Status_t InitMstAmpOsc()
        {
            return phdlMstAmpOsc_InitMstAmpOsc(m_pDataParams);
        }

        public Status_t Cal()
        {
            return phdlMstAmpOsc_Cal(m_pDataParams);
        }

        public Status_t SetFieldStrength(
                                    ushort wFieldStrength
                                       )
        {
            return phdlMstAmpOsc_SetFieldStrength(m_pDataParams, wFieldStrength);
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region AR75A250
    public class AR75A250 : dlMstAmpOsc.Generic
    {
        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pDlAmpDataParams;         /**< Pointer to the parameter structure of the dlAmp component. */
            public IntPtr pDlOsciDataParams;        /**< Pointer to the parameter structure of the dlOsci component. */
            public ushort wMinFieldStrengthAmp;     /**< Min achieveable fieldstrength with amplifier. */
            public ushort wMaxFieldStrengthAmp;     /**< Max achieveable fieldstrength with amplifier. */
            public byte bChannel;                   /**< Selected osci channel for field strength measurement. */
            public byte bPrecision;                 /**< Set the required precision for set field strength. */
            public byte bCheckPrecision;            /**< Flag indicating if precision has to be checked. */
            public byte bCal;                       /**< Flag indicating if calibration has been performed. */
            public ushort wMinGainVal;              /**< Determines the minimum value for the gain settings of the dlAmp. */
            public ushort wMaxGainVal;              /**< Determines the maximum value for the gain settings of the dlAmp. */
        };
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_AR75A250_Init(
            ref DataParams_t m_pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,               /**< [In] Specifies the size of the data parameter structure */
            IntPtr pAlAmpDataParams,   				/**< [In] Pointer to the parameter structure of the alAmp component. */
            IntPtr pAlOsciDataParams,               /**< [In] Pointer to the parameter structure of the alOsci component. */
            byte bChannel                           /**< [In] Specifies the oscilloscope channel for the field strength signal. */
            );

        #endregion

        #region INIT
        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            byte bChannel
            )
        {
            return phdlMstAmpOsc_AR75A250_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pDlAmp.m_pDataParams, pDlOsci.m_pDataParams, bChannel);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public AR75A250()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~AR75A250()
        {

            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion

    #region ISOSetup
    public class ISOSetup : dlMstAmpOsc.Generic
    {
        #region DEFINES

        public const byte PHDL_MSTAMPOSC_ID_CNT = 4;

        public enum Config : ushort
        {
            FIND_GAIN_VALS = (CustomCodes.CONFIG_BEGIN),    /**< Configure if during calibration the maximum range gain values are determined. */
            USE_MAX_GAIN,                                   /**< Defines if the highest gain setting is used as well. */
            GAIN_0,                                         /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
            GAIN_1,                                         /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
            GAIN_2,                                         /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
            GAIN_3,                                         /**< Defines the (initial) value for gain 0 (is modified if find gain vals is ON). */
            MAX_FIELD_FIND,                                 /**< Defines the minimum value for the maximum field strength that is tried to reach during find gain values. */
            MIN_FIELD_OVERLAP,                              /**< Defines the minimum overlap of the field for the different gain values. */
        };
        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                              /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pDlAmpDataParams;                                 /**< Pointer to the parameter structure of the dlAmp component. */
            public IntPtr pDlOsciDataParams;                                /**< Pointer to the parameter structure of the dlOsci component. */
            public IntPtr pDlRdFpgaDataParams;                              /**< Pointer to the parameter structure of the dlRdFpga component. */
            public ushort wMinFieldStrengthLowField;                        /**< Min achieveable fieldstrength in low field region for this component. */
            public fixed ushort wMinFieldStrength[PHDL_MSTAMPOSC_ID_CNT];   /**< Min achieveable fieldstrength with poti for amplifier setting[index]. */
            public fixed ushort wMaxFieldStrength[PHDL_MSTAMPOSC_ID_CNT];   /**< Max achieveable fieldstrength with poti for amplifier setting[index]. */
            public fixed ushort wGainValue[PHDL_MSTAMPOSC_ID_CNT];          /**< Gain value for different Gain Ids. */
            public byte bCurrGainId;                                        /**< Current set Gain Identifer. */
            public byte bUseMaxGain;                                        /**< Flag defining if the maximum gain ID is used (12 A/m). */
            public byte bChannel;                                           /**< Selected osci channel for field strength measurement. */
            public byte bPrecision;                                         /**< Set the required precision for set field strength. */
            public byte bCheckPrecision;                                    /**< Flag indicating if precision has to be checked. */
            public byte bCal;                                               /**< Flag indicating if calibration has been performed. */
            public ushort wFieldRxGainLowerThresh;                          /**< Determines the field strength for the lower threshold of the Rx Gain (standart value -1). */
            public ushort wFieldRxGainUpperThresh;                          /**< Determines the field strength for the upper threshold of the Rx Gain (standart value +1). */
            public byte bRxGainStandard;                                    /**< Sets the standart value for the RxGain. */
            public byte bFindGainVals;                                      /**< Determines whether the best gain values are determined during calibration. */
            public ushort wMinPotiVal;                                      /**< Determines the minimum value for the poti settings of the dlRdFpga. */
            public ushort wMaxPotiVal;                                      /**< Determines the maximum value for the poti settings of the dlRdFpga. */
            public ushort wSNRLimitPotiVal;                                 /**< Determines the SNR limit value for the poti settings of the dlRdFpga. */
            public ushort wMinGainVal;                                      /**< Determines the minimum value for the gain settings of the dlAmp. */
            public ushort wMaxGainVal;                                      /**< Determines the maximum value for the gain settings of the dlAmp. */
            public ushort wMaxFindFieldStrength;                            /**< Determines the maximum value that is tried to be reached during calibration if FIND_GAIN_VALS is active. */
            public ushort wMinOverlap;                                      /**< Determines the minumum value for the field to overlap for the different gain values. */
        };

        #endregion

        #region DLLIMPORTS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_ISOSetup_Init(
            ref DataParams_t m_pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,               /**< [In] Specifies the size of the data parameter structure */
            IntPtr pAlAmpDataParams,   				/**< [In] Pointer to the parameter structure of the alAmp component. */
            IntPtr pAlOsciDataParams,               /**< [In] Pointer to the parameter structure of the alOsci component. */
            IntPtr pDlRdFpgaDataParams,				/**< [In] Pointer to the parameter structure of the dlRdFpga component. */
            byte bChannel,                          /**< [In] Specifies the oscilloscope channel for the field strength signal. */
            ushort wFieldRxGainLowerThresh,         /**< [In] Specifies the upper fieldstrenght threashold for the RxGain. */
            ushort wFieldRxGainUpperThresh,         /**< [In] Specifies the Lower fieldstrenght threashold for the RxGain. */
            byte bRxGainStandard				    /**< [In] Specifies the standart RxGain. */
            );
        #endregion

        #region INIT
        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            dlRdFpga.Generic pDLRdFpga,
            byte bChannel,
            ushort wFieldRxGainLowerThresh,
            ushort wFieldRxGainUpperThresh,
            byte bRxGainStandard
            )
        {
            return phdlMstAmpOsc_ISOSetup_Init(ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pDlAmp.m_pDataParams, pDlOsci.m_pDataParams,
                pDLRdFpga.m_pDataParams,
                bChannel,
            wFieldRxGainLowerThresh,
            wFieldRxGainUpperThresh, bRxGainStandard
            );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public ISOSetup()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~ISOSetup()
        {


            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        public ushort[] MinFieldStrength
        {
            get
            {
                ushort[] MinFieldStrength = new ushort[PHDL_MSTAMPOSC_ID_CNT];
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            MinFieldStrength[i] = pDataParams->wMinFieldStrength[i];
                        }
                    }
                }
                return MinFieldStrength;
            }
            set
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            pDataParams->wMinFieldStrength[i] = value[i];
                        }
                    }
                }
            }
        }
        public ushort[] MaxFieldStrength
        {
            get
            {
                ushort[] MaxFieldStrength = new ushort[PHDL_MSTAMPOSC_ID_CNT];
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            MaxFieldStrength[i] = pDataParams->wMaxFieldStrength[i];
                        }
                    }
                }
                return MaxFieldStrength;
            }
            set
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            pDataParams->wMaxFieldStrength[i] = value[i];
                        }
                    }
                }
            }
        }
        public ushort[] GainValue
        {
            get
            {
                ushort[] GainValue = new ushort[PHDL_MSTAMPOSC_ID_CNT];
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            GainValue[i] = pDataParams->wGainValue[i];
                        }
                    }
                }
                return GainValue;
            }
            set
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_ID_CNT; i++)
                        {
                            pDataParams->wGainValue[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion

    #region MP300
    public class Mp300 : dlMstAmpOsc.Generic
    {
        #region DEFINES

        public enum Config : ushort
        {
            MAX_FIELDSTRENGTH_USED =        (CustomCodes.CONFIG_BEGIN),     /**< Defines if the highest gain setting is used as well. */
            USE_FINE_CONTROL =              (CustomCodes.CONFIG_BEGIN + 1), /**< Defines if the fine control is used */
            DEFAULT_MP300_GAIN =            (CustomCodes.CONFIG_BEGIN + 2), /**< Defines the default gain that is used with MP300 */
            RX_GAIN_TARGET_MODE =           (CustomCodes.CONFIG_BEGIN + 3), /**< Defines the rx gain target mode */
            RX_GAIN_TARGET_MIN_FS =         (CustomCodes.CONFIG_BEGIN + 4), /**< Set the wMinMp300RxGainFS value */
            RX_GAIN_TARGET_MAX_FS =         (CustomCodes.CONFIG_BEGIN + 5), /**< Set the wMaxMp300RxGainFS value */
            RX_GAIN_TARGET_MIN =            (CustomCodes.CONFIG_BEGIN + 6), /**< Defines the rx gain min value */
            RX_GAIN_TARGET_MAX =            (CustomCodes.CONFIG_BEGIN + 7), /**< Defines the rx gain max value */
            FAST_RECALIBRATE =              (CustomCodes.CONFIG_BEGIN + 8), /**< Define if fast recalibrate is used */
            USE_NORMAL_CAL_IF_FAST_FAIL =   (CustomCodes.CONFIG_BEGIN + 9), /**< Define if a normal calibration should be used as fallback if fast calibration fails */
            SENSE_COIL_BALANCE_CHANNEL  =   (CustomCodes.CONFIG_BEGIN + 10),/**< configs the sense coil balance channel */
            CANCELLATION_TOKEN          =   (CustomCodes.CONFIG_BEGIN + 11) /**< configs the cancellation token */
        };

        public enum RxGainTargetMode : ushort
        {
            DISABLED          = 0,
            CONSTANT          = 1,
            LINEAR            = 2,
            LINEAR_LIMIT      = 3,
            LINEAR_LIMIT_UP   = 4,
            LINEAR_LIMIT_DOWN = 5
        };
        #endregion

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                          /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pDlAmpDataParams;             /**< Pointer to the parameter structure of the alAmp component. */
            public IntPtr pDlOsciDataParams;            /**< Pointer to the parameter structure of the alOsci component. */
            public IntPtr pDlRdGainDataParams;          /**< Pointer to the parameter structure of the device supporting setting gain of the reader component. */
            public IntPtr pdGainValues;                 /**< FS for the Gain values. */
            public IntPtr pdFieldstrength;              /**< Array with points where the FS is measured do interpolate the curve */
            public IntPtr pdbH2Gain;                    /**< FS for the Gain values. */
            public IntPtr pdcH2Gain;                    /**< Array with points where the FS is measured do interpolate the curve */
            public IntPtr pddH2Gain;                    /**< FS for the Gain values. */
            public IntPtr pdbGain2H;                    /**< FS for the Gain values. */
            public IntPtr pdcGain2H;                    /**< Array with points where the FS is measured do interpolate the curve */
            public IntPtr pddGain2H;                    /**< FS for the Gain values. */
            public IntPtr pdTempBuffer;                 /**< Buffer used to store tmp values */
            public ushort wMaxFieldstrengthUsed;        /**< Max fieldstrength that is allowed (also used for calibrate). */
            public byte bNumberPoints;                  /**< Number of supplied (and inserted) points maximal used for spline interpolation (full range) */
            public byte bMaxPoints;                     /**< Number of points maximal used for spline interpolation (limited by buffer size) */
            public byte bFieldstrengthValuesSupplied;   /**< If fieldstrength values are supplied this value will be set to 1 */
            public byte bMaxMeasuredInterpolationPoint; /**< Index of the last measured Interpolation point in calibration */
            public byte bChannel;                       /**< Selected osci channel for field strength measurement. */
            public byte bPrecision;                     /**< Set the required precision for set field strength. */
            public byte bCheckPrecision;                /**< Flag indicating if precision has to be checked. */
            public byte bFastRecalibrateEnabled;        /**< Flag indicating if recalibrate use a fast mode and did not switch if the RF field */
            public byte bUseNormalCalIfFastFail;        /**< Flag indicating that in case of the fast recalibrate fails a normal calibration is performed */
            public byte bFineControl;                   /**< Flag indicating if the scope should measure back the fs and adjust the reader to ajive better results */
            public byte bCal;                           /**< Flag indicating if calibration has been performed. */
            public ushort wDefaultMp300Gain;            /**< Saves the default gain the MP300 use (in Permil). */
            public ushort wCurrAmpGain;                 /**< Saves the current gain the Amplifier. */
            public ushort wMinAmpGain;                  /**< Saves the minimal gain of the Amplifier. */
            public byte wRxGainMode;                    /**< Mode used for set rx gain target. */
            public ushort wMinMp300RxGainFS;            /**< fs point were the wMinMp300RxGain value should be achived. */
            public ushort wMaxMp300RxGainFS;            /**< fs point were the wMaxMp300RxGain value should be achived. */
            public ushort wMinMp300RxGain;              /**< gain target at fs = wMinMp300RxGainFS. */
            public ushort wMaxMp300RxGain;              /**< gain target at fs = wMaxMp300RxGainFS. */
            public byte bSenseCoilBalanceChannel;       /**< if the sense coil balance channel is set it is checked if the probes are connected in a wrong way. 0 indicated that this is not used. This is done by checking that the channel is not higher than wMaxFieldstrengthUsed and that the cal coil is > sense coil starting from bSenseCoilBalanceCheckLimit */
            public ushort bSenseCoilBalanceCheckLimit;  /**< starting from this fieldstrength it is checked that the sense coil < cal coil */
            public byte bCancellationToken;             /**< if set to one the calibration operation should be canceled. */
        };

        #endregion

        #region DLLIMPORTS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_Mp300_Init(
            ref DataParams_t m_pDataParams,         /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,               /**< [In] Specifies the size of the data parameter structure */
            IntPtr pAlAmpDataParams,   				/**< [In] Pointer to the parameter structure of the alAmp component. */
            IntPtr pAlOsciDataParams,               /**< [In] Pointer to the parameter structure of the alOsci component. */
            IntPtr pDlRdGainDataParams,				/**< [In] Pointer to the parameter structure of the device supporting setting gain of the reader component. */
            byte bChannel,                          /**< [In] Specifies the oscilloscope channel for the field strength signal. */
            byte bNumberPoints,                     /**< [In] Number of points unsed do interpolate the caracteristic curve */
            IntPtr pGainValues,                     /**< [In] Gain values that should be used for calibrate or are the points where the fieldstrength is already measured. If it is NULL bNumberPoints will be used linear (this value can be deleted after init) */
            IntPtr pFieldstrengthValues,            /**< [In] Fieldstrenght values. If they are NULL a calibration will be done. Else this values will be used for interpolation (this value can be deleted after init) */
            IntPtr pSplineParameterStorage,         /**< [In] Array where all spline parameters are stored. This array must have a size of at least (18*bNumberPoints). (bNumberPoints*(Gain + FS + 6Parameter + Tmp)*2(to allow insert of new points) */
            ushort wSizeOfSplineParameterStorage    /**< [In] size of supplied storage (in bytes) */
            );
        #endregion

        #region INIT

        private double[] m_bSplineParameterStorage;
        private GCHandle m_pSplineParameterStorage;

        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            object pMp300,
            byte bChannel
            )
        {
            /* AR75A250 use this 13 points */
            if (pDlAmp != null && pDlAmp.GetType() == typeof(dlAmp.AR75A250))
            {
                double[] points = new double[] {0, 500, 1000, 1500, 2000, 2500, 3000, 3500, 3700, 3850, 3950, 4025, 4095};
                return Init(pDlAmp, pDlOsci, pMp300, bChannel, points);
            }
            return Init(pDlAmp, pDlOsci, pMp300, bChannel, 11);
        }

        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            object pMp300,
            byte bChannel,
            byte bNumberPoints
            )
        {
            // Free Buffers
            if (this.m_pSplineParameterStorage.IsAllocated)
            {
                this.m_pSplineParameterStorage.Free();
            }

            // Allocate buffers
            m_bSplineParameterStorage = new double[(18 * bNumberPoints)];

            // Link given buffers
            m_pSplineParameterStorage = GCHandle.Alloc(m_bSplineParameterStorage, GCHandleType.Pinned);

            var DlReaderGainDataParamsIntPtr = System.IntPtr.Zero;
            if (pMp300 is Hal.Mp300)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.Mp300) pMp300).m_pDataParams;
            }
            else if (pMp300 is Hal.ProxiLAB)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.ProxiLAB) pMp300).m_pDataParams;
            }
            else if (pMp300 is dlRdFpga.Generic)
            {
                DlReaderGainDataParamsIntPtr = ((dlRdFpga.Generic) pMp300).m_pDataParams;
            }

            // Call init function
            return phdlMstAmpOsc_Mp300_Init(ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                (pDlAmp != null) ? pDlAmp.m_pDataParams : System.IntPtr.Zero,
                (pDlOsci != null) ? pDlOsci.m_pDataParams : System.IntPtr.Zero,
                DlReaderGainDataParamsIntPtr,
                bChannel,
                bNumberPoints,
                System.IntPtr.Zero,
                System.IntPtr.Zero,
                m_pSplineParameterStorage.AddrOfPinnedObject(),
                (ushort)(Marshal.SizeOf(typeof(double)) * m_bSplineParameterStorage.Length)
            );
        }

        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            object pMp300,
            byte bChannel,
            double[] pGainValues
            )
        {
            GCHandle m_pGainValues;

            // Free Buffers
            if (this.m_pSplineParameterStorage.IsAllocated)
            {
                this.m_pSplineParameterStorage.Free();
            }

            // Allocate buffers
            m_bSplineParameterStorage = new double[(18 * pGainValues.Length)];

            // Link given buffers
            m_pSplineParameterStorage = GCHandle.Alloc(m_bSplineParameterStorage, GCHandleType.Pinned);
            m_pGainValues = GCHandle.Alloc(pGainValues, GCHandleType.Pinned);

            var DlReaderGainDataParamsIntPtr = System.IntPtr.Zero;
            if (pMp300 is Hal.Mp300)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.Mp300)pMp300).m_pDataParams;
            }
            else if (pMp300 is Hal.ProxiLAB)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.ProxiLAB)pMp300).m_pDataParams;
            }
            else if (pMp300 is dlRdFpga.Generic)
            {
                DlReaderGainDataParamsIntPtr = ((dlRdFpga.Generic)pMp300).m_pDataParams;
            }

            // Call init function
            Status_t status = phdlMstAmpOsc_Mp300_Init(ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                (pDlAmp != null) ? pDlAmp.m_pDataParams : System.IntPtr.Zero,
                (pDlOsci != null) ? pDlOsci.m_pDataParams : System.IntPtr.Zero,
                DlReaderGainDataParamsIntPtr,
                bChannel,
                (byte)pGainValues.Length,
                m_pGainValues.AddrOfPinnedObject(),
                IntPtr.Zero,
                m_pSplineParameterStorage.AddrOfPinnedObject(),
                (ushort)(Marshal.SizeOf(typeof(double)) * m_bSplineParameterStorage.Length)
            );

            // Free Buffer
            if (m_pGainValues.IsAllocated)
            {
                m_pGainValues.Free();
            }

            return status;
        }

        public Status_t Init(
            dlAmp.Generic pDlAmp,
            dlOsci.Generic pDlOsci,
            object pMp300,
            byte bChannel,
            double [] pGainValues,
            double [] pFieldstrengthValues
            )
        {
            GCHandle m_pGainValues;
            GCHandle m_pFieldstrengthValues;

            if (pGainValues == null || pFieldstrengthValues == null)
            {
                return new Status_t(Error_CompCode.DL_MSTAMPOSC, Error_Param.INVALID_PARAMETER);
            }

            if (pGainValues.Length != pFieldstrengthValues.Length)
            {
                return new Status_t(Error_CompCode.DL_MSTAMPOSC, Error_Param.INVALID_PARAMETER);
            }

            // Free Buffers
            if (this.m_pSplineParameterStorage.IsAllocated)
            {
                this.m_pSplineParameterStorage.Free();
            }

            // Allocate buffers
            m_bSplineParameterStorage = new double[(18 * pGainValues.Length)];

            // Link given buffers
            m_pSplineParameterStorage = GCHandle.Alloc(m_bSplineParameterStorage, GCHandleType.Pinned);
            m_pGainValues = GCHandle.Alloc(pGainValues, GCHandleType.Pinned);
            m_pFieldstrengthValues = GCHandle.Alloc(pFieldstrengthValues, GCHandleType.Pinned);

            var DlReaderGainDataParamsIntPtr = System.IntPtr.Zero;
            if (pMp300 is Hal.Mp300)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.Mp300)pMp300).m_pDataParams;
            }
            else if (pMp300 is Hal.ProxiLAB)
            {
                DlReaderGainDataParamsIntPtr = ((Hal.ProxiLAB)pMp300).m_pDataParams;
            }
            else if (pMp300 is dlRdFpga.Generic)
            {
                DlReaderGainDataParamsIntPtr = ((dlRdFpga.Generic)pMp300).m_pDataParams;
            }

            // Call init function
            Status_t status = phdlMstAmpOsc_Mp300_Init(ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                (pDlAmp != null) ? pDlAmp.m_pDataParams : System.IntPtr.Zero,
                (pDlOsci != null) ? pDlOsci.m_pDataParams : System.IntPtr.Zero,
                DlReaderGainDataParamsIntPtr,
                bChannel,
                (byte)pGainValues.Length,
                m_pGainValues.AddrOfPinnedObject(),
                m_pFieldstrengthValues.AddrOfPinnedObject(),
                m_pSplineParameterStorage.AddrOfPinnedObject(),
                (ushort)(Marshal.SizeOf(typeof(double)) * m_bSplineParameterStorage.Length)
            );

            // Free Buffer
            if (m_pGainValues.IsAllocated)
            {
                m_pGainValues.Free();
            }
            if (m_pFieldstrengthValues.IsAllocated)
            {
                m_pFieldstrengthValues.Free();
            }

            return status;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Mp300()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Mp300()
        {
            // Free Buffers
            if (this.m_pSplineParameterStorage.IsAllocated)
            {
                this.m_pSplineParameterStorage.Free();
            }

            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        public ushort[] GainValues
        {
            get
            {
                ushort[] GainValues = new ushort[DataParams.bNumberPoints];
                unsafe
                {
                    double* pGain = (double*)DataParams.pdGainValues.ToPointer();
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < DataParams.bNumberPoints; i++)
                        {
                            GainValues[i] = (ushort)pGain[i];
                        }
                    }
                }
                return GainValues;
            }
            set
            {
                unsafe
                {
                    double* pGain = (double*)DataParams.pdGainValues.ToPointer();
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < DataParams.bNumberPoints; i++)
                        {
                            pGain[i] = value[i];
                        }
                    }
                }
            }
        }

        public ushort[] FieldStrengthValues
        {
            get
            {
                ushort[] FieldStrengthValues = new ushort[DataParams.bNumberPoints];
                unsafe
                {
                    double* pFieldStrengthValues = (double*)DataParams.pdFieldstrength.ToPointer();
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < DataParams.bNumberPoints; i++)
                        {
                            FieldStrengthValues[i] = (ushort)pFieldStrengthValues[i];
                        }
                    }
                }
                return FieldStrengthValues;
            }
            set
            {
                unsafe
                {
                    double* pFieldStrengthValues = (double*)DataParams.pdFieldstrength.ToPointer();
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < DataParams.bNumberPoints; i++)
                        {
                            pFieldStrengthValues[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion

    #region ISO3
    public class ISO3 : dlMstAmpOsc.Generic
    {
        #region DEFINES
        public const byte PHDL_MSTAMPOSC_CONF_CNT = 6;

        public enum StepAttConfig : int
        {
            STEP_ATT_0DB = 0,
            STEP_ATT_6DB = 1,
            STEP_ATT_12DB = 2,
            STEP_ATT_18DB = 3,
            STEP_ATT_24DB = 4,
            STEP_ATT_CURRENT = 5
        }

        public enum Config : ushort
        {
            STEP_ATT_0DB = (CustomCodes.CONFIG_BEGIN),    /**< Step attenuator 0db damping bits. */
            STEP_ATT_6DB,                                 /**< Step attenuator 6db damping bits. */
            STEP_ATT_12DB,                                /**< Step attenuator 12db damping bits. */
            STEP_ATT_18DB,                                /**< Step attenuator 18db damping bits. */
            STEP_ATT_24DB,                                /**< Step attenuator 24db damping bits.  */
            STEP_ATT_CURRENT,                             /**< Step attenuator current damping. */
            DAC_GAIN,                                     /**< Gain DAC multiplicator. */
            CUR_FIELD,                                    /**< read only - current field strength in mA/m. */
            CUR_ANTENNA_TYPE,                             /**< Type of currently used antenna. */
            RX_THRESHOLD_TARGET_MODE,                     /**< Defines the rx threshold target mode */
            RX_THRESHOLD_TARGET_MIN_FS,                   /**< Set the wMinRxThresholdFS value */
            RX_THRESHOLD_TARGET_MAX_FS,                   /**< Set the wMaxRxThresholdFS value */
            RX_THRESHOLD_TARGET_MIN,                      /**< Defines the rx threshold min value */
            RX_THRESHOLD_TARGET_MAX                       /**< Defines the rx threshold max value */
        };

        public enum RxThresholdTargetMode : ushort
        {
            DISABLED = 0,
            CONSTANT = 1,
            LINEAR = 2,
            LINEAR_LIMIT = 3,
            LINEAR_LIMIT_UP = 4,
            LINEAR_LIMIT_DOWN = 5
        };

        public enum Antenna : byte
        {
            Type_1 = 0,
            Type_2 = 1,
            Type_3 = 2
        }

        #endregion

        #region DATA_STRUCTURE
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                              /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pHalIso3DataParams;                               /**< Pointer to the parameter structure of the hal iso3 component. */
            public ushort wMaxFieldStrength;                                /**< Max field strength in mA/m from last calibration or configured limit */
            public ushort wMaxFieldStrengthConfigured;                      /**< Max field strength in mA/m that was configurd as limit */
            public ushort wMinFieldStrength;                                /**< Min field strength in mA/m from last calibration */
            public uint dwDacGain;                                          /**< Gain factor for envelope to control field strength */
            public uint dwFieldStrengthRatioA1;                             /**< To convert measured amplitude into field strength values for antenna type 1 */
            public uint dwFieldStrengthRatioA2;                             /**< To convert measured amplitude into field strength values for antenna type 2 */
            public uint dwFieldStrengthRatioA3;                             /**< To convert measured amplitude into field strength values for antenna type 3 */
            public fixed ushort wStepAttConfigs[PHDL_MSTAMPOSC_CONF_CNT];   /**< set commands for every attenuation value */
            public byte bCurrentAntennaType;                                /**< Type of antenna currently used in our ISO setup V3 */
            public byte bRxThresholdMode;                                   /**< Mode used for set rx threshold target. */
            public ushort wMinRxThresholdFS;                                /**< fs point were the wMinRxThreshold value should be achived. */
            public ushort wMaxRxThresholdFS;                                /**< fs point were the wMaxRxThreshold value should be achived. */
            public ushort wMinRxThreshold;                                  /**< threshold target at fs = wMinRxThresholdFS. */
            public ushort wMaxRxThreshold;                                  /**< threshold target at fs = wMaxRxThresholdFS. */
        };
        #endregion

        #region DLLIMPORTS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_ISO3_Init(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,                   /**< [In] Specifies the size of the data parameter structure */
            IntPtr pAlAmpDataParams                     /**< [In] Pointer to the parameter structure of the hal iso3 component. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_ISO3_Cmd_SetFieldStrengthRatio(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            uint dwFieldStrengthRatio,                  /**< [In] Field strength ratio */
            byte bAntennaType                           /**< [In] Antenna type */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlMstAmpOsc_ISO3_Cmd_GetFieldStrengthRatio(
            ref DataParams_t m_pDataParams,             /**< [In] Pointer to this layers parameter structure. */
            ref uint dwFieldStrengthRatio,              /**< [In] Field strength ratio */
            byte bAntennaType                           /**< [In] Antenna type */
            );
        #endregion

        #region INIT
        public Status_t Init(
            Hal.ISO3 pHalIso3
            )
        {
            return phdlMstAmpOsc_ISO3_Init(ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pHalIso3.m_pDataParams);
        }

       #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Cmd_SetFieldStrengthRatio(
            uint fieldStrengthRatio,
            byte antennaType
            )
        {
            return phdlMstAmpOsc_ISO3_Cmd_SetFieldStrengthRatio(
                ref m_DataParamsInt[0], fieldStrengthRatio, antennaType);
        }

        public Status_t Cmd_GetFieldStrengthRatio(
            out uint fieldStrengthRatio,
            byte antennaType
            )
        {
            Status_t status;
            uint wTemp = 0;
            status = phdlMstAmpOsc_ISO3_Cmd_GetFieldStrengthRatio(ref m_DataParamsInt[0], ref wTemp, antennaType);
            fieldStrengthRatio = wTemp;
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public ISO3()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~ISO3()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        public ushort[] StepAttConfigs
        {
            get
            {
                ushort[] MaxFieldStrength = new ushort[PHDL_MSTAMPOSC_CONF_CNT];
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_CONF_CNT; i++)
                        {
                            StepAttConfigs[i] = pDataParams->wStepAttConfigs[i];
                        }
                    }
                }
                return MaxFieldStrength;
            }
            set
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < PHDL_MSTAMPOSC_CONF_CNT; i++)
                        {
                            pDataParams->wStepAttConfigs[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion
}
#endif
