/*
 * Copyright 2013 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet
{
    public class Log
    {
        #region DEFINES

        private const byte DATATYPE_BUFFER = 0x00;
        private const byte DATATYPE_VALUE = 0x01;

        public enum eLogType
        {
            INFO = 0x00,    /**< Log Type : Info */
            ERROR,          /**< Log Type : Error */
            WARN,           /**< Log Type : Warn */
            DEBUG           /**< Log Type : Debug */
        };

        public enum eLogOption
        {
            CATEGORY_ENTER = 0x01,  /**< Execute Option: Logging takes place at function entry. */
            CATEGORY_GEN = 0x02,    /**< Execute Option: Logging takes place in the middle of a function. */
            CATEGORY_LEAVE = 0x03   /**< Execute Option: Logging takes place before leaving the function. */
        };

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Definition of a singe Log entry.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct LogEntry_t
        {
            public string pString;
            public byte[] pData;
            public eLogType bLogType;
        };

        /// <summary>
        /// Definition of a singe Log entry (internal).
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct LogEntryInt_t
        {
            public byte bLogType;
            public IntPtr pString;
            public IntPtr pData;
            public ushort wDataLen;
            public byte bDataType;
        };

        /// <summary>
        /// Definition of a Register Entry (internal).
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        private struct RegisterEntryInt_t
        {
            public IntPtr pDataParams;
            public IntPtr pLogEntries;
            public ushort wNumEntries;
            public ushort wMaxEntries;
            public ushort wLogEntriesStart;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phLog_Init(
            IntPtr pLogCallback,        /**< [In] Pointer to a log callback function. */
            IntPtr pRegisterEntries,    /**< [In] An array of \ref phLog_RegisterEntry_t memory. */
            ushort wNumRegisterEntries  /**< [In] Size of the given \ref phLog_RegisterEntry_t array. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phLog_Register(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            IntPtr pLogEntries,    /**< [In] An array of \ref phLog_LogEntry_t memory. */
            ushort wNumEntries  /**< [In] Size of the given \ref phLog_LogEntry_t array. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddString(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pString      /**< [In] The Null-terminated string to add. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddString(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            byte[] pString      /**< [In] The Null-terminated string to add. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint8(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint8(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            byte[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint16(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint16(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            byte[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint32(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Uint32(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            byte[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Buffer(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam,    /**< [In] Pointer to the memory where the parameter resides. */
            ushort wLength      /**< [In] The length in bytes to the supplied buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_AddParam_Buffer(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            byte[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam,    /**< [In] Pointer to the memory where the parameter resides. */
            ushort wLength      /**< [In] The length in bytes to the supplied buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern void phLog_Execute(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bOption        /**< [In] Option byte (one of the PH_LOG_OPTION* defines). */
            );

        #endregion

        #region DELEGATES

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void CallbackDelegate(
            IntPtr pDataParams,
            eLogOption bOption,
            LogEntry_t[] pLogEntries
            );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        private delegate void CallbackDelegateInt(
            IntPtr pDataParams,
            byte bOption,
            IntPtr pLogEntries,
            ushort wLogEntries
            );

        #endregion

        #region CALLBACKS

        private void CallbackInt(
            IntPtr pDataParams,
            byte bOption,
            IntPtr pLogEntries,
            ushort wLogEntries
            )
        {
            byte value;
            int offset;


            // Create temporary LogEntry array
            LogEntry_t[] logEntries = new LogEntry_t[wLogEntries];

            // Perform the following for all entries
            for (int index = 0; index < wLogEntries; index++)
            {
                LogEntryInt_t logEntry = (LogEntryInt_t)Marshal.PtrToStructure(pLogEntries, typeof(LogEntryInt_t));
                pLogEntries = new IntPtr(pLogEntries.ToInt64() + Marshal.SizeOf(typeof(LogEntryInt_t)));

                // Copy the string
                offset = 0;
                do
                {
                    value = Marshal.ReadByte(logEntry.pString, offset++);
                    if (value != 0)
                    {
                        logEntries[index].pString += (char)value;
                    }
                }
                while (value != 0);

                // Copy the data
                logEntries[index].pData = new byte[(logEntry.pData == IntPtr.Zero) ? 0 :(int)logEntry.wDataLen];
                if (logEntry.bDataType == DATATYPE_BUFFER)
                {
                    for (offset = 0; offset < logEntries[index].pData.Length; offset++)
                    {
                        logEntries[index].pData[offset] = Marshal.ReadByte(logEntry.pData, offset);
                    }
                }
                else
                {
                    /* assume we are using little endian. */
                    value = 0;
                    for (offset = logEntries[index].pData.Length; offset > 0; offset--)
                    {
                        logEntries[index].pData[value++] = Marshal.ReadByte(logEntry.pData, offset - 1);
                    }
                }

                // Copy the log type
                logEntries[index].bLogType = (eLogType)logEntry.bLogType;
            }

            // Execute external Log Callback
            this.m_LogCallback(pDataParams, (eLogOption)bOption, logEntries);
        }

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public void AddString(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pString      /**< [In] The Null-terminated string to add. */
            )
        {
            phLog_AddString(pDataParams, bLogType, pString);
        }

        public void AddString(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            string pString      /**< [In] The Null-terminated string to add. */
            )
        {
            if (!GCHandleDict.ContainsKey(pDataParams))
                GCHandleDict[pDataParams] = new List<GCHandle>();
            // Allocate internal data parameters and pointer to them
            this.m_StringBufferInt = new byte[pString.Length + 1];
            for (int i = 0; i < pString.Length; i++)
            {
                this.m_StringBufferInt[i] = (byte)pString[i];
            }
            this.m_StringBufferInt[pString.Length] = (byte)0;
            GCHandle m_pStringBufferInt = GCHandle.Alloc(this.m_StringBufferInt, GCHandleType.Pinned); /* Just alloc handle do avoid GC */
            GCHandleDict[pDataParams].Add(m_pStringBufferInt);
            phLog_AddString(pDataParams, bLogType, m_StringBufferInt);
        }

        public void AddParam_Uint8(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            phLog_AddParam_Uint8(pDataParams, bLogType, pName, ref pParam);
        }

        public void AddParam_Uint8(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            string pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            if (!GCHandleDict.ContainsKey(pDataParams))
                GCHandleDict[pDataParams] = new List<GCHandle>();
            // Allocate internal data parameters and pointer to them
            this.m_StringBufferInt = new byte[pName.Length + 1];
            for (int i = 0; i < pName.Length; i++)
            {
                this.m_StringBufferInt[i] = (byte)pName[i];
            }
            this.m_StringBufferInt[pName.Length] = (byte)0;
            GCHandle m_pStringBufferInt = GCHandle.Alloc(this.m_StringBufferInt, GCHandleType.Pinned); /* Just alloc handle do avoid GC */
            GCHandleDict[pDataParams].Add(m_pStringBufferInt);
            phLog_AddParam_Uint8(pDataParams, bLogType, m_StringBufferInt, ref pParam);
        }

        public void AddParam_Uint16(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            phLog_AddParam_Uint16(pDataParams, bLogType, pName, ref pParam);
        }

        public void AddParam_Uint16(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            string pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            if (!GCHandleDict.ContainsKey(pDataParams))
                GCHandleDict[pDataParams] = new List<GCHandle>();
            // Allocate internal data parameters and pointer to them
            this.m_StringBufferInt = new byte[pName.Length + 1];
            for (int i = 0; i < pName.Length; i++)
            {
                this.m_StringBufferInt[i] = (byte)pName[i];
            }
            this.m_StringBufferInt[pName.Length] = (byte)0;
            GCHandle m_pStringBufferInt = GCHandle.Alloc(this.m_StringBufferInt, GCHandleType.Pinned); /* Just alloc handle do avoid GC */
            GCHandleDict[pDataParams].Add(m_pStringBufferInt);
            phLog_AddParam_Uint16(pDataParams, bLogType, m_StringBufferInt, ref pParam);
        }

        public void AddParam_Uint32(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            phLog_AddParam_Uint32(pDataParams, bLogType, pName, ref pParam);
        }

        public void AddParam_Uint32(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            string pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam     /**< [In] Pointer to the memory where the parameter resides. */
            )
        {
            if (!GCHandleDict.ContainsKey(pDataParams))
                GCHandleDict[pDataParams] = new List<GCHandle>();
            // Allocate internal data parameters and pointer to them
            this.m_StringBufferInt = new byte[pName.Length + 1];
            for (int i = 0; i < pName.Length; i++)
            {
                this.m_StringBufferInt[i] = (byte)pName[i];
            }
            this.m_StringBufferInt[pName.Length] = (byte)0;
            GCHandle m_pStringBufferInt = GCHandle.Alloc(this.m_StringBufferInt, GCHandleType.Pinned); /* Just alloc handle do avoid GC */
            GCHandleDict[pDataParams].Add(m_pStringBufferInt);
            phLog_AddParam_Uint32(pDataParams, bLogType, m_StringBufferInt, ref pParam);
        }

        public void AddParam_Buffer(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            char[] pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam,    /**< [In] Pointer to the memory where the parameter resides. */
            ushort wLength      /**< [In] The length in bytes to the supplied buffer. */
            )
        {
            phLog_AddParam_Buffer(pDataParams, bLogType, pName, ref pParam, wLength);
        }

        public void AddParam_Buffer(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            byte bLogType,      /**< [In] Type of Entry (one of the \c PHLOG_LOG_TYPE_* values). */
            string pName,       /**< [In] The Null-terminated name of the parameter. */
            ref byte pParam,    /**< [In] Pointer to the memory where the parameter resides. */
            ushort wLength      /**< [In] The length in bytes to the supplied buffer. */
            )
        {
            if (!GCHandleDict.ContainsKey(pDataParams))
                GCHandleDict[pDataParams] = new List<GCHandle>();
            // Allocate internal data parameters and pointer to them
            this.m_StringBufferInt = new byte[pName.Length + 1];
            for (int i = 0; i < pName.Length; i++)
            {
                this.m_StringBufferInt[i] = (byte)pName[i];
            }
            this.m_StringBufferInt[pName.Length] = (byte)0;
            GCHandle m_pStringBufferInt = GCHandle.Alloc(this.m_StringBufferInt, GCHandleType.Pinned); /* Just alloc handle do avoid GC */
            GCHandleDict[pDataParams].Add(m_pStringBufferInt);
            phLog_AddParam_Buffer(pDataParams, bLogType, m_StringBufferInt, ref pParam, wLength);
        }

        public void Execute(
            IntPtr pDataParams, /**< [In] The DataParams of the calling function. */
            eLogOption bOption  /**< [In] Option byte (one of the PH_LOG_OPTION* defines). */
            )
        {
            phLog_Execute(pDataParams, (byte)bOption);
            if (!GCHandleDict.ContainsKey(pDataParams))
            {
                foreach (GCHandle handle in GCHandleDict[pDataParams])
                {
                    // Free allocated pointer to data params
                    if (handle.IsAllocated)
                    {
                        handle.Free();
                    }
                }
                GCHandleDict[pDataParams].Clear();
            }
        }

        #endregion

        #region MEMORY_MAPPING

        private byte[] m_StringBufferInt = null;
        private Dictionary<IntPtr, List<GCHandle>> GCHandleDict = new Dictionary<IntPtr, List<GCHandle>>();

        private RegisterEntryInt_t[] m_RegisterEntriesInt;
        private GCHandle m_pRegisterEntriesInt;

        private GCHandle[] m_pLogEntriesArrays;
        private int m_NumLogEntriesArrays;

        private CallbackDelegateInt m_LogCallbackInt;
        private CallbackDelegate m_LogCallback;

        public Status_t Init(
            CallbackDelegate LogCallback,   /**< [In] Pointer to a log callback function. */
            ushort NumRegisterEntries       /**< [In] Size of the desired \ref phLog_RegisterEntry_t array. */
            )
        {
            IntPtr pLogCallbackInt;
            IntPtr pRegisterEntriesInt;

            if (LogCallback != null)
            {
                // Free allocated pointer to data params
                if (this.m_pRegisterEntriesInt.IsAllocated)
                {
                    this.m_pRegisterEntriesInt.Free();
                }

                // Free allocated pointer to logEntries
                if (this.m_pLogEntriesArrays != null)
                {
                    foreach (GCHandle logEntries in this.m_pLogEntriesArrays)
                    {
                        if (logEntries.IsAllocated)
                        {
                            logEntries.Free();
                        }
                    }
                }

                // Allocate internal data parameters and pointer to them
                this.m_RegisterEntriesInt = new RegisterEntryInt_t[NumRegisterEntries];
                this.m_pRegisterEntriesInt = GCHandle.Alloc(this.m_RegisterEntriesInt, GCHandleType.Pinned);

                // Allocate LogEntry Array
                this.m_pLogEntriesArrays = new GCHandle[NumRegisterEntries];
                this.m_NumLogEntriesArrays = 0;

                // Set pointers
                pLogCallbackInt = Marshal.GetFunctionPointerForDelegate(m_LogCallbackInt);
                pRegisterEntriesInt = m_pRegisterEntriesInt.AddrOfPinnedObject();
            }
            else
            {
                // Set zero pointers
                pLogCallbackInt = IntPtr.Zero;
                pRegisterEntriesInt = IntPtr.Zero;
                NumRegisterEntries = 0;
            }

            // Assign external callback
            this.m_LogCallback = LogCallback;

            // Execute init function
            return phLog_Init(
                pLogCallbackInt,
                pRegisterEntriesInt,
                NumRegisterEntries);
        }

        public Status_t Register(
            IntPtr nativePointer,   /**< [In] Component for which logging shall be performed. */
            int NumEntries      /**< [In] Size of the given \ref phLog_LogEntry_t array. */
            )
        {
            // Create and pin new entry array
            if (this.m_NumLogEntriesArrays < this.m_pLogEntriesArrays.Length)
            {
                this.m_pLogEntriesArrays[m_NumLogEntriesArrays] = GCHandle.Alloc(new LogEntryInt_t[NumEntries], GCHandleType.Pinned);
            }

            // Call register function
            Status_t status = phLog_Register(
                nativePointer,
                this.m_pLogEntriesArrays[m_NumLogEntriesArrays].AddrOfPinnedObject(),
                (ushort)NumEntries);

            // Increment Array count
            if (status.Equals(new Status_t()))
            {
                ++m_NumLogEntriesArrays;
            }

            return status;
        }

        public Status_t Register(
            object Component,   /**< [In] Component for which logging shall be performed. */
            int NumEntries      /**< [In] Size of the given \ref phLog_LogEntry_t array. */
            )
        {
            Status_t status;
            IntPtr nativePointer;

            #region BAL
            if (Component.GetType().BaseType == typeof(Bal.Generic))
            {
                nativePointer = ((Bal.Generic)Component).m_pDataParams;
            }
            #endregion BAL

            #region HAL
            else if (Component.GetType().BaseType == typeof(Hal.Generic))
            {
                nativePointer = ((Hal.Generic)Component).m_pDataParams;
            }
            #endregion HAL

#if PACKAGE_INTERNAL
            #region CONTACT_HAL
            else if (Component.GetType().BaseType == typeof(HalContact.Generic))
                        {
                            nativePointer = ((HalContact.Generic)Component).m_pDataParams;
            }
            #endregion CONTACT_HAL
#endif

            #region PAL
            else if ( Component.GetType ().BaseType == typeof ( palI14443p3a.Generic ) )
            {
                nativePointer = ( ( palI14443p3a.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palI14443p3b.Generic ) )
            {
                nativePointer = ( ( palI14443p3b.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palI14443p4a.Generic ) )
            {
                nativePointer = ( ( palI14443p4a.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palI14443p4.Generic ) )
            {
                nativePointer = ( ( palI14443p4.Generic ) Component ).m_pDataParams;
            }
#if PACKAGE_INTERNAL
            else if (Component.GetType().BaseType == typeof(palI7816p4.Generic))
            {
                nativePointer = ((palI7816p4.Generic)Component).m_pDataParams;
            }
            else if (Component.GetType().BaseType == typeof(palI7816p4a.Generic))
            {
                nativePointer = ((palI7816p4a.Generic)Component).m_pDataParams;
            }
#endif
            else if ( Component.GetType ().BaseType == typeof ( palMifare.Generic ) )
            {
                nativePointer = ( ( palMifare.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palSli15693.Generic ) )
            {
                nativePointer = ( ( palSli15693.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palI18000p3m3.Generic ) )
            {
                nativePointer = ( ( palI18000p3m3.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palI18092mPI.Generic ) )
            {
                nativePointer = ( ( palI18092mPI.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palEpcUid.Generic ) )
            {
                nativePointer = ( ( palEpcUid.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( palFelica.Generic ) )
            {
                nativePointer = ( ( palFelica.Generic ) Component ).m_pDataParams;
            }
            #endregion PAL

            #region AL
            else if ( Component.GetType ().BaseType == typeof ( alMful.Generic ) )
            {
                nativePointer = ( ( alMful.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfc.Generic ) )
            {
                nativePointer = ( ( alMfc.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfp.Generic ) )
            {
                nativePointer = ( ( alMfp.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfpEVx.Generic ) )
            {
                nativePointer = ( ( alMfpEVx.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfdf.Generic ) )
            {
                nativePointer = ( ( alMfdf.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfdfEVx.Generic ) )
            {
                nativePointer = ( ( alMfdfEVx.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfDuoX.Generic ) )
            {
                nativePointer = ( ( alMfDuoX.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alNtagXDna.Generic ) )
            {
                nativePointer = ( ( alNtagXDna.Generic ) Component ).m_pDataParams;
            }
#if PACKAGE_INTERNAL
            else if ( Component.GetType ().BaseType == typeof ( alMfIdentity.Generic ) )
            {
                nativePointer = ( ( alMfIdentity.Generic ) Component ).m_pDataParams;
            }
#endif

            else if ( Component.GetType ().BaseType == typeof ( alMfdfLight.Generic ) )
            {
                nativePointer = ( ( alMfdfLight.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMifarePrimeNtag.Generic ) )
            {
                nativePointer = ( ( alMifarePrimeNtag.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alMfNtag42XDna.Generic ) )
            {
                nativePointer = ( ( alMfNtag42XDna.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alICode.Generic ) )
            {
                nativePointer = ( ( alICode.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alVca.Generic ) )
            {
                nativePointer = ( ( alVca.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alFelica.Generic ) )
            {
                nativePointer = ( ( alFelica.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alI18000p3m3.Generic ) )
            {
                nativePointer = ( ( alI18000p3m3.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alP40CmdPriv.Generic ) )
            {
                nativePointer = ( ( alP40CmdPriv.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( alP40CmdPub.Generic ) )
            {
                nativePointer = ( ( alP40CmdPub.Generic ) Component ).m_pDataParams;
            }
            #endregion AL

#if PACKAGE_INTERNAL
            #region DL
            else if ( Component.GetType ().BaseType == typeof ( dlAmp.Generic ) )
            {
                nativePointer = ( ( dlAmp.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( dlMstAmpOsc.Generic ) )
            {
                nativePointer = ( ( dlMstAmpOsc.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( dlOsci.Generic ) )
            {
                nativePointer = ( ( dlOsci.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( dlRdFpga.Generic ) )
            {
                nativePointer = ( ( dlRdFpga.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( dlThstrm.Generic ) )
            {
                nativePointer = ( ( dlThstrm.Generic ) Component ).m_pDataParams;
            }
            else if (Component.GetType().BaseType == typeof(dlStepper.Generic))
            {
                nativePointer = ((dlStepper.Generic)Component).m_pDataParams;
            }
            #endregion DL
#endif

            #region Common
            else if ( Component.GetType ().BaseType == typeof ( CidManager.Generic ) )
            {
                nativePointer = ( ( CidManager.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( CryptoRng.Generic ) )
            {
                nativePointer = ( ( CryptoRng.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( CryptoSym.Generic ) )
            {
                nativePointer = ( ( CryptoSym.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( CryptoASym.Generic ) )
            {
                nativePointer = ( ( CryptoASym.Generic ) Component ).m_pDataParams;
            }
            else if ( Component.GetType ().BaseType == typeof ( KeyStore.Generic ) )
            {
                nativePointer = ( ( KeyStore.Generic ) Component ).m_pDataParams;
            }
            #endregion Common

            else
            {
                return new Status_t(Error_CompCode.LOG, Error_Param.INVALID_PARAMETER);
            }

            // Create and pin new entry array
            if (this.m_NumLogEntriesArrays < this.m_pLogEntriesArrays.Length)
            {
                this.m_pLogEntriesArrays[m_NumLogEntriesArrays] = GCHandle.Alloc(new LogEntryInt_t[NumEntries], GCHandleType.Pinned);
            }

            // Call register function
            status = phLog_Register(
                nativePointer,
                this.m_pLogEntriesArrays[m_NumLogEntriesArrays].AddrOfPinnedObject(),
                (ushort)NumEntries);

            // Increment Array count
            if (status.Equals(new Status_t()))
            {
                ++m_NumLogEntriesArrays;
            }

            return status;
        }

        /// <summary>
        /// Create the class instance.
        /// </summary>
        public Log()
        {
            // Assign internal callback
            this.m_LogCallbackInt = new CallbackDelegateInt(CallbackInt);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Log()
        {
            // Remove logging callback
            phLog_Init(IntPtr.Zero, IntPtr.Zero, 0);

            // Free allocated pointer to data params
            if (this.m_pRegisterEntriesInt.IsAllocated)
            {
                this.m_pRegisterEntriesInt.Free();
            }

            // Free allocated pointer to logEntries
            if (this.m_pLogEntriesArrays != null)
            {
                foreach (GCHandle logEntries in this.m_pLogEntriesArrays)
                {
                    if (logEntries.IsAllocated)
                    {
                        logEntries.Free();
                    }
                }
            }
        }

        #endregion
    }
}
