/*
 * Copyright 2013 - 2018, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.KeyStore
{
    #region Enumerations
    #region Configuration
    /// <summary>
    /// Definitions for KeyStore layer configuration
    /// </summary>
    public enum Config : int
    {
        /// <summary> Reset all bit of SET param. </summary>
        SET_DEFAULT = 0xFFFF
    }
    #endregion

    #region KeyType
    /// <summary>
    /// Definitions for Symmetric and ASymmetric Key types
    /// </summary>
    public enum KeyType : ushort
    {
        /// <summary> Symmetric AES 128 Key [16]. </summary>
        AES128 = 0x0000,

        /// <summary> Symmetric AES 192 Key [24]. </summary>
        AES192,

        /// <summary> Symmetric AES 256 Key [32]. </summary>
        AES256,

        /// <summary> Symmetric DES Single Key. </summary>
        DES,

        /// <summary> Symmetric 2 Key Triple Des. </summary>
        DES2K3,

        /// <summary> Symmetric 3 Key Triple Des. </summary>
        DES3K3,

        /// <summary> Symmetric MIFARE (R) Key. </summary>
        MIFARE,

        /// <summary> ASymmetric ECC (Elliptical Curve Cryptography) key type. </summary>
        ECC = 0x0200,

        /// <summary> ID used for various parameters as a invalid default </summary>
        INAVLID = 0xFFFF
    }
    #endregion

    #region KeyPair
    /// <summary>
    /// Definitions for ASymmetric Key pair
    /// </summary>
    public enum KeyPair : ushort
    {
        /// <summary> ASymmetric key pair Invalid. </summary>
        INVALID = 0xC000,

        /// <summary> ASymmetric key pair as Private Key. </summary>
        PRIVATE_KEY = 0x1000,

        /// <summary> ASymmetric key pair as Public Key. </summary>
        PUBLIC_KEY = 0x2000
    }
    #endregion

    #region CurveID
    /// <summary>
    /// Supported ECC Curves Group ID's
    /// The ID values matched with the underlying crypto library being used.
    /// </summary>
    public enum CurveID : byte
    {
        /// <summary> Field to indicate the curve group id as none. </summary>
        NONE = 0x00,

        /// <summary> Field to indicate the curve group id as secp256r1. </summary>
        SECP256R1 = 0x03,

        /// <summary> Field to indicate the curve group id as secp384r1. </summary>
        SECP384R1 = 0x04,

        /// <summary> Field to indicate the curve group id as brainpoolp256r1. </summary>
        BRAINPOOL256R1 = 0x06,

        /// <summary> Field to indicate the curve group id as brainpoolp384r1. </summary>
        BRAINPOOL384R1 = 0x07,
    }
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Generic KeyStore Component Wrapper of the Reader Library Framework.
    /// </summary>
    public abstract class Generic
    {
        #region Public Variables
        /// <summary> Maximum length of Key information. </summary>
        public const byte MAX_NUM_KEY_INFO_LENGTH = 3;

        /// <summary> Maximum size of a Symmetric and ASymmetric Key. </summary>
        public const int MAX_KEY_SIZE = 48;
        #endregion

        #region DLL Imports
        #region Common Methods
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_FormatKeyEntry ( IntPtr pDataParams, ushort wKeyNo, KeyType wNewKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetKUC ( IntPtr pDataParams, ushort wKeyNo, ushort wRefNoKUC );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetKUC ( IntPtr pDataParams, ushort wRefNoKUC, ref uint dwLimit, ref uint pdwCurVal );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_ChangeKUC ( IntPtr pDataParams, ushort wRefNoKUC, uint dwLimit );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetConfigStr ( IntPtr pDataParams, ushort wConfig, byte[] pBuffer, ushort wBufferLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetConfigStr ( IntPtr pDataParams, ushort wConfig, ref IntPtr ppBuffer,
            ref ushort pBufferLength );
        #endregion

        #region Methods for Symmetric Keys
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetKey ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVersion, KeyType wKeyType,
            byte[] pNewKey, ushort wNewKeyVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetKeyAtPos ( IntPtr pDataParams, ushort wKeyNo, ushort wPos, KeyType wKeyType,
            byte[] pNewKey, ushort wNewKeyVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetFullKeyEntry ( IntPtr pDataParams, ushort wNoOfKeys, ushort wKeyNo,
            ushort wNewRefNoKUC, KeyType wNewKeyType, byte[] pNewKeys, ushort[] pNewKeyVersionList );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetKey ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVersion, byte bKeyBufSize,
            byte[] pKey, ref KeyType pKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetKeyEntry ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVersionBufSize,
            ushort[] wKeyVersion, ref ushort wKeyVersionLength, ref KeyType pKeyType );
        #endregion

        #region Methods for ASymmetric Keys
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SetKeyASym ( IntPtr pDataParams, ushort wKeyNo, ushort wPos, KeyType wKeyType, ushort wKeyInfo,
            byte[] pKey, ushort wKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetKeyASym ( IntPtr pDataParams, ushort wKeyNo, ushort wPos, KeyPair wKeyPair, ref KeyType pKeyType,
            ref CurveID pCurveID, byte[] pKey, ref ushort pKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_GetCurveID ( IntPtr pDataParams, ushort wKeyNo, ushort wPos, ref CurveID pCurveID );
        #endregion
        #endregion

        #region Properties
        /// <summary> Returns the count of configured KeyVersionPair. </summary>
        public abstract int m_wNumKeyVersionPairs
        {
            get;
        }
        #endregion

        #region Wrapper Functions
        #region Common Methods
        /// <summary>
        /// Format a key entry to a new KeyType.
        /// </summary>
        ///
        /// <param name="wKeyNo">KeyEntry number to be Formatted.</param>
        /// <param name="wNewKeyType">New Key type of the KeyEntry (predefined type of KeyType).
        ///                             Symmetric KeyTypes
        ///                                 <see cref="KeyType.AES128"/>
        ///                                 <see cref="KeyType.AES192"/>
        ///                                 <see cref="KeyType.AES256"/>
        ///                                 <see cref="KeyType.DES"/>
        ///                                 <see cref="KeyType.DES2K3"/>
        ///                                 <see cref="KeyType.DES3K3"/>
        ///                                 <see cref="KeyType.MIFARE"/>
        ///
        ///                             ASymmetric KeyTypes
        ///                                 <see cref="KeyType.ECC"/>
        /// </param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t FormatKeyEntry ( int wKeyNo, KeyType wNewKeyType )
        {
            Status_t oStatus;
            oStatus = phKeyStore_FormatKeyEntry ( m_pDataParams, ( ushort ) wKeyNo, wNewKeyType );
            return oStatus;
        }

        /// <summary>
        /// Change the KUC of a key entry.
        /// </summary>
        ///
        /// <param name="wKeyNo">KeyEntry number.</param>
        /// <param name="wRefNoKUC">Reference Number of the key usage counter used together with that key.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetKUC ( int wKeyNo, int wRefNoKUC )
        {
            Status_t oStatus;
            oStatus = phKeyStore_SetKUC ( m_pDataParams, ( ushort ) wKeyNo, ( ushort ) wRefNoKUC );
            return oStatus;
        }

        /// <summary>
        /// Obtain a key usage counter entry.
        /// </summary>
        ///
        /// <param name="wRefNoKUC">Number of the key usage counter to be looked at (00h to 0Fh)</param>
        /// <param name="pdwLimit">Currently Set Limit in the KUC</param>
        /// <param name="pdwCurVal">Currently set value in the KUC</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKUC ( int wRefNoKUC, out long pdwLimit, out long pdwCurVal )
        {
            Status_t oStatus;
            uint dwLimit = 0;
            uint dwCurVal = 0;

            oStatus = phKeyStore_GetKUC ( m_pDataParams, ( ushort ) wRefNoKUC, ref dwLimit, ref dwCurVal );
            pdwLimit = ( long ) dwLimit;
            pdwCurVal = ( long ) dwCurVal;

            return oStatus;
        }

        /// <summary>
        /// Change a key usage counter entry.
        /// </summary>
        ///
        /// <param name="wRefNoKUC">Number of key entry.</param>
        /// <param name="dwLimit">Limit of the Key Usage Counter.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeKUC ( int wRefNoKUC, long dwLimit )
        {
            return phKeyStore_ChangeKUC ( m_pDataParams, ( ushort ) wRefNoKUC, ( uint ) dwLimit );
        }

        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="wValue">Configuration Value</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( int wConfig, int wValue )
        {
            return phKeyStore_SetConfig ( m_pDataParams, ( ushort ) wConfig, ( ushort ) wValue );
        }

        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier.</param>
        /// <param name="pBuffer">Buffer containing the configuration string.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfigStr ( int wConfig, byte[] pBuffer )
        {
            return phKeyStore_SetConfigStr ( m_pDataParams, ( ushort ) wConfig, pBuffer,
                ( ushort ) ( ( pBuffer == null ) ? 0 : pBuffer.Length ) );
        }

#if DEBUG
        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier.</param>
        /// <param name="pBuffer">Buffer containing the configuration string.</param>
        /// <param name="bBuffLen">Length of configuration string.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfigStr ( int wConfig, byte[] pBuffer, byte bBuffLen )
        {
            return phKeyStore_SetConfigStr ( m_pDataParams, ( ushort ) wConfig, pBuffer, bBuffLen );
        }
#endif

        /// <summary>
        /// Get configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="pValue">Configuration Value</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( int wConfig, out int pValue )
        {
            Status_t oStatus;
            ushort wValue = 0;
            oStatus = phKeyStore_GetConfig ( m_pDataParams, ( ushort ) wConfig, ref wValue );
            pValue = ( int ) wValue;
            return oStatus;
        }

        /// <summary>
        /// Get configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration Identifier</param>
        /// <param name="pBuffer">Pointer to the buffer containing the configuration string.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfigStr ( int wConfig, out byte[] pBuffer )
        {
            Status_t oStatus;
            IntPtr BufferInt = IntPtr.Zero;
            ushort LengthInt = 0;
            pBuffer = null;

            oStatus = phKeyStore_GetConfigStr ( m_pDataParams, ( ushort ) wConfig, ref BufferInt, ref LengthInt );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                pBuffer = new byte[LengthInt];
                Marshal.Copy ( BufferInt, pBuffer, 0, LengthInt );
            }

            return oStatus;
        }
        #endregion

        #region Methods for Symmetric Keys
        /// <summary>
        /// Change a symmetric key entry at a given version.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the key to be loaded.</param>
        /// <param name="wKeyVersion">Key version of the key to be loaded.</param>
        /// <param name="wKeyType">New Key type of the KeyEntry (predefined type of KeyType).
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        /// <param name="pNewKey">Pointer to the key itself.</param>
        /// <param name="wNewKeyVersion">New Key version of the key to be updated.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetKey ( int wKeyNo, int wKeyVersion, KeyType wKeyType, byte[] pNewKey, int wNewKeyVersion )
        {
            return phKeyStore_SetKey ( m_pDataParams, ( ushort ) wKeyNo, ( ushort ) wKeyVersion, wKeyType, pNewKey, ( ushort ) wNewKeyVersion );
        }

        /// <summary>
        /// Change a symmetric key entry at the specified position.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the key to be loaded.</param>
        /// <param name="wPos">Key position to be updated.</param>
        /// <param name="wKeyType">New Key type of the KeyEntry (predefined type of KeyType).
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        /// <param name="pNewKey">Pointer to the key itself.</param>
        /// <param name="wNewKeyVersion">New Key version of the key to be updated.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetKeyAtPos ( int wKeyNo, int wPos, KeyType wKeyType, byte[] pNewKey, int wNewKeyVersion )
        {
            return phKeyStore_SetKeyAtPos ( m_pDataParams, ( ushort ) wKeyNo, ( ushort ) wPos, wKeyType, pNewKey, ( ushort ) wNewKeyVersion );
        }

        /// <summary>
        /// Change a full symmetric key entry.
        /// </summary>
        ///
        /// <param name="wNoOfKeys">Number of keys in pNewKeys.</param>
        /// <param name="wKeyNo">Key number of the key to be loaded.</param>
        /// <param name="wNewRefNoKUC">Number of the key usage counter used together with that key.</param>
        /// <param name="wNewKeyType">Key type of the key (if the current keyType of KeyEntry is different, error).
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        /// <param name="pNewKeys">Array of Keys to load.</param>
        /// <param name="pNewKeyVersionList">KeyVersionList of the key to be loaded.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetFullKeyEntry ( int wNoOfKeys, int wKeyNo, int wNewRefNoKUC, KeyType wNewKeyType, byte[] pNewKeys,
            int[] pNewKeyVersionList )
        {
            ushort[] wKeyVersion_Int = new ushort[(pNewKeyVersionList == null) ? 0 : pNewKeyVersionList.Length];

            for ( int i = 0; i < wKeyVersion_Int.Length; i++ )
                wKeyVersion_Int[i] = ( ushort ) pNewKeyVersionList[i];

            return phKeyStore_SetFullKeyEntry ( m_pDataParams, ( ushort ) wNoOfKeys, ( ushort ) wKeyNo, ( ushort ) wNewRefNoKUC,
                wNewKeyType, pNewKeys, wKeyVersion_Int );
        }

        /// <summary>
        /// Get a symmetric key.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the key to be retrieved.param>
        /// <param name="wKeyVersion">Key version of the key to be retrieved.</param>
        /// <param name="pKey">Pointer to the key itself.</param>
        /// <param name="pKeyType">Type of the key. One of the below values.
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKey ( int wKeyNo, int wKeyVersion, out byte[] pKey, out KeyType pKeyType )
        {
            pKeyType = KeyType.INAVLID;
            pKey = new byte[255];
            Status_t oStatus = phKeyStore_GetKey(m_pDataParams, (ushort)wKeyNo, (ushort)wKeyVersion, (byte)pKey.Length, pKey, ref pKeyType);
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                switch ( pKeyType )
                {
                    case KeyType.AES128:
                    case KeyType.DES2K3:
                        System.Array.Resize<byte> ( ref pKey, 16 );
                        break;
                    case KeyType.AES192:
                    case KeyType.DES3K3:
                        System.Array.Resize<byte> ( ref pKey, 24 );
                        break;
                    case KeyType.AES256:
                        System.Array.Resize<byte> ( ref pKey, 32 );
                        break;
                    case KeyType.DES:
                        System.Array.Resize<byte> ( ref pKey, 8 );
                        break;
                    case KeyType.MIFARE:
                        System.Array.Resize<byte> ( ref pKey, 12 );
                        break;
                    default:
                        System.Array.Resize<byte> ( ref pKey, 0 );
                        break;
                }
            }
            return oStatus;
        }

#if DEBUG
        /// <summary>
        /// Get a symmetric key.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the key to be retrieved.param>
        /// <param name="wKeyVersion">Key version of the key to be retrieved.</param>
        /// <param name="wReceiveBufferSize">Size of the key buffer.</param>
        /// <param name="pKey">Pointer to the key itself.</param>
        /// <param name="pKeyType">Type of the key. One of the below values.
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKey ( int wKeyNo, int wKeyVersion, int wReceiveBufferSize, out byte[] pKey,
            out KeyType pKeyType )
        {
            pKeyType = KeyType.INAVLID;
            pKey = new byte[255];
            Status_t oStatus = phKeyStore_GetKey(m_pDataParams, (ushort)wKeyNo, (ushort)wKeyVersion, (byte)wReceiveBufferSize, pKey, ref pKeyType);
            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                switch ( pKeyType )
                {
                    case KeyType.AES128:
                    case KeyType.DES2K3:
                        System.Array.Resize<byte> ( ref pKey, 16 );
                        break;
                    case KeyType.AES192:
                    case KeyType.DES3K3:
                        System.Array.Resize<byte> ( ref pKey, 24 );
                        break;
                    case KeyType.AES256:
                        System.Array.Resize<byte> ( ref pKey, 32 );
                        break;
                    case KeyType.DES:
                        System.Array.Resize<byte> ( ref pKey, 8 );
                        break;
                    case KeyType.MIFARE:
                        System.Array.Resize<byte> ( ref pKey, 12 );
                        break;
                    default:
                        System.Array.Resize<byte> ( ref pKey, 0 );
                        break;
                }
            }
            return oStatus;
        }
#endif

        /// <summary>
        /// Get a symmetric key entry information block.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the key entry of interest.</param>
        /// <param name="wKeyVersion">Array for version information.</param>
        /// <param name="pKeyType">Type of the key. One of the below values.
        ///                             <see cref="KeyType.AES128"/>
        ///                             <see cref="KeyType.AES192"/>
        ///                             <see cref="KeyType.AES256"/>
        ///                             <see cref="KeyType.DES"/>
        ///                             <see cref="KeyType.DES2K3"/>
        ///                             <see cref="KeyType.DES3K3"/>
        ///                             <see cref="KeyType.MIFARE"/>
        /// </param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKeyEntry ( int wKeyNo, out int[] wKeyVersion, out KeyType pKeyType )
        {
            ushort[] wKeyVersion_Int = new ushort[m_wNumKeyVersionPairs];
            ushort wKeyEntryLength = 0;

            pKeyType = KeyType.AES128;
            wKeyVersion = new int[m_wNumKeyVersionPairs];

            Status_t oStatus = phKeyStore_GetKeyEntry(m_pDataParams, (ushort)wKeyNo, (ushort)(wKeyVersion.Length * 2),
                wKeyVersion_Int, ref wKeyEntryLength, ref pKeyType);

            Array.Copy ( wKeyVersion_Int, wKeyVersion, wKeyVersion_Int.Length );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                System.Array.Resize<int> ( ref wKeyVersion, wKeyEntryLength );
            }
            else
            {
                System.Array.Resize<int> ( ref wKeyVersion, 0 );
            }
            return oStatus;
        }
        #endregion

        #region Method for ASymmetric Keys
        /// <summary>
        /// Change a Asymmetric key entry.
        ///
        /// Note
        ///     Before changing the information of the key entry, the key entry should be formatted using
        ///     <see cref="FormatKeyEntry"/> method.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the KeyStore to be loaded.</param>
        /// <param name="wPos">Position at which the Key should be updated.</param>
        /// <param name="eKeyType">New Key type of the KeyEntry (predefined type of KeyType).
        ///                             <see cref="KeyType.ECC"/>
        /// </param>
        /// <param name="eKeyPair">Format of key available in aKey buffer.
        ///                         <see cref="KeyPair.PRIVATE_KEY"/>
        ///                         <see cref="KeyPair.PUBLIC_KEY"/>
        /// </param>
        /// <param name="eFormat">The format of ASymmetric key.
        ///                         <see cref="KeyFormat.PEM"/>
        ///                         <see cref="KeyFormat.DER"/>
        ///                         <see cref="KeyFormat.BIN"/>
        /// </param>
        /// <param name="aKey">Pointer to the key itself.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetKeyASym ( ushort wKeyNo, ushort wPos, KeyType eKeyType, KeyPair eKeyPair, CurveID eCurveID, byte[] aKey )
        {
            ushort wKeyInfo = ( ushort) ( ( ushort ) eKeyPair | ( ushort ) eCurveID);

            return phKeyStore_SetKeyASym ( m_pDataParams, wKeyNo, wPos, eKeyType, wKeyInfo, aKey,
                ( ushort ) ( ( aKey == null ) ? 0 : aKey.Length ) );
        }

        /// <summary>
        /// Get a asymmetric key.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the KeyStore to be retrieved.</param>
        /// <param name="wPos">Position at which the Key should be retrieved.</param>
        /// <param name="eKeyType">Type of key being returned. One of the below values.
        ///                             <see cref="KeyType.ECC"/>
        /// </param>
        /// <param name="eKeyPair">Format of key available in aKey buffer.
        ///                         <see cref="KeyPair.PRIVATE_KEY"/>
        ///                         <see cref="KeyPair.PUBLIC_KEY"/>
        /// </param>
        /// <param name="aKey">Pointer to the key itself.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKeyASym ( ushort wKeyNo, ushort wPos, KeyPair eKeyPair, out KeyType eKeyType, out CurveID eCurveID,
            out byte[] aKey )
        {
            byte[] aKeyTmp = new byte[(MAX_KEY_SIZE * 2) + 1];
            ushort wKeyLen = 0;

            aKey = null;
            eKeyType = KeyType.INAVLID;
            eCurveID = CurveID.NONE;

            Status_t oStatus = phKeyStore_GetKeyASym ( m_pDataParams, wKeyNo, wPos, eKeyPair, ref eKeyType, ref eCurveID, aKeyTmp, ref wKeyLen );
            if ( oStatus.Equals ( new Status_t () ) )
            {
                aKey = new byte[wKeyLen];
                Array.Copy ( aKeyTmp, 0, aKey, 0, wKeyLen );
            }

            return oStatus;
        }

        /// <summary>
        /// Get a asymmetric key ECC Curve ID.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number of the KeyStore to be retrieved.</param>
        /// <param name="wPos">Position at which the Key should be retrieved.</param>
        /// <param name="eCurveID">Type of key being returned. One of the below values.
        ///                             <see cref="KeyType.ECC"/>
        /// </param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetCurveID ( ushort wKeyNo, ushort wPos, out CurveID eCurveID )
        {
            eCurveID = CurveID.NONE;
            return phKeyStore_GetCurveID ( m_pDataParams, wKeyNo, wPos, ref eCurveID );
        }
        #endregion
        #endregion

        #region Memory Mapping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion

    #region Software
    /// <summary>
    /// Class for Software Mode layer initialization interface and data params.
    /// </summary>
    public class Sw : Generic
    {
        #region Data Structure
        /// <summary>
        /// Software KeyVersionPair structure for Symmetric and ASymmetric keys.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct KeyVersionPair_t
        {
            /// <summary> Array containing a Symmetric and ASymmetric Key. </summary>
            public fixed byte bKey[MAX_KEY_SIZE];

            /// <summary> Versions related to the Symmetric Key. Not applicable for ASymmetric keys storage. </summary>
            public ushort wVersion;

            /// <summary> Array containing a Symmetric and ASymmetric Key. </summary>
            public fixed byte aPubKey[(MAX_KEY_SIZE * 2) + 1];

            /// <summary> KeyPair being used for key to be updated. </summary>
            public ushort wKeyPairType;

            /// <summary> Curve Type of Private / Public Key. One of the values mentioned in <see cref="CurveID"/>. </summary>
            public byte bCurveID;
        };

        /// <summary>
        /// Software KeyEntry structure.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct KeyEntry_t
        {
            /// <summary> Type of the keys in <see cref="KeyVersionPair_t"/>. </summary>
            public ushort wKeyType;

            /// <summary> Key usage counter number of the keys in pKeys. </summary>
            public ushort wRefNoKUC;
        };

        /// <summary>
        /// Software KeyUsageCounter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct KUCEntry_t
        {
            /// <summary> Limit of the Key Usage Counter. </summary>
            public uint dwLimit;

            /// <summary> Current Value of the KUC. </summary>
            public uint dwCurVal;
        };

        /// <summary>
        /// Software parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Key entry storage, size = sizeof(#phKeyStore_Sw_KeyEntry_t) * wNumKeyEntries. </summary>
            public IntPtr pKeyEntries;

            /// <summary> Key version pairs, size = sizeof(#phKeyStore_Sw_KeyVersionPair_t)* wNumKeyEntries * wNumVersions. </summary>
            public IntPtr pKeyVersionPairs;

            /// <summary> Number of key entries in that storage. </summary>
            public ushort wNoOfKeyEntries;

            /// <summary> Number of versions in each key entry. </summary>
            public ushort wNoOfVersions;

            /// <summary> Key usage counter entry storage, size = sizeof(#phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries. </summary>
            public IntPtr pKUCEntries;

            /// <summary> Number of Key usage counter entries. </summary>
            public ushort wNoOfKUCEntries;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pKeyEntries,
            ushort wNumKeyEntries, IntPtr pKeyVersionPairs, ushort wNumKeyVersionPairs, IntPtr pKUCEntries, ushort wNumKUCEntries );
        #endregion

        #region Initialization
        private KeyEntry_t[] m_KeyEntries;
        private GCHandle m_pKeyEntries;

        private KUCEntry_t[] m_KUCEntries;
        private GCHandle m_pKUCEntries;

        private KeyVersionPair_t[] m_KeyVersionPairs;
        private GCHandle m_pKeyVersionPairs;

        private ushort m_wNumKeyVersionPairs_Int;
        private ushort m_wNumKeyEntries;
        private ushort m_wNumKeyUsageCounters;

        /// <summary>
        /// Initializes the KeyStore component as software component
        /// </summary>
        ///
        /// <param name="wNumKeyEntries">Size of pKeyEntries.</param>
        /// <param name="wNumKeyVersionPairs">Amount of key versions available in each key entry.</param>
        /// <param name="wNumKeyUsageCounters">Number of Key usage counter entries.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wNumKeyEntries, int wNumKeyVersionPairs, int wNumKeyUsageCounters )
        {
            // Free allocated pointer to reader params
            if ( m_pKeyEntries.IsAllocated )
            {
                m_pKeyEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKUCEntries.IsAllocated )
            {
                m_pKUCEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKeyVersionPairs.IsAllocated )
            {
                m_pKeyVersionPairs.Free ();
            }

            // Allocate internal key entries and pointer to them
            m_wNumKeyVersionPairs_Int = ( ushort ) wNumKeyVersionPairs;
            m_KeyVersionPairs = new KeyVersionPair_t[wNumKeyVersionPairs * wNumKeyEntries];
            m_pKeyVersionPairs = GCHandle.Alloc ( m_KeyVersionPairs, GCHandleType.Pinned );

            m_wNumKeyEntries = ( ushort ) wNumKeyEntries;
            m_KeyEntries = new KeyEntry_t[wNumKeyEntries];
            m_pKeyEntries = GCHandle.Alloc ( m_KeyEntries, GCHandleType.Pinned );

            m_wNumKeyUsageCounters = ( ushort ) wNumKeyUsageCounters;
            m_KUCEntries = new KUCEntry_t[wNumKeyUsageCounters];
            m_pKUCEntries = GCHandle.Alloc ( m_KUCEntries, GCHandleType.Pinned );

            return phKeyStore_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                m_pKeyEntries.AddrOfPinnedObject (), ( ushort ) wNumKeyEntries, m_pKeyVersionPairs.AddrOfPinnedObject (),
                ( ushort ) wNumKeyVersionPairs, m_pKUCEntries.AddrOfPinnedObject (), ( ushort ) wNumKeyUsageCounters );
        }

#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component as software component
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="wNumKeyEntries">Size of pKeyEntries.</param>
        /// <param name="wNumKeyVersionPairs">Amount of key versions available in each key entry.</param>
        /// <param name="wNumKeyUsageCounters">Number of Key usage counter entries.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, int wNumKeyEntries, int wNumKeyVersionPairs, int wNumKeyUsageCounters )
        {
            // Free allocated pointer to reader params
            if ( m_pKeyEntries.IsAllocated )
            {
                m_pKeyEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKUCEntries.IsAllocated )
            {
                m_pKUCEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKeyVersionPairs.IsAllocated )
            {
                m_pKeyVersionPairs.Free ();
            }

            // Allocate internal key entries and pointer to them
            m_wNumKeyVersionPairs_Int = ( ushort ) wNumKeyVersionPairs;
            m_KeyVersionPairs = new KeyVersionPair_t[wNumKeyVersionPairs * wNumKeyEntries];
            m_pKeyVersionPairs = GCHandle.Alloc ( m_KeyVersionPairs, GCHandleType.Pinned );

            m_wNumKeyEntries = ( ushort ) wNumKeyEntries;
            m_KeyEntries = new KeyEntry_t[wNumKeyEntries];
            m_pKeyEntries = GCHandle.Alloc ( m_KeyEntries, GCHandleType.Pinned );

            m_wNumKeyUsageCounters = ( ushort ) wNumKeyUsageCounters;
            m_KUCEntries = new KUCEntry_t[wNumKeyUsageCounters];
            m_pKUCEntries = GCHandle.Alloc ( m_KUCEntries, GCHandleType.Pinned );

            return phKeyStore_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, m_pKeyEntries.AddrOfPinnedObject (),
                ( ushort ) wNumKeyEntries, m_pKeyVersionPairs.AddrOfPinnedObject (), ( ushort ) wNumKeyVersionPairs,
                m_pKUCEntries.AddrOfPinnedObject (), ( ushort ) wNumKeyUsageCounters );
        }

        /// <summary>
        /// Initializes the KeyStore component as software component
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="wNumKeyEntries">Size of pKeyEntries.</param>
        /// <param name="KeyEntery">Pointer to a storage containing the key entries.</param>
        /// <param name="wNumKeyVersionPairs">Amount of key versions available in each key entry.</param>
        /// <param name="KeyVersionPair">Pointer to a storage containing the key version pairs.</param>
        /// <param name="wNumKeyUsageCounters">Number of Key usage counter entries.</param>
        /// <param name="KUCEntery">Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, int wNumKeyEntries, KeyEntry_t[] KeyEntery, int wNumKeyVersionPairs, KeyVersionPair_t[] KeyVersionPair,
            int wNumKeyUsageCounters, KUCEntry_t[] KUCEntery )
        {
            // Allocate internal key entries and pointer to them
            m_wNumKeyVersionPairs_Int = ( ushort ) wNumKeyVersionPairs;
            m_KeyVersionPairs = KeyVersionPair;
            m_pKeyVersionPairs = GCHandle.Alloc ( m_KeyVersionPairs, GCHandleType.Pinned );

            m_wNumKeyEntries = ( ushort ) wNumKeyEntries;
            m_KeyEntries = KeyEntery;
            m_pKeyEntries = GCHandle.Alloc ( m_KeyEntries, GCHandleType.Pinned );

            m_wNumKeyUsageCounters = ( ushort ) wNumKeyUsageCounters;
            m_KUCEntries = KUCEntery;
            m_pKUCEntries = GCHandle.Alloc ( m_KUCEntries, GCHandleType.Pinned );

            return phKeyStore_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, m_pKeyEntries.AddrOfPinnedObject (),
                ( ushort ) wNumKeyEntries, m_pKeyVersionPairs.AddrOfPinnedObject (), ( ushort ) wNumKeyVersionPairs,
                m_pKUCEntries.AddrOfPinnedObject (), ( ushort ) wNumKeyUsageCounters );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );

        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKeyEntries.IsAllocated )
            {
                m_pKeyEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKUCEntries.IsAllocated )
            {
                m_pKUCEntries.Free ();
            }

            // Free allocated pointer to reader params
            if ( m_pKeyVersionPairs.IsAllocated )
            {
                m_pKeyVersionPairs.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region Sam
    /// <summary>
    /// KeyStore component to updating and retrieving information from SAM (AV4 and future SAM's) KeyStore component.
    /// This support Sam AV3 for backward compatibility features and SAM (AV4 and future SAM's).
    /// with features of existing Sam AV3 and new features of SAM (AV4 and future SAM's).
    /// </summary>
    public class Sam : Generic
    {
        #region Enumerations
        #region Configuration Options
        /// <summary>
        /// Configuration options to Set / Get the SET, ExtSET and other configurable information.
        /// </summary>
        public enum Config : ushort
        {
            #region SET Configuration Options
            /// <summary> Enable or Disable SAM_ChangeKeyMIFARE and SAM_DumpSessionKey command. </summary>
            ALLOW_DUMP_SESSION_KEY = 0x0000,

            /// <summary> Enable or Disable the reset of init vector after a crypto command. </summary>
            KEEP_IV = 0x0001,

            /// <summary> Enable or Disable the Host key type to provide permissions for Cmd.PLExec execution. </summary>
            PL_KEY = 0x0002,

            /// <summary> Enable or Disable Host Authentication with key other that MasterKey. </summary>
            AUTH_KEY = 0x0003,

            /// <summary> Enable or Disable Key Entry. </summary>
            DISABLE_KEY_ENTRY = 0x0004,

            /// <summary> Enable or Disable LockUnlock. </summary>
            LOCK_KEY = 0x0005,

            /// <summary> Enable or Disable writing the key to a PICC. </summary>
            DISABLE_CHANGE_KEY_PICC = 0x0006,

            /// <summary> Enable or Disable SAM_DecipherData command. </summary>
            DISABLE_DECRYPTION = 0x0007,

            /// <summary> Enable or Disable SAM_EncipherData command. </summary>
            DISABLE_ENCRYPTION = 0x0008,

            /// <summary> Enable or Disable SAM_VerifyMAC command. </summary>
            DISABLE_VERIFY_MAC = 0x0009,

            /// <summary> Enable or Disable SAM_GenerateMAC command. </summary>
            DISABLE_GENERATE_MAC = 0x000A,
            #endregion

            #region ExtSET Configuration Options
            /// <summary> Key Class used in key store. </summary>
            KEY_CLASS = 0x000B,

            /// <summary> Enable or Disable SAM_DumpSecretKey command. </summary>
            ALLOW_DUMP_SECRET_KEY = 0x000C,

            /// <summary> Mandate or not key diversification. </summary>
            MANDATE_KEY_DIVERSIFICATION = 0x000D,

            /// <summary> Enable or disable the Key Entry for Sam Personalization. </summary>
            RESERVED_SAM_PERSONALIZATION = 0x000E,

            /// <summary> Enable or disable the Key Entry usage by Internal Host. </summary>
            KEY_USAGE_INTERNAL_HOST = 0x000F,

            /// <summary> Enable or disable the Key Entry change by Internal Host. </summary>
            KEY_CHANGE_INTERNAL_HOST = 0x0010,

            /// <summary> Enable or disable the Session Key usage by Internal Host. </summary>
            SESSION_KEY_USAGE_INTERNAL_HOST = 0x0011,

            /// <summary> Enable or disable the dumping of Secret Key by Internal Host. </summary>
            ALLOW_DUMP_SECRET_KEY_INTERNAL_HOST = 0x0012,

            /// <summary> Enable or disable the dumping of Session Key by Internal Host. </summary>
            ALLOW_DUMP_SESSION_KEY_INTERNAL_HOST = 0x0013,
            #endregion

            #region Key Storage Table's Key Entry Configuration Options
            /// <summary> DESFire application ID. </summary>
            DF_AID = 0x0014,

            /// <summary> DESFire key number. </summary>
            DF_KEY_NO = 0x0015,

            /// <summary> Key Number of Change Entry key. </summary>
            KEYNO_CEK = 0x0016,

            /// <summary> Key Version of Change Entry key. </summary>
            KEYV_CEK = 0x0017,

            /// <summary> Reference number of key usage counter. </summary>
            REF_NO_KUC = 0x0018,

            /// <summary> Key Number of Access Entry key. </summary>
            KEYNO_AEK = 0x0019,

            /// <summary> Key Version of Access Entry key. </summary>
            KEYV_AEK = 0x001A,

            /// <summary> Key Number of Change KUC. </summary>
            KEYNO_CKUC = 0x001B,

            /// <summary> Key Version of Change KUC. </summary>
            KEYV_CKUC = 0x001C,

            /// <summary> Option for single DES and 2 Key Triple DES keys. </summary>
            DES_KEY_OPTION = 0x001D,

            /// <summary> The AES key to be used is for LRP algorithm. </summary>
            ENABLE_LRP = 0x001E
            #endregion
        }
        #endregion

        #region KeyClass
        /// <summary>
        /// The type of key classes to be used for SAM (AV4 and future SAM's).
        /// </summary>
        public enum KeyClass : byte
        {
            /// <summary> Configuring key entry as Host. </summary>
            HOST = 0x00,

            /// <summary> Configuring key entry as PICC. </summary>
            PICC = 0x01,

            /// <summary> Configuring key entry as Offline Change. </summary>
            OFFLINE_CHANGE = 0x02,

            /// <summary> Configuring key entry as Offline Crypto. </summary>
            OFFLINE_CRYPTO = 0x04,

            /// <summary> Configuring key entry as Offline Upload. </summary>
            OFFLINE_UPLOAD = 0x05,

            /// <summary> Configuring key entry as Offline Perso. </summary>
            OFFLINE_PERSO = 0x06
        }
        #endregion

        #region DESFire Key Options
        /// <summary>
        /// The additional option to indicate the type of DESFire key.
        /// </summary>
        public enum DesKeyOption : byte
        {
            /// <summary> DESFire 4 compatibility mode. </summary>
            DESFIRE4 = 0x00,

            /// <summary> ISO 10116 mode with CRC16 protection and 4 bytes MAC. </summary>
            ISO_CRC16 = 0x01,

            /// <summary> ISO 10116 mode with CRC32 protection and 8 bytes MAC. </summary>
            ISO_CRC32 = 0x02
        }
        #endregion
        #endregion

        #region Data Structure
        /// <summary>
        /// SAM (AV4 and future SAM's) parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            public IntPtr pHalDataParams;

            /// <summary> Configuration settings. </summary>
            public fixed byte aSet[2];

            /// <summary> Extended configuration settings. </summary>
            public fixed byte aExtSet[2];

            /// <summary> DESFire application ID. </summary>
            public fixed byte aDFAid[3];

            /// <summary> DESFire key number. </summary>
            public byte bDFKeyNo;

            /// <summary> Key Number of Change Entry Key. </summary>
            public byte bKeyNoCEK;

            /// <summary> Key Version of Change Entry Key. </summary>
            public byte bKeyVCEK;

            /// <summary> Reference number of key usage counter. </summary>
            public byte bRefNoKUC;

            /// <summary> Key Number of Access Entry Key. </summary>
            public byte bKeyNoAEK;

            /// <summary> Key Version of Access Entry Key. </summary>
            public byte bKeyVAEK;

            /// <summary> Key Number of Change KUC. </summary>
            public byte bKeyNoCKUC;

            /// <summary> Key Version of Change KUC. </summary>
            public byte bKeyVCKUC;

            /// <summary>
            /// Option for single DES and 2 Key Triple DES keys.
            /// Can be set either to
            ///     <see cref="DesKeyOption.DESFIRE4"/>
            ///     <see cref="DesKeyOption.ISO_CRC16"/>
            ///     <see cref="DesKeyOption.ISO_CRC32"/>
            /// </summary>
            public byte b2K3DESOption;

            /// <summary> Option for LRP key type. If set indicated that the AES key is of LRP type. </summary>
            public byte bIsLRPKey;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_Sam_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion

        #region Initialization
#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component to communicate with Sam KeyStore.
        /// </summary>
        ///
        /// <param name="wDataParamsSize">The size of Sam DataParams structure.</param>
        /// <param name="oHal">Object reference of Sam Hal layer.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( ushort wDataParamsSize, Hal.Sam oHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Sam_Init ( ref m_DataParamsInt[0], wDataParamsSize, oHal.m_pDataParams );
        }
#endif

        /// <summary>
        /// Initializes the KeyStore component to communicate with Sam KeyStore.
        /// </summary>
        ///
        /// <param name="oHal">Object reference of Sam Hal layer.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( Hal.Sam oHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Sam_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sam ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 3;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sam ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }

        #endregion

        #region Structure Member Access
        /// <summary>
        /// Gets / Sets the KeyStore component identifier.
        /// </summary>
        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        /// <summary>
        /// Gets Pointer information of HAL component.
        /// </summary>
        public Hal.Generic HalSamDataParams
        {
            get { return ( ( GCHandle ) m_DataParamsInt[0].pHalDataParams ).Target as Hal.Sam; }
        }

        /// <summary>
        /// Gets the SET configuration as bytes.
        /// </summary>
        public byte[] SET
        {
            get
            {
                byte[] aSet = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aSet[i] = pDataParams->aSet[i];
                        }
                    }
                }
                return aSet;
            }
        }

        /// <summary>
        /// Gets Extended SET configuration as bytes.
        /// </summary>
        public byte[] ExtSET
        {
            get
            {
                byte[] aExtSet = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aExtSet[i] = pDataParams->aExtSet[i];
                        }
                    }
                }
                return aExtSet;
            }
        }

        /// <summary>
        /// Gets the DESFire Application identifier.
        /// </summary>
        public byte[] DF_Aid
        {
            get
            {
                byte[] aDF_Aid = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aDF_Aid[i] = pDataParams->aDFAid[i];
                        }
                    }
                }
                return aDF_Aid;
            }
        }

        /// <summary>
        /// Sets / Gets the DESFire Key number.
        /// </summary>
        public byte DF_KeyNo
        {
            get { return m_DataParamsInt[0].bDFKeyNo; }
            set { m_DataParamsInt[0].bDFKeyNo = value; }
        }

        /// <summary>
        /// Gets / Sets Change Entry Key number.
        /// </summary>
        public byte KeyNo_CEK
        {
            get { return m_DataParamsInt[0].bKeyNoCEK; }
            set { m_DataParamsInt[0].bKeyNoCEK = value; }
        }

        /// <summary>
        /// Gets / Sets Change Entry Key version.
        /// </summary>
        public byte KeyV_CEK
        {
            get { return m_DataParamsInt[0].bKeyVCEK; }
            set { m_DataParamsInt[0].bKeyVCEK = value; }
        }

        /// <summary>
        /// Gets / Sets reference KUC entry number.
        /// </summary>
        public byte RefNoKUC
        {
            get { return m_DataParamsInt[0].bRefNoKUC; }
            set { m_DataParamsInt[0].bRefNoKUC = value; }
        }

        /// <summary>
        /// Gets / Sets AEK Key number.
        /// </summary>
        public byte KeyNo_AEK
        {
            get { return m_DataParamsInt[0].bKeyNoAEK; }
            set { m_DataParamsInt[0].bKeyNoAEK = value; }
        }

        /// <summary>
        /// Gets / Sets AEK Key version.
        /// </summary>
        public byte KeyV_AEK
        {
            get { return m_DataParamsInt[0].bKeyVAEK; }
            set { m_DataParamsInt[0].bKeyVAEK = value; }
        }

        /// <summary>
        /// Gets / Sets CKUC Key number.
        /// </summary>
        public byte KeyNo_CKUC
        {
            get { return m_DataParamsInt[0].bKeyNoCKUC; }
            set { m_DataParamsInt[0].bKeyNoCKUC = value; }
        }

        /// <summary>
        /// Gets / Sets CKUC Key version.
        /// </summary>
        public byte KeyV_CKUC
        {
            get { return m_DataParamsInt[0].bKeyVCKUC; }
            set { m_DataParamsInt[0].bKeyVCKUC = value; }
        }
        #endregion
    }
    #endregion
#endif

    #region SamAV2
    /// <summary>
    /// Class for SAM AV2 Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV2 : Generic
    {
        #region Enumerations
        #region Config
        /// <summary>
        /// Enumerations for configuration
        /// </summary>
        public enum Config : int
        {
            /// <summary> Enable or Disable SAM_ChangeKeyMIFARE and SAM_DumpSessionKey command. </summary>
            ALLOW_DUMP_SESSION_KEY = 0x0000,

            /// <summary> Enable or Disable crypto based on the secret key. </summary>
            ALLOW_CRYPTO_SECRET_KEY = 0x0001,

            /// <summary> Enable or Disable SAM_DumpSecretKey command. </summary>
            ALLOW_DUMP_SECRET_KEY = 0x0002,

            /// <summary> Enable or Disable the reset of init vector after a crypto command. </summary>
            KEEP_IV = 0x0003,

            /// <summary> Enable or Disable Higher Command Security Level. </summary>
            ENABLE_HIGH_SEC_LEVEL = 0x0004,

            /// <summary> Enable or Disable Host Authentication. </summary>
            ENABLE_HOST_AUTH = 0x0005,

            /// <summary> Enable or Disable Host Authentication after reset for AV1 Master Key. </summary>
            ENABLE_HOST_AUTH_AFTER_RESET = 0x0006,

            /// <summary> Enable or Disable LockUnlock for AV2 mode. </summary>
            ENABLE_LOCK_UNLOCK = 0x0007,

            /// <summary> Enable or Disable SAM_ChangeKeyPICC command. </summary>
            DISABLE_CHANGE_KEY_PICC = 0x0008,

            /// <summary> Enable or Disable SAM_DecipherData command. </summary>
            DISABLE_ENCRYPTION = 0x0009,

            /// <summary> Enable or Disable SAM_EncipherData command. </summary>
            DISABLE_DECRYPTION = 0x000A,

            /// <summary> Enable or Disable SAM_VerifyMAC command. </summary>
            DISABLE_VERIFY_MAC = 0x000B,

            /// <summary> Enable or Disable SAM_GenerateMAC command. </summary>
            DISABLE_GENERATE_MAC = 0x000C,

            /// <summary> Mandate or not key diversification. </summary>
            MANDATE_KEY_DIVERSIFICATION = 0x000D,

            /// <summary> Enable plain exchange of ChangeKey functions for AV1 mode. </summary>
            CHANGE_KEY_PLAIN = 0x000E,

            /// <summary> Key Class used in key store. </summary>
            KEYCLASS = 0x000F,

            /// <summary> Key Number of Change Entry key. </summary>
            KEYNO_CEK = 0x0010,

            /// <summary> Key Version of Change Entry key. </summary>
            KEYV_CEK = 0x0011,

            /// <summary> Key Number of Change  KUC. </summary>
            KEYNO_CKUC = 0x0012,

            /// <summary> Key Version of Change KUC. </summary>
            KEYV_CKUC = 0x0013,

            /// <summary> Enable or Disable Key Entry. </summary>
            DISABLE_KEY_ENTRY = 0x0014,

            /// <summary> DESFire application ID. </summary>
            DF_AID = 0x0015,

            /// <summary> DESFire key number. </summary>
            DF_KEY_NO = 0x0016,

            /// <summary> Reference number of key usage counter. </summary>
            REF_NO_KUC = 0x0017,

            /// <summary> Option for single DES and 2 Key Triple DES keys. </summary>
            DES_KEY_OPTION = 0x0018,

            /// <summary> Key Number used for MIFARE key A diversification. </summary>
            KEYNO_MF_DIV_A = 0x0019,

            /// <summary> Key Version used for MIFARE key A diversification. </summary>
            KEYV_MF_DIV_A = 0x001A,

            /// <summary> Key Number used for MIFARE key B diversification. </summary>
            KEYNO_MF_DIV_B = 0x001B,

            /// <summary> Key Version used for MIFARE key B diversification. </summary>
            KEYV_MF_DIV_B = 0x001C
        }
        #endregion

        #region KeyClass
        /// <summary>
        /// Enumeration for KeyClass
        /// </summary>
        public enum KeyClass : byte
        {
            /// <summary> Field for KeyClass enumeration as HOST. </summary>
            HOST = 0x00,

            /// <summary> Field for KeyClass enumeration as PICC. </summary>
            PICC = 0x01,

            /// <summary> Field for KeyClass enumeration as OFFLINE_CHANGE. </summary>
            OFFLINE_CHANGE = 0x02,

            /// <summary> Field for KeyClass enumeration as OFFLINE_CRYPTO. </summary>
            OFFLINE_CRYPTO = 0x04
        }
        #endregion

        #region DesKeyOption
        /// <summary>
        /// Enumeration to differentiate DES Keys.
        /// </summary>
        public enum DesKeyOption : byte
        {
            /// <summary> Field for DesKeyOption enumeration as DESFIRE4. </summary>
            DESFIRE4 = 0x00,

            /// <summary> Field for DesKeyOption enumeration as ISO_CRC16. </summary>
            ISO_CRC16 = 0x01,

            /// <summary> Field for DesKeyOption enumeration as ISO_CRC32. </summary>
            ISO_CRC32 = 0x02
        }
        #endregion
        #endregion

        #region Data Structure
        /// <summary>
        /// SAM AV2 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            IntPtr pHalDataParams;

            /// <summary> Configuration settings. </summary>
            fixed byte pSet[2];

            /// <summary> Extended configuration settings. </summary>
            byte bExtSet;

            /// <summary> Use plain transmission for ChangeKey operations in AV1 mode. </summary>
            byte bChangeKeyPlain;

            /// <summary> DESFire application ID. </summary>
            fixed byte pDFAid[3];

            /// <summary> DESFire key number. </summary>
            byte bDFKeyNo;

            /// <summary> Key Number of Change Entry Key. </summary>
            byte bKeyNoCEK;

            /// <summary> Key Version of Change Entry Key. </summary>
            byte bKeyVCEK;

            /// <summary> Reference number of key usage counter. </summary>
            byte bRefNoKUC;

            /// <summary> Key Number of Change KUC. </summary>
            byte bKeyNoCKUC;

            /// <summary> Key Version of Change KUC. </summary>
            byte bKeyVCKUC;

            /// <summary> Key Number used for MIFARE key A diversification (has to point to a DES key). </summary>
            byte bKeyNoMfDivA;

            /// <summary> Key Version used for MIFARE key A diversification (has to point to a DES key). </summary>
            byte bKeyVMfDivA;

            /// <summary> Key Number used for MIFARE key B diversification (has to point to a DES key). </summary>
            byte bKeyNoMfDivB;

            /// <summary> Key Version used for MIFARE key B diversification (has to point to a DES key). </summary>
            byte bKeyVMfDivB;

            /// <summary>
            /// Option for single DES and 2 Key Triple DES keys.
            /// Can be set either to
            ///     <see cref="DesKeyOption.DESFIRE4"/>
            ///     <see cref="DesKeyOption.ISO_CRC16"/>
            ///     <see cref="DesKeyOption.ISO_CRC32"/>
            /// </summary>
            byte b2K3DESOption;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SamAV2_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component as SAM AV2 component.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV2 pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_SamAV2_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SamAV2 ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 3;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SamAV2 ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }


        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region SamAV3
    /// <summary>
    /// KeyStore component to updating and collecting information from SAM.
    /// This support SamAV2 for backward compatibility features and SamAV3
    /// with features of existing SamAV2 and new features of SamAV3.
    /// </summary>
    public class SamAV3 : Generic
    {
        #region Enumerations
        #region Configuration Options
        /// <summary>
        /// Configuration options to Set / Get the SET, ExtSET and other configurable information.
        /// </summary>
        public enum Config : ushort
        {
            #region SET Configuration Options
            /// <summary> Enable or Disable SAM_ChangeKeyMIFARE and SAM_DumpSessionKey command. </summary>
            ALLOW_DUMP_SESSION_KEY = 0x0000,

            /// <summary> Enable or Disable the reset of init vector after a crypto command. </summary>
            KEEP_IV = 0x0001,

            /// <summary> Enable or Disable the Host key type to provide permissions for Cmd.PLExec execution. </summary>
            PL_KEY = 0x0002,

            /// <summary> Enable or Disable Host Authentication with key other that MasterKey. </summary>
            AUTH_KEY = 0x0003,

            /// <summary> Enable or Disable Key Entry. </summary>
            DISABLE_KEY_ENTRY = 0x0004,

            /// <summary> Enable or Disable LockUnlock. </summary>
            LOCK_KEY = 0x0005,

            /// <summary> Enable or Disable writing the key to a PICC. </summary>
            DISABLE_CHANGE_KEY_PICC = 0x0006,

            /// <summary> Enable or Disable SAM_DecipherData command. </summary>
            DISABLE_DECRYPTION = 0x0007,

            /// <summary> Enable or Disable SAM_EncipherData command. </summary>
            DISABLE_ENCRYPTION = 0x0008,

            /// <summary> Enable or Disable SAM_VerifyMAC command. </summary>
            DISABLE_VERIFY_MAC = 0x0009,

            /// <summary> Enable or Disable SAM_GenerateMAC command. </summary>
            DISABLE_GENERATE_MAC = 0x000A,
            #endregion SET Configuration Options

            #region ExtSET Configuration Options
            /// <summary> Key Class used in key store. </summary>
            KEY_CLASS = 0x000B,

            /// <summary> Enable or Disable SAM_DumpSecretKey command. </summary>
            ALLOW_DUMP_SECRET_KEY = 0x000C,

            /// <summary> Mandate or not key diversification. </summary>
            MANDATE_KEY_DIVERSIFICATION = 0x000D,

            /// <summary> Enable or disable the Key Entry for Sam Personalization. </summary>
            RESERVED_SAM_PERSONALIZATION = 0x000E,

            /// <summary> Enable or disable the Key Entry usage by Internal Host. </summary>
            KEY_USAGE_INTERNAL_HOST = 0x000F,

            /// <summary> Enable or disable the Key Entry change by Internal Host. </summary>
            KEY_CHANGE_INTERNAL_HOST = 0x0010,

            /// <summary> Enable or disable the Session Key usage by Internal Host. </summary>
            SESSION_KEY_USAGE_INTERNAL_HOST = 0x0011,

            /// <summary> Enable or disable the dumping of Secret Key by Internal Host. </summary>
            ALLOW_DUMP_SECRET_KEY_INTERNAL_HOST = 0x0012,

            /// <summary> Enable or disable the dumping of Session Key by Internal Host. </summary>
            ALLOW_DUMP_SESSION_KEY_INTERNAL_HOST = 0x0013,
            #endregion ExtSET Configuration Options

            #region Key Storage Table's Key Entry Configuration Options
            /// <summary> DESFire application ID. </summary>
            DF_AID = 0x0014,

            /// <summary> DESFire key number. </summary>
            DF_KEY_NO = 0x0015,

            /// <summary> Key Number of Change Entry key. </summary>
            KEYNO_CEK = 0x0016,

            /// <summary> Key Version of Change Entry key. </summary>
            KEYV_CEK = 0x0017,

            /// <summary> Reference number of key usage counter. </summary>
            REF_NO_KUC = 0x0018,

            /// <summary> Key Number of Access Entry key. </summary>
            KEYNO_AEK = 0x0019,

            /// <summary> Key Version of Access Entry key. </summary>
            KEYV_AEK = 0x001A,

            /// <summary> Key Number of Change KUC. </summary>
            KEYNO_CKUC = 0x001B,

            /// <summary> Key Version of Change KUC. </summary>
            KEYV_CKUC = 0x001C,

            /// <summary> Option for single DES and 2 Key Triple DES keys. </summary>
            DES_KEY_OPTION = 0x001D,

            /// <summary> Key Number used for MIFARE key A diversification. </summary>
            KEYNO_MF_DIV_A = 0x001E,

            /// <summary> Key Version used for MIFARE key A diversification. </summary>
            KEYV_MF_DIV_A = 0x001F,

            /// <summary> Key Number used for MIFARE key B diversification. </summary>
            KEYNO_MF_DIV_B = 0x0020,

            /// <summary> Key Version used for MIFARE key B diversification. </summary>
            KEYV_MF_DIV_B = 0x0021,

            /// <summary> The AES key to be used is for LRP algorithm. </summary>
            ENABLE_LRP = 0x0022
            #endregion Key Storage Table's Key Entry Configuration Options
        }
        #endregion Configuration Options

        #region KeyClass
        /// <summary>
        /// The type of key classes to be used for SamAV2 or SamAV3.
        /// </summary>
        public enum KeyClass : byte
        {
            /// <summary> Configuring key entry as Host. </summary>
            HOST = 0x00,

            /// <summary> Configuring key entry as PICC. </summary>
            PICC = 0x01,

            /// <summary> Configuring key entry as Offline Change. </summary>
            OFFLINE_CHANGE = 0x02,

            /// <summary> Configuring key entry as Offline Crypto. </summary>
            OFFLINE_CRYPTO = 0x04,

            /// <summary> Configuring key entry as Offline Upload. </summary>
            OFFLINE_UPLOAD = 0x05,

            /// <summary> Configuring key entry as Offline Perso. </summary>
            OFFLINE_PERSO = 0x06
        }
        #endregion KeyClass

        #region DESFire Key Options
        /// <summary>
        /// The additional option to indicate the type of DESFire key.
        /// </summary>
        public enum DesKeyOption : byte
        {
            /// <summary> DESFire 4 compatibility mode. </summary>
            DESFIRE4 = 0x00,

            /// <summary> ISO 10116 mode with CRC16 protection and 4 bytes MAC. </summary>
            ISO_CRC16 = 0x01,

            /// <summary> ISO 10116 mode with CRC32 protection and 8 bytes MAC. </summary>
            ISO_CRC32 = 0x02
        }
        #endregion DESFire Key Options
        #endregion Enumerations

        #region Data Structure
        /// <summary>
        /// SAM AV3 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            public IntPtr pHalDataParams;

            /// <summary> Configuration settings. </summary>
            public fixed byte aSet[2];

            /// <summary> Extended configuration settings. </summary>
            public fixed byte aExtSet[2];

            /// <summary> DESFire application ID. </summary>
            public fixed byte aDFAid[3];

            /// <summary> DESFire key number. </summary>
            public byte bDFKeyNo;

            /// <summary> Key Number of Change Entry Key. </summary>
            public byte bKeyNoCEK;

            /// <summary> Key Version of Change Entry Key. </summary>
            public byte bKeyVCEK;

            /// <summary> Reference number of key usage counter. </summary>
            public byte bRefNoKUC;

            /// <summary> Key Number of Access Entry Key. </summary>
            public byte bKeyNoAEK;

            /// <summary> Key Version of Access Entry Key. </summary>
            public byte bKeyVAEK;

            /// <summary> Key Number of Change KUC. </summary>
            public byte bKeyNoCKUC;

            /// <summary> Key Version of Change KUC. </summary>
            public byte bKeyVCKUC;

            /// <summary> Key Number used for MIFARE key A diversification (has to point to a DES key). </summary>
            public byte bKeyNoMfDivA;

            /// <summary> Key Version used for MIFARE key A diversification (has to point to a DES key). </summary>
            public byte bKeyVMfDivA;

            /// <summary> Key Number used for MIFARE key B diversification (has to point to a DES key). </summary>
            public byte bKeyNoMfDivB;

            /// <summary> Key Version used for MIFARE key B diversification (has to point to a DES key). </summary>
            public byte bKeyVMfDivB;

            /// <summary>
            /// Option for single DES and 2 Key Triple DES keys.
            /// Can be set either to
            ///     <see cref="DesKeyOption.DESFIRE4"/>
            ///     <see cref="DesKeyOption.ISO_CRC16"/>
            ///     <see cref="DesKeyOption.ISO_CRC32"/>
            /// </summary>
            public byte b2K3DESOption;

            /// <summary> Option for LRP key type. If set indicated that the AES key is of LRP type. </summary>
            public byte bIsLRPKey;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_SamAV3_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component to communicate with SamAV3 KeyStore.
        /// </summary>
        ///
        /// <param name="wDataParamsSize">The size of SamAV3 DataParams structure.</param>
        /// <param name="pHal">Object reference of SamAV3 Hal layer.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( ushort wDataParamsSize, Hal.SamAV3 pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_SamAV3_Init ( ref m_DataParamsInt[0], wDataParamsSize, pHal.m_pDataParams );
        }

        /// <summary>
        /// Initializes the KeyStore component to communicate with SamAV3 KeyStore.
        /// </summary>
        ///
        /// <param name="pHal">Object reference of SamAV3 Hal layer.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( Hal.SamAV3 pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_SamAV3_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SamAV3 ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 3;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SamAV3 ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }

        #endregion

        #region Structure Member Access
        /// <summary>
        /// Gets / Sets the KeyStore component identifier.
        /// </summary>
        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        /// <summary>
        /// Gets Pointer information of HAL component.
        /// </summary>
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get { return ( ( GCHandle ) m_DataParamsInt[0].pHalDataParams ).Target as NxpRdLibNet.Hal.SamAV3; }
        }

        /// <summary>
        /// Gets the SET configuration as bytes.
        /// </summary>
        public byte[] SET
        {
            get
            {
                byte[] aSet = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aSet[i] = pDataParams->aSet[i];
                        }
                    }
                }
                return aSet;
            }
        }

        /// <summary>
        /// Gets Extended SET configuration as bytes.
        /// </summary>
        public byte[] ExtSET
        {
            get
            {
                byte[] aExtSet = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aExtSet[i] = pDataParams->aExtSet[i];
                        }
                    }
                }
                return aExtSet;
            }
        }

        /// <summary>
        /// Gets the DESFire Application identifier.
        /// </summary>
        public byte[] DF_Aid
        {
            get
            {
                byte[] aDF_Aid = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            aDF_Aid[i] = pDataParams->aDFAid[i];
                        }
                    }
                }
                return aDF_Aid;
            }
        }

        /// <summary>
        /// Sets / Gets the DESFire Key number.
        /// </summary>
        public byte DF_KeyNo
        {
            get { return m_DataParamsInt[0].bDFKeyNo; }
            set { m_DataParamsInt[0].bDFKeyNo = value; }
        }

        /// <summary>
        /// Gets / Sets Change Entry Key number.
        /// </summary>
        public byte KeyNo_CEK
        {
            get { return m_DataParamsInt[0].bKeyNoCEK; }
            set { m_DataParamsInt[0].bKeyNoCEK = value; }
        }

        /// <summary>
        /// Gets / Sets Change Entry Key version.
        /// </summary>
        public byte KeyV_CEK
        {
            get { return m_DataParamsInt[0].bKeyVCEK; }
            set { m_DataParamsInt[0].bKeyVCEK = value; }
        }

        /// <summary>
        /// Gets / Sets reference KUC entry number.
        /// </summary>
        public byte RefNoKUC
        {
            get { return m_DataParamsInt[0].bRefNoKUC; }
            set { m_DataParamsInt[0].bRefNoKUC = value; }
        }

        /// <summary>
        /// Gets / Sets AEK Key number.
        /// </summary>
        public byte KeyNo_AEK
        {
            get { return m_DataParamsInt[0].bKeyNoAEK; }
            set { m_DataParamsInt[0].bKeyNoAEK = value; }
        }

        /// <summary>
        /// Gets / Sets AEK Key version.
        /// </summary>
        public byte KeyV_AEK
        {
            get { return m_DataParamsInt[0].bKeyVAEK; }
            set { m_DataParamsInt[0].bKeyVAEK = value; }
        }

        /// <summary>
        /// Gets / Sets CKUC Key number.
        /// </summary>
        public byte KeyNo_CKUC
        {
            get { return m_DataParamsInt[0].bKeyNoCKUC; }
            set { m_DataParamsInt[0].bKeyNoCKUC = value; }
        }

        /// <summary>
        /// Gets / Sets CKUC Key version.
        /// </summary>
        public byte KeyV_CKUC
        {
            get { return m_DataParamsInt[0].bKeyVCKUC; }
            set { m_DataParamsInt[0].bKeyVCKUC = value; }
        }

        /// <summary>
        /// Gets / Sets MIFARE Diversification A Key number.
        /// </summary>
        public byte KeyNo_MFDivA
        {
            get { return m_DataParamsInt[0].bKeyNoMfDivA; }
            set { m_DataParamsInt[0].bKeyNoMfDivA = value; }
        }

        /// <summary>
        /// Gets / Sets MIFARE Diversification A Key version.
        /// </summary>
        public byte KeyV_MFDivA
        {
            get { return m_DataParamsInt[0].bKeyVMfDivA; }
            set { m_DataParamsInt[0].bKeyVMfDivA = value; }
        }

        /// <summary>
        /// Gets / Sets MIFARE Diversification B Key number.
        /// </summary>
        public byte KeyNo_MFDivB
        {
            get { return m_DataParamsInt[0].bKeyNoMfDivB; }
            set { m_DataParamsInt[0].bKeyNoMfDivB = value; }
        }

        /// <summary>
        /// Gets / Sets MIFARE Diversification B Key version.
        /// </summary>
        public byte KeyV_MFDivB
        {
            get { return m_DataParamsInt[0].bKeyVMfDivB; }
            set { m_DataParamsInt[0].bKeyVMfDivB = value; }
        }
        #endregion
    }
    #endregion
#endif

    #region Rc632
    /// <summary>
    /// Class for RC632 Mode layer initialization interface and data params.
    /// </summary>
    public class Rc632 : Generic
    {
        #region Data Structure
        /// <summary>
        /// RC632 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            public IntPtr pHalDataParams;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_Rc632_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component as software component
        /// </summary>
        ///
        /// <param name="pHal">Pointer to the parameter structure of the underlying layer.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Rc632_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component as software component
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to a HAL based on NXP RC632 IC.</param>
        ///
        /// <returns>Returns SUCCESS for successful operations.
        ///          Other depending on the layer's implementation.</returns>
        public Status_t Init ( int wDataParamSize, Hal.Generic pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Rc632_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, pHal.m_pDataParams );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rc632 ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 1;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rc632 ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }

        #endregion
    }
    #endregion

    #region Rc663
    /// <summary>
    /// Class for RC663 Mode layer initialization interface and data params.
    /// </summary>
    public class Rc663 : Generic
    {
        #region Data Structure
        /// <summary>
        /// RC663 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the underlying layer. </summary>
            public IntPtr pHalDataParams;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_Rc663_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component as RC663 component.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to a HAL based on NXP RC663 IC.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Generic pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Rc663_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ), pHal.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component as RC663 component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHal">Pointer to a HAL based on NXP RC663 IC.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.Generic pHal )
        {
            // Allocate internal key entries and pointer to them
            return phKeyStore_Rc663_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, pHal.m_pDataParams );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rc663 ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 1;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rc663 ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }

        #endregion
    }
    #endregion

    #region Rd710
    /// <summary>
    /// Class for RD710 Mode layer initialization interface and data params.
    /// </summary>
    public class Rd710 : Generic
    {
        #region Data Structure
        /// <summary>
        /// RD710 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the Hal parameter structure. </summary>
            IntPtr pHalDataParams;
        };

        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_Rd710_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component as RD710 component.
        /// </summary>
        ///
        /// <param name="pHal">Pointer to a HAL based on NXP RD710 IC.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.Rd710 pHal )
        {
            return phKeyStore_Rd710_Init ( ref m_DataParamsInt[0],
                                            ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                            pHal.m_pDataParams
                                            );
        }

#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component as RD710 component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure</param>
        /// <param name="pHal">Pointer to a HAL based on NXP RD710 IC.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.Rd710 pHal )
        {
            return phKeyStore_Rd710_Init ( ref m_DataParamsInt[0],
                                            ( ushort ) wDataParamSize,
                                            pHal.m_pDataParams
                                            );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort m_wNumKeyVersionPairs_Int;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rd710 ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
            m_wNumKeyVersionPairs_Int = 3;
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rd710 ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        private DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return m_wNumKeyVersionPairs_Int;
            }
        }
        #endregion
    }
    #endregion

    #region DUT
    /// <summary>
    /// Class for DUT Mode layer initialization interface and DataParams.
    /// </summary>
    public class DUT : Generic
    {
#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Enumerations
        #region Provision
        /// <summary>
        /// Option indicating the type of key to be provisioned.
        /// </summary>
        public enum Provision : ushort
        {
            /// <summary>
            /// Option to exclude of provisioning of AES 128Bit and AES 256Bit Application Root Key.
            /// </summary>
            DISABLED = 0x0000,

            /// <summary>
            /// Option to provision both AES 128Bit and 256Bit Application Root Keys.
            /// </summary>
            APPLICATION_ROOT_KEYS = 0x0001
        }
        #endregion

        #region Configuration
        /// <summary>
        /// Option to update the provisioning data like Expected Decrypted data, Initialization Vector for Wrapping,
        /// Application Root Keys,etc...
        /// </summary>
        public enum Config : ushort
        {
            /// <summary>
            /// Updates the existing AES 128-Bit Application root key used during component <see cref="Provision_Init"/>
            ///
            /// Note:
            ///     - Using this configuration, the key will be stored in the DataParams for further
            ///       Provisioning of Fixed or ASymmetric Keys.
            ///     - Application Root Key configured will be used from next operations.
            /// </summary>
            APP_ROOT_KEY_AES128_BIT = 0x00A1,

            /// <summary>
            /// Updates the existing AES 256-Bit Application root key used during component <see cref="Provision_Init"/>
            ///
            /// Note:
            ///     - Using this configuration, the key will be stored in the DataParams for further
            ///       Provisioning of Fixed or ASymmetric Keys.
            ///     - Application Root Key configured will be used from next operations.
            /// </summary>
            APP_ROOT_KEY_AES256_BIT,

            /// <summary>
            /// Updates the existing Derivation message used for opening a Session or wrapping key during
            /// component <see cref="Provision_Init"/>
            ///
            /// Note:
            ///     - Using this configuration, the key will be stored in the DataParams for further
            ///       Provisioning of Fixed or ASymmetric Keys.
            ///     - Application Root Key configured will be used from next operations.
            /// </summary>
            DERIV_MSG,

            /// <summary>
            /// Expected Updates the existing Expected Decryption Data used for opening a Session or wrapping key during
            /// component <see cref="Provision_Init"/>
            ///
            /// Note:
            ///     - Using this configuration, the key will be stored in the DataParams for further
            ///       Provisioning of Fixed or ASymmetric Keys.
            ///     - Application Root Key configured will be used from next operations.
            /// </summary>
            EXPECTED_DEC_DATA,

            /// <summary>
            /// Updates the existing Initialization Vector (WIV) used for opening a Session or wrapping key during
            /// component <see cref="Provision_Init"/>
            ///
            /// Note:
            ///     - Using this configuration, the key will be stored in the DataParams for further
            ///       Provisioning of Fixed or ASymmetric Keys.
            ///     - Application Root Key configured will be used from next operations.
            /// </summary>
            WRAP_IV
        }
        #endregion
        #endregion
#endif

        #region Data Structure
        /// <summary>
        /// RD710 parameter structure
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public struct DataParams_t
        {
            /// <summary> Layer ID for this HAL component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the Hal parameter structure. </summary>
            IntPtr pHalDataParams;
        };

        #endregion

        #region DLL Imports
        #region Initialization
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalDataParams, ushort wNoOfKeyEntries );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_Provision_Init ( ref DataParams_t m_pDataParams, ushort wOption, byte[] pTransportKey_AES128,
            byte[] pTransportKey_AES256, byte[] pAppRootKey_AES128, byte[] pAppRootKey_AES256, byte[] pExpDecData, byte[] pDervMsg,
            byte[] pWrapIV );
#endif
        #endregion

        #region De-Initialization
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_DeInit ( ref DataParams_t m_pDataParams );

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern void phKeyStore_DUT_Provision_DeInit ( ref DataParams_t m_pDataParams );
#endif
        #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Key Provisioning
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_SetConfig_Ext ( ref DataParams_t m_pDataParams, ushort wConfig, byte[] pValue, ushort wValueLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_GetConfig_Ext ( ref DataParams_t m_pDataParams, ushort wConfig, byte[] pValue, ref ushort pValueLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phKeyStore_DUT_Provision_AppFixedKeys ( ref DataParams_t m_pDataParams, byte bKeyIndex, ushort wKeyType,
            byte[] pNewKey );
        #endregion
#endif
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the KeyStore component as DUT component.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to a DUT HAL.</param>
        /// <param name="wNoOfKeyEntries"></param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.DUT oHal, ushort wNoOfKeyEntries = 16 )
        {
            this.wNoOfKeyEntries = wNoOfKeyEntries;
            return phKeyStore_DUT_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oHal.m_pDataParams, wNoOfKeyEntries );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// Initializes DUT KeyStore component for Key provisioning.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to a DUT HAL.</param>
        /// <param name="wOption">Options to perform the following.
        ///                         - <see cref="Provision.DISABLED"/>: Provisioning of Application Root Key not required.
        ///                         - <see cref="Provision.APPLICATION_ROOT_KEYS"/>: Provision both 128-Bit and 256-Bit Application Root Key.
        /// </param>
        /// <param name="aTransportKey_AES128">Array containing AES 128-Bit [16 Byte] Transport Key.
        ///                                    Can be NULL if \b wOption = <see cref="Provision.DISABLED"/>
        /// </param>
        /// <param name="aTransportKey_AES256">Array containing AES 256-Bit [32 Byte] Transport Key.
        ///                                    Can be NULL if \b wOption = <see cref="Provision.DISABLED"/>
        /// </param>
        /// <param name="aAppRootKey_AES128">Array containing AES 128-Bit [16 Byte]  Application Root Key. Should not be Null.</param>
        /// <param name="aAppRootKey_AES256">Array containing AES 256-Bit [32 Byte]  Application Root Key. Should not be Null.</param>
        /// <param name="aExpDecData">Array containing AES 128-Bit  [16 Byte] expected Decrypted data to be used while
        ///                        opening the Session. Should not be NULL.
        /// </param>
        /// <param name="aDervMsg">Array containing derivation message. Should not be NULL.
        ///                             - Derivation message to derive a wrapping key from existing Transport Key
        ///                               (NXP_TPT_KEY) or Application Root Key ( APP_ROOT_KEY)
        ///                             - Will be used while opening a session and provisioning the Keys.
        ///                             - Should be 24 bytes in length.
        /// </param>
        /// <param name="aWrapIV">IV used for wrapping the key. This information will be used while provisioning the Key.
        ///                         - Should be 16 bytes in length.
        ///                         - If NULL, Zero IV will be used for wrapping the key.
        /// </param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Provision_Init ( Hal.DUT oHal, ushort wOption, byte[] aTransportKey_AES128, byte[] aTransportKey_AES256, byte[] aAppRootKey_AES128,
            byte[] aAppRootKey_AES256, byte[] aExpDecData, byte[] aDervMsg, byte[] aWrapIV )
        {
            return phKeyStore_DUT_Provision_Init ( ref m_DataParamsInt[0], wOption, aTransportKey_AES128, aTransportKey_AES256, aAppRootKey_AES128,
                aAppRootKey_AES256, aExpDecData, aDervMsg, aWrapIV );
        }
#endif

#if DEBUG
        /// <summary>
        /// Initializes the KeyStore component as DUT component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure</param>
        /// <param name="oHal">Pointer to a DUT HAL.</param>
        /// <param name="wNoOfKeyEntries"></param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( int wDataParamSize, Hal.DUT oHal, ushort wNoOfKeyEntries = 16 )
        {
            this.wNoOfKeyEntries = wNoOfKeyEntries;
            return phKeyStore_DUT_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, oHal.m_pDataParams, wNoOfKeyEntries );
        }
#endif
        #endregion

        #region De-Initialization
        /// <summary>
        /// De-Initialize the DUT component.
        /// NOTE: Its must to call this interface to Un-initialize any used global context from other libraries.
        /// If not called, there might be unusual behavior for the next executions.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to a DUT HAL.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t DeInit ( Hal.DUT oHal )
        {
            return phKeyStore_DUT_DeInit ( ref m_DataParamsInt[0] );
        }

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        /// <summary>
        /// De-Initializes DUT KeyStore component for Key provisioning.
        /// </summary>
        ///
        /// <param name="oHal">Pointer to a DUT HAL.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public void Provision_DeInit ( Hal.DUT oHal )
        {
            phKeyStore_DUT_Provision_DeInit ( ref m_DataParamsInt[0] );
        }
#endif
        #endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
        #region Key Provisioning
        /// <summary>
        /// Set configuration parameter.
        /// </summary>
        ///
        /// <param name="wConfig"> Configuration Identifier based on the DUT being used.
        ///                         - <see cref="Config.APP_ROOT_KEY_AES128_BIT"/>
        ///                         - <see cref="Config.APP_ROOT_KEY_AES256_BIT"/>
        ///                         - <see cref="Config.DERIV_MSG"/>
        ///                         - <see cref="Config.EXPECTED_DEC_DATA"/>
        ///                         - <see cref="Config.WRAP_IV"/>
        /// </param>
        /// <param name="aValue">Configuration Value to update.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig_Ext ( ushort wConfig, byte[] aValue )
        {
            return phKeyStore_DUT_SetConfig_Ext ( ref m_DataParamsInt[0], wConfig, aValue, ( ushort ) ( ( aValue == null ) ? 0 : aValue.Length ) );
        }

        /// <summary>
        /// Get configuration parameter
        /// </summary>
        ///
        /// <param name="wConfig"> Configuration Identifier based on the DUT being used.
        ///                         - <see cref="Config.APP_ROOT_KEY_AES128_BIT"/>
        ///                         - <see cref="Config.APP_ROOT_KEY_AES256_BIT"/>
        ///                         - <see cref="Config.DERIV_MSG"/>
        ///                         - <see cref="Config.EXPECTED_DEC_DATA"/>
        ///                         - <see cref="Config.WRAP_IV"/>
        /// </param>
        /// <param name="aValue">Configuration Value to retrieve.</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig_Ext ( ushort wConfig, out byte[] aValue )
        {
            ushort wValLen = 0;
            aValue = new byte[512];

            Status_t oStatus = phKeyStore_DUT_GetConfig_Ext ( ref m_DataParamsInt[0], wConfig, aValue, ref wValLen );
            if ( oStatus.Equals ( new Status_t () ) )
                Array.Resize ( ref aValue, wValLen );

            return oStatus;
        }

        /// <summary>
        /// Provision Application Fixed Keys.
        /// </summary>
        ///
        /// <param name="bKeyIndex">Key number of the hardware keys store to be provisioned.
        ///                             - For Symmetric Keys it will be 10h - 1Ah
        ///                             - For ASymmetric Keys it will be 1Bh - 21h
        /// </param>
        /// <param name="wKeyType">Key type of the key to be loaded. Should be one of the following
        ///                         - <see cref="KeyType.AES128"/>
        ///                         - <see cref="KeyType.AES256"/>
        /// </param>
        /// <param name="aNewKey">Pointer to the key itself</param>
        ///
        /// <returns>Returns Success status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        public Status_t Provision_AppFixedKeys ( byte bKeyIndex, ushort wKeyType, byte[] aNewKey )
        {
            return phKeyStore_DUT_Provision_AppFixedKeys ( ref m_DataParamsInt[0], bKeyIndex, wKeyType, aNewKey );
        }
        #endregion
#endif

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;
        private ushort wNoOfKeyEntries = 0;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public DUT ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~DUT ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }

        /// <summary>
        /// Override m_numVersions from abstract base class
        /// </summary>
        public override int m_wNumKeyVersionPairs
        {
            get
            {
                return wNoOfKeyEntries;
            }
        }
        #endregion
    }
    #endregion
}
