/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alP40CmdPub
{
    #region DEFINES

    public enum Error : byte
    {
        /// <summary> P40 Bad Command Format Error. </summary>
        FORMAT = (CustomCodes.ERROR_BEGIN),
        /// <summary> P40 Invalid Parameters Error. </summary>
        PARAM,
        /// <summary> P40 Not Authorized Error. </summary>
        NOT_AUTHORIZED,
        /// <summary> P40 Internal Error </summary>
        INTERNAL,
        /// <summary> P40 Unknown Command Error. </summary>
        COMMAND,
        /// <summary> P40 Generic Error. </summary>
        UNKNOWN
    }

    public enum Application : byte
    {
        App1 = 0x01,
        App2 = 0x02
    }

    public enum LcSwitch : byte
    {
        AppMgmtToPreRelease = 0x00,
        AppMgmtToRelease = 0x01,
        PreReleaseToRelease = 0x02
    }

    public enum FastTestType : byte
    {
        All = 0x00
    }

    public enum KeyType : byte
    {
        CardMaster = 0,
        App1 = 1,
        App2 = 2,
        Originality = 3

    }
    #endregion

    #region VERSION_INFO
    public class VersionInfo
    {
        //define the struct used for passing
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct VersionInfo_t
        {
            public ushort wIcVer;                    /**<  IC version version */
            public ushort wRomCodeVer;      /**< ROM code version */
            public ushort wWaferTestVer;      /**< WaferTest Program version */
            public byte bLifecycle;              /**< Current Lifeycle */
            /* reserved bytes for struct so total size is 16*/
            public fixed byte bReserve[9];
        };

        #endregion

        #region GETTERS/SETTERS
        // Setter & Getter for VersionInfo_t struct
        public VersionInfo_t Data
        {
            get
            {
                return this.m_version;
            }
            set
            {
                this.m_version = value;
            }
        }

        public ushort Ic
        {
            get
            {
                return this.m_version.wIcVer;
            }
        }

        public ushort RomCode
        {
            get
            {
                return this.m_version.wRomCodeVer;
            }
        }

        public ushort WaferTest
        {
            get
            {
                return this.m_version.wWaferTestVer;
            }
        }

        public byte Lifecycle
        {
            get
            {
                return this.m_version.bLifecycle;
            }
        }

        #endregion


        private VersionInfo_t m_version;

        public VersionInfo()
        {
            m_version = new VersionInfo_t();
        }
    }
    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_GetVersion(
                                       IntPtr pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                                       byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                       ref VersionInfo.VersionInfo_t pVersionData   /**< [Out] Pointer to store GetVersion return data. */
                                       );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_DownloadCode(
                                        IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                        ushort wAddr,			  /**< [In] specify starting address to write code. */
                                        byte[] pCode,            /**< [In] Pointer to code to be written. */
                                        ushort wCodeLength        /**< [In] Size of the code to be written. */
                                        );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_DownloadConst(
                                                IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                                ushort wAddr,			    /**< [In] specify starting address to write code. */
                                                byte[] pConst,             /**< [In] Pointer to const to be written. */
                                                ushort wConstLength         /**< [In] Size of the const to be written. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_DownloadEE(
                                                IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
                                                ushort wAddr,			    /**< [In] specify starting address to write code. */
                                                byte[] pEEData,            /**< [In] Pointer to EE data to be written. */
                                                ushort wEEDataLength        /**< [In] Size of the EE data  to be written. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_DownloadPatch(
                                                IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                byte bPatchNr,          /**< [In] Number of patch to be written */
                                                ushort wPatchAddress,   /**< [In] Address in the code area to be patched */
                                                byte[] pPatch,           /**< [In] Pointer to patch code to be written. */
                                                ushort wPatchLength       /**< [In] Size of patch code to be written. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_DownloadFlash(
                                                IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                ushort wAddr,           /**< [In] specify starting address to write code. */
                                                byte[] pFlashData,     /**< [In] Pointer to Flash data to be written. */
                                                ushort wFlashDataLength /**< [In] Size of the Flash data  to be written. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_LockApplication(
                                                IntPtr pDataParams        /**< [In] Pointer to this layer's parameter structure. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_FormatApplication(
                                                IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                byte bApplicationId	  /**< [In] Application ID to be formatted.*/
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_SwitchLifeCycle(
                                                IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bLcType                /**< [In] Life cycle type to be executed.*/
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_ExecuteTrapdoor(
                                                IntPtr pDataParams,        /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId      /**< [In] Current LifeCycle . */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_SetConfiguration(
                                                IntPtr pDataParams,                /**< [In] Pointer to this layer's parameter structure. */
                                                byte bDisableCmdTestFuse, /**< [In] Enables or disables cmd based test fuse.*/
                                                byte bDisableTPTestFuse,     /**< [In] Enables or disables test pad based test fuse.*/
                                                byte bUseExtendedConfig,   /**< [In] If set, the configs that follow are also sent to the card*/
                                                ushort wApp1CodeSize,       /**< [In] Code size of application 1. */
                                                ushort wApp1ConstSize,       /**< [In] Const size of application 1. */
                                                ushort wApp2CodeSize,       /**< [In] Code size of application 2. */
                                                ushort wApp2ConstSize,       /**< [In] Const size of application 2. */
                                                ushort wCommBufferSize,     /**< [In] Communication Buffer size. */
                                                byte bNumKeys,                    /**< [In] Number of keys in RAM keystore. */
                                                byte bDefaultApp                   /**< [In] Default Application. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_GetConfiguration(
                                                IntPtr pDataParams,                        /**< [In] Pointer to this layer's parameter structure. */
                                                ref byte pDisableCmdTestFuse,    /**< [Out] Enables or disables cmd based test fuse.*/
                                                ref byte pDisableTPTestFuse,       /**< [Out] Enables or disables test pad based test fuse.*/
                                                ref ushort pApp1CodeSize,           /**< [Out] Code size of application 1. */
                                                ref ushort pApp1ConstSize,          /**< [Out] Const size of application 1. */
                                                ref ushort pApp2CodeSize,           /**< [Out] Code size of application 2. */
                                                ref ushort pApp2ConstSize,  	        /**< [Out] Const size of application 2. */
                                                ref ushort pCommBufferSize,	        /**< [Out] Communication Buffer size. */
                                                ref byte pNumKeys,                       /**< [Out] Number of keys in RAM keystore. */
                                                ref byte pDefaultApp                      /**< [Out] Default Application. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_FastTest(
                                                IntPtr pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bFastTestType	  /**< [In] Fast Test type to be executed.*/
                                                );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_GetChallenge(
                                                IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bChallengeLen,/**< [In] Requested  Challenge length. */
                                                byte[] pChallenge     /**< [Out] returned  challenge string. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_ExternalAuthenticate(
                                                IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bCryptoMethod,      /**< [In] The crypto method to use  i.e ECB or CBC . */
                                                byte bKeyId,      /**< [In] Ident of Key to authenticate to. */
                                                byte[] pEdata,            /**< [In] encrypted RND pcd-picc string.. */
                                                byte bEdataLen       /**< [In] Length of Edata. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_InternalAuthenticate(
                                                IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bCryptoMethod,      /**< [In] The crypto method to use  i.e ECB or CBC . */
                                                byte bKeyId,      /**< [In] Ident of Key to authenticate to. */
                                                byte[] pEdata,            /**< [In] encrypted RND pcd-picc string.. */
                                                byte bEdataLen       /**< [In] Length of Edata. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_Authenticate(
                                                IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                                byte bLifeCycleId,      /**< [In] Current LifeCycle . */
                                                byte bKeyId,      /**< [In] Ident of Key to authenticate to. */
                                                ushort wKeyNumber,     /**< [In] Key Storage number. */
                                                ushort wKeyVersion    /**< [In] Key Storage version. */
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_MaintainanceReqa(
                                                IntPtr pDataParams /**< [In] Pointer to this layer's parameter structure. */
                                                );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t GetVersion(
            int bLc,
            out VersionInfo Ver)
        {
            Status_t status;

            Ver = new VersionInfo();
            VersionInfo.VersionInfo_t vi = new VersionInfo.VersionInfo_t();
            status = phalP40CmdPub_GetVersion(m_pDataParams, (byte)(bLc - AppMgmtLifecycle), ref vi);
            Ver.Data = vi;
            return status;
        }

        public Status_t DownloadCode(
            int wAddr,
            byte[] pCode)
        {
            return phalP40CmdPub_DownloadCode(m_pDataParams, (ushort)wAddr, pCode, (ushort)pCode.Length);
        }

        public Status_t DownloadConst(
            int wAddr,
            byte[] pConst)
        {
            return phalP40CmdPub_DownloadConst(m_pDataParams, (ushort)wAddr, pConst, (ushort)pConst.Length);
        }

        public Status_t DownloadEE(
            int wAddr,
            byte[] pEEData)
        {
            return phalP40CmdPub_DownloadEE(m_pDataParams, (ushort)wAddr, pEEData, (ushort)pEEData.Length);
        }

        public Status_t DownloadPatch(
            int bPatchNr,
            int wPatchAddress,
            byte[] pPatch)
        {
            return phalP40CmdPub_DownloadPatch(m_pDataParams, (byte)bPatchNr, (ushort)wPatchAddress, pPatch, (ushort)pPatch.Length);
        }

        public Status_t DownloadFlash(
            int wAddr,
            byte[] pFlashData)
        {
            return phalP40CmdPub_DownloadFlash(m_pDataParams, (ushort)wAddr, pFlashData, (ushort)pFlashData.Length);
        }

        public Status_t LockApplication(
            )
        {
            return phalP40CmdPub_LockApplication(m_pDataParams);
        }

        public Status_t FormatApplication(
            int bApplicationId
            )
        {
            return phalP40CmdPub_FormatApplication(m_pDataParams, (byte)bApplicationId);
        }

        public Status_t SwitchLifeCycle(
            int bLcCurrent,
            int bLcTo
            )
        {
            return phalP40CmdPub_SwitchLifeCycle(m_pDataParams, (byte)(bLcCurrent - AppMgmtLifecycle), (byte)bLcTo);
        }

        public Status_t ExecuteTrapdoor(
            int bLcCurrent
            )
        {
            return phalP40CmdPub_ExecuteTrapdoor(m_pDataParams, (byte)(bLcCurrent - AppMgmtLifecycle));
        }

        public Status_t SetConfiguration(
            int bDisableCmdTestFuse,
            int bDisableTPTestFuse,
            int bUseExtendedConfig,
            int wApp1CodeSize,
            int wApp1ConstSize,
            int wApp2CodeSize,
            int wApp2ConstSize,
            int wCommBufferSize,
            int bNumKeys,
            int bDefaultApp
            )
        {
            return phalP40CmdPub_SetConfiguration(
                m_pDataParams,
                (byte)bDisableCmdTestFuse, (byte)bDisableTPTestFuse, (byte)bUseExtendedConfig,
                (ushort)wApp1CodeSize, (ushort)wApp1ConstSize, (ushort)wApp2CodeSize, (ushort)wApp2ConstSize,
                (ushort)wCommBufferSize, (byte)bNumKeys, (byte)bDefaultApp);
        }

        public Status_t GetConfiguration(
            out byte pDisableCmdTestFuse,
            out byte pDisableTPTestFuse,
            out ushort pApp1CodeSize,
            out ushort pApp1ConstSize,
            out ushort pApp2CodeSize,
            out ushort pApp2ConstSize,
            out ushort pCommBufferSize,
            out byte pNumKeys,
            out byte pDefaultApp
            )
        {
            pDisableCmdTestFuse = 0;
            pDisableTPTestFuse = 0;
            pApp1CodeSize = 0;
            pApp1ConstSize = 0;
            pApp2CodeSize = 0;
            pApp2ConstSize = 0;
            pCommBufferSize = 0;
            pNumKeys = 0;
            pDefaultApp = 0;
            return phalP40CmdPub_GetConfiguration(m_pDataParams, ref pDisableCmdTestFuse, ref pDisableTPTestFuse,
                                                ref pApp1CodeSize, ref pApp1ConstSize, ref pApp2CodeSize, ref pApp2ConstSize, ref pCommBufferSize,
                                                ref pNumKeys, ref pDefaultApp);
        }

        public Status_t FastTest(
            int bLcCurrent,
            int bFastTestType
            )
        {
            return phalP40CmdPub_FastTest(m_pDataParams, (byte)(bLcCurrent - AppMgmtLifecycle), (byte)bFastTestType);
        }

        public Status_t GetChallenge(
            int bLcCurrent,
            int bLen,
            out byte[] pOut
            )
        {
            Status_t status;
            byte[] tmpBuf = new byte[bLen];

            status = phalP40CmdPub_GetChallenge(m_pDataParams,
                                                                                (byte)(bLcCurrent - AppMgmtLifecycle),
                                                                                (byte)bLen,
                                                                                tmpBuf);
            if (!status.Equals(Error_Gen.SUCCESS))
            {
                pOut = new byte[0];
            }
            else
            {
                pOut = tmpBuf;
            }
            return status;
        }

        public Status_t ExternalAuthenticate(
            int bLcCurrent,
            int bCryptoMethod,
            int bKeyId,
            byte[] pEdata
            )
        {
            Status_t status;
            status = phalP40CmdPub_ExternalAuthenticate( m_pDataParams,
                                                                                            (byte)(bLcCurrent - AppMgmtLifecycle),
                                                                                            (byte)bCryptoMethod,
                                                                                            (byte)bKeyId,
                                                                                            pEdata,
                                                                                            (byte)pEdata.Length);
            return status;
        }

        public Status_t Authenticate(
            int bLcCurrent,
            int bKeyId,
            int wKeyNumber,
            int wKeyVersion
            )
        {
            return phalP40CmdPub_Authenticate(  m_pDataParams,
                                                                            (byte)(bLcCurrent - AppMgmtLifecycle),
                                                                            (byte)bKeyId,
                                                                            (ushort)wKeyNumber,
                                                                            (ushort)wKeyVersion);
        }

        public Status_t MaintainanceReqa(
            )
        {
            return phalP40CmdPub_MaintainanceReqa(m_pDataParams);
        }


        #endregion

        #region MEMORY_MAPPING
        protected const byte AppMgmtLifecycle = 3;
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }
    #endregion

    #region Sw

    public class Sw : Generic
    {

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct phalP40CmdPub_Sw_CurrentApdu_t
        {
            ushort wLc;	 /**< Total number of command bytes to be sent */
            ushort wLe; /**< Total number of command bytes to be received*/
            byte bSw1; /**< Returned SW1 byte of the current command*/
            byte bSw2; /**< Returned SW2 byte of the current command*/
            byte bFlags; /**< Bit flags for command*/
        } ;

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            ushort wId;		  /**< Layer ID for this component, NEVER MODIFY! */
            IntPtr pPalMifare;    /**< Pointer to the parameter structure of the underlying palMifare layer. */
            IntPtr pKeyStore;     /**< Pointer to the parameter structure of the key store component. */
            IntPtr pHal;  /**< Pointer to the parameter structure of the underlying hal layer. */
            IntPtr pCryptoEnc;   /**< Pointer to the parameter structure of the Crypto layer for encryption. */
            IntPtr pCryptoRng; /**< Pointer to the parameter structure of the CryptoRng layer. */
            fixed byte bSessionKey[24];                    /**< Session key for this authentication */
            fixed byte bIv[16];                            /**< Max size of IV can be 16 bytes */
            short wCrc;           /**< 2 Byte CRC initial value in Authenticate mode. */
            phalP40CmdPub_Sw_CurrentApdu_t sCurrentApdu;
        };

        #endregion

        #region DLLIMPORTS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPub_Sw_Init(
                                          ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure. */
                                          ushort wDataParamSize,                      /**< [In] Specifies the size of the data parameter structure. */
                                          IntPtr pPalMifareDataParams,              /**< [In] Pointer to the parameter structure of the pal Mifare layer. */
                                          IntPtr pKeyStoreDataParams,              /**< [In] Pointer to the parameter structure of the keystore layer. */
                                          IntPtr pHalDataParams,                        /**< [In] Pointer to the parameter structure of the hal layer. */
                                          IntPtr pCryptoDataParamsEnc,            /**< [In] Pointer to a Crypto component context for encryption. */
                                          IntPtr pCryptoRngDataParams            /**< [In] Pointer to a CryptoRng component context. */
            );
        #endregion

        #region INIT

        public Status_t Init(
                        palMifare.Generic pPalMifare,
                        KeyStore.Generic pKeyStore,
                        Hal.Generic pHal,
                        CryptoSym.Generic pCryptoEnc,
                        CryptoRng.Generic pCryptoRng
                        )
        {

            return phalP40CmdPub_Sw_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pPalMifare.m_pDataParams,
                pKeyStore.m_pDataParams,
                pHal.m_pDataParams,
                pCryptoEnc.m_pDataParams,
                pCryptoRng.m_pDataParams);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion

    }

    #endregion

}

