/*
 * Copyright 2013 - 2014, 2018 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfdf
{
	#region Enumerations
	#region Error Codes
	/// <summary>
	/// Custom error codes equivalent to C library error codes.
	/// </summary>
	public enum Error : byte
	{
		/// <summary> MF DF Response - No changes done to backup files. </summary>
		PHAL_MFDF_NO_CHANGES = 0x80,

		/// <summary> MF DF Response - Insufficient NV-Memory. </summary>
		PHAL_MFDF_ERR_OUT_OF_EEPROM_ERROR,

		/// <summary> MF DF Invalid key number specified. </summary>
		PHAL_MFDF_ERR_NO_SUCH_KEY,

		/// <summary> MF DF Current configuration/status does not allow the requested command. </summary>
		PHAL_MFDF_ERR_PERMISSION_DENIED,

		/// <summary> MF DF Requested AID not found on PICC. </summary>
		PHAL_MFDF_ERR_APPLICATION_NOT_FOUND,

		/// <summary> MF DF Attempt to read/write data from/to beyond the files/record's limits. </summary>
		PHAL_MFDF_ERR_BOUNDARY_ERROR,

		/// <summary> MF DF Previous cmd not fully completed. Not all frames were requested or provided by the PCD. </summary>
		PHAL_MFDF_ERR_COMMAND_ABORTED,

		/// <summary> MF DF Num. of applns limited to 28. No additional applications possible. </summary>
		PHAL_MFDF_ERR_COUNT,

		/// <summary> MF DF File/Application with same number already exists. </summary>
		PHAL_MFDF_ERR_DUPLICATE,

		/// <summary> MF DF Specified file number does not exist. </summary>
		PHAL_MFDF_ERR_FILE_NOT_FOUND,

		/// <summary> MF DF Crypto error returned by PICC. </summary>
		PHAL_MFDF_ERR_PICC_CRYPTO,

		/// <summary> MF DF Parameter value error returned by PICC. </summary>
		PHAL_MFDF_ERR_PARAMETER_ERROR,

		/// <summary> MF DF DesFire Generic error. Check additional Info. </summary>
		PHAL_MFDF_ERR_DF_GEN_ERROR,

		/// <summary> MF DF ISO 7816 Generic error. Check Additional Info. </summary>
		PHAL_MFDF_ERR_DF_7816_GEN_ERROR
	};
	#endregion Error Codes
	#endregion Enumerations

	#region Generic
	/// <summary>
	/// Class having the wrapper for C command.
	/// </summary>
	public abstract class Generic
	{
		#region Constants
		public const byte READ_BLOCK_LENGTH = 16;
		public const byte WRITE_BLOCK_LENGTH = 4;
		public const byte COMPWRITE_BLOCK_LENGTH = 16;

		public const byte DES_BLOCK_SIZE = 8;
		public const byte DES_KEY_LENGTH = 16;
		#endregion

		#region DLL Imports

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_Authenticate (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 ushort wOption,    /**< [In] whether NEW, OLD or NO diversification is used */
								 ushort wKeyNo,        /**< [In] 1 byte key number in key store to authenticate with */
								 ushort wKeyVer,       /**< [In] Key version in the key store */
								 byte bKeyNoCard,    /**< [In] Key number on card */
								 byte[] pDivInput,   /**< [In] Diversification input. Can be NULL */
								 byte bDivLen        /**< [In] Length of diversification input max 31B */
								 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_AuthenticateISO (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 ushort wOption,   /**< [In] whether NEW, OLD or NO diversification is used */
								 ushort wKeyNo,        /**< [In] 1 byte key number in key store to authenticate with */
								 ushort wKeyVer,       /**< [In] Key version in the key store */
								 byte bKeyNoCard,    /**< [In] Key number on card */
								 byte[] pDivInput,   /**< [In] Diversification input. Can be NULL */
								 byte bDivLen        /**< [In] Length of diversification input max 31B */
								 );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_AuthenticateAES (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 ushort wOption,   /**< [In] whether NEW, OLD or NO diversification is used */
								 ushort wKeyNo,        /**< [In] 1 byte key number in key store to authenticate with */
								 ushort wKeyVer,       /**< [In] Key version in the key store */
								 byte bKeyNoCard,    /**< [In] Key number on card */
								 byte[] pDivInput,   /**< [In] Diversification input. Can be NULL */
								 byte bDivLen        /**< [In] Length of diversification input max 31B */
								 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ChangeKeySettings (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 byte bKeySettings        /**< [In] New key settings */
								 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetKeySettings (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 byte[] pResponse       /**< [Out]: 3 bytes + MAC size */
								 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ChangeKey (
								IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
								ushort wOption,     /**< [In] whether NEW, OLD or NO diversification is used */
								ushort wKeyNo,      /**< [In] Old key number in keystore. */
								ushort wKeyVer,     /**< [In] Old key version in keystore. */
								ushort wNewKeyNo,   /**< [In] New key number in keystore. */
								ushort bNewKeyVer,  /**< [In] New key version in keystore. */
								byte bKeyNoCard, /**< [In] Card key number. Key type of new key is coded in the fist two bits for PICC master key */
								byte[] pDivInput, /**< [In] Diversification input. Can be NULL */
								byte bDivLen /**< [In] Length of diversification input max 31B */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetKeyVersion (
								 IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
								 byte bKeyNo, /**< [In]: 1 byte key number */
								 byte[] pResponse /**< [Out]: 1 byte key version */
								 );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateApplication (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption,/**< bOption=01 means wISOFileId is supplied, bOption=02 means pISODFName is present.
						                        bOption=03 means both of these are present, bOption=00 means both not present */
								byte[] pAid, /**< [In] array of 3 bytes */
								byte bKeySettings1, /**< [In] 1 byte */
								byte bKeySettings2, /**< [In] 1 byte */
								byte[] pISOFileId, /**< [In] 2 btyes ISO File ID */
								byte[] pISODFName, /**< [In] 1 to 16 Bytes. Can also be NULL */
								byte bISODFNameLen /**< [In] Size of pISODFName if that is present */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_DeleteApplication (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pAid /**< [In] 3 byte array */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetApplicationIDs (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pAidBuff, /**< [Out] should be >= 96B. (3 * 28) */
								ref byte pNumAIDs /**< [Out] Number of AIDs read */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetDFNames (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption,         /**< PH_EXCHANGE_DEFAULT or PH_EXCHANGE_RXCHAINING */
								byte[] pDFBuffer, /**< [Out] Should be atleast 21 * 28 = 588+28 = 616 bytes */
								ref byte bSize /**< [Out] for this API to fill the number of DFs filled in df buffer */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SelectApplication (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pAppId /**< [In] 3 byte App Id. LSB First */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_FormatPICC (
								IntPtr pDataParams /**< [In] Pointer to this layers param structure */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetVersion (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pVerInfo /**< [Out] 28bytes of version info. User has to parse this */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_FreeMem (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pMemInfo
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SetConfiguration (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption, /**< [In] 0x00 or 0x01 or 0x02 */
								byte[] pData, /**< [In] max predictible is 24 bytes + 1 byte for
														 option 0x01. Minimum size is 1 byte for option 0x00
														 Unspecified size for option 0x02, if ATS is
														 set longer than 16 bytes. */
								byte bDataSize /**< [In] 8, 16 or 24 bytes */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetCardUID (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pUid /**< [Out] size = 4 bytes */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetFileIDs (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pFid, /**< [Out] size >= 33 bytes */
								ref byte bNumFIDs /**< [Out] Number of fidS read */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetISOFileIDs (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte[] pFidBuffer, /**< [Out] size = 64 bytes */
								ref byte bNumFIDs /**< [Out] Number of fidS read */
								);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetFileSettings (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bFileNo, /**< [In] file number */
								byte[] pFSBuffer, /**< [Out] size = 17 bytes*/
								ref byte bBufferLen /**< [Out] size of data put in pFSBuffer */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ChangeFileSettings (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption, /**< [In] bOption = ENCIPHERED means to be sent enciphered. Otherwise plain */
								byte bFileNo, /**< [In] file number */
								byte bCommSett, /**< [In] communication settings */
								byte[] pAccessRights /**< [In] 2 byte access rights */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateStdDataFile (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption,/**< option parameter. 0x00 means wISOFileId is not provided. 0x01 means wISOFileId is provided and is valid */
								byte bFileNo, /**< [In] file number */
								byte[] pISOFileId, /**< [In] ISO File ID */
								byte bCommSett, /**< [In] communication settings */
								byte[] pAccessRights, /**< [In] 2 byte access rights. Sent LSB first to PICC */
								byte[] pFileSize /**< [In] 3bytes. Sent LSB first to PICC */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateBackupDataFile (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption, /**< option parameter. 0x00 means wISOFileId is not provided. 0x01 means wISOFileId is provided and is valid */
								byte bFileNo, /**< [In] file number */
								byte[] pISOFileId, /**< [In] ISO File ID */
								byte bCommSett, /**< [In] communication settings */
								byte[] pAccessRights, /**< [In] 2 byte access rights. Sent LSB first to PICC */
								byte[] pFileSize /**< [In] 3bytes. Sent LSB first to PICC */
								);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateValueFile (
							   IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
							   byte bFileNo,            /**< [In] file number. */
							   byte bCommSett,          /**< [In] communication settings. */
							   byte[] bAccessRights,    /**< [In] 2 byte access rights. Sent LSB first to PICC. */
							   byte[] bLowerLmit,       /**< [In] 4 byte Lower limit value. Sent LSB first to PICC. */
							   byte[] bUpperLmit,       /**< [In] 4 byte Upper limit value. Sent LSB first to PICC. */
							   byte[] bValue,           /**< [In] 4 byte Value. Sent LSB first to PICC. */
							   byte bLimitedCredit      /**< [In] Limited Credit enabled or not. */
							   );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateLinearRecordFile (
										IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
										byte bOption,         /**< [In] Indicates ISO file ID is present or not. */
										byte bFileNo,         /**< [In] Linear record File No. */
										byte[] pIsoFileId,     /**< [In] 2 Byte IsoFileId. Sent LSB first to PICC. */
										byte bCommSett,        /**< [In] communication settings */
										byte[] pAccessRights,  /**< [In] 2 byte access rights. Sent LSB first to PICC */
										byte[] pRecordSize,    /**< [In] 3 byte Record Size. Sent LSB first to PICC */
										byte[] pMaxNoOfRec    /**< [In] 3 byte Max Number of Records. Sent LSB first to PICC */
										);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CreateCyclicRecordFile (
								IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								byte bOption,         /**< [In] Indicates ISO file ID is present or not. */
								byte bFileNo,         /**< [In] Linear record File No. */
								byte[] pIsoFileId,     /**< [In] 2 Byte IsoFileId. Sent LSB first to PICC. */
								byte bCommSett,        /**< [In] communication settings */
								byte[] pAccessRights,  /**< [In] 2 byte access rights. Sent LSB first to PICC */
								byte[] pRecordSize,    /**< [In] 2 byte Record Size. Sent LSB first to PICC */
								byte[] pMaxNoOfRec    /**< [In] 3 byte Max Number of Records. Sent LSB first to PICC */
								);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_DeleteFile (
								  IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
								  byte bFileNo /**< [In] 1 byte file number */
								  );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ReadData (
									IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
									byte bOption, /**< [In] Is either PH_EXCHANGE_DEFAULT or PH_EXCHANGE_RXCHAINING */
									byte bFileNo, /**< [In] 1 byte file number */
									byte[] pOffset, /**< [In] 3 bytes offset. LSB First */
									byte[] pLength, /**< [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
									ref IntPtr ppRxdata, /**< [Out] Pointer to HAL Rx buffer returned back to user. */
									ref ushort pRxdataLen /**< [Out] Length of Rxdata. LSB first */
									);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_WriteData (
									IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
									byte bOption, /**< [In] option provided to this function */
									byte bFileNo, /**< [In] 1 byte file number */
									byte[] pOffset, /**< [In] 3 bytes offset. LSB First */
									byte[] pTxData, /**< [in] data to be written. */
									byte[] pTxDataLen /**< [in] 3 bytes. length of data to be written. */
									);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetValue (
									IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
									byte bCommOption, /**< [In] Communication option plain, Macd or Enc. */
									byte bFileNo, /**< [In] 1 byte file number */
									byte[] pValue /**< [Out] 4 Byte array to store the value read out. LSB First */
									);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_Credit (
									  IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
									  byte bCommOption, /**< [In] Communication option plain, Macd or Enc. */
									  byte bFileNo,     /**< [In] 1 byte file number */
									  byte[] pValue     /**< [In] 4 byte value array. LSB first */
									  );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_Debit (
						 IntPtr pDataParams,/**< [In] Pointer to this layers param structure. */
						 byte bCommOption,  /**< [In] communication option. Plain, Mac'd or encrypted. */
						 byte bFileNo,      /**< [In] 1 byte file number. */
						 byte[] pValue      /**< [In] 4 byte value array. LSB first. */
						 );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_LimitedCredit (
						 IntPtr pDataParams,    /**< [In] Pointer to this layers param structure. */
						 byte bCommOption,      /**< [In] communication option. Plain, Mac'd or encrypted. */
						 byte bFileNo,          /**< [In] 1 byte file number. */
						 byte[] pValue          /**< [In] 4 byte value array. LSB first. */
						 );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_WriteRecord (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						byte bCommOption,   /**< [In] Communication option. Plain, Macd or Enc. */
						byte bFileNo, /**< [In] 1 byte file number */
						byte[] pOffset, /**< [In] 3 bytes offset. LSB First */
						byte[] pData, /**< [In] data to be written. */
						byte[] pDataLen /**< [In] 3 bytes. length of data to be written. */
						);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ReadRecords (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						byte bOption,   /**< [In] Default or chaining option. */
						byte bFileNo, /**< [In] 1 byte file number */
						byte[] pOffset, /**< [In] 3 bytes offset to the record. LSB First */
						byte[] pNumRec, /**< [In] 3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read. */
						byte[] pRecSize, /**< [In] 3 byte record size, LSB First. */
						ref IntPtr ppRxdata, /**< [Out] pointer to the HAL buffer that stores the read data. */
						ref ushort pRxdataLen /**< [Out] number of bytes read (= number of records read * size of record) */
						);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ClearRecordFile (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						byte bFileNo    /**< [In] 1 byte file number */
						);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_CommitTransaction (
						IntPtr pDataParams /**< [In] Pointer to this layers param structure */
						);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_AbortTransaction (
						IntPtr pDataParams /**< [In] Pointer to this layers param structure */
						);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoSelectFile (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						byte bOption, /**< [In] If bOption == 00 FCI is returned. If 0x0C no FCI returned */
						byte bSelector,/**< [In] bSelector = 0x00 => Selection by 2 byte file Id.
                                              bSelector = 0x02 => Select EF under current DF. wFid = EF id
                                               bSelector = 0x04 => Selection by DF Name. DFName and len is then valid
                                                 */
						byte[] pFid, /**< [In] two byte file id */
						byte[] pDFname, /**< [In] DFName upto 16 bytes. valid only when bOption = 0x04 */
						byte DFnameLen, /**< [In] Length of DFName string provided by the user */
						ref IntPtr ppFCI, /**< [Out] File control information. */
						ref ushort pwFCILen /**< [Out] Length of FCI returned. */
						);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoReadBinary (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						ushort wOption, /**< [In] #PH_EXCHANGE_DEFAULT or #PH_EXCHANGE_RXCHAINING. */
						byte bOffset, /**< [In] Offset from where to read */
						byte bSfid, /**< [In] Short ISO File Id bits 0..4 only code this value */
						byte bBytesToRead, /**< [In] number of bytes to read. If 0, then entire file to be read */
						ref IntPtr pRxBuffer, /**< [Out] buffer where the read bytes will be stored */
						ref ushort bBytesRead /**< [Out] number of bytes read */
						);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoUpdateBinary (
					   IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
					   byte bOffset, /**< [In] Offset from where to write */
					   byte bSfid, /**< [In] Short ISO File Id bits 0..4 only code this value */
					   byte[] pData, /**< [In] data to be written */
					   byte bDataLen /**< [In] number of bytes to write */
					   );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoReadRecords (
						IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
						ushort wOption, /**< [In] #PH_EXCHANGE_DEFAULT or #PH_EXCHANGE_RXCHAINING. */
						byte bRecNo, /**< [In] Record to read / from where to read. */
						byte bReadAllFromP1, /** [In] Whether to read all records from P1 or just one */
						byte bSfid, /**< [In] Short ISO File Id bits 0..4 only code this value */
						byte bBytesToRead, /**< [In] number of bytes to read. Multiple of record size */
						ref IntPtr pRxBuffer, /**< [Out] buffer where the read bytes will be stored */
						ref ushort bBytesRead /**< [Out] number of bytes read */
						);

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoAppendRecord (
							IntPtr pDataParams,  /**< [In] Pointer to this layers param structure */
							byte bSfid,          /**< [In] Short Iso File Id bits 0..4 only code this value */
							byte[] pData,        /**< [In] data to write */
							byte bDataLen        /**< [In] number of bytes to write */
							);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoGetChallenge (
							IntPtr pDataParams, /**< [In] Pointer to this layers param structure */
							ushort wKeyNo,    /**< [In] Key number in key store */
							ushort wKeyVer,   /**< [In] Key version in key store */
							byte bLe, /**< [In] Length of expected challenge RPICC1 */
							byte[] pRPICC1 /**< [Out] RPICC1 returned from PICC */
							);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoExternalAuthenticate (
							IntPtr pDataParams, /**< [In] Pointer to this layers param structure. */
							byte[] pInput,   /**< [In] Input data. */
							byte bInputLen,  /**< [In] Length of pInput. */
							byte[] pDataOut, /**< [Out] Returns Rnd number PCD2 in sam non x mode. Nothing in S/W mode. */
							ref byte pOutLen   /**< [Out] Length of data returned in pDataOut. */
							);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoInternalAuthenticate (
											IntPtr pDataParams,     /**< [In] Pointer to this layers param structure */
											byte[] pInput,   /**< [In] Input data. */
											byte bInputLen,  /**< [In] Length of pInput. */
											byte[] pDataOut, /**< [Out] RRPICC2||RPCD2 after decryption in S/W mode. Nothing in Sam non x mode. */
											ref byte pOutLen   /**< [Out] Length of data returned in pDataOut. */
											);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_IsoAuthenticate (
									IntPtr pDataParams,     /**< [In] Pointer to this layers param structure */
									ushort wKeyNo,         /**< [In] DESFire key number or SAM Key entry number */
									ushort wKeyVer,        /**< [In] Key version */
									byte bKeyNoCard,      /**< [In] Key number on card. 0x0 to 0xD */
									byte isPICCkey          /**< [In] Is it PICC Master key? 1=YES. */
									);
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_GetConfig (
									  IntPtr pDataParams,  /**< [In] Pointer to this layers parameter structure. */
									  ushort wConfig,    /**< [In] Item to read. */
									  ref ushort pValue    /**< [In] Read value. */
									  );
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SetConfig (
									  IntPtr pDataParams,  /**< [In] Pointer to this layers parameter structure. */
									  ushort wConfig,    /**< [In] Item to read. */
									  ushort wValue      /**< [In] Read value. */
									  );

		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_ResetAuthStatus (
									  IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
									  );
		#endregion

		#region Wrapper Functions
		#region Security Commands
		public Status_t Authenticate ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
		{
			return phalMfdf_Authenticate ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
		}

		public Status_t AuthenticateISO ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
		{
			return phalMfdf_AuthenticateISO ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
		}

		public Status_t AuthenticateAES ( ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput )
		{
			return phalMfdf_AuthenticateAES ( m_pDataParams, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
		}

		public Status_t ChangeKeySettings ( byte bKeySettings )
		{
			return phalMfdf_ChangeKeySettings ( m_pDataParams, bKeySettings );
		}

		public Status_t GetKeySettings ( out byte[] pResponse )
		{
			pResponse = new byte[2];
			return phalMfdf_GetKeySettings ( m_pDataParams, pResponse );
		}

		public Status_t ChangeKey ( ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo, ushort bNewKeyVer, byte bKeyNoCard, byte[] pDivInput )
		{
			return phalMfdf_ChangeKey ( m_pDataParams, wOption, wKeyNo, wKeyVer, wNewKeyNo, bNewKeyVer, bKeyNoCard, pDivInput,
				( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
		}

		public Status_t GetKeyVersion ( byte bKeyNo, out byte[] pResponse )
		{
			pResponse = new byte[1];
			return phalMfdf_GetKeyVersion ( m_pDataParams, bKeyNo, pResponse );
		}
		#endregion Security Commands

		#region PICC Level commands
		public Status_t CreateApplication ( byte bOption, byte[] pAid, byte bKeySettings1, byte bKeySettings2, byte[] pISOFileId, byte[] pISODFName )
		{
			return phalMfdf_CreateApplication (
								m_pDataParams,
								bOption,
								 pAid,
								 bKeySettings1,
								 bKeySettings2,
								 pISOFileId,
								 pISODFName,
								 ( byte ) ( ( pISODFName == null ) ? 0 : pISODFName.Length ) );
		}

		public Status_t DeleteApplication ( byte[] pAid )
		{
			return phalMfdf_DeleteApplication (
								m_pDataParams,
								 pAid
								);
		}

		public Status_t GetApplicationIDs ( out byte[] pAidBuff )
		{
			Status_t status;
			pAidBuff = new byte[96];
			byte NumAIDs = 0;

			status = phalMfdf_GetApplicationIDs (
								m_pDataParams,
								 pAidBuff,
								 ref NumAIDs
								);
			Array.Resize<byte> ( ref pAidBuff, 3 * NumAIDs );

			return status;

		}

		public Status_t GetDFNames ( byte bOption, out byte[] pDFBuffer )
		{
			Status_t status;
			pDFBuffer = new byte[21];
			byte bSize = 0;
			status = phalMfdf_GetDFNames (
								m_pDataParams,
								bOption,
								pDFBuffer,
								ref bSize
								);
			Array.Resize<byte> ( ref pDFBuffer, bSize );
			return status;
		}

		public Status_t SelectApplication ( byte[] pAppId )
		{
			return phalMfdf_SelectApplication (
								m_pDataParams,
								 pAppId
								);
		}

		public Status_t FormatPICC ()
		{
			return phalMfdf_FormatPICC (
								m_pDataParams
								);
		}

		public Status_t GetVersion ( out byte[] pVerInfo )
		{
			pVerInfo = new byte[28];
			return phalMfdf_GetVersion (
								m_pDataParams,
								 pVerInfo
								);
		}

		public Status_t FreeMem ( out byte[] pMemInfo )
		{
			pMemInfo = new byte[3];
			return phalMfdf_FreeMem (
								m_pDataParams,
								pMemInfo
								);
		}

		public Status_t SetConfiguration ( byte bOption, byte[] pData )
		{
			return phalMfdf_SetConfiguration (
								m_pDataParams,
								 bOption,
								 pData,
								 ( byte ) pData.Length
								);
		}

		public Status_t GetCardUID ( out byte[] pUid )
		{
			pUid = new byte[7];

			return phalMfdf_GetCardUID (
								m_pDataParams,
								 pUid
								);
		}
		#endregion PICC Level commands

		#region Application Level commands
		public Status_t GetFileIDs ( out byte[] pFid )
		{
			byte bNumFIDs = 0;
			Status_t status;
			pFid = new byte[32];

			status = phalMfdf_GetFileIDs (
								m_pDataParams,
								 pFid,
								 ref bNumFIDs
								);
			Array.Resize<byte> ( ref pFid, bNumFIDs );
			return status;
		}

		public Status_t GetISOFileIDs ( out byte[] pFidBuffer )
		{
			Status_t status;
			byte bNumFIDs = 0;

			pFidBuffer = new byte[64];

			status = phalMfdf_GetISOFileIDs (
								m_pDataParams,
								 pFidBuffer,
								 ref bNumFIDs
								);

			Array.Resize<byte> ( ref pFidBuffer, bNumFIDs * 2 );
			return status;
		}

		public Status_t GetFileSettings ( byte bFileNo, out byte[] pFSBuffer )
		{
			Status_t status;
			byte bBufferLen = 0;

			pFSBuffer = new byte[17];

			status = phalMfdf_GetFileSettings (
								m_pDataParams,
								 bFileNo,
								 pFSBuffer,
								 ref bBufferLen
								);
			Array.Resize<byte> ( ref pFSBuffer, bBufferLen );
			return status;
		}

		public Status_t ChangeFileSettings ( byte bOption, byte bFileNo, byte bCommSett, byte[] pAccessRights )
		{
			return phalMfdf_ChangeFileSettings (
								m_pDataParams,
								 bOption,
								 bFileNo,
								 bCommSett,
								 pAccessRights
								);
		}

		public Status_t CreateStdDataFile ( byte bOption, byte bFileNo, byte[] pISOFileId, byte bCommSett, byte[] pAccessRights, byte[] pFileSize )
		{
			return phalMfdf_CreateStdDataFile (
								m_pDataParams,
								 bOption,
								 bFileNo,
								 pISOFileId,
								 bCommSett,
								 pAccessRights,
								 pFileSize
								);
		}

		public Status_t CreateBackupDataFile ( byte bOption, byte bFileNo, byte[] pISOFileId, byte bCommSett, byte[] pAccessRights, byte[] pFileSize )
		{
			return phalMfdf_CreateBackupDataFile (
								m_pDataParams,
								 bOption,
								 bFileNo,
								 pISOFileId,
								 bCommSett,
								 pAccessRights,
								pFileSize
					);
		}

		public Status_t CreateValueFile ( byte bFileNo, byte bCommSett, byte[] bAccessRights, byte[] bLowerLmit, byte[] bUpperLmit, byte[] bValue, byte bLimitedCredit )
		{
			return phalMfdf_CreateValueFile (
								 m_pDataParams,
								 bFileNo,
								 bCommSett,
					 bAccessRights,
					 bLowerLmit,
					 bUpperLmit,
					 bValue,
					 bLimitedCredit
								 );
		}

		public Status_t CreateLinearRecordFile ( byte bOption, byte bFileNo, byte[] pIsoFileId, byte bCommSett, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec )
		{
			return phalMfdf_CreateLinearRecordFile (
										 m_pDataParams,
										 bOption,
										 bFileNo,
										 pIsoFileId,
										 bCommSett,
										 pAccessRights,
										 pRecordSize,
										 pMaxNoOfRec
										);
		}

		public Status_t CreateCyclicRecordFile ( byte bOption, byte bFileNo, byte[] pIsoFileId, byte bCommSett, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec )
		{
			return phalMfdf_CreateCyclicRecordFile (
								 m_pDataParams,
								 bOption,
								 bFileNo,
								 pIsoFileId,
								 bCommSett,
								 pAccessRights,
								 pRecordSize,
								 pMaxNoOfRec
								);
		}

		public Status_t DeleteFile ( byte bFileNo )
		{
			return phalMfdf_DeleteFile (
								  m_pDataParams,
								  bFileNo
								  );
		}
		#endregion Application Level commands

		#region Data Manipulation commands
		public Status_t ReadData ( byte wOption, byte bFileNo, byte[] pOffset, byte[] pLength, out byte[] pRxdata )
		{
			Status_t status;
			IntPtr ppRxdata = IntPtr.Zero;
			byte[] pRxdataLen;
			ushort rxDataLen = 0;

			pRxdataLen = new byte[3];

			status = phalMfdf_ReadData (
									m_pDataParams,
									 wOption,
									 bFileNo,
									 pOffset,
									 pLength,
									 ref ppRxdata,
									 ref rxDataLen
									);
			if ( rxDataLen != 0 )
			{
				pRxdata = new byte[rxDataLen];
				Marshal.Copy ( ppRxdata, pRxdata, 0, rxDataLen );
			}
			else
			{
				pRxdata = null;
			}
			return status;
		}

		public Status_t WriteData ( byte wOption, byte bFileNo, byte[] pOffset, byte[] pTxData, byte[] pTxDataLen )
		{
			return phalMfdf_WriteData (
									m_pDataParams,
									 wOption,
									 bFileNo,
									 pOffset,
									 pTxData,
									 pTxDataLen
									);
		}

		public Status_t GetValue ( byte bCommOption, byte bFileNo, out byte[] pValue )
		{
			pValue = new byte[4];
			return phalMfdf_GetValue (
									m_pDataParams,
									 bCommOption,
									 bFileNo,
									 pValue
									);
		}

		public Status_t Credit ( byte bCommOption, byte bFileNo, byte[] pValue )
		{
			return phalMfdf_Credit (
									  m_pDataParams,
									   bCommOption,
									   bFileNo,
									   pValue
									  );
		}

		public Status_t Debit ( byte bCommOption, byte bFileNo, byte[] pValue )
		{
			return phalMfdf_Debit (
						 m_pDataParams,
						  bCommOption,
						  bFileNo,
						  pValue
						 );
		}

		public Status_t LimitedCredit ( byte bCommOption, byte bFileNo, byte[] pValue )
		{
			return phalMfdf_LimitedCredit (
						 m_pDataParams,
						  bCommOption,
						  bFileNo,
						  pValue
						 );
		}

		public Status_t WriteRecord ( byte bCommOption, byte bFileNo, byte[] pOffset, byte[] pData, byte[] pDataLen )
		{
			return phalMfdf_WriteRecord (
						m_pDataParams,
						 bCommOption,
						 bFileNo,
						 pOffset,
						 pData,
						 pDataLen
						);
		}

		public Status_t ReadRecords ( byte bCommOption, byte bFileNo, byte[] pOffset, byte[] pNumRec, byte[] pRecSize, out byte[] pRxdata )
		{
			Status_t status;
			IntPtr ppRxdata = IntPtr.Zero;
			ushort pRxDataLen = 0;

			status = phalMfdf_ReadRecords (
						m_pDataParams,
						 bCommOption,
						 bFileNo,
						 pOffset,
						 pNumRec,
						 pRecSize,
						 ref ppRxdata,
						 ref pRxDataLen
						);

			if ( pRxDataLen != 0 )
			{
				pRxdata = new byte[pRxDataLen];
				Marshal.Copy ( ppRxdata, pRxdata, 0, pRxDataLen );
			}
			else
			{
				pRxdata = null;
			}
			return status;

		}

		public Status_t ClearRecordFile ( byte bFileNo )
		{
			return phalMfdf_ClearRecordFile (
						m_pDataParams,
						 bFileNo
						);
		}
		public Status_t CommitTransaction ()
		{
			return phalMfdf_CommitTransaction (
						m_pDataParams
						);
		}
		public Status_t AbortTransaction ()
		{

			return phalMfdf_AbortTransaction (
						m_pDataParams
						);
		}
		#endregion Data Manipulation commands

		#region ISO7816 commands
		public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, out byte[] pFCI )
		{
			Status_t status;
			IntPtr ppRxBuffer = IntPtr.Zero;
			ushort wBytesRead = 0;
			byte bDfNameLen = 0x00;
			if ( pDFname != null )
			{
				bDfNameLen = ( byte ) pDFname.Length;
			}

			status = phalMfdf_IsoSelectFile (
						m_pDataParams,
						bOption,
						 bSelector,
						 pFid,
						 pDFname,
						 bDfNameLen,
						 ref ppRxBuffer,
						 ref wBytesRead
						);
			if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
			{
				pFCI = new byte[wBytesRead];
				Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
			}
			else
			{
				pFCI = null;
			}
			return status;
		}

		public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, byte bBytesToRead, out byte[] pRxBuffer )
		{
			Status_t status;
			IntPtr ppRxBuffer = IntPtr.Zero;
			ushort bBytesRead = 0;

			status = phalMfdf_IsoReadBinary (
						m_pDataParams,
						 wOption,
						 bOffset,
						 bSfid,
						 bBytesToRead,
						 ref ppRxBuffer,
						 ref bBytesRead
						);
			if ( ( ppRxBuffer != IntPtr.Zero ) && ( bBytesRead != 0 ) )
			{
				pRxBuffer = new byte[bBytesRead];
				Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, bBytesRead );
			}
			else
			{
				pRxBuffer = null;
			}
			return status;

		}

		public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] pData )
		{

			return phalMfdf_IsoUpdateBinary (
					   m_pDataParams,
						bOffset,
						bSfid,
						pData,
						( byte ) pData.Length
					   );
		}

		public Status_t IsoReadRecords ( ushort wOption, byte bRecNo, byte bReadAllFromP1, byte bSfid, byte bBytesToRead, out byte[] pRxBuffer )
		{
			Status_t status;
			IntPtr ppRxBuffer = IntPtr.Zero;
			ushort bBytesRead = 0;

			status = phalMfdf_IsoReadRecords (
						m_pDataParams,
						wOption,
						 bRecNo,
						 bReadAllFromP1,
						 bSfid,
						 bBytesToRead,
						 ref ppRxBuffer,
						 ref bBytesRead
						);
			if ( ( ppRxBuffer != IntPtr.Zero ) && ( bBytesRead != 0 ) )
			{
				pRxBuffer = new byte[bBytesRead];
				Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, bBytesRead );
			}
			else
			{
				pRxBuffer = null;
			}
			return status;
		}

		public Status_t IsoAppendRecord ( byte bSfid, byte[] pData )
		{
			return phalMfdf_IsoAppendRecord (
							m_pDataParams,
							 bSfid,
							 pData,
							 ( byte ) pData.Length
							);
		}

		public Status_t IsoGetChallenge ( ushort wKeyNo, ushort wKeyVer, byte bLe, out byte[] pRPICC1 )
		{
			pRPICC1 = new byte[16];
			return phalMfdf_IsoGetChallenge (
							m_pDataParams,
							wKeyNo,
							wKeyVer,
							 bLe,
							 pRPICC1
							);

		}

		public Status_t IsoExternalAuthenticate ( byte[] pInput, byte bInputLen, out byte[] pDataOut )
		{
			byte OutLen = 0;
			Status_t status;
			pDataOut = new byte[255];
			status = phalMfdf_IsoExternalAuthenticate (
													m_pDataParams,
													pInput,
													bInputLen,
													pDataOut,
													ref OutLen
													);
			Array.Resize<byte> ( ref pDataOut, OutLen );
			return status;
		}

		public Status_t IsoInternalAuthenticate ( byte[] pInput, byte bInputLen, out byte[] pDataOut )
		{
			byte OutLen = 0;
			Status_t status;
			pDataOut = new byte[255];
			status = phalMfdf_IsoInternalAuthenticate (
													m_pDataParams,
													pInput,
													bInputLen,
													pDataOut,
													ref OutLen
													);
			Array.Resize<byte> ( ref pDataOut, OutLen );
			return status;
		}

		public Status_t IsoAuthenticate ( ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte isPICCkey )
		{
			return phalMfdf_IsoAuthenticate (
									m_pDataParams,
									 wKeyNo,
									 wKeyVer,
									 bKeyNoCard,
									isPICCkey
									);
		}
		#endregion ISO7816 commands

		#region Miscellaneous commands
		public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
		{
			return phalMfdf_GetConfig (
									  m_pDataParams,
									  wConfig,
									  ref pValue
									  );
		}

		public Status_t SetConfig ( ushort wConfig, ushort wValue )
		{
			return phalMfdf_SetConfig (
									  m_pDataParams,
									  wConfig,
									  wValue
									  );
		}

		public Status_t ResetAuthStatus ()
		{
			return phalMfdf_ResetAuthStatus ( m_pDataParams );
		}
		#endregion Miscellaneous commands
		#endregion

		#region Memory Maping

		protected GCHandle m_pDataParamsInt;

		/// <summary>
		/// Retrieve private data storage of underlying C Object.
		/// </summary>
		public IntPtr m_pDataParams
		{
			get
			{
				return this.m_pDataParamsInt.AddrOfPinnedObject ();
			}
		}

		#endregion
	}
	#endregion Generic

	#region Software
	/// <summary>
	/// Class for software layer initialization interface and data params.
	/// </summary>
	[ClassInterface ( ClassInterfaceType.AutoDual )]
	public class Sw : alMfdf.Generic
	{
		#region Constants
		private const byte AID_LENGTH = 3;
		private const byte IV_LENGTH = 16;
		private const byte SESSION_KEY_LENGTH = 24;
		#endregion

		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE DESfire Software layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare component. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
			public IntPtr pKeyStoreDataParams;

			/// <summary> Pointer to the parameter structure of the Crypto layer for encryption. </summary>
			public IntPtr pCryptoDataParamsEnc;

			/// <summary> Pointer to the parameter structure of the CryptoRng layer. </summary>
			public IntPtr pCryptoRngDataParams;

			/// <summary> Pointer to the HAL parameters structure. </summary>
			public IntPtr pHalDataParams;

			/// <summary> Session key for this authentication. </summary>
			public fixed byte bSessionKey[SESSION_KEY_LENGTH];

			/// <summary> Key number against which this authentication is done. </summary>
			public byte bKeyNo;

			/// <summary> Max size of IV can be 16 bytes. </summary>
			public fixed byte bIv[IV_LENGTH];

			/// <summary> Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA). </summary>
			public byte bAuthMode;

			/// <summary> Aid of the currently selected application. </summary>
			public fixed byte pAid[AID_LENGTH];

			/// <summary> DES,3DES, 3K3DES or AES. </summary>
			public byte bCryptoMethod;

			/// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. </summary>
			public byte bWrappedMode;

			/// <summary> 2 Byte CRC initial value in Authenticate mode. </summary>
			public ushort wCrc;

			/// <summary> 4 Byte CRC initial value in 0x1A, 0xAA mode. </summary>
			public uint dwCrc;

			/// <summary> Specific error codes for Desfire generic errors. </summary>
			public ushort wAdditionalInfo;

			/// <summary> When reading unspecified length of data. </summary>
			public ushort wPayLoadLen;

			/// <summary> Buffer to store last Block of encrypted data in case of chaining. </summary>
			public fixed byte bLastBlockBuffer[16];

			/// <summary> Last Block Buffer Index. </summary>
			public byte bLastBlockIndex;
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams, IntPtr pKeyStoreDataParams,
			IntPtr pCryptoDataParamsEnc, IntPtr pCryptoRngDataParams, IntPtr pHalDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for MIFARE DESFire software component.
		/// </summary>
		///
		/// <param name="pPalMifare">Pointer to a palMifare component context</param>
		/// <param name="pKeyStore">Pointer to a KeyStore component context</param>
		/// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption</param>
		/// <param name="pCryptoMac">Pointer to a Crypto component context for Macing</param>
		/// <param name="pCryptoRng">Pointer to a CryptoRng component context</param>
		/// <param name="pHalparams">Pointer to Hal parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoRng.Generic pCryptoRng, Hal.Generic pHalparams )
		{
			return phalMfdf_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
				( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams, ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
				( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
		}

#if DEBUG
		/// <summary>
		/// Initialization API for MIFARE DESFire software component.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pPalMifare">Pointer to a palMifare component context</param>
		/// <param name="pKeyStore">Pointer to a KeyStore component context</param>
		/// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption</param>
		/// <param name="pCryptoMac">Pointer to a Crypto component context for Macing</param>
		/// <param name="pCryptoRng">Pointer to a CryptoRng component context</param>
		/// <param name="pHalparams">Pointer to Hal parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoRng.Generic pCryptoRng,
			Hal.Generic pHalparams )
		{
			return phalMfdf_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
				( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams, ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
				( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
		}

#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public Sw ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~Sw ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}
		#endregion

		#region Parameter Access
		public byte bAuthMode
		{
			get { return m_DataParamsInt[0].bAuthMode; }
			set { m_DataParamsInt[0].bAuthMode = value; }
		}
		public byte[] pAid
		{
			get
			{
				byte[] bValue = new byte[AID_LENGTH];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < AID_LENGTH; i++ )
						{
							bValue[i] = pDataParams->pAid[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > 3 )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->pAid[i] = value[i];
						}
					}
				}
			}
		}
		public byte[] bIv
		{
			get
			{
				byte[] bValue = new byte[IV_LENGTH];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < IV_LENGTH; i++ )
						{
							bValue[i] = pDataParams->bIv[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > IV_LENGTH )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->bIv[i] = value[i];
						}
					}
				}
			}
		}
		public byte[] bSessionKey
		{
			get
			{
				byte[] bValue = new byte[SESSION_KEY_LENGTH];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
						{
							bValue[i] = pDataParams->bSessionKey[i];
						}
					}
				}
				return bValue;
			}
		}
		public byte bWrappedMode
		{
			get { return m_DataParamsInt[0].bWrappedMode; }
			set { m_DataParamsInt[0].bWrappedMode = value; }
		}
		#endregion
	}
	#endregion Software

	#region Sam_NonX
	#region SamAV2
	/// <summary>
	/// Class for Sam AV2 NonX Mode layer initialization interface and data params.
	/// </summary>
	public class SamAV2 : alMfdf.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE DESFire SamAV2 NonX layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Pointer to the parameter structure of the hal of the reader. </summary>
			public IntPtr pHalDataParams;

			/// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
			public byte bAuthMode;

			/// <summary> Key number against which authenticated. </summary>
			public byte bKeyNo;

			/// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. </summary>
			public byte bWrappedMode;

			/// <summary> Currently selected application Id. </summary>
			public fixed byte pAid[3];

			/// <summary> Specific error codes for Desfire generic errors. </summary>
			public ushort wAdditionalInfo;
		};
		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SamAV2_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
			IntPtr pHalDataParams, IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV2 in NonX Mode.
		/// </summary>
		///
		/// <param name="pHalSam">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="pHal">Pointer to HAL param structure of contactless reader.</param>
		/// <param name="pPalMifare">Pointer to the parameter structure of the underlying palMIFARE layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV2 pHalSam, Hal.Generic pHal, palMifare.Generic pPalMifare )
		{
			ushort wStatus = phalMfdf_SamAV2_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
			return wStatus;
		}

#if DEBUG
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV2 in NonX Mode.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHalSam">Pointer to Hal SamAV2 parameter structure.</param>
		/// <param name="pHal">Pointer to HAL param structure of contactless reader.</param>
		/// <param name="pPalMifare">Pointer to the parameter structure of the underlying palMIFARE layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( ushort wDataParamSize, Hal.SamAV2 pHalSam, Hal.Generic pHal, palMifare.Generic pPalMifare )
		{
			return phalMfdf_SamAV2_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams,
				( pHal == null ) ? IntPtr.Zero : pHal.m_pDataParams, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV2 ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV2 ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}

		#endregion

		#region Parameter Access
		public NxpRdLibNet.Hal.Generic HalSamDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
				return ( Handle.Target as NxpRdLibNet.Hal.SamAV2 );
			}
		}

		public NxpRdLibNet.palMifare.Generic PalMifareDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pPalMifareDataParams;
				return ( Handle.Target as NxpRdLibNet.palMifare.Generic );
			}
		}

		public ushort ID
		{
			get { return m_DataParamsInt[0].wId; }
			set { m_DataParamsInt[0].wId = value; }
		}

		public byte AuthMode
		{
			get { return m_DataParamsInt[0].bAuthMode; }
			set { m_DataParamsInt[0].bAuthMode = value; }
		}

		public byte KeyNo
		{
			get { return m_DataParamsInt[0].bKeyNo; }
			set { m_DataParamsInt[0].bKeyNo = value; }
		}

		public byte WrappedMode
		{
			get { return m_DataParamsInt[0].bWrappedMode; }
			set { m_DataParamsInt[0].bWrappedMode = value; }
		}

		public ushort AdditionalInfo
		{
			get { return m_DataParamsInt[0].wAdditionalInfo; }
			set { m_DataParamsInt[0].wAdditionalInfo = value; }
		}

		public byte[] Aid
		{
			get
			{
				byte[] bValue = new byte[3];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < 3; i++ )
						{
							bValue[i] = pDataParams->pAid[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > 3 )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->pAid[i] = value[i];
						}
					}
				}
			}
		}
		#endregion
	}
	#endregion

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
	#region SamAV3
	/// <summary>
	/// Class for Sam AV3 NonX Mode layer initialization interface and data params.
	/// </summary>
	public class SamAV3_NonX : alMfdf.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE DESFire SamAV3 NonX layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this HAL component, NEVER MODIFY!. </summary>
			public ushort wId;

			/// <summary> Pointer to the parameter structure of the palMifare layer. </summary>
			public IntPtr pPalMifareDataParams;

			/// <summary> Pointer to the parameter structure of the hal of the SAM layer. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
			public byte bAuthMode;

			/// <summary> Key number against which authenticated. </summary>
			public byte bKeyNo;

			/// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. </summary>
			public byte bWrappedMode;

			/// <summary> Currently selected application Id. </summary>
			public fixed byte pAid[3];

			/// <summary> Specific error codes for Desfire generic errors. </summary>
			public ushort wAdditionalInfo;
		};
		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
			IntPtr pPalMifareDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV2 in NonX Mode.
		/// </summary>
		///
		/// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="pPalMifare">Pointer to the parameter structure of the underlying palMIFARE layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare )
		{
			ushort wStatus = phalMfdf_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
			return wStatus;
		}

#if DEBUG
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV3 in NonX Mode.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
		/// <param name="pPalMifare">Pointer to the parameter structure of the underlying palMIFARE layer.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam, palMifare.Generic pPalMifare )
		{
			return phalMfdf_SamAV3_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams,
				( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams );
		}
#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV3_NonX ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV3_NonX ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}
		#endregion#if DEBUG

		#region Parameter Access
		public NxpRdLibNet.Hal.Generic HalSamDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
				return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
			}
		}

		public NxpRdLibNet.palMifare.Generic PalMifareDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pPalMifareDataParams;
				return ( Handle.Target as NxpRdLibNet.palMifare.Generic );
			}
		}

		public ushort ID
		{
			get { return m_DataParamsInt[0].wId; }
			set { m_DataParamsInt[0].wId = value; }
		}

		public byte AuthMode
		{
			get { return m_DataParamsInt[0].bAuthMode; }
			set { m_DataParamsInt[0].bAuthMode = value; }
		}

		public byte KeyNo
		{
			get { return m_DataParamsInt[0].bKeyNo; }
			set { m_DataParamsInt[0].bKeyNo = value; }
		}

		public byte WrappedMode
		{
			get { return m_DataParamsInt[0].bWrappedMode; }
			set { m_DataParamsInt[0].bWrappedMode = value; }
		}

		public ushort AdditionalInfo
		{
			get { return m_DataParamsInt[0].wAdditionalInfo; }
			set { m_DataParamsInt[0].wAdditionalInfo = value; }
		}

		public byte[] Aid
		{
			get
			{
				byte[] bValue = new byte[3];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < 3; i++ )
						{
							bValue[i] = pDataParams->pAid[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > 3 )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->pAid[i] = value[i];
						}
					}
				}
			}
		}
		#endregion
	}
	#endregion
#endif
	#endregion Sam_NonX

	#region Sam_X
	#region SamAV2
	/// <summary>
	/// Class for Sam AV2 X Mode layer initialization interface and data params.
	/// </summary>
	public class SamAV2_X : alMfdf.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE DESFire SamAV3 X layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this component. </summary>
			public ushort wId;

			/// <summary> Pointer to SamAV3 parameter structure. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
			public byte bAuthMode;

			/// <summary> Key number against which authenticated. </summary>
			public byte bKeyNo;

			/// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. </summary>
			public byte bWrappedMode;

			/// <summary> Currently selected application Id. </summary>
			public fixed byte pAid[3];

			/// <summary> Specific error codes for Desfire generic errors. </summary>
			public ushort wAdditionalInfo;
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SamAV2_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV3 in X Mode.
		/// </summary>
		///
		/// <param name="pHalSam">Pointer to Hal SamAV2 parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV2 pHalSam )
		{
			ushort wStatus = phalMfdf_SamAV2_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
			return wStatus;
		}

#if DEBUG
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV3 in X Mode.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHalSam">Pointer to Hal SamAV2 parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( ushort wDataParamSize, Hal.SamAV2 pHalSam )
		{
			return phalMfdf_SamAV2_X_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
		}
#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV2_X ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV2_X ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}
		#endregion

		#region Parameter Access
		public NxpRdLibNet.Hal.Generic HalSamDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
				return ( Handle.Target as NxpRdLibNet.Hal.SamAV2 );
			}
		}

		public ushort ID
		{
			get { return m_DataParamsInt[0].wId; }
			set { m_DataParamsInt[0].wId = value; }
		}

		public byte AuthMode
		{
			get { return m_DataParamsInt[0].bAuthMode; }
			set { m_DataParamsInt[0].bAuthMode = value; }
		}

		public byte KeyNo
		{
			get { return m_DataParamsInt[0].bKeyNo; }
			set { m_DataParamsInt[0].bKeyNo = value; }
		}

		public byte WrappedMode
		{
			get { return m_DataParamsInt[0].bWrappedMode; }
			set { m_DataParamsInt[0].bWrappedMode = value; }
		}

		public ushort AdditionalInfo
		{
			get { return m_DataParamsInt[0].wAdditionalInfo; }
			set { m_DataParamsInt[0].wAdditionalInfo = value; }
		}

		public byte[] Aid
		{
			get
			{
				byte[] bValue = new byte[3];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < 3; i++ )
						{
							bValue[i] = pDataParams->pAid[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > 3 )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->pAid[i] = value[i];
						}
					}
				}
			}
		}
		#endregion
	}
	#endregion SamAV2	`

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
	#region SamAV3
	/// <summary>
	/// Class for Sam AV3 X Mode layer initialization interface and data params.
	/// </summary>
	public class SamAV3_X : alMfdf.Generic
	{
		#region Data Structure
		/// <summary>
		/// Data structure for MIFARE DESFire SamAV3 X layer implementation.
		/// </summary>
		[StructLayout ( LayoutKind.Sequential, Pack = 1 )]
		public unsafe struct DataParams_t
		{
			/// <summary> Layer ID for this component. </summary>
			public ushort wId;

			/// <summary> Pointer to SamAV3 parameter structure. </summary>
			public IntPtr pHalSamDataParams;

			/// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
			public byte bAuthMode;

			/// <summary> Key number against which authenticated. </summary>
			public byte bKeyNo;

			/// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus. </summary>
			public byte bWrappedMode;

			/// <summary> Currently selected application Id. </summary>
			public fixed byte pAid[3];

			/// <summary> Specific error codes for Desfire generic errors. </summary>
			public ushort wAdditionalInfo;
		};

		#endregion

		#region DLL Imports
		[DllImport ( Common.IMPORT_LIBRARY_NAME )]
		private static extern ushort phalMfdf_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams );
		#endregion

		#region Initialization
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV3 in X Mode.
		/// </summary>
		///
		/// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( Hal.SamAV3 pHalSam )
		{
			ushort wStatus = phalMfdf_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
				( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
			return wStatus;
		}

#if DEBUG
		/// <summary>
		/// Initialization API for AL MIFARE DESFire to communicate with SamAV3 in X Mode.
		/// </summary>
		///
		/// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
		/// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
		///
		/// <returns>Returns Success status for successfull operation.
		///			 Other Depending on implementation and underlaying component.</returns>
		public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam )
		{
			return phalMfdf_SamAV3_X_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams );
		}
#endif

		#endregion

		#region Memory Maping
		private DataParams_t[] m_DataParamsInt;

		/// <summary>
		/// Constructor
		/// </summary>
		public SamAV3_X ()
		{
			// Allocate internal data parameters and pointer to them
			this.m_DataParamsInt = new DataParams_t[1];
			this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
		}

		/// <summary>
		/// Destructor
		/// </summary>
		~SamAV3_X ()
		{
			// Free allocated pointer to data params
			if ( this.m_pDataParamsInt.IsAllocated )
			{
				this.m_pDataParamsInt.Free ();
			}
		}

		/// <summary>
		/// Setter & Getter for DataParams structure
		/// </summary>
		public DataParams_t DataParams
		{
			set
			{
				this.m_DataParamsInt[0] = value;
			}
			get
			{
				return this.m_DataParamsInt[0];
			}
		}
		#endregion

		#region Parameter Access
		public NxpRdLibNet.Hal.Generic HalSamDataParams
		{
			get
			{
				GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
				return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
			}
		}

		public ushort ID
		{
			get { return m_DataParamsInt[0].wId; }
			set { m_DataParamsInt[0].wId = value; }
		}

		public byte AuthMode
		{
			get { return m_DataParamsInt[0].bAuthMode; }
			set { m_DataParamsInt[0].bAuthMode = value; }
		}

		public byte KeyNo
		{
			get { return m_DataParamsInt[0].bKeyNo; }
			set { m_DataParamsInt[0].bKeyNo = value; }
		}

		public byte WrappedMode
		{
			get { return m_DataParamsInt[0].bWrappedMode; }
			set { m_DataParamsInt[0].bWrappedMode = value; }
		}

		public ushort AdditionalInfo
		{
			get { return m_DataParamsInt[0].wAdditionalInfo; }
			set { m_DataParamsInt[0].wAdditionalInfo = value; }
		}

		public byte[] Aid
		{
			get
			{
				byte[] bValue = new byte[3];
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < 3; i++ )
						{
							bValue[i] = pDataParams->pAid[i];
						}
					}
				}
				return bValue;
			}
			set
			{
				if ( value.Length > 3 )
					throw new ArgumentException ();
				unsafe
				{
					fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
					{
						for ( int i = 0; i < value.Length; i++ )
						{
							pDataParams->pAid[i] = value[i];
						}
					}
				}
			}
		}
		#endregion
	}
	#endregion SamAV3
#endif
	#endregion Sam_X
}
