/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alI18000p3m3
{
    #region DEFINES

    /// <summary>
    /// Custom Errors
    /// </summary>
    public enum Error : byte
    {
        /// <summary> Other Error. </summary>
        OTHER = (CustomCodes.ERROR_BEGIN),
        /// <summary> Memory overrun. </summary>
        MEMORY_OVERRUN,
        /// <summary> Memory locked. </summary>
        MEMORY_LOCKED,
        /// <summary> Insufficient power. </summary>
        INSUFFICIENT_POWER,
        /// <summary> Non-specific error. </summary>
        NON_SPECIFIC
    }

    /// <summary>
    /// ReqRn Options
    /// </summary>
    public enum ReqRnOption : byte
    {
        USE_CRC = 0x00,
        USE_HANDLE = 0x01
    }

    /// <summary>
    /// Access Command Options
    /// </summary>
    public enum ACOption : byte
    {
        NO_COVER_CODING = 0x00,
        USE_COVER_CODING = 0x01
    }

    /// <summary>
    /// Memory Banks
    /// </summary>
    public enum MemBanks : byte
    {
        RESERVERD = 0x00,
        UII = 0x01,
        TID = 0x02,
        USER = 0x03
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Ack(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to Tag data and, if required, PacketCRC. */
            ref ushort pRxBitLength /**< [Out] Tag response length in bits. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_ReqRn(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bOption,           /**< [In] Option parameter. */
            ref IntPtr ppRxBuffer   /**< [Out] New RN16 or handle. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Read(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bMemBank,          /**< [In] Memory bank where the read shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting read adress. */
            byte bWordPtrLength,    /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte bWordCount,        /**< [In] Number of bytes to read. */
            ref IntPtr ppRxBuffer,  /**< [Out] Header and requested memory words. */
            ref ushort pRxBitLength /**< [Out] Number of received bits. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Write(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bOption,           /**< [In] Option parameter. */
            byte bMemBank,          /**< [In] Memory bank where the write shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting write adress. */
            byte bWordPtrLength,    /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte[] pData            /**< [In] Word to write; byte[2]. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Kill(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte bOption,       /**< [In] Option parameter. */
            byte[] pPassword,   /**< [In] Full kill password; byte[4] */
            byte bRecom         /**< [In] Recommissioning bits. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Lock(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte[] pMask,       /**< [In] 10bit Action Field Mask; byte[2]. */
            byte[] pAction      /**< [In] 10bit Action Field; byte[2]. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Access(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte bOption,       /**< [In] Option parameter. */
            byte[] pPassword    /**< [In] Full access password; byte[4] */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_BlockWrite(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bMemBank,          /**< [In] Memory bank where the write shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting write adress. */
            byte bWordPtrLength,    /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte bWordCount,        /**< [In] Number of blocks to write. */
            byte[] pData            /**< [In] Words to write; byte[2 * \c bWordCount]. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_BlockErase(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bMemBank,          /**< [In] Memory bank where the erase shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting erase adress. */
            byte bWordPtrLength,    /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte bWordCount         /**< [In] Number of blocks to erase. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_BlockPermaLock(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bRFU,              /**< [In] RFU, shall be set to \c 0. */
            byte bReadLock,         /**< [In] Whether the permalock states shall be retrieved (\c 0) or the blocks shall be permalocked (\c 1). */
            byte bMemBank,          /**< [In] Memory bank where the erase shall be performed. */
            byte[] pBlockPtr,       /**< [In] Starting erase adress. */
            byte bBlockPtrLength,   /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte bBlockRange,       /**< [In] Mask range, specified in units of 16 blocks. */
            byte[] pMask,           /**< [In] Specifies which memory blocks a tag permalocks; byte[2 * \c bBlockRange] Ignored if \c bReadLock is \c 0 */
            ref IntPtr ppRxBuffer,  /**< [Out] Header and Permalock bits if \c bReadLock is \c 0 or NULL otherwise. */
            ref ushort pRxBitLength /**< [Out] Number of received bits if \c bReadLock is \c 0. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_SetHandle(
            IntPtr pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
            byte[] pHandle              /**< [In] Handle to the Card. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Ack(
            out byte[] pRxBuffer,   /**< [Out] Pointer to received data */
            out int pRxBitLength    /**< [Out] number of received data bits. */
            )
        {
            Status_t status;
            ushort wRxBitLength = 0;
            int wRxByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phalI18000p3m3_Ack(m_pDataParams, ref pRxBufferInt, ref wRxBitLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxBitLength > 0))
            {
                pRxBitLength = (int)wRxBitLength;
                wRxByteLength = pRxBitLength >> 3;
                if ((pRxBitLength & 0x07) != 0)
                {
                    ++wRxByteLength;
                }
                pRxBuffer = new byte[wRxByteLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxByteLength);
            }
            else
            {
                pRxBitLength = 0;
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t ReqRn(
            byte bOption,           /**< [In] Option parameter. */
            out byte[] pRxBuffer    /**< [Out] New RN16 or handle. */
            )
        {
            Status_t status;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phalI18000p3m3_ReqRn(m_pDataParams, bOption, ref pRxBufferInt);

            if (pRxBufferInt != IntPtr.Zero)
            {
                pRxBuffer = new byte[2];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, 2);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Read(
            byte bMemBank,          /**< [In] Memory bank where the read shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting read adress. */
            byte bWordCount,        /**< [In] Number of bytes to read. */
            out byte[] pRxBuffer,   /**< [Out] Header and requested memory words. */
            out int pRxBitLength    /**< [Out] Number of received bits. */
            )
        {
            Status_t status;
            ushort wRxBitLength = 0;
            int wRxByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;
            byte bPointerLength;
            if ((pWordPtr != null) && (pWordPtr.Length != 0))
            {
                bPointerLength = (byte)(pWordPtr.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            status = phalI18000p3m3_Read(m_pDataParams, bMemBank, pWordPtr, bPointerLength, bWordCount, ref pRxBufferInt, ref wRxBitLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxBitLength > 0))
            {
                pRxBitLength = (int)wRxBitLength;
                wRxByteLength = pRxBitLength >> 3;
                if ((pRxBitLength & 0x07) != 0)
                {
                    ++wRxByteLength;
                }
                pRxBuffer = new byte[wRxByteLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxByteLength);
            }
            else
            {
                pRxBitLength = 0;
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Write(
            byte bOption,           /**< [In] Option parameter. */
            byte bMemBank,          /**< [In] Memory bank where the write shall be performed. */
            byte[] pWordPtr,        /**< [In] Starting write adress. */
            byte[] pData            /**< [In] Word to write; byte[2]. */
            )
        {
            byte bPointerLength;
            if ((pWordPtr != null) && (pWordPtr.Length != 0))
            {
                bPointerLength = (byte)(pWordPtr.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            return phalI18000p3m3_Write(m_pDataParams, bOption, bMemBank, pWordPtr, bPointerLength, pData);
        }

        public Status_t Kill(
            byte bOption,       /**< [In] Option parameter. */
            byte[] pPassword,   /**< [In] Full kill password; byte[4] */
            byte bRecom         /**< [In] Recommissioning bits. */
            )
        {
            return phalI18000p3m3_Kill(m_pDataParams, bOption, pPassword, bRecom);
        }

        public Status_t Lock(
            byte[] pMask,   /**< [In] 10bit Action Field Mask; byte[2]. */
            byte[] pAction  /**< [In] 10bit Action Field; byte[2]. */
            )
        {
            return phalI18000p3m3_Lock(m_pDataParams, pMask, pAction);
        }

        public Status_t Access(
            byte bOption,       /**< [In] Option parameter. */
            byte[] pPassword    /**< [In] Full access password; byte[4] */
            )
        {
            return phalI18000p3m3_Access(m_pDataParams, bOption, pPassword);
        }

        public Status_t BlockWrite(
            byte bMemBank,      /**< [In] Memory bank where the write shall be performed. */
            byte[] pWordPtr,    /**< [In] Starting write adress. */
            byte bWordCount,    /**< [In] Number of blocks to write. */
            byte[] pData        /**< [In] Words to write; byte[2 * \c bWordCount]. */
            )
        {
            byte bPointerLength;
            if ((pWordPtr != null) && (pWordPtr.Length != 0))
            {
                bPointerLength = (byte)(pWordPtr.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            return phalI18000p3m3_BlockWrite(m_pDataParams, bMemBank, pWordPtr, bPointerLength, bWordCount, pData);
        }

        public Status_t BlockErase(
            byte bMemBank,      /**< [In] Memory bank where the erase shall be performed. */
            byte[] pWordPtr,    /**< [In] Starting erase adress. */
            byte bWordCount     /**< [In] Number of blocks to erase. */
            )
        {
            byte bPointerLength;
            if ((pWordPtr != null) && (pWordPtr.Length != 0))
            {
                bPointerLength = (byte)(pWordPtr.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            return phalI18000p3m3_BlockErase(m_pDataParams, bMemBank, pWordPtr, bPointerLength, bWordCount);
        }

        public Status_t BlockPermaLock(
            byte bRFU,              /**< [In] RFU, shall be set to \c 0. */
            byte bReadLock,         /**< [In] Whether the permalock states shall be retrieved (\c 0) or the blocks shall be permalocked (\c 1). */
            byte bMemBank,          /**< [In] Memory bank where the erase shall be performed. */
            byte[] pBlockPtr,       /**< [In] Starting erase adress. */
            byte bBlockRange,       /**< [In] Mask range, specified in units of 16 blocks. */
            byte[] pMask,           /**< [In] Specifies which memory blocks a tag permalocks; byte[2 * \c bBlockRange] Ignored if \c bReadLock is \c 0 */
            out byte[] pRxBuffer,   /**< [Out] Header and Permalock bits if \c bReadLock is \c 0 or NULL otherwise. */
            out int pRxBitLength    /**< [Out] Number of received bits if \c bReadLock is \c 0. */
            )
        {
            Status_t status;
            ushort wRxBitLength = 0;
            int wRxByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;
            byte bPointerLength;
            if ((pBlockPtr != null) && (pBlockPtr.Length != 0))
            {
                bPointerLength = (byte)(pBlockPtr.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            status = phalI18000p3m3_BlockPermaLock(
                m_pDataParams,
                bRFU,
                bReadLock,
                bMemBank,
                pBlockPtr,
                bPointerLength,
                bBlockRange,
                pMask,
                ref pRxBufferInt,
                ref wRxBitLength
                );

            if ((pRxBufferInt != IntPtr.Zero) && (wRxBitLength > 0))
            {
                pRxBitLength = (int)wRxBitLength;
                wRxByteLength = pRxBitLength >> 3;
                if ((pRxBitLength & 0x07) != 0)
                {
                    ++wRxByteLength;
                }
                pRxBuffer = new byte[wRxByteLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxByteLength);
            }
            else
            {
                pRxBitLength = 0;
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t SetHandle(
            byte[] pHandle            /**< [In] Handle to the Card. */
            )
        {
            return phalI18000p3m3_SetHandle(m_pDataParams, pHandle);
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : alI18000p3m3.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                      /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pPalI18000p3m3DataParams; /**< Pointer to the parameter structure of the underlying PAL layer. */
            public fixed byte abHandle[2];          /**< Handle retrieved by ReqRn(StoredCRC). */
            public byte bHandleValid;               /**< Whether Handle is valid or not. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalI18000p3m3_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pPalI18000p3m3DataParams /**< [In] Pointer to the parameter structure of the underlying PAL layer. */
            );

        #endregion

        #region INIT

        public Status_t Init(palI18000p3m3.Generic pPalI18000p3m3)
        {
            return phalI18000p3m3_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pPalI18000p3m3.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, palI18000p3m3.Generic pPalI18000p3m3)
        {
            return phalI18000p3m3_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pPalI18000p3m3.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
