/*
 * Copyright 2013, 2015, 2017 - 2018, 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Sli15693 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHPALSLI15693_H
#define PHPALSLI15693_H

#include <ph_Status.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHPAL_SLI15693_SW
/***************************************************************************************************************************************/
/* Software DataParams and Initialization Interface.                                                                                   */
/***************************************************************************************************************************************/

/** \defgroup phpalSli15693_Sw Component : Software
 * \brief Sub-Component to initialize PAL ISO15693 Generic component as Software.
 * @{
 */
#define PHPAL_SLI15693_SW_ID                                            0x01U   /**< ID for Software layer implementation. */

/** \brief Software Sli15693 PAL parameter structure */
typedef struct
{
    uint16_t wId;                                                               /**< ID for this component, not to be modified by user. This is initialized with #PHPAL_SLI15693_SW_ID in the
                                                                                 *   \ref phpalSli15693_Sw_Init "Initialization" function. This is used for validating the data parameter structure
                                                                                 *   and to make sure that APIs are called after initialization.
                                                                                 */
    void  * pHalDataParams;                                                     /**< Pointer to the HAL data parameter. This pointer is initialized with the address of HAL data parameter
                                                                                 *   structure by \ref phpalSli15693_Sw_Init "Initialization" interface.
                                                                                 */
    uint16_t wAdditionalInfo;                                                   /**< Holds the last error code sent by the VICC. But for Authentication Scenario contains the Flag Byte[non-error]
                                                                                 *   as well. When a response is received from the VICC with the error flag #PHPAL_SLI15693_FLAG_RESP_ERROR set, the
                                                                                 *   error code returned by VICC is stored here.
                                                                                 */
    uint8_t bFlags;                                                             /**< This holds the value of \ref req_flags "Request Flag" field used in the commands sent by the library. */
    uint8_t bResFlags;                                                          /**< This holds the value of \ref res_flags "Response Flag" field used in the commands sent by the library. */
    uint8_t pUid[8];                                                            /**< The VICC's Unique Identifier, which always is of 8 bytes. */
    uint8_t bUidBitLen;                                                         /**< The current length of the UID in bits. */
    uint8_t bExplicitlyAddressed;                                               /**< Indicator of API to be in explicit Addressed mode
                                                                                 *      - This indicates that the API is explicitly asked to be used in addressed mode.
                                                                                 *      - This parameter differentiate the understanding of explicitly addressed mode from the APIs like
                                                                                 *        \ref phpalSli15693_Select "Select" and \ref phpalSli15693_ResetToReady "Reset to Ready" which are
                                                                                 *        implicitly addressed
                                                                                 */
    uint8_t bOpeMode;                                                           /**< Operation Mode. Will be one of the following
                                                                                 *      - \ref RD_LIB_MODE_NFC "NFC"
                                                                                 *      - \ref RD_LIB_MODE_EMVCO "EMVCo"
                                                                                 *      - \ref RD_LIB_MODE_ISO "ISO"
                                                                                 */
    uint8_t bBuffering;                                                         /**< Enable or disable the data buffering. Supported values are
                                                                                 *      - \ref PH_OFF "Disable buffering of data"
                                                                                 *      - \ref PH_ON "Enable buffering of data"
                                                                                 */
    uint8_t bMaxRetryCount;                                                     /**< Set / Get Maximum Retry Count. */
    uint8_t bIcMfgCode;                                                         /**< Set / Get IC Manufacturer Code. */
} phpalSli15693_Sw_DataParams_t;

/**
 * \brief Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
 * This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "Configuration Options" for supported configurations
 * and their default values.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *          - Size of data parameter structure during \ref phpalSli15693_Sw_DataParams_t "Initialization",
 *            specified in wSizeOfDataParams is wrong.
 *          - If any of the DataParams are null.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phpalSli15693_Sw_Init(
        phpalSli15693_Sw_DataParams_t * pDataParams,                            /**< [In] Pointer to this layers parameter structure. */
        uint16_t wSizeOfDataParams,                                             /**< [In] Specifies the size of the data parameter structure. */
        void * pHalDataParams                                                   /**< [In] Pointer to the HAL data parameter structure. */
    );

/**
 * end of group phpalSli15693_Sw
 * @}
 */
#endif /* NXPBUILD__PHPAL_SLI15693_SW */





#ifdef NXPBUILD__PHPAL_SLI15693_RD710
/***************************************************************************************************************************************/
/* RD710 DataParams and Initialization Interface.                                                                                      */
/***************************************************************************************************************************************/

#include <phhalHw_Rd710_Cmd.h>

/** \defgroup phpalSli15693_Rd710 Component : Rd710
 * \brief Sub-Component to initialize PAL ISO15693 Generic component as RD710. Here the implementation of ISO15693 protocol is performed
 * by RD710 reader.
 * @{
 */

#define PHPAL_SLI15693_RD710_ID                                         0x02U   /**< ID for RD710 layer implementation. */

 /** \brief RD710 Hardware Sli15693 PAL parameter structure */
typedef struct
{
    uint16_t  wId;                                                              /**< ID for this component, not to be modified by user. This is initialized with #PHPAL_SLI15693_RD710_ID in the
                                                                                 *   \ref phpalSli15693_Rd710_Init "Initialization" function. This is used for validating the data parameter structure
                                                                                 *   and to make sure that APIs are called after initialization.
                                                                                 */
    phhalHw_Rd710_DataParams_t * pHalDataParams;                                /**< Pointer to the HAL Rd710 data parameter. This pointer is initialized with the address of Rd710 HAL data
                                                                                 *   parameter structure by \ref phpalSli15693_Rd710_Init "Initialization" interface.
                                                                                 */
    uint16_t wAdditionalInfo;                                                   /**< Holds the last error code sent by the VICC. But for Authentication Scenario contains the Flag Byte[non-error]
                                                                                 *   as well. When a response is received from the VICC with the error flag #PHPAL_SLI15693_FLAG_RESP_ERROR set, the
                                                                                 *   error code returned by VICC is stored here.
                                                                                 */
    uint8_t bFlags;                                                             /**< This holds the value of \ref req_flags "Request Flag" field used in the commands sent by the library. */
    uint8_t pUid[8];                                                            /**< The VICC's Unique Identifier, which always is of 8 bytes. */
    uint8_t bUidBitLen;                                                         /**< The current length of the UID in bits. */
} phpalSli15693_Rd710_DataParams_t;

/**
 * \brief Initialize PAL Sli15693 layer. This should be called before using any of the PAL Sli15693 APIs to do initialization of the layer.
 * This configures the Sli15693 PAL with default configurations. Refer \ref config_palsli "Configuration Options" for supported configurations
 * and their default values.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 *          - Size of data parameter structure during \ref phpalSli15693_Sw_DataParams_t "Initialization",
 *            specified in wSizeOfDataParams is wrong.
 *          - If any of the DataParams are null.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phpalSli15693_Rd710_Init(
        phpalSli15693_Rd710_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wSizeOfDataParams,                                             /**< [In] Specifies the size of the data parameter structure. */
        phhalHw_Rd710_DataParams_t * pHalDataParams                             /**< [In] Pointer to phhalHw_Rd710_DataParams_t parameter structure. */
    );

/**
 * end of group phpalSli15693_Rd710
 * @}
 */
#endif /* NXPBUILD__PHPAL_SLI15693_RD710 */





#ifdef NXPBUILD__PHPAL_SLI15693

/** \defgroup phpalSli15693 ISO/IEC 15693 / ISO/IEC 18000-3m1
 * \brief These component implements the protocol layer (activation) commands of ISO/IEC 15693 (ISO/IEC 18000-3M1) vicinity protocol.
 *
 * This implementation is based on ISO/IEC 15693-3:2009(E) specification. This layer also implements all activation related custom
 * commands of NXP's ICODE SLI series of tags.
 *
 * Different states of a VICC (Vicinity Integrated Circuit Card) and corresponding commands are shown below.
 *
 * \image html PalSli15693_States.png
 *
 * \b Note: The "VCD waiting time before sending a subsequent request" <em>(see 9.3, ISO/IEC 15693-3:2009(E))</em>
 *          is handled in the respective functions.
 *
 * @{
 */

/** \defgroup phpalSli15693_Errors ErrorCodes
 * \brief Description of error codes returned by VICC and Reader Library.
 * @{
 */

/**
 * \defgroup phpalSli15693_ErrorCodes_VICC ErrorCodes_VICC
 * \brief Description of actual error codes received from VICC. These codes are for reference purpose only.
 *
 * \note
 *  - All the interfaces will return codes mentioned in \ref phpalSli15693_ErrorCodes_Mapped "Library Mapped ErrorCodes"
 *    and not these codes.
 *  - Response Error codes according to ISO15693. Codes ranging from A0 - DF are custom command error codes.
 * @{
 */
#define PHPAL_SLI15693_FLAG_ERROR_NOT_SUPPORTED                         0x01U   /**< The command is not supported, i.e. the request code is not recognized. */
#define PHPAL_SLI15693_FLAG_ERROR_FORMAT                                0x02U   /**< The command is not recognized, for example: a format error occurred. */
#define PHPAL_SLI15693_FLAG_ERROR_OPTION                                0x03U   /**< The command option is not supported. */
#define PHPAL_SLI15693_FLAG_ERROR_GENERIC                               0x0FU   /**< Error with no information given or a specific error code is not supported. */
#define PHPAL_SLI15693_FLAG_ERROR_BLOCK_NA                              0x10U   /**< The specified block is not available (doesn't exist). */
#define PHPAL_SLI15693_FLAG_ERROR_BLOCK_LOCKED1                         0x11U   /**< The specified block is already locked and thus cannot be locked again. */
#define PHPAL_SLI15693_FLAG_ERROR_BLOCK_LOCKED2                         0x12U   /**< The specified block is locked and its content cannot be changed. */
#define PHPAL_SLI15693_FLAG_ERROR_BLOCK_WRITE1                          0x13U   /**< The specified block was not successfully programmed. */
#define PHPAL_SLI15693_FLAG_ERROR_BLOCK_WRITE2                          0x14U   /**< The specified block was not successfully locked. */
/**
 * end of group phpalSli15693_ErrorCodes_VICC
 * @}
 */

/** \defgroup phpalSli15693_ErrorCodes_Mapped ErrorCodes_Mapped
 * \brief Description of mapped error codes from Library with respect to VICC Error codes.
 * @{
 */
#define PHPAL_SLI15693_ERR_ISO15693                 (PH_ERR_CUSTOM_BEGIN + 0U)  /**< Generic error code from library to represent the VICC error codes mentioned in
                                                                                 *   \ref phpalSli15693_ErrorCodes_VICC "VICC Error Codes". The actual VICC error code can
                                                                                 *   be accessed via \ref phpalSli15693_SetConfig "SetConfig" interface using
                                                                                 *   \ref PHPAL_SLI15693_CONFIG_ADD_INFO "Additional Information" as Configuration.
                                                                                 */
#define PHPAL_SLI15693_ERR_IN_PROCESS_REPLY_PART1       \
            (PH_ERR_CUSTOM_BEGIN + 1U)                                          /**< Generic error code from library to represent the VICC error codes mentioned in
                                                                                 *   \ref phpalSli15693_ErrorCodes_VICC "VICC Error Codes" for In-Process reply part
                                                                                 *   1 response. The actual VICC error code can be accessed via \ref phpalSli15693_SetConfig
                                                                                 *   "SetConfig" interface using \ref PHPAL_SLI15693_CONFIG_ADD_INFO "Additional Information"
                                                                                 *   as Configuration.
                                                                                 */
#define PHPAL_SLI15693_ERR_IN_PROCESS_REPLY_PART2       \
            (PH_ERR_CUSTOM_BEGIN + 2U)                                          /**< Generic error code from library to represent the VICC error codes mentioned in
                                                                                 *   \ref phpalSli15693_ErrorCodes_VICC "VICC Error Codes" for In-Process reply part
                                                                                 *   2 response. The actual VICC error code can be accessed via \ref phpalSli15693_SetConfig
                                                                                 *   "SetConfig" interface using \ref PHPAL_SLI15693_CONFIG_ADD_INFO "Additional Information"
                                                                                 *   as Configuration.
                                                                                 */
/**
 * end of group phpalSli15693_ErrorCodes_Mapped
 * @}
 */

/**
 * end of group phpalSli15693_Errors
 * @}
 */

/**
 * \defgroup phpalSli15693_CommonDefs Defines
 * @{
 */

/**
 * \defgroup phpalSli15693_CommonDefs_BufferSize Buffer Size
 * @{
 */
#define PHPAL_SLI15693_UID_LENGTH                                       0x08U   /**< Macro to represent the UID size. */
#define PHPAL_SLI15693_BLOCK_LENGTH                                     0x04U   /**< Block length in bytes. */
#define PHPAL_SLI15693_PAGE_LENGTH                                      0x10U   /**< Number of bytes per page. */
/**
 * end of group phpalSli15693_CommonDefs_BufferSize
 * @}
 */

/** \defgroup phpalSli15693_Flags Flags
 * \brief Description about the Request Flags.
 * @{
 */

/** \defgroup phpalSli15693_Flags_Request Flags_Request
 * \brief Request Flags 1 and 8 according to ISO15693 specification. Refer \ref phpalSli15693_Flags_Common "Common Flags",
 * \ref phpalSli15693_Flags_INV_NotSet "Flags when Inventory is not SET" and \ref phpalSli15693_Flags_INV_Set
 * "Flags when Inventory is SET" for complete description and usage on the Flags.
 * \anchor req_flags
 * @{
 */

/** \defgroup phpalSli15693_Flags_Common Common
 * \brief Flags that are common and are part of bits 1 - 4 ( i.e. 0 - 3) according to ISO15693 specification.
 * \anchor req_flags_common
 * @{
 */
#define PHPAL_SLI15693_FLAG_TWO_SUB_CARRIERS                            0x01U   /**< Sub-carrier_flag. If set two sub-carriers shall be used (VICC to VCD). Else a single sub-carrier
                                                                                 *   frequency shall be used by the VICC.
                                                                                 */
#define PHPAL_SLI15693_FLAG_DATA_RATE                                   0x02U   /**< Data_rate_flag. If set high data rate shall be used (VICC to VCD). Else low data rate shall
                                                                                 *   be used.
                                                                                 */
#define PHPAL_SLI15693_FLAG_INVENTORY                                   0x04U   /**< Inventory_flag. Shall be set for inventory request. If set flags 5 to 8 meaning is as per this
                                                                                 *   \ref phpalSli15693_Flags_INV_Set "Inventory Set" list. If not set flags 5 to 8 meaning is defined
                                                                                 *   by this \ref phpalSli15693_Flags_INV_NotSet "Inventory Not Set" list.
                                                                                 */
#define PHPAL_SLI15693_FLAG_PROTOCOL_EXTENSION                          0x08U   /**< Protocol_Extension_flag. If set protocol format is extended. Reserved for future use. No supported
                                                                                 *   in current version.
                                                                                 */
#define PHPAL_SLI15693_FLAG_OPTION                                      0x40U   /**< Option_flag. Meaning is defined by the command description. It shall be set to 0 if not otherwise
                                                                                 *   defined by the command.
                                                                                 */
#define PHPAL_SLI15693_FLAG_FAST_DATA_RATE                              0x80U   /**< RFU as per ISO15693 but this flag should be utilized for NTAG5 product only. This flag is used to
                                                                                 *   set the hardware RX DataRate to Fast_High (~53 kbps) for Fast_Low (~13 kbps)
                                                                                 */
/**
 * end of group phpalSli15693_Flags_Common
 * @}
 */

/** \defgroup phpalSli15693_Flags_INV_NotSet Inventory_NotSet
 * \brief Flags when Inventory is NOT SET. These are part for bits 5 - 8 ( i.e\. 4 - 7) according to ISO15693 specification.
 * \anchor req_flags_inv_notset
 * @{
 */
#define PHPAL_SLI15693_FLAG_SELECTED                                    0x10U   /**< Select_flag. If set, only VICC in selected state shall respond. The \ref #PHPAL_SLI15693_FLAG_ADDRESSED
                                                                                 *   "Address Flag" shall be set to 0 and the UID field will not be included in the request. If not set
                                                                                 *   Request shall be executed by any VICC according to the setting of \ref #PHPAL_SLI15693_FLAG_ADDRESSED
                                                                                 *   "Address Flag".
                                                                                 */
#define PHPAL_SLI15693_FLAG_ADDRESSED                                   0x20U   /**< Address_flag. If set request, is addressed. UID field is included in request. It shall be executed only
                                                                                 *   by the VICC whose UID matches the UID specified in the request. If not set request is not addressed.
                                                                                 *   UID field is not included in request. It shall be executed by any VICC.
                                                                                 */
#define PHPAL_SLI15693_FLAG_NON_ADDRESSED                               0x00U   /**< Address_flag. If not set request, is addressed. UID field is not included in request. It shall be executed
                                                                                 *   on any VICC.
                                                                                 */
/**
 * end of group phpalSli15693_Flags_INV_NotSet
 * @}
 */

/**
 * \defgroup phpalSli15693_Flags_INV_Set Inventory_Set
 * \brief Flags when Inventory is SET. These are part for bits 5 - 8 ( i.e\. 4 - 7) according to ISO15693 specification.
 * \anchor req_flags_inv_set
 * @{
 */
#define PHPAL_SLI15693_MASK_INVENTORY_FLAGS                             0xF0U   /**< Inventory flags mask. */
#define PHPAL_SLI15693_FLAG_AFI                                         0x10U   /**< AFI_flag. If set AFI field is present in request. If not set AFI field is not present in request. */
#define PHPAL_SLI15693_FLAG_NBSLOTS                                     0x20U   /**< Nb_slots_flag. If set Use 1 slot instead of 16 slots for inventory. If not set use 16 slots. */
/**
 * end of group phpalSli15693_Flags_INV_Set
 * @}
 */

/**
 * end of group phpalSli15693_Flags_Request
 * @}
 */

/** \defgroup phpalSli15693_Flags_Response Flags_Response
 * \brief Response Flags 1 and 4 according to ISO15693. All others flags are RFU and shall not be interpreted.
 * \anchor res_flags
 * @{
 */
#define PHPAL_SLI15693_FLAG_RESP_ERROR                                  0x01U   /**< Error_flag.
                                                                                 *   Error detected. Error code is in the "Error" field of response. When a response is received from the VICC
                                                                                 *   with the error flag set, library returns the \ref PHPAL_SLI15693_ERR_ISO15693 "Custom Error Code". The
                                                                                 *   configuration option \ref PHPAL_SLI15693_CONFIG_ADD_INFO "Additional Information" can be used to
                                                                                 *   get the actual \ref phpalSli15693_ErrorCodes_Mapped "Mapped Error Codes" returned by the VICC.
                                                                                 */
#define PHPAL_SLI15693_FLAG_RESP_EXTERNSION                             0x08U   /**< Extension_flag. Protocol format is extended. Reserved for future use. */
/**
 * end of group phpalSli15693_Flags_Response
 * @}
 */

/**
 * end of group phpalSli15693_Flags
 * @}
 */

/**
 * \defgroup phpalSli15693_CommonDefs_ActivateCard ActivateCard
 * \brief Description of options for \ref phpalSli15693_ActivateCard "Activate Card" command.
 * @{
 */
#define PHPAL_SLI15693_ACTIVATE_DEFAULT                                 0x0000U /**< Default option. Same as #PHPAL_SLI15693_ACTIVATE_ADDRESSED. */
#define PHPAL_SLI15693_ACTIVATE_ADDRESSED                               0x0000U /**<  Card will be in addressed state (i.e. SELECT command is not send) after \ref phpalSli15693_ActivateCard
                                                                                 *    "Activate Card" command. After activate card with this option, to bring VICC to SELECTED state,
                                                                                 *    \ref phpalSli15693_Select "Select" command can be used.
                                                                                 */
#define PHPAL_SLI15693_ACTIVATE_SELECTED                                0x0001U /**<  Card will be in selected state (i.e. SELECT command is send) after \ref phpalSli15693_ActivateCard
                                                                                 *    "Activate Card" command.
                                                                                 */
/**
 * end of group phpalSli15693_CommonDefs_ActivateCard
 * @}
 */

/**
 * \defgroup phpalSli15693_CommonDefs_EOF EOF
 * \brief Description of EOF options for \ref phpalSli15693_SendEof "SendEOF" command.
 * @{
 */
#define PHPAL_SLI15693_EOF_NEXT_SLOT                                    0x00U   /**< Send an EOF with timing configuration for an \ref phpalSli15693_Inventory "Inventory" round. */
#define PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ                           0x01U   /**< Send an EOF with timing configuration for below mentioned commands
                                                                                 *      - \ref phpalSli15693_InventoryRead "Inventory Read"
                                                                                 *      - \ref phpalSli15693_FastInventoryRead "Fast Inventory Read"
                                                                                 *      - \ref phpalSli15693_InventoryPageRead "Inventory Page Read"
                                                                                 *      - \ref phpalSli15693_FastInventoryPageRead "Fast Inventory Page Read".
                                                                                 */
#define PHPAL_SLI15693_EOF_WRITE_ALIKE                                  0x02U   /**< Send an EOF with timing configuration for write alike commands without #PHPAL_SLI15693_TIMEOUT_LONG_US
                                                                                 *   "Long waiting time".
                                                                                 */
#define PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT                        0x03U   /**< Send an EOF with timing configuration for write alike commands with #PHPAL_SLI15693_TIMEOUT_LONG_US
                                                                                 *   "Long waiting time".
                                                                                 */
/**
 * end of group phpalSli15693_CommonDefs_EOF
 * @}
 */

/**
 * \defgroup phpalSli15693_CommonDefs_Timeout Timeout
 * \brief Description of Timeout options.
 * @{
 */
#define PHPAL_SLI15693_TIMEOUT_SHORT_US                                 324U    /**< Timeout T1 in microseconds as defined in ISO/IEC 15693, 9.1.4. */
#define PHPAL_SLI15693_TIMEOUT_DELTA_US                                 00000U  /**< Tolerance Timeout Delta in microseconds. */
#define PHPAL_SLI15693_TIMEOUT_NTAG5_I2C_US                             162U    /**< Timeout T1 in microseconds, 160.5 is taken as 162us Supported only by NTAG5 */
#define PHPAL_SLI15693_TIMEOUT_NTAG5_I2C_81_US                          81U     /**< Timeout T1 in microseconds, 80.2 is taken as 81us Supported only by NTAG5 */
#define PHPAL_SLI15693_TIMEOUT_LONG_US                                  20000U  /**< Extended timeout value in microseconds for long timeouts. */
#define PHPAL_SLI15693_TIMEOUT_EOF_US                                   16850U  /**< Special frame EOF delay time in microseconds (FDTv, EOF). */
#define PHPAL_SLI15693_TIMEOUT_MAX2_US                                  19950U  /**< FDT time max2 in microseconds (FDTv, listen, max2). */
/**
 * end of group phpalSli15693_CommonDefs_Timeout
 * @}
 */

/** \defgroup phpalSli15693_CommonDefs_DataRate DataRate
 * \brief The baud Rates supported by NTAG5 product.
 * @{
 */
#define PHPAL_SLI15693_26KBPS_DATARATE                                  0x1AU   /**< The default baud rate is 26kbps baud rate, is supported in both \ref RD_LIB_MODE_NFC
                                                                                 *   "NFC" and \ref RD_LIB_MODE_ISO "ISO" mode.
                                                                                 *   This is just for representation Purpose, a \ref phpalSli15693_GetConfig "GetConfig" after
                                                                                 *   applying this baud rate will return \ref PHHAL_HW_RF_TX_DATARATE_1_OUT_OF_4 "1 out of 4"
                                                                                 *    for TX and \ref PHHAL_HW_RF_RX_DATARATE_HIGH "High DataRate" for RX.
                                                                                 */
#define PHPAL_SLI15693_53KBPS_DATARATE                                  0x1BU   /**< 53kbps baud rate, supported in TX as well as RX but only for \ref RD_LIB_MODE_ISO "ISO"
                                                                                 *   mode. For RX this is just for representation Purpose, a \ref phpalSli15693_GetConfig "GetConfig"
                                                                                 *   after applying this baud rate will return \ref PHPAL_SLI15693_53KBPS_DATARATE "53 kbps" for TX and
                                                                                 *   \ref PHHAL_HW_RF_RX_DATARATE_FAST_HIGH "Fast High DataRate" for RX.
                                                                                 */
#define PHPAL_SLI15693_106KBPS_DATARATE                                 0x1CU   /**< 106kbps baud rate, supported in TX as well as RX but only for \ref RD_LIB_MODE_ISO "ISO" mode. */
#define PHPAL_SLI15693_212KBPS_DATARATE                                 0x1DU   /**< 212kbps baud rate, supported only in TX and \ref RD_LIB_MODE_ISO "ISO" mode. */
/**
 * end of group phpalSli15693_CommonDefs_DataRate
 * @}
 */

/**
 * end of group phpalSli15693_CommonDefs
 * @}
 */





/* ISO15693 Mandatory commands ------------------------------------------------------------------------------------- */
/**
 * \defgroup phpalSli15693_Mandatory Commands_Mandatory
 * \brief Describes the Mandatory commands of ISO15693 products.
 * @{
 */

/**
 * \brief Perform a ISO15693 Inventory command.
 *
 *  - When receiving the Inventory request, the VICC shall perform the anti-collision sequence.
 *    This is the fist command in the activation/detection sequence of VICC.
 *
 *  - When issuing the inventory command, the reader shall set the Nb_slots_flag \ref PHPAL_SLI15693_FLAG_NBSLOTS
 *     "NB_Slots Flag" to the desired setting to select 1 slot or 16 slots for the inventory process.
 *
 *  - The mask length indicates the number of significant bits of the mask value. It can have any value between 0
 *    and 60 when 16 slots are used and any value between 0 and 64 when 1 slot is used. The mask value is contained
 *    in an integer number of bytes. If the mask length is not a multiple of 8 (bits), the mask value MSB shall be
 *    padded with the required number of null (set to 0) bits so that the mask value is contained in an integer
 *    number of bytes.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_Inventory(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within \b pMask buffer. */
        uint8_t * pDsfid,                                                       /**< [Out] Data Storage Format Identifier. */
        uint8_t * pUid                                                          /**< [Out] Received UID. Should be #PHPAL_SLI15693_UID_LENGTH long. */
    );

/**
 * \brief Perform a ISO15693 StayQuiet command.
 *
 *  - When receiving the Stay quiet command, the VICC shall enter the quiet state and shall not send back a response.
 *    There is no response to the stay quiet command.
 *
 *  - In quiet state, the VICC shall not process any request where \ref PHPAL_SLI15693_FLAG_INVENTORY "Inventory Flag"
 *    is set and the VICC shall process any addressed request. The VICC shall exit the quiet state when reset (power off),
 *    receiving a \ref phpalSli15693_Select "Select request" (It shall then go to the selected state if supported or return
 *    an error if not supported), receiving a \ref phpalSli15693_ResetToReady "Reset to Ready" (It shall then go to the Ready
 *    state).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_StayQuiet(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );
/**
 * end of group phpalSli15693_Mandatory
 * @}
 */





/* ISO15693 Optional commands -------------------------------------------------------------------------------------- */
/**
 * \defgroup phpalSli15693_Optional Commands_Optional
 * \brief Describes the Optional commands of ISO15693 products.
 * @{
 */

/**
 * \brief Perform a ISO15693 Select command.
 *
 * When receiving the Select command, if the UID is equal to its own UID, the VICC shall enter the selected state and
 * shall send a response. If the UID is different to its own and in selected state, the VICC shall return to the Ready
 * state and shall not send a response. If the UID is different to its own and not in selected state, the VICC shall
 * remain in its state and shall not send a response.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_Select(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );

/**
 * \brief Perform a ISO15693 ResetToReady command.
 *
 * When receiving a Reset to ready command, the VICC shall return to the Ready state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid response received.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phpalSli15693_ResetToReady(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );
/**
 * end of group phpalSli15693_Optional
 * @}
 */





/* ISO15693 Custom commands ---------------------------------------------------------------------------------------- */
/**
 * \defgroup phpalSli15693_Custom Commands_Custom
 * \brief Describes the Custom commands of ISO15693 products.
 * @{
 */

/** \defgroup phpalSli15693_Custom_Defines Defines
 * \brief Description of options to be used for Custom command sets.
 * @{
 */

/** \defgroup phpalSli15693_Custom_Defines_ExtFlags ExtendedInventoryFlags
 * \brief Description of extended flags. These flags should be used for the below mentioned commands.
 *      - \ref phpalSli15693_InventoryReadExtended "InventoryRead Extended"
 *      - \ref phpalSli15693_FastInventoryReadExtended "Fast InventoryRead Extended"
 * @{
 */
#define PHPAL_SLI15693_FLAG_INVENTORY_READ_EXTENSION                    0x80U   /** Inventory Read Extension Flag
                                                                                 *      - If set the additional features of inventory mentioned along with the command
                                                                                 *        will be performed by the VICC.
                                                                                 *      - This flags needs to be set as the MSB bit for Mask Length.
                                                                                 */

#define PHPAL_SLI15693_EXTENDED_FLAGS_EAS_MODE                          0x01U   /**< Extended Inventory Read Options Flag.
                                                                                 *   Option to indicate that the VICC with EAS enabled will respond.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE                          0x02U   /**< Extended Inventory Read Options Flag.
                                                                                 *   If set the complete UID is transmitted by the VICC to the reader irrespective
                                                                                 *   of the mask length.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE                       0x04U   /**< Extended Inventory Read Options Flag.
                                                                                 *   If set the 16-bit CID will be transmitted in the command and only tags with the
                                                                                 *   same CID will respond.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE                      0x08U   /**< Extended Inventory Read Options Flag.
                                                                                 *   If set the Custom ID(CID) is transmitted by the VICC to the reader.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA                         0x10U   /**< Extended Inventory Read Options Flag.
                                                                                 *      - If set no user memory data are requested from the tag.
                                                                                 *      - If not set the tag will add the user memory blocks in the response as requested
                                                                                 *        in command.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_QUIET                        0x20U   /**< Extended Inventory Read Options Flag.
                                                                                 *   If set the VICC will go to Quiet state after receiving the command.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_PERSISTENT_QUIET             0x40U   /**< Extended Inventory Read Options Flag.
                                                                                 *   If set the VICC will go to Persistent Quiet state after receiving the command.
                                                                                 */
#define PHPAL_SLI15693_EXTENDED_FLAGS_RFU3                              0x80U

/**
 * end of group phpalSli15693_Custom_Defines_ExtFlags
 * @}
 */

/**
 * end of group phpalSli15693_Custom_Defines
 * @}
 */

/**
 * \brief Perform ICODE SLI InventoryRead command.
 *
 *  - When receiving the INVENTORY READ request, the ICODE SLI tag performs the same as the anti-collision sequence,
 *    with the difference that instead of the UID and the DSFID, the requested memory content is re-transmitted from
 *    the ICODE SLI. If an error is detected, the ICODE SLI tag remains silent.
 *
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is not set, n blocks of data are re-transmitted.
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is set, n blocks of data and the part of the UID which is
 *    not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead
 *    of padding with zeros up to the next byte boundary, the corresponding bits of the UID are returned.
 *
 *  - The number of bits of the re-transmitted UID can be calculated as follows:
 *      - 16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
 *      - 1 slot: 64 bits - mask length rounded up to the next byte boundary.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_InventoryRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bBlockNo,                                                       /**< [In] Block number of first block to read. */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ICODE SLI InventoryRead command in extended mode.
 *
 *  - When receiving the INVENTORY READ request, the ICODE SLI tag performs the same as the anti-collision sequence,
 *    with the difference that instead of the UID and the DSFID, the requested memory content is re-transmitted from
 *    the ICODE SLI. If an error is detected, the ICODE SLI tag remains silent.
 *
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is not set, n blocks of data are re-transmitted.
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is set, n blocks of data and the part of the UID which is
 *    not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots, are returned. Instead
 *    of padding with zeros up to the next byte boundary, the corresponding bits of the UID are returned.
 *
 *  - The number of bits of the re-transmitted UID can be calculated as follows:
 *      - 16 slots: 60 bits (bit 64 to bit 4), mask length rounded up to the next byte boundary.
 *      - 1 slot: 64 bits, mask length rounded up to the next byte boundary.
 *
 *  - Depending on different values of Extended Options the API behaves differently.
 *      | **Bit** | **Value** | **Description**                                                                                             |
 *      |:-------:|:---------:|:------------------------------------------------------------------------------------------------------------|
 *      |    b1   |     0     | Label responds independent from the EAS status.                                                             |
 *      |    ^    |     1     | Only labels will respond which have the EAS enabled.                                                        |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_EAS_MODE "EAS"**                           |
 *      |    b2   |     0     | UID will be transmitted as in regular mode.                                                                 |
 *      |    ^    |     1     | Complete UID will be transmitted.                                                                           |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE "UID"**                           |
 *      |    b3   |     0     | No CID is transmitted in the command.                                                                       |
 *      |    ^    |     1     | 16 bits CID will be transmitted in the command and only matching tags should reply.                         |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE "CID In Command"**             |
 *      |    b4   |     0     | CID will not be transmitted in the response from the tag.                                                   |
 *      |    ^    |     1     | CID will be transmitted in the response from the tag.                                                       |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE "CID In Response"**           |
 *      |    b5   |     0     | Tag will add the user memory block in the response as requested.                                            |
 *      |    ^    |     1     | No user memory data is requested from the tag.                                                              |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA "Skip Data"**                    |
 *      | b6 & b7 |     00    | Remain in current state.                                                                                    |
 *      |    ^    |     01    | Go to persistent quiet state after response.                                                                |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_PERSISTENT_QUIET "Persistent Quiet"** |
 *      |    ^    |     10    | Go to quiet state after response.                                                                           |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_QUIET "Quiet"**                       |
 *      |    ^    |     11    | Only tags in the PERSISTENT QUIET state will respond to the command.                                        |
 *      |    b8   |     0     | RFU                                                                                                         |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_InventoryReadExtended(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bExtendedOptions,                                               /**< [In] Refer \ref phpalSli15693_Custom_Defines_ExtFlags "Extended Inventory Flags"
                                                                                 *        for supported values.
                                                                                 */
        uint8_t * pCID,                                                         /**< [In] Two byte CID -> if marked in extended options */
        uint8_t bBlockNo,                                                       /**< [In] Block Number from where start reading */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to read */
        uint8_t * pCidOut,                                                      /**< [Out] Received CustomId */
        uint8_t * pUid,                                                         /**< [Out] Received UID. */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received Data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 FastInventoryRead command.
 *
 * When receiving the Fast Inventory Read command the ICODE SLI tag behaves the same as the \ref phpalSli15693_InventoryRead
 * "Inventory Read" command with the exceptions that the data rate in the direction ICODE SLI tag to the reader is twice
 * than defined in ISO/IEC 15693-3 depending on the \ref PHPAL_SLI15693_FLAG_DATA_RATE "DataRate" flag 53 kbit (high data rate)
 *   or 13 kbit (low data rate).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_FastInventoryRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bBlockNo,                                                       /**< [In] Block number of first block to read. */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ICODE SLI FastInventoryRead command in extended mode.
 *
 *  - When receiving the Fast Inventory Read command the ICODE SLI tag behaves the same as the \ref phpalSli15693_InventoryReadExtended
 *   "Inventory Read Extended" command with the exceptions that the data rate in the direction ICODE SLI tag to the reader is twice
 *   than defined in ISO/IEC 15693-3 depending on the \ref PHPAL_SLI15693_FLAG_DATA_RATE "DataRate" flag 53 kbit (high data rate)
 *   or 13 kbit (low data rate).
 *
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is not set, n blocks of data are re-transmitted.
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is set, n blocks of data and the part of the UID
 *    which is not part of the mask are re-transmitted. Additionally the slot number in case of 16 slots,
 *    are returned. Instead of padding with zeros up to the next byte boundary, the corresponding bits
 *    of the UID are returned.
 *
 *  - The number of bits of the re-transmitted UID can be calculated as follows:
 *      - 16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
 *      - 1 slot: 64 bits - mask length rounded up to the next byte boundary.
 *
 *  - Depending on different values of Extended Options the API behaves differently.
 *      | **Bit** | **Value** | **Description**                                                                                             |
 *      |:-------:|:---------:|:------------------------------------------------------------------------------------------------------------|
 *      |    b1   |     0     | Label responds independent from the EAS status.                                                             |
 *      |    ^    |     1     | Only labels will respond which have the EAS enabled.                                                        |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_EAS_MODE "EAS"**                           |
 *      |    b2   |     0     | UID will be transmitted as in regular mode.                                                                 |
 *      |    ^    |     1     | Complete UID will be transmitted.                                                                           |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_UID_MODE "UID"**                           |
 *      |    b3   |     0     | No CID is transmitted in the command.                                                                       |
 *      |    ^    |     1     | 16 bits CID will be transmitted in the command and only matching tags should reply.                         |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_COMPARE "CID In Command"**             |
 *      |    b4   |     0     | CID will not be transmitted in the response from the tag.                                                   |
 *      |    ^    |     1     | CID will be transmitted in the response from the tag.                                                       |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_CID_RESPONSE "CID In Response"**           |
 *      |    b5   |     0     | Tag will add the user memory block in the response as requested.                                            |
 *      |    ^    |     1     | No user memory data is requested from the tag.                                                              |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_SKIP_DATA "Skip Data"**                    |
 *      | b6 & b7 |     00    | Remain in current state.                                                                                    |
 *      |    ^    |     01    | Go to persistent quiet state after response.                                                                |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_PERSISTENT_QUIET "Persistent Quiet"** |
 *      |    ^    |     10    | Go to quiet state after response.                                                                           |
 *      |    ^    |     ^     | **Corresponding definition is \ref PHPAL_SLI15693_EXTENDED_FLAGS_GOTO_QUIET "Quiet"**                       |
 *      |    ^    |     11    | Only tags in the PERSISTENT QUIET state will respond to the command.                                        |
 *      |    b8   |     0     | RFU                                                                                                         |
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_FastInventoryReadExtended(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bExtendedOptions,                                               /**< [In] Refer \ref phpalSli15693_Custom_Defines_ExtFlags
                                                                                 *        "Extended Inventory Flags" for supported values.
                                                                                 */
        uint8_t * pCID,                                                         /**< [In] Two byte CID -> if marked in extended options */
        uint8_t bBlockNo,                                                       /**< [In] Block Number from where start reading */
        uint16_t wNoOfBlocks,                                                   /**< [In] Number of blocks to read */
        uint8_t * pCIDOut,                                                      /**< [Out] Received UID. */
        uint8_t * pUid,                                                         /**< [Out] Received UID. */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received Data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 InventoryPageRead command.
 *
 * - When receiving the Inventory Page Read request, the ISO15693 tag performs the same as in the anti-collision sequence,
 *   with the difference that instead of the UID and the DSFID the requested memory content is re-transmitted from the tag.
 *
 *  - If an error is detected the ISO15693 tag remains silent.
 *
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is not set, n pages of data including page protection status
 *    (password protection condition) are re-transmitted.
 *  - If the \ref PHPAL_SLI15693_FLAG_OPTION "Option Flag" is set, n pages (4 blocks = 16 byte) of data including page
 *    protection status (password protection condition) and the part of the UID which is not part of the mask are
 *    re-transmitted. Additionally the slot number in case of 16 slots, are returned.
 *
 *  - The number of bits of the re-transmitted UID can be calculated as follows:
 *      - 16 slots: 60 bits (bit 64 to bit 4) - mask length rounded up to the next byte boundary.
 *      - 1 slot: 64 bits - mask length rounded up to the next byte boundary.
 *
 *  - If a requested page is protected with the Read password and the valid Read password has not been transmitted
 *    before, the page is not read and only protection status (0x0F) is returned.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_InventoryPageRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bPageNo,                                                        /**< [In] Block number of first page to read. */
        uint16_t wNoOfPages,                                                    /**< [In] Number of pages to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 FastInventoryPageRead command.
 *
 *  - When receiving the Fast Inventory Page Read command the ICODE SLI tag behaves the same as the \ref phpalSli15693_InventoryPageRead
 *   "Inventory Page Read" command with the exceptions that the data rate in the direction ICODE SLI tag to the reader is twice
 *   than defined in ISO/IEC 15693-3 depending on the \ref PHPAL_SLI15693_FLAG_DATA_RATE "DataRate" flag 53 kbit (high data rate)
 *   or 13 kbit (low data rate).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
*/
phStatus_t phpalSli15693_FastInventoryPageRead(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bFlags,                                                         /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t bAfi,                                                           /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t bMaskBitLen,                                                    /**< [In] Number of UID bits within pMask. */
        uint8_t bPageNo,                                                        /**< [In] Block number of first page to read. */
        uint16_t wNoOfPages,                                                    /**< [In] Number of pages to read */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ICODE SLI Stay quite persistent command.
 *
 *  - When receiving the Stay Quiet Persistent command, the tag enters the persistent quiet state and
 *    will not send back a response.
 *
 *  - The Stay Quiet Persistent command provides the same behavior as the mandatory \ref phpalSli15693_StayQuiet
 *    "STAY QUIET" command with the only difference at a reset (power off), the tag will turn to the ready state,
 *    if the power off time is exceeding the persistent time. Refer the corresponding ICODE SLI tag specification
 *    for the value of persistent time.
 *
 * The tag will exit the persistent quiet state when:
 *      - Reset (power off) exceeding the persistent time.
 *      - Receiving a SELECT request. It shall then go to the Selected state.
 *      - Receiving a RESET TO READY request. It shall then go to the Ready state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_StayQuietPersistent(
        void * pDataParams                                                      /**< [In] Pointer to this layers parameter structure. */
    );
/**
 * end of group phpalSli15693_Custom
 * @}
 */





/* ISO15693 Common commands ---------------------------------------------------------------------------------------- */
/** \defgroup phpalSli15693_Common Commands_Commmon
 * \brief Common commands that performs Activation of the VICC, Exchange to VICC, etc...
 * @{
 */

/**
 * \brief Perform ISO15693 ActivateCard command.
 *
 * - This function activates the first VICC detected during the collision resolution and optionally move it to selected
 *   state based on the wOption specified.
 *
 * - \b bOption can be one of:
 *      - \ref PHPAL_SLI15693_ACTIVATE_DEFAULT      "Default"
 *      - \ref PHPAL_SLI15693_ACTIVATE_SELECTED     "Selected"
 *      - \ref PHPAL_SLI15693_ACTIVATE_ADDRESSED    "Addressed"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_ActivateCard(
        void *   pDataParams,                                                   /**< [In] Pointer to this layers parameter structure. */
        uint8_t  bOption,                                                       /**< [In] Selected or addressed. Refer description for supported options. */
        uint8_t  bFlags,                                                        /**< [In] Request flags byte. Refer below set of flags
                                                                                 *          - \ref phpalSli15693_Flags_Common "Common"
                                                                                 *          - \ref phpalSli15693_Flags_INV_NotSet "Inventory_NotSet"
                                                                                 *          - \ref phpalSli15693_Flags_INV_Set "Inventory_Set"
                                                                                 */
        uint8_t  bAfi,                                                          /**< [In] Application Family Identifier. */
        uint8_t * pMask,                                                        /**< [In] UID mask, holding known UID bits. */
        uint8_t  bMaskBitLen,                                                   /**< [In] Number of UID bits within pMask. */
        uint8_t * pDsfid,                                                       /**< [Out] Data Storage Format Identifier. */
        uint8_t * pUid,                                                         /**< [Out] Received UID of first found card */
        uint8_t * pMoreCardsAvailable                                           /**< [Out] If there are more cards available this value is different from zero. */
    );

/**
 * \brief Send a ISO15693 EOF (End of Frame).
 *
 * - When 16 slots are used in the inventory process, to switch to the next slot, the reader shall send an EOF with option
 *   \ref PHPAL_SLI15693_EOF_NEXT_SLOT "EOF Next Slot".
 *
 * - To send EOF for write alike commands, \ref PHPAL_SLI15693_EOF_WRITE_ALIKE "Write Alike" or \ref PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT
 *   "Write Alike with Wait" shall be used. For ISO15693 custom commands like Inventory Read, \ref PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ
 *   "EOF Next Slot with Inventory Read" option shall be used.
 *
 * - \b bOption can be one of
 *      - \ref PHPAL_SLI15693_EOF_NEXT_SLOT                 "EOF Next Slot"
 *      - \ref PHPAL_SLI15693_EOF_WRITE_ALIKE               "EOF Write Alike"
 *      - \ref PHPAL_SLI15693_EOF_NEXT_SLOT_INV_READ        "EOF Next Slot with Inventory Read"
 *      - \ref PHPAL_SLI15693_EOF_WRITE_ALIKE_WITH_WAIT     "EOF Write Alike with Wait"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_SendEof(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t bOption,                                                        /**< [In] Option parameter. Refer description for supported options. */
        uint8_t * pDsfid,                                                       /**< [Out] Data Storage Format Identifier. */
        uint8_t * pUid,                                                         /**< [Out] Received UID */
        uint8_t * pUidLen,                                                      /**< [Out] Length of bytes available in \b pUid buffer. */
        uint8_t * pData,                                                        /**< [Out] Received data. */
        uint16_t * pDataLen                                                     /**< [Out] Length of bytes available in \b pData buffer. */
    );

/**
 * \brief Perform ISO15693 Data Exchange with VICC.
 *
 *  - \b wOption can be one of:
 *      - \ref PH_EXCHANGE_DEFAULT      "Default. VICC Exchange is performed."
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Information. VICC Exchange is not performed."
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer intermediate information. VICC Exchange is not performed."
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer last information (if any) and Exchange to VICC."
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_PROTOCOL_ERROR       Invalid response received.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_Exchange(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wOption,                                                       /**< [In] Buffering options. Refer description for supported options. */
        uint8_t * pTxBuffer,                                                    /**< [In] Data to be transmitted to VICC.
                                                                                 *          \note: Flags, IC MFC code and UID will be added automatically by
                                                                                 *                 the library internally.
                                                                                 */
        uint16_t wTxBuffLen,                                                    /**< [In] Length of bytes available in \b pTxBuffer buffer. */
        uint8_t ** ppRxBuffer,                                                  /**< [Out] Received information from VICC. */
        uint16_t * pRxBuffLen                                                   /**< [Out] Length of bytes available in \b ppRxBuffer buffer. */
    );


/**
 * end of group phpalSli15693_Common
 * @}
 */





/* ISO15693 Utility commands --------------------------------------------------------------------------------------- */
/**
 * \defgroup phpalSli15693_Utils Utilities
 * \brief Describes the utility interfaces. These interfaces are actually not part of ISO15693
 * These are generic interfaces as part of reader library for internal purpose.
 * @{
 */

/** \defgroup phpalSli15693_Utility_Defines Defines
 * \brief Configuration and Option definitions. These options shall be used along with
 * \ref phpalSli15693_SetConfig "SetConfig" and \ref phpalSli15693_GetConfig "GetConfig" to
 * configure PAL Sli15693 layer.
 * \anchor config_palsli
 * @{
 */
#define PHPAL_SLI15693_CONFIG_FLAGS                                     0x0000U /**< Set or get the flags byte. This shall be used to set or get the value of
                                                                                 *   \ref phpalSli15693_Flags_Request "Request Flag" field used in the commands
                                                                                 *   sent by the library.
                                                                                 */
#define PHPAL_SLI15693_CONFIG_ADD_INFO                                  0x0001U /**< Get the additional info word. This shall be used to set or get the value of
                                                                                 *   \ref phpalSli15693_Flags_Response "Response Flag" for authentication commands
                                                                                 *   or the error code field used in the response received by the library.
                                                                                 */
#define PHPAL_SLI15693_CONFIG_TIMEOUT_US                                0x0002U /**< Set or get the current timeout in microseconds. */
#define PHPAL_SLI15693_CONFIG_TIMEOUT_MS                                0x0003U /**< Set or get the current timeout in milliseconds. */
#define PHPAL_SLI15693_CONFIG_TXDATARATE                                0x0004U /**< Set or get the current TX Data rate. */
#define PHPAL_SLI15693_CONFIG_RXDATARATE                                0x0005U /**< Set or get the current RX Data rate. */
#define PHPAL_SLI15693_CONFIG_T1_PARAMETER                              0x0006U /**< Set or get the current T1 timeout after Parameter Select. */
#define PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING                          0x0007U /**< Option to Get / Set the buffering of response data to be enabled or disabled.
                                                                                 *   By default this flag will be disabled.
                                                                                 */
#define PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT                             0x0008U /**< Set / Get Maximum Retry Count. */
#define PHPAL_SLI15693_CONFIG_ICMFGCODE                                 0x0009U /**< Set / Get IC Manufacturer Code. */
#define PHPAL_SLI15693_CONFIG_TRANSPARENT                               0x0010U /**< Set or get the Transparent mode. */
/**
 * end of group phpalSli15693_Utility_Defines
 * @}
 */

/**
 * \brief Retrieve the serial number (UID).
 *
 * This API is used to get the UID of current VICC with which this layer is communicating.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval #PH_ERR_USE_CONDITION        No UID available.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_GetSerialNo(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t * pUid,                                                         /**< [Out] Buffer, holding the UID (8 bytes) */
        uint8_t * bUidLen                                                       /**< [Out] Length of bytes available in \b pUid buffer. */
    );

/**
 * \brief Set serial number (UID).
 *
 * When multiple VICCs are detected, this API shall be used to set a specific UID to this layer from the list of detected UIDs.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  If DataParams is null.
 * \retval #PH_ERR_INVALID_PARAMETER    If the buffers are null.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_SetSerialNo(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint8_t * pUid,                                                         /**< [In] Buffer, holding the UID (8 bytes) */
        uint8_t bUidLen                                                         /**< [In] Length of UID in bytes. */
    );

/**
 * \brief Set value of the specified \ref phpalSli15693_Utility_Defines "Configuration Identifier".
 *
 *  - The SetConfig command is used to set values for the different configuration parameters of this layer.
 *
 *  - wConfig can be one of:
 *      - \ref PHPAL_SLI15693_CONFIG_FLAGS              "Flags"
 *      - \ref PHPAL_SLI15693_CONFIG_ADD_INFO           "Additional Information"
 *      - \ref PHPAL_SLI15693_CONFIG_TIMEOUT_US         "Timeout in us"
 *      - \ref PHPAL_SLI15693_CONFIG_TIMEOUT_MS         "Timeout in ms"
 *      - \ref PHPAL_SLI15693_CONFIG_TXDATARATE         "Tx DataRate"
 *      - \ref PHPAL_SLI15693_CONFIG_RXDATARATE         "Rx DataRate"
 *      - \ref PHPAL_SLI15693_CONFIG_T1_PARAMETER       "T1 Parameter"
 *      - \ref PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING   "Enable Buffering"
 *      - \ref PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT      "Maximum Retry Count"
 *      - \ref PHPAL_SLI15693_CONFIG_ICMFGCODE          "IC Manufacturer Code"
 *      - \ref PHPAL_SLI15693_CONFIG_TRANSPARENT        "Transparent"
 *
 * \b NOTE: Both the flags values \ref PHPAL_SLI15693_FLAG_DATA_RATE "DataRate" and \ref PHPAL_SLI15693_FLAG_FAST_DATA_RATE
 *          "Fast DataRate" should not be combined, it should be passed separately along with other flags.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS      If DataParams is null.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Invalid wConfig.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_SetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to configure. */
        uint16_t wValue                                                         /**< [In] Value to set. */
    );

/**
 * \brief Get value of the specified \ref phpalSli15693_Utility_Defines "Configuration Identifier".
 *
 *  - The SetConfig command is used to set values for the different configuration parameters of this layer.
 *
 *  - wConfig can be one of:
 *      - \ref PHPAL_SLI15693_CONFIG_FLAGS              "Flags"
 *      - \ref PHPAL_SLI15693_CONFIG_ADD_INFO           "Additional Information"
 *      - \ref PHPAL_SLI15693_CONFIG_TIMEOUT_US         "Timeout in us"
 *      - \ref PHPAL_SLI15693_CONFIG_TIMEOUT_MS         "Timeout in ms"
 *      - \ref PHPAL_SLI15693_CONFIG_TXDATARATE         "Tx DataRate"
 *      - \ref PHPAL_SLI15693_CONFIG_RXDATARATE         "Rx DataRate"
 *      - \ref PHPAL_SLI15693_CONFIG_T1_PARAMETER       "T1 Parameter"
 *      - \ref PHPAL_SLI15693_CONFIG_ENABLE_BUFFERING   "Enable Buffering"
 *      - \ref PHPAL_SLI15693_CONFIG_MAXRETRYCOUNT      "Maximum Retry Count"
 *      - \ref PHPAL_SLI15693_CONFIG_ICMFGCODE          "IC Manufacturer Code"
 *      - \ref PHPAL_SLI15693_CONFIG_TRANSPARENT        "Transparent"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS      If DataParams is null.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Invalid wConfig.
 * \retval XXXX
 *          - Depending on status codes return by PICC.
 *          - Other Depending on implementation and underlying component.
 */
phStatus_t phpalSli15693_GetConfig(
        void * pDataParams,                                                     /**< [In] Pointer to this layers parameter structure. */
        uint16_t wConfig,                                                       /**< [In] Configuration to read. */
        uint16_t * pValue                                                       /**< [In] Read value. */
    );

/**
 * end of group phpalSli15693_Utils
 * @}
 */

/**
 * end of phpalSli15693
 * @}
 */
#endif /* NXPBUILD__PHPAL_SLI15693 */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHPALSLI15693_H */
