/*
 * Copyright 2013, 2015, 2017, 2019, 2020, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHTOOLS_H
#define PHTOOLS_H

#include <ph_Status.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

/** \defgroup phTools Tools
*
* \brief This component implements tools that are necessary for many different software modules
* @{
*/

/**
* \name ToolsConfigs
*/
/** @{ */
#define PH_TOOLS_CRC16_PRESET_ISO14443A     0x6363U     /**< CRC16 Preset for ISO14443-A. */
#define PH_TOOLS_CRC16_PRESET_ISO14443B     0xFFFFU     /**< CRC16 Preset for ISO14443-B. */
#define PH_TOOLS_CRC16_PRESET_FELICA        0x0000U     /**< CRC16 Preset for Felica. */
#define PH_TOOLS_CRC16_PRESET_ISO15693      0xFFFFU     /**< CRC16 Preset for ISO15693. */
#define PH_TOOLS_CRC16_PRESET_CONTACTBASED  0xFFFFU     /**< CRC16 Preset for Contactbased Cards. */
#define PH_TOOLS_CRC16_POLY_ISO14443        0x8408U     /**< CRC16 Polynom for ISO14443. */
#define PH_TOOLS_CRC16_POLY_ISO15693        0x8408U     /**< CRC16 Polynom for ISO15693. */
#define PH_TOOLS_CRC16_POLY_CONTACTBASED    0x8408U     /**< CRC16 Polynom for Contactbased Cards. */
#define PH_TOOLS_CRC32_PRESET_ISO14443      0xFFFFFFFFU /**< CRC32 Preset for ISO14443 Enhanced Frame. */
#define PH_TOOLS_CRC32_POLY_ISO14443        0x04C11DB7U /**< CRC32 Polynom for ISO14443 Enhanced Frame. */
#define PH_TOOLS_CRC32_PRESET_DF8           0xFFFFFFFFU /**< CRC32 Preset for DesFire. */
#define PH_TOOLS_CRC32_POLY_DF8             0xEDB88320U /**< CRC32 Polynom for DesFire. */
#define PH_TOOLS_CRC8_PRESET_EPC            0xFFU       /**< Preset value for EPC CRC-8 generation. */
#define PH_TOOLS_CRC8_PRESET_UID            0xFDU       /**< Preset value for UID CRC-8 generation. */
#define PH_TOOLS_CRC8_POLY_EPCUID           0x1DU       /**< Polynomial for EPC/UID CRC-8 generation: x^8 + x^4 + x^3 + x^2 + 1 (MSB first). */
#define PH_TOOLS_CRC16_PRESET_EPCUID        0xFFFFU     /**< Preset value for EPC/UID CRC-16 generation. */
#define PH_TOOLS_CRC16_POLY_EPCUID          0x1021U     /**< Polynomial for EPC/UID CRC-16 generation: x^16 + x^12 + x^5 + 1 (MSB first). */
#define PH_TOOLS_CRC5_PRESET_I18000P3       0x09U       /**< Preset value for ISO18000-3 CRC-5 generation. */
#define PH_TOOLS_CRC5_POLY_I18000P3         0x09U       /**< Polynomial for ISO18000-3 CRC-5 generation: x^5 + x^3 + 1 (LSB first). */
#define PH_TOOLS_CRC16_PRESET_I18000P3      0xFFFFU     /**< Preset value for ISO18000-3 CRC-16 generation. */
#define PH_TOOLS_CRC16_POLY_I18000P3        0x8408U     /**< Polynomial for ISO18000-3 CRC-16 generation: x^16 + x^12 + x^5 + 1 (LSB first). */
#define PH_TOOLS_CRC16_POLY_FELICA          0x1021U     /**< Polynomial for Felica CRC-16 generation: x^16 + x^12 + x^5 + 1 (MSB first). */
#define PH_TOOLS_PARITY_OPTION_EVEN         0x00U       /**< Encode/Decode even parity bits. */
#define PH_TOOLS_PARITY_OPTION_ODD          0x01U       /**< Encode/Decode odd parity bits. */
#define PH_TOOLS_PARITY_OPTION_DONT_CARE    0x02U       /**< Decode don't care parity bits. (just remove) */

#define PH_TOOLS_CRC_OPTION_DEFAULT         0x00U       /**< Default CRC operation. */
#define PH_TOOLS_CRC_OPTION_OUPUT_INVERTED  0x01U       /**< Inverts the bits after calculation. */
#define PH_TOOLS_CRC_OPTION_MSB_FIRST       0x02U       /**< Calculates CRC with MSB(it) first. */
#define PH_TOOLS_CRC_OPTION_BITWISE         0x04U       /**< Bitwise CRC, \c wDataLength represents the Bitlength. */
#define PH_TOOLS_CRC_OPTION_MASK            0x07U       /**< Mask of valid option bits. */

#define PH_TOOLS_CRC32_OPTION_REVERSE_DATA_BYTE 0x08U       /**< Bitwise CRC, \c wDataLength represents the Bitlength. */
#define PH_TOOLS_CRC32_OPTION_REVERSE_CRC       0x10U       /**< Bitwise CRC, \c wDataLength represents the Bitlength. */
#define PH_TOOLS_CRC32_OPTION_MASK              0x1FU       /**< Mask of valid option bits for CRC 32. */
/** @} */

/**
* \brief Calculate even or odd parity.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_EncodeParity(
                                uint8_t bOption,                /**< [In] Parity option; e.g. #PH_TOOLS_PARITY_OPTION_EVEN. */
                                uint8_t * pInBuffer,            /**< [In] Array to input data. */
                                uint16_t wInBufferLength,       /**< [In] Length of input data in bytes. */
                                uint8_t bInBufferBits,          /**< [In] Number of valid bits in last byte of pInBuffer. */
                                uint16_t wOutBufferSize,        /**< [In] Size of the output buffer. */
                                uint8_t * pOutBuffer,           /**< [Out] Output buffer. */
                                uint16_t * pOutBufferLength,    /**< [Out] Number of valid bytes in pOutBuffer. */
                                uint8_t * pOutBufferBits        /**< [Out] Number of valid bits in last byte of pOutBuffer. */
                                );

/**
* \brief Verify and Remove even or odd parity.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_DecodeParity(
                                uint8_t bOption,                /**< [In] Parity option; e.g. #PH_TOOLS_PARITY_OPTION_EVEN. */
                                uint8_t * pInBuffer,            /**< [In] Array to input data. */
                                uint16_t wInBufferLength,       /**< [In] Length of input data in bytes. */
                                uint8_t bInBufferBits,          /**< [In] Number of valid bits in last byte of pInBuffer. */
                                uint16_t wOutBufferSize,        /**< [In] Size of the output buffer. */
                                uint8_t * pOutBuffer,           /**< [Out] Output buffer. */
                                uint16_t * pOutBufferLength,    /**< [Out] Number of valid bytes in pOutBuffer. */
                                uint8_t * pOutBufferBits        /**< [Out] Number of valid bits in last byte of pOutBuffer. */
                                );

/**
* \brief Calculate a CRC 5
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_CalculateCrc5(
                                 uint8_t bOption,       /**< [In] Specifies whether the output should be inverted or not. */
                                 uint8_t bPreset,       /**< [In] Preset used for CRC calculation, e.g. #PH_TOOLS_CRC5_PRESET_I18000P3. */
                                 uint8_t bPolynom,      /**< [In] Polynom used for reduction, e.g. #PH_TOOLS_CRC5_POLY_I18000P3. */
                                 uint8_t * pData,       /**< [In] Array to input data. */
                                 uint16_t wDataLength,  /**< [In] Length of input data. */
                                 uint8_t * pCrc         /**< [Out] Resulting CRC. */
                                 );

/**
* \brief Calculate a CRC 8
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_CalculateCrc8(
                                 uint8_t bOption,       /**< [In] Specifies whether the output should be inverted or not. */
                                 uint8_t bPreset,       /**< [In] Preset used for CRC calculation, e.g. #PH_TOOLS_CRC8_PRESET_UID. */
                                 uint8_t bPolynom,      /**< [In] Polynom used for reduction, e.g. #PH_TOOLS_CRC8_POLY_EPCUID. */
                                 uint8_t * pData,       /**< [In] Array to input data. */
                                 uint16_t wDataLength,  /**< [In] Length of input data. */
                                 uint8_t * pCrc         /**< [Out] Resulting CRC. */
                                 );

/**
* \brief Calculate a CRC 16
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_CalculateCrc16(
                                  uint8_t bOption,      /**< [In] Specifies whether the output should be inverted or not. */
                                  uint16_t wPreset,     /**< [In] Preset used for CRC calculation, e.g. #PH_TOOLS_CRC16_PRESET_ISO14443A. */
                                  uint16_t wPolynom,    /**< [In] Polynom used for reduction, e.g. #PH_TOOLS_CRC16_POLY_ISO14443. */
                                  uint8_t * pData,      /**< [In] Array to input data. */
                                  uint16_t wDataLength, /**< [In] Length of input data. */
                                  uint16_t * pCrc       /**< [Out] Resulting CRC. */
                                  );

/**
* \brief Calculate a CRC 32
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_CalculateCrc32(
                                  uint8_t bOption,      /**< [In] Specifies whether the output should be inverted or not. */
                                  uint32_t dwPreset,    /**< [In] Preset used for CRC calculation, e.g. #PH_TOOLS_CRC32_PRESET_DF8. */
                                  uint32_t dwPolynom,   /**< [In] Polynom used for reduction, e.g. #PH_TOOLS_CRC32_POLY_DF8. */
                                  uint8_t * pData,      /**< [In] Array to input data. */
                                  uint32_t dwDataLength, /**< [In] Length of input data. */
                                  uint32_t * pCrc       /**< [Out] Resulting CRC. */
                                  );

/**
* \brief Calculate the LRC
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_CalculateLrc(
                                uint8_t * pData,        /**< [In] Array to input data. */
                                uint16_t wDataLength,   /**< [In] Length of input data. */
                                uint8_t * pLrc          /**< [Out] Resulting LRC. */
                                );

/**
* \brief add Hamming Byte to data
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_HammingEncode(
                                 uint8_t* pData,                  /**< [In] Data to be encoded with Hamming (must be 7 bytes) */
                                 uint8_t* pEnhancedSubBlock,      /**< [Out] Data including the calculated Hamming Byte (8th Byte) */
                                 uint16_t wEnhancedSubBlockSize,  /**< [In] Size of the output buffer */
                                 uint16_t wEnhancedSubBlockOffset /**< [In] Offset where the encoded block should be stored */
                                 );

/**
* \brief add Hamming Byte to data with length > 7 bytes
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_HammingEncodeMoreBlocks(
                                           uint8_t*  pData,                  /**< [In] Data to be encoded with Hamming */
                                           uint16_t  wDataLen,               /**< [In] Length of Data */
                                           uint8_t*  pEnhancedBlocks,        /**< [Out] Data including the calculated Hamming Byte (8th Byte) */
                                           uint16_t  wEnhancedBlocksSize,    /**< [In] Size of pEnhancedBlock */
                                           uint16_t* pwEnhancedBlocksLen     /**< [Out] Size of the Blocks encoded */
                                           );

/**
* \brief check Hamming Byte and correct 1 bit error if occured
* \return Status code 0 or error is correction was not possible
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_HammingDecode(
                                        uint8_t* pEnhancedSubBlock,         /**< [In] Data to be checked for Hamming (must be 8 bytes) */
                                        uint8_t* pDecodedData,              /**< [Out] Corrected data */
                                        uint16_t wDecodedDataSize,          /**< [In] Size of the output buffer */
                                        uint16_t wDecodedDataOffset         /**< [In] Offset where the encoded block should be stored */
                                        );


/**
* \brief check Hamming Byte and correct 1 bit error if occured - with data length > 7
* \return Status code 0 or error is correction was not possible
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_HammingDecodeMoreBlocks(
                                           uint8_t* pEnhancedBlocks,         /**< [In] Data to be checked for Hamming */
                                           uint16_t wEnhancedBlocksLen,      /**< [In] Length of Data */
                                           uint8_t*  pDecodedBlocks,         /**< [Out] (corrected) data without Hamming Byte */
                                           uint16_t  wDecodedBlocksSize,     /**< [In] Size of pDecodedBlocks */
                                           uint16_t* wDecodedBlocksLen       /**< [Out] Legth of the decoded data */
                                           );
/**
* \brief Sleep for certain amount of time
* \return Status code 0 or error
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phTools_Sleep(
                        uint32_t dwTimeMilliSeconds    /**< [In] Time in Milliseconds to call Sleep */
                        );

/**
* end of phTools group
* @}
*/

#ifdef __cplusplus
}/*Extern C*/
#endif

#endif /* PHTOOLS_H */
