/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for configuring MiFare Ultralight, MiFare Classic, MiFare DesFire as NFC tag.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
* Header for this file
*/
#include "Example-Rd710_NDEF_Demo.h"

/**
* Standard Header
*/
#ifdef _WIN32
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#endif
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#ifdef _WIN32
#pragma warning(pop)            /* PRQA S 3116 */
#endif

/**
* Pegoda 2 configurations
*/
#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */

#define GLOBAL_BUFFER_SIZE     0x012CU  /**< global size of transmit and recieve buffer */

#define NDEF_SIZE_UL  19   /**< Size of the NDEF_msg array */

#define CHECK_SUCCESS(x)  {if ((x) != PH_ERR_SUCCESS) {printf("An error occured (0x%04X), press any key to exit...", (x)); _getch(); return 0;}}

/* define Key Store constants */
#define NUMBER_OF_KEYENTRIES 2
#define NUMBER_OF_KEYVERSIONPAIRS 1
#define NUMBER_OF_KUCENTRIES 1

/*******************************************************************************
**   NDEF (NFC Data Exchange Format) message for Mifare Ultralight
********************************************************************************
**   Here we will see an example of how to write an NDEF message into a
**   MIFARE(R) Ultralight card.
**   In addition to the explanation below you can get very detailed information
**   on http://www.nfc-forum.org/specs/spec_list
**
**   The purpose of writing NDEF messages onto a Tag is to provide
**   some information that every NFC device can read. That information can be a
**   URL or some text.
**   But now let's take a look at the example below:
**
**   When writing NDEF messages you have to differentiate between a static
**   memory structure and a dynamic memory structure. That depends on the
**   memory size of the tag.
**   * A static memory structure is used for tags with a memory size equal
**     to 64 bytes (like our MIFARE(R) Ultralight).
**   * A dynamic memory structure is used for tags with a memory size bigger
**     than 64 bytes.
**   Because of using a MIFARE(R) Ultralight card we have a static memory
**   structure.
**
**   The first 3 blocks(0-2) are reserved for the UID, some internal data and
**   the lock bits of the memory. Therefore we begin writing at Block 3.
**   Keep in mind, that we can only write if the
**   lock bits (Block 2, Byte 2 and 3) are all set to 0b (read an write
**   permission).
**
**   Block 3 describes the CC (Capability Container). If a bit is set to logic
**   1 it cannot be changed back to 0b anymore.
**   The meaning of Block 3 in this example is the following:
**   Byte 0 is equal to E1h to indicate that NFC Forum defined data is
**          stored in the data area.
**   Byte 1 is the version number of the document we are taking as a guide for
**          writing NDEF messages. For the following example I'm using the
**          version 1.0 of the document supported by the Type 2 tag platform.
**          The 4 MSB (Most Significant Bits) indicate the major version
**          number, and the 4 LSB (Least Significant Bits) indicate the
**          minor version number. So we use 10h for Byte 1.
**   Byte 2 indicates the memory size of the data area. MIFARE(R) Ultralight
**          cards provide a size of 48 Bytes for the data. 48 divided by 8 is 6.
**          So Byte 2 is 0x06.
**   Byte 3 indicates the read and write access capability of the data and CC
**          area of the Type 2 tag platform. For the read and write access we
**          need to have the value of 0x00.
**
**   Block 4 describes the data area. Beginning with Byte 0 we can start writing
**   our TLV (Tag Length Value). Our valid TLV consists of the following:
**   Byte 0 describes the type of the content of the V (Value). Here we
**          need 0x03 in order to mark an NDEF message.
**   Byte 1 describes the size of the Value field in Bytes. In our example
**          we have a size of 12 Bytes. So Byte 1 is 0x0C.
**   Byte 2 is always set to 0xD1 for NDEF messages.
**   Byte 3 contains the length of the Record Type. For URI we have 0x01.
**
**   Block 5 continues with the Value.
**   Byte 0 indicates the length of the payload. In our example we have 7 Bytes
**          for "nxp.com" . However, we have to add 1 for the protocol
**          abbreviation. So we set Byte 0 to 0x08.
**   Byte 1 describes the type of the NDEF record. Here we have "URI", so we
**          set Byte 1 to "U", which is 0x55 in the ASCII table.
**   Byte 2 describes the type of URI we have. There are different hex codes for
**          "http://www.", "ftp://ftp.", "http://" and so on. So we set Byte 1
**          to 0x01 for "http://www.".
**   Byte 3 last but not least our NDEF message starts here. At the end of the
**          message we have to remember to write the closing 0xFE after
**          our NDEF message.
/*******************************************************************************
**   NDEF (NFC Data Exchange Format) message for Mifare Classic
********************************************************************************
**   In order to write data into the blocks of MiFare classic cards, first authentication should be done.
**   For authentication, this application uses KeyB which should be 0xFFFFFFFFFFFF.
**   Block0 of MiFare classic contains Manufacturer data.
**   Block1 and Block2 should contain MiFare application directory(MAD).
**   MAD should contain CRC, Info byte along with MAD data
**      MAD data = 0x03E103E103E103E103E103E103E103E103E103E103E103E103E103E103E1
**      Info byte = 0x01
**      CRC = 0x14
**   In order to access the MAD of Mifare Classic card, it should be written with a public KeyA
**      KeyA = 0xA0A1A2A3A4A5
**   Sector condition code for sector 0(block0 to block3) should be 0x787788
**   GPB = 0xC1
**   Block 4 describes the data area. Beginning with Byte 0 we can start writing
**   our TLV (Tag Length Value). Our valid TLV consists of the following:
**   Byte 0 describes the type of the content of the V (Value). Here we
**          need 0x03 in order to mark an NDEF message.
**   Byte 1 describes the size of the Value field in Bytes. In our example
**          we have a size of 12 Bytes. So Byte 1 is 0x0C.
**   Byte 2 is always set to 0xD1 for NDEF messages.
**   Byte 3 contains the length of the Record Type. For URI we have 0x01.
**   Byte 4 indicates the length of the payload. In our example we have 7 Bytes
**          for "nxp.com" . However, we have to add 1 for the protocol
**          abbreviation. So we set Byte 0 to 0x08.
**   Byte 5 describes the type of the NDEF record. Here we have "URI", so we
**          set Byte 1 to "U", which is 0x55 in the ASCII table.
**   Byte 6 describes the type of URI we have. There are different hex codes for
**          "http://www.", "ftp://ftp.", "http://" and so on. So we set Byte 1
**          to 0x01 for "http://www.".
**   Byte 7 last but not least our NDEF message starts here. At the end of the
**          message we have to remember to write the closing 0xFE after
**          our NDEF message.
**   NDEF message can be extracted from aNDEF_msg. i.e.,NDEF message is from aNDEF_msg[4] to aNDEF_msg[18]
**   To access the NDEF message stored in sector 1, block7 should be written with public KeyA
**      KeyA = 0xD3F7D3F7D3F7
**   Sector condition code for sector0 and sector1 should be 0x7F0788
**   While writing the sector trailer, KeyB should be equal to 0xFFFFFFFFFFFF.
**   By retaining the KeyB we can reuse the sectors.
/*******************************************************************************
**   NDEF (NFC Data Exchange Format) message for Mifare DesFire
********************************************************************************
**   Create Application with the follwing parameters
**       App ID : 0x000001
**       Key setting1 : 0x0F
**                    - Key Access Rights = 0
**                    - Configuration changeable
**                    - PICC master key not required for create/delete file
**                    - Free Directory list access without master key
**                    - Allow changing the master key
**       Key setting2 : 0x21
**                    - Allow ISO file IDs within application
**                    - No. of keys is 0x01
**       ISO File ID : 0x05E1
**       ISO DF Name : 0xD2760000850101
**   Select application with ID : 0x000001
**   Create Std file(for Capability Container) with following parameters
**       File No. : 0x01
**       ISO File ID : 0xE103
**       Communication setting : 0x00 (No encryption)
**       Access Rights : 0xEEEE
**       File Size : 0x0F
**   Create Std file(for NDEF) with following parameters
**       File No. : 0x02
**       ISO File ID : 0xE104
**       Communication setting : 0x00 (No encryption)
**       Access Rights : 0xEEEE
**       File Size : 0x0F
**   Write data for CC File with following parameters
**       File No. : 0x01
**       Offset : 0x000000
**       Length : 0x0F
**       Data : 0x000F20003B00340406E10410000000
**   Write data for NDEF File with following parameters
**       File No. : 0x02
**       Offset : 0x000000
**       Length : 0x0E
**       Data : aNDEF_msg[4] = 0x00
**   NDEF message is from aNDEF_msg[4] to aNDEF_msg[18]
*******************************************************************************/

int __cdecl main()
{
    phStatus_t status;

    uint8_t aAtr[256];                                /**< Atr */
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];   /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];   /**< HAL Reader receive buffer */
    uint8_t pUid[10];                                 /**< Array for Uid */
    uint8_t bLenUid;                                  /**< Uid Length variable */
    uint8_t bSak;                                     /**< SAK variable */
    uint8_t bMoreCardsAvaliable;                      /**< Variable to indicate the single/multiple cards in the field */
    uint8_t aNDEF_msg[NDEF_SIZE_UL] = {0xE1, 0x10, 0x06, 0x00,  /* Block 3 (CC) NDEF message */
        0x03, 0x0C, 0xD1, 0x01,  /* Block 4 */
        0x08, 'U',  0x01, 'n',   /* Block 5 */
        'x',  'p',  '.',  'c',   /* Block 6 */
        'o',  'm',  0xFE};       /* Block 7 */
    uint8_t pAts[255];                                /**< Answer to select buffer */
    uint8_t aAID[3] = {0x01, 0x00, 0x00};             /**< Application Id for Desfire cards */
    uint8_t aISOFileId_A[2] = {0x05, 0xE1};           /**< ISO file Id for application */
    uint8_t aISODFName[7] = {0xD2, 0x76, 0x00,        /**< DF Name buffer */
        0x00, 0x85, 0x01,
        0x01};
    uint8_t bDFLen = 7;                               /**< DF Name buffer length */
    uint8_t bCidEnable;                               /**< Card Identifier enable variable */
    uint8_t bCid;                                     /**< Card Identifier */
    uint8_t bNadSupported;                            /**< Node address support variable */
    uint8_t bFwi;                                     /**< Frame wait integer */
    uint8_t bFsdi;                                    /**< Frame size integer of PCD */
    uint8_t bFsci;                                    /**< Frame size integer of PICC */
    uint8_t aISOFileId_F[2] = {0x03, 0xE1};	          /**< ISOFileId for file */
    uint8_t aAccessRights[2] = {0xEE, 0xEE};          /**< Access Rights */
    uint8_t aFileSize[3] = {0x0F, 0x00, 0x00};        /**< File size specifier */
    uint8_t aOffset[3] = {0x00, 0x00, 0x00};          /**< Offset for write data */
    uint8_t aData[15] = {0x00, 0x0F, 0x20,            /**< Data buffer */
        0x00, 0x3B, 0x00,
        0x34, 0x04, 0x06,
        0xE1, 0x04, 0x10,
        0x00, 0x00, 0x00};
    uint8_t aDataLen[3] = {0x0F, 0x00, 0x00};          /**< Valid data buffer length */
    uint8_t bI;                                        /**< Used as counter */
    uint8_t bBlockNr;                                  /**< Block number variable */
    int sel;                                           /**< Selection variable for Ul/DF */

    /* Bfl data parameter storage */
    phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
    phhalHw_Rd710_DataParams_t halRd710;              /**< Rd710 HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t I14443p3a_Sw;      /**< Software PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t I14443p4a_Sw;      /**< Software PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t I14443p4_Sw;        /**< Software PAL-ISO14443P4 parameter structure */
    phpalMifare_Sw_DataParams_t palMifare_Sw;         /**< Software PAL-MIFARE parameter structure */
    phalMful_Sw_DataParams_t alMifareUl;              /**< Software AL-MIFARE-ULTRALIGHT parameter structure */
    phalMfdf_Sw_DataParams_t alMfdf;                  /**< Software AL-MIFARE-DESFIRE parameter structure */
    phalMfc_Sw_DataParams_t alMfc;                    /**< Software AL-MIFARE-Classic parameter structure */
    phKeyStore_Sw_DataParams_t KeyStore;              /**< Software KeyStore parameter structure */

#ifdef NXPBUILD__PH_CRYPTOSYM

    phCryptoSym_Sw_DataParams_t CryptoEnc;            /**< CryptoSym parameter structure for ENC */
    phCryptoSym_Sw_DataParams_t CryptoSymRnd;         /**< CryptoSym parameter structure for SymRng */
    phCryptoRng_Sw_DataParams_t CryptoRng;            /**< CryptoRng parameter structure for Rng */

#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* variables used by the AL (Mifare) component */
    uint8_t  bBlockNo, bKeyType;
    uint16_t wKeyNumber, wKeyVersion;
    uint8_t pBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];

    /* variables used by the KeyStore component */
    uint16_t wKeyStoreKeyType;
    uint8_t pNewKey[12];         /* KeyA & KeyB */
    phKeyStore_Sw_KeyEntry_t        pKeyEntries[NUMBER_OF_KEYENTRIES];
    uint16_t wNoOfKeyEntries =      NUMBER_OF_KEYENTRIES;
    phKeyStore_Sw_KeyVersionPair_t  pKeyVersionPairs[NUMBER_OF_KEYVERSIONPAIRS * NUMBER_OF_KEYENTRIES];
    uint16_t wNoOfKeyVersionPairs = NUMBER_OF_KEYVERSIONPAIRS;
    phKeyStore_Sw_KUCEntry_t        pKUCEntries[NUMBER_OF_KUCENTRIES];
    uint16_t wNoOfKUCEntries =      NUMBER_OF_KUCENTRIES;

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /* Init. PCSC BAL */
    status = phbalReg_PcscWin_Init(&balPcsc,     /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phbalReg_PcscWin_DataParams_t),   /**< [In] Specifies the size of the data parameter structure. */
        aAtr,                                    /**< [In] ATR buffer. */
        sizeof(aAtr));                           /**< [In] ATR buffer size. */
    CHECK_SUCCESS(status);

    /* set the BAL Communication Configuration values */
    status = phbalReg_SetConfig(&balPcsc,              /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,             /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);   /**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);

    /* set the BAL Communication Configs */
    status = phbalReg_SetConfig(&balPcsc,        /**< [In] Pointer to this layer's parameter structure. */
        PHBAL_REG_PCSCWIN_CONFIG_SHARE,          /**< BAL Communication Configs */
        PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);   /**< BAL Communication Configuration values */
    CHECK_SUCCESS(status);

    /* Select Port to be used (no SAM reader) */
    status = phbalReg_SetPort(&balPcsc,          /**< [In] Pointer to this layer's parameter structure. */
        (uint8_t*)PCSC_READER_P2_NO_SAM_NAME);   /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

    /* Initialise the Rd710 HAL component */
    status = phhalHw_Rd710_Init(&halRd710,    /**< [In] Pointer to this layer's parameter structure. */
        sizeof(phhalHw_Rd710_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
        &balPcsc,                             /**< [In] Pointer to the lower layers parameter structure */
        0,                                    /**< [In] Slot number */
        aHalBufferReaderTx,                   /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderTx),           /**< [In] Size of the global transmit buffer. */
        aHalBufferReaderRx,                   /**< [In] Pointer to global receive buffer used by the Exchange() function. */
        sizeof(aHalBufferReaderRx));          /**< [In] Size of the global receive buffer. */
    CHECK_SUCCESS(status);

    /* Open the PCSC port */
    status = phbalReg_OpenPort(&balPcsc);   /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

	/* Init the 710 reader */
	status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710); /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /* init the 14443-3A component */
    status = phpalI14443p3a_Sw_Init(&I14443p3a_Sw,   /**< [In] PAL-ISO14443P3A parameter structure */
        sizeof(phpalI14443p3a_Sw_DataParams_t),      /**< [In] Specifies the size of the data parameter structure */
        &halRd710);                                  /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /* init the 14443-4A component */
    status = phpalI14443p4a_Sw_Init(&I14443p4a_Sw,   /**< [In] PAL-ISO14443P4A parameter structure */
        sizeof(phpalI14443p4a_Sw_DataParams_t),      /**< [In] Specifies the size of the data parameter structure */
        &halRd710);                                  /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /* init the 14443-4 component */
    status = phpalI14443p4_Sw_Init(&I14443p4_Sw,   /**< [In] PAL-ISO14443P4 parameter structure */
        sizeof(phpalI14443p4_Sw_DataParams_t),     /**< [In] Specifies the size of the data parameter structure */
        &halRd710);                                /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /* Initialize Mifare pal	*/
    status = phpalMifare_Sw_Init(&palMifare_Sw,   /**< [In] PAL-MIFARE parameter structure */
        sizeof(phpalMifare_Sw_DataParams_t),      /**< [In] Specifies the size of the data parameter structure */
        &halRd710,                                /**< [In] Pointer to the parameter structure of the underlying layer.*/
        &I14443p4_Sw);                            /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
    CHECK_SUCCESS(status);

    /* Perform Field Reset */
    status = phhalHw_FieldReset(&halRd710);   /**< [In] Pointer to the parameter structure of the underlying layer */
    CHECK_SUCCESS(status);

    /* Configure HAL for typeA cards */
    status = phhalHw_ApplyProtocolSettings(&halRd710,   /**< [In] Pointer to the parameter structure of the underlying layer.*/
        PHHAL_HW_CARDTYPE_ISO14443A);                   /**< [In] Card type is IS014443A. */
    CHECK_SUCCESS(status);

    /* Perform Field Reset */
    status = phhalHw_FieldReset(&halRd710);   /**< [In] Pointer to the parameter structure of the underlying layer */
    CHECK_SUCCESS(status);

    /* Menu for card selection */
    printf("Enter your card type to write NDEF message to\n");
    printf("1.Mifare Ultralight\n");
    printf("2.Mifare Classic\n");
    printf("3.Mifare Desfire\n");
    sel = _getch();

    if (sel == '1')
    {
        /* Initialize Mifare Ultralight */
        status = phalMful_Sw_Init(&alMifareUl,   /**< [In] AL-Mifare Ultralight parameter structure */
            sizeof(alMifareUl),                  /**< [In] Specifies the size of the data parameter structure */
            &palMifare_Sw,                       /**< [In] PAL-MIFARE parameter structure */
            NULL,                                /**<[In] Keystore parameter structure */
            NULL,                                /**<[In] Crypto parameter structure */
            NULL);                               /**<[In] CryptoRng parameter structure */
        CHECK_SUCCESS(status);

        /* Activate card */
        status = phpalI14443p3a_ActivateCard(&I14443p3a_Sw,   /**< [In] PAL-ISO14443P3A parameter structure */
            pUid,                                             /**< [In] Known Uid */
            0x00,                                             /**< [In] Known Uid length */
            pUid,                                             /**< [Out] Complete Uid */
            &bLenUid,                                         /**< [Out] Lenght Uid **/
            &bSak,                                            /**< [out] SAK */
            &bMoreCardsAvaliable);                            /**< [Out] Whether there are more cards in the field or not */
        CHECK_SUCCESS(status);

        /* Write the NFC tag data to the blocks	*/
        for (bI = 0, bBlockNr = 3; bI < NDEF_SIZE_UL; bI += 4, bBlockNr++)
        {
            status = phalMful_Write(&alMifareUl,
                bBlockNr,
                &aNDEF_msg[bI]);
            CHECK_SUCCESS(status);
        }
    }
    else if (sel == '2')
    {
        /* initialise the Key Store: */
        status = phKeyStore_Sw_Init(&KeyStore,   /**< [In] Keystore parameter structure */
            sizeof(KeyStore),                    /**< [In] Size of Keystore parameter structure */
            pKeyEntries,                         /**< [In] Array of key entries */
            wNoOfKeyEntries,                     /**< [In] Number of key entries */
            pKeyVersionPairs,                    /**< [In] Array of key version pairs */
            wNoOfKeyVersionPairs,                /**< [In] Number of key version pairs */
            pKUCEntries,                         /**< [In] Key usage counter entry storage */
            wNoOfKUCEntries);                    /**< [In] Number of Key usage counter entries */
        CHECK_SUCCESS(status);

        /* load a Key to the Store */
        wKeyNumber = 1;
        wKeyVersion = 0;
        wKeyStoreKeyType = PH_KEYSTORE_KEY_TYPE_MIFARE;
        status = phKeyStore_FormatKeyEntry(&KeyStore, wKeyNumber, wKeyStoreKeyType);
        CHECK_SUCCESS(status);

        /* Set the Key */
        memcpy(pNewKey, "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF", PH_KEYSTORE_KEY_TYPE_MIFARE_SIZE);

        /* Store the key in keystore */
        status = phKeyStore_SetKey(&KeyStore,   /**< [In] Keystore parameter structure */
            wKeyNumber,                         /**< [In] Key number */
            wKeyVersion,                        /**< [In] Key version */
            wKeyStoreKeyType,                   /**< [In] MiFare type key */
            pNewKey,                            /**< [In] Key array */
            wKeyVersion);                       /**< [In] Key version */
        CHECK_SUCCESS(status);

        /* Initialise the MiFare Classic */
        status = phalMfc_Sw_Init(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            sizeof(alMfc),                 /**< [In] size of this layer's parameter structure */
            &palMifare_Sw,                 /**< [In] Pointer to a palMifare component */
            &KeyStore);                    /**< [In] Pointer to a Keystore component */
        CHECK_SUCCESS(status);

        /* Activate card */
        status = phpalI14443p3a_ActivateCard(&I14443p3a_Sw,   /**< [In] PAL-ISO14443P3A parameter structure */
            pUid,                                             /**< [In] Known Uid */
            0x00,                                             /**< [In] Known Uid length */
            pUid,                                             /**< [Out] Complete Uid */
            &bLenUid,                                         /**< [Out] Lenght Uid */
            &bSak,                                            /**< [out] SAK */
            &bMoreCardsAvaliable);                            /**< [Out] Whether there are more cards in the field or not */
        CHECK_SUCCESS(status);

        /* Authenticate for block0 to block3 with KEYB */
        bBlockNo = 1;
        bKeyType = PHPAL_MIFARE_KEYB;
        wKeyNumber = 1;
        wKeyVersion = 0;

        status = phalMfc_Authenticate(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                           /**< [In] Block number to be authenticated */
            bKeyType,                           /**< [In] KeyB, MiFare type */
            wKeyNumber,                         /**< [In] Key number */
            wKeyVersion,                        /**< [In] Key Version */
            pUid,                               /**< [In] Uid of the card */
            bLenUid);                           /**< [In] Length of the Uid specified */
        CHECK_SUCCESS(status);

        /* Copy the MAD to be written in block1 */
        memcpy(pBlockData, "\x14\x01\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1", sizeof(pBlockData));

        /* Write the MAD to block1 */
        status = phalMfc_Write(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                    /**< [In] Block No.1 */
            pBlockData);                 /**< [In] Block data to be written */
        CHECK_SUCCESS(status);

        /* Copy the MAD to be written in block2 */
        memcpy(pBlockData, "\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1\x03\xE1", sizeof(pBlockData));

        bBlockNo = 2;

        /* Write the MAD to block2 */
        status = phalMfc_Write(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                    /**< [In] Block No.2 */
            pBlockData);                 /**< [In] Block data to be written */
        CHECK_SUCCESS(status);

        /* Copy KeyA, AC, GPB, KeyB to be written in block3(Sector trailer) */
        memcpy(pBlockData, "\xA0\xA1\xA2\xA3\xA4\xA5\x78\x77\x88\xC1\xFF\xFF\xFF\xFF\xFF\xFF", sizeof(pBlockData));

        bBlockNo = 3;

        /* Write sector trailer to block3 */
        status = phalMfc_Write(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                    /**< [In] Block No.3 */
            pBlockData);                 /**< [In] Block data to be written */
        CHECK_SUCCESS(status);

        /* Authenticate for block4 to block7 with KeyB */
        bBlockNo = 4;
        bKeyType = PHPAL_MIFARE_KEYB;
        wKeyNumber = 1;
        wKeyVersion = 0;
        status = phalMfc_Authenticate(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                           /**< [In] Block No.4 */
            bKeyType,                           /**< [In] KeyB Mifare type */
            wKeyNumber,                         /**< [In] Key number = 1 */
            wKeyVersion,                        /**< [In] Key Version */
            pUid,                               /**< [In] Card Uid */
            bLenUid);                           /**< [In] Card Uid length */
        CHECK_SUCCESS(status);

        /* Write NDEF message to block3 */
        status = phalMfc_Write(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                    /**< [In] Block No.4 */
            &aNDEF_msg[4]);              /**< [In] NDEF message to be written */
        CHECK_SUCCESS(status);

        /* Copy KeyA, AC, GPB, KeyB to be written in block7(Sector trailer) */
        memcpy(pBlockData, "\xD3\xF7\xD3\xF7\xD3\xF7\x7F\x07\x88\x40\xFF\xFF\xFF\xFF\xFF\xFF", sizeof(pBlockData));

        bBlockNo = 7;

        /* Write sector trailer to block3 */
        status = phalMfc_Write(&alMfc,   /**< [In] Pointer to application layer's parameter structure */
            bBlockNo,                    /**< [In] Block No.7 */
            pBlockData);                 /**< [In] Sector trailer data */
        CHECK_SUCCESS(status);
    }
    else if (sel == '3')
    {
#ifdef NXPBUILD__PH_CRYPTOSYM

        /* initialise the Key Store: */
        status = phKeyStore_Sw_Init(&KeyStore,  /**< [In] Keystore parameter structure */
            sizeof(KeyStore),                   /**< [In] Size of Keystore parameter structure */
            pKeyEntries,                        /**< [In] Array of key entries */
            wNoOfKeyEntries,                    /**< [In] Number of key entries */
            pKeyVersionPairs,                   /**< [In] Array of key version pairs */
            wNoOfKeyVersionPairs,               /**< [In] Number of key version pairs */
            pKUCEntries,                        /**< [In] Key usage counter entry storage */
            wNoOfKUCEntries);                   /**< [In] Number of Key usage counter entries */
        CHECK_SUCCESS(status);

        /*Intialize the crypto components */
        status = phCryptoSym_Sw_Init(&CryptoEnc,    /**< [In] Pointer to this layer's parameter structure. */
            sizeof(phCryptoSym_Sw_DataParams_t),    /**< [In] Specifies the size of the data parameter structure */
            &KeyStore);                             /**< [In] Pointer to a key store structure (can be null).*/
        CHECK_SUCCESS(status);

        status = phCryptoSym_Sw_Init(&CryptoSymRnd, /**< [In] Pointer to this layer's parameter structure. */
            sizeof(phCryptoSym_Sw_DataParams_t),    /**< [In] Specifies the size of the data parameter structure */
            &KeyStore);                             /**< [In] Pointer to a key store structure (can be null).*/
        CHECK_SUCCESS(status);

        status = phCryptoRng_Sw_Init(&CryptoRng,    /**< [In] Pointer to this layer's parameter structure. */
            sizeof(phCryptoRng_Sw_DataParams_t),    /**< [In] Specifies the size of the data parameter structure */
            &CryptoSymRnd);                         /**< [In] Pointer to the parameter structure of the symmetric crypto layer. */
        CHECK_SUCCESS(status);

#endif /* NXPBUILD__PH_CRYPTOSYM */

        /* Activate Layer 3 card. In loop till a card is detected. */
        status = phpalI14443p3a_ActivateCard(&I14443p3a_Sw,   /**< [In] PAL-ISO14443P3A parameter structure */
            NULL,                                             /**< [In] Known Uid */
            0x00,                                             /**< [In] Known Uid length */
            pUid,                                             /**< [Out] Complete Uid */
            &bLenUid,                                         /**< [Out] Lenght Uid **/
            &bSak,                                            /**< [out] SAK */
            &bMoreCardsAvaliable);                            /**< [Out] Whether there are more cards in the field or not */
        CHECK_SUCCESS(status);

        /* Send RATS */
        status = phpalI14443p4a_Rats(&I14443p4a_Sw,   /**< [In] PAL-ISO14443P4A parameter structure */
            0x08,                                     /**< [In] Frame size integer */
            0x01,                                     /**< [In] Card Identifier */
            pAts);                                    /**< [Out] Answer to select */
        CHECK_SUCCESS(status);

        /* Send PPS */
        status = phpalI14443p4a_Pps(&I14443p4a_Sw,   /**< [In] PAL-ISO14443P4A parameter structure */
            0x00,                                    /**< [In] Divisor Receive (PCD to PICC) Integer */
            0x00);                                   /**< [In] Divisor Send (PICC to PCD) Integer */
        CHECK_SUCCESS(status);

        status = phpalI14443p4a_GetProtocolParams(&I14443p4a_Sw,   /**< [In] PAL-ISO14443P4A parameter structure */
            &bCidEnable,                                           /**< [Out] Card identifier Enable Specifier*/
            &bCid,                                                 /**< [Out] Card identifier */
            &bNadSupported,                                        /**< [Out] Node Address support Variable */
            &bFwi,                                                 /**< [Out] Frame waiting Integer */
            &bFsdi,                                                /**< [Out] PCD frame size integer */
            &bFsci);                                               /**< [Out] PICC frame size integer */
        CHECK_SUCCESS(status);

        status = phpalI14443p4_SetProtocol(&I14443p4_Sw,   /**< [In] PAL-ISO14443P4 parameter structure */
            bCidEnable,                                    /**< [In] Card identifier Enable Specifier*/
            bCid,                                          /**< [In] Card identifier */
            bNadSupported,                                 /**< [In] Node Address support Variable */
            0x00,                                          /**< [In] Node Address */
            bFwi,                                          /**< [In] Frame waiting Integer */
            bFsdi,                                         /**< [In] PCD frame size integer */
            bFsci);                                        /**< [In] PICC frame size integer */
        CHECK_SUCCESS(status);

#ifndef NXPBUILD__PH_CRYPTOSYM

        /* Initialize the MF DesFire EV1 component */
        status = phalMfdf_Sw_Init(&alMfdf,      /**< [In] Pointer to this layer's parameter structure */
            sizeof(phalMfdf_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
            &palMifare_Sw,                      /**< [In] Pointer to a palMifare component context */
            NULL,                               /**< [In] Pointer to Key Store data parameters */
            NULL,                               /**< [In] Pointer to a Crypto component context for encryption */
            NULL,                               /**< [In] Pointer to a CryptoRng component context */
            &halRd710);                         /**< [In] Pointer to the HAL parameters structure */
        CHECK_SUCCESS(status);

#endif /* NXPBUILD__PH_CRYPTOSYM */

#ifdef NXPBUILD__PH_CRYPTOSYM

        /* Initialize the MF DesFire EV1 component */
        status = phalMfdf_Sw_Init(&alMfdf,      /**< [In] Pointer to this layer's parameter structure */
            sizeof(phalMfdf_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
            &palMifare_Sw,                      /**< [In] Pointer to a palMifare component context */
            &KeyStore,                          /**< [In] Pointer to Key Store data parameters */
            &CryptoEnc,                         /**< [In] Pointer to a Crypto component context for encryption */
            &CryptoRng,                         /**< [In] Pointer to a CryptoRng component context */
            &halRd710);                         /**< [In] Pointer to the HAL parameters structure */
        CHECK_SUCCESS(status);

#endif /* NXPBUILD__PH_CRYPTOSYM */

        /* Create Application */
        status = phalMfdf_CreateApplication(&alMfdf,   /**< [In] Pointer to parameters data structure */
            0x03,                                      /**< [In] Option to indicate whether this application has ISO Fid and DF */
            aAID,                                      /**< [In] 3 byte AID, LSB First */
            0x0F,                                      /**< [In] Key Settings 1 - 1 Byte */
            0x21,                                      /**< [In] Key Settings 2 - 1 Byte */
            aISOFileId_A,                              /**< [In] Two byte ISO File Id, LSB First */
            aISODFName,                                /**< [In] ISO DF Name. Maximum 16 bytes */
            bDFLen);                                   /**< [In] Length of DF Name provided above */
        CHECK_SUCCESS(status);

        /* Select the AFC Application */
        status = phalMfdf_SelectApplication(&alMfdf,   /**< [In] Pointer to parameters data structure */
            aAID);                                     /**< [In] 3 byte AID, LSB First */
        CHECK_SUCCESS(status);

        /* Create Std File for CC */
        status = phalMfdf_CreateStdDataFile(&alMfdf,   /**< [In] Pointer to parameters data structure */
            0x01,                                      /**< [In] Option Parameter */
            0x01,                                      /**< [In] File No. */
            aISOFileId_F,                              /**< [In] ISO File Id */
            PHAL_MFDF_COMMUNICATION_PLAIN,             /**< [In] Communication option */
            aAccessRights,                             /**< [In] Access Rights */
            aFileSize);                                /**< [In] File size array */
        CHECK_SUCCESS(status);

        /* Now to create a file with ISO file ID 0xE104 */
        aISOFileId_F[0] = 0x04;

        /* Create Std File for NDEF */
        status = phalMfdf_CreateStdDataFile(&alMfdf,   /**< [In] Pointer to parameters data structure */
            0x01,                                      /**< [In] Option Parameter */
            0x02,                                      /**< [In] File No. */
            aISOFileId_F,                              /**< [In] ISO File Id */
            PHAL_MFDF_COMMUNICATION_PLAIN,             /**< [In] Communication option */
            aAccessRights,                             /**< [In] Access Rights */
            aFileSize);                                /**< [In] File size array */
        CHECK_SUCCESS(status);

        /* Write data to CC file */
        status = phalMfdf_WriteData(&alMfdf,   /**< [In] Pointer to parameters data structure */
            PHAL_MFDF_COMMUNICATION_PLAIN,     /**< [In] Communication option */
            0x01,                              /**< [In] File No. */
            aOffset,                           /**< [In] 3 bytes offset */
            aData,                             /**< [In] Data buffer */
            aDataLen);                         /**< [In] Length of the data */
        CHECK_SUCCESS(status);

        /* Now to Write NDEF msg in File No.2 */
        aDataLen[0] = 0x0E;
        aNDEF_msg[4] = 0x00;

        /* Write data to NDEF file */
        status = phalMfdf_WriteData(&alMfdf,   /**< [In] Pointer to parameters data structure */
            PHAL_MFDF_COMMUNICATION_PLAIN,     /**< [In] Communication option */
            0x02,                              /**< [In] File No. */
            aOffset,                           /**< [In] 3 bytes offset */
            &aNDEF_msg[4],                     /**< [In] Data buffer */
            aDataLen);                         /**< [In] Length of the data */
        CHECK_SUCCESS(status);
    }
    else
    {
        /* Invalid Choice */
        printf("Invalid selection\n");
        return 0;
    }

    status = phhalHw_FieldReset(&halRd710);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
    CHECK_SUCCESS(status);

    status = phbalReg_ClosePort(&balPcsc);    /**< [In] Pointer to the parameter structure of the underlying BAL layer. */
    CHECK_SUCCESS(status);

    printf("NFC tag written successfully\n");
    printf("Press any key to exit ....\n");
    _getch();

    /* Returns zero if the procedure was successful */
    return 0;
}
