/*
 * Copyright 2015 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for MIFARE DESFire EV1 / EV2 using PCSC Reader.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/* Header for this file. */
#include "Example-Pcsc_Mfdf.h"

#undef NXPBUILD__PH_LOG			/* Disabling Log. To enable logging, comment this line. */

#ifdef NXPBUILD__PH_LOG
	phLog_RegisterEntry_t logRegisterEntries[16];
	phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

int __cdecl main()
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint32_t dwChoice = 0;
	uint32_t dwChoice_Valid = PH_OFF;

	uint8_t aAtr[256];
	uint8_t aTx_HalBuffer[512];
	uint8_t aRx_HalBuffer[512];

	/* Initialize software keystore component. */
	wStatus = phKeyStore_Sw_Init ( &stKeyStore, sizeof ( stKeyStore ), aKeyEntry, 0xFF, aKeyVersion, 0x01, aKeyUsageCounter, 0xFF );
	CHECK_SUCCESS ( wStatus );

	/* Initialize CryptoSym for encryption. */
	wStatus = phCryptoSym_Sw_Init ( &stCryptoEnc, sizeof ( stCryptoEnc ), NULL );
	CHECK_SUCCESS ( wStatus );

	/* Initialize CryptoSym for random number generation. */
	wStatus = phCryptoSym_Sw_Init ( &stCryptoSymRng, sizeof ( stCryptoSymRng ), NULL );
	CHECK_SUCCESS ( wStatus );

	/* Initialize Crypto for random number generation. */
	wStatus = phCryptoRng_Sw_Init ( &stCryptoRng, sizeof ( stCryptoRng ), &stCryptoSymRng );
	CHECK_SUCCESS ( wStatus );

	/* Initialize the reader BAL component. */
	wStatus = phbalReg_PcscWin_Init ( &stBal_Pcsc, sizeof ( stBal_Pcsc ), aAtr, 256 );
	CHECK_SUCCESS ( wStatus );

	/* Set the BAL Communication Configuration values */
	wStatus = phbalReg_SetConfig ( &stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL, ( PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T0 | PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T1 ) );
	CHECK_SUCCESS ( wStatus );

	wStatus = phbalReg_SetConfig ( &stBal_Pcsc, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_SHARED );
	CHECK_SUCCESS ( wStatus );

	printf ( "Please ensure that Reader is connected and MIFARE DESFire EV1 or EV2 card is placed on the reader.\n" );
	printf ( "Select any one of the PCSC Readers mentioned below \n" );
	printf ( "Press 1 for %s.\n", OMNIKEY_CL_READER_PORT );
	printf ( "Press 2 for %s.\n", IDENTIV_CL_READER_PORT );
	printf ( "Press 3 for %s.\n", PR533_CL_READER_PORT );
	printf ( "Press 4 for %s.\n", REINERSCT_CL_READER_PORT );
	printf ( "Press x To Quit.\n\n" );

	do
	{
		dwChoice = _getch ();
		switch(dwChoice)
		{
			case '1':
				dwChoice_Valid = PH_ON;
				wStatus = phbalReg_SetPort ( &stBal_Pcsc, ( uint8_t * ) OMNIKEY_CL_READER_PORT );
				break;

			case '2':
				dwChoice_Valid = PH_ON;
				wStatus = phbalReg_SetPort ( &stBal_Pcsc, ( uint8_t *) IDENTIV_CL_READER_PORT );
				break;

			case '3' :
				dwChoice_Valid = PH_ON;
				wStatus = phbalReg_SetPort ( &stBal_Pcsc, ( uint8_t * ) PR533_CL_READER_PORT );
				break;

			case '4':
				dwChoice_Valid = PH_ON;
				wStatus = phbalReg_SetPort ( &stBal_Pcsc, ( uint8_t * ) REINERSCT_CL_READER_PORT );
				break;

			case 'x':
			case 'X':
				exit ( 0 );

			default:
				printf ( "Invalid option selected.\n" );
				break;
		}
		CHECK_SUCCESS ( wStatus );

	}while( !dwChoice_Valid );

	/* Connect to th Reader. */
	wStatus = phbalReg_OpenPort ( &stBal_Pcsc );
	CHECK_SUCCESS ( wStatus );
	printf ( "Successfully Connected with the reader.\n\n" );

	printf ( "Build up the library stack... " );
	/* Initialize HAL component. */
	wStatus = phhalHw_Pcsc_Init ( &stHal_Pcsc, sizeof ( stHal_Pcsc ), &stBal_Pcsc, &stKeyStore, aTx_HalBuffer, sizeof ( aTx_HalBuffer ),
		aRx_HalBuffer, sizeof ( aRx_HalBuffer ) );
	CHECK_SUCCESS ( wStatus );

	/* Initialize ISO14443-3A component. */
	wStatus = phpalI14443p3a_Pcsc_Init ( &stI14443p3a, sizeof ( stI14443p3a ), &stHal_Pcsc );
	CHECK_SUCCESS ( wStatus );

	/* Initialize ISO14443-4A component. */
	wStatus = phpalI14443p4a_Pcsc_Init ( &stI14443p4a, sizeof ( stI14443p4a ), &stHal_Pcsc );
	CHECK_SUCCESS ( wStatus );

	/* Initialize ISO14443-4 generic component. */
	wStatus = phpalI14443p4_Pcsc_Init ( &stI14443p4, sizeof ( stI14443p4 ), &stHal_Pcsc );
	CHECK_SUCCESS ( wStatus );

	/* Initialize Mifare component. */
	wStatus = phpalMifare_Pcsc_Init ( &stMifare, sizeof ( stMifare ), &stHal_Pcsc, &stI14443p4 );
	CHECK_SUCCESS ( wStatus );

	/* Initialize logging */
#ifdef NXPBUILD__PH_LOG
    /* Register the component for logging */
	wStatus = phLog_Init ( Log, logRegisterEntries, ( uint16_t ) ( sizeof ( logRegisterEntries ) / sizeof ( phLog_RegisterEntry_t ) ) );
	CHECK_SUCCESS ( wStatus );

    wStatus = phLog_Register ( &stMfdf, logEntries, ( uint16_t ) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) );
    CHECK_SUCCESS ( wStatus );
#endif /* NXPBUILD__PH_LOG */

	/* Initilize Mfdf component. */
	wStatus = phalMfdf_Sw_Init ( &stMfdf, sizeof ( stMfdf ), &stMifare, &stKeyStore, &stCryptoEnc, &stCryptoRng, &stHal_Pcsc );
	CHECK_SUCCESS ( wStatus );

	printf ( "Success\n\n" );

	do
	{
		/* Clear the console. */
		system ( "cls" );

		/* Perform RfReset. */
		wStatus = phbalReg_ClosePort ( &stBal_Pcsc );
		CHECK_SUCCESS ( wStatus );
		wStatus = phbalReg_OpenPort ( &stBal_Pcsc );
		CHECK_SUCCESS ( wStatus );

		/* Reset the authentication state. */
		wStatus = phalMfdf_ResetAuthStatus ( &stMfdf );
		CHECK_SUCCESS ( wStatus );

		printf ( "Press 1 for Native Commands using AL component.\n" );
		printf ( "Press 2 for Iso7816 Wrapped Commands using AL component.\n" );
		printf ( "Press 3 for Standard Iso7816 Commands using AL component.\n" );
		printf ( "Press x to exit the demonstration. \n" );

		printf ( "\n\n" );
		printf ( "NOTE: Make sure the PICC is formated.\n" );

		dwChoice = _getch ();
		printf ( "\n" );

		switch ( dwChoice )
		{
			case '1':
				printf ( "Native Commands using AL component -------------------------------------------------------\n" );

				wStatus = ExecuteDemo ( dwChoice, COMMAND_FORMAT_NATIVE );
				CHECK_SUCCESS ( wStatus );
				break;

			case '2':
				printf ( "Iso7816 Wrapped Commands using AL component ----------------------------------------------\n" );

				wStatus = ExecuteDemo ( dwChoice, COMMAND_FORMAT_WRAPPED );
				CHECK_SUCCESS ( wStatus );
				break;

			case '3':
				printf ( "Standard Iso7816 Commands using AL component ---------------------------------------------\n" );

				wStatus = ExecuteDemo ( dwChoice, COMMAND_FORMAT_STANDARD_ISO7816 );
				CHECK_SUCCESS ( wStatus );
				break;

			case 'x':
			case 'X':
				exit ( 0 );

			default:
				printf ( "\n\n" );
				printf ( "Invalid option selected.\n" );
				break;
		}

		printf ( "Press any key to continue with the demonstration.\n" );
		printf ( "Press x to exit the demonstration.\n" );
		dwChoice = _getch ();
	}while( ( dwChoice != 'x' ) && ( dwChoice != 'X' ) );

	/* Returns zero if the procedure was successful. */
    return 0;
}

void PrintData ( uint8_t* pBuffer, uint32_t dwLength, char* pFormat, char* pSpecialChar )
{
    uint32_t dwIndex;

    for ( dwIndex = 0; dwIndex < dwLength; ++dwIndex )
        printf ( pFormat, pBuffer[dwIndex] );

    printf ( pSpecialChar );
}

phStatus_t CreateApplication ( uint32_t dwChoice, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t * pAid = NULL;
	uint8_t * pIsoFileID = NULL;
	uint8_t * pIsoDFName = NULL;

	/* Set Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;
	if ( bMode == COMMAND_FORMAT_WRAPPED )
		stMfdf.bWrappedMode = PH_ON;

	/* Create Application. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tCreate Application           : " );
#endif /* NXPBUILD__PH_LOG */

	pAid = ( dwChoice == '1' ) ? aAID_1 : ( dwChoice == '2' ) ? aAID_2 : ( dwChoice == '3' ) ? aAID_3 : NULL;
	pIsoFileID = ( dwChoice == '1' ) ? aIsoFileID_App_1 : ( dwChoice == '2' ) ? aIsoFileID_App_2 : ( dwChoice == '3' ) ? aIsoFileID_App_3 : NULL;
	pIsoDFName = ( dwChoice == '1' ) ? aIsoDFName_1 : ( dwChoice == '2' ) ? aIsoDFName_2 : ( dwChoice == '3' ) ? aIsoDFName_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_CreateApplication ( &stMfdf, PHAL_MFDF_ISO_FILE_ID_DF_NAME_AVAILABLE, pAid, 0x0F, 0x25, pIsoFileID,
		pIsoDFName, 4 ) );

#ifndef NXPBUILD__PH_LOG
	printf ( "SUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	/* Clear Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;

	return wStatus;
}

phStatus_t SelectApplication ( uint32_t dwChoice, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t * pAid = NULL;

	/* Set Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;
	if ( bMode == COMMAND_FORMAT_WRAPPED )
		stMfdf.bWrappedMode = PH_ON;

	/* Select Application. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tSelect Application           : " );
#endif /* NXPBUILD__PH_LOG */

	pAid = ( dwChoice == '1' ) ? aAID_1 : ( dwChoice == '2' ) ? aAID_2 : ( dwChoice == '3' ) ? aAID_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_SelectApplication ( &stMfdf, pAid ) );

#ifndef NXPBUILD__PH_LOG
	printf ( "SUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	/* Clear Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;

	return wStatus;
}

phStatus_t CreateFile ( uint32_t dwChoice, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t bFileID = 0;
	uint8_t * pIsoFileID = NULL;

	/* Set Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;
	if ( bMode == COMMAND_FORMAT_WRAPPED )
		stMfdf.bWrappedMode = PH_ON;

	/* Create File. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tCreateFile                   : " );
#endif /* NXPBUILD__PH_LOG */

	bFileID = ( dwChoice == '1' ) ? bFileID_1 : ( dwChoice == '2' ) ? bFileID_2 : ( dwChoice == '3' ) ? bFileID_3 : NULL;
	pIsoFileID = ( dwChoice == '1' ) ? aIsoFileID_File_1 : ( dwChoice == '2' ) ? aIsoFileID_File_2 : ( dwChoice == '3' ) ? aIsoFileID_File_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_CreateStdDataFile ( &stMfdf, PHAL_MFDF_ISO_FILE_ID_AVAILABLE, bFileID, pIsoFileID,
		PHAL_MFDF_COMMUNICATION_PLAIN, aAccessRights, aFileSize ) );

#ifndef NXPBUILD__PH_LOG
	printf ( "SUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	/* Clear Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;

	return wStatus;
}

phStatus_t WriteData ( uint32_t dwChoice, uint8_t bReset, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t bFileID = 0;
	uint8_t aTx_Data_Tmp[16];

	/* Clear the temp buffer. */
	memset ( aTx_Data_Tmp, 0x00, sizeof ( aTx_Data_Tmp ) );

	/* Set the Data */
	memcpy ( aTx_Data_Tmp, aTx_Data, sizeof ( aTx_Data_Tmp ) );

	/* Set Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;
	if ( bMode == COMMAND_FORMAT_WRAPPED )
		stMfdf.bWrappedMode = PH_ON;

	/* Write Data. */
#ifndef NXPBUILD__PH_LOG
	if ( !bReset )
		printf ( "\tWriteData                    : " );
#endif /* NXPBUILD__PH_LOG */

	bFileID = ( dwChoice == '1' ) ? bFileID_1 : ( dwChoice == '2' ) ? bFileID_2 : ( dwChoice == '3' ) ? bFileID_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_WriteData ( &stMfdf, PHAL_MFDF_COMMUNICATION_PLAIN, bFileID, aOffset, aTx_Data_Tmp, aLength ) );

#ifndef NXPBUILD__PH_LOG
	if ( !bReset )
	{
		printf ( "Data = " );
		PrintData ( aTx_Data_Tmp, sizeof ( aTx_Data_Tmp ), "%02X ", "" );
		printf ( "\tSUCCESS\n" );
	}
#endif /* NXPBUILD__PH_LOG */

	/* Clear Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;

	return wStatus;
}

phStatus_t ReadData ( uint32_t dwChoice, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t bFileID = 0;
	uint16_t wRxLen = 0;

	/* Set Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;
	if ( bMode == COMMAND_FORMAT_WRAPPED )
		stMfdf.bWrappedMode = PH_ON;

	/* Read Data. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tReadData                     : " );
#endif /* NXPBUILD__PH_LOG */

	bFileID = ( dwChoice == '1' ) ? bFileID_1 : ( dwChoice == '2' ) ? bFileID_2 : ( dwChoice == '3' ) ? bFileID_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_ReadData ( &stMfdf, PHAL_MFDF_COMMUNICATION_PLAIN, bFileID, aOffset, aLength, &pRx_Data, &wRxLen ) );

#ifndef NXPBUILD__PH_LOG
	printf ( "Data = " );
	PrintData ( pRx_Data, wRxLen, "%02X ", "" );
	printf ( "\tSUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	/* Clear Wrapped mode. */
	stMfdf.bWrappedMode = PH_OFF;

	return wStatus;
}

phStatus_t IsoSelectFile ( uint32_t dwChoice )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t * pIsoFileID_App = NULL;
	uint8_t * pIsoFileID_File = NULL;
	uint8_t * pIsoDFName = NULL;

	/* Select Application. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tIsoSelectFile (Application)  : " );
#endif /* NXPBUILD__PH_LOG */

	pIsoFileID_File = ( dwChoice == '1' ) ? aIsoFileID_App_1 : ( dwChoice == '2' ) ? aIsoFileID_App_2 : ( dwChoice == '3' ) ? aIsoFileID_App_3 : NULL;
	pIsoDFName = ( dwChoice == '1' ) ? aIsoDFName_1 : ( dwChoice == '2' ) ? aIsoDFName_2 : ( dwChoice == '3' ) ? aIsoDFName_3 : NULL;
	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_IsoSelectFile ( &stMfdf, PHAL_MFDF_FCI_NOT_RETURNED, PHAL_MFDF_SELECTOR_4, pIsoFileID_App,
		pIsoDFName, 4, &pFCI, &wFCILen ) );

#ifndef NXPBUILD__PH_LOG
	printf ( "SUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	/* Select File. */
	if ( wStatus == PH_ERR_SUCCESS )
	{
#ifndef NXPBUILD__PH_LOG
		printf ( "\tIsoSelectFile (File)         : " );
#endif /* NXPBUILD__PH_LOG */

		pIsoFileID_File = ( dwChoice == '1' ) ? aIsoFileID_File_1 : ( dwChoice == '2' ) ? aIsoFileID_File_2 : ( dwChoice == '3' ) ? aIsoFileID_File_3 : NULL;
		PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_IsoSelectFile ( &stMfdf, PHAL_MFDF_FCI_NOT_RETURNED, PHAL_MFDF_SELECTOR_0, pIsoFileID_File,
			NULL, 0, &pFCI, &wFCILen ) );
		CHECK_SUCCESS ( wStatus );

#ifndef NXPBUILD__PH_LOG
		printf ( "SUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */
	}

	return wStatus;
}

phStatus_t IsoUpdateBinary ( uint8_t bReset )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint8_t aTx_Data_Tmp[16];

	/* Clear the temp buffer. */
	memset ( aTx_Data_Tmp, 0x00, sizeof ( aTx_Data_Tmp ) );

	/* Set the Data */
	memcpy ( aTx_Data_Tmp, aTx_Data, sizeof ( aTx_Data_Tmp ) );

	/* Write Data. */
#ifndef NXPBUILD__PH_LOG
	if ( !bReset )
		printf ( "\tIsoUpdateBinary              : " );
#endif /* NXPBUILD__PH_LOG */

	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_IsoUpdateBinary ( &stMfdf, 0x00, 0x00, aTx_Data_Tmp, ( uint8_t ) sizeof ( aTx_Data_Tmp ) ) );

#ifndef NXPBUILD__PH_LOG
	if ( !bReset )
	{
		printf ( "Data = " );
		PrintData ( aTx_Data_Tmp, sizeof ( aTx_Data_Tmp ), "%02X ", "" );
		printf ( "\tSUCCESS\n" );
	}
#endif /* NXPBUILD__PH_LOG */

	return wStatus;
}

phStatus_t IsoReadBinary ()
{
	phStatus_t wStatus = PH_ERR_SUCCESS;
	uint16_t wRxLen = 0;

	/* Read Data. */
#ifndef NXPBUILD__PH_LOG
	printf ( "\tIsoReadBinary                : " );
#endif /* NXPBUILD__PH_LOG */

	PH_CHECK_SUCCESS_FCT ( wStatus, phalMfdf_IsoReadBinary ( &stMfdf, PH_EXCHANGE_DEFAULT, 0x00, 0x00, 16, &pRx_Data, &wRxLen ) );
	printf ( "Data = " );
	PrintData ( pRx_Data, wRxLen, "%02X ", "" );

#ifndef NXPBUILD__PH_LOG
	printf ( "\tSUCCESS\n" );
#endif /* NXPBUILD__PH_LOG */

	return wStatus;
}

phStatus_t ExecuteDemo ( uint32_t dwChoice, uint8_t bMode )
{
	phStatus_t wStatus = PH_ERR_SUCCESS;

	/* Create Application. */
	PH_CHECK_SUCCESS_FCT ( wStatus, CreateApplication ( dwChoice, bMode ) );

	/* Select Application. */
	PH_CHECK_SUCCESS_FCT ( wStatus, SelectApplication ( dwChoice, bMode ) );

	/* Create File. */
	PH_CHECK_SUCCESS_FCT ( wStatus, CreateFile ( dwChoice, bMode ) );
	printf ( "\n" );

	/* Native / Wrapped Commands */
	if ( bMode != COMMAND_FORMAT_STANDARD_ISO7816 )
	{
		/* ReadData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, ReadData ( dwChoice, bMode ) );

		/* WriteData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, WriteData ( dwChoice, PH_OFF, bMode ) );

		/* ReadData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, ReadData ( dwChoice, bMode ) );

		/* WriteData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, WriteData ( dwChoice, PH_ON, bMode ) );
	}
	else
	{
		/* Perform RfReset. */
		wStatus = phbalReg_ClosePort ( &stBal_Pcsc );
		CHECK_SUCCESS ( wStatus );
		wStatus = phbalReg_OpenPort ( &stBal_Pcsc );
		CHECK_SUCCESS ( wStatus );

		/* Select Application and File. */
		PH_CHECK_SUCCESS_FCT ( wStatus, IsoSelectFile ( dwChoice ) );

		/* ReadData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, IsoReadBinary () );

		/* WriteData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, IsoUpdateBinary ( PH_OFF ) );

		/* ReadData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, IsoReadBinary () );

		/* WriteData. */
		PH_CHECK_SUCCESS_FCT ( wStatus, IsoUpdateBinary ( PH_ON ) );
	}

	printf ( "\n" );

	return wStatus;
}

#ifdef NXPBUILD__PH_LOG
void Log ( void* pDataParams, uint8_t bOption, phLog_LogEntry_t* pEntries, uint16_t wEntryCount )
{
    uint16_t wIteration;
	PH_UNUSED_VARIABLE ( bOption );
    PH_UNUSED_VARIABLE ( pDataParams );

    for ( wIteration = 0; wIteration < wEntryCount; wIteration++ )
    {
        if ( wIteration != 0 )
        {
            printf ( "%s : ", pEntries[wIteration].pString );
        }
        else
        {
            printf ( "-----------------------------------------\n" );
            printf ( "%s\n", pEntries[0].pString );
            printf ( "-----------------------------------------\n" );
        }
		if ( pEntries[wIteration].wDataLen != 0 )
			PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "\t" );
		else
			PrintData ( ( uint8_t * ) ( pEntries[wIteration].pData ), pEntries[wIteration].wDataLen, "%02X ", "" );
    }
    printf("\n\n");
}
#endif  /* NXPBUILD__PH_LOG */
