/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (Secure Access Module) internal implementation via TDA interface for Reader Library
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 08. Feb 2024
 *
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHBAL_REG_SAM

#include "phbalReg_Sam_TDA.h"
#include "../phbalReg_Sam_Int.h"

/* Supported Fi/Di values */
static const uint8_t PH_MEMLOC_CONST_ROM PH_MEMLOC_CONST_ROM gaphbalReg_Sam_SupportedFiDiValues[] =
{
    0x11,
    0x12,
    0x13,
    0x18,
    0x95,
    0x96
};

phStatus_t phbalReg_Sam_Int_Exchange_TDA(phbalReg_Sam_DataParams_t * pDataParams, uint8_t bCommType, uint8_t bCmd, uint8_t * pData,
    uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wOption = 0;
    uint16_t    PH_MEMLOC_REM wCmd = 0;
    uint16_t    PH_MEMLOC_REM wTotRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRemRespLen = 0;
    uint16_t    PH_MEMLOC_REM wTxLen = 0;
    uint16_t    PH_MEMLOC_REM wReadLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bRetries = 0U;
    uint8_t     PH_MEMLOC_REM bExchangeTx = PH_ON;
    uint8_t     PH_MEMLOC_REM bSplitRead = PH_OFF;

    /* Clear Transmit and Receive buffer */
    (void) memset(pDataParams->pTxBuffer, 0x00U, pDataParams->wTxBufSize);
    (void) memset(pDataParams->pRxBuffer, 0x00U, pDataParams->wRxBufSize);

    /* Add command to TX buffer. */
    pDataParams->pTxBuffer[wTxLen++] = bCommType;
    pDataParams->pTxBuffer[wTxLen++] = bCmd;

    /* Add additional info to TxBuffer based on command. */
    switch(bCmd)
    {
        case PHBAL_SAM_CONFIG_RD_OPS_SET_PCSC_MODE:
            pDataParams->pTxBuffer[wTxLen++] = 0x01U;
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;
            pDataParams->pTxBuffer[wTxLen++] = 0x01U;
            break;

        case PHBAL_SAM_CMD_ACTIVATE:
        case PHBAL_SAM_CMD_DEACTIVATE:
        case PHBAL_SAM_CMD_COLD_RESET:
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;
            break;

        case PHBAL_SAM_CMD_SEND_PPS:
            pDataParams->pTxBuffer[wTxLen++] = 0x03U;
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;
            pDataParams->pTxBuffer[wTxLen++] = 0x00U;
            pDataParams->pTxBuffer[wTxLen++] = 0x11U;
            break;

        case PHBAL_SAM_CMD_TRANSMIT_DATA:
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) (wDataLen & 0xFF);
            pDataParams->pTxBuffer[wTxLen++] = (uint8_t) (wDataLen >> 8U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    /* Add data if applicable. */
    (void) memcpy(&pDataParams->pTxBuffer[wTxLen], pData, wDataLen);
    wTxLen += wDataLen;

    /* Frame the actual command. */
    wCmd = (uint16_t) ((bCommType << 8U) | bCmd);

    /* Set Option info to be provided for lower BAL. */
    wOption = (uint16_t) PH_EXCHANGE_DEFAULT;

    /* Update information based on Lower BAL. */
    if(PH_GET_COMPID(pDataParams->pLowerBalDataParams) == PHBAL_REG_SERIALWIN_ID)
    {
        wOption |= (uint16_t) PHBAL_REG_SERIALWIN_SUPRESS_CHECKS;

        /* Read the Header Information. */
        wRemRespLen = PHBAL_SAM_FRAME_HEADER_LEN;

        /* Set Loop Termination flag. */
        bSplitRead = PH_ON;
    }
    else
    {
        /* Read complete information. */
        wRemRespLen = pDataParams->wRxBufSize;

        /* Set Loop Termination flag. */
        bSplitRead = PH_OFF;
    }

    do
    {
        wStatus = phbalReg_Exchange(
            pDataParams->pLowerBalDataParams,
            wOption,
            bExchangeTx ? pDataParams->pTxBuffer : NULL,
            (uint16_t) (bExchangeTx ? wTxLen : 0U),
            wRemRespLen,
            &pDataParams->pRxBuffer[wTotRespLen],
            &wRspLen);

        wTotRespLen += wRspLen;
        wRemRespLen -= wRspLen;
        bExchangeTx = PH_OFF;

        /* Break the loop in case of error and max retries reached. */
        bRetries += (uint8_t) ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS);
        if((bRetries == 10U) || (bSplitRead == PH_OFF))
            break;

    } while(wTotRespLen != PHBAL_SAM_FRAME_HEADER_LEN);

    /* Update the remaining bytes to read. */
    wRemRespLen = (uint16_t) (pDataParams->pRxBuffer[4] | (pDataParams->pRxBuffer[5] << 8U));
    wReadLen = wRemRespLen;

    /* Read the payload is available. */
    if((wRemRespLen != 0U) && (bSplitRead == PH_ON))
    {
        do
        {
            wStatus = phbalReg_Exchange(
                pDataParams->pLowerBalDataParams,
                wOption,
                NULL,
                0U,
                wReadLen,
                &pDataParams->pRxBuffer[wTotRespLen],
                &wRspLen);

            wTotRespLen += wRspLen;
            wReadLen -= wRspLen;
        } while(((wTotRespLen - PHBAL_SAM_FRAME_HEADER_LEN) != wRemRespLen) && (wReadLen > 0U));
    }

    /* Check response */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_CheckResponse(
        wCmd,
        pDataParams->pRxBuffer,
        wTotRespLen,
        NULL,
        &wRspLen));

    /* Move the response to parameters. */
    *ppResponse = &pDataParams->pRxBuffer[PHBAL_SAM_FRAME_HEADER_LEN];
    *pRspLen = wRspLen;

    return wStatus;
}

phStatus_t phbalReg_Sam_ActivateSam_TDA(phbalReg_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t *   PH_MEMLOC_REM pAtr = NULL;
    uint16_t    PH_MEMLOC_REM wAtrLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Set PCSC mode in case if Pegoda - 2 reader is configured for PCSC
     * mode of communication.
     */
    if(PH_GET_COMPID(pDataParams->pLowerBalDataParams) == PHBAL_REG_PCSCWIN_ID)
    {
        /* Perform Mode change */
        PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_TDA(
            pDataParams,
            PHBAL_SAM_CMD_READER_OPERATION,
            PHBAL_SAM_CONFIG_RD_OPS_SET_PCSC_MODE,
            NULL,
            0U,
            &pResponse,
            &wRspLen));
    }
    else
    {
        /* PCSC Mode setting is not required. */
    }

    /* Perform Activate Contact Card */
    wStatus = phbalReg_Sam_Int_Exchange_TDA(
        pDataParams,
        PHBAL_SAM_CMD_COMMUNICATION_TDA,
        PHBAL_SAM_CMD_ACTIVATE,
        NULL,
        0U,
        &pAtr,
        &wAtrLen);

    /* Perform Cold Reset in case if Activate Fails. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_TDA(
            pDataParams,
            PHBAL_SAM_CMD_COMMUNICATION_TDA,
            PHBAL_SAM_CMD_COLD_RESET,
            NULL,
            0U,
            &pAtr,
            &wAtrLen));
    }

    /* Store ATR */
    if(wAtrLen < pDataParams->wMaxAtrBufSize)
    {
        (void) memcpy(pDataParams->pAtrBuffer, pAtr, wAtrLen);
        pDataParams->wAtrBufSize = wAtrLen;
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    return wStatus;
}

phStatus_t phbalReg_Sam_Pps_TDA(phbalReg_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;

    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bPps1 = 0;
    uint8_t     PH_MEMLOC_REM bSpecificMode = 0;
    uint8_t     PH_MEMLOC_REM bDoPps = 0;
    uint8_t     PH_MEMLOC_REM * pResponse = NULL;

    /* Set PCSC mode in case if Pegoda - 2 reader is configured. */
    if(PH_GET_COMPID(pDataParams->pLowerBalDataParams) == PHBAL_REG_PCSCWIN_ID)
    {
        /* Parse ATR */
        PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_ParseAtr(pDataParams->pAtrBuffer, pDataParams->wAtrBufSize,
            &bPps1, &bSpecificMode));

        if((!bSpecificMode) && (bPps1 != 0x11))
        {
            /* Check if Fi/Di values are supported */
            for(bIndex = 0; bIndex < sizeof(gaphbalReg_Sam_SupportedFiDiValues); ++bIndex)
            {
                if(gaphbalReg_Sam_SupportedFiDiValues[bIndex] == bPps1)
                {
                    bDoPps = 1;
                    break;
                }
            }
        }

        /* Perform PPS if necessary */
        if(bDoPps)
        {
            /* Perform PPS */
            PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_TDA(
                pDataParams,
                PHBAL_SAM_CMD_COMMUNICATION_TDA,
                PHBAL_SAM_CMD_SEND_PPS,
                &bPps1,
                1U,
                &pResponse,
                &wRspLen));
        }
    }
    else
    {
        /* PCSC Mode setting is not required. */
    }

    wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);

    return wStatus;
}

phStatus_t phbalReg_Sam_DeActivateSam_TDA(phbalReg_Sam_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Perform De-Activate Contact Card */
    wStatus = phbalReg_Sam_Int_Exchange_TDA(
        pDataParams,
        PHBAL_SAM_CMD_COMMUNICATION_TDA,
        PHBAL_SAM_CMD_DEACTIVATE,
        NULL,
        0U,
        &pResponse,
        &wRspLen);

    /* Payload length should be 0 */
    if(wRspLen != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_BAL);
    }
    else
    {
        /* Forcing status to SUCCESS as post De-Activate, there will be no response from reader. */
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
    }

    return wStatus;
}

phStatus_t phbalReg_Sam_TransmitData_TDA(phbalReg_Sam_DataParams_t * pDataParams, uint8_t * pTxBuffer, uint16_t wTxBufLen,
    uint16_t wRxBufSize, uint8_t * pRxBuffer, uint16_t * pRxBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;

    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Perform command exchange */
    PH_CHECK_SUCCESS_FCT(wStatus, phbalReg_Sam_Int_Exchange_TDA(
        pDataParams,
        PHBAL_SAM_CMD_COMMUNICATION_TDA,
        PHBAL_SAM_CMD_TRANSMIT_DATA,
        pTxBuffer,
        wTxBufLen,
        &pResponse,
        &wRspLen));

    /* Check if RxBuffer is big enough */
    if(wRxBufSize < wRspLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
    }

    /* Copy received contents */
    (void) memcpy(pRxBuffer, pResponse, wRspLen);
    *pRxBufLen = wRspLen;

    return wStatus;
}

#endif /* NXPBUILD__PHBAL_REG_SAM */
