/*
 * Copyright 2013 - 2014, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * TMIUtils Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phTMIUtils.h>

#ifdef NXPBUILD__PH_TMIUTILS

#include <ph_Status.h>
#include <ph_RefDefs.h>

phStatus_t phTMIUtils_Init(phTMIUtils_t * pDataParams, uint8_t * pTMIBuffer, uint32_t dwBufLen)
{
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_TMIUTILS);
	PH_ASSERT_NULL_PARAM(pTMIBuffer, PH_COMP_TMIUTILS);

	/* Check parameters */
	if (dwBufLen == 0)
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);

	pDataParams->pTMIBuffer = pTMIBuffer;
	pDataParams->dwTMIBufLen = dwBufLen;
	pDataParams->dwTMIbufIndex = 0;
	pDataParams->bTMIStatus = PH_OFF;
	pDataParams->dwOffsetInTMI = 0;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

phStatus_t phTMIUtils_ActivateTMICollection(phTMIUtils_t *pDataParams, uint8_t bOption)
{
	if ((pDataParams->dwTMIBufLen == 0) || (pDataParams == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);
	}

	switch (bOption)
	{
		/* Deactivate TMI Collection */
		case PH_TMIUTILS_DEACTIVATE_TMI:
			pDataParams->dwTMIBufLen = 0;
			pDataParams->dwOffsetInTMI = 0;
			pDataParams->bTMIStatus = PH_OFF;
			break;

		/* Activate/Resume TMI collection */
		case PH_TMIUTILS_ACTIVATE_TMI:
		case PH_TMIUTILS_RESUME_TMI:
			pDataParams->bTMIStatus = PH_ON;
			break;

		/* Pause TMI collection */
		case PH_TMIUTILS_PAUSE_TMI:
			pDataParams->bTMIStatus = PH_OFF;
			break;

		/* Reset TMI collection buffer index to 0 */
		case PH_TMIUTILS_RESET_TMI:
			pDataParams->dwTMIbufIndex = 0;
			pDataParams->dwOffsetInTMI = 0;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);

	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

phStatus_t phTMIUtils_GetTMI(phTMIUtils_t * pDataParams, uint8_t ** ppTMIBuffer, uint32_t * dwTMILen)
{
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_TMIUTILS);

	*ppTMIBuffer = pDataParams->pTMIBuffer;
	*dwTMILen = pDataParams->dwTMIbufIndex;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

phStatus_t phTMIUtils_CollectTMI(phTMIUtils_t * pDataParams, uint8_t bOption, uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t * pData,
	uint32_t dwDataLen, uint16_t wBlockSize)
{
	uint32_t    PH_MEMLOC_REM dwTmp = 0;

	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_TMIUTILS);

	/* Parameter validation */
    if ((pCmdBuff == NULL) && (pData == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);
    }

	if ((pDataParams->dwTMIbufIndex + wCmdLen + (wBlockSize - (wCmdLen % wBlockSize)) + dwDataLen + (wBlockSize - (dwDataLen % wBlockSize))) > pDataParams->dwTMIBufLen)
	{
		return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_TMIUTILS);
	}

	if (wCmdLen)
	{
		/* Copy the command buffer */
		memcpy(&pDataParams->pTMIBuffer[pDataParams->dwTMIbufIndex], pCmdBuff, wCmdLen); /* PRQA S 3200 */
		pDataParams->dwTMIbufIndex += wCmdLen;

		if ((bOption & PH_TMIUTILS_ZEROPAD_CMDBUFF) && (pDataParams->dwTMIbufIndex % wBlockSize))
		{
			/* Zero padding */
			memset(&pDataParams->pTMIBuffer[pDataParams->dwTMIbufIndex], 0x00, (wBlockSize - (pDataParams->dwTMIbufIndex % wBlockSize))); /* PRQA S 3200 */
			pDataParams->dwTMIbufIndex += (wBlockSize - (pDataParams->dwTMIbufIndex % wBlockSize));
		}
	}
	if (bOption & PH_TMIUTILS_READ_INS)
	{
		if (pDataParams->dwOffsetInTMI)
		{
			/* Update the Length field Offset in pDataParams */
			dwTmp = pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI + 1];
			dwTmp <<= 8;
			dwTmp |= pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI];
			dwTmp += dwDataLen;
			pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI] = (uint8_t) dwTmp;
			pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI + 1] = (uint8_t) (dwTmp >> 8);
		}
		else
		{
			/* Store the Length field Offset in pDataParams */
			pDataParams->dwOffsetInTMI = (pDataParams->dwTMIbufIndex - 11);
		}
	}

	if (dwDataLen)
	{
		/* Copy the command buffer */
		memcpy(&pDataParams->pTMIBuffer[pDataParams->dwTMIbufIndex], pData, dwDataLen); /* PRQA S 3200 */
		pDataParams->dwTMIbufIndex += dwDataLen;

		if ((bOption & PH_TMIUTILS_ZEROPAD_DATABUFF) && (pDataParams->dwTMIbufIndex % wBlockSize))
		{
			/* Zero padding */
			memset(&pDataParams->pTMIBuffer[pDataParams->dwTMIbufIndex], 0x00, (wBlockSize - (pDataParams->dwTMIbufIndex % wBlockSize))); /* PRQA S 3200 */
			pDataParams->dwTMIbufIndex += (wBlockSize - (pDataParams->dwTMIbufIndex % wBlockSize));
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

phStatus_t phTMIUtils_SetConfig(phTMIUtils_t * pDataParams, uint16_t wConfig, uint32_t dwValue)
{
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_TMIUTILS);

	switch (wConfig)
	{
		case PH_TMIUTILS_TMI_OFFSET_LENGTH:
			pDataParams->dwOffsetInTMI = dwValue;
			break;

		case PH_TMIUTILS_TMI_OFFSET_VALUE:
			pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI] = (uint8_t) dwValue;
			pDataParams->pTMIBuffer[pDataParams->dwOffsetInTMI + 1] = (uint8_t) (dwValue >> 8);
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);
			break;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

phStatus_t phTMIUtils_GetConfig(phTMIUtils_t * pDataParams, uint16_t wConfig, uint32_t *pValue)
{
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_TMIUTILS);

	switch (wConfig)
	{
		case PH_TMIUTILS_TMI_STATUS:
			*pValue = pDataParams->bTMIStatus;
			break;

		case PH_TMIUTILS_TMI_OFFSET_LENGTH:
			*pValue = pDataParams->dwOffsetInTMI;
			break;

		case PH_TMIUTILS_TMI_BUFFER_INDEX:
			*pValue = pDataParams->dwTMIbufIndex;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TMIUTILS);
			break;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TMIUTILS);
}

#endif /* NXPBUILD__PH_TMIUTILS */
