/*
 * Copyright 2013, 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Stepper Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLSTEPPER_H
#define PHDLSTEPPER_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_STEPPER_V1

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_V1 Component : V1
* @{
*/

#define PH_COMP_DL_STEPPER_V1_ID    0x01U   /**< ID for Hardware Stepper V1 Architecture layer. */

/**
* \brief phdlStepper_V1 parameter structure
*/
typedef struct
{
    uint16_t wId;                   /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;       /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bInit;                 /**< Indicates the state of the connection */
    uint8_t bState;                /**< Current state parameter */
    uint16_t wRamp_delay;           /**< Indicates the number of ms, one single speed is kept during the ramp. */
    uint8_t bRamp_on_of;            /**< Indicates, if an acceleration ramp is used (0 = off, 1 = start & stor ramp). */
    uint8_t bDirection;             /**< Indicates the direction of the stepper (0 = down, 1 = up). */
    uint16_t wStart_Speed;          /**< Indicates the direction of the stepper (0 = down, 1 = up) */
    uint16_t wMax_Speed;            /**< Maximal speed of the ramp. */
    uint16_t wSteps_Way_Ratio;      /**< Ratio between steps of the motor and moved mm of the stepper. */
    uint16_t wDistance;             /**< Distance in mm the stepper move in DRIVE_DISTANCE mode. */
    uint8_t bMaxCurrent;            /**< Indicates the maximal current in %. */
    uint8_t bStdby_Current;         /**< Indicates the standby motor hold current in %. */
    uint8_t	 bStdby_Current_Flag;   /**< Indicates if motor current is on during non-drive. */
    uint32_t dwMax_Position;        /**< Indicates the maximal reachable position of the stepper. */
}phdlStepper_V1_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_V1_Init(
                               phdlStepper_V1_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams,                  /**< [In] Specifies the size of the data parameter structure */
                               void * pBalRegDataParams                     /**< [In] Pointer to the parameter structure of the balReg component. */
                               );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_STEPPER_V1 */

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_Robot Component : Robot
* @{
*/

#define PH_COMP_DL_STEPPER_ROBOT_ID    0x02U   /**< ID for Hardware Stepper V1 Architecture layer. */

/**
* \brief phdlStepper_Robot parameter structure
*/
typedef struct
{
    uint16_t wId;                   /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;       /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bInit;                 /**< Indicates the state of the connection */
    uint16_t wMaxSpeed;            /**< Maximal speed of the ramp. */
    uint16_t wStepsWayRatio;      /**< Ratio between steps of the motor and moved mm of the stepper. */
    uint32_t dwMaxPosition;        /**< Indicates the maximal reachable position of the stepper. */
    uint16_t wMaxGetPositionRetrys; /**< How often the get Position loop is called if an error occours */
    uint16_t wWaitPositionTimeout;  /**< Number of loops used if position limit not is used */
}phdlStepper_Robot_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_Robot_Init(
                               phdlStepper_Robot_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams,                  /**< [In] Specifies the size of the data parameter structure */
                               void * pBalRegDataParams                     /**< [In] Pointer to the parameter structure of the balReg component. */
                               );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_MICROBOT

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_Microbot Component : Microbot
* @{
*/

#define PH_COMP_DL_STEPPER_MICROBOT_ID    0x03U   /**< ID for Hardware Stepper Microbot Architecture layer. */

/**
* \brief phdlStepper_Microbot parameter structure
*/
typedef struct
{
    uint16_t wId;                  /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;      /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bInit;                 /**< Indicates the state of the connection */
    uint16_t wStepsWayRatio;       /**< Ratio between steps of the motor and moved mm of the stepper. */
    uint8_t bPositionMode;         /**< How the position is set */
    uint8_t bActiveAxis;           /**< Which Axis is used with the set Position functions */
    int32_t dwPositionX;          /**< Current X Position. */
    int32_t dwPositionY;          /**< Current Y Position. */
    int32_t dwPositionZ;          /**< Current Z Position. */
    int16_t wAngle;          /**< Current Angle Position. */
}phdlStepper_Microbot_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_Microbot_Init(
                               phdlStepper_Microbot_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams,                  /**< [In] Specifies the size of the data parameter structure */
                               void * pBalRegDataParams                     /**< [In] Pointer to the parameter structure of the balReg component. */
                               );

/** @} \endcond */

/**
* \name Microbot Stepper Configs
*/
/*@{*/
#define PHDL_STEPPER_MICROBOT_CONFIG_POSITION_MODE	(PH_CONFIG_CUSTOM_BEGIN) /**< number of milliseconds one single speed is kept during the ramp. */
#define PHDL_STEPPER_MICROBOT_CONFIG_ACTIVE_AXIS 	(PH_CONFIG_CUSTOM_BEGIN+1) /**< Axis used in Set Commands. */
/*@}*/

/**
* \name Microbot Stepper Config32
*/
/*@{*/
#define PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_X   (PH_CONFIG_CUSTOM_BEGIN+2) /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Y   (PH_CONFIG_CUSTOM_BEGIN+3) /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_MICROBOT_CONFIG32_POSITION_Z   (PH_CONFIG_CUSTOM_BEGIN+4) /**< indicates the numbers of steps the carriage is away from a defined zero point. */
/*@}*/

/**
* \name Position Mode Config Values
*/
/*@{*/
enum PositionMode
{
	PHDL_STEPPER_MICROBOT_POS_MODE_SET_CURRENT_AS_ZERO = 0x01U,   /**< Sets the current positon as Zero Position */
	PHDL_STEPPER_MICROBOT_POS_MODE_REMOTE = 0x02U,                /**< The Position is done with remote command */
	PHDL_STEPPER_MICROBOT_POS_MODE_LOCAL = 0x03U,                 /**< Position is done with buttons on Mircobot */
	PHDL_STEPPER_MICROBOT_POS_MODE_COLLECT = 0x04U,               /**< Position set with SetConfig32 are only stored but is not send to the robot */
	PHDL_STEPPER_MICROBOT_POS_MODE_START = 0x05U,                 /**< The robot moves to the actual stored position and the Mode is set to REMOTE */
    PHDL_STEPPER_MICROBOT_POS_MODE_START_NONBLOCKING = 0x06U      /**< The robot moves to the actual stored position and the Mode is set to REMOTE with non-blocking flag */
};
/*@}*/

/**
* \name Active Axis Values
*/
/*@{*/
enum ActiveAxis
{
	PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_X = 0x01U,   /**< Axis X is used as default */
	PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Y = 0x02U,   /**< Axis Y is used as default */
	PHDL_STEPPER_MICROBOT_ACTIVE_AXIS_Z = 0x03U    /**< Axis Z is used as default */
};
/*@}*/

#endif /* NXPBUILD__PHDL_STEPPER_MICROBOT */

#ifdef NXPBUILD__PHDL_STEPPER_HIGHZ

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_HighZ Component : HighZ
* @{
*/

#define PH_COMP_DL_STEPPER_HIGHZ_ID    0x06U   /**< ID for Hardware Stepper HighZ Architecture layer. */

/**
* \brief phdlStepperHighZ parameter structure
*/
typedef struct
{
    uint16_t wId;                  /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;      /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bInit;                 /**< Indicates the state of the connection */
    uint8_t bPowerIsOn;            /**< Indicates if the engines are powered */
    uint16_t wMicrometerPerTurn[3];/**< Ratio between steps of the motor and moved um of the stepper. */
    uint16_t wStepsPerTurn[3];     /**< Ratio between steps of the motor and moved um of the stepper. */

    int32_t dwMaxRangeX;           /**< Max value in um for X axis. */
    int32_t dwMaxRangeY;           /**< Max value in um for Y axis. */
    int32_t dwMaxRangeZ;           /**< Max value in um for Z axis. */

    uint16_t wRampDelayMin;         /**< Min delay for movement. */
    uint16_t wRampDelayMax;         /**< Max delay for movement. */
    uint16_t wRampSpeed;            /**< Speed of the acceleration ramp. */
    uint8_t bRampEnabled;           /**< Flag if acceleration is enabled */

    int32_t dwPaddingX;             /**< ep_switches_offset for X axis */
    int32_t dwPaddingY;             /**< ep_switches_offset for Y axis */
    int32_t dwPaddingZ;             /**< ep_switches_offset for Z axis */

    uint32_t dwDriveTimeout;        /**< maximum time that the drives is allowed to be done, depends on delay and #steps */

    uint8_t bPositionMode;         /**< How the position is set */
    uint8_t bActiveAxis;           /**< Which Axis is used with the set Position functions */
    int32_t dwPositionX;          /**< Current X Position used for collect mode. */
    int32_t dwPositionY;          /**< Current Y Position used for collect mode. */
    int32_t dwPositionZ;          /**< Current Z Position used for collect mode. */
}phdlStepper_HighZ_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_HighZ_Init(
                               phdlStepper_HighZ_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams,                  /**< [In] Specifies the size of the data parameter structure */
                               void * pBalRegDataParams                     /**< [In] Pointer to the parameter structure of the balReg component. */
                               );

/** @} \endcond */

/**
* \name HighZ Stepper Configs
*/
/*@{*/
#define PHDL_STEPPER_HIGHZ_CONFIG_POSITION_MODE	      (PH_CONFIG_CUSTOM_BEGIN)    /**< mode used for set multiple axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_ACTIVE_AXIS 	      (PH_CONFIG_CUSTOM_BEGIN+1)  /**< Axis used in Set Commands. */
#define PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MIN      (PH_CONFIG_CUSTOM_BEGIN+11) /**< Delay between steps for full speed. */
#define PHDL_STEPPER_HIGHZ_CONFIG_RAMP_DELAY_MAX      (PH_CONFIG_CUSTOM_BEGIN+12) /**< Delay between steps for min speed. */
#define PHDL_STEPPER_HIGHZ_CONFIG_ROBOT_SPEED         (PH_CONFIG_CUSTOM_BEGIN+13) /**< Speed of the robot, value is then calculated to delay, not all values are possible. */
#define PHDL_STEPPER_HIGHZ_CONFIG_RAMP_SPEED          (PH_CONFIG_CUSTOM_BEGIN+14) /**< Speed of the acceleration ramp. */
#define PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_X (PH_CONFIG_CUSTOM_BEGIN+15) /**< Distance in um that is performed per full turn of the robot for x axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Y (PH_CONFIG_CUSTOM_BEGIN+16) /**< Distance in um that is performed per full turn of the robot for y axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_DISTANCE_PER_TURN_Z (PH_CONFIG_CUSTOM_BEGIN+17) /**< Distance in um that is performed per full turn of the robot for z axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_X    (PH_CONFIG_CUSTOM_BEGIN+18) /**< Steps of the engine to perform a full turn of the x axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Y    (PH_CONFIG_CUSTOM_BEGIN+19) /**< Steps of the engine to perform a full turn of the y axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG_STEPS_PER_TURN_Z    (PH_CONFIG_CUSTOM_BEGIN+20) /**< Steps of the engine to perform a full turn of the z axis. */

/*@}*/

/**
* \name HighZ Stepper Config32
*/
/*@{*/
#define PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_X   (PH_CONFIG_CUSTOM_BEGIN+2) /**< indicates the position in um the carriage is away from a defined zero point. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Y   (PH_CONFIG_CUSTOM_BEGIN+3) /**< indicates the position in um the carriage is away from a defined zero point. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_POSITION_Z   (PH_CONFIG_CUSTOM_BEGIN+4) /**< indicates the position in um the carriage is away from a defined zero point. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_MAX_X		 (PH_CONFIG_CUSTOM_BEGIN+5) /**< max position for the x axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Y		 (PH_CONFIG_CUSTOM_BEGIN+6) /**< max position for the y axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_MAX_Z		 (PH_CONFIG_CUSTOM_BEGIN+7) /**< max position for the z axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_X	 (PH_CONFIG_CUSTOM_BEGIN+8) /**< padding for the x axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Y	 (PH_CONFIG_CUSTOM_BEGIN+9) /**< padding for the y axis. */
#define PHDL_STEPPER_HIGHZ_CONFIG32_PADDING_Z	 (PH_CONFIG_CUSTOM_BEGIN+10)/**< padding for the z axis. */

/*@}*/

/**
* \name Position Mode Config Values
*/
/*@{*/
enum PositionModeHighZ
{
	PHDL_STEPPER_HIGHZ_POS_MODE_SET_CURRENT_AS_ZERO = 0x01U,   /**< Sets the current positon as Zero Position */
	PHDL_STEPPER_HIGHZ_POS_MODE_CLEAR_CURRENT_ZERO = 0x07U,    /**< Clears the current Zero Position */
	PHDL_STEPPER_HIGHZ_POS_MODE_REMOTE = 0x02U,                /**< The Position is done with remote command */
	PHDL_STEPPER_HIGHZ_POS_MODE_COLLECT = 0x04U,               /**< Position set with SetConfig32 are only stored but is not send to the robot */
	PHDL_STEPPER_HIGHZ_POS_MODE_START = 0x05U,                 /**< The robot moves to the actual stored position and the Mode is set to REMOTE */
};
/*@}*/

/**
* \name Active Axis Values
*/
/*@{*/
enum ActiveAxisHighZ
{
	PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_X = 0x01U,   /**< Axis X is used as default */
	PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Y = 0x02U,   /**< Axis Y is used as default */
	PHDL_STEPPER_HIGHZ_ACTIVE_AXIS_Z = 0x03U    /**< Axis Z is used as default */
};
/*@}*/

#endif /* NXPBUILD__PHDL_STEPPER_HIGHZ */

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_DensoVS60 Component : DensoVS60
* @{
*/

#define PH_COMP_DL_STEPPER_DENSO_VS60_ID    0x04U   /**< ID for Hardware Stepper Denso VS 60 Architecture layer. */
#define DENSO_VS60_MAX_IP_LENGTH    16              /**< Maximum number of chars in the ip adresse string */
#define DENSO_VS60_IO_DEVICE_MAX_NUMBER   40        /**< The maximum number of possible IO devices ports */

/**
* \brief phdlStepper_DensoVS60 parameter structure for offset of the X-Y axis cooridnates
*/
typedef struct
{
    double dwPositionX;          /**< Current X Position offset.      */
    double dwPositionY;          /**< Current Y Position offset.      */
    double dwPositionZ;          /**< Current Z Position offset.      */
    double dwPositionRx;         /**< Current RX Position offset.     */
    double dwPositionRy;         /**< Current RY Position offset.     */
    double dwPositionRz;         /**< Current RZ Position offset.     */
    double wFigure;              /**< Current Figure Position offset. */

}phdlStepper_DensoVS60_DataParams_Offset_XY_coord_t;

/**
* \brief phdlStepper_DensoVS60 parameter structure for the X-Y axis coordinates
*/
typedef struct
{
    int32_t dwPositionX;          /**< Current X Position      */
    int32_t dwPositionY;          /**< Current Y Position      */
    int32_t dwPositionZ;          /**< Current Z Position      */
    int32_t dwPositionRx;         /**< Current RX Position     */
    int32_t dwPositionRy;         /**< Current RY Position     */
    int32_t dwPositionRz;         /**< Current RZ Position     */
    int16_t wFigure;              /**< Current Figure Position */
    phdlStepper_DensoVS60_DataParams_Offset_XY_coord_t offsetCoords;   /**< Void pointer to the array of robt arm offset values. */

}phdlStepper_DensoVS60_DataParams_XY_coord_t;

/**
* \brief phdlStepper_DensoVS60 parameter structure for offset of the Joint axis coordinates
*/
typedef struct
{
    double dwPositionJ1;         /**< Current J1 Position offset. */
    double dwPositionJ2;         /**< Current J2 Position offset. */
    double dwPositionJ3;         /**< Current J3 Position offset. */
    double dwPositionJ4;         /**< Current J4 Position offset. */
    double dwPositionJ5;         /**< Current J5 Position offset. */
    double dwPositionJ6;         /**< Current J6 Position offset. */
    double dwPositionJ7;         /**< Current J7 Position offset. */
    double dwPositionJ8;         /**< Current J8 Position offset. */

}phdlStepper_DensoVS60_DataParams_Offset_Joint_coord_t;

/**
* \brief phdlStepper_DensoVS60 parameter structure for the Joint axis cooridnates
*/
typedef struct
{
    int32_t dwPositionJ1;         /**< Current J1 Position. */
    int32_t dwPositionJ2;         /**< Current J2 Position. */
    int32_t dwPositionJ3;         /**< Current J3 Position. */
    int32_t dwPositionJ4;         /**< Current J4 Position. */
    int32_t dwPositionJ5;         /**< Current J5 Position. */
    int32_t dwPositionJ6;         /**< Current J6 Position. */
    int32_t dwPositionJ7;         /**< Current J7 Position. */
    int32_t dwPositionJ8;         /**< Current J8 Position. */
    phdlStepper_DensoVS60_DataParams_Offset_Joint_coord_t offsetCoords;

}phdlStepper_DensoVS60_DataParams_Joint_coord_t;


/**
* \brief phdlStepper_DensoVS60 parameter structure
*/
typedef struct
{
    uint16_t wId;                  /**< Layer ID for this component, NEVER MODIFY! */
    uint8_t  bInit;                /**< Indicates the state of the connection */
    uint16_t wStepsWayRatio;       /**< Ratio between steps of the motor and moved mm of the stepper. */
    uint8_t  bPositionMode;        /**< How the position is set */
    uint8_t  bPositionModeCorrect; /**< Set 0 if collect mode takes correct axis, 1 otherwise */
    uint8_t  bActiveAxisType;      /**< Which Axis type is set as active 0 - XY, 1 - Joint */
    uint8_t  bActiveAxis;          /**< Which Axis is used with the set Position functions */
    uint8_t  bArmTaken;            /**< Arm control flag: 1 - taken, 0 -free */
    uint8_t  bActiveRotation;      /**< Active rotation control flag: 1 - taken, 0 -free */
    uint16_t wRobotSpeed;          /**< Robot arm speed */
    uint8_t  bRobotAcceeration;    /**< Robot arm acceleration */
    uint8_t  pPortName[DENSO_VS60_MAX_IP_LENGTH];        /**< Robot arm ip adresse pointer */
    void * pEngine;                /**< Void pointer to the ICaoEngine object of the Denso Robot. */
    void * pContoller;             /**< Void pointer to the ICaoContoler object of the Denso Robot. */
    void * pWorkspaces;            /**< Void pointer to the ICaoWorkspaces object of the Denso Robot. */
    void * pWorkspace;             /**< Void pointer to the ICaoWorkspaces object of the Denso Robot. */
    void * pArm;                   /**< Void pointer to the ICaoRobot object of the Denso Robot. */
    uint64_t pDevicesArray[DENSO_VS60_IO_DEVICE_MAX_NUMBER];    /**< Array of pointers to the IO devices of the robot arm. */
    phdlStepper_DensoVS60_DataParams_XY_coord_t structTargetXY;
    phdlStepper_DensoVS60_DataParams_Joint_coord_t structTargetJoint;

}phdlStepper_DensoVS60_DataParams_t;


/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_DensoVS60_Init(
                               phdlStepper_DensoVS60_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams                          /**< [In] Specifies the size of the data parameter structure */
                               );

/** @} \endcond */

/**
* \name DensoVS60 Stepper Configs
*/
/*@{*/
#define PHDL_STEPPER_DENSO_VS60_CONFIG_POSITION_MODE        (PH_CONFIG_CUSTOM_BEGIN)      /**< number of milliseconds one single speed is kept during the ramp. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_AXIS 	        (PH_CONFIG_CUSTOM_BEGIN+1)    /**< Axis used in Set Commands. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_LASER       (PH_CONFIG_CUSTOM_BEGIN+2)    /**< Turn on/off IO extensions IO10 */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_PUMP        (PH_CONFIG_CUSTOM_BEGIN+3)    /**< Turn on/off IO extensions IO25 */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_CHECK_HEAD_SENSOR    (PH_CONFIG_CUSTOM_BEGIN+4)    /**< Turn on/off IO extensions IO26 */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_ROTATION      (PH_CONFIG_CUSTOM_BEGIN+20)   /**< Turn on/off the rotation of the end point can not be reached over X-Y mode */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_SPEED          (PH_CONFIG_CUSTOM_BEGIN+21)   /**< Robot arm speed */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_ACCELERATION   (PH_CONFIG_CUSTOM_BEGIN+22)   /**< Robot arm acceleration */
#define PHDL_STEPPER_DENSO_VS60_CONFIG_CORRECT_COLLECT      (PH_CONFIG_CUSTOM_BEGIN+23)   /**< Check if the collect mode takes correct axis */
/*@}*/

/**
* \name DensoVS60 Stepper Config32
*/
/*@{*/
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_X   (PH_CONFIG_CUSTOM_BEGIN+5)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Y   (PH_CONFIG_CUSTOM_BEGIN+6)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Z   (PH_CONFIG_CUSTOM_BEGIN+7)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RX  (PH_CONFIG_CUSTOM_BEGIN+8)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RY  (PH_CONFIG_CUSTOM_BEGIN+9) /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RZ  (PH_CONFIG_CUSTOM_BEGIN+10) /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_FIG  (PH_CONFIG_CUSTOM_BEGIN+11) /**< indicates the numbers of steps the carriage is away from a defined zero point. */

#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J1  (PH_CONFIG_CUSTOM_BEGIN+12)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J2  (PH_CONFIG_CUSTOM_BEGIN+13)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J3  (PH_CONFIG_CUSTOM_BEGIN+14)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J4  (PH_CONFIG_CUSTOM_BEGIN+15)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J5  (PH_CONFIG_CUSTOM_BEGIN+16)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J6  (PH_CONFIG_CUSTOM_BEGIN+17)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J7  (PH_CONFIG_CUSTOM_BEGIN+18)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J8  (PH_CONFIG_CUSTOM_BEGIN+19)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */


#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_X   (PH_CONFIG_CUSTOM_BEGIN+20)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Y   (PH_CONFIG_CUSTOM_BEGIN+21)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Z   (PH_CONFIG_CUSTOM_BEGIN+22)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RX  (PH_CONFIG_CUSTOM_BEGIN+23)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RY  (PH_CONFIG_CUSTOM_BEGIN+24)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RZ  (PH_CONFIG_CUSTOM_BEGIN+25)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_FIG  (PH_CONFIG_CUSTOM_BEGIN+26) /**< indicates the numbers of steps the carriage is away from a defined zero point. */

#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J1  (PH_CONFIG_CUSTOM_BEGIN+27)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J2  (PH_CONFIG_CUSTOM_BEGIN+28)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J3  (PH_CONFIG_CUSTOM_BEGIN+29)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J4  (PH_CONFIG_CUSTOM_BEGIN+30)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J5  (PH_CONFIG_CUSTOM_BEGIN+31)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J6  (PH_CONFIG_CUSTOM_BEGIN+32)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J7  (PH_CONFIG_CUSTOM_BEGIN+33)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J8  (PH_CONFIG_CUSTOM_BEGIN+34)  /**< indicates the numbers of steps the carriage is away from a defined zero point. */

/*@}*/

/**
* \name Position Mode Config Values
*/
/*@{*/
enum PositionModeDensoVS60
{
	PHDL_STEPPER_DENSO_VS60_POS_MODE_SET_CURRENT_AS_ZERO = 0x01U,     /**< Sets the current positon as Zero Position */
    PHDL_STEPPER_DENSO_VS60_POS_MODE_SET_CLEAR_ZERO_OFFSET = 0x02U,   /**< Clear the offset and set the zero position to the default value */
	PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE = 0x03U,                  /**< The Position is done with remote command */
	PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL = 0x04U,                   /**< Position is done with buttons on Mircobot */
	PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT = 0x05U,                 /**< Position set with SetConfig32 are only stored but is not send to the robot */
    PHDL_STEPPER_DENSO_VS60_POS_MODE_START = 0x06U,                   /**< The robot moves to the actual stored position and the Mode is set to REMOTE with blocking flag*/
    PHDL_STEPPER_DENSO_VS60_POS_MODE_START_NONBLOCKING = 0x07U,       /**< The robot moves to the actual stored position and the Mode is set to REMOTE with non-blocking flag */
    PHDL_STEPPER_DENSO_VS60_POS_MODE_HALT_MOVE = 0x08U                /**< The robot stops its movement */
};
/*@}*/

/**
* \name Active Axis Values
*/
/*@{*/
enum ActiveAxisDensoVS60
{
	PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_X   = 0x01U,   /**< Axis X  is used as default */
	PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Y   = 0x02U,   /**< Axis Y  is used as default */
	PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z   = 0x03U,   /**< Axis Z  is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RX  = 0x04U,   /**< Axis RX is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RY  = 0x05U,   /**< Axis RY is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RZ  = 0x06U,   /**< Axis RZ is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_FIG = 0x07U,   /**< Figure is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J1  = 0x08U,   /**< Axis J1 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J2  = 0x09U,   /**< Axis J2 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J3  = 0x10U,   /**< Axis J3 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J4  = 0x11U,   /**< Axis J4 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J5  = 0x12U,   /**< Axis J5 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J6  = 0x13U,   /**< Axis J6 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J7  = 0x14U,   /**< Axis J7 is used as default */
    PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J8  = 0x15U    /**< Axis J8 is used as default */
};

/*@}*/

#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */

#ifdef NXPBUILD__PHDL_STEPPER_WACHLER

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper_Wachler Component : Wachler
* @{
*/

#define PH_COMP_DL_STEPPER_WACHLER_ID    0x05U   /**< ID for Hardware Wachler Architecture layer. */

#define PH_COMP_DL_STEPPER_WACHLER_IDN_SIZE 128
#define PH_COMP_DL_STEPPER_WACHLER_DEFAULT_NUM_STEPS 10
#define PH_COMP_DL_STEPPER_WACHLER_MAX_NUM_STEPS 10
#define PH_COMP_DL_STEPPER_WACHLER_DEFAULT_UP_DOWN_LEVEL 4.0f
#define PH_COMP_DL_STEPPER_WACHLER_DEFAULT_POS_LEVEL 2.0f

/**
* \brief phdlStepper_Wachler parameter structure
*/
typedef struct
{
    uint16_t wId;                                       /**< Layer ID for this component, NEVER MODIFY! */
    void * pBalRegDataParams;                           /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bInit;                                      /**< Indicates the state of the connection */
    float fMaxPositionDegree;                           /**< Indicates the maximal reachable position of the wachler. */
    float fMinPositionDegree;                           /**< Indicates the minimum reachable position of the wachler. */
    uint32_t actPosition;                               /**< Actual position */
    uint8_t bNumSteps;                                  /**< Number of steps supported by the wachler */
    float fModeUpDownLevel;                             /**< Level used in Up/Down */
    float fModePosLevel;                                /**< Level used in Pos */
    uint8_t bPwmSpeed;                                  /**< current Speed set with PWM */
    uint8_t bIdn[PH_COMP_DL_STEPPER_WACHLER_IDN_SIZE];  /**< This value stores the idn string of the wachler */
}phdlStepper_Wachler_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlStepper_Wachler_Init(
                               phdlStepper_Wachler_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                               uint16_t wSizeOfDataParams,                  /**< [In] Specifies the size of the data parameter structure */
                               void * pBalRegDataParams                     /**< [In] Pointer to the parameter structure of the balReg component. */
                               );

/**
* \name Position Mode Config Values
*/
/*@{*/
enum PositionModeWachler
{
	PHDL_STEPPER_WACHLER_MODE_RUN = 0x00U,     /**< Moves all the time */
    PHDL_STEPPER_WACHLER_MODE_STOP = 0x01U,    /**< Stop until a new position is set */
	PHDL_STEPPER_WACHLER_MODE_CAL = 0x02U,     /**< Calibrate the min and max values */
	PHDL_STEPPER_WACHLER_MODE_PWMCAL = 0x03U,  /**< Calibrate the PWM values */
	PHDL_STEPPER_WACHLER_MODE_AUTO = 0x04U,    /**< Move up and down and stop at both positions */
	PHDL_STEPPER_WACHLER_MODE_BSTOP = 0x05U,   /**< Move up and down and stop at bottom positions */
	PHDL_STEPPER_WACHLER_MODE_POS = 0x06U,     /**< position was set manual */
};
/*@}*/

/**
* \name Wachler Stepper Configs
*/
/*@{*/
#define PHDL_STEPPER_WACHLER_CONFIG_MODE            (PH_CONFIG_CUSTOM_BEGIN)            /**< Operation mode of the wachler. */
#define PHDL_STEPPER_WACHLER_CONFIG_REMOTE          (PH_CONFIG_CUSTOM_BEGIN+1)          /**< Enable/Disable remote mode. */
#define PHDL_STEPPER_WACHLER_CONFIG_NUMBER_STEPS    (PH_CONFIG_CUSTOM_BEGIN+2)          /**< Number of steps supported by the robot. */
#define PHDL_STEPPER_WACHLER_CONFIG_UP_DOWN_LEVEL   (PH_CONFIG_CUSTOM_BEGIN+3)          /**< Level that is used on up/down mode. (valueDegree = value/100) */
#define PHDL_STEPPER_WACHLER_CONFIG_POS_LEVEL       (PH_CONFIG_CUSTOM_BEGIN+4)          /**< Level that is used on pos mode. (valueDegree = value/100) */
#define PHDL_STEPPER_WACHLER_CONFIG_DELAY           (PH_CONFIG_CUSTOM_BEGIN+5)          /**< Delay used on AUTO and BSTOP Mode */
#define PHDL_STEPPER_WACHLER_CONFIG_POS_MIN         (PH_CONFIG_CUSTOM_BEGIN+6)          /**< Get the minimum position in 100*degree. */
#define PHDL_STEPPER_WACHLER_CONFIG_POS_MAX         (PH_CONFIG_CUSTOM_BEGIN+7)          /**< Get the maximum position in 100*degree. */
#define PHDL_STEPPER_WACHLER_CONFIG_COUNTER         (PH_CONFIG_CUSTOM_BEGIN+8)          /**< If this config is set to 0 the counter is reseted. Any other value is not supported. */
#define PHDL_STEPPER_WACHLER_CONFIG_POSITION_Z      (PH_CONFIG_CUSTOM_BEGIN+9)          /**< Set/Get the current Z point. */
#define PHDL_STEPPER_WACHLER_CONFIG_PWM             (PH_CONFIG_CUSTOM_BEGIN+10)         /**< Enable or disable PWM Mode */
#define PHDL_STEPPER_WACHLER_CONFIG_PWM_SPEED       (PH_CONFIG_CUSTOM_BEGIN+11)         /**< Set PWM speed */
#define PHDL_STEPPER_WACHLER_CONFIG_PWM_CALIBRATION (PH_CONFIG_CUSTOM_BEGIN+12)         /**< Set PWM calibration value for minimum speed. The value must be between 0 and 100 */
/*@}*/

/** @} \endcond */
#endif /* NXPBUILD__PHDL_STEPPER_WACHLER */

#ifdef NXPBUILD__PHDL_STEPPER

/** \cond INCLUDE_INTERNALS \defgroup phdlStepper Stepper
* \brief These Components implement the Stepper.
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHDL_STEPPER_ERR_UPPER_LIMIT_INDICATOR  (PH_ERR_CUSTOM_BEGIN+0) /**< Stepper touch the upper limit indicator. */
#define PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR  (PH_ERR_CUSTOM_BEGIN+1) /**< Stepper touch the lower limit indicator. */
#define PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS      (PH_ERR_CUSTOM_BEGIN+2) /**< If the requested position is out of reach. */
#define PHDL_STEPPER_ERR_DRIVE_FAILURE          (PH_ERR_CUSTOM_BEGIN+3) /**< Stepper Hardware does not work. */
#define PH_STEPPER_DENSO_VS60_ERR_CUSTOMERROR   (PH_ERR_CUSTOM_BEGIN+4) /**< An default error for the Denso VS60 robot*/
/*@}*/

/**
* \name Stepper Configs
*/
/*@{*/
#define PHDL_STEPPER_CONFIG_RAMP_DELAY				0x0000U /**< number of milliseconds one single speed is kept during the ramp. */
#define PHDL_STEPPER_CONFIG_RAMP_ON_OFF				0x0001U /**< indicates if an acceleration ramp is used. */
#define PHDL_STEPPER_CONFIG_DIR						0x0002U /**< indicates the direction the stepper will move. */
#define PHDL_STEPPER_CONFIG_START_SPEED				0x0003U /**< indicates the speed the carriage will begin to move. */
#define PHDL_STEPPER_CONFIG_MAX_SPEED				0x0004U /**< speed of the carriage after the acceleration ramp. */
#define PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO			0x0005U /**< ratio between the steps of the motor and the moved distance (in mm). */
#define PHDL_STEPPER_CONFIG_DISTANCE				0x0006U /**< distance in mm the stepper move in DRIVE_DISTANCE mode. */
#define PHDL_STEPPER_CONFIG_MAX_CURRENT				0x0007U /**< The maximal current indicate the maximal current in %. */
#define PHDL_STEPPER_CONFIG_STDBY_CURRENT			0x0008U /**< Standby current indicates the standby motor hold current in %. */
#define PHDL_STEPPER_CONFIG_STDBY_CURRENT_FLAG		0x0009U /**< The standby current flag indicate if the motor current is switched on while the motor is not driving. */
/*@}*/

/**
* \name Stepper Config32
*/
/*@{*/
#define PHDL_STEPPER_CONFIG32_POSITION_COUNTER      0x0010U /**< indicates the numbers of steps the carriage is away from a defined zero point. */
#define PHDL_STEPPER_CONFIG32_MAX_POSITION			0x0011U /**< indicates the maximum position of the stepper. */
/*@}*/

/**
* \name Stepper Directions
*/
/*@{*/
#define PHDL_STEPPER_DIR_DOWN   0U  /**< direction down. */
#define PHDL_STEPPER_DIR_UP     1U  /**< direction up. */
/*@}*/

/**
* \brief Stepper moves wDistance [um] in bDirection with wSpeed [um/s]. If  bBlocking is 1 it the funktion waits till
the stepper stops.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_MoveDistance(
                                    void * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                    uint16_t wSpeed,    /**< [In] Moving speed. */
                                    uint8_t bDirection, /**< [In] Moving direction. */
                                    uint32_t wDistance, /**< [In] Moving distance. */
                                    uint8_t bBlocking   /**< [In] Indicates (if = 1) if the funktion waits till the movement is complited before it returns. */
                                    );

/**
* \brief Stepper moves wDistance [um] in bDirection with wSpeed [um/s]. If  bBlocking is 1 it the funktion waits till
the stepper stops.
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_MoveSteps(
                                 void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wSpeed,       /**< [In] Moving speed. */
                                 uint8_t bDirection,    /**< [In] Moving direction. */
                                 uint32_t dwSteps,      /**< [In] Moving steps. */
                                 uint8_t bBlocking      /**< [In] Indicates (if = 1) if the funktion waits till the movement is complited before it returns. */
                                 );

/**
* \brief Stepper moves to wPosition [um] (Point of origin is the upper limit indicator) with wSpeed [um/s]. If  bBlocking is 1 it the funktion waits till
the stepper stops.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_GoToPosition(
                                    void * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                                    uint16_t wSpeed,    /**< [In] Moving speed. */
                                    uint32_t wPosition, /**< [In] Desired position. */
                                    uint8_t bBlocking   /**< [In] Indicates if the funktion waits till the movement is complited before it returns. */
                                    );

/**
* \brief Set the 16 bit parameter specified by Config.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_SetConfig(
                                 void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wConfig,      /**< [In] Configuration Identifier */
                                 uint16_t wValue        /**< [In] Configuration Value */
                                 );

/**
* \brief Set the 32 bit parameter specified by Config.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_SetConfig32(
                                   void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wConfig,    /**< [In] Configuration Identifier */
                                   uint32_t dwValue     /**< [In] Configuration Value */
                                   );

/**
* \brief Get the 16- bitparameter specified by bCommand form the Stepper.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_GetConfig(
                                 void * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
                                 uint16_t wConfig,      /**< [In] Configuration Identifier */
                                 uint16_t * pValue      /**< [Out] Return parameter value. */
                                 );

/**
* \brief Get the 32- bitparameter specified by bCommand form the Stepper.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_GetConfig32(
                                   void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                                   uint16_t wConfig,    /**< [In] Configuration Identifier */
                                   uint32_t * pValue    /**< [Out] Return parameter value. */
                                   );

/**
* \brief Gets the position of the upper (POI) and lower (Max_Position) limit indicatior.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlStepper_Initialize(
                                  void * pDataParams    /**< [In] Pointer to this layer's parameter structure. */
                                  );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_STEPPER*/

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLSTEPPER_H */
