/*
 * Copyright 2013, 2017, 2019 - 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Oscilloscope Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLOSCI_H
#define PHDLOSCI_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_OSCI_LW64XI

/** \cond INCLUDE_INTERNALS \defgroup phdlOsci_LW64Xi Component : L64Xi
* @{
*/
#define PHDL_OSCI_LW64XI_ID  0x01U                              /**< ID for Lecroy Waverunner L64Xi device layer. */

#define PHDL_OSCI_LW64XI_SELECT_FUNC_CHANNEL        0x80        /**< Mask to select function channel. */
#define PHDL_OSCI_LW64XI_SET_VDIV_DELAY              400U       /**< The delay in ms after VDIV is changed. */

/**
* \name LeCroy specific configs
*/
/*@{*/
/**
* \brief Specific Settings.
*/
#define PHDL_OSCI_LW64XI_CONFIG_AUTOADJUSTCHANNELS             (PH_CONFIG_CUSTOM_BEGIN+0)      /**< Sets Flag in DataParams to auto adjust channels when initializing the osci */
#define PHDL_OSCI_LW64XI_CONFIG_NOISE_FILTER                   (PH_CONFIG_CUSTOM_BEGIN+1)	   /**< Noise Filter Option */
#define PHDL_OSCI_LW64XI_CONFIG64_WAVEFORM_SAMPLE_SIZE         (PH_CONFIG_CUSTOM_BEGIN+2)	   /**< Number samples that should be transfered when get waveform */
/*@}*/

/**
* \name Osci Noise Filter Settings
*/
/*@{*/
#define PHDL_OSCI_LW64XI_NOISE_FILTER_NONE                 0x0000U /**< Noise Filter None */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_HALF_BIT             0x0001U /**< Noise Filter None */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_ONE_BIT              0x0002U /**< Noise Filter 1 Bit */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_ONEHALF_BIT          0x0003U /**< Noise Filter None */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_TWO_BIT              0x0004U /**< Noise Filter 2 Bit */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_TWOHALF_BIT          0x0005U /**< Noise Filter None */
#define PHDL_OSCI_LW64XI_NOISE_FILTER_THREE_BIT            0x0006U /**< Noise Filter 3 Bit */
/*@}*/

/**
* \brief Oscilloscope Software parameter structure
*/
typedef struct
{
    uint16_t wId;                                       /**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Visa_DataParams_t * pBalRegDataParams;     /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wFieldStrengthMultiplier;                  /**< Defines the conversion coefficient for the field Strengh. */
    uint16_t wAverageFact;                              /**< Defines the number of averages for the measurements. */
    uint16_t wRangeMin;                                 /**< Defines the minimum range for SetCorrRange. */
    uint16_t wRangeMax;                                 /**< Defines the maximum range for SetCorrRange. */
    uint8_t bMeasurementOptions;                        /**< Store additional options for performing measurement. */
    uint8_t bSelectedChannel;                           /**< Defines the channel that is selected to apply settings */
    uint8_t bAutoAdjustChannels;                        /**< Auto Adjusts Channels on InitOsci if set */
    uint32_t dwMaxSamplesMinValue;                      /**< The minimal value that can be set for MaxSamples Config */
    uint32_t dwMaxSamplesMaxValue;                      /**< The maximum value that can be set for MaxSamples Config */
} phdlOsci_LW64Xi_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlOsci_LW64Xi_Init(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,	    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,						/**< [In] Specifies the size of the data parameter structure. */
    phbalReg_Visa_DataParams_t * pBalRegDataParams  /**< [In] Pointer to the parameter structure of the balReg component. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_OSCI_LW64XI */


#ifdef NXPBUILD__PHDL_OSCI_DSO7052A

/** \cond INCLUDE_INTERNALS \defgroup phdlOsci_DSO7052A Component : DSO7052A
* @{
*/
#define PHDL_OSCI_DSO7052A_ID  0x02U                        /**< ID for Agilent DSO7052A device layer. */

#define PHDL_OSCI_DSO7052A_AC   0x01U                       /**< Custom Config AC or DC. AC. */
#define PHDL_OSCI_DSO7052A_DC   0x02U                       /**< Custom Config AC or DC. DC. */

/**
* \name DSO7052A specific configs
*/
/*@{*/
/**
* \brief Specific Settings.
*/
#define PHDL_OSCI_DSO7052A_CONFIG_VOLT_AVERAGE                      (PH_CONFIG_CUSTOM_BEGIN)      /**< Get config hack for Ralph. */
#define PHDL_OSCI_DSO7052A_CONFIG_SAVE_CSV                          (PH_CONFIG_CUSTOM_BEGIN+1)      /**< Save current Image as CSV on USB Stick connected to Oscilloscope. */
/*@}*/

/**
* \brief Oscilloscope Software parameter structure
*/
typedef struct
{
    uint16_t wId;                                       /**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Visa_DataParams_t * pBalRegDataParams;     /**< Pointer to the parameter structure of the balReg component. */
    uint16_t wFieldStrengthMultiplier;                  /**< Defines the conversion coefficient for the field Strengh. */
    uint16_t wCurrentTriggerMode;                       /**< Represents the current Trigger setting in use. */
    uint16_t wAverageFact;                              /**< Defines the number of averages for the measurements. */
    uint16_t wRangeMin;                                 /**< Defines the minimum range for SetCorrRange. */
    uint16_t wRangeMax;                                 /**< Defines the maximum range for SetCorrRange. */
    uint8_t bMeasurementOptions;                        /**< Store additional options for performing measurement. */
    uint8_t bSelectedChannel;                           /**< Defines the channel that is selected for apply settings */
    uint8_t bAcOrDcMeasurementCh1;                      /**< Defines if the following measurements are on AC or DC on Channel 1. */
    uint8_t bAcOrDcMeasurementCh2;                      /**< Defines if the following measurements are on AC or DC on Channel 2. */
    uint8_t bAcOrDcMeasurementCh3;                      /**< Defines if the following measurements are on AC or DC on Channel 3. */
    uint8_t bAcOrDcMeasurementCh4;                      /**< Defines if the following measurements are on AC or DC on Channel 4. */
    uint8_t bNumberOfChannels;                          /**< The number of channels available depending on Model. */
} phdlOsci_DSO7052A_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlOsci_DSO7052A_Init(
    phdlOsci_DSO7052A_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                         /**< [In] Specifies the size of the data parameter structure. */
    phbalReg_Visa_DataParams_t * pBalRegDataParams      /**< [In] Pointer to the parameter structure of the balReg component. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_OSCI_DSO7052A */

#ifdef NXPBUILD__PHDL_OSCI_PICOSCOPE6000

/** \cond INCLUDE_INTERNALS \defgroup phdlOsci_PicoScope6000 Component : PicoScope6000
* @{
*/
#define PHDL_OSCI_PICOSCOPE6000_ID   0x03U                       /**< ID for PicoScope PS6000 device layer. */

#define PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS 5

#define PHDL_OSCI_PICOSCOPE6000_CHANNEL_ENABLED     0x01U        /**< Channel Enabled Value */
#define PHDL_OSCI_PICOSCOPE6000_CHANNEL_DISABLED    0x00U        /**< Channel Disabled Value */


/**
* \brief Oscilloscope Software parameter structure
*/
typedef struct
{
    uint16_t                               wId;                                                     /**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Ps6000Usb_DataParams_t      * pBalRegDataParams;                                       /**< Pointer to the parameter structure of the balReg component. */
    uint16_t                               wAdditionalInfo;                                         /**< Storage for additional error information. */
    uint16_t                               wFieldStrengthMultiplier;                                /**< Defines the conversion coefficient for the field Strengh. */
    uint16_t                               wCurrentTriggerMode;                                     /**< Represents the current Trigger setting in use. */
    uint16_t                               wTriggerAutoTriggerMilliseconds;                         /**< The time in milliseconds after which the trigger fires when trigger condition not reached */
    uint16_t                               wAverageFact;                                            /**< Defines the number of averages for the measurements. */
    uint16_t                               wRangeMin;                                               /**< Defines the minimum range for SetCorrRange. */
    uint16_t                               wRangeMax;                                               /**< Defines the maximum range for SetCorrRange. */
    uint8_t                                bMeasurementOptions;                                     /**< Store additional options for performing measurement. */
    uint32_t                               dwLastSamplingTimebase;                                  /**< Timebase used for the last sampling progress */
    /* Osci derivate parameter */
    uint8_t                                bNumberOfChannels;                                       /**< The number of channels available depending on Model. */
    uint16_t                               wBandwidthMHz;                                           /**< Bandwidth of the device in MHz. */
    uint16_t                               wMaxMemoryMS;                                            /**< Available memory size of the device in Mega Samples. */
    uint16_t                               wSamplingRateMSs;                                        /**< Max Sampling rate of the device in Mega Samples per second. */
    uint8_t                                bAWG;                                                    /**< Is a AWG available for the current device. */
    /* Channel setting */
    uint8_t                                bSelectedChannelIndex;                                   /**< Defines the selected channel index starting with zero */
    uint8_t                                bChannelEnabled[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS];   /**< Defines if a Channel is enabled */
    uint8_t                                bChannelCoupling[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS];  /**< Defines if the channel coupling is AC or DC. */
    uint8_t                                bChannelImpedance[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS]; /**< Defines the impedance of a channel. */
    uint8_t                                bChannelBWLimit[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS];   /**< Defines if the BW Limit is enabled for the channel. */
    uint8_t                                bChannelInvert[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS];    /**< Defines if the channel is inverted */
    uint16_t                               wChannelProbeGain[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS]; /**< Gain of the probe connected to this Channel */
    uint8_t                                bChannelRangeIndex[PHDL_OSCI_PICOSCOPE6000_MAX_CHANNELS];/**< Range index of the Channel */
    /* Time Setting */
    int64_t                                qwTimebaseRangeNs;                                       /**< The full scale timebase that should be used */
    int64_t                                qwTimebasePositionNs;                                    /**< The offset of the timebase that should be used */
    int64_t                                qwSamplePointsTarget;                                    /**< The target number of points, the real number of points could be also
                                                                                                     *   higher if the target could not be reached with the sampling rate and downsampling or
                                                                                                     *   lower if the maximum sampling rate is already used and the time is to short */
    /* Trigger Settings */
    uint8_t                                bTriggerType;                                            /**< current trigger type */
    uint8_t                                bTriggerSourceIndex;                                     /**< current trigger source */
    uint8_t                                bTriggerEdgeSlope;                                       /**< current trigger slope for edge type */
    int64_t                                qwTriggerLevelmV;                                        /**< current trigger level */
} phdlOsci_PicoScope6000_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlOsci_PicoScope6000_Init(
    phdlOsci_PicoScope6000_DataParams_t * pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                           /**< [In] Specifies the size of the data parameter structure. */
    phbalReg_Ps6000Usb_DataParams_t * pBalRegDataParams   /**< [In] Pointer to the parameter structure of the balReg component. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_OSCI_PICOSCOPE6000 */

#ifdef NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE

#define PHDL_OSCI_ISO3SOFTSCOPE_ID                               0x04U      /**< ID for ISO3SoftScope device layer. */

#define PHDL_OSCI_ISO3SOFTSCOPE_MAX_BUFFER_SIZE                 0x40000000  /**< Max buffer size in bytes - In samples: 0x20000000*/
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_BIG_BUFFER              0x0001      /**< 1GB soft scope buffer */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_RING_BUFFER_TRG1        0x0002      /**< 4KB ring Buffer of Trigger 1 */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_RING_BUFFER_TRG2        0x0003      /**< 4KB ring Buffer of Trigger 2 */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_RISE               0x0004      /**< 4KB ring Buffer of Trigger 2 */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_RISE               0x0005      /**< 4KB ring Buffer of Trigger 2 */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG1_FALL               0x0006      /**< 4KB ring Buffer of Trigger 2 */
#define PHDL_OSCI_ISO3SOFTSCOPE_CHANNEL_TRG2_FALL               0x0007      /**< 4KB ring Buffer of Trigger 2 */

#define PHDL_OSCI_ISO3SOFTSCOPE_RING_BUFFER_SAMPLE_COUNT        0x800U      /**< Number of 16-bit addresses reserved for buffer of trigger */
#define PHDL_OSCI_ISO3SOFTSCOPE_SAMPLE_SIZE                     0x2U        /**< Number of bytes of 1 waveshape sample */


/** \name ISO3 Osci Channels
*/
/*@{*/
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_RX                  (PH_CONFIG_CUSTOM_BEGIN) + 0  /**< RX from ADC_IN (FPGA input from ADC) */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_CAL                 (PH_CONFIG_CUSTOM_BEGIN) + 1  /**< CAL from ADC_IN (FPGA input from ADC) */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_COS                 (PH_CONFIG_CUSTOM_BEGIN) + 2  /**< internal cos wave */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_SIN                 (PH_CONFIG_CUSTOM_BEGIN) + 3  /**< internal sinus wave */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_AGC                 (PH_CONFIG_CUSTOM_BEGIN) + 4  /**< AGC gain output */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_IQ                  (PH_CONFIG_CUSTOM_BEGIN) + 5  /**< IQ output from modulator, before gain control */
#define PHDL_OSCI_ISO3SOFTSCOPE_DATA_SOURCE_DAC                 (PH_CONFIG_CUSTOM_BEGIN) + 6  /**< Final input to DAC from FPGA */
/*@}*/

/** \name ISO3 Soft scope configs
*/
/*@{*/
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_DATA_SRC                             (PH_CONFIG_CUSTOM_BEGIN) + 0        /**< Set/Get current data source */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_COND_RISE                  (PH_CONFIG_CUSTOM_BEGIN) + 1        /**< Event condition for trigger 1 rise */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_COND_FALL                  (PH_CONFIG_CUSTOM_BEGIN) + 2        /**< Event condition for trigger 1 fall */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_COND_RISE                  (PH_CONFIG_CUSTOM_BEGIN) + 3        /**< Event condition for trigger 2 rise */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_COND_FALL                  (PH_CONFIG_CUSTOM_BEGIN) + 4        /**< Event condition for trigger 2 fall */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_1_EXTERN_ENABLE              (PH_CONFIG_CUSTOM_BEGIN) + 5        /**< Output trigger 1 to external oscilloscope */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_2_EXTERN_ENABLE              (PH_CONFIG_CUSTOM_BEGIN) + 6        /**< Output trigger 2 to external oscilloscope */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_TRIGGER_SRC               (PH_CONFIG_CUSTOM_BEGIN) + 7        /**< Specify the event which should be written to the big soft scope buffer */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_BIG_BUFFER_SIZE_NUM_SAMPLES          (PH_CONFIG_CUSTOM_BEGIN) + 8        /**< Soft scope buffer size in samples, default = 655kb */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG1_SIZE_NUM_SAMPLES    (PH_CONFIG_CUSTOM_BEGIN) + 9        /**< Ring buffer size for Trigger 1 (in samples) */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_RING_BUFFER_TRG2_SIZE_NUM_SAMPLES    (PH_CONFIG_CUSTOM_BEGIN) + 10       /**< Ring buffer size for Trigger 2 (in samples) */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_RISE_BIT_NUMB                (PH_CONFIG_CUSTOM_BEGIN) + 11       /**< */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIGGER_FALL_BIT_NUMB                (PH_CONFIG_CUSTOM_BEGIN) + 12       /**< */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_HEADER_ENABLED              (PH_CONFIG_CUSTOM_BEGIN) + 13       /**< If enabled a header is included at the beginning of the waveform buffer */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_WAVEFORM_SAMPLE_SIZE                 (PH_CONFIG_CUSTOM_BEGIN) + 14       /**< Softscope Waveform Size in Samples */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_FDT_MEASUREMENT                      (PH_CONFIG_CUSTOM_BEGIN) + 15       /**< Activates FDT measurement */
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG1_SIGNAL_ID                      (PH_CONFIG_CUSTOM_BEGIN  + 16)
#define PHDL_OSCI_ISO3SOFTSCOPE_CONFIG_TRIG2_SIGNAL_ID                      (PH_CONFIG_CUSTOM_BEGIN  + 17)
/*@}*/

/** \name ISO3 Soft scope log targets
*/
/*@{*/
#define PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_MLS                   0x1  /**< Capture when MLS enabled and detected */
#define PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_FORCE                 0x4  /**< Force to capture */
#define PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_1             0x8  /**< Trigger 1 will be logged into the soft scope buffer */
#define PHDL_OSCI_ISO3SOFTSCOPE_LOG_TARGET_TRIGGER_2             0x10 /**< Trigger 2 will be logged into the soft scope buffer */
/*@}*/

/** \name ISO3 Soft scope trigger conditions
*/
/*@{*/
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_FIELD_OFF               0x00U  /**< Trigger event when switching off carrier */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_IDLE                    0x01U  /**< Trigger event when transmitter enters IDLE STATE */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_SOC                     0x02U  /**< Trigger event when transmitter modulates SOC */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_LAST_BIT                0x03U  /**< Trigger event when transmitter modulates last bit of a frame */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_RX_AFTER_LAST_TX_BIT    0x04U  /**< Trigger event when transmitter activates receiver after last TX bit */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_SOC_RX                  0x05U  /**< Trigger event when SOC was detected by receiver */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_TX_LMA_RX_EVENT            0x06U  /**< Trigger event when LMA event (8 subcarrier modulations) was detected */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_FIRST_STATE_AFTER_IDLE  0x09U  /**< Trigger event when receiver changes state after IDLE */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_FIFTH_BIT               0x0AU  /**< Trigger event when receiver receives fifth bit. This Trigger is immune to EMD */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_1ST_MOD                 0x0CU  /**< Trigger event when receiver detected first modulation */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_SOC                     0x0DU  /**< Trigger event when receiver detected SOC */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_BIT_NUM                 0x0EU  /**< Trigger event when received a certain bit (see TRIGGER_CTRL1 */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_EOC                     0x0BU  /**< Trigger event when recieved EOC */
/* #define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_RX_EOC                     0x0FU  *< Trigger event when recieved EOC - Obsolete */
#define PHDL_OSCI_ISO3SOFTSCOPE_TRIGGER_COND_MLS_START                  0x10U  /**< Trigger event when modulator starts to play maximum length sequence */
/*@}*/

/**
* \brief ISO3 SoftScope Software parameter structure
*/
typedef struct
{
    uint16_t                        wId;                                                                /**< Layer ID for this component, NEVER MODIFY! */
    void     *                      pHalISO3DataParams;                                                 /**< Pointer to the parameter structure of the HAL ISO3 component */
    uint32_t                        dwLogCtrlReg;                                                       /**< Register to configure soft scope signals and mls trigger */
    uint32_t                        dwLogSizeReg;                                                       /**< Register to configure the size (number of 16-bit samples) of the soft scope buffer */
    uint32_t                        dwTriggerCtrl0Reg;                                                  /**< Control register to set trigger events */
    uint32_t                        dwTriggerCtrl1Reg;                                                  /**< Register to read number of received bits */
    uint32_t                        pdwRingBuffer[PHDL_OSCI_ISO3SOFTSCOPE_RING_BUFFER_SAMPLE_COUNT];    /**< 32-bit ring buffer array for Trigger 1 and Trigger 2 */
    uint32_t                        dwLastExchangeCounter;                                              /**< Saves last value of HAL ISO3 exchange counter */
    uint8_t                         bWaveFormHeaderEnabled;                                             /**< Indicates whether waveform header is included by GetWaveForm function or not */
    uint32_t                        dwWaveFormSampleSize;                                               /**< Softscope Waveform Size in Samples */
} phdlOsci_ISO3SoftScope_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlOsci_ISO3SoftScope_Init(
    phdlOsci_ISO3SoftScope_DataParams_t * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,                          /**< [In] Specifies the size of the data parameter structure. */
    void * pBalRegDataParams                             /**< [In] Pointer to the parameter structure of the balReg component. */
    );
/** @} \endcond */
#endif /* NXPBUILD__PHDL_OSCI_ISO3SOFTSCOPE */

#ifdef NXPBUILD__PHDL_OSCI

/** \cond INCLUDE_INTERNALS \defgroup phdlOsci Oscilloscope
* \brief These Components implement the Oscilloscope
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHDL_OSCI_ERR_COULD_NOT_CONVERT_DIVISOR_INTO_CMD        (PH_ERR_CUSTOM_BEGIN+0)     /**< Osciloscope could not convert divisor into command error. */
#define PHDL_OSCI_ERR_CORR_RANGE_FAIL                           (PH_ERR_CUSTOM_BEGIN+1)     /**< Osciloscope setting correct range failed error. */
#define PHDL_OSCI_ERR_AMPLITUDE_VAL                             (PH_ERR_CUSTOM_BEGIN+2)     /**< Osciloscope wrong amplitude value measured error. */

/* Command Errors */
#define PHDL_OSCI_ERR_CMD_UNKNOWN_COMMAND                       (PH_ERR_CUSTOM_BEGIN+3)     /**< Osciloscope unknown command error. */
#define PHDL_OSCI_ERR_CMD_ILLEGAL_HEADER_PATH                   (PH_ERR_CUSTOM_BEGIN+4)     /**< Osciloscope illegal header path error. */
#define PHDL_OSCI_ERR_CMD_ILLEGAL_NUMBER                        (PH_ERR_CUSTOM_BEGIN+5)     /**< Osciloscope illegal number error. */
#define PHDL_OSCI_ERR_CMD_ILLEGAL_NUMBER_SUFFIX                 (PH_ERR_CUSTOM_BEGIN+6)     /**< Osciloscope illegal number suffix error. */
#define PHDL_OSCI_ERR_CMD_UNKNOWN_KEYWORD                       (PH_ERR_CUSTOM_BEGIN+7)     /**< Osciloscope unknown keyword error. */
#define PHDL_OSCI_ERR_CMD_STRING_ERROR                          (PH_ERR_CUSTOM_BEGIN+8)     /**< Osciloscope string error. */
#define PHDL_OSCI_ERR_CMD_GET_EMBEDDED_IN_ANOTHER_MESSAGE       (PH_ERR_CUSTOM_BEGIN+9)     /**< Osciloscope get embedded in another message error. */
#define PHDL_OSCI_ERR_CMD_ARBITRARY_DATA_BLOCK_EXPECTED         (PH_ERR_CUSTOM_BEGIN+10)    /**< Osciloscope arbitrary data block expected error. */
#define PHDL_OSCI_ERR_CMD_NONDIGIT_CHAR_IN_BYTE_COUNT_FIELD     (PH_ERR_CUSTOM_BEGIN+11)    /**< Osciloscope nondigit char in bytecount field error. */
#define PHDL_OSCI_ERR_CMD_EOI_DETECTED                          (PH_ERR_CUSTOM_BEGIN+12)    /**< Osciloscope EOI detected error. */
#define PHDL_OSCI_ERR_CMD_EXTRA_BYTES_DETECTED                  (PH_ERR_CUSTOM_BEGIN+13)    /**< Osciloscope extra bytes detected error. */

/* Execution Errors */
#define PHDL_OSCI_ERR_EXE_PERMISSION_ERROR                      (PH_ERR_CUSTOM_BEGIN+14)    /**< Osciloscope permission error. */
#define PHDL_OSCI_ERR_EXE_ENVIRONMENT_ERROR                     (PH_ERR_CUSTOM_BEGIN+15)    /**< Osciloscope environment error. */
#define PHDL_OSCI_ERR_EXE_OPTION_ERROR                          (PH_ERR_CUSTOM_BEGIN+16)    /**< Osciloscope option error. */
#define PHDL_OSCI_ERR_EXE_UNRESOLVED_PARSING                    (PH_ERR_CUSTOM_BEGIN+17)    /**< Osciloscope unresolved parsing error. */
#define PHDL_OSCI_ERR_EXE_PARAMETER_ERROR                       (PH_ERR_CUSTOM_BEGIN+18)    /**< Osciloscope parameter error. */
#define PHDL_OSCI_ERR_EXE_CMD_NOT_IMPLEMENTED                   (PH_ERR_CUSTOM_BEGIN+19)    /**< Osciloscope command not implemented error. */
#define PHDL_OSCI_ERR_EXE_PARAM_MISSING                         (PH_ERR_CUSTOM_BEGIN+20)    /**< Osciloscope parameter missing error. */
#define PHDL_OSCI_ERR_EXE_HEX_DATA_ERROR                        (PH_ERR_CUSTOM_BEGIN+21)    /**< Osciloscope hex data error. */
#define PHDL_OSCI_ERR_EXE_WAVEFORM_ERROR                        (PH_ERR_CUSTOM_BEGIN+22)    /**< Osciloscope waveform error. */
#define PHDL_OSCI_ERR_EXE_WAVEFORM_DESCRIPTION_ERROR            (PH_ERR_CUSTOM_BEGIN+23)    /**< Osciloscope waveform description error. */
#define PHDL_OSCI_ERR_EXE_WAVEFORM_TEXT_ERROR                   (PH_ERR_CUSTOM_BEGIN+24)    /**< Osciloscope waveform text error. */
#define PHDL_OSCI_ERR_EXE_WAVEFORM_TIME_ERROR                   (PH_ERR_CUSTOM_BEGIN+25)    /**< Osciloscope waveform time error. */
#define PHDL_OSCI_ERR_EXE_WAVEFORM_DATA_ERROR                   (PH_ERR_CUSTOM_BEGIN+26)    /**< Osciloscope waveform data error. */
#define PHDL_OSCI_ERR_EXE_PANEL_SETUP_ERROR                     (PH_ERR_CUSTOM_BEGIN+27)    /**< Osciloscope panel setup error. */
#define PHDL_OSCI_ERR_EXE_NO_MASS_STORAGE_PRESENT               (PH_ERR_CUSTOM_BEGIN+28)    /**< Osciloscope no mass storage present error. */
#define PHDL_OSCI_ERR_EXE_MASS_STORAGE_NOT_FORMATTED            (PH_ERR_CUSTOM_BEGIN+29)    /**< Osciloscope mass storage not formated error. */
#define PHDL_OSCI_ERR_EXE_MASS_STORAGE_WAS_WRITE_PROTECTED      (PH_ERR_CUSTOM_BEGIN+30)    /**< Osciloscope mass storage write protected error. */
#define PHDL_OSCI_ERR_EXE_BAD_MASS_STORAGE_DETECTED             (PH_ERR_CUSTOM_BEGIN+31)    /**< Osciloscope bad mass storage detected error. */
#define PHDL_OSCI_ERR_EXE_MASS_STOR_ROOT_DIR_FULL               (PH_ERR_CUSTOM_BEGIN+32)    /**< Osciloscope mass storage root directory full error. */
#define PHDL_OSCI_ERR_EXE_MASS_STOR_FULL                        (PH_ERR_CUSTOM_BEGIN+33)    /**< Osciloscope mass storage directory full error. */
#define PHDL_OSCI_ERR_EXE_MASS_STOR_FILE_SEQ_NUM_EXHAUSTED      (PH_ERR_CUSTOM_BEGIN+34)    /**< Osciloscope mass storage sequence numbers exhausted error. */
#define PHDL_OSCI_ERR_EXE_MASS_STOR_FILE_NOT_FOUND              (PH_ERR_CUSTOM_BEGIN+35)    /**< Osciloscope mass storage file not found error. */
#define PHDL_OSCI_ERR_EXE_REQ_DIR_NOT_FOUND                     (PH_ERR_CUSTOM_BEGIN+36)    /**< Osciloscope requested directory not found error. */
#define PHDL_OSCI_ERR_EXE_FILENAME_ILLEGAL                      (PH_ERR_CUSTOM_BEGIN+37)    /**< Osciloscope illegal filename error. */
#define PHDL_OSCI_ERR_EXE_MASS_STORAGE_FILENAME_ALREADY_EXISTS  (PH_ERR_CUSTOM_BEGIN+38)    /**< Osciloscope mass storage filename allready exists error. */
#define PHDL_OSCI_ERR_READ_WAVEFORM_BUFFER_TOO_SMALL            (PH_ERR_CUSTOM_BEGIN+39)    /**< Buffer for reading waveform too small error. */
#define PHDL_OSCI_ERR_READ_WAVEFORM_HEADER_PARSE_INVALID        (PH_ERR_CUSTOM_BEGIN+40)    /**< Could NOT parse header of waveform data correctly. */
#define PHDL_OSCI_ERR_PICOSCOPE_DLL                             (PH_ERR_CUSTOM_BEGIN+41)     /**< PicoScope DLL driver threw an non standard error read wAddionalInfo to get error code */
#define PHDL_OSCI_ERR_TRIGGER_NOT_FIRED                         (PH_ERR_CUSTOM_BEGIN+42)    /**< PicoScope Data Capturing not successfull */
#define PHDL_OSCI_ERR_COULD_NOT_DETERMINE_MAX_POINTS            (PH_ERR_CUSTOM_BEGIN+43)    /**< Number of Maximum Points coult not been evaluated */
/*@}*/

/**
* \name Osci Configs
*/
/*@{*/
#define PHDL_OSCI_CONFIG_TRIGGER                 0x0000U /**< Select the trigger mode */
#define PHDL_OSCI_CONFIG_MODE                    0x0001U /**< Select the configuration mode, eg. FDT_106 */
#define PHDL_OSCI_CONFIG_FIELD_MULTIPLIER        0x0002U /**< Select the Field Multiplier value, eg. RMS = FieldStrength * 0.318(318/1000) of small antennas 0.118 (118/1000) */
#define PHDL_OSCI_CONFIG_AVERAGE_FACT            0x0003U /**< Select the averaging factor for the measurements */
#define PHDL_OSCI_CONFIG_RANGE_MIN               0x0004U /**< Set the minimum range for SetCorrRange [Div/10] */
#define PHDL_OSCI_CONFIG_RANGE_MAX               0x0005U /**< Set the minimum range for SetCorrRange [Div/10] */
/*#define PHDL_OSCI_CONFIG_TRIGGER_DELAY           0x0006U /**< Set Trigger Delay */
#define PHDL_OSCI_CONFIG_TIMEBASE_RANGE_US       0x0007U /**< Set the Timebase */

#define PHDL_OSCI_CONFIG_CHANNEL_ACTIVE          0x0008U /**< To which channel the settings for channel should be applied. PHDL_OSCI_CHANNEL_1 .. PHDL_OSCI_CHANNEL_4 */
#define PHDL_OSCI_CONFIG_CHANNEL_BW_LIMIT        0x0009U /**< Is the BW Limit enabled? PH_ON or PH_OFF */
#define PHDL_OSCI_CONFIG_CHANNEL_COUPLING        0x000AU /**< Coupling PHDL_OSCI_COUPLING_AC or PHDL_OSCI_COUPLING_DC */
#define PHDL_OSCI_CONFIG_CHANNEL_DISPLAY         0x000BU /**< Is the selected channel displayed? PH_ON or PH_OFF*/
#define PHDL_OSCI_CONFIG_CHANNEL_IMPEDANCE       0x000CU /**< The impedance of the selected channel. PHDL_OSCI_IMPEDANCE_FIFTY or PHDL_OSCI_IMPEDANCE_ONE_MEGA */
#define PHDL_OSCI_CONFIG_CHANNEL_INVERT          0x000DU /**< Is the channel inverted. PH_ON or PH_OFF*/
#define PHDL_OSCI_CONFIG64_CHANNEL_OFFSET_MV     0x000EU /**< The offset of the channel in mV */
#define PHDL_OSCI_CONFIG_CHANNEL_PROBE           0x000FU /**< The atenuation of the attached probe. PHDL_OSCI_PROBE_ATTENUATION_1, PHDL_OSCI_PROBE_ATTENUATION_10, PHDL_OSCI_PROBE_ATTENUATION_100 */
#define PHDL_OSCI_CONFIG64_CHANNEL_SCALE_MV      0x0010U /**< Scale in mV/div */
#define PHDL_OSCI_CONFIG64_CHANNEL_RANGE_MV      0x0011U /**< Range in mV (4 times scale) */
#define PHDL_OSCI_CONFIG_CHANNEL_UNITS           0x0012U /**< The unit of the measured value. PHDL_OSCI_UNIT_VOLT, PHDL_OSCI_UNIT_AMPERE */
#define PHDL_OSCI_CONFIG_CHANNEL_VERNIER         0x0013U /**< Vernier enabled. PH_ON or PH_OFF */

#define PHDL_OSCI_CONFIG_TRIGGER_TYPE            0x0014U /**< Which trigger mode is used */
#define PHDL_OSCI_CONFIG_TRIGGER_SOURCE          0x0015U /**< The source that is used for the current trigger type. PHDL_OSCI_CHANNEL_1 .. PHDL_OSCI_CHANNEL_4, PHDL_OSCI_CHANNEL_EXTERNAL, PHDL_OSCI_CHANNEL_LINE, PHDL_OSCI_CHANNEL_DIGITAL1 .. PHDL_OSCI_CHANNEL_DIGITAL15 */
#define PHDL_OSCI_CONFIG64_TRIGGER_LEVEL_MV      0x0016U /**< The level that is used for the trigger in mV */
#define PHDL_OSCI_CONFIG_TRIGGER_EDGE_SLOPE      0x0017U /**< Which edge should be used. PHDL_OSCI_EDGE_SLOPE_POSITIVE, PHDL_OSCI_EDGE_SLOPE_NEGATIVE, PHDL_OSCI_EDGE_SLOPE_EITHER, PHDL_OSCI_EDGE_SLOPE_ALTERNATE*/

#define PHDL_OSCI_CONFIG_TIMEBASE_MODE           0x0018U /**< Mode of the timebase. PHDL_OSCI_TIMEBASE_MODE_MAIN, PHDL_OSCI_TIMEBASE_MODE_WINDOW, PHDL_OSCI_TIMEBASE_MODE_XY, PHDL_OSCI_TIMEBASE_MODE_ROLL */
#define PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS  0x0019U /**< Delay of the trigger point in ns */
#define PHDL_OSCI_CONFIG64_TIMEBASE_SCALE_NS     0x001AU /**< Scale of the timebase in ns/div */
#define PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS     0x001BU /**< Range of the full scale timebase in ns */
#define PHDL_OSCI_CONFIG_TIMEBASE_VERNIER        0x001CU /**< Vernier enabled. PH_ON or PH_OFF */
#define PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES  0x001DU /**< Fixed Memory Max Sample Points for adjusting the Sampling Rate, the real used points may depend on the scope and sampling time/rate */
#define PHDL_OSCI_CONFIG64_CURRENT_SAMPLE_RATE   0x001EU /**< Getter for the current sample rate in Hz */
#define PHDL_OSCI_CONFIG_NUMBER_CHANNELS         0x001FU /**< Getter for the number of data channels of the current osci */
#define PHDL_OSCI_CONFIG_MEASUREMENT_OPTIONS     0x0020U /**< Configure additional options when perform measurement. PHDL_OSCI_MEASUREMENT_ADJUST_TIME, PHDL_OSCI_MEASUREMENT_ADJUST_RANGE, PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER, PHDL_OSCI_MEASUREMENT_WAIT_STOP */
/*@}*/

/** \name Osci Coupling Modes
*/
/*@{*/
#define PHDL_OSCI_COUPLING_AC               0x0000U /**< Coupling AC */
#define PHDL_OSCI_COUPLING_DC               0x0001U /**< Coupling DC */
#define PHDL_OSCI_COUPLING_GND              0x0002U /**< Coupling GND */
/*@}*/

/** \name Osci input impedance
*/
/*@{*/
#define PHDL_OSCI_IMPEDANCE_FIFTY           0x0000U /**< 50 Ohm imput */
#define PHDL_OSCI_IMPEDANCE_ONE_MEGA        0x0001U /**< 1 Mega Ohm imput */
/*@}*/

/** \name Attenuator of the attached probe
*/
/*@{*/
#define PHDL_OSCI_PROBE_ATTENUATION_1       0x0000U /**< 1:1 probe */
#define PHDL_OSCI_PROBE_ATTENUATION_10      0x000AU /**< 1:10 probe */
#define PHDL_OSCI_PROBE_ATTENUATION_100     0x0064U /**< 1:100 probe */
/*@}*/

/** \name Osci Units
*/
/*@{*/
#define PHDL_OSCI_UNIT_VOLT                 0x0000U /**< The unit of the channel is Volt */
#define PHDL_OSCI_UNIT_AMPERE               0x0001U /**< The unit of the channel is Ampere */
/*@}*/

/** \name Osci Units
*/
/*@{*/
#define PHDL_OSCI_TRIGGER_TYPE_EDGE         0x0000U /**< A edge is used for trigger */
#define PHDL_OSCI_TRIGGER_TYPE_WIDTH        0x0001U /**< A width is used for trigger */
#define PHDL_OSCI_TRIGGER_TYPE_TEQ          0x0002U /**< Qualified is used for trigger */
#define PHDL_OSCI_TRIGGER_TYPE_PATTERN      0x0003U /**< A Pattern is used for trigger */
#define PHDL_OSCI_TRIGGER_TYPE_TV           0x0004U /**< TV is used for trigger */
#define PHDL_OSCI_TRIGGER_TYPE_SMART        0x0005U /**< Smart mode is used for trigger */
/*@}*/

/** \name Osci Channels
*/
/*@{*/
#define PHDL_OSCI_CHANNEL_1                 0x0001U /**< Channel 1 */
#define PHDL_OSCI_CHANNEL_2                 0x0002U /**< Channel 2 */
#define PHDL_OSCI_CHANNEL_3                 0x0003U /**< Channel 3 */
#define PHDL_OSCI_CHANNEL_4                 0x0004U /**< Channel 4 */
#define PHDL_OSCI_CHANNEL_EXTERNAL          0x0005U /**< External in */
#define PHDL_OSCI_CHANNEL_LINE              0x0006U /**< Line in */
#define PHDL_OSCI_CHANNEL_DIGITAL0          0x0007U /**< Digital input 0 */
#define PHDL_OSCI_CHANNEL_DIGITAL1          0x0008U /**< Digital input 1 */
#define PHDL_OSCI_CHANNEL_DIGITAL2          0x0009U /**< Digital input 2 */
#define PHDL_OSCI_CHANNEL_DIGITAL3          0x000AU /**< Digital input 3 */
#define PHDL_OSCI_CHANNEL_DIGITAL4          0x000BU /**< Digital input 4 */
#define PHDL_OSCI_CHANNEL_DIGITAL5          0x000CU /**< Digital input 5 */
#define PHDL_OSCI_CHANNEL_DIGITAL6          0x000DU /**< Digital input 6 */
#define PHDL_OSCI_CHANNEL_DIGITAL7          0x000EU /**< Digital input 7 */
#define PHDL_OSCI_CHANNEL_DIGITAL8          0x000FU /**< Digital input 8 */
#define PHDL_OSCI_CHANNEL_DIGITAL9          0x0010U /**< Digital input 9 */
#define PHDL_OSCI_CHANNEL_DIGITAL10         0x0011U /**< Digital input 10 */
#define PHDL_OSCI_CHANNEL_DIGITAL11         0x0012U /**< Digital input 11 */
#define PHDL_OSCI_CHANNEL_DIGITAL12         0x0013U /**< Digital input 12 */
#define PHDL_OSCI_CHANNEL_DIGITAL13         0x0014U /**< Digital input 13 */
#define PHDL_OSCI_CHANNEL_DIGITAL14         0x0015U /**< Digital input 14 */
#define PHDL_OSCI_CHANNEL_DIGITAL15         0x0016U /**< Digital input 15 */
/*@}*/

/** \name Osci Trigger Edge Slope
*/
/*@{*/
#define PHDL_OSCI_EDGE_SLOPE_POSITIVE       0x0000U /**< Use a positive slope for trigger */
#define PHDL_OSCI_EDGE_SLOPE_NEGATIVE       0x0001U /**< Use a negative slope for trigger */
#define PHDL_OSCI_EDGE_SLOPE_EITHER         0x0002U /**< Use a either slope for trigger */
#define PHDL_OSCI_EDGE_SLOPE_ALTERNATE      0x0003U /**< Use a alternate slope for trigger */
/*@}*/

/** \name Osci Timebase Modes
*/
/*@{*/
#define PHDL_OSCI_TIMEBASE_MODE_MAIN        0x0000U /**< The normal time base mode is the main time base. */
#define PHDL_OSCI_TIMEBASE_MODE_WINDOW      0x0001U /**< In the WINDow (zoomed or delayed) time base mode, measurements are made in the zoomed time base if possible */
#define PHDL_OSCI_TIMEBASE_MODE_XY          0x0002U /**< In the XY mode */
#define PHDL_OSCI_TIMEBASE_MODE_ROLL        0x0003U /**< In the ROLL mode, data moves continuously across the display from left to right. The oscilloscope runs continuously and is untriggered. */
/*@}*/

/** \name Osci Trigger Modes
*/
/*@{*/
#define PHDL_OSCI_TRIGGER_AUTO              0x0000U /**< Trigger setting Auto */
#define PHDL_OSCI_TRIGGER_NORMAL            0x0001U /**< Trigger setting Normal */
#define PHDL_OSCI_TRIGGER_SINGLE            0x0002U /**< Trigger setting Single */
#define PHDL_OSCI_TRIGGER_STOP              0x0003U /**< Trigger setting Stop */
/*@}*/

/** \name Osci Measurement Options
*/
/*@{*/
#define PHDL_OSCI_MEASUREMENT_ADJUST_TIME   0x0001U /**< Auto adjust timescale for measurements */
#define PHDL_OSCI_MEASUREMENT_ADJUST_RANGE  0x0002U /**< Auto adjust range for measurements */
#define PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER 0x0004U /**< Suppress the force trigger command */
#define PHDL_OSCI_MEASUREMENT_WAIT_STOP     0x0008U /**< Wait until aquisition is in stopped */
#define PHDL_OSCI_MEASUREMENT_MASK          0x0003U /**< Mask to code valid values */
#define PHDL_OSCI_MEASUREMENT_MASK_LW64Xi   0x000FU /**< Mask to code valid values for LeCroy Scope */
/*@}*/

/**
* \name Osci Mode Settings
*/
/*@{*/
#define PHDL_OSCI_MODE_FDT_PCD              0x0000U /**< Frame Delay Time PCD setting */
#define PHDL_OSCI_MODE_FDT_PICC_106_L3      0x0001U /**< Frame Delay Time PICC 106 Layer 3 setting */
#define PHDL_OSCI_MODE_FDT_PICC_106_L4      0x0002U /**< Frame Delay Time PICC 106 Layer 4 setting */
#define PHDL_OSCI_MODE_FDT_PICC_212_L4      0x0003U /**< Frame Delay Time PICC 212 Layer 4 setting */
#define PHDL_OSCI_MODE_FDT_PICC_424_L4      0x0004U /**< Frame Delay Time PICC 424 Layer 4 setting */
#define PHDL_OSCI_MODE_FDT_PICC_848_L4      0x0005U /**< Frame Delay Time PICC 848 Layer 4 setting */
#define PHDL_OSCI_MODE_LIVE                 0x0006U /**< Live setting */
#define PHDL_OSCI_MODE_SET                  0x0007U /**< SET setting */
#define PHDL_OSCI_MODE_GLOBAL               0x0008U /**< GLOBAL setting */
#define PHDL_OSCI_MODE_HFPAUSE_S            0x0009U /**< HFPause small setting */
#define PHDL_OSCI_MODE_HFPAUSE_M            0x000AU /**< HFPause medium setting */
#define PHDL_OSCI_MODE_HFPAUSE_L            0x000BU /**< HFPause large setting */
#define PHDL_OSCI_MODE_HFPAUSE_XS           0x000CU /**< HFPause xsmall setting */
#define PHDL_OSCI_MODE_ANTITEARING          0x000DU /**< AntiTearing mode setting */
#define PHDL_OSCI_MODE_LABELRESETWITHCOMMAND 0x000EU /**< LabelResetWithCommand mode setting */
#define PHDL_OSCI_MODE_ANTITEARINGDESELECT  0x000FU /**< AntiTering mode setting*/
#define PHDL_OSCI_MODE_VHD                  0x0010U /**< VHD setting */
#define PHDL_OSCI_MODE_EMD                  0x0011U /**< EMD setting */
/*@}*/

/**
* \name Osci Conversion Constants
*/
/*@{*/
#define PHDL_OSCI_FIELD_STRENGTH_DIVISOR    1000U /**< Divisor for resolution of field strength (1/1000 = 1mA/m). */
#define PHDL_OSCI_VOLTAGE_DIVISOR           1000U /**< Divisor for resolution of amplitude (1/1000 = 1mV). */
#define PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR  1000U /**< Divisor for resolution of field multiplier (1/1000 resolution == 1000). */
#define PHDL_OSCI_RANGE_DIVISOR               10U /**< Divisor for range factor. */
/*@}*/

/**
* \brief Initialize oscilloscope settings
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/

phStatus_t phdlOsci_InitOsci(
        void * pDataParams          /**< [In] Pointer to this layer's parameter structure. */
        );

/**
* \brief Set Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/

phStatus_t phdlOsci_SetConfig(
        void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,       /**< [In] Specifies parameter identifier. */
        uint16_t wValue             /**< [In] Specifies parameter value. */
        );

/**
* \brief Get Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetConfig(
        void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,       /**< [In] Specifies parameter identifier. */
        uint16_t * pwValue          /**< [Out] Return parameter value. */
        );

/**
* \Extended Set Config for 64 bit values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_SetConfig64(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        int64_t qValue                      /**< [In] Specifies parameter value. */
        );

/**
* \Extended Get Config for 64 bit values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetConfig64(
        void * pDataParams,                 /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,               /**< [In] Specifies parameter identifier. */
        int64_t * pqValue                   /**< [Out] Return parameter value. */
        );

/**
* \brief Get the current fieldstrength (mA/m)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetFieldStrength(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        uint16_t * pwFieldStrength      /**< [Out] FieldStrength value. */
        );

/**
* \brief Get the current amplitude (mV)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetAmpl(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        uint16_t * pwAmplitude          /**< [Out] Amplitude value. */
        );

/**
* \brief Get the current peak peak value (mV)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetPeakPeak(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        float64_t * pdfPkPk             /**< [Out] Peak Peak value. */
        );

/**
* \brief Get the current RMS value (mV)
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetRMS(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        uint16_t * pwRMS                /**< [Out] RMS value. */
        );

/**
* \brief Get the current RMS value (mV) as double value
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetRMSDouble(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        float64_t * pdfRMS              /**< [Out] RMS value. */
        );

/**
* \brief Oscilloscope is set to the correct range
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_SetCorrRange(
        void * pDataParams,         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel            /**< [In] Specifies the channel. */
        );

/**
* \brief get the current displayed waveform
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlOsci_GetWaveForm(
        void * pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel,               /**< [In] Specifies the channel. */
        uint32_t wWaveFormBufferSize,   /**< [In] Specifies the buffer size. */
        uint32_t * pwWaveFormLength,    /**< [Out] Specifies the waveform size. */
        uint8_t * pWaveFormBuffer,      /**< [Out] Waveform buffer  */
        uint8_t * pbHeaderOffset        /**< [Out] Header offset from Osci*/
        );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_OSCI */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLOSCI_H */
