/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software implementation of P40 CmdPub layer. Uses I.PublicCommands  in P40
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALP40CMDPUB_H
#define PHALP40CMDPUB_H

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_P40CMDPUB_SW

/** \defgroup phalP40CmdPub_Sw Component: Software
* @{
*/

#define PHAL_P40CMDPUB_SW_ID          0x01    /**< ID for  P40 CmdPub software component. */

/**
* \brief Private parameter structure
*/
/**
* \brief Internal tracker of current command/ response pair
*/
typedef struct
{
    uint16_t wLc;	 /**< Total number of command bytes to be sent */
    uint16_t wLe; /**< Total number of command bytes to be received*/
    uint8_t  bSw1; /**< Returned SW1 byte of the current command*/
    uint8_t  bSw2; /**< Returned SW2 byte of the current command*/
    uint8_t  bFlags; /**< Bit flags for command*/
} phalP40CmdPub_Sw_CurrentApdu_t;

typedef struct
{
    uint16_t wId;               /**< Layer ID for this component, NEVER MODIFY! */
    void *	pPalMifare;  /**< Pointer to the parameter structure of the underlying palMifare layer. */
    void *  pKeyStore;          /**< Pointer to the parameter structure of the key store component. */
    void *  pHal;  /**< Pointer to the parameter structure of the underlying hal layer. */
    void * pCryptoEnc;    /**< Pointer to the parameter structure of the Crypto layer for encryption. */
    void * pCryptoRng;    /**< Pointer to the parameter structure of the CryptoRng layer. */
    uint8_t bSessionKey[24];        /**< Session key for this authentication */
    uint8_t bIv[16];                /**< Max size of IV can be 16 bytes */
    uint16_t wCrc;                  /**< 2 Byte CRC initial value in Authenticate mode. */
    phalP40CmdPub_Sw_CurrentApdu_t  sCurrentApdu;

} phalP40CmdPub_Sw_DataParams_t;


/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_Sw_Init(
        phalP40CmdPub_Sw_DataParams_t * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
        uint16_t    wDataParamSize,            /**< [In] Specifies the size of the data parameter structure */
        void*       pPalMifareDataParams,    /**< [In]Pointer to the parameter structure of the palMifare layer. */
        void *      pKeyStoreDataParams,    /**< [In] Pointer to the parameter structure of the keystore layer. */
        void *      pHalDataParams,              /**< [In] Pointer to the parameter structure of the hal layer. */
        void *      pCryptoEncDataParams,  /**< Pointer to the parameter structure of the Crypto layer for encryption. */
        void *      pCryptoRngDataParams   /**< Pointer to the parameter structure of the CryptoRng layer. */
	);

/*@}*/
#endif  /*NXPBUILD__PHAL_P40CMDPUB_SW*/

#ifdef NXPBUILD__PHAL_P40CMDPUB


/** \defgroup phalP40CmdPub P40 Public Commands
* \brief This is the RdLib side of the public commands defined in the P40 CmdPub.
*
* <b>Defined format:</b>
* \verbatim:
* --> [CLA] [INS] [P1] [P2] [LC] [DATA] [LE]
* <-- [Data] [SW1SW2]
* \endverbatim
* \n
* It is the responsibilty of the application to activate the card accordingly before issuing the command.
* @{
*/

/** \name Custom Error Codes
*/
/*@{*/
#define PHAL_P40CMDPUB_ERR_COMMAND          (PH_ERR_CUSTOM_BEGIN+0)     /**< P40 bad command error. */
#define PHAL_P40CMDPUB_ERR_PARAM            (PH_ERR_CUSTOM_BEGIN+1)     /**< P40 bad parameter error. */
#define PHAL_P40CMDPUB_ERR_FORMAT           (PH_ERR_CUSTOM_BEGIN+2)     /**< P40 bad cmd format error. */
#define PHAL_P40CMDPUB_ERR_INTERNAL         (PH_ERR_CUSTOM_BEGIN+3)     /**< P40 execution error. */
#define PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED   (PH_ERR_CUSTOM_BEGIN+4)     /**< P40 not authorized error. */
#define PHAL_P40CMDPUB_ERR_RESP_LENGTH		(PH_ERR_CUSTOM_BEGIN+5)     /**< P40 response length incorrect. */
#define PHAL_P40CMDPUB_ERR_UNKNOWN          (PH_ERR_CUSTOM_BEGIN+6)     /**< P40 unknown error returned . */
/*@}*/

/** \name App management Authentication keytypes
*/
/*@{*/
#define PHAL_P40CMDPUB_AUTHKEY_CARDMASTER  0x00U   /**< Auth key for card master. */
#define PHAL_P40CMDPUB_AUTHKEY_APP1                   0x01U   /**< Auth key for application 1. */
#define PHAL_P40CMDPUB_AUTHKEY_APP2                   0x02U   /**< Auth key for application 2. */
#define PHAL_P40CMDPUB_AUTHKEY_ORIGINALITY     0x03U   /**< Auth key for wafer test. */
/*@}*/

/** \name Application Selection
*/
/*@{*/
#define PHAL_P40CMDPUB_APPLICATION_1   0x01U   /**< Application 1. */
#define PHAL_P40CMDPUB_APPLICATION_2   0x02U   /**< Application 2. */
/*@}*/

/** \name Lifecycle Switch
*/
/*@{*/
#define PHAL_P40CMDPUB_SWITCH_APPMGMT_TO_PRE_RELEASE   0x00U   /**< Switch from lifecycle application management to lifecycle pre release. */
#define PHAL_P40CMDPUB_SWITCH_APPMGMT_TO_RELEASE	   0x01U   /**< Switch from lifecycle application management to lifecycle release. */
#define PHAL_P40CMDPUB_SWITCH_PRE_RELEASE_TO_RELEASE   0x02U   /**< Switch from lifecycle pre release to lifecycle release. */
/*@}*/

/** \name Public Lifecycle Idents used to anticipate current activation level and use appropriate layer 3 or layer 4 exchange
*/
/*@{*/
#define PHAL_P40CMDPUB_LIFECYCLE_APPMGMT           0x00U   /**< Chip is in ApplicationManagement lifecycle and requires L4 exchange. */
#define PHAL_P40CMDPUB_LIFECYCLE_PRE_RELEASE   0x01U   /**< Chip is in Prerelease lifecycle and requires L3 exchange. */
#define PHAL_P40CMDPUB_LIFECYCLE_RELEASE            0x02U   /**< Chip is in Release lifecycle and requires L3 exchange. */
/*@}*/

/** \name Fast Test Type
*/
/*@{*/
#define PHAL_P40CMDPUB_FASTTEST_ALL    0x00U   /**< Execute all available tests. */
/*@}*/

/** \name Test fuse Type
*/
/*@{*/
/* TestFuse defines*/
#define PHAL_P40CMDPUB_TESTFUSE_DISABLE  0x00U   /**< Disable testfuse and checking. */
#define PHAL_P40CMDPUB_TESTFUSE_ENABLE   0xFFU  /**< Enable testfuse and checking. */

/*@}*/

#define PHAL_P40CMDPUB_VERSIONINFO_SIZE		 0x0010U
#define PHAL_P40CMDPUB_CONFIGURATION_SIZE    0x000DU


/**
* \brief Send GetVersion command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
*/
phStatus_t phalP40CmdPub_GetVersion(
                    void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,   /**< [In] Lifecycle Ident for correct exchange format. */
                    uint8_t * pVersionData   /**< [Out] Pointer to store GetVersion return data. */
                    );

/**
* \brief Send DownloadCode command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_DownloadCode(
                    void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                    uint16_t wAddr,              /**< [In] specify starting address to write code. */
                    uint8_t * pCode,             /**< [In] Pointer to code to be written. */
                    uint16_t wCodeLength  /**< [In] Size of the code to be written. */
                    );

/**
* \brief Send DownloadConst command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_DownloadConst(
                    void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                    uint16_t wAddr,              /**< [In] specify starting address to write code. */
                    uint8_t * pConst,             /**< [In] Pointer to const to be written. */
                    uint16_t wConstLength  /**< [In] Size of the const to be written. */
                    );

/**
* \brief Send DownloadEE command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_DownloadEE(
                    void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                    uint16_t wAddr,             /**< [In] specify starting address to write code. */
                    uint8_t * pEEData,        /**< [In] Pointer to EE data to be written. */
                    uint16_t wEEDataLength    /**< [In] Size of the EE data  to be written. */
                    );

/**
* \brief Send DownloadPatch command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_DownloadPatch(
                    void * pDataParams,           /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t bPatchNr,                /**< [In] Number of patch to be written */
                    uint16_t wPatchAddress,   /**< [In] Address in the code area to be patched */
                    uint8_t * pPatch,                 /**< [In] Pointer to patch code to be written. */
                    uint16_t wPatchLength      /**< [In] Size of patch code to be written. */
                    );

/**
* \brief Send DownloadFlash command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_DownloadFlash(
                    void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                    uint16_t wAddr,              /**< [In] specify starting address to write code. */
                    uint8_t * pFlashData,     /**< [In] Pointer to Flash data to be written. */
                    uint16_t wFlashDataLength /**< [In] Size of the Flash data  to be written. */
                    );

/**
* \brief Send LockApplication command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_LockApplication(
                    void * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
                    );

/**
* \brief Send FormatApplication command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_FormatApplication(
                    void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t bApplicationId	  /**< [In] Application ID to be formatted.*/
                    );

/**
* \brief Send SwitchLifeCycle command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_SwitchLifeCycle(
                    void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,   /**< [In] Lifecycle Ident for correct exchange format. */
                    uint8_t bLcType       /**< [In] Life cycle type to be executed.*/
                    );

/**
* \brief Send ExecuteTrapdoor command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_ExecuteTrapdoor(
                    void * pDataParams,   /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId   /**< [In] Lifecycle Ident for correct exchange format. */
                    );

/**
* \brief Send SetConfiguration command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_SetConfiguration(
                    void * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t bDisableCmdTestFuse, /**< [In] Enables or disables cmd based test fuse.*/
                    uint8_t bDisableTPTestFuse,    /**< [In] Enables or disables test pad based test fuse.*/
                    uint8_t bUseExtendedConfig,   /**< [In] If set, the configs that follow are also sent to the card*/
                    uint16_t wApp1CodeSize,         /**< [In] Code size of application 1. */
                    uint16_t wApp1ConstSize,         /**< [In] Const size of application 1. */
                    uint16_t wApp2CodeSize,         /**< [In] Code size of application 2. */
                    uint16_t wApp2ConstSize,         /**< [In] Const size of application 2. */
                    uint16_t wCommBufferSize,      /**< [In] Communication Buffer size. */
                    uint8_t bNumKeys,                     /**< [In] Number of keys in RAM keystore. */
                    uint8_t bDefaultApp                    /**< [In] Default Application. */
                    );

/**
* \brief Send GetConfiguration command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_GetConfiguration(
                    void * pDataParams,                       /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t * pDisableCmdTestFuse,  /**< [Out] Enables or disables cmd based test fuse.*/
                    uint8_t * pDisableTPTestFuse,      /**< [Out] Enables or disables test pad based test fuse.*/
                    uint16_t * pApp1CodeSize,           /**< [Out] Code size of application 1. */
                    uint16_t * pApp1ConstSize,           /**< [Out] Const size of application 1. */
                    uint16_t * pApp2CodeSize,           /**< [Out] Code size of application 2. */
                    uint16_t * pApp2ConstSize,           /**< [Out] Const size of application 2. */
                    uint16_t * pCommBufferSize,         /**< [Out] Communication Buffer size. */
                    uint8_t * pNumKeys,                      /**< [Out] Number of keys in RAM keystore. */
                    uint8_t * pDefaultApp                     /**< [Out] Default Application. */
                    );

/**
* \brief Send FastTest command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PHAL_P40CMDPUB_ERR_FORMAT
* \retval #PHAL_P40CMDPUB_ERR_PARAM
* \retval #PHAL_P40CMDPUB_ERR_NOT_AUTHORIZED
*/
phStatus_t phalP40CmdPub_FastTest(
                    void * pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,    /**< [In] Lifecycle Ident for correct exchange format. */
                    uint8_t bFastTestType /**< [In] Fast Test type to be executed.*/
                    );

/**
* \brief Send VerifyDownload command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_VerifyDownload(
                     void * pDataParams, /**< [In] Pointer to this layer's parameter structure. */
                     uint16_t wSHA5        /**< [In] The expected hash code. */
                     );

/**
* \brief Send GetChallenge command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_GetChallenge(
                     void * pDataParams,          /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,          /**< [In] Lifecycle Ident for correct exchange format. */
                     uint8_t bRequestLen,        /**< [In] Requested length of challenge string. */
                     uint8_t* pChallenge          /**< [Out] Challenge string received. */
                     );

/**
* \brief Send ExternalAuthenticate command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_ExternalAuthenticate(
                    void * pDataParams,      /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,     /**< [In] Lifecycle Ident for correct exchange format. */
                    uint8_t bCryptoMethod, /**< [In] Crypto method used to authenticate. */
                    uint8_t bKeyId,               /**< [In] Ident of Key to authenticate to. */
                    uint8_t* pEdata,            /**< [In] Encrypted string to decipher.. */
                    uint8_t bEdataLen       /**< [In] Length of encrypted string. */
                    );

/**
* \brief Send InternalAuthenticate command to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_InternalAuthenticate(
                     void * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
                     uint8_t  bLifeCycleId,     /**< [In] Lifecycle Ident for correct exchange format. */
                     uint8_t bCryptoMethod, /**< [In] Crypto method used to authenticate. */
                     uint8_t bKeyId,               /**< [In] Ident of Key to authenticate to. */
                     uint8_t* pEdata,            /**< [In] Encrypted string to decipher.. */
                     uint8_t bEdataLen       /**< [In] Length of encrypted string. */
                      );

/**
* \brief Send Maintainance Request A
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_MaintainanceReqa(
                    void * pDataParams             /**< [In] Pointer to this layer's parameter structure. */
                    );

/**
* \brief Sends Authenticate command sequence to P40 I.PublicCommand interface implementation
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phalP40CmdPub_Authenticate(
                    void * pDataParams,          /**< [In] Pointer to this layer's parameter structure. */
                    uint8_t  bLifeCycleId,         /**< [In] Lifecycle Ident for correct exchange format. */
                    uint8_t bKeyId,                   /**< [In] Ident of Key to authenticate to. */
                    uint16_t wKeyNumber,     /**< [In] Key Storage number. */
                    uint16_t wKeyVersion       /**< [In] Key Storage version. */
                    );

/** @} */
#endif /* NXPBUILD__PHAL_P40CMDPUB */

#endif /* PHALP40CMDPUB_H */

