/*
 * Copyright 2019 - 2020, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHTOOLS_MICROPROSS_H
#define PHTOOLS_MICROPROSS_H

#include <ph_Status.h>
#include <phhalHwContact.h>
#include <WinSock2.h>
#include <WS2tcpip.h>

#ifdef __cplusplus
extern "C" {
#endif    /* __cplusplus */

#define PHTOOLS_MICROPROSS_ERR_MICROPROSS                           (PH_ERR_CUSTOM_BEGIN+0)    /**< Non standard error. Read wAddionalInfo to get error code. */

#define PHTOOLS_MICROPROSS_ERR_OK                                   0x000U      /**< Everything OK. */

#define PHTOOLS_MICROPROSS_ERR_INVALID_PARAMETER                    0x003U      /**< invalid parameters. */

#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_INVALID_PIN_STATE          0x118U      /**< This pin is in unexpected state. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_FULL                       0x119U      /**< No more ram the sequencer is full. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_NO_SUCH_SEQUENCEID         0x11AU      /**< Invalid sequence ID. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_NO_MORE_BUFFER_SPACE       0x11BU      /**< Instruction string is too long. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_RUNNING                    0x11DU      /**< Sequencer running. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_MAX_SEQUENCE_REACHED       0x11EU      /**< More than 10 sequences have been opened. */
#define PHTOOLS_MICROPROSS_ERR_SEQUENCER_TIMEOUT                    0x11FU      /**< Timeout has been reached for the execution of a blocking sequence. */

#define PHTOOLS_MICROPROSS_ERR_TIMEOUTRX                            0x190U      /**< Receive timeout has been reached. */
#define PHTOOLS_MICROPROSS_ERR_COLLISION                            0x191U      /**< Collision TYPE A detected. */
#define PHTOOLS_MICROPROSS_ERR_ANTICOL_FAILED                       0x192U      /**< Anticollision type A has failed. */
#define PHTOOLS_MICROPROSS_ERR_PARITY                               0x193U      /**< Parity error detected (type A). */
#define PHTOOLS_MICROPROSS_ERR_BITCOUNTERR                          0x195U      /**< Received invalid number of bits (type A). */
#define PHTOOLS_MICROPROSS_ERR_SELECT_FAILED                        0x196U      /**< Select command failed (type A). */
#define PHTOOLS_MICROPROSS_ERR_CRC                                  0x197U      /**< CRC received failed. */
#define PHTOOLS_MICROPROSS_ERR_ANSWER_TO_HALT_DETECTED              0x199U      /**< The Picc type A responds to HALT command. */

#define PHTOOLS_MICROPROSS_ERR_ATS_INVALID_LENGTH                   0x19AU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_CRC                      0x19BU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_T0                       0x19CU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_FSCI                     0x19DU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_FWI                      0x19EU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_TC1                      0x19FU
#define PHTOOLS_MICROPROSS_ERR_INVALID_ATS_TA1                      0x1A0U

#define PHTOOLS_MICROPROSS_ERR_INCONSISTENT_RECEIVED_FRAME          0x1A1U     /**< Type B. */
#define PHTOOLS_MICROPROSS_ERR_INVALID_ANSWER_TO_HALT               0x1A2U     /**< Answer to halt command is not valid. */
#define PHTOOLS_MICROPROSS_ERR_WRONG_ATTRIB                         0x1A3U     /**< Attrib content is wrong or not supported (MPC_SendATTRIB). */
#define PHTOOLS_MICROPROSS_ERR_DESELECT_SENT                        0x1A4U     /**< DESELECT command sent due to invalid exchange in the protocol 14443-4. */
#define PHTOOLS_MICROPROSS_ERR_DATA_RATE_NOT_SUPPORTED              0x1A5U     /**< Data rate not supported by the CLASS (API MPC_SelectDataRate). */
#define PHTOOLS_MICROPROSS_ERR_INVALID_ANSWER_TO_DESELECT           0x1A6U     /**< DESELECT command return error. */
#define PHTOOLS_MICROPROSS_ERR_TIMEOUTRX_CMD1                       0x1A9U     /**< Fo answer after the first command (API MPC_TEST ( TEST_FDT_PICCPCD_A/B). */
#define PHTOOLS_MICROPROSS_ERR_TIMEOUTTX                            0x1AAU     /**< End of transmission not detected (API MPC_TEST). */
#define PHTOOLS_MICROPROSS_ERR_UNEXPECTED_RX                        0x1ABU     /**< Unexpected RX frame received. */

#define PHTOOLS_MICROPROSS_ERR_FELICA_INVALID_LENGTH                430    /**< Received invalid length (type Felica). */
#define PHTOOLS_MICROPROSS_ERR_FELICA_INVALID_SYNC_CODE             431    /**< Received invalid sync code (type Felica). */

#define PHTOOLS_MICROPROSS_ERR_PPS_INVALID_PARAMETER                432    /**< Invalid parameter passed to the API MPC_SendPPS. */
#define PHTOOLS_MICROPROSS_ERR_PPS_INVALID_ANSWER                   433    /**< Received invalid answer from the PPS command. */

#define PHTOOLS_MICROPROSS_ERR_CALIBRATION_1                        500
#define PHTOOLS_MICROPROSS_ERR_CALIBRATION_2                        501
#define PHTOOLS_MICROPROSS_ERR_CALIBRATION_3                        502
#define PHTOOLS_MICROPROSS_ERR_CALIBRATION_4                        503
#define PHTOOLS_MICROPROSS_ERR_INVALID_IMPEDANCE_CALIBRATION_TABLE  504
#define PHTOOLS_MICROPROSS_ERR_NO_CABLE_INFORMATION                 505
#define PHTOOLS_MICROPROSS_ERR_NO_HEAD_INFORMATION                  506

#define PHTOOLS_MICROPROSS_ERR_RESONANCE_FREQUENCY_MEASUREMENT      507       /**< system can not determine the resonance frequency. */
#define PHTOOLS_MICROPROSS_ERR_ADJUSTRX_CHANNEL_2                   508       /**< System can not adjust Rx channel 2. */

#define PHTOOLS_MICROPROSS_ERR_NO_CABLE_PLUS_HEAD_INFORMATION       509
#define PHTOOLS_MICROPROSS_ERR_NO_SHAPE_SET                         510

#define PHTOOLS_MICROPROSS_ERR_PROTOCOL_ERROR_DETECTED              513
#define PHTOOLS_MICROPROSS_ERR_PROTOCOL_STOP_TIMEOUT_DETECTED       514

#define PHTOOLS_MICROPROSS_ERR_CARD_TIMEOUT                         202     /**< Timeout occurs during card data read */
#define PHTOOLS_MICROPROSS_ERR_CARD_MUTE                            217     /**< Card is in mute state - is there even a card available? */

#define PHTOOLS_MICROPROSS_ERR_I2C_NOT_OPEN                         870     /**< I2C mode not selected */
#define PHTOOLS_MICROPROSS_ERR_I2C_RX_ERROR                         871     /**< Error in I2C reception */
#define PHTOOLS_MICROPROSS_ERR_I2C_RX_OVERRUN                       872     /**< Overrun during I2C reception */
#define PHTOOLS_MICROPROSS_ERR_I2C_TIMEOUT                          873     /**< Timeout during I2C exchange */
#define PHTOOLS_MICROPROSS_ERR_I2C_TRANSMISSION_ERROR               874     /**< Error in I2C transmission */

typedef struct
{
    uint8_t    ID[4];
    uint16_t   DataFormat;
    uint16_t   Version;
    uint8_t    MachineID[32];
    uint8_t    MachineVers[32];
    uint16_t   TimeBase;
    uint32_t   ClockDivider;
    uint8_t    TraceType;
    uint8_t    Protocol;
    uint32_t   NumberOfEvents;
    uint16_t   ConfigFlags;
    uint16_t   EventsMask1;
    uint16_t   EventsMask2;
    uint16_t   EventsMask3;
    uint16_t   VccThresholdLow;
    uint16_t   VccThresholdHigh;
    uint16_t   VIL;
    uint16_t   VIH;
    uint32_t   FICC;
    uint8_t    StateOfPins;
    uint8_t    Thresholds;         /* File Version >= 2 */
    uint8_t    MeasuredOnPin;      /* File Version >= 2 */
    uint8_t    SpySensitivity;     /* File Version >= 3 */
    uint32_t   OpenLogEventsMask;  /* File Version >= 3 */
    uint16_t   SWPS2Threshold;     /* File Version >= 3 */
    uint8_t    InitStateSWPLines;  /* File Version >= 4 */
    uint32_t   BurstDelay;         /* File Version >= 5 */
    uint8_t    RFU[9];
}SpyHeaderContact;

typedef struct
{
    uint8_t    ID[4];
    uint16_t   DataFormat;
    uint16_t   Version;
    uint8_t    MachineID[32];
    uint8_t    ModuleID[32];
    uint16_t   TimeBase;
    uint32_t   RFU1;
    uint8_t    TraceType;
    uint8_t    RFU2;
    uint32_t   NumberOfEvents;
    uint16_t   ConfigFlags;
    uint32_t   EventsMasks;
    uint16_t   RFU3;
    uint32_t   FRfMeasure;         /* File Version >= 3 */
    uint32_t   BurstDelay;         /* File Version >= 4 */
    uint8_t    RFU[28];
}SpyHeaderContactless;

/** \defgroup phToolsMicropross Mircopross
*
* \brief This component implements tools that are needed for some mircropross functionality
* @{
*/

phStatus_t phToolsMicropross_GetStatus(
                                       phhalHwContact_Micropross_DataParams_t * pDataParams,
                                       uint8_t * pCmd
                                       );

phStatus_t phToolsMicropross_EvaluateStatus(
                                            phhalHwContact_Micropross_DataParams_t * pDataParams,
                                            uint16_t wStatus
                                            );

phStatus_t phToolsMicropross_SendCommand(
                                         phhalHwContact_Micropross_DataParams_t * pDataParams
                                         );

phStatus_t phToolsMicropross_GetParam_HexString(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData,
    uint16_t wDataSize,
    uint16_t * pwDataLen
    );

phStatus_t phToolsMicropross_GetParam_Uint8(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData
    );

phStatus_t phToolsMicropross_GetParam_Uint16(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint16_t * pData
    );

phStatus_t phToolsMicropross_GetParam_Uint32(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint32_t * pData
    );

phStatus_t phToolsMicropross_GetParam_Int32(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    int32_t * pData
);

phStatus_t phToolsMicropross_GetParam_Uint8Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint8_t * pData
    );

phStatus_t phToolsMicropross_GetParam_Uint16Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint16_t * pData
    );

phStatus_t phToolsMicropross_GetParam_Uint32Hex(
    phhalHwContact_Micropross_DataParams_t * pDataParams,
    uint32_t * pData
    );

phStatus_t phToolsMicropross_OpenResource(phhalHwContact_Micropross_DataParams_t * pDataParams);

phStatus_t phToolsMicropross_SpyClose(phhalHwContact_Micropross_DataParams_t * pDataParams);

phStatus_t phToolsMicropross_GetResourceID(phhalHwContact_Micropross_DataParams_t * pDataParams, uint8_t bCouplerNumber, uint32_t * pdwResourceID);

phStatus_t phToolsMicropross_GetRxBuffer(
                                         phhalHwContact_Micropross_DataParams_t * pDataParams,
                                         uint8_t ** pRxBuffer,
                                         uint16_t * pRxBufferLen,
                                         uint16_t * pRxBufferSize
                                         );

phStatus_t phToolsMicropross_AddToIntBufferHexString(
                                                     phhalHwContact_Micropross_DataParams_t * pDataParams,
                                                     uint8_t * pData,
                                                     uint16_t wLen
                                                     );

phStatus_t phToolsMicropross_GetReaderVersions(phhalHwContact_Micropross_DataParams_t * pDataParams);

phStatus_t phToolsMicropross_GetEepromSys(phhalHwContact_Micropross_DataParams_t * pDataParams, uint8_t bParamId, uint8_t * pbParamValue, uint16_t wParamSize);

phStatus_t phToolsMicropross_SendAndCheckCmd(phhalHwContact_Micropross_DataParams_t * pDataParams, const char *format, ...);

phStatus_t phToolsMicropross_GetLocalIp(uint32_t NetworkAddress, uint32_t SubnetMask, uint8_t *ip);

phStatus_t phToolsMicropross_CheckSuccessCloseSpy(phhalHwContact_Micropross_DataParams_t * pDataParams, phStatus_t status);

phStatus_t phToolsMicropross_CreateServerSocket(void *GEvtReady, uint8_t *pExpectedClientSocketHostName, SOCKET *ClientSocket, SOCKET *ServerSocket);

void phToolsMicropross_SwapHeaderContact(SpyHeaderContact *gHeader);

void phToolsMicropross_SwapHeaderContactless(SpyHeaderContactless *gHeader);

void phToolsMicropross_SwapEvents16(uint16_t *buf, uint32_t n);

void phToolsMicropross_SwapEvents32(uint32_t *buf, uint32_t n);

void phToolsMicropross_SwapEvents64(uint64_t *buf, uint32_t n);

/**
* end of phToolsMicropross group
* @}
*/

#ifdef __cplusplus
}/*Extern C*/
#endif

#endif /* PHTOOLS_MICROPROSS_H */
