/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Rd710/Rd852 SAM PC/SC.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd710_Rd852_SamX_Pcsc.h"

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phalMfc.h>
#include <phalMfp.h>
#include <phhalHw_Rd710_Cmd.h>


#define GLOBAL_BUFFER_SIZE     0x012CU  /**< global size of transmit and recieve buffer */

/**
 * Pegoda 2 configurations
 */
#define PCSC_READER_P2_NON_X_NAME   "NXP Pegoda S 0 0"        /**< Pegoda 2: SAM in non X-Mode */

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[2];
phLog_LogEntry_t logEntries[32];
#endif /* NXPBUILD__PH_LOG */

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

#ifdef NXPBUILD__PH_LOG
void Log_Callback(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    phStatus_t status;
    uint8_t bIsFirstString;
    uint8_t bIsFirstParam;
    uint16_t wIndex;
    uint16_t wByteIndex;
    uint16_t wOption;
    char sOutput[512];
    char sTmp[7];

    /* satisfy compiler */
    if (pDataParams);

    /* Ignore some calls */
    if ((strncmp((char*)pEntries[0].pString, "phhalHw_SetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_GetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_ReadReg", 15) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_WriteReg", 16) == 0))
    {
        return;
    }

    /* Clear output string */
    sprintf(sOutput, "");

    /* Init. states */
    bIsFirstString = 1;
    bIsFirstParam = 1;

    if (bOption != PH_LOG_OPTION_CATEGORY_GEN)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }

    for (wIndex = 0; wIndex < wEntryCount; ++wIndex)
    {
        if ((pEntries[wIndex].bLogType == PH_LOG_LOGTYPE_INFO) && pEntries[wIndex].wDataLen == 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            if (bIsFirstString)
            {
                switch (bOption)
                {
                case PH_LOG_OPTION_CATEGORY_ENTER:
                    strcat_s(sOutput, sizeof(sOutput), " [ENTRY] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_GEN:
                    strcat_s(sOutput, sizeof(sOutput), " [GEN] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_LEAVE:
                    strcat_s(sOutput, sizeof(sOutput), " [LEAVE] (");
                    break;
                }
                bIsFirstString = 0;
            }
        }
        else
        {
            if (!bIsFirstParam)
            {
                if ((strcmp((char*)pEntries[wIndex].pString, "status") != 0) && (pEntries[wIndex].wDataLen > 0))
                {
                    strcat_s(sOutput, sizeof(sOutput), ", ");
                }
            }
            else
            {
                bIsFirstParam = 0;
            }
        }

        if ((strcmp((char*)pEntries[wIndex].pString, "wOption") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            wOption = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            strcat_s(sOutput, sizeof(sOutput), "wOption = ");
            switch (wOption & PH_EXCHANGE_MODE_MASK)
            {
            case PH_EXCHANGE_DEFAULT:
                strcat_s(sOutput, sizeof(sOutput), "DEFAULT");
                break;
            case PH_EXCHANGE_TXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "TXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING_BUFSIZE:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING_BUFSIZE");
                break;
            }
            if ((wOption & PH_EXCHANGE_BUFFERED_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | BUFFERED_BIT");
            }
            if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | LEAVE_BUFFER_BIT");
            }
        }
        else if ((strcmp((char*)pEntries[wIndex].pString, "status") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            strcat_s(sOutput, sizeof(sOutput), ") = ");
            status = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            sprintf_s(sTmp, sizeof(sTmp), "0x%04X", status);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
        }
        else if (pEntries[wIndex].wDataLen > 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            strcat_s(sOutput, sizeof(sOutput), "(");
            sprintf_s(sTmp, sizeof(sTmp), "%d", pEntries[wIndex].wDataLen);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
            strcat_s(sOutput, sizeof(sOutput), ") = ");

            for (wByteIndex = 0; wByteIndex < pEntries[wIndex].wDataLen; ++wByteIndex)
            {
                sprintf_s(sTmp, sizeof(sTmp), "%02X", ((uint8_t*)pEntries[wIndex].pData)[wByteIndex]);
                strcat_s(sOutput, sizeof(sOutput), sTmp);
                if ((wByteIndex + 1) < pEntries[wIndex].wDataLen)
                {
                    strcat_s(sOutput, sizeof(sOutput), " ");
                }
            }
        }
    }

    if (bOption == PH_LOG_OPTION_CATEGORY_LEAVE)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }
    else
    {
        strcat_s(sOutput, sizeof(sOutput), ")");
    }
    printf(sOutput);
}

#endif /* NXPBUILD__PH_LOG */

int __cdecl main()
{
    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t bRdStatus[9];
    uint8_t aAtr[256];                /**< Atr */
    uint16_t wAtrLen;
	uint8_t aTxBuffer[9];
	uint16_t wTxBufferLength;
	uint8_t aRxBuffer[255];
	uint16_t wRxBufferLength;
	uint8_t aPPS[4];
	uint8_t *pBuffer;

    /**
	 * Bfl data parameter storage
	 */
    phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
	phhalHw_Rd710_DataParams_t halRd710;              /**< Rd710 HAL parameter structure */

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
	status = phLog_Init(Log_Callback, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&balPcsc, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&halRd710, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

    printf("----------------\n");
    printf("Connected Pegoda 2: SAM in non X-mode\n");
    printf("----------------\n");
	printf("\n\n");

 	/**
	 * Init. PCSC BAL
	 */
	status = phbalReg_PcscWin_Init(&balPcsc,                                /**< [In] Pointer to this layer's parameter structure. */
		                           sizeof(phbalReg_PcscWin_DataParams_t),   /**< [In] Specifies the size of the data parameter structure. */
                                   aAtr,                                    /**< [In] ATR buffer. */
                                   sizeof(aAtr));                           /**< [In] ATR buffer size. */
	CHECK_SUCCESS(status);

	/**
	 * Initialise the Rd710 HAL component:
	 *   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
	 */
	status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
		                        sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
								&balPcsc,   /**< [In] Pointer to the lower layers parameter structure */
								0,          /**< [In] Slot number */
								aHalBufferReaderTx,  /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
								sizeof(aHalBufferReaderTx),        /**< [In] Size of the global transmit buffer. */
								aHalBufferReaderRx,  /**< [In] Pointer to global receive buffer used by the Exchange() function. */
								sizeof(aHalBufferReaderRx));       /**< [In] Size of the global receive buffer. */
	CHECK_SUCCESS(status);

	/**
	 * set the BAL Communication Configuration values
	 */
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		                        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,  /**< [In] BAL Communication Configs */
			                    PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED); /**< [In] BAL Communication Configuration values */
	CHECK_SUCCESS(status);
	/**
	 * set the BAL Communication Configs
	 */
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		                        PHBAL_REG_PCSCWIN_CONFIG_SHARE,  /**< [In] BAL Communication Configs */
							    PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< [In] BAL Communication Configuration values */
	CHECK_SUCCESS(status);

	/**
	 * Select Port to be used (SAM in non X-Mode)
	 */
    status = phbalReg_SetPort(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
		                      (uint8_t*)PCSC_READER_P2_NON_X_NAME);  /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

	/**
	 * Open the BAL Rd710 SAM Port
	 */
	status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

	/**
	 * Init reader
	 */
    status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

	/**
	 * Deactivate reader
	 */
    status = phhalHw_Rd710_Cmd_CC_Deactivate(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	 * Activates card and returns ATR
	 */
    status = phhalHw_Rd710_Cmd_CC_Activate(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
										   aAtr,       /**< [Out] returns ATR */
										   &wAtrLen);  /**< [Out] length of the ATR */
    CHECK_SUCCESS(status);

    /**
	 * Performs the PPS with contact card
	 */
    aPPS[0] = 0x11;  /**< PPS0 */
	aPPS[1] = 0x11;  /**< PPS1 372/1 */
	status = phhalHw_Rd710_Cmd_CC_SendPPS(&halRd710,    /**< [In] Pointer to this layer's parameter structure. */
		                                  0x00,         /**< [In] Option byte: 0x00 = send PPS, 0x01 = only set the speed */
										  aPPS,         /**< [In] PPS */
										  aRxBuffer);   /**< [Out] recieve buffer */
    CHECK_SUCCESS(status);

    /* Get Reader status */
    status = phhalHw_Rd710_Cmd_RD_GetReaderStatus(&halRd710, bRdStatus);
    CHECK_SUCCESS(status);
    printf("RdStatus: %02x %02x %02x %02x%02x%02x \n\n", bRdStatus[0],bRdStatus[1],bRdStatus[2],bRdStatus[3],bRdStatus[4],bRdStatus[5]);

	/**
	 * Send a getVersion to the SAM
	 */
    aTxBuffer[PHHAL_HW_SAMAV2_ISO7816_CLA_POS] = PHHAL_HW_SAMAV2_ISO7816_CLA_BYTE;  /**< Cla Byte of SamAV2 Commands */
    aTxBuffer[PHHAL_HW_SAMAV2_ISO7816_INS_POS] = PHHAL_HW_SAMAV2_CMD_GET_VERSION_INS;  /**< CMD Byte for GetVersion command */
    aTxBuffer[PHHAL_HW_SAMAV2_ISO7816_P1_POS] = PHHAL_HW_SAMAV2_ISO7816_DEFAULT_P1_BYTE;  /**< Default Ins Byte of SamAV2 Commands */
    aTxBuffer[PHHAL_HW_SAMAV2_ISO7816_P2_POS] = PHHAL_HW_SAMAV2_ISO7816_DEFAULT_P2_BYTE;  /**< Default Ins Byte of SamAV2 Commands */
    aTxBuffer[PHHAL_HW_SAMAV2_ISO7816_LE_NO_LC_POS] = PHHAL_HW_SAMAV2_ISO7816_DEFAULT_LE_BYTE;  /**< Default Le Byte of SamAV2 Commands */
    wTxBufferLength = 0x05;
	pBuffer = aRxBuffer;
    status = phhalHw_Rd710_Cmd_CC_TransmitData(&halRd710, aTxBuffer, wTxBufferLength, &pBuffer, &wRxBufferLength);
    CHECK_SUCCESS(status);

    /**
	 * Switch off the field
	 */
    status = phhalHw_FieldOff(&halRd710);
    CHECK_SUCCESS(status);

    /**
	 * we are finished, release bal handle
	 */
    status = phbalReg_ClosePort(&balPcsc);
    CHECK_SUCCESS(status);

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    _getch();

    /**
	 * Returns zero if the procedure was successful.
	 */
    return 0;
}

