/*
 * Copyright 2013, 2016, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/* Header for this file */
#include "Example-Rd70xBasic.h"

/* Reader Library Headers */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
/**
 * Standard Header
 */
#ifdef _WIN32
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#endif
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#ifdef _WIN32
#pragma warning(pop)            /* PRQA S 3116 */
#endif

#pragma warning(disable:4996)

#define KEYCOUNT 2
#define KEYVERSIONS 1

static const uint8_t bSamHostKey[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
static const uint8_t bUlcKey[] = {0x49, 0x45, 0x4D, 0x4B, 0x41, 0x45, 0x52, 0x42, 0x21, 0x4E, 0x41, 0x43, 0x55, 0x4F, 0x59, 0x46};

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

void PRINT_BUFFER(uint8_t * pBuffer, uint8_t bLength)
{
    uint8_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}

int __cdecl main(int argc, char *argv[])
{
    phStatus_t status;
    uint8_t bReaderListPegoda[512];
    uint8_t bReaderListSam[512];
    uint8_t bHalBufferSam[300];
    uint8_t bHalBufferReader[0xFFFF];
	uint8_t bAtr[256];
    uint8_t bSak[1];
    uint8_t bUid[10];
    uint8_t bData[256];
    uint8_t pSeed[8];
    uint8_t bMoreCardsAvaliable;
    uint8_t bIndex;
    uint8_t bLength;
    uint8_t bUseSam;
    uint16_t wCount;
    char * pArgument;
    int iChoice;
    int dIndex;

    phKeyStore_Sw_KeyEntry_t keyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t keyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t keyUsage[KEYCOUNT];

    /* Bfl data parameter storage */
    phbalReg_PcscWin_DataParams_t balSam;
    phbalReg_Rd70xUsbWin_DataParams_t balReader;
    phhalHw_SamAV2_DataParams_t halSam;
    phhalHw_Rd70x_DataParams_t halReader;
    phpalI14443p3a_Sw_DataParams_t I14443p3a;
    phpalI14443p4a_Sw_DataParams_t I14443p4a;
    phpalI14443p4_Sw_DataParams_t I14443p4;
    phpalMifare_Sw_DataParams_t palMifare;
    phCryptoSym_Sw_DataParams_t cryptoEnc;
    phCryptoSym_Sw_DataParams_t cryptoMac;
    phCryptoSym_Sw_DataParams_t cryptoSymRnd;
    phCryptoRng_Sw_DataParams_t cryptoRng;
    phKeyStore_Sw_DataParams_t keyStore;
    phalMful_Sw_DataParams_t alMful;
    void * pHal;

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /* Iterate through all arguments */
    for (bIndex = 0; bIndex < (uint8_t)argc; ++bIndex);
    {
        /* Retrive next argument */
        pArgument = argv[bIndex];

        /* */
        /* Insert argument handling here */
        /* */
    }

    /* init. crypto */
    status = phCryptoSym_Sw_Init(&cryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoMac, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRnd, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng, sizeof(phCryptoRng_Sw_DataParams_t), &cryptoSymRnd);
    CHECK_SUCCESS(status);
    memset(pSeed, 0x00, 0x08);
    status = phCryptoRng_Seed(&cryptoRng, pSeed, 8);
    CHECK_SUCCESS(status);

    /* init. keystore */
    status = phKeyStore_Sw_Init(
        &keyStore,
        sizeof(phKeyStore_Sw_DataParams_t),
        keyEntry,
        KEYCOUNT,
        keyVersion,
        KEYVERSIONS,
        keyUsage,
        KEYCOUNT);

    CHECK_SUCCESS(status);

    /* Initialise the Reader BAL component */
    status = phbalReg_Rd70xUsbWin_Init(&balReader, sizeof(phbalReg_Rd70xUsbWin_DataParams_t));
    CHECK_SUCCESS(status);

    /* Get list of connected pegoda readers */
    status = phbalReg_GetPortList(&balReader, sizeof(bReaderListPegoda), bReaderListPegoda, &wCount);
    CHECK_SUCCESS(status);

    /* Connect to the first reader */
    status = phbalReg_SetPort(&balReader, bReaderListPegoda);
    CHECK_SUCCESS(status);

    /* open the reader port */
    status = phbalReg_OpenPort(&balReader);
    CHECK_SUCCESS(status);

    /* init the Reader HAL component */
    status = phhalHw_Rd70x_Init(
        &halReader,
        sizeof(phhalHw_Rd70x_DataParams_t),
        &balReader,
        bHalBufferReader,
        sizeof(bHalBufferReader),
        bHalBufferReader,
        sizeof(bHalBufferReader));

    CHECK_SUCCESS(status);

    /* The default reader is the Pegoda */
    pHal = &halReader;

    printf("----------------\n");
    printf("SAM Mode\n");
    printf("----------------\n");
    printf("[x] No SAM\n");
    printf("[1] SAM Non-X\n");
    printf("----------------\n");
	printf("Choice:");
    iChoice = _getch();
	printf("\n\n");

    if (iChoice == '1')
    {
        bUseSam = 1;

        /* Init. PCSC BAL */
        status = phbalReg_PcscWin_Init(&balSam, sizeof(phbalReg_PcscWin_DataParams_t), bAtr, sizeof(bAtr));
        CHECK_SUCCESS(status);

        /* Get list of connected PCSC readers */
        status = phbalReg_GetPortList(&balSam, sizeof(bReaderListSam), bReaderListSam, &wCount);
        CHECK_SUCCESS(status);

        /* Print PCSC driver list */
        printf("------------------\n");
        printf("Available Readers:\n");
        printf("------------------\n");
        pArgument = (char*)bReaderListSam;
        for (bIndex = 0; bIndex < wCount; bIndex++)
        {
            printf("[%d] %s\n", bIndex, pArgument);
            pArgument += strlen(pArgument) + 1;
        }
        printf("------------------------------------\n");
        printf("Choice:");
        dIndex = _getch() - '0';
        printf("\n\n");

        /* Advance to selected reader */
        pArgument = (char*)bReaderListSam;
        for (dIndex; dIndex > 0; --dIndex)
        {
            pArgument += strlen(pArgument) + 1;
        }

        /* Select the desired reader */
        status = phbalReg_SetPort(&balSam, (uint8_t*)pArgument);
        CHECK_SUCCESS(status);

        /* open the SAM port */
        status = phbalReg_OpenPort(&balSam);
        CHECK_SUCCESS(status);

        /* init the SAM HAL component */
        status = phhalHw_SamAV2_Init(
            &halSam,
            sizeof(phhalHw_SamAV2_DataParams_t),
            &balSam,
            pHal,
            &keyStore,
            &cryptoEnc,
            &cryptoMac,
            &cryptoRng,
            PHHAL_HW_SAMAV2_OPMODE_NON_X,
            0x00,
            bHalBufferSam,
            sizeof(bHalBufferSam),
            bHalBufferSam,
            sizeof(bHalBufferSam));

        CHECK_SUCCESS(status);

        /* Change the reader to SAM */
        pHal = &halSam;

        /* Load SAM AES128 key at position 0 in KeyStore */
        status = phKeyStore_FormatKeyEntry(&keyStore, 0, PH_KEYSTORE_KEY_TYPE_AES128);
        CHECK_SUCCESS(status);
        status = phKeyStore_SetKeyAtPos(&keyStore, 0x00, 0, PH_KEYSTORE_KEY_TYPE_AES128, (uint8_t*)bSamHostKey, 0);
        CHECK_SUCCESS(status);

        /* Detect SAM settings */
        status = phhalHw_SamAV2_DetectMode(&halSam);
        CHECK_SUCCESS(status);

        /* Perform HOST AUTHENTICATION */
        status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(
            &halSam,
            0x00,       /* AuthType: Default */
            0,          /* Host Key Number */
            0x00,       /* Host Key Version */
            0,          /* Sam Key Number */
            0x00,       /* Sam Key Version */
            NULL,       /* No Diversification input */
            0);

        CHECK_SUCCESS(status);
    }
    else
    {
        bUseSam = 0;
    }

    /* init the 14443-3A component */
    status = phpalI14443p3a_Sw_Init(&I14443p3a, sizeof(phpalI14443p3a_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init the 14443-4A component */
    status = phpalI14443p4a_Sw_Init(&I14443p4a, sizeof(phpalI14443p4a_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init the 14443-4 component */
    status = phpalI14443p4_Sw_Init(&I14443p4, sizeof(phpalI14443p4_Sw_DataParams_t), pHal);
    CHECK_SUCCESS(status);

    /* init. mifare pal */
    status = phpalMifare_Sw_Init(&palMifare, sizeof(phpalMifare_Sw_DataParams_t), pHal, &I14443p4);
    CHECK_SUCCESS(status);

    /* init. Ultralight-C component */
    status = phalMful_Sw_Init(&alMful, sizeof(phalMful_Sw_DataParams_t), &palMifare, &keyStore, &cryptoEnc, &cryptoRng);
    CHECK_SUCCESS(status);

    /* Configure HAL for Type-A cards */
    status = phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO14443A);
    CHECK_SUCCESS(status);

    printf("--------------------\n");
    printf("ISO 14443-3A Example\n");
    printf("--------------------\n");
    printf("[x] Skip\n");
    printf("[1] Run\n");
    printf("----------\n");
    printf("Choice:");
    iChoice = _getch();
    printf("\n\n");

    if (iChoice == '1')
    {
        /* Switch on / reset the field */
        status = phhalHw_FieldReset(pHal);
        CHECK_SUCCESS(status);

        /* Do some activation tests */
        do
        {
            /* Activate Layer 3 card */
            status = phpalI14443p3a_ActivateCard(&I14443p3a, NULL, 0x00, bUid, &bLength, bSak, &bMoreCardsAvaliable);
            CHECK_SUCCESS(status);

            /* Halt Layer 3 card */
            status = phpalI14443p3a_HaltA(&I14443p3a);
            CHECK_SUCCESS(status);
        }
        while (bMoreCardsAvaliable);

        /* switch on time mesaurement */
        status = phhalHw_SetConfig(&halReader, PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_COMM);
        CHECK_SUCCESS(status);

        /* Wake the last card up */
        status = phpalI14443p3a_ActivateCard(&I14443p3a, bUid, bLength, bUid, &bLength, bSak, &bMoreCardsAvaliable);
        CHECK_SUCCESS(status);

        /* get the timing */
        status = phhalHw_GetConfig(&halReader, PHHAL_HW_CONFIG_TIMING_US, &wCount);
        CHECK_SUCCESS(status);
        printf("Timing: %dus\n", wCount);

        /* switch off time mesaurement */
        status = phhalHw_SetConfig(&halReader, PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_OFF);
        CHECK_SUCCESS(status);
    }

    printf("---------------------------\n");
    printf("MIFARE Ultralight-C Example\n");
    printf("---------------------------\n");
    printf("[x] Skip\n");
    printf("[1] Run\n");
    printf("----------\n");
    printf("Choice:");
    iChoice = _getch();
    printf("\n\n");

    if (iChoice == '1')
    {
        /* Switch on / reset the field */
        status = phhalHw_FieldReset(pHal);
        CHECK_SUCCESS(status);

        /* Activate Layer 3 card */
        status = phpalI14443p3a_ActivateCard(&I14443p3a, NULL, 0x00, bUid, &bLength, bSak, &bMoreCardsAvaliable);
        CHECK_SUCCESS(status);

        /* Load Mifare Ultralight key at position 2 in KeyStore */
        status = phKeyStore_FormatKeyEntry(&keyStore, 1, PH_KEYSTORE_KEY_TYPE_2K3DES);
        CHECK_SUCCESS(status);
        status = phKeyStore_SetKeyAtPos(&keyStore, 1, 0, PH_KEYSTORE_KEY_TYPE_2K3DES, (uint8_t*)bUlcKey, 0x00);
        CHECK_SUCCESS(status);

        /* Perform Ultralight-C Authentication */
        printf("Ultralight C Authenticate:\n");
        status = phalMful_UlcAuthenticate(&alMful, 0x00, 1, 0, NULL, 0);
        CHECK_SUCCESS(status);

        /* Perform Ultralight Read */
        printf("Ultralight Read:\n");
        for (bIndex = 0x00; bIndex <= 0x28; ++bIndex)
        {
            status = phalMful_Read(&alMful, bIndex, bData);
            CHECK_SUCCESS(status);
            PRINT_BUFFER(bData, 16);
        }

        /* Perform Ultralight Write */
        printf("Ultralight Write...\n");
        for (bIndex = 0x04; bIndex <= 0x27; ++bIndex)
        {
            memcpy(bData, "\x01\x02\x03\x04", 4);
            status = phalMful_Write(&alMful, bIndex, bData);
            CHECK_SUCCESS(status);
        }

        /* Perform Ultralight Read */
        printf("Ultralight Read:\n");
        for (bIndex = 0x04; bIndex <= 0x27; ++bIndex)
        {
            status = phalMful_Read(&alMful, bIndex, bData);
            CHECK_SUCCESS(status);
            PRINT_BUFFER(bData, 16);
        }

        /* Perform Ultralight Compatibility Write */
        printf("Ultralight Compatibility Write...\n");
        for (bIndex = 0x04; bIndex <= 0x27; ++bIndex)
        {
            memcpy(bData, "\x0A\x0B\x0C\x0D", 4);
            status = phalMful_CompatibilityWrite(&alMful, bIndex, bData);
            CHECK_SUCCESS(status);
        }

        /* Perform Ultralight Read */
        printf("Ultralight Read:\n");
        for (bIndex = 0x04; bIndex <= 0x27; ++bIndex)
        {
            status = phalMful_Read(&alMful, bIndex, bData);
            CHECK_SUCCESS(status);
            PRINT_BUFFER(bData, 16);
        }
    }

    /* Switch off the field */
    status = phhalHw_FieldOff(pHal);
    CHECK_SUCCESS(status);

    if (bUseSam)
    {
        /* we are finished, release bal handle */
        status = phbalReg_ClosePort(&balSam);
        CHECK_SUCCESS(status);
    }

    /* we are finished, release bal handle */
    status = phbalReg_ClosePort(&balReader);
    CHECK_SUCCESS(status);

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    iChoice = _getch();

    /* Returns zero if the procedure was successful. */
    return 0;
}
