/*
 * Copyright 2013 - 2018, 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for MIFARE DESFire EVx.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/*******************************************************************************************************************************
 * case 1: creating an Application ,Authenticating using EVxSecureMessaging,creating a value file and transactions on value file
 * case 2: creation of a  Delegated Application,creating a standard file and checking the Free& Format commands
 * case 3: creating an  Application with keysetparameters ,creating a standard data file and  read &write of data into that file
 * case 4: creation of  a Delegated Application
 * case 5: creation of two applications and accessing files in each application by selecting both applications at a time
 * case 6: Updating the records in Linear Record File
 * case 7: Demonstrating Originality Check in Symmetric mode
 * case 8: Demonstrating Originality Check in ASymmetric mode
 * case 9: Demonstrating ISO 7816 commands with Extended APDU Lengths(Lc > 1 and Le > 1).
 *******************************************************************************************************************************/

/* Header for this file */


/* Reader Library Headers */
#include <phbalReg.h>
#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phpalMifare.h>
#include <phalVca.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoASym.h>
#include <phCryptoSym.h>
#include <time.h>
#include "../../intfs/phalMfdfEVx.h"
#include "../../intfs/phTMIUtils.h"
#include "../../intfs/phalVca.h"
#include "Example-MfdfEVx.h"

#pragma warning(disable:4996)

#define KEYCOUNT 21
#define KEYVERSIONS 1
#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */
#ifdef NXPBUILD__PH_LOG


phLog_RegisterEntry_t logRegisterEntries[16];
phLog_LogEntry_t logEntries[48];
#endif /* NXPBUILD__PH_LOG */

/* Utility function to check return status */
#define CHECK_SUCCESS(x)                        \
	if ((x) != PH_ERR_SUCCESS)                  \
{                                               \
	printf("An error occurred (0x%04X)", (x));	\
	return 0;                                   \
}

static uint8_t gaHalRxReader[512U];
static uint8_t gaHalTxReader[512U];
static uint8_t gbGenBuffer[1024];
static uint8_t pISOFileName[16];
static uint8_t bISOFileLen;
static uint8_t bRunOnMifareApplet;

/* Utility function to print a buffer */
void PRINT_BUFFER(uint8_t * pBuffer, uint16_t wLength)
{
	uint16_t wIndex;

	for (wIndex = 0; wIndex < wLength; ++wIndex)
	{
		printf("%02X ", pBuffer[wIndex]);
	}
	printf("\n");
}

/* Logging function. Turned OFF currently */
#ifdef NXPBUILD__PH_LOG
void phalMfdfEVx_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
	uint16_t i;
	PHAL_MFDFEVX_UNUSED_VARIABLE(bOption);
	PHAL_MFDFEVX_UNUSED_VARIABLE(pDataParams);

	for (i = 0; i < wEntryCount; i++ )
	{
		if ( i != 0 )
		{
			printf("%s : ", pEntries[i].pString);
		}
		else
		{
			printf("-----------------------------------------\n");
			printf("%s \n", pEntries[0].pString);
			printf("-----------------------------------------\n");
		}
		PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
	}
	printf("\n\n");
}
void phalvca_log(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
	uint16_t i;
	PHAL_MFDFEVX_UNUSED_VARIABLE(bOption);
	PHAL_MFDFEVX_UNUSED_VARIABLE(pDataParams);
	for (i = 0; i < wEntryCount; i++ )
	{
		if ( i != 0 )
		{
			printf("%s : ", pEntries[i].pString);
		}
		else
		{
			printf("-----------------------------------------\n");
			printf("%s \n", pEntries[0].pString);
			printf("-----------------------------------------\n");
		}
		PRINT_BUFFER((uint8_t *)(pEntries[i].pData), pEntries[i].wDataLen);
	}
	printf("\n\n");
}
#endif /* NXPBUILD__PH_LOG */

#define AMKEY           0       /* Application Master Key number. */
#define ACKEY           1       /* Application Common Key number. */
#define ARKEY           2       /* Application Read Key number. */
#define AWKEY           3       /* Application Write Key number. */
#define AAKEY           4       /* Application Accreditation Key number. */
#define PICCKEY         5       /* PICC Key entry number in key store. */
#define PICCAESKEY      6       /* New PICC (AES128) Key entry number in key store. */
#define AMKEY_NEW       7       /* New Application Master Key entry number in Key store. */
#define SYSTEM_KEY      8       /* System key entry number in key store. */
#define NPKEY0          9       /* Next PACS App Key 0. */
#define NPKEY1          10      /* Next PACS App Key 1. */
#define SYSTEM_KEY_NP   11      /* Next PACS System key. */
#define DEFAULT_3K3DES  12      /* Default 3K3DES KEY. All zeros. */
#define NPKEY_FOR_UID   13      /* Used to retrieve UID */
#define AFCKEY_FOR_UID  14      /* Used to retrieve UID */
#define PICCDAMAUTHKEY  15      /* PICC DAM Auth Key for DAM */
#define PICCORIGINALITYKEY 16	/* PICC Originality Key */
#define VCCONFIGKEY        17	/* PICC VC Configuration Key */
#define VCPROXIMITYKEY     18	/* PICC VC Proximity Key */
#define VCSELECTMACKEY     19	/* PICC VC Select MAC Key */
#define VCSELECTENCKEY     20	/* PICC VC Select ENC Key */

#define STDDATAFILE1    1       /* File number of Standard data file 1. */
#define STDDATAFILE2    2       /* File number of Standard data file 2. */
#define BCKUPDATAFILE1  6       /* File number od Backup data file.*/
#define BCKUPDATAFILE2  7       /* File number of backup datafile. */
#define VALUEFILE       3       /* File number of Value file. */
#define RECORDFILE      4       /* File number of Record file. */
#define TMACFILE        5       /* File number of Record file. */

#define NPCARDOBJ       1       /* File number of NP Card object. */
#define NPPACSOBJ       2       /* File number of NP PACS object. */
#define TMI_BUFFER_SIZE 255     /* TMI Buffer Size */
#define IID_KEY_COUNT   0x13U   /* number of IID keys */
#define VCA_AUTH_RND_LEN   36   /* Size of the Virtual Card and Proximity Check Random numbers */

/* Default PICC key */
static uint8_t bDefaultPICCKey[16] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

/* AES PICC key */
static uint8_t bAESPICCKey[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Master key of AFC App */
static uint8_t bAMKey[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Master key of AFC App */
static uint8_t bACKey[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static uint8_t bPICCKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static uint8_t bPICCORIGNINALITYKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Configuration Key */
static uint8_t bVCCONFIGKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Proximity Key */
static uint8_t bVCPROXIMITYKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Select MAC Key */
static uint8_t bVCSELECTMACKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC VC Select ENC Key */
static uint8_t bVCSELECTENCKEY[16] =     {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* PICC DAM Auth Key for DAM */
static uint8_t  bPICCDAMAUTHKEY[16]= {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Application Common Key of AFC App */
static uint8_t bACKEY[16] =     {
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
	0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11
};
/* Application Read Key of AFC App */
static uint8_t bARKey[16] =     {
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
	0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};
/* Application Write Key of AFC App */
static uint8_t bAWKey[16] =     {
	0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33,
	0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x33
};
/* Application Accreditation Key of AFC App */
static uint8_t bAAKey[16] =     {
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44
};
/* New Application Master key of AFC App */
static uint8_t bAMKey_New[16] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* AFC App Key to retrieve UID */
static uint8_t bAfcKeyN[24]  =   {
	0x1A, 0x2B, 0x3C, 0x4D, 0x5E, 0x6F, 0x70, 0x81,
	0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
	0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98
};

/* System key of AFC App */
static uint8_t bSysKey[16] =    {
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Next PACs System key */
static uint8_t bDEFAULT_3K3DES[24] =  {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
/* Default 3K 3DES Key */
static uint8_t bSysKeyNP[24] =  {
	0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xCA, 0xCB,
	0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
	0x09, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16
};
/* Data for AFC App data file 1 */
static uint8_t bData1[32] =     {
	0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8,
	0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
	0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8,
	0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8
};
/* Data for AFC App data file 2 */
static uint8_t bData2[28] =     {
	0x4D, 0x55, 0x53, 0x54, 0x45, 0x52, 0x3C, 0x53,
	0x41, 0x4D, 0x50, 0x4C, 0x45, 0x3C, 0x3C, 0x01,
	0x07, 0x80, 0x01, 0x07, 0x09, 0x30, 0x06, 0x10,
	0x00, 0x00, 0x00, 0x00
};
/* NextPac key 0 */
static uint8_t bNPKey0[24]  =   {
	0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x00, 0x11,
	0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
	0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};
/* NextPac key 1 */
static uint8_t bNPKey1[24]  =   {
	0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF, 0xE0, 0xE1,
	0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9,
	0xC0, 0xD0, 0xE0, 0xF0, 0x20, 0x30, 0x40, 0x50
};

/* NP Key to retrieve UID */
static uint8_t bNPKey2[24]  =   {
	0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xC0, 0xA1,
	0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9,
	0x10, 0x20, 0x30, 0x40, 0x50, 0x60, 0x70, 0x80
};


/* MASTER APPLICATION ID */
static uint8_t bAfcMApp[3] = {0x00, 0x00, 0x00};
/* AFC APPLICATION id */
static uint8_t bAfcApp[3] = {0x2A, 0x8C, 0xF1};
/* AFC APPLICATION2 id */
static uint8_t bAfcApp2[3] = {0x02, 0x02, 0x02};

/* Delegated AFC APPLICATION id */
static uint8_t bAfcDelApp[3] = {0x01, 0x01, 0x01};

static uint8_t bCardUid[10];

static uint8_t bVersionString[28];

static uint8_t aBuffer[512];							  /**< ASymmetric Crypto Buffer */

/* Public Key */
static uint8_t gaPublicKey_DFEV2[] = {
    0x04, 0xB3, 0x04, 0xDC, 0x4C, 0x61, 0x5F, 0x53, 0x26, 0xFE, 0x93, 0x83, 0xDD, 0xEC, 0x9A, 0xA8,
    0x92, 0xDF, 0x3A, 0x57, 0xFA, 0x7F, 0xFB, 0x32, 0x76, 0x19, 0x2B, 0xC0, 0xEA, 0xA2, 0x52, 0xED,
    0x45, 0xA8, 0x65, 0xE3, 0xB0, 0x93, 0xA3, 0xD0, 0xDC, 0xE5, 0xBE, 0x29, 0xE9, 0x2F, 0x13, 0x92,
    0xCE, 0x7D, 0xE3, 0x21, 0xE3, 0xE5, 0xC5, 0x2B, 0x3A
};

static uint8_t gaPublicKey_DFEV3[] = {
    0x04, 0x1D, 0xB4, 0x6C, 0x14, 0x5D, 0x0A, 0x36, 0x53, 0x9C, 0x65, 0x44, 0xBD, 0x6D, 0x9B, 0x0A,
    0xA6, 0x2F, 0xF9, 0x1E, 0xC4, 0x8C, 0xBC, 0x6A, 0xBA, 0xE3, 0x6E, 0x00, 0x89, 0xA4, 0x6F, 0x0D,
    0x08, 0xC8, 0xA7, 0x15, 0xEA, 0x40, 0xA6, 0x33, 0x13, 0xB9, 0x2E, 0x90, 0xDD, 0xC1, 0x73, 0x02,
    0x30, 0xE0, 0x45, 0x8A, 0x33, 0x27, 0x6F, 0xB7, 0x43
};

/* This 33 bytes string is used for Originality check.
 * NOTE: This should be provided by product team once the key is finalized.
 */
const char *publickey_str = "046F70AC557F5461CE5052C8E4A7838C11C7A236797E8A0730A101837C004039C";

/* TMC & TMV */
uint8_t bTMC[4] = {0x00,0x00,0x00,0x00};
uint8_t bTMV[8] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
uint8_t bTMIBuffer[TMI_BUFFER_SIZE];

/* Creates and pre personalizes the AFC Application */
phStatus_t PrePersonalization_AFCApp(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx )
{
	phStatus_t status;
	uint8_t bLength;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint8_t bLLimit[4];
	uint8_t bULimit[4];
	uint8_t bValue[4];
	uint8_t bAccessRights[2];
	uint8_t bLimitedCredit = 0;
	uint8_t bConfigByte;
	uint8_t bLenPcdCapsIn=6;                                    /**< [In] Length of PcdCapsIn. Always zero for following authentication */
	uint8_t bPcdCapsIn[6] = {0x00,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsOut[6] ;                                    /**< [Out] PCD Capabilities. 6 bytes. */
	uint8_t bPdCapsOut[6] ;                                     /**< [Out] PD Capabilities. 6 bytes. */
	uint8_t bTMKey[16];
	uint8_t bTMKeyVer;
	printf("Performing Pre Personalization ...... \n");
	/* ####################Pre personalization############################### */

	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCKEY,                                    /* [In]KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x00,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/* Set Configuration - default key and version */
	bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION1; /* option to set default key and version */
	memset(gbGenBuffer, 0x00, 24);             /* Default key */
	gbGenBuffer[24] = 0x00;                    /* default key version */
	bLength = 25;                              /* length */
	status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
		bOption,
		gbGenBuffer,
		bLength);
	CHECK_SUCCESS(status);

	/* Set Configuration - Configure for Random UID */
	bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION0;          /* option to set configuration byte */
	bConfigByte = 0x02;                                 /* 00000010 : Random ID Enabled */
	status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
		bOption,
		&bConfigByte,
		0x01);
	CHECK_SUCCESS(status);
	/* Get the card UID - wk423, As of now it is not working with Csharp Simulator, rev 5311. */
	/* Please uncomment later once it is working*/
	status = phalMfdfEVx_GetCardUID(pAlMfdfEVx,      /* [In] Pointer to parameters data structure */
					0x00,
					0x00,
					bCardUid);                          /* [Out] 7 Byte Card UID */
	CHECK_SUCCESS(status);

	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;
	/* Iso File Id = 0x00AA. Sent LSB first */
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;
	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);
	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;
	/* Keysettings 2 = 10100110
	* AES crypto + supports 2 byte ISO fids +
	* max 6 keys in this app
	*/
	bKeySettings2 = 0xB6;
	/* Keysetting 3 = 0000 0001. b[0] indicates App key set presence */
	bKeySettings3 = 0x01;
	/* Key Set Values */
	bKeySetValues[0] = 0x01;         /* byte0 = AKS ver */
	bKeySetValues[1] = 0x02;         /* byte1 = #keysets */
	bKeySetValues[2] = 0x10;         /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00;         /* byte3 = AppKeySetSettings */
	/* Create AFC Application */
	status = phalMfdfEVx_CreateApplication( pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                           /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                           /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                     /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                     /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                     /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                 /* [In] 4 byte keyset values.  */
		bISOFileId,                                        /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                        /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                           /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);
	/* Select the AFC Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,
		0x00,     /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,  /* [In] 3 byte AID, LSB First */
		0x00);    /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	/* EV2 First Authenticate */
	status = phalMfdfEVx_AuthenticateEv2(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		0x01,                                         /* first authentication */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);
	bLenPcdCapsIn=0;
	/* EV2 Following Authenticate */
	status = phalMfdfEVx_AuthenticateEv2(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		0x00,                                         /* Nonfirst Authentication*/
		PHAL_MFDFEVX_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCAESKEY,                                   /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0,                                            /* [In] Diversification input length */
		bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
		bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
		bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
		bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
	CHECK_SUCCESS(status);
	memset(bTMKey,0x00,16);
	bTMKeyVer = 0x00;
	/* accessRights For Transaction MAC File*/
	bAccessRights[0] = 0xF0;                          /* RW access = AAK, Change access rights = AMK  */
	bAccessRights[1] = 0x0F;                          /* Read access = AWKey, Write access = AWKey */

	/* create Transaction MAC File*/
	status = phalMfdfEVx_CreateTransactionMacFile(pAlMfdfEVx, /* [In] Pointer to parameters data structure */
		0,
		TMACFILE,                                             /* [In] File number */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                       /* [In] Communication settings of the file */
		bAccessRights,                                        /* [In] 2 Byte access rights. LSB First */
		PICCAESKEY,											  /* [In] Key no of the TMKey to be used. */
		PHAL_MFDFEVX_KEY_TYPE_AES128,                         /* [In] 1byte. Bit0, 1 indicate key type.. */
		bTMKey,                                               /* [In] Key value. 16 bytes. */
		bTMKeyVer,
		NULL,
		0
		);                                           /* [In] TM Key version. Value should be valid for AES key. Ignored for other key types. */
	CHECK_SUCCESS(status);

	/* Create Value File */
	bAccessRights[0] = 0x00;                         /* RW access = AAK, Change access rights = AMK  */
	bAccessRights[1] = 0x00;                         /* Read access = AWKey, Write access = AWKey */

	/* Create Value File */
	memset(bLLimit, 0x00, 4);
	memset(bULimit, 0x00, 4);
	memset(bValue, 0x00, 4);

	bLLimit[0] = 0x14;   /*Lower Limit  20. */
	bValue[0] = 0x14;    /*Initial value 20. */
	bULimit[0] = 0xF4;
	bULimit[1] = 0x01;   /*500 is UpperLimit. */

	/* FreeGetValue Enabled */
	bLimitedCredit = PHAL_MFDFEVX_ENABLE_FREE_GETVALUE;
	/* Create Value File */
	status = phalMfdfEVx_CreateValueFile( pAlMfdfEVx,         /* [In] Pointer to parameters data structure */
		VALUEFILE,                                            /* [In] File number */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                       /* [In] Communication settings of the file */
		bAccessRights,                                        /* [In] 2 Byte access rights. LSB First */
		bLLimit,                                              /* [In] Lower Limit. 4 Byte value LSB First */
		bULimit,                                              /* [In] Upper Limit. 4 Byte value LSB First */
		bValue,                                               /* [In] 4 Byte value LSB First */
		bLimitedCredit                                        /* [In] Limited Credit and free getvalue setting. */
		);
	CHECK_SUCCESS(status);
	/* ####################End Pre personalization########################### */
	printf("\tPre Personalization DONE! \n");
	return 0;
}



/* Credits a given value to the AFC Application */
phStatus_t Accreditation_WithValue(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx,
	uint8_t * pValue)
{
	phStatus_t status;
	uint8_t bValue[4];
	uint32_t dwVal1;

	/* ########################## Accreditation ################################### */
	printf("Performing Accreditation in AFC App.... \n");
	/* Authenticate with the Write Key of Value File */
	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,               /* [In] Diversification option */
		PICCAESKEY,                                    /* [In] KEY number in key store */
		0x00,                                          /* [In] Key version in key store */
		0x00,                                          /* [In] Key number on Card */
		NULL,                                          /* [In] Diversification input */
		0);                                            /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	memset(bValue, 0, 4);
	/* Get Current Value */
	status = phalMfdfEVx_GetValue(pAlMfdfEVx,         /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,             /* [In] Communication option. */
		VALUEFILE,                                    /* [In] 1 byte file number. */
		bValue);                                      /* [Out] 4 Byte array to store the value read out. LSB First. */
	CHECK_SUCCESS(status);

	dwVal1 = bValue[3];
	dwVal1 = dwVal1 << 8 | bValue[2];
	dwVal1 = dwVal1 << 8 | bValue[1];
	dwVal1 = dwVal1 << 8 | bValue[0];

	/* Credit the value provided */
	memcpy(bValue, pValue, 4);

	/* Add money (100) to the account */
	status = phalMfdfEVx_Credit(pAlMfdfEVx,     /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,         /* [In] Communication option. */
		VALUEFILE,                              /* [In] 1 byte file number. */
		bValue);                                /* [In] 4 byte value array. LSB first. */
	CHECK_SUCCESS(status);
	/* Commit after credit*/
	status = phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		0x01,                                            /* [In] Set to 1 to get TMC & TMV */
		bTMC,                                            /* [Out] 4 byte TMAC counter. */
		bTMV);                                           /* [Out] 8 byte TMAC value. */
	CHECK_SUCCESS(status);
	/* Performing Get Value After Credit */
	status = phalMfdfEVx_GetValue(pAlMfdfEVx,    /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,        /* [In] Communication option. */
		VALUEFILE,                               /* [In] 1 byte file number. */
		bValue);                                 /* [Out] 4 Byte array to store the value read out. LSB First. */
	CHECK_SUCCESS(status);

	printf(" The amount in your account  After credit is %x %x %x %x \n", bValue[3], bValue[2], bValue[1], bValue[0]);
	/* ########################## End Accreditation ############################### */
	printf("\tAccreditation DONE! \n");
	return 0;
}

/* Credits 100 units to the AFC Application */
phStatus_t Accreditation(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
	uint8_t bValue[4] = {100, 0, 0, 0};
	return Accreditation_WithValue(pAlMfdfEVx,
		bValue);
}

/* Checkout of AFC App. Debits 10 units from the AFC Application */
phStatus_t CheckOutAfcApp(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t bValue[4] = {0, 0, 0, 0};
	uint32_t dwVal1;

	dwVal1 = bValue[3];
	dwVal1 = dwVal1 << 8 | bValue[2];
	dwVal1 = dwVal1 << 8 | bValue[1];
	dwVal1 = dwVal1 << 8 | bValue[0];

	/* Prepare value to be debited */
	memset(bValue, 0, 4);
	bValue[0] = 10;
	/* Debit 10 units */
	status = phalMfdfEVx_Debit(pAlMfdfEVx,       /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,          /* [In] Communication option. */
		VALUEFILE,                               /* [In] 1 byte file number. */
		bValue);                                 /* [In] 4 byte value array. LSB first. */
	CHECK_SUCCESS(status);

	/* Commit after  Debiting the value */
	status = phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		0x01,                                            /* [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                            /* [Out] 4 byte TMAC counter. */
		bTMV);                                           /* [Out] 8 byte TMAC value. */
	CHECK_SUCCESS(status);
	/* GetValue After Debiting The Value */
	status = phalMfdfEVx_GetValue(pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,      /* [In] Communication option. */
		VALUEFILE,                             /* [In] 1 byte file number. */
		bValue);                               /* [Out] 4 Byte array to store the value read out. LSB First. */
	CHECK_SUCCESS(status);

	printf(" The amount in your account After Debit is %x %x %x %x \n", bValue[3], bValue[2], bValue[1], bValue[0]);
	printf("\tCheckOutAfcApp DONE! \n");
	return 0;
}
/*  This Function will create a Delegated Application and a Standard datafile of size>2KB
 *  And Checking the availability of memory  before and after formatting the card
 */

phStatus_t DemoGarbageCollection(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx){
	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint16_t wKeyVerDAMEnc= 0x00;
	uint16_t wKeyNoAppDAMDefault= 0x00;
	uint16_t wKeyVerAppDAMDefault=0x00;
	uint8_t bAppDAMDefaultKeyVer=0x00;
	uint8_t pDAMEncKey[32];
	uint16_t wKeyNoDAMMAC = 0x00;  /*TBD: Value to be decided */
	uint16_t wKeyVerDAMMAC = 0x00; /*TBD: Value to be decided */
	uint8_t bDamParams[5] = {0x00,0x01,0x01,0x80,0x00};
	uint8_t bDAMMAC[8];
	uint8_t bMemInfo[3];
	uint32_t dwFreeMem;
	uint8_t pISOFileId[2];
	uint8_t bFileSize[3];
	uint8_t bAccessRights[2];

	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,           /* [In] Diversification option */
		PICCKEY,                                   /* [In]KEY number in key store */
		0x00,                                      /* [In] Key version in key store */
		0x00,                                      /* [In] Key number on Card */
		NULL,                                      /* [In] Diversification input */
		0);                                        /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);
	/*Initilaise DAM Auth Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		PICCKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		PICCKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x10,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/*Initilaise DAM MAC Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		PICCKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		PICCKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x11,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/*Initilaise DAM ENC Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		PICCKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		PICCKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x12,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first */
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;

	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);

	/* Application masterkey settings 00001111
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;

	/* Keysettings 2 = 00100110
	* 2K 3DES crypto + supports 2 byte ISO fids +
	* max 6 keys in this app
	*/
	bKeySettings2 = 0x26;

	/*	Keysetting 3 = 0000 0000
	*	bit[0] = 1 - Key set values present
	*/
	bKeySettings3 = 0x00;
	/* Key Set Values */
	bKeySetValues[0] = 0xAA; /* byte0 = AKS ver */
	bKeySetValues[1] = 0x00; /* byte1 = #keysets */
	bKeySetValues[2] = 0x00; /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00; /* byte3 = AppKeySetSettings */
	/* generate DAM encryption key */
	status = phalMfdfEVx_GenerateDAMEncKey( pAlMfdfEVx,
		PICCDAMAUTHKEY,
		wKeyVerDAMEnc,
		wKeyNoAppDAMDefault,
		wKeyVerAppDAMDefault,
		bAppDAMDefaultKeyVer,
		pDAMEncKey
		);
	CHECK_SUCCESS(status);
	/* Generate DAMMAC key first */
	status = phalMfdfEVx_GenerateDAMMAC(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		bOption,                                      /* [In] Option to indicate whether this application has ISO Fid and DF */
		wKeyNoDAMMAC,                                 /* [In] Key number in key store of DAM MAC Key */
		wKeyVerDAMMAC,                                /* [In] Key version in key store of DAM MAC Key*/
		bAfcDelApp,                                   /* [In] 3 byte AID, LSB First */
		bDamParams,                                   /* [In] 4 byte array. Byte0 = DAM Slot, Byte1 = DAMSlotVersion, Byte2, Byte 3  = QuotaLimit. Multiple of 32 byte block. */
		bKeySettings1,                                /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                            /* [In] 4 byte keyset values.  */
		bISOFileId,                                   /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                   /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen,                                       /* [In] Length of DF Name provided above */
		pDAMEncKey,                                   /* [In] 32 byte Encrypted initial application key. */
		bDAMMAC);                                     /* [Out] 8 byte DAM MAC. */
	CHECK_SUCCESS(status);
	/* Authenticate with the AUTH key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,           /* [In] Diversification option */
		PICCKEY,                                   /* [In]KEY number in key store */
		0x00,                                      /* [In] Key version in key store */
		0x10,                                      /* [In] Key number on Card */
		NULL,                                      /* [In] Diversification input */
		0);                                        /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Creating Delegated Application */
	status = phalMfdfEVx_CreateDelegatedApplication(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		bOption,                                                 /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcDelApp,                                              /* [In] 3 byte AID, LSB First */
		bDamParams,                                              /* [In] 4 byte array. Byte0 = DAM Slot, Byte1 = DAMSlotVersion, Byte2, Byte 3  = QuotaLimit. Multiple of 32 byte block. */
		bKeySettings1,                                           /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                           /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                           /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                       /* [In] 4 byte keyset values.  */
		bISOFileId,                                              /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                              /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen,                                                  /* [In] Length of DF Name provided above */
		pDAMEncKey,                                              /* [In] 32 byte Encrypted initial application key. */
		bDAMMAC);                                                /* [In] 8 byte DAM MAC. */
	CHECK_SUCCESS(status);

	/* Select this app, auth & formatPICC */
	/* Select the AFC Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                            /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcDelApp,                                      /* [In] 3 byte AID, LSB First */
		0x00);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);

	/*Standard File creation.*/
	pISOFileId[0] = 0x01;
	pISOFileId[1] = 0x01;

	/* File Size = 32 bytes */
	bFileSize[0] = 0x08;
	bFileSize[1] = 0x08;
	bFileSize[2] = 0x00;
	/* File AccessRights */
	bAccessRights[0] = 0x00;        /* Change access rights = AMK, RW access = never(for now AMK) */
	bAccessRights[1] = 0x10;        /* Read access = ACKey, Write access = never (for now AMK) */

	/* Create standard data file # 01 */
	status = phalMfdfEVx_CreateStdDataFile(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		0x01,                                             /* [In] Option to indicate ISO File Id provided */
		STDDATAFILE1,                                     /* [In] File number. 1 byte */
		pISOFileId,                                       /* [In] Two byte ISO File ID. LSB First */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                   /* [In] Communication settings of the file */
		bAccessRights,                                    /* [In] 2 Byte access rights. LSB First */
		bFileSize                                         /* [In] 3 Byte file size. LSB First */
		);
	CHECK_SUCCESS(status);

	memset(bMemInfo,0,3);
	/*  Get Available FreeMemory */
	status = phalMfdfEVx_FreeMem(pAlMfdfEVx,
		bMemInfo);                                       /* [Out] 3 bytes memory info. LSB first. */
	CHECK_SUCCESS(status);

	dwFreeMem = bMemInfo[2];
	dwFreeMem = dwFreeMem << 8 | bMemInfo[1];
	dwFreeMem = dwFreeMem << 8 | bMemInfo[0];
	printf("\n Available memory after creating the delegated Application  is %d bytes \n",dwFreeMem);

	/* Select the master Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                            /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcMApp,                                        /* [In] 3 byte AID, LSB First */
		0x00);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);

	/*Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,             /* [In] Diversification option */
		PICCKEY,                                     /* [In]TBD: Give DAMAuthKey no here */
		0x00,                                        /* [In] Key version in key store */
		0x00,                                        /* [In] Key number on Card */
		NULL,                                        /* [In] Diversification input */
		0);                                          /* [In] Diversification input length */

	CHECK_SUCCESS(status);

	/* Format card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/*   Get Available FreeMemory */
	status = phalMfdfEVx_FreeMem(pAlMfdfEVx,
		bMemInfo);                            /* [Out] 3 bytes memory info. LSB first. */
	CHECK_SUCCESS(status);

	dwFreeMem = bMemInfo[2];
	dwFreeMem = dwFreeMem << 8 | bMemInfo[1];
	dwFreeMem = dwFreeMem << 8 | bMemInfo[0];

	printf("\n Available memory after formating the PICC is %d bytes \n",dwFreeMem);

	return 0;
}

phStatus_t StoreKeysInKeyStore(phKeyStore_Sw_DataParams_t * pKeyStore)
{
	phStatus_t status;
	phKeyStore_Sw_DataParams_t keyStore;
	uint8_t bDefKey[24];

	keyStore = *pKeyStore;

	/* Set the default PICC Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCKEY, PH_CRYPTOSYM_KEY_TYPE_2K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_2K3DES,
		bPICCKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY, PH_CRYPTOSYM_KEY_TYPE_2K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AMKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_2K3DES,
		bAMKey,
		0x00);
	CHECK_SUCCESS(status);

	/* PICC DAM Auth Key for DAM */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCDAMAUTHKEY, PH_CRYPTOSYM_KEY_TYPE_DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCDAMAUTHKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_DES,
		bPICCDAMAUTHKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Common key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, ACKEY, PH_CRYPTOSYM_KEY_TYPE_DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		ACKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_DES,
		bACKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Read key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, ARKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		ARKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bARKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Write key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AWKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AWKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAWKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the Application Accreditation key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AAKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AAKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAAKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC AES128 key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCAESKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCAESKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAESPICCKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC ORIGINALITY CHECK AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, PICCORIGINALITYKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		PICCORIGINALITYKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bPICCORIGNINALITYKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new PICC VC Configuration Key AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCCONFIGKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCCONFIGKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCCONFIGKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC VC PROXIMITY KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCPROXIMITYKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCPROXIMITYKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCPROXIMITYKEY,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the new PICC VC Select MAC KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCSELECTMACKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCSELECTMACKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCSELECTMACKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new PICC VC Select ENC KEY AES128 key in the key store. */
	status = phKeyStore_FormatKeyEntry(&keyStore, VCSELECTENCKEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);
	status = phKeyStore_SetKeyAtPos(&keyStore,
		VCSELECTENCKEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bVCSELECTENCKEY,
		0x00);
	CHECK_SUCCESS(status);
	/* Set the new Application Master key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AMKEY_NEW, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AMKEY_NEW,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAMKey_New,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the SYSTEM key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		SYSTEM_KEY,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bSysKey,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the UID retriever key in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, AFCKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_AES128);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		AFCKEY_FOR_UID,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_AES128,
		bAfcKeyN,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NPKEY0 in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY0, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY0,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey0,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NPKEY1 in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY1, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY0,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey1,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the NP system KEY in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, SYSTEM_KEY_NP, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		SYSTEM_KEY_NP,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bSysKeyNP,
		0x00);
	CHECK_SUCCESS(status);

	memset(bDefKey, 0, 24);
	/* Set the default 3K3DES KEY in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, DEFAULT_3K3DES, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		DEFAULT_3K3DES,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bDEFAULT_3K3DES,
		0x00);
	CHECK_SUCCESS(status);

	/* Set the UID retriever KEY for NP App in the key store */
	status = phKeyStore_FormatKeyEntry(&keyStore, NPKEY_FOR_UID, PH_CRYPTOSYM_KEY_TYPE_3K3DES);
	CHECK_SUCCESS(status);

	status = phKeyStore_SetKeyAtPos(&keyStore,
		NPKEY_FOR_UID,
		0x00,
		PH_CRYPTOSYM_KEY_TYPE_3K3DES,
		bNPKey2,
		0x00);
	CHECK_SUCCESS(status);

	return 0;
}

/* Function to demonstrate Multiple Keysets and Keysets Rolling.                                   */
/* Steps:                                                                                          */
/* CreateApplication with additional key set parameters. Crypto type 2K3DES, Number of KeySets = 4,*/
/* MaxKeySize = 24, Specify the RollKeySett key in this command.                                   */
/* Authenticate with the Application Master key                                                    */
/* Create a data file that supports encrypted data transfer.                                       */
/* Perform read and write encrypted data after authentication with the corresponding key           */
/* InitializeKeySet for keyset number AKS + 1 as 3K3DES                                            */
/* Authenticate with ChangeKey                                                                     */
/* Invoke FinalizeKeySet                                                                           */
/* Re authenticate with the Application master key or the File Read key. This is now 3K3DES crypto.*/
/* Perform read/write operations on the file                                                       */
phStatus_t DemoKeySetRolling(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx){
	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint8_t bAccessRights[2];
	uint8_t pISOFileId[2];
	uint8_t bFileSize[3];
	uint8_t bKeySetNo = 0x1;
	uint8_t pTxData[4];
	uint8_t pTxDataLen[3];
	uint8_t  pOffset[3];
	uint8_t  *pRxData;
	uint16_t pRxDataLen;

	printf("Starting KeySetRolling application ...... \n");

	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCKEY,                                    /* [In]KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x00,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first */
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;

	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);

	/* Application masterkey settings 00011111
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x01
	*/
	bKeySettings1 = 0x1F;

	/* Keysettings 2 = 0011 0110
	* 00   - 2K3DES crypto
	* 1    - supports 2 byte ISO fids
	* 1    - Keyset parameters present
	* 0110 - max 6 keys in this app
	*/
	bKeySettings2 = 0x36;
	bKeySettings3 = 0x01;

	/* Key Set Values */
	bKeySetValues[0] = 0x00;        /* byte0 = AKS ver */ /* TBD */
	bKeySetValues[1] = 0x04;        /* byte1 = #keysets */
	bKeySetValues[2] = 0x18;        /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x01;        /* byte3 = AppKeySetSettings */

	/* Creating Application */
	status = phalMfdfEVx_CreateApplication( pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                           /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                           /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                     /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                     /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                     /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                 /* [In] 4 byte keyset values.  */
		bISOFileId,                                        /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                        /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                           /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);

	/* Select the AFC Application */
	status = phalMfdfEVx_SelectApplication( pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                             /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,                                          /* [In] 3 byte AID, LSB First */
		0x00);                                            /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);

	/* Authenticate */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                /* [In] Diversification option */
		AMKEY,                                          /* [In] KEY number in key store */
		0x00,                                           /* [In] Key version in key store */
		0x00,                                           /* [In] Key number on Card */
		NULL,                                           /* [In] Diversification input */
		0);                                             /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	/* Iso File ID of file */
	pISOFileId[0] = 0x01;
	pISOFileId[1] = 0x01;
	/* File Size = 32 bytes */
	bFileSize[0] = 0x20;
	bFileSize[1] = 0x00;
	bFileSize[2] = 0x00;
	/* File Access Rights */
	bAccessRights[0] = 0x11;        /* Change access rights = AMK, RW access = never(for now AMK) */
	bAccessRights[1] = 0x11;        /* Read access = ACKey, Write access = never (for now AMK) */

	/* Create standard data file # 01 */
	status = phalMfdfEVx_CreateStdDataFile( pAlMfdfEVx,     /* [In] Pointer to parameters data structure */
		0x01,                                               /* [In] Option to indicate ISO File Id provided */
		STDDATAFILE1,                                       /* [In] File number. 1 byte */
		pISOFileId,                                         /* [In] Two byte ISO File ID. LSB First */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                     /* [In] Communication settings of the file */
		bAccessRights,                                      /* [In] 2 Byte access rights. LSB First */
		bFileSize                                           /* [In] 3 Byte file size. LSB First */
		);
	CHECK_SUCCESS(status);
	/* Data Length -LSB First */
	pTxDataLen[0]=0x03;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;
	/* Data */
	pTxData[0]=10;
	pTxData[1]=0;
	pTxData[2]=11;
	/* Offset address */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;
	/* Authenticate  with Write Key of Standard Data File*/
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,             /* [In] Diversification option */
		ACKEY,                                       /* [In] KEY number in key store */
		0x00,                                        /* [In] Key version in key store */
		0x01,                                        /* [In] Key number on Card */
		NULL,                                        /* [In] Diversification input */
		0);                                          /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Writting Data into Standard File */
	status= phalMfdfEVx_WriteData(pAlMfdfEVx,      /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,            /* [In] Communication Mode. Plain, Mac'd or encrypted. */
		0x00,                                      /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		STDDATAFILE1,                              /* [In] 1 byte file number + SAI. */
		pOffset,                                   /* [In] 3 bytes offset. LSB First. */
		pTxData,                                   /* [in] Data to be written. */
		pTxDataLen);                               /* [in] 3 bytes. length of data to be written. */
	CHECK_SUCCESS(status);

	/* Reading The data */
	status= phalMfdfEVx_ReadData(pAlMfdfEVx,      /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,           /* [In] Is either plain or encrypted or MAC'd. */
		0x00,                                     /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		STDDATAFILE1,                             /* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,                                  /* [In] 3 bytes offset. LSB First. */
		pTxDataLen,                               /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
		&pRxData,                                 /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen                               /* [Out] Pointer to Length of RxData. */
		);
	CHECK_SUCCESS(status);

	/* printf("Application with crypto type 2K3DES and encrypted data file is created and credited with a value 0x100... \n"); */
	printf("InitializeKeySet to next and chageKey... \n");

	/* KeySetNo Bit3-0 : KeySet number within targeted application */
	/* It was zero earlier and hence changing to 1 */
	bKeySetNo = 0x01;
	/* Authenticate  with ChangeKeyaccess Key Number*/
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,             /* [In] Diversification option */
		AMKEY,                                       /* [In] KEY number in key store */
		0x00,                                        /* [In] Key version in key store */
		0x01,                                        /* [In] Key number on Card */
		NULL,                                        /* [In] Diversification input */
		0);                                          /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	/* Initialize Keyset */
	status = phalMfdfEVx_InitializeKeySet(pAlMfdfEVx,   /* [In] Pointer to this layer's parameter structure. */
		bKeySetNo,                                      /* [In] KeySet Number. Bit 7 indicates SAI. */
		PHAL_MFDFEVX_KEY_TYPE_3K3DES);                  /* [In] 1 byte Card key type. it can be PHAL_MFDFEV2_KEY_TYPE_AES128/KEY_TYPE_2K3DES/KEY_TYPE_3K3DES*/
	CHECK_SUCCESS(status);
	/* Finalize KeySet */
	status = phalMfdfEVx_FinalizeKeySet(pAlMfdfEVx,     /* [In] Pointer to this layer's parameter structure. */
		bKeySetNo,                                      /* [In] KeySet number. Can be 1 to 15. Bit 7 indicates SAI */
		PHAL_MFDFEVX_KEY_TYPE_3K3DES);                  /* [In] KeySet version of the key set to be finalized. */
	CHECK_SUCCESS(status);
	/* Authenticate  with ChangeKeyaccess Key Number*/
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,       /*[In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                /* [In] Diversification option */
		AMKEY,                                          /* [In] KEY number in key store */
		0x00,                                           /* [In] Key version in key store */
		0x01,                                           /* [In] Key number on Card */
		NULL,                                           /* [In] Diversification input */
		0);                                             /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* RollKey KeySet */
	status = phalMfdfEVx_RollKeySet(pAlMfdfEVx,        /* [In] Pointer to this layer's parameter structure. */
		bKeySetNo);                                    /* [In] KeySet number. Can be 1 to 15. Bit 7 indicates SAI */
	CHECK_SUCCESS(status);
	printf("Now application is modified to 3K3DES and finalized keyset... \n");
	/* Reauth with App master key */
	/* Authenticate with Modified Key*/
	status = phalMfdfEVx_AuthenticateISO(pAlMfdfEVx,     /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                 /* [In] Diversification option */
		DEFAULT_3K3DES,                                  /* [In] KEY number in key store */
		0x00,                                            /* [In] Key version in key store */
		0x00,                                            /* [In] Key number on Card */
		NULL,                                            /* [In] Diversification input */
		0);                                              /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Authenticate with File Read/write Key */
	status = phalMfdfEVx_AuthenticateISO(pAlMfdfEVx,     /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                 /* [In] Diversification option */
		DEFAULT_3K3DES,                                  /* [In] KEY number in key store */
		0x00,                                            /* [In] Key version in key store */
		0x01,                                            /* [In] Key number on Card */
		NULL,                                            /* [In] Diversification input */
		0);                                              /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Writting Data into Standard File */
	status= phalMfdfEVx_WriteData(pAlMfdfEVx,          /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                /* [In] Communication Mode. Plain, Mac'd or encrypted. */
		0x00,                                          /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		STDDATAFILE1,                                  /* [In] 1 byte file number + SAI. */
		pOffset,                                       /* [In] 3 bytes offset. LSB First. */
		pTxData,                                       /* [in] Data to be written. */
		pTxDataLen);                                   /* [in] 3 bytes. length of data to be written. */
	CHECK_SUCCESS(status);
	/* Reading The data */
	status= phalMfdfEVx_ReadData(pAlMfdfEVx,      /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,           /* [In] Is either plain or encrypted or MAC'd. */
		0x00,                                     /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		STDDATAFILE1,                             /* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,                                  /* [In] 3 bytes offset. LSB First. */
		pTxDataLen,                               /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
		&pRxData,                                 /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen                               /* [Out] Pointer to Length of RxData. */
		);
	CHECK_SUCCESS(status);
	return 0;

}
phStatus_t DemoDAM(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx){
	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint8_t pDAMEncKey[32];
	uint8_t bDamParams[5] = {0x02,0x00,0x01,0x20,0x00};
	uint8_t bDAMMAC[8];
	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,	/* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCKEY,                                    /* [In]KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x00,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);
	/*Initilaise DAM Auth Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		AMKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		AMKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x10,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/*Initilaise DAM MAC Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /*  [In] Diversification option. */
		AMKEY,                                    /*  [In] Old key number in keystore. */
		0,                                          /*  [In] Old key version in keystore. */
		AMKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x11,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/*Initilaise DAM ENC Key */
	status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
		AMKEY,                                    /* [In] Old key number in keystore. */
		0,                                          /* [In] Old key version in keystore. */
		AMKEY,                                    /* [In] New key number in keystore. */
		0,                                          /* [In] New key version in keystore. */
		0x12,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
		NULL,                                       /* [In] Diversification input. Can be NULL. */
		0);                                         /* [In] Length of diversification input max 31B. */
	CHECK_SUCCESS(status);
	/* Authenticate with the DAM Auth key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,	/* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		AMKEY,                                      /* [In]KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x10,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* generate DAM encryption key */
	status = phalMfdfEVx_GenerateDAMEncKey( pAlMfdfEVx,   /* [In] Pointer to this layers parameter structure. */
		AMKEY,			                                  /* [In] Key number in key store of DAM Encryption key. */
		0x00,		                                      /* [In] Key version in key store of DAM Encryption key. */
		DEFAULT_3K3DES,		                              /* [In] Key number in key store of DAM default key. */
		0x00,				                              /* [In] Key version in key store of DAM default key. */
		0x00,				                              /* [In] DAM Default Key version. Value should be valid for AES key. Ignored for other key types. */
		pDAMEncKey                                        /* [Out] Encrypted DAM Default key data. */
		);
	CHECK_SUCCESS(status);
		/* DF Name = AFCAPPLICATION */
	bDFLen = 15;
	memcpy(pISODFName, "AFCAPPLICATION2", bDFLen);
	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first */
	bISOFileId[0] = 0x11;
	bISOFileId[1] = 0x22;

	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;

	/* Keysettings 2 = 00100110
	* 2K 3DES crypto + supports 2 byte ISO fids +
	* max 6 keys in this app
	*/
	bKeySettings2 = 0x35;

	/*	Keysetting 3 = 0000 0000
		No keyset values passed
	*/
	bKeySettings3 = 0x01;
	/* Key Set Values */
	bKeySetValues[0] = 0x01; /* byte0 = AKS ver */
	bKeySetValues[1] = 0x04; /* byte1 = #keysets */
	bKeySetValues[2] = 0x10; /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x01; /* byte3 = AppKeySetSettings */
	/* Generate DAMMAC key first */
	status = phalMfdfEVx_GenerateDAMMAC(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		bOption,                                          /* [In] Option to indicate whether this application has ISO Fid and DF */
		AMKEY,		                                      /* [In] Key number in key store of DAM MAC Key */
		0x00,		                                      /* [In] Key version in key store of DAM MAC Key*/
		bAfcDelApp,                                       /* [In] 3 byte AID, LSB First */
		bDamParams,                                       /* [In] 4 byte array. Byte0 = DAM Slot, Byte1 = DAMSlotVersion, Byte2, Byte 3  = QuotaLimit. Multiple of 32 byte block. */
		bKeySettings1,                                    /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                    /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                    /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                /* [In] 4 byte keyset values.  */
		bISOFileId,                                       /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                       /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen,                                           /* [In] Length of DF Name provided above */
		pDAMEncKey,                                       /* [In] 32 byte Encrypted initial application key. */
		bDAMMAC);                                         /* [Out] 8 byte DAM MAC. */
	CHECK_SUCCESS(status);
	printf("Creating delegated application ...... \n");
	/* create Delegated Application*/
	status = phalMfdfEVx_CreateDelegatedApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		bOption,                                                      /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcDelApp,                                                   /* [In] 3 byte AID, LSB First */
		bDamParams,                                                   /* [In] 4 byte array. Byte0 = DAM Slot, Byte1 = DAMSlotVersion, Byte2, Byte 3  = QuotaLimit. Multiple of 32 byte block. */
		bKeySettings1,                                                /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                                /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                                /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                            /* [In] 4 byte keyset values.  */
		bISOFileId,                                                   /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                                   /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen,                                                       /* [In] Length of DF Name provided above */
		pDAMEncKey,                                                   /* [In] 32 byte Encrypted initial application key. */
		bDAMMAC);                                                     /* [In] 8 byte DAM MAC. */
	CHECK_SUCCESS(status);
	/* Select the AFC Delegated Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		0x00,                                                /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcDelApp,                                          /* [In] 3 byte AID, LSB First */
		0x00);                                               /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	/* Authenticate with the Application Master Key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,            /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                     /* [In] Diversification option */
		AMKEY,                                               /* [In]KEY number in key store */
		0x00,                                                /* [In] Key version in key store */
		0x00,                                                /* [In] Key number on Card */
		NULL,                                                /* [In] Diversification input */
		0);                                                  /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	return 0;
}
phStatus_t SharedAppDemo(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx){

	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bISOFileId2[2];
	uint8_t pISODFName2[16];
	uint8_t bDFLen;
	uint8_t bAccessRights[2];
	uint8_t pISOFileId[2];
	uint8_t pISOFileId2[2];
	uint8_t bFileSize[3];
	uint8_t bFileSize2[3];
	uint8_t pTxData[4];
	uint8_t pTxDataLen[3];
	uint8_t  pOffset[3];
	uint8_t *pRxData;
	uint16_t pRxDataLen;

	printf(" Starting SharedDemoApp.........");
	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,           /* [In] Diversification option */
		PICCKEY,                                   /* [In]KEY number in key store */
		0x00,                                      /* [In] Key version in key store */
		0x00,                                      /* [In] Key number on Card */
		NULL,                                      /* [In] Diversification input */
		0);                                        /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first  For Application1*/
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;

	/* Iso File Id =0x00BB.sent LSB First For Application 2 */
	bISOFileId2[0]=0xBB;
	bISOFileId2[1]=0x00;

	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);
	memcpy(pISODFName2,"APPLICATIONNO2",bDFLen);

	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;

	/* Keysettings 2 = 1010 0110
	* 10   - AES crypto
	* 1    - supports 2 byte ISO fids
	* 1    - Keyset parameters not present
	* 0110 - max 6 keys in this app
	*/
	bKeySettings2 = 0xA6;

	/*	Keysetting 3 = 0000 0001
	*	bit[0] = 1 - Key set values present
	*/
	bKeySettings3 = 0x01;

	/* Key Set Values */
	bKeySetValues[0] = 0x00; /* byte0 = AKS ver */ /* TBD */
	bKeySetValues[1] = 0x04; /* byte1 = #keysets */
	bKeySetValues[2] = 0x24; /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00; /* byte3 = AppKeySetSettings */
	/* creating Application 1*/
	status = phalMfdfEVx_CreateApplication(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                          /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                          /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                    /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                    /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                    /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                /* [In] 4 byte keyset values.  */
		bISOFileId,                                       /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                       /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                          /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);
	/* creating Application 2 */
	status = phalMfdfEVx_CreateApplication(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                          /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp2,                                         /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                    /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                    /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                    /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                /* [In] 4 byte keyset values.  */
		bISOFileId2,                                      /* [In] Two byte ISO File Id, LSB First */
		pISODFName2,                                      /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                          /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);
	/* Select the AFC Application 1*/
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                            /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,                                         /* [In] 3 byte AID, LSB First */
		0x00);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);

	/* ISOFile ID of File*/
	pISOFileId[0] = 01;
	pISOFileId[1] = 01;

	/* File Size = 32 bytes */
	bFileSize[0] = 0x20;
	bFileSize[1] = 0x00;
	bFileSize[2] = 0x00;
	/* AccessRights of File */
	bAccessRights[0] = 0x11;            /* Change access rights = AMK, RW access = never(for now AMK) */
	bAccessRights[1] = 0x11;            /* Read access = ACKey, Write access = never (for now AMK) */
	/* Create backup file in Application 1 */
	status = phalMfdfEVx_CreateBackupDataFile(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		0x01,                                                /* [In] Option to indicate ISO File Id provided */
		BCKUPDATAFILE1,                                      /* [In] File number. 1 byte */
		pISOFileId,                                          /* [In] Two byte ISO File ID. LSB First */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                      /* [In] Communication settings of the file */
		bAccessRights,                                       /* [In] 2 Byte access rights. LSB First */
		bFileSize);                                          /* [In] 3 Byte file size. LSB First */
	CHECK_SUCCESS(status);
	/* Select the AFC Application 2*/
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,      /* [In] Pointer to parameters data structure */
		0x00,                                               /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp2,                                           /* [In] 3 byte AID, LSB First */
		0x00);                                              /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);

	/* ISOFile ID of File*/
	pISOFileId2[0] = 02;
	pISOFileId2[1] = 02;

	/* File Size = 32 bytes */
	bFileSize2[0] = 0x20;
	bFileSize2[1] = 0x00;
	bFileSize2[2] = 0x00;
	/* AccessRights of File */
	bAccessRights[0] = 0x11;          /* Change access rights = AMK, RW access = never(for now AMK) */
	bAccessRights[1] = 0x11;          /* Read access = ACKey, Write access = never (for now AMK) */

	/* Create backup file in Application 2 */
	status = phalMfdfEVx_CreateBackupDataFile(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		0x01,                                                /* [In] Option to indicate ISO File Id provided */
		BCKUPDATAFILE2,                                      /* [In] File number. 1 byte */
		pISOFileId2,                                         /* [In] Two byte ISO File ID. LSB First */
		PHAL_MFDFEVX_COMMUNICATION_ENC,                      /* [In] Communication settings of the file */
		bAccessRights,                                       /* [In] 2 Byte access rights. LSB First */
		bFileSize2);                                         /* [In] 3 Byte file size. LSB First */
	CHECK_SUCCESS(status);
	/* Select Both Applications */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		0x01,                                                /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,                                             /* [In] 3 byte AID, LSB First */
		bAfcApp2);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	/* Authenticate with the write key  of Application 1*/
	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,        /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,                    /* [In] Diversification option */
		AMKEY_NEW,                                          /* [In] KEY number in key store */
		0x00,                                               /* [In] Key version in key store */
		0x01,                                               /* [In] Key number on Card */
		NULL,                                               /* [In] Diversification input */
		0);                                                 /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Data Length -3 bytes LSB first */
	pTxDataLen[0]=0x03;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;
	/* Data */
	pTxData[0]=10;
	pTxData[1]=0;
	pTxData[2]=11;
	/* Offset */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;

	/* writing the data into file */
	/* In this demo, while performing WriteData */
	/* we get HAL_IO_TIMEOUT error from the Card*/
	status= phalMfdfEVx_WriteData(pAlMfdfEVx,        /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_COMMUNICATION_ENC,              /* [In] Communication Mode. Plain, Mac'd or encrypted. */
		0x00,                                        /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		BCKUPDATAFILE1,                              /* [In] 1 byte file number + SAI. */
		pOffset,                                     /* [In] 3 bytes offset. LSB First. */
		pTxData,                                     /* [in] Data to be written. */
		pTxDataLen);                                 /* [in] 3 bytes. length of data to be written. */
	CHECK_SUCCESS(status);
	/* Authenticating the file with write key of Application2 */
	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,               /* [In] Diversification option */
		AMKEY_NEW,                                     /* [In] KEY number in key store */
		0x00,                                          /* [In] Key version in key store */
		0x81,                                          /* [In] Key number on Card */
		NULL,                                          /* [In] Diversification input */
		0);                                            /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* data length -LSB first */
	pTxDataLen[0]=0x03;
	pTxDataLen[1]=0x00;
	pTxDataLen[2]=0x00;
	/*data*/
	pTxData[0]=11;
	pTxData[1]=15;
	pTxData[2]=44;
	/*Offset*/
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;
	/* writing the data into Backup File in Application 2*/
	status= phalMfdfEVx_WriteData(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_COMMUNICATION_ENC,          /* [In] Communication Mode. Plain, Mac'd or encrypted. */
		0x00,                                    /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		0x87,                                    /* [In] 1 byte file number + SAI. *//*File number with SAI Identifier set */
		pOffset,                                 /* [In] 3 bytes offset. LSB First. */
		pTxData,                                 /* [in] Data to be written. */
		pTxDataLen);                             /* [in] 3 bytes. length of data to be written. */
	CHECK_SUCCESS(status);
	/* commiting The transaction */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /**< [In] Pointer to this layers param structure. */
		0x00,                                           /**< [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                           /**< [Out] 4 byte TMAC counter. */
		bTMV 	                                        /**< [Out] 8 byte TMAC value. */
		);
	CHECK_SUCCESS(status);
	/* Authenticate with the write key of Application 1*/
	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,               /* [In] Diversification option */
		AMKEY_NEW,                                     /* [In] KEY number in key store */
		0x00,                                          /* [In] Key version in key store */
		0x01,                                          /* [In] Key number on Card */
		NULL,                                          /* [In] Diversification input */
		0);                                            /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Reading The data from backupFile in Application 1 */
	status= phalMfdfEVx_ReadData(pAlMfdfEVx,         /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,              /* [In] Is either plain or encrypted or MAC'd. */
		0x00,                                        /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		BCKUPDATAFILE1,                              /* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,                                     /* [In] 3 bytes offset. LSB First. */
		pTxDataLen,                                  /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
		&pRxData,                                    /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen                                  /* [Out] Pointer to Length of RxData. */
		);
	CHECK_SUCCESS(status);
	/* Authenticate with the write key of Application 2 */
	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,              /* [In] Diversification option */
		AMKEY_NEW,                                    /* [In] KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x81,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0);                                           /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Reading The data from backupFile in Application 2 */
	status= phalMfdfEVx_ReadData(pAlMfdfEVx,         /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_ENC,              /* [In] Is either plain or encrypted or MAC'd. */
		0x00,                                        /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		0x87,                                        /* [In] 1 byte file number. 7th bit indicates SAI. */
		pOffset,                                     /* [In] 3 bytes offset. LSB First. */
		pTxDataLen,                                  /* [In] 3 bytes. length of data to be read. If 00, entire file will be read. */
		&pRxData,                                    /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen                                  /* [Out] Pointer to Length of RxData. */
		);
	CHECK_SUCCESS(status);
	return 0;

}
phStatus_t AddAccessRightsDemo(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint8_t bAccessRights[2];
	uint8_t pISOFileId[2];
	uint8_t bFileSize[3];
	printf(" Starting AdditionalAccessRightsDemoApp.........");
	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,           /* [In] Diversification option */
		PICCKEY,                                   /* [In]KEY number in key store */
		0x00,                                      /* [In] Key version in key store */
		0x00,                                      /* [In] Key number on Card */
		NULL,                                      /* [In] Diversification input */
		0);                                        /* [In] Diversification input length */
	CHECK_SUCCESS(status);

	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first  For Application1*/
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;


	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);

	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;

	/* Keysettings 2 = 0010 0110
	* 00   - 2K3DES crypto
	* 1    - supports 2 byte ISO fids
	* 1    - Keyset parameters not present
	* 0110 - max 6 keys in this app
	*/
	bKeySettings2 = 0x26;
	/*	Keysetting 3 = 0000 0001
	*	bit[0] = 1 - Key set values present
	*/
	bKeySettings3 = 0x01;

	/* Key Set Values */
	bKeySetValues[0] = 0x00; /* byte0 = AKS ver */ /* TBD */
	bKeySetValues[1] = 0x04; /* byte1 = #keysets */
	bKeySetValues[2] = 0x24; /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00; /* byte3 = AppKeySetSettings */

	/* creating Application 1*/
	status = phalMfdfEVx_CreateApplication(pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                          /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                          /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                    /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                    /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                    /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                                /* [In] 4 byte keyset values.  */
		bISOFileId,                                       /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                       /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                          /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);
	/* Select Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                            /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,                                         /* [In] 3 byte AID, LSB First */
		0x00);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	/* ISO File Id of File */
	pISOFileId[0] = 01;
	pISOFileId[1] = 01;

	/* File Size = 32 bytes */
	bFileSize[0] = 0x20;
	bFileSize[1] = 0x00;
	bFileSize[2] = 0x00;
	/* AccessRights of The File */
	bAccessRights[0] = 0x11;          /* Change access rights = AMK, RW access = never(for now AMK) */
	bAccessRights[1] = 0x11;          /* Read access = ACKey, Write access = never (for now AMK) */
	/* Create backup file */
	status = phalMfdfEVx_CreateBackupDataFile(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		0x01,                                              /* [In] Option to indicate ISO File Id provided */
		BCKUPDATAFILE1,                                    /* [In] File number. 1 byte */
		pISOFileId,                                        /* [In] Two byte ISO File ID. LSB First */
		0x80,                                              /* [In] Communication settings of the file ,enabled additional accessrights*/
		bAccessRights,                                     /* [In] 2 Byte access rights. LSB First */
		bFileSize                                          /* [In] 3 Byte file size. LSB First */
		);
	CHECK_SUCCESS(status);
	return 0;

}
phStatus_t UpdateLinearRecordFile(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx){

	phStatus_t status;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t pISODFName[16];
	uint8_t bDFLen;
	uint8_t pISOFileId[2];
	uint8_t pAccessRights[2];
	uint8_t pRecordSize[3];   /**< [In] 3 byte Record Size. Sent LSB first to PICC. */
	uint8_t pMaxNoOfRec[3];
	uint8_t pOffset[3];       /**< [In] 3 bytes offset. LSB First. */
	uint8_t pData[3];         /**< [In] data to be written. */
	uint8_t pDataLen[3];      /**< [In] 3 bytes. length of data to be written. */
	uint8_t *pRxData;
	uint16_t pRxDataLen;
	uint8_t RecNo[3]= {0x00,0x00,0x00};
	uint8_t RecCount[3] = {0x00,0x00,0x00};
	printf(" Starting Linear Record file Demo App.....");

	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_AuthenticateISO(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,              /* [In] Diversification option */
		PICCKEY,                                      /* [In]KEY number in key store */
		0x00,                                         /* [In] Key version in key store */
		0x00,                                         /* [In] Key number on Card */
		NULL,                                         /* [In] Diversification input */
		0);                                           /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);

	/* Create AFC Application */
	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;

	/* Iso File Id = 0x00AA. Sent LSB first  For Application1*/
	bISOFileId[0] = 0xAA;
	bISOFileId[1] = 0x00;

	/* DF Name = AFCAPPLICATION */
	bDFLen = 14;
	memcpy(pISODFName, "AFCAPPLICATION", bDFLen);

	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;

	/* Keysettings 2 = 0010 0110
	* 00   - 2K3DES crypto
	* 1    - supports 2 byte ISO fids
	* 1    - Keyset parameters not present
	* 0110 - max 6 keys in this app
	*/
	bKeySettings2 = 0x26;
	/*	Keysetting 3 = 0000 0001
	*	bit[0] = 1 - Key set values present
	*/
	bKeySettings3 = 0x01;

	/* Key Set Values */
	bKeySetValues[0] = 0x00; /* byte0 = AKS ver */ /* TBD */
	bKeySetValues[1] = 0x04; /* byte1 = #keysets */
	bKeySetValues[2] = 0x24; /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00; /* byte3 = AppKeySetSettings */
	/* creating Application 1*/
	status = phalMfdfEVx_CreateApplication( pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
		bOption,                                         /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                         /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                   /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                   /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,                                   /* [In] Key Settings 3 - 1 Byte */
		&bKeySetValues[0],                               /* [In] 4 byte keyset values.  */
		bISOFileId,                                      /* [In] Two byte ISO File Id, LSB First */
		pISODFName,                                      /* [In] ISO DF Name. Maximum 16 bytes */
		bDFLen);                                         /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);
	/* Select the AFC Application 1*/
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		0x00,                                            /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcApp,                                         /* [In] 3 byte AID, LSB First */
		0x00);                                           /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	bOption=0x01;                                        /* Indicates The presence of IsoFileId */
	/* File Iso File Id */
	pISOFileId[0]=0x12;
	pISOFileId[1]=0x34;
	/* File Access Rights */
	pAccessRights[0]=0xEE;
	pAccessRights[1]=0xEE;
	/* Record size */
	pRecordSize[0] = 0x03;
	pRecordSize[1] = 0x00;
	pRecordSize[2] = 0x00;
	/* Number of records in record file */
	pMaxNoOfRec[0]=0x08;
	pMaxNoOfRec[1]=0x00;
	pMaxNoOfRec[2]=0x00;
	/* create Linear Record File */
	status= phalMfdfEVx_CreateLinearRecordFile(pAlMfdfEVx,  /* [In] Pointer to this layers param structure */
		bOption,                                            /* [In] Indicates ISO file ID is present or not. */
		RECORDFILE,                                         /* [In] Linear record file Number + SAI. */
		pISOFileId,                                         /* [In] 2 Byte IsoFileId. Sent LSB first to PICC. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN ,                  /* [In] communication settings + AAR indicator. */
		pAccessRights,                                      /* [In] 2 byte access rights. Sent LSB first to PICC. */
		pRecordSize,                                        /* [In] 3 byte Record Size. Sent LSB first to PICC. */
		pMaxNoOfRec                                         /* [In] 3 byte Max Number of Records. Sent LSB first to PICC. */
		);
	CHECK_SUCCESS(status);
	/* offset of the file */
	pOffset[0]=0x00;
	pOffset[1]=0x00;
	pOffset[2]=0x00;
	/* length of data to be written into file */
	pDataLen[0]=0x03;
	pDataLen[1]=0x00;
	pDataLen[2]=0x00;
	/* data to be written into the file */
	pData[0]=0x10;
	pData[1]=0x00;
	pData[2]=0x11;
	/* writting the data into Linear Record File */
	status= phalMfdfEVx_WriteRecord(pAlMfdfEVx, /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,       /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                   /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                             /* [In] 1 byte file number and SAI.. */
		pOffset,                                /* [In] 3 bytes offset. LSB First. */
		pData,                                  /* [In] data to be written. */
		pDataLen                                /* [In] 3 bytes. length of data to be written. */
		);
	CHECK_SUCCESS(status);
	/* commiting the data */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		0x00,                                           /* [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                           /* [Out] 4 byte TMAC counter. */
		bTMV 	                                        /* [Out] 8 byte TMAC value. */
		);
	CHECK_SUCCESS(status);
	/* Data to write in Record File */
	pData[0]=0x10;
	pData[1]=0x00;
	pData[2]=0x22;
	/* writting the data into Linear Record File */
	status= phalMfdfEVx_WriteRecord(pAlMfdfEVx,     /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,           /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                       /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                                 /* [In] 1 byte file number and SAI.. */
		pOffset,                                    /* [In] 3 bytes offset. LSB First. */
		pData,                                      /* [In] data to be written. */
		pDataLen                                    /* [In] 3 bytes. length of data to be written. */
		);
	CHECK_SUCCESS(status);
	/* commiting the data */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		0x00,                                           /* [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                           /* [Out] 4 byte TMAC counter. */
		bTMV 	                                        /* [Out] 8 byte TMAC value. */
		);
	CHECK_SUCCESS(status);
	/* Data to write in Record File */
	pData[0]=0x10;
	pData[1]=0x00;
	pData[2]=0x44;
	/* writting the data into Linear Record File */
	status= phalMfdfEVx_WriteRecord(pAlMfdfEVx,   /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,         /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                     /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                               /* [In] 1 byte file number and SAI.. */
		pOffset,                                  /* [In] 3 bytes offset. LSB First. */
		pData,                                    /* [In] data to be written. */
		pDataLen                                  /* [In] 3 bytes. length of data to be written. */
		);
	CHECK_SUCCESS(status);
	/* commiting the data */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx, /* [In] Pointer to this layers param structure. */
		0x00,                                          /* [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                          /* [Out] 4 byte TMAC counter. */
		bTMV 	                                       /* [Out] 8 byte TMAC value. */
		);
	CHECK_SUCCESS(status);
	/* Data to write in Record File */
	pData[0]=0x10;
	pData[1]=0x00;
	pData[2]=0x44;
	/* writting the data into Linear Record File */
	status= phalMfdfEVx_WriteRecord(pAlMfdfEVx,      /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,            /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                        /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                                  /* [In] 1 byte file number and SAI.. */
		pOffset,                                     /* [In] 3 bytes offset. LSB First. */
		pData,							             /* [In] data to be written. */
		pDataLen                                     /* [In] 3 bytes. length of data to be written. */
		);
	CHECK_SUCCESS(status);
	/* commiting the data */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		0x00,                                           /* [In] Lower nibble : Return Calculated TMAC or not.*/
		bTMC,                                           /* [Out] 4 byte TMAC counter. */
		bTMV 	                                        /* [Out] 8 byte TMAC value. */
		);
	/* Read records from Record File */
	status= phalMfdfEVx_ReadRecords(pAlMfdfEVx, /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,       /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                   /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                             /* [In] 1 byte file number and SAI. */
		RecNo,                                  /* [In] 3 bytes LSB first. Record number of the newest record targeted, starting to count from the latest record written. */
		RecCount,                               /* [In] 3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read. */
		pRecordSize,                            /* [In] Record size. 3Bytes LSB first. */
		&pRxData,                               /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen);                           /* [Out] Pointer to Length of RxData. */
	CHECK_SUCCESS(status);
	/* Record Number of the file to be updated */
	RecNo[0]=0x01;
	RecNo[1]=0x00;
	RecNo[2]=0x00;
	/* data to be updated */
	pData[0]=0x10;
	pData[1]=0x00;
	pData[2]=0x33;
	/* Updating RecordFile */
	status = phalMfdfEVx_UpdateRecord(pAlMfdfEVx,   /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,           /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                       /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                                 /* [In] 1 byte file number and SAI. */
		RecNo,                                      /* [In] 3 byte record number. */
		pOffset,                                    /* [In] 3 bytes offset. LSB First. */
		pData,                                      /* [In] data to be written. */
		pDataLen);                                  /* [In] 3 bytes. length of data to be written. */
	CHECK_SUCCESS(status);
	/* commiting the data */
	status= phalMfdfEVx_CommitTransaction( pAlMfdfEVx,
		0x00,
		bTMC,        /**< [Out] 4 byte TMAC counter. */
		bTMV 	     /**< [Out] 8 byte TMAC value. */
		);
	CHECK_SUCCESS(status);
	RecNo[0]=0x00;
	RecNo[1]=0x00;
	RecNo[2]=0x00;
	/* Reading The records After Updating */
	status= phalMfdfEVx_ReadRecords(pAlMfdfEVx,  /* [In] Pointer to this layers param structure. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN,        /* [In] Communication option. Plain, Mac'd or enc. */
		0x00,                                    /* [In] If set, uses ISO 14443-4 chaining instead of DESFire application chaining. */
		RECORDFILE,                              /* [In] 1 byte file number and SAI. */
		RecNo,                                   /* [In] 3 bytes LSB first. Record number of the newest record targeted, starting to count from the latest record written. */
		RecCount,                                /* [In] 3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read. */
		pRecordSize,                             /* [In] Record size. 3Bytes LSB first. */
		&pRxData,                                /* [Out] Pointer to HAL Rx buffer returned back to user. */
		&pRxDataLen);                            /* [Out] Pointer to Length of RxData. */
	CHECK_SUCCESS(status);
	return 0;

}

uint8_t OriginalityCheckDemo_Asymmetric(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx, phCryptoASym_mBedTLS_DataParams_t * pCryptoASym)
{
    phStatus_t wStatus = 0;
    uint16_t wOption = 0;
    uint8_t * pSignature = NULL;

    uint8_t bAddr = 0x0;
	uint8_t aUid[10];
    uint16_t wUid_Len = 0;

    uint8_t bSigStat = PH_OFF;

	/* Authenticate with the PICC Master key */
    wStatus = phalMfdfEVx_Authenticate(pAlMfdfEVx, PHAL_MFDFEVX_NO_DIVERSIFICATION, PICCKEY, 0x00, 0x00, NULL, 0);
	CHECK_SUCCESS(wStatus);

	/* Get Card UID. This is VCUID which is of 7 bytes */
    wStatus =  phalMfdfEVx_GetCardUID(pAlMfdfEVx, 0x00, 0x00, aUid);
	CHECK_SUCCESS(wStatus);

    /* Get the UID Length. */
    wStatus = phalMfdfEVx_GetConfig(pAlMfdfEVx, PHAL_MFDFEVX_ADDITIONAL_INFO, &wUid_Len);
    CHECK_SUCCESS(wStatus);

	 /* Read the signature from the PICC */
    wStatus = phalMfdfEVx_ReadSign(pAlMfdfEVx, bAddr, &pSignature);
    CHECK_SUCCESS(wStatus);

    /* Load the Public Key. */
    wOption = (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | PH_CRYPTOASYM_CURVE_ID_SECP224R1);
    wStatus = phCryptoASym_ECC_LoadKeyDirect(pCryptoASym, wOption, gaPublicKey_DFEV2, (uint16_t) sizeof(gaPublicKey_DFEV2));
    CHECK_SUCCESS(wStatus);

	/* Signature verification for the UID sent */
    wStatus = phCryptoASym_ECC_Verify(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE, aUid, (uint8_t) wUid_Len,
        pSignature, (uint16_t) sizeof(pSignature));
    bSigStat = (uint8_t) ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS);

    /* Load the Public Key. */
    wOption = (uint16_t) (PH_CRYPTOASYM_PUBLIC_KEY | PH_CRYPTOASYM_CURVE_ID_SECP224R1);
    wStatus = phCryptoASym_ECC_LoadKeyDirect(pCryptoASym, wOption, gaPublicKey_DFEV3, (uint16_t) sizeof(gaPublicKey_DFEV3));
    CHECK_SUCCESS(wStatus);

    /* Signature verification for the UID sent */
    wStatus = phCryptoASym_ECC_Verify(pCryptoASym, PH_EXCHANGE_DEFAULT, PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE, aUid, (uint8_t) wUid_Len,
        pSignature, (uint16_t) sizeof(pSignature));
    bSigStat = (uint8_t) (((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS) << 1);

    if((bSigStat & 0x03U) != 0)
    {
        printf("Asymmetric Originality Check SUCCESS!! ... \n");
        return PH_ERR_SUCCESS;
    }
    else
    {
        printf("Asymmetric Originality Check FAILED!! ... \n");
        return 1;
    }
}

phStatus_t OriginalityCheckDemo_Symmetric(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
    phStatus_t status;
	/* The ref arch ver 0.14 says,
	 * 1. The authentication procedure for AES keys can be used to authenticate to KeyID.OriginalityKeys and check
	 * whether the PICC is a genuine NXP product or a genuine licensed NXP product.
	 * 2. Since the authentication requires the use of a key derived from the master key, a Secure Access Module will be
	 * required to calculate the derived key from the Master key, securely stored in the SAM.
	 * The master will be defined by NXP towards licensee
	 *
	 * This means the Originality key will be generated and stored in SAM using the master key. This key will be provided later, till
	 * then a dummy originality key will be used.
	 */

	status = phalMfdfEVx_AuthenticateAES(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCORIGINALITYKEY,                         /* [In] KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x01,                                       /* [In] Key No 0x01= OriginalityKey-1*/
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* If the authentication succeeds then the OC is passed */
	printf("\t Symmetric Originality Check  SUCCESS! \n");
	return status;
}
/**
 * Virtual Card Example Demo Application
 */
phStatus_t VirtualCardDemo(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx,
			phalVca_Sw_DataParams_t * pAlVca)
{
    phStatus_t status;
	uint16_t wFCILen = 0;
	uint8_t  FCI[40];
	uint8_t bSelector;
	uint8_t bOption;
	uint8_t bExtendedLenApdu;
	uint8_t pISODFName[16] = {0xA0, 0x00, 0x00, 0x03, 0x96, 0x56, 0x43, 0x41, 0x03, 0xF0, 0x15, 0x40, 0x20, 0xD4, 0x00, 0x0D};
	uint8_t pISODFName_default[7] = {0xD2, 0x76, 0x00, 0x00, 0x85, 0x01, 0x00 };
    uint16_t wKeyNumber = VCSELECTENCKEY;
	uint16_t wKeyVersion = 0x00;
	uint8_t bRandChal[16];
	uint8_t bVCData[16];
	uint8_t bLenPcdCapsIn=6;                                    /**< [In] Length of PcdCapsIn. Always zero for following authentication */
	uint8_t bPcdCapsIn[6] = {0x00,0x00,0x00,0x00,0x00,0x00};    /**< [In] PCD Capabilities. Upto 6 bytes. */
	uint8_t bPcdCapsOut[6] ;                                    /**< [Out] PCD Capabilities. 6 bytes. */
	uint8_t bPdCapsOut[6] ;                                     /**< [Out] PD Capabilities. 6 bytes. */
	uint8_t bConfig;

	/* The below portion of code is not required if tested in Mifare applet sample cards */
	if(!bRunOnMifareApplet)
	{
		/*Do Master Authentication before changing authentication key to VC Config Key*/
		/* Authenticate with the PICC Master key */
		status = phalMfdfEVx_Authenticate(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,           /* [In] Diversification option */
			PICCKEY,                                   /* [In]KEY number in key store */
			0x00,                                      /* [In] Key version in key store */
			0x00,                                      /* [In] Key number on Card */
			NULL,                                      /* [In] Diversification input */
			0);                                        /* [In] Diversification input length */
		CHECK_SUCCESS(status);

		/*Initialise VC Auth Key i.e VC Configuration key (VCConfigurationKey)*/
		status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
			VCCONFIGKEY,                                /* [In] Old key number in keystore. */
			0,                                          /* [In] Old key version in keystore. */
			VCCONFIGKEY,                                /* [In] New key number in keystore. */
			0,                                          /* [In] New key version in keystore. */
			0x20,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
			NULL,                                       /* [In] Diversification input. Can be NULL. */
			0);                                         /* [In] Length of diversification input max 31B. */
		CHECK_SUCCESS(status);

		/* Do Authentication with VC Config Key*/
		status = phalMfdfEVx_AuthenticateEv2(pAlMfdfEVx,  /* [In] Pointer to parameters data structure */
			0x01,                                         /* first authentication */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,              /* [In] Diversification option */
			VCCONFIGKEY,                                   /* [In] KEY number in key store */
			0x00,                                         /* [In] Key version in key store */
			0x20,                                         /* [In] Key number on Card */
			NULL,                                         /* [In] Diversification input */
			0,                                            /* [In] Diversification input length */
			bLenPcdCapsIn,                                /* [In] Length of PcdCapsIn. Always zero for following authentication */
			bPcdCapsIn,                                   /* [In] PCD Capabilities. Upto 6 bytes. */
			bPcdCapsOut,                                  /* [Out] PCD Capabilities. 6 bytes. */
			bPdCapsOut);                                  /* [Out] PD Capabilities. 6 bytes. */
		CHECK_SUCCESS(status);

		/*Initilaise VC Proximity Key */
		status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
			VCPROXIMITYKEY,                             /* [In] Old key number in keystore. */
			0,                                          /* [In] Old key version in keystore. */
			VCPROXIMITYKEY,                             /* [In] New key number in keystore. */
			0,                                          /* [In] New key version in keystore. */
			0x21,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
			NULL,                                       /* [In] Diversification input. Can be NULL. */
			0);                                         /* [In] Length of diversification input max 31B. */
		CHECK_SUCCESS(status);
		/*Initilaise VC MAC Key */
		status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
			VCSELECTMACKEY,                             /* [In] Old key number in keystore. */
			0,                                          /* [In] Old key version in keystore. */
			VCSELECTMACKEY,                             /* [In] New key number in keystore. */
			0,                                          /* [In] New key version in keystore. */
			0x22,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
			NULL,                                       /* [In] Diversification input. Can be NULL. */
			0);                                         /* [In] Length of diversification input max 31B. */
		CHECK_SUCCESS(status);
		/*Initilaise VC ENC Key */
		status = phalMfdfEVx_ChangeKey(pAlMfdfEVx,      /* [In] Pointer to this layer's parameter structure. */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option. */
			VCSELECTENCKEY,                             /* [In] Old key number in keystore. */
			0,                                          /* [In] Old key version in keystore. */
			VCSELECTENCKEY,                             /* [In] New key number in keystore. */
			0,                                          /* [In] New key version in keystore. */
			0x23,                                       /* [In] One byte Card number to be changed. Bit 7 will indicate SAI if card master app is not selected else bit 7,6 indicate key type. */
			NULL,                                       /* [In] Diversification input. Can be NULL. */
			0);                                         /* [In] Length of diversification input max 31B. */
		CHECK_SUCCESS(status);

		/* Select the master app */
		status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
			0x00,                                                /* [In] bOption-0x00 for single and 0x01 for second application */
			bAfcMApp,                                            /* [In] 3 byte AID, LSB First */
			0x00);                                               /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
		CHECK_SUCCESS(status);
		/* Authenticate with the PICC Master Key */
		status = phalMfdfEVx_Authenticate(pAlMfdfEVx,            /* [In] Pointer to parameters data structure */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,                     /* [In] Diversification option */
			AMKEY,                                               /* [In]KEY number in key store */
			0x00,                                                /* [In] Key version in key store */
			0x00,                                                /* [In] Key number on Card */
			NULL,                                                /* [In] Diversification input */
			0);                                                  /* [In] Diversification input length */
		CHECK_SUCCESS(status);

		/* Set Configuration */
		bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION0; /* option 0x00 for PICC configuration */
		bConfig = 0x0C;
		status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
							bOption,
							&bConfig,
							0x01);
		CHECK_SUCCESS(status);
	}
	/* Select the master app */
		status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
			0x00,                                                /* [In] bOption-0x00 for single and 0x01 for second application */
			bAfcMApp,                                            /* [In] 3 byte AID, LSB First */
			0x00);                                               /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
		CHECK_SUCCESS(status);
		/* Authenticate with the PICC Master Key */
		status = phalMfdfEVx_Authenticate(pAlMfdfEVx,            /* [In] Pointer to parameters data structure */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,                     /* [In] Diversification option */
			AMKEY,                                               /* [In]KEY number in key store */
			0x00,                                                /* [In] Key version in key store */
			0x00,                                                /* [In] Key number on Card */
			NULL,                                                /* [In] Diversification input */
			0);                                                  /* [In] Diversification input length */
		CHECK_SUCCESS(status);
	/* ISO Select for Virtual Card */
	/* bOption == 0x00 FCI is returned. If 0x0C no FCI returned */
	bOption = 0x00;
	/* bSelector equals 0x04 for selecting the virtual card only with ISO DF Name */
	bSelector = 0x04;
	/* Flag for Extended Length APDU. 0x01 for Extended Length APDUs. 0x00 or any other value for Short APDUs. */
	bExtendedLenApdu = 0x00;
	memset(FCI, 0, sizeof(FCI));
	status = phalVca_IsoSelect(pAlVca,		    /**< [In] Pointer to VCA param structure. */
								bSelector,	    /**< [In] bSelector equals 0x04 for selecting only with DF Name*/
								bOption,		    /**< [In] If bOption == 00 FCI is returned. If 0x0C no FCI returned. */
								bRunOnMifareApplet ? 16 : 7, /**< [In] Length of DFName string provided by the user. */
								bRunOnMifareApplet ? pISODFName : pISODFName_default, /**< [In] DFName upto 16 bytes. valid only when bOption = 0x04. */
								NULL,
								0x00,
								0x00,
								0x00,
								0x00,
								0x00,
								FCI,				/**< [Out] File control information. */
								&wFCILen		/**< [Out] Length of FCI returned. */
								);
	CHECK_SUCCESS(status);
   /* AuthVCMandatory is true so IsoExternalAuthenticate is compulsory
      and if AuthVCMandatory is false then IsoExternalAuthenticate is not required */
   if(wFCILen == VCA_AUTH_RND_LEN)
   {
   /* Authenticating the Selected Virtual Card */
    /* Decrypting the response data got from ISOSelect API */
    status = phalVca_DecryptResponse(pAlVca,		/**< [In] Pointer to VCA param structure. */
								wKeyNumber,
								wKeyVersion,
								FCI,
								bRandChal,
								bVCData
								);
   memcpy(&FCI[0], bRandChal,16);
   memcpy(&FCI[16], bVCData,16);
   wKeyNumber = VCSELECTMACKEY;
   status = phalVca_IsoExternalAuthenticate(pAlVca,		/**< [In] Pointer to VCA param structure. */
								FCI,
								wKeyNumber,
								wKeyVersion
								);
    CHECK_SUCCESS(status);
   }
   /* TODO: Read/Write operation can be done after selecting and authenticating a specific virtual card */
   return 0;
}
phStatus_t ProximityCheckDemo(phalVca_Sw_DataParams_t * pAlVca)
{
	phStatus_t status;
	uint8_t bGenerateRndC = 0x01;
    uint8_t pPrndC[8];
	uint8_t bOption = 0;
	uint8_t pPubRespTime[2];
	uint8_t aResponse[10];
	uint16_t wRespLen = 0;
    uint8_t bNumSteps = 0x01;
    uint16_t wKeyNumber = VCPROXIMITYKEY;
    uint16_t wKeyVersion = 0x00;
    uint8_t pCumRndRC[16];

 	/* Setting the wrapped mode */
	status = phalVca_SetConfig(pAlVca, PHAL_VCA_WRAPPED_MODE, 0x01);
	/* Calling Proximity Check all-in-one function which consits of all the three operation (PreparePC, Proximity check & VerifyPC) */
    status = phalVca_ProximityCheckNew(pAlVca,
					bGenerateRndC,
					pPrndC,
					bNumSteps,
					wKeyNumber,
					wKeyVersion,
					NULL,
					0x00,
					&bOption,
					pPubRespTime,
					aResponse,
				    &wRespLen,
					pCumRndRC
					);
    CHECK_SUCCESS(status);

	return 0;
}
/**
 * Generate DAMMAC for Set Configuration option 0x06 Demo Application
 */
phStatus_t GenerateDAMMACSetConfigDemo(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
	phStatus_t status;
	uint16_t wOldDFNameLen;				/* [In] Length of existing DF Name */
	uint8_t pOldISODFName[16];	    	/* [In] This means already created delegated app ISO DF Name. Maximum 16 bytes */
	uint16_t wNewDFNameLen;	    		/* [In] Length of new DF Name */
	uint8_t pNewISODFName[16];  		/* [In] This means new delegated app ISO DF Name which will replace the existing one. Maximum 16 bytes */
	uint8_t pDAMMAC[8];     			/* [Out] generated 8 bytes DAM MAC for setconfig option 0x06 */
	uint8_t bOption;					/* [In] Option for Set configuration */
	uint8_t bDataLen;
	uint8_t pDFBuffer[50];	       	    /**< [Out] One DF Name at a time. Should be 21(3+2+16) bytes long. */
    uint8_t bDFInfoLen;	     	        /**< [Out] The size of the DF info returned in this frame. */
	/* Calling Case DEMODAM for Set configuration option 0x06 for creating a delegated application with ISODFName "AFCAPPLICATION2" */
	status = DemoDAM(pAlMfdfEVx);
    CHECK_SUCCESS(status);

	/* Old ISODFName = AFCAPPLICATION2 */
	wOldDFNameLen = 15;
	memcpy(pOldISODFName, "AFCAPPLICATION2", wOldDFNameLen);

	/* New ISODFName = NEWAPPLICATION */
	wNewDFNameLen = 14;
	memcpy(pNewISODFName, "NEWAPPLICATION", wNewDFNameLen);

	/* GenerateDAMMACSetConfig */
    status = phalMfdfEVx_GenerateDAMMACSetConfig(pAlMfdfEVx,
						AMKEY,
						0x00,
						wOldDFNameLen,
						pOldISODFName,
						wNewDFNameLen,
						pNewISODFName,
						pDAMMAC
					);
    CHECK_SUCCESS(status);
	/* Set Configuration - Changing the exixsting DF Name for a delegated applicattion using DAMAMC generated by phalMfdfEVx_GenerateDAMMACSetConfig() */
	bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION6; /* option 0x06 for delegated app ISODFname change */
	/* Building Buffer */
	memset(gbGenBuffer, 0x00, 25);				/* IsoDFName buffer */
	/* OldISODFNameLength being 16-bit length value as per the ref arch v 15 */
	gbGenBuffer[0] = 0x0E;						/* Length of existing ISODFName */
	memcpy(&gbGenBuffer[1], pNewISODFName, wNewDFNameLen);	/* Existing IsoDFName for the delegated app */
	memcpy(&gbGenBuffer[17], pDAMMAC, 8);		/* DAMMAC for changing DFName for the existing Delegated App */
	bDataLen = 25;								/* Data length passed */
	status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
						bOption,
						gbGenBuffer,
						bDataLen);
	CHECK_SUCCESS(status);
	/* Select the AFC Delegated Application */
	status = phalMfdfEVx_SelectApplication(pAlMfdfEVx,       /* [In] Pointer to parameters data structure */
		0x00,                                                /* [In] bOption-0x00 for single and 0x01 for second application */
		bAfcMApp,                                            /* [In] 3 byte AID, LSB First */
		0x00);                                               /* [In] Second AID...NA in this case; we do not need this as bOption is 0x00 */
	CHECK_SUCCESS(status);
	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,	/* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCKEY,                                    /* [In] KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x00,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Checking whether ISO DF Name of the delegated app has been changed or not after Set Config option 0x06 */
	status = phalMfdfEVx_GetDFNames(pAlMfdfEVx,
						PH_EXCHANGE_DEFAULT,
						pDFBuffer,
						&bDFInfoLen);
	CHECK_SUCCESS(status);
    return 0;
}

/* Create appplication with ISODFName and ISO File ID. This app will be used to demonstrate ISO7816 extended APDU */
phStatus_t Create_Application(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx )
{
	phStatus_t status;
	uint8_t bLength;
	uint8_t bOption;
	uint8_t bKeySettings1;
	uint8_t bKeySettings2;
	uint8_t bKeySettings3;
	uint8_t bKeySetValues[4];
	uint8_t bISOFileId[2];
	uint8_t bConfigByte;
	printf("Performing App Creation ...... \n");

	/* Authenticate with the PICC Master key */
	status = phalMfdfEVx_Authenticate(pAlMfdfEVx,   /* [In] Pointer to parameters data structure */
		PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
		PICCKEY,                                    /* [In]KEY number in key store */
		0x00,                                       /* [In] Key version in key store */
		0x00,                                       /* [In] Key number on Card */
		NULL,                                       /* [In] Diversification input */
		0);                                         /* [In] Diversification input length */
	CHECK_SUCCESS(status);
	/* Get a fresh card */
	status = phalMfdfEVx_Format(pAlMfdfEVx);
	CHECK_SUCCESS(status);
	/* Set Configuration - default key and version */
	bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION1; /* option to set default key and version */
	memset(gbGenBuffer, 0x00, 24);             /* Default key */
	gbGenBuffer[24] = 0x00;                    /* default key version */
	bLength = 25;                              /* length */
	status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
		bOption,
		gbGenBuffer,
		bLength);
	CHECK_SUCCESS(status);
	bOption = PHAL_MFDFEVX_SET_CONFIG_OPTION0;          /* option to set configuration byte */
	bConfigByte = 0x02;                                 /* 00000010 : Random ID Enabled */
	status = phalMfdfEVx_SetConfiguration(pAlMfdfEVx,
		bOption,
		&bConfigByte,
		0x01);
	CHECK_SUCCESS(status);

	/* bOption = 0x03 indicates that application has both ISO Fid and DF Name */
	bOption = 0x03;
	/* Iso File Id = 0x00AA. Sent LSB first */
	bISOFileId[0] = 0x12;
	bISOFileId[1] = 0x34;
	/* DF Name = AFCAPPLICATION */
	bISOFileLen = 14;
	memcpy(pISOFileName, "AFCAPPLICATION", bISOFileLen);
	/* Application masterkey settings 00001011
	* =  conf. changeable +
	*    create/delete file requires authentication +
	*   directory access is free +
	*   App MK is changeable
	*   Change access rights = 0x00 (AMK)
	*/
	bKeySettings1 = 0x0F;
	/* Keysettings 2 = 10100110
	* AES crypto + supports 2 byte ISO fids +
	* max 6 keys in this app
	*/
	bKeySettings2 = 0xA6;
	/* Key Set Values */
	bKeySetValues[0] = 0xAA;         /* byte0 = AKS ver */
	bKeySetValues[1] = 0x00;         /* byte1 = #keysets */
	bKeySetValues[2] = 0x00;         /* byte2 = MaxKeysize */
	bKeySetValues[3] = 0x00;         /* byte3 = AppKeySetSettings */
	bKeySettings3 = 0x00;
	/* Create AFC Application */
	status = phalMfdfEVx_CreateApplication( pAlMfdfEVx,    /* [In] Pointer to parameters data structure */
		bOption,                                           /* [In] Option to indicate whether this application has ISO Fid and DF */
		bAfcApp,                                           /* [In] 3 byte AID, LSB First */
		bKeySettings1,                                     /* [In] Key Settings 1 - 1 Byte */
		bKeySettings2,                                     /* [In] Key Settings 2 - 1 Byte */
		bKeySettings3,									   /* [In] Key Settings 2 - 1 Byte */
		&bKeySetValues[0],                                 /* [In] 4 byte keyset values.  */
		bISOFileId,                                        /* [In] Two byte ISO File Id, LSB First */
		pISOFileName,                                        /* [In] ISO DF Name. Maximum 16 bytes */
		bISOFileLen);                                           /* [In] Length of DF Name provided above */
	CHECK_SUCCESS(status);

	printf("\tApp creation DONE! \n");
	return 0;
}

/* Select the application using ISOSelectFile command */
phStatus_t ISO_SelectFile(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t 	pwFCILen = 0;
	uint8_t pFid[2] = {0x12, 0x34};
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	status = phalMfdfEVx_IsoSelectFile(
                                     pAlMfdfEVx,
                                     0x0C, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
									 &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("***** IsoSelectFile SUCCESS!!*******\n");
	return status;
}

/* Create a standard data file with ISOFile ID */
phStatus_t CreateStandardFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t fileNo = 0x1F;
	uint8_t IsoFileID[2] = {0x56, 0x78};
	uint8_t fileOption = 0x00;
	uint8_t accessRights[2] = {0xE0, 0x00};
	uint8_t fileSize[3] = {0xFF, 0x00, 0x00};
	status  = phalMfdfEVx_CreateStdDataFile(
		pAlMfdfEVx,
		0x01,
		fileNo,
		IsoFileID,
		fileOption,
		accessRights,
		fileSize);
	CHECK_SUCCESS(status);
	printf("***** Creating standard data file SUCCESS!!*******\n");
	return status;
}

phStatus_t SelectStandardFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t 	pwFCILen = 0;
	uint8_t pFid[2] = {0x56, 0x78};
	phStatus_t status;
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	status = phalMfdfEVx_IsoSelectFile(
                                     pAlMfdfEVx,
                                     0x0C, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
                                     &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("***** Selecting Standard Data file SUCCESS !! *****\n");
	return status;
}

/* Write some data to standard data file */
phStatus_t UpdateBinary(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t sfid = 0x00; /* supply the offset not short file id */
	uint32_t length = 0xFF; /* 255 bytes of data */
	uint8_t offset = 0x00; /* Start from 0th position */
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t data[256] = {'\0'};
	uint16_t index;
	for(index =0 ; index < sizeof(data); index++)
	{
		data[index] = (uint8_t)index;
	}
	status =  phalMfdfEVx_IsoUpdateBinary(
										   pAlMfdfEVx,
										   offset,
										   sfid,
										   bExtendedLenApdu,
										   data,
										   length
 										   );
	CHECK_SUCCESS(status);
	printf("***** ISOUpdateBinary SUCCESS !! *****\n");
	return(status);
}
phStatus_t ReadBinary(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx )
{
	phStatus_t status;
	uint8_t offset = 0x00;
	uint8_t sfid = 0x00;
	uint32_t bytestoread = 0xFF;
	uint8_t readbuf[500] =  {'\0'};
	uint32_t bytes_read = 0;
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t *pp_readbuf = readbuf;
	status  =  phalMfdfEVx_IsoReadBinary(
										 pAlMfdfEVx,
										 PH_EXCHANGE_DEFAULT,
										 offset,
										 sfid,
										 bytestoread,
										 bExtendedLenApdu,
										 &pp_readbuf,
										 &bytes_read
										 );
	CHECK_SUCCESS(status);
	printf("***** ISOReadBinary SUCCESS !! *****\n");
	return status;
}
phStatus_t CreateLinearRecordFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t bISOFileId[2];
	uint8_t pAccessRights[2];
	uint8_t pRecordSize[3];   /**< [In] 3 byte Record Size. Sent LSB first to PICC. */
	uint8_t pMaxNoOfRec[3];
	uint8_t bOption;
	bOption=0x01;
	bISOFileId[0]=0x12;
	bISOFileId[1]=0x34;
	/* File Access Rights */
	pAccessRights[0]=0xEE;
	pAccessRights[1]=0xEE;
	/* Record size */
	pRecordSize[0] = 0xFF;
	pRecordSize[1] = 0x00;
	pRecordSize[2] = 0x00;
	/* Number of records in record file */
	pMaxNoOfRec[0]=0x08;
	pMaxNoOfRec[1]=0x00;
	pMaxNoOfRec[2]=0x00;
	status= phalMfdfEVx_CreateLinearRecordFile(pAlMfdfEVx,  /* [In] Pointer to this layers param structure */
		bOption,                                            /* [In] Indicates ISO file ID is present or not. */
		RECORDFILE,                                         /* [In] Linear record file Number + SAI. */
		bISOFileId,                                         /* [In] 2 Byte IsoFileId. Sent LSB first to PICC. */
		PHAL_MFDFEVX_COMMUNICATION_PLAIN ,                  /* [In] communication settings + AAR indicator. */
		pAccessRights,                                      /* [In] 2 byte access rights. Sent LSB first to PICC. */
		pRecordSize,                                        /* [In] 3 byte Record Size. Sent LSB first to PICC. */
		pMaxNoOfRec                                         /* [In] 3 byte Max Number of Records. Sent LSB first to PICC. */
		);
	CHECK_SUCCESS(status);
	printf("***** Creating Linear Record File SUCCESS !! *****\n");
	return status;
}

phStatus_t SelectLinearRecordFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	uint8_t pFCI[256] = {'\0'};
	uint8_t *ppFCI = pFCI;
	uint16_t 	pwFCILen = 0;
	uint8_t pFid[2];
	phStatus_t status;
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	pFid[0]=0x12;
	pFid[1]=0x34;
	status = phalMfdfEVx_IsoSelectFile(
                                     pAlMfdfEVx,
                                     0x0C, /* Select by DF name */
                                     0x00, /* Return FCI */
                                     pFid,
                                     pISOFileName,
                                     bISOFileLen,
									 bExtendedLenApdu,
                                     &ppFCI,
                                     &pwFCILen
                                     );
	CHECK_SUCCESS(status);
	printf("***** Selecting Linear Record File SUCCESS !! *****\n");
	return status;
}

phStatus_t AppendLinearRecordFile(phalMfdfEVx_Sw_DataParams_t *pAlMfdfEVx)
{
	phStatus_t status;
	uint8_t sfid = 0x00; /* bit(3-7) ==>0x00 = currently selected file, 0x01 - 0x1E = short file ID, 0x1F=RFU
						  * bit(0-2) ==>000 = first record
						  */
	uint32_t length = 0xFF; /* 255 bytes of data */
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t data[256] = {'\0'};
	uint16_t index;
	for(index =0 ; index < sizeof(data); index++)
	{
		data[index] = (uint8_t)index;
	}
	status =  phalMfdfEVx_IsoAppendRecord(
										   pAlMfdfEVx,
										   sfid,
										   data,
										   length,
										   bExtendedLenApdu
 										   );
	CHECK_SUCCESS(status);
	printf("***** ISOAppendRecord SUCCESS !! *****\n");
	return(status);
}

phStatus_t ReadRecords(phalMfdfEVx_Sw_DataParams_t * pAlMfdfEVx )
{
	phStatus_t status;
	uint8_t record_number = 0x00; /* This is represented in P1. 0x00 means current record or This can take value from 0x01....ExistingRecordCount */
	uint8_t sfid = 0x00;  /* Currently selected File */
	uint8_t total_rec_toread = 0x04; /* 0100 Read record from rec num mentioned in P1, 0101 Read all records from P1 up to the last */
	uint32_t bytestoread = 0xFF;
	uint8_t readbuf[500] =  {'\0'};
	uint32_t bytes_read = 0;
	uint8_t bExtendedLenApdu = 0x01; /* 0x00 = short apdu, 0x01 = extended apdu */
	uint8_t *pp_readbuf = readbuf;
	status  =  phalMfdfEVx_IsoReadRecords(
										 pAlMfdfEVx,
										 PH_EXCHANGE_DEFAULT,
										 record_number,
										 total_rec_toread,
										 sfid,
										 bytestoread,
										 bExtendedLenApdu,
										 &pp_readbuf,
										 &bytes_read
										 );
	CHECK_SUCCESS(status);
	printf("***** ISOReadRecords SUCCESS !! *****\n");
	return status;
}

int __cdecl main(int argc, char *argv[])
{
	phStatus_t status;
	uint8_t bReaderListPegoda[512];
	uint8_t bCidEnable, bCid, bNadSupported, bFwi, bFsdi, bFsci;
	uint8_t bSak[1];
	uint8_t bUid[10];
	uint8_t pSeed[8];
	uint8_t bMoreCardsAvaliable;
	uint8_t bLength;
	uint16_t wCount;
	int iChoice,choice;
	uint8_t bIndex=0;
	phKeyStore_Sw_KeyEntry_t keyEntry[KEYCOUNT];
	phKeyStore_Sw_KeyVersionPair_t keyVersion[KEYCOUNT * KEYVERSIONS];
	phKeyStore_Sw_KUCEntry_t keyUsage[KEYCOUNT];

	/**
	 * Virtual Card
	 */
	phalVca_Sw_DataParams_t alVca;                               /**< VCA Software parameter structure */
	phalVca_Sw_IidTableEntry_t aIidTableStorage[IID_KEY_COUNT];  /**< Pointer to the Iid Table storage for the layer. */
	uint16_t wNumIidTableStorageEntries = IID_KEY_COUNT;         /**< Number of possible Iid table entries in the storage. */
	phalVca_Sw_CardTableEntry_t aCardTableStorage[IID_KEY_COUNT];/**< Pointer to the Card Table storage for the layer. */
	uint16_t wNumCardTableStorageEntries = IID_KEY_COUNT;        /**< Number of possible Card table entries in the storage. */
	/* Bfl data parameter storage */
	phbalReg_Rd70xUsbWin_DataParams_t balReader;
	phhalHw_Rd70x_DataParams_t halReader;
	phpalI14443p3a_Sw_DataParams_t I14443p3a;
	phpalI14443p4a_Sw_DataParams_t I14443p4a;
	phpalI14443p4_Sw_DataParams_t I14443p4;
	phpalMifare_Sw_DataParams_t palMifare;
	phCryptoSym_Sw_DataParams_t cryptoEnc;
	phCryptoSym_Sw_DataParams_t cryptoSymRnd;
	phCryptoSym_Sw_DataParams_t cryptoMac;
	phCryptoRng_Sw_DataParams_t cryptoRng;
    phCryptoASym_mBedTLS_DataParams_t stCryptoASym;
	phKeyStore_Sw_DataParams_t keyStore;
	phalMfdfEVx_Sw_DataParams_t alMfdfEVx;
	phTMIUtils_t TMIDataParams;
	void * pHal= NULL;
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */
	uint8_t pAts[255];
	phbalReg_SocketWin_DataParams_t balSocket;
	phhalHw_RdCardSim_DataParams_t  halRdSim;
	uint8_t pPortName[] = {'1','2','7','.','0','.','0','.','1','\0','1','5','2','0','0','\0','0','0','0','1','\0'};
	uint8_t TxBufferSocket[0x800];
	uint8_t RxBufferSocket[0x200];
	uint8_t TxBufferCardSim[0x800];
	uint8_t RxBufferCardSim[0x800];
	uint8_t IntBufferCardSim[0xff00];

	phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
    phhalHw_Rd710_DataParams_t	  halRd710;           /**< Rd710 HAL parameter structure */
	uint8_t aAtr[256];								  /**< Atr */

	PHAL_MFDFEVX_UNUSED_VARIABLE(argc);
	PHAL_MFDFEVX_UNUSED_ARRAY(argv);
	printf("\nNxpRdLib ANSI-C Example MIFARE DesFire EV2 Program\n\n");
	printf("Performing startup...\n\n");
	memset(&bTMIBuffer[0],0x00,TMI_BUFFER_SIZE);

	/* init. keystore */
	status = phKeyStore_Sw_Init(&keyStore,
		sizeof(phKeyStore_Sw_DataParams_t),
		keyEntry,
		KEYCOUNT,
		keyVersion,
		KEYVERSIONS,
		keyUsage,
		KEYCOUNT);
	CHECK_SUCCESS(status);
	pKeyStore = &keyStore;
	/* init. crypto */
    status = phCryptoASym_mBedTLS_Init(&stCryptoASym, sizeof(phCryptoASym_mBedTLS_DataParams_t), &keyStore, aBuffer, sizeof(aBuffer));
    CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoEnc, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoMac, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoSymRnd, sizeof(phCryptoSym_Sw_DataParams_t), &keyStore);
	CHECK_SUCCESS(status);
	status = phCryptoRng_Sw_Init(&cryptoRng, sizeof(phCryptoRng_Sw_DataParams_t), &cryptoSymRnd);
	CHECK_SUCCESS(status);
	memset(pSeed, 0x00, 0x08);
	status = phCryptoRng_Seed(&cryptoRng, pSeed, 8);
	CHECK_SUCCESS(status);

	printf("Press 1 To run with CLRd70x.\n\n");
	printf("Press 2 To run with  Reader = RdCardSim, Card = C# Simulator.\n\n");
	printf("Press 3 To run with  Reader CM-1 or Rd710.\n\n");
	printf("Press x To Quit.\n\n");
	choice=_getch();
	switch(choice)
	{
	 case('1') :
	    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
	    printf("Firmware Version 2.11 or higher must be used!\n\n");

	    /* Initialise the Reader BAL component */
	    status = phbalReg_Rd70xUsbWin_Init(&balReader, sizeof(phbalReg_Rd70xUsbWin_DataParams_t));
	    CHECK_SUCCESS(status);

	    /* Get list of connected pegoda readers */
	    status = phbalReg_GetPortList(&balReader, sizeof(bReaderListPegoda), bReaderListPegoda, &wCount);
	    CHECK_SUCCESS(status);

	    /* Connect to the first reader */
	    status = phbalReg_SetPort(&balReader, bReaderListPegoda);
	    CHECK_SUCCESS(status);

	    /* open the reader port */
	    status = phbalReg_OpenPort(&balReader);
	    CHECK_SUCCESS(status);

	    /* init the Reader HAL component */
	    status = phhalHw_Rd70x_Init(&halReader,
		    sizeof(phhalHw_Rd70x_DataParams_t),
		    &balReader,
		    gaHalRxReader,
		    sizeof(gaHalRxReader),
		    gaHalTxReader,
		    sizeof(gaHalTxReader));

	    CHECK_SUCCESS(status);

	    /* The default reader is the Pegoda */
	    pHal = &halReader;
	    break;

	case('2'):
		printf("Please ensure that Simulator is open and is in waiting state.\n\n");
		printf("Simulator version 5311 must be used!\n\n");

		/* Initialise the Reader BAL component */
		status =  phbalReg_SocketWin_Init(&balSocket, sizeof(phbalReg_SocketWin_DataParams_t), 0, sizeof(TxBufferSocket), TxBufferSocket,sizeof(RxBufferSocket), RxBufferSocket);
		CHECK_SUCCESS(status);

		status = phbalReg_SetPort(&balSocket, pPortName);
		CHECK_SUCCESS(status);
		/* init the Reader HAL component */
		status =   phhalHw_RdCardSim_Init(&halRdSim,
			sizeof(halRdSim),
			&balSocket,
			&keyStore,
			TxBufferCardSim,
			sizeof(TxBufferCardSim),
			RxBufferCardSim,
			sizeof(RxBufferCardSim),
			IntBufferCardSim,
			sizeof(IntBufferCardSim));

		pHal = &halRdSim;

		/* open the reader port */
		status = phbalReg_OpenPort(&balSocket);
		CHECK_SUCCESS(status);
		break;

	case('3'):
		printf("\nNxpRdLib ANSI-C Rd710 CM1 reader Example Program \n\n");
		printf("Please ensure that a Pegoda-2 CM1 reader is connected and in working condition.\n\n");
		printf("Performing startup...\n\n");
		printf("-------------------------\n");

		/* Init. PCSC BAL */
		status = phbalReg_PcscWin_Init(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
			sizeof(phbalReg_PcscWin_DataParams_t),  /**< [In] Specifies the size of the data parameter structure. */
			aAtr,                                   /**< [In] Pointer to buffer used for storing the ATR after card activation. */
			256);                                   /**< [In] Size of the ATR buffer. */
		CHECK_SUCCESS(status);

		/* Set the BAL Communication Configuration values */
		status = phbalReg_SetConfig(&balPcsc,           /**< [In] Pointer to this layer's parameter structure. */
			PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,          /**< BAL Communication Configs */
			PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED);/**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);

		/* set the BAL Communication Configs */
		status = phbalReg_SetConfig(&balPcsc,       /**< [In] Pointer to this layer's parameter structure. */
			PHBAL_REG_PCSCWIN_CONFIG_SHARE,         /**< BAL Communication Configs */
			PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);

		/* Select Port to be used (no SAM reader) */
		status = phbalReg_SetPort(&balPcsc,         /**< [In] Pointer to this layer's parameter structure. */
			(uint8_t*)PCSC_READER_P2_NO_SAM_NAME);  /**< [In] Port Name as String. */
		CHECK_SUCCESS(status);

		/**
		* Initialise the Rd710 HAL component:
		*   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
		*/
		status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
			sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
			&balPcsc,                           /**< [In] Pointer to the lower layers parameter structure */
			0,                                  /**< [In] Slot number */
			gaHalTxReader,                 /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
			sizeof(gaHalTxReader),         /**< [In] Size of the global transmit buffer. */
			gaHalRxReader,                 /**< [In] Pointer to global receive buffer used by the Exchange() function. */
			sizeof(gaHalRxReader));        /**< [In] Size of the global receive buffer. */
		CHECK_SUCCESS(status);

		/**< Set HAL pointer */
		pHal = &halRd710;

		/* Open the PCSC port */
		status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);

		/* Standard Rd710 Cmd Init. */
		status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);
		break;

	default:
		printf(" Quitting ... \n");
		exit(0);

	}

	/* init the 14443-3A component */
	status = phpalI14443p3a_Sw_Init(&I14443p3a, sizeof(phpalI14443p3a_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init the 14443-4A component */
	status = phpalI14443p4a_Sw_Init(&I14443p4a, sizeof(phpalI14443p4a_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init the 14443-4 component */
	status = phpalI14443p4_Sw_Init(&I14443p4, sizeof(phpalI14443p4_Sw_DataParams_t), pHal);
	CHECK_SUCCESS(status);

	/* init. mifare pal */
	status = phpalMifare_Sw_Init(&palMifare, sizeof(phpalMifare_Sw_DataParams_t), pHal, &I14443p4);
	CHECK_SUCCESS(status);

	status = phTMIUtils_Init(&TMIDataParams,&bTMIBuffer[0],TMI_BUFFER_SIZE);
	CHECK_SUCCESS(status);


	/* Initialize logging */
#ifdef NXPBUILD__PH_LOG
	/* Register the component for logging */
	status = phLog_Init(phalMfdfEVx_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
	status = phLog_Init(phalvca_log, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
	/*status = phLog_Register(&cryptoRng, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);
	status = phLog_Register(&cryptoRng, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);*/
	status = phLog_Register(&alMfdfEVx, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	status = phLog_Register(&alVca, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
	CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

	/* ####################Card Initialization########################### */
	do
	{
		/* Switch on the field */
		status = phhalHw_FieldOn(pHal);
		CHECK_SUCCESS(status);

		/* Configure HAL for Type-A cards */
		status = phhalHw_ApplyProtocolSettings(pHal, PHHAL_HW_CARDTYPE_ISO14443A);
		CHECK_SUCCESS(status);

		/* Activate Layer 3 card. In loop till a card is detected. */
		do
		{
			status = phpalI14443p3a_ActivateCard(&I14443p3a, NULL, 0x00, bUid, &bLength, bSak, &bMoreCardsAvaliable);

		} while (status != PH_ERR_SUCCESS);


		/* switch on time mesaurement */
		/* status = phhalHw_SetConfig(&halRdSim, PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_COMM);
		CHECK_SUCCESS(status);*/

		/* Send RATS */
		status = phpalI14443p4a_Rats(&I14443p4a, 0x08, 0x01, pAts);
		CHECK_SUCCESS(status);

		/* Obtain the protocol parameters from layer 4a and then store them in layer 4 */
		/*
		bCidEnable indicates if the card supports Cid. if 0, then no cid is supported.
		bNadSupported indicates if card supports Node address. if 0, NAD is not supported.
		bFwi => Framw waiting integer. can be zero to maximum value can be 0 TO 15.
		bFsdi -> Frame size indicator of pcd 0 to 8
		bFsci -> Frame size indicator of Card 0 - 8
		*/
		status = phpalI14443p4a_GetProtocolParams(&I14443p4a, &bCidEnable, &bCid, &bNadSupported, &bFwi, &bFsdi, &bFsci);
		CHECK_SUCCESS(status);

		status = phpalI14443p4_SetProtocol(&I14443p4, bCidEnable, bCid, bNadSupported, 0, bFwi, bFsdi, bFsci);
		CHECK_SUCCESS(status);
		/* ####################End Card Initialization########################### */

		/* Initialize the VCA component */
		status = phalVca_Sw_Init(&alVca,
			sizeof(alVca),					/**< [In] Specifies the size of the data parameter structure */
			&palMifare,						/**< [In] Pointer to a palMifare component context. */
			&keyStore,						/**< [In] Pointer to the parameter structure of the KeyStore component. */
			&cryptoEnc,						/**< [In] Pointer to a Crypto component context for encryption. */
			&cryptoRng,						/**< [In] Pointer to a CryptoRng component context. */
			aIidTableStorage,				/**< [In] Pointer to the Iid Table storage for the layer. */
			wNumIidTableStorageEntries,		/**< [In] Number of possible Iid table entries in the storage. */
			aCardTableStorage,				/**< [In] Pointer to the Card Table storage for the layer. */
			wNumCardTableStorageEntries  	/**< [In] Number of possible Card table entries in the storage. */
			);
		CHECK_SUCCESS(status);

		/* Initialize the MF DesFire EV1 component */
		status = phalMfdfEVx_Sw_Init(&alMfdfEVx,    /* [In] Pointer to this layer's parameter structure. */
			sizeof(phalMfdfEVx_Sw_DataParams_t),    /* [In] Specifies the size of the data parameter structure */
			&palMifare,                             /* [In] Pointer to a palMifare component context. */
			&keyStore,                              /* [In] Pointer to Key Store data parameters. */
			&cryptoEnc,                             /* [In] Pointer to a Crypto component context for encryption. */
			&cryptoMac,                             /* [In] Pointer to a CryptoMAC context */
			&cryptoRng,                             /* [In] Pointer to a CryptoRng component context. */
			&TMIDataParams,                         /* [In] Pointer to TMIUtils dataparams */
			&alVca,									/* [In] Pointer to Virtual Card dataparams */
			pHal);                                  /* [In] Pointer to the HAL parameters structure. */
		CHECK_SUCCESS(status);

		status = phalVca_SetApplicationType(&alVca, &alMfdfEVx);
		CHECK_SUCCESS(status);

		/* Set all the keys in the S/W key store */
		status = StoreKeysInKeyStore(&keyStore);
		CHECK_SUCCESS(status);

		/* Clear the console. */
		system ( "cls" );
		printf("A Ticket is discovered \n");
		printf("Press 1 for Personalization and Valuefile with TMAC demo App \n");
		printf("Press 2 for Garbage Collection demo App \n");
		printf("Press 3 for Keyset rolling demo App \n");
		printf("Press 4 for Delegated application management demo App ****This will throw AUTH_ERROR**** \n");
		printf("Press 5 for Shared Application Demo App\n");
		printf("Press 6 for UpdateRecordFile DemoApp\n");
		/*printf("Press 7 for Additional AccessRights DemoApp\n");*/
		printf("Press 7 for Symmetric Originality Check \n");
		printf("Press 8 for ASymmetric Originality Check \n");
		printf("Press 9 for Demo on ISO7816 commands with extended APDU Length \n");
		printf("Press a for Virtual Card Demo App\n");
		printf("Press b for Proximity Check Demo App\n");
		printf("Press c for Generate DAMMAC for Set Configuration option 0x06\n");
		printf("Press d for Get card UID \n");
		printf("Press e for GetVersion\n");
		printf("Press x to quit \n");
		iChoice = _getch();

		/* To See The Complete Log ,right Click on Console Window,Go to Properties->Layout->Screen Buffer Size->999 */
		switch(iChoice)
		{
		case('1'):
			/* Personalization and Valuefile with TMAC demo App */
			/* Pre personalization */

			status = PrePersonalization_AFCApp(&alMfdfEVx);
			CHECK_SUCCESS(status);
			printf("DOOR OPENED !!! \n");

			/* Accreditation */
			status = Accreditation(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Debit & Commit */
			status = CheckOutAfcApp(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('2'):
			/* Garbage collection */
			status = DemoGarbageCollection(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('3'):
			/* Keyset Rolling */
			status = DemoKeySetRolling(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('4'):
			/* Delegated Application management */
			/* This Function will exit with AUTH_ERROR as authentication of
			Delegated Application feature is not available with C# Simulator(version 5311)*/
			status = DemoDAM(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('5'):
			/* Shared Application Demo App */

			status = SharedAppDemo(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('6'):
			/* UpdateRecordFile demoAPP */
			status = UpdateLinearRecordFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

			/*March11, 2014 comment: Shall be worked on later*/
			/*case('7'):
			/*  Additional AccessRights demoAPP */
			/* status = AddAccessRightsDemo(&alMfdfEVx);
			CHECK_SUCCESS(status);*/

			/* Deselect */
			/*phhalHw_FieldReset(&halReader);
			break;*/

		case ('7'):
			printf("Selected Symmetric Originality Check demo(AES128 + OCKEYS) \n");
			OriginalityCheckDemo_Symmetric(&alMfdfEVx);
			break;

		case ('8'):
			printf("Selected ASymmetric Originality Check demo(ECDSA) \n");
			OriginalityCheckDemo_Asymmetric(&alMfdfEVx, &stCryptoASym);
			break;

		case ('9'):
			printf("Demo on ISO7816 commands with extended APDU Length \n");
			/* Create an Application */
			status = Create_Application(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Select the created application using ISOSelectFile command with extended APDU. */
			status = ISO_SelectFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* After selecting app, Create Standard data file */
			status = CreateStandardFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Select the standard data file using ISOSelectFile command with extended APDU*/
			status = SelectStandardFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Add data to standard data file using ISOUpdateBinary command with extended APDU */
			status = UpdateBinary(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Read the contents of the standard data file using ISOReadBinary command with extended APDU */
			status = ReadBinary(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Create a linear record file within the same application */
			status = CreateLinearRecordFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Select Linear record file using ISOSelectFile command with extended APDU. */
			status = SelectLinearRecordFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Write some data to the record file using ISOAppendRecord command with extended APDU */
			status = AppendLinearRecordFile(&alMfdfEVx);
			CHECK_SUCCESS(status);

			/* Read the data from record file using ISOReadRecord command with extended APDU */
			status = ReadRecords(&alMfdfEVx);
			CHECK_SUCCESS(status);
			break;

		case('a'):
			printf("\n\n******Do you want to run on MIFARE APPLET sample cards(1:Yes, 0:No)******\n");
			bRunOnMifareApplet = ('0' == _getch()) ? 0x00 : 0x01;

			/* Virtual Card Demo APP */
			status = VirtualCardDemo(&alMfdfEVx, &alVca);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;

		case('b'):
			printf("\n\n******Do you want to run on MIFARE APPLET sample cards(1:Yes, 0:No)******\n");
			bRunOnMifareApplet = ('0' == _getch()) ? 0x00 : 0x01;

			/* Virtual Card Demo APP */
			status = VirtualCardDemo(&alMfdfEVx, &alVca);
			CHECK_SUCCESS(status);

			/* Proximity Check Demo APP */
			status = ProximityCheckDemo(&alVca);
			CHECK_SUCCESS(status);

			/* Deselect */
			phhalHw_FieldReset(&halReader);
			break;
		case('c'):
			status = GenerateDAMMACSetConfigDemo(&alMfdfEVx);
			CHECK_SUCCESS(status);
			phhalHw_FieldReset(&halReader);
			break;

		case('d'):
			/* Authenticate with the PICC Master key */
			status = phalMfdfEVx_Authenticate(&alMfdfEVx,   /* [In] Pointer to parameters data structure */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
			PICCKEY,                                    /* [In] KEY number in key store */
			0x00,                                       /* [In] Key version in key store */
			0x00,                                       /* [In] Key number on Card */
			NULL,                                       /* [In] Diversification input */
			0);                                         /* [In] Diversification input length */
		CHECK_SUCCESS(status);

		status = phalMfdfEVx_GetCardUID(&alMfdfEVx,      /* [In] Pointer to parameters data structure */
						0x00,
						0x00,
						bCardUid);                          /* [Out] 7 Byte Card UID */
		CHECK_SUCCESS(status);

		break;

		case('e'):
			/* Authenticate with the PICC Master key */
			status = phalMfdfEVx_Authenticate(&alMfdfEVx,   /* [In] Pointer to parameters data structure */
			PHAL_MFDFEVX_NO_DIVERSIFICATION,            /* [In] Diversification option */
			PICCKEY,                                    /* [In] KEY number in key store */
			0x00,                                       /* [In] Key version in key store */
			0x00,                                       /* [In] Key number on Card */
			NULL,                                       /* [In] Diversification input */
			0);                                         /* [In] Diversification input length */
		CHECK_SUCCESS(status);

		status = phalMfdfEVx_GetVersion(&alMfdfEVx,      /* [In] Pointer to parameters data structure */
						bVersionString);                          /* [Out] 7 Byte Card UID */
		CHECK_SUCCESS(status);
		for(bIndex = 0; bIndex < sizeof(bVersionString); bIndex++)
		{
			printf("bVersionString[%d] = 0x%x\n", bIndex, bVersionString[bIndex]);
		}
		break;

		case 'x':
		case 'X':
			exit ( 0 );

		default:
			printf ( "\n\n" );
			printf ( "Invalid option selected.\n" );
			break;
		}

		printf("\n");
		printf ( "Press any key to continue with the demonstration.\n" );
		printf ( "Press x to exit the demonstration.\n" );
		iChoice = _getch ();
	}while( ( iChoice != 'x' ) && ( iChoice != 'X' ) );

	/* Returns zero if the procedure was successful. */
	return 0;
}