/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software ISO7816 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHwContact.h>
#include <phpalI7816p4.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I7816P4_SW

#include "phpalI7816p4_Sw_Int.h"
#include "phpalI7816p4_Sw.h"

phStatus_t phpalI7816p4_Sw_Cmd_SendI2CSBlock(
                                             phpalI7816p4_Sw_DataParams_t * pDataParams,
                                             uint8_t bType,
                                             uint8_t * pData,
                                             uint16_t wDataLength,
                                             uint8_t ** ppResponse,
                                             uint16_t * pResponseLength
                                             )
{
    phStatus_t status;
    phStatus_t statusTmp;
    uint8_t sblock[PHPAL_I7816P4_SW_GP_S_MAX_LEN];
    uint16_t sblockLen = 0;
    uint8_t * ptmpResponse;
    uint16_t wtmpResponseLen;
    uint32_t dwOldNumExpectedBytes;
    uint32_t dwRedundancyMode = 0;
    uint32_t dwCommunicationChannel = 0;
    uint16_t remainingBytes = 0;
    uint16_t wPayloadOffset = 0;

    ppResponse = NULL;
    pResponseLength = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL,
        &dwCommunicationChannel));

    /* Only supported for I2C/SPI communication */
    if (dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_I2C && dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI && dwCommunicationChannel != PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_SPI_EXCHANGE)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    /* CIP not available for i2c protocol t1overi2c */
    if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_PROTOCOL) == PHPAL_I7816P4_SW_I2C_PROTOCOL_T1OVERI2C &&
        bType == PHPAL_I7816P4_SW_GP_S_CIP)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    /* End of APDU session and get atr not available for i2c protocol global platform */
    if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_PROTOCOL) == PHPAL_I7816P4_SW_I2C_PROTOCOL_GLOBALPLATFORM &&
       (bType == PHPAL_I7816P4_SW_T1OI_S_ENDOFAPDUSESS || bType == PHPAL_I7816P4_SW_T1OI_S_GETATR))
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_I7816P4A);
    }

    if(bType == PHPAL_I7816P4_SW_S_RESYNCH ||
       bType == PHPAL_I7816P4_SW_S_ABORT ||
       bType == PHPAL_I7816P4_SW_S_SWR ||
       bType == PHPAL_I7816P4_SW_GP_S_CIP ||
       bType == PHPAL_I7816P4_SW_GP_S_RELEASE ||
       bType == PHPAL_I7816P4_SW_T1OI_S_ENDOFAPDUSESS ||
       bType == PHPAL_I7816P4_SW_T1OI_S_SECHIPRESET ||
       bType == PHPAL_I7816P4_SW_T1OI_S_GETATR)
    {
        /* no INF allowed */
        if(wDataLength != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }

        /* set length bytes to zero */
        sblock[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] =     0x00;     /* Length Field */

        if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_TWO)
        {
            sblock[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = 0x00;     /* Length Field */
        }
    }
    else
    {
        if((bType == PHPAL_I7816P4_SW_S_WTX && wDataLength != 1) ||
           (bType == PHPAL_I7816P4_SW_S_IFS && wDataLength != 1 && wDataLength != 2))
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_I7816P4);
        }

        if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
        {
             /* set length byte */
            sblock[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] = (uint8_t)wDataLength; /* Length Field */

            /* copy data to buffer */
            memcpy(&sblock[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE], pData, wDataLength);
        }
        else
        {
            /* set length bytes */
            sblock[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] =     0x00;                  /* Length Field */
            sblock[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1] = (uint8_t)wDataLength;  /* Length Field */

            /* copy data to buffer */
            memcpy(&sblock[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO], pData, wDataLength);
        }

        sblockLen += wDataLength;
    }

    /* Build S-Block */
    sblock[PHPAL_I7816P4_SW_HEADER_NAD_OFFSET] = pDataParams->bNad;                      /* NAD */
    sblock[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] = PHPAL_I7816P4_SW_S_REQUEST | bType;     /* PCB Byte */

    wPayloadOffset = ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE) ?
                     PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_ONE : PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO;
    sblockLen += wPayloadOffset;

    /* Set Tx only mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, &dwRedundancyMode));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, (dwRedundancyMode & (PHHAL_HW_CONTACT_MODE_MASK_TX | PHHAL_HW_CONTACT_MODE_MASK_TX_SWAPPED))));

    /* set number expected bytes */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, &dwOldNumExpectedBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, wPayloadOffset));

    /* Perform Exchange */
    status = phhalHwContact_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        sblock,
        sblockLen,
        &ptmpResponse,
        &wtmpResponseLen);

    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Got Header now request remaining bytes if there are any */
        if ((dwRedundancyMode & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
        {
            remainingBytes = 1;
        }
        else if ((dwRedundancyMode & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            remainingBytes = 2;
        }
        else
        {
            remainingBytes = 0;
        }

        if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
        {
            remainingBytes += ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET];
        }
        else
        {
            remainingBytes += (ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) + ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1];
        }

        if(remainingBytes > 0)
        {
            /* Set Full buffer check mode and only rx mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE,
                ((dwRedundancyMode & (PHHAL_HW_CONTACT_MODE_MASK_RX | PHHAL_HW_CONTACT_MODE_MASK_RX_SWAPPED)) | PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)));

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams,
                PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS, wPayloadOffset));

            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, remainingBytes));

            status = phhalHwContact_Exchange(
                pDataParams->pHalDataParams,
                PH_EXCHANGE_DEFAULT,
                NULL,
                0,
                &ptmpResponse,
                &wtmpResponseLen);
        }
    }

    /* Reset settings */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, dwRedundancyMode));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES, dwOldNumExpectedBytes));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams->pHalDataParams, PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS, 0));

    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* request and response type have to match */
        if(ptmpResponse[PHPAL_I7816P4_SW_HEADER_PCB_OFFSET] != (PHPAL_I7816P4_SW_S_RESPONSE | bType))
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
        }

        if(bType == PHPAL_I7816P4_SW_S_RESYNCH ||
           bType == PHPAL_I7816P4_SW_S_ABORT ||
           (bType == PHPAL_I7816P4_SW_S_SWR && ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_PROTOCOL) == PHPAL_I7816P4_SW_I2C_PROTOCOL_GLOBALPLATFORM)) ||
           bType == PHPAL_I7816P4_SW_GP_S_RELEASE ||
           bType == PHPAL_I7816P4_SW_T1OI_S_ENDOFAPDUSESS ||
           bType == PHPAL_I7816P4_SW_T1OI_S_SECHIPRESET)
        {
            /* no INF allowed */
            if(wtmpResponseLen != wPayloadOffset)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
            }
        }
        else
        {
            /* inf field has to match */
            if(bType == PHPAL_I7816P4_SW_S_WTX || bType == PHPAL_I7816P4_SW_S_IFS)
            {
                if(wDataLength != (wtmpResponseLen - wPayloadOffset))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }

                if((ptmpResponse[wPayloadOffset] != pData[0]) ||
                   (wtmpResponseLen == PHPAL_I7816P4_SW_GP_S_MAX_LEN && ptmpResponse[PHPAL_I7816P4_SW_HEADER_PAY_OFFSET_LEN_TWO + 1] != pData[1]))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }
            }

            if ((pDataParams->wI2CProtocol & PHPAL_I7816P4_SET_CONFIG_MASK_LEN_SIZE) == PHPAL_I7816P4_SW_I2C_PROTOCOL_LEN_ONE)
            {
               /* LEN field has to match the actual received length */
                if(ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] != (wtmpResponseLen - wPayloadOffset))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }
            }
            else
            {
                /* LEN field has to match the actual received length */
                if(((ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET] << 8) + ptmpResponse[PHPAL_I7816P4_SW_HEADER_LEN_OFFSET + 1]) != (wtmpResponseLen - wPayloadOffset))
                {
                    return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_I7816P4);
                }
            }
        }

        /* success so copy buffers */
        memcpy(pDataParams->pIntBuffer, ptmpResponse, wtmpResponseLen);
        pDataParams->wIntBufferLen = wtmpResponseLen;
        ppResponse = (uint8_t**)&pDataParams->pIntBuffer;
        pResponseLength = &pDataParams->wIntBufferLen;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_PAL_I7816P4);
}

#endif /* NXPBUILD__PHPAL_I7816P4_SW */
